package com.engisis.sysphs.serialization.simulink;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SConnectionPoint;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SLibrary;
import com.engisis.sysphs.language.simulink.SLine;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SSystemParameter;
import com.engisis.sysphs.language.simulink.SSystemParameterAssignment;
import com.engisis.sysphs.util.TextWriter;

/**
 * Class in charge of writing Simulink models into the MDL format
 * @author barbau
 *
 */
public class SimulinkMDLWriter extends TextWriter {
    
    /**
     * Constructs a Simulink MDL writer
     * @param w Object to which the model is written into
     */
	protected SimulinkMDLWriter(Writer w) {
		super(w);
	}
	
	/**
	 * Calculates a graphical position for the block
	 * @param sblock block for which the position is calculated
	 * @throws IOException
	 */
	public void positionBlock(SBlock sblock) throws IOException
	{
		if (sblock.getOwningSystem() != null)
		{
			int pos = sblock.getOwningSystem().getOwnedBlocks().indexOf(sblock);
			int w = 20;
			int h = 20;
			int m = 20;
			int x1 = (m + m * pos) + (pos*w);
			int x2 = x1+w;
			int y1 = m;
			int y2 = y1+h;
			String position = "[" + x1 + "," + y1 + "," + x2 + "," + y2 + "]";
			append("Position " + position, TL);
		}
	}
	
	public void introBlock() throws IOException
	{
        append("Block", TL);
		append("{", TL);
        indent();
	}
	public void defaultBlock(SBlock sblock) throws IOException
	{
        if (sblock.getName() != null && !sblock.getName().isEmpty())
        {
            append("Name \"" + sblock.getName() + "\"", TL);
        }
        if (sblock.getSID() != null && !sblock.getSID().isEmpty())
        {
        	append("SID \"" + sblock.getSID() + "\"", TL);
        }
	}
	
	public void outroBlock() throws IOException
	{
		unindent();
        append("}", TL);
	}

	public void visit(SExpressionValue object) throws IOException {
		append(object.getValue());
	}

	public void visit(SFunction1Block object) throws IOException {
		introBlock();
        append("BlockType S-Function", TL);
		defaultBlock(object);
        if (object.getSfunction() != null && object.getSfunction().getName() != null
                && !object.getSfunction().getName().isEmpty())
            append("FunctionName \"" + object.getSfunction().getName() + "\"", TL);
        if (object.getSfunction().getOwnedBlocks().size() == 0)
        	append("Ports [1,1]", TL);
        else
        {
        	int i = 0;
        	int o = 0;
        	for(SBlock sbl : object.getSfunction().getOwnedBlocks())
        	{
        		if (sbl instanceof SInport)
        			i++;
        		else if (sbl instanceof SOutport)
        			o++;
        	}
        	append("Ports [" + i + "," + o + "]", TL);
        }
		positionBlock(object);
        outroBlock();
	}

	public void visit(SFunction2Block object) throws IOException {
		introBlock();
        append("BlockType M-S-Function", TL);
		defaultBlock(object);
        if (object.getSfunction() != null && object.getSfunction().getName() != null
                && !object.getSfunction().getName().isEmpty())
            append("FunctionName \"" + object.getSfunction().getName() + "\"", TL);
        append("Ports ["+ object.getSfunction().getInputs().size() + "," + 
            object.getSfunction().getOutputs().size() + "]", TL);
		positionBlock(object);
        outroBlock();
	}

	public void visit(SInport object) throws IOException {
		introBlock();
        append("BlockType Inport", TL);
		defaultBlock(object);
		positionBlock(object);
        String unit = object.getUnit();
        if (unit != null)
            append("Unit \"" + unit + "\"");
        outroBlock();
	}

	public void visit(SInterface object) throws IOException {
		introBlock();
    	SSystem ssystem = object.getSystem();
        if (ssystem != null)
        {
            append("BlockType " + ssystem.getName(), TL);
        	defaultBlock(object);
        	//if (ssystem.getInports().size() != 1 || ssystem.getOutports().size() != 1)
        		append("Ports [" + ssystem.getInports().size() + ","
        				+ ssystem.getOutports().size() + "]", TL);
    		positionBlock(object);
            for(SSystemParameter sparameter : object.getSystem().getSystemparameters())
            {
            	SDataValue val = sparameter.getValue();
            	for(SSystemParameterAssignment sassignment : object.getAssignments())
            		if (sassignment.getParameter() == sparameter)
            			val = sassignment.getValue();

            	if (val != null)
            	{
	            	append(sparameter.getName() + " \"", T);
	            	val.accept(getDispatcher());
	            	append("\"", L);
            	}
            }
        }
        outroBlock();
	}

	public void visit(SLibrary object) throws IOException {
        append("Library", TL);
        append("{", TL);
        indent();
        if (object.getName() != null && !object.getName().isEmpty())
            append("Name \"" + object.getName() + "\"", TL);
        append("Version 8.2", TL);
        append("MdlSubVersion 0", TL);
        if (object.getSystem() != null)
            object.getSystem().accept(getDispatcher());
        
        unindent();
        append("}", TL);
	}

	public void visit(SLine object) throws IOException {
        append("Line", TL);
        append("{", TL);
        indent();
        
        SBlock sbsrcblock = object.getSource().getBlock();
        SBlock sbsrcport = object.getSource().getPort();
        
		append("SrcBlock \"" + sbsrcblock.getName() + "\"", TL);
        
        int srcport = 0;
		if (sbsrcport == null)
            srcport = 1;
		else
		{
            if (sbsrcport instanceof SInport)
                srcport = sbsrcport.getOwningSystem().getInports()
                .indexOf(sbsrcport) + 1;
            else if (sbsrcport instanceof SOutport)
                srcport = sbsrcport.getOwningSystem().getOutports()
                .indexOf(sbsrcport) + 1;
		}
        append("SrcPort " + srcport, TL);
        
        for (SConnectionPoint sdestination : object.getDestinations())
        {
            SBlock sbdstblock = sdestination.getBlock();
            SBlock sbdstport = sdestination.getPort();
            
			String dstblock = sbdstblock.getName();
            int dstport = 0;
			if (sbdstport == null)
                dstport = 1;
			else
			{
                if (sbdstport instanceof SInport)
                	dstport = sbdstport.getOwningSystem().getInports()
                    .indexOf(sbdstport) + 1;
                else if (sbdstport instanceof SOutport)
                	dstport = sbdstport.getOwningSystem().getOutports()
                    .indexOf(sbdstport) + 1;
			}
            if (object.getDestinations().size() > 1)
            {
                append("Branch", TL);
                append("{", TL);
                indent();
                append("DstBlock \"" + dstblock + "\"", TL);
                append("DstPort " + dstport, TL);
                unindent();
                append("}", TL);
            }
            else
            {
                append("DstBlock \"" + dstblock + "\"", TL);
                append("DstPort " + dstport, TL);
            }
        }
        unindent();
        append("}", TL);
	}

	public void visit(SModel object) throws IOException {
        append("Model", TL);
        append("{", TL);
        indent();
        if (object.getName() != null && !object.getName().isEmpty())
            append("Name \"" + object.getName() + "\"", TL);
        append("Version 8.2", TL);
        append("MdlSubVersion 0", TL);
        if (object.getSystem() != null)
            object.getSystem().accept(getDispatcher());
        append("}", TL);
	}

	public void visit(SOutport object) throws IOException {
		introBlock();
        append("BlockType Outport", TL);
		defaultBlock(object);
		positionBlock(object);
        String unit = object.getUnit();
        if (unit != null)
            append("Unit \"" + unit + "\"");
        outroBlock();
	}

	public void visit(SDoubleValue object) throws IOException {
		append(String.valueOf(object.getValue()));
	}

	public void visit(SReference object) throws IOException {
		introBlock();
        append("BlockType Reference", TL);
		defaultBlock(object);
        if (object.getSystem() != null)
        {
            append("Ports [" + object.getSystem().getInports().size() + ","
                    + object.getSystem().getOutports().size() + "]", TL);
            append("SourceBlock \"" + object.getSystem().getPath() + "\"", TL);
            append("SourceType \"SubSystem\"", TL);
        }
		positionBlock(object);
        outroBlock();
	}

	public void visit(SSubsystem object) throws IOException {
		introBlock();
        append("BlockType SubSystem", TL);
		defaultBlock(object);
        if (object.getSystem() != null)
        {
            append("Ports [" + object.getSystem().getInports().size() + ","
                    + object.getSystem().getOutports().size() + "]", TL);
            object.getSystem().accept(getDispatcher());
        }
		positionBlock(object);
        outroBlock();
	}

	public void visit(SSystem object) throws IOException {
        append("System", TL);
        append("{", TL);
        indent();
        printSystem(object);
        unindent();
        append("}", TL);
	}
	
	protected void printSystem(SSystem ssystem) throws IOException
	{
        if (ssystem.getName() != null && !ssystem.getName().isEmpty())
            append("Name \"" + ssystem.getName() + "\"", TL);
        
        for (SBlock sblock : ssystem.getOwnedBlocks())
            sblock.accept(getDispatcher());
        for (SLine sline : ssystem.getOwnedLines())
            sline.accept(getDispatcher());
        
        // additional code depending on the block
        for (SBlock sblock : ssystem.getOwnedBlocks())
        {
        	printSystemCode(sblock);
        }
	}
	
	protected void printSystemCode(SBlock sblock) throws IOException
	{
		if (sblock instanceof SFunction2Block)
		{
			SFunction2Block sf2block = (SFunction2Block)sblock;
	        // additional lines to connect inputs/outputs to sfunctions
	        List<SFInputVariable> inputs = sf2block.getSfunction().getInputs();
	        List<SFOutputVariable> outputs = sf2block.getSfunction().getOutputs();
	        
	        if (inputs != null)
	        {
	            for (int i = 0; i < inputs.size(); i++)
	            {
	                SFInputVariable sinputvariable = inputs.get(i);
	                SInport sinport = sinputvariable.getInport();
	                if (sinport != null)
	                {
	                    append("Line", TL);
	                    append("{", TL);
	                    append("SrcBlock \"" + sinport.getName() + "\"", TL);
	                    append("SrcPort 1", TL);
	                    append("DstBlock \"" + sf2block.getName() + "\"", TL);
	                    append("DstPort " + (i + 1), TL);
	                    append("}", TL);
	                }
	            }
	        }
	        if (outputs != null)
	        {
	            for (int i = 0; i < outputs.size(); i++)
	            {
	                SFOutputVariable soutputvariable = outputs.get(i);
	                SOutport soutport = soutputvariable.getOutport();
	                if (soutport != null)
	                {
	                	append("Line", TL);
	                	append("{", TL);
	                	append("SrcBlock \"" + sf2block.getName() + "\"", TL);
	                	append("SrcPort " + (i + 1), TL);
	                    append("DstBlock \"" + soutport.getName() + "\"", TL);
	                    append("DstPort 1", TL);
	                    append("}", TL);
	                }
	            }
	        }
		}
	}
}
