package com.engisis.sysphs.serialization.simulink;

import java.io.IOException;
import java.io.Writer;
import java.util.LinkedList;
import java.util.List;

import com.engisis.sysphs.language.simscape.SBranch;
import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SConnection;
import com.engisis.sysphs.language.simscape.SConversion;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SEquation;
import com.engisis.sysphs.language.simscape.SInput;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAccess;
import com.engisis.sysphs.language.simscape.SMemberAssignment;
import com.engisis.sysphs.language.simscape.SMemberPath;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SOutput;
import com.engisis.sysphs.language.simscape.SParameter;
import com.engisis.sysphs.language.simscape.SPropagation;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.util.TextWriter;

/**
 * Class in charge of writing Simscape models
 * 
 * @author barbau
 *
 */
public class SimscapeWriter extends TextWriter
{
    /**
     * Constructs a Simscape writer
     * 
     * @param w
     *            object to write the model into
     */
    public SimscapeWriter(Writer w)
    {
        super(w);
    }
    
    public void visit(SBranch object) throws IOException
    {
        append(object.getVariable().getName() + ": ", T);
        if (object.getFrom() != null)
            object.getFrom().accept(getDispatcher());
        else
            append("*");
        append("->");
        if (object.getTo() != null)
            object.getTo().accept(getDispatcher());
        else
            append("*");
        append(";", L);
    }
    
    public void visit(SComponent object) throws IOException
    {
        
        append("component", T);
        if (object.isHidden()
                || (object.getPropagation() != null && object.getPropagation() != SPropagation.PROPAGATES))
        {
            boolean first = true;
            append("(");
            if (object.isHidden())
            {
                append("Hidden=true");
                first = false;
            }
            if (object.getPropagation() != null && object.getPropagation() != SPropagation.PROPAGATES)
            {
                if (!first)
                    append(",");
                append("Propagation=");
                switch (object.getPropagation())
                {
                case BLOCKS:
                    append("blocks");
                    break;
                case PROPAGATES:
                    append("propagates");
                    break;
                case SOURCE:
                    append("source");
                    break;
                default:
                    break;
                }
            }
            append(")");
        }
        append(" " + object.getName());
        if (object.getBaseComponent() != null)
            append(" < " + object.getBaseComponent().getQualifiedName());
        append("", L);
        indent();
        
        // Members
        LinkedList<SMember> smembers = new LinkedList<SMember>(object.getOwnedMembers());
        while (smembers.size() != 0)
        {
            SMember smember = smembers.get(0);
            getHeader(smember);
            indent();
            smember.accept(getDispatcher());
            
            int i = 1;
            while (i < smembers.size())
            {
                if (shareHeader(smember, smembers.get(i)))
                {
                    smembers.get(i).accept(getDispatcher());
                    smembers.remove(i);
                    continue;
                }
                i++;
            }
            smembers.remove(0);
            unindent();
            append("end", TL);
        }
        
        // Across, Through, Assignments
        append("function setup", TL);
        indent();
        for (SMemberAssignment sassignment : object.getOwnedAssignments())
            sassignment.accept(getDispatcher());
        unindent();
        append("end", TL);
        
        // Branches
        if (object.getBranches().size() != 0)
        {
            append("branches", TL);
            indent();
            for (SBranch sbranch : object.getBranches())
                sbranch.accept(getDispatcher());
            unindent();
            append("end", TL);
        }
        
        // Connections
        if (object.getOwnedConnections().size() != 0)
        {
            append("connections", TL);
            indent();
            for (SConnection sconnection : object.getOwnedConnections())
            {
                append("connect(", T);
                sconnection.accept(getDispatcher());
                append(");", L);
            }
            unindent();
            append("end", TL);
        }
        
        // Equations
        if (object.getOwnedEquations().size() != 0)
        {
            append("equations", TL);
            indent();
            for (SEquation sequation : object.getOwnedEquations())
                sequation.accept(getDispatcher());
            unindent();
            append("end", TL);
        }
        unindent();
        append("end", TL);
    }
    
    public void getHeader(SMember smember) throws IOException
    {
        if (smember instanceof SComponentReference)
            append("components", T);
        else if (smember instanceof SParameter)
            append("parameters", T);
        else if (smember instanceof SVariable)
            append("variables", T);
        else if (smember instanceof SInput)
            append("inputs", T);
        else if (smember instanceof SOutput)
            append("outputs", T);
        else if (smember instanceof SNode)
            append("nodes", T);
        
        List<String> options = new LinkedList<String>();
        if (smember.isHidden())
            options.add("Hidden=true");
        if (smember.getAccess() != null && smember.getAccess() != SMemberAccess.PUBLIC)
            options.add("Access=" + smember.getAccess().name().toLowerCase());
        if (smember instanceof SParameter)
            if (((SParameter) smember).getConversion() != SConversion.ABSOLUTE)
                options.add("Conversion=relative");
        if (smember instanceof SVariable)
            if (((SVariable) smember).isBalancing())
                options.add("Balancing=true");
            
        if (options.size() > 0)
        {
            append("(");
            for (int i = 0; i < options.size(); i++)
            {
                if (i != 0)
                    append(",");
                append(options.get(i));
            }
            append(")");
        }
        append("", L);
    }
    
    public static boolean shareHeader(SMember smember1, SMember smember2)
    {
        if (smember1 == null || smember2 == null)
            throw new IllegalArgumentException("The compared members must not be null");
        if (smember1.getClass() != smember2.getClass())
            return false;
        if (smember1 instanceof SParameter
                && ((SParameter) smember1).getConversion() != ((SParameter) smember2).getConversion())
            return false;
        if (smember1 instanceof SVariable
                && ((SVariable) smember1).isBalancing() != ((SVariable) smember2).isBalancing())
            return false;
        return smember1.isHidden() == smember2.isHidden() && smember1.getAccess() == smember2.getAccess();
    }
    
    public void visit(SComponentReference object) throws IOException
    {
        append(object.getName() + "=" + object.getComponent().getQualifiedName() + ";", TL);
    }
    
    public void visit(SConnection object) throws IOException
    {
        for (int i = 0; i < object.getPoints().size(); i++)
        {
            if (i != 0)
                append(",");
            object.getPoints().get(i).accept(getDispatcher());
        }
    }
    
    public void visit(SMemberPath object) throws IOException
    {
        for (int i = 0; i < object.getPath().size(); i++)
        {
            if (i != 0)
                append(".");
            append(object.getPath().get(i).getName());
        }
    }
    
    public void visit(SDomain object) throws IOException
    {
        append("domain " + object.getName(), TL);
        // Members
        indent();
        LinkedList<SParameter> sparameters = new LinkedList<SParameter>(object.getParameters());
        while (sparameters.size() != 0)
        {
            SParameter sparameter = sparameters.get(0);
            getHeader(sparameter);
            indent();
            sparameter.accept(getDispatcher());
            int i = 1;
            while (i < sparameters.size())
            {
                if (shareHeader(sparameter, sparameters.get(i)))
                {
                    sparameters.get(i).accept(getDispatcher());
                    sparameters.remove(i);
                    continue;
                }
                i++;
            }
            sparameters.remove(0);
            unindent();
            append("end", TL);
        }
        
        // Members
        LinkedList<SVariable> svariables = new LinkedList<SVariable>(object.getVariables());
        while (svariables.size() != 0)
        {
            SVariable svariable = svariables.get(0);
            // domain variables have no initial value
            svariable.setValue(null);
            getHeader(svariable);
            svariable.accept(getDispatcher());
            indent();
            int i = 1;
            while (i < svariables.size())
            {
                if (shareHeader(svariable, svariables.get(i)))
                {
                    SVariable svariable2 = svariables.get(i);
                    svariable2.setValue(null);
                    svariable2.accept(getDispatcher());
                    svariables.remove(i);
                    continue;
                }
                i++;
            }
            svariables.remove(0);
            unindent();
            append("end", TL);
        }
        unindent();
        append("end", TL);
    }
    
    public void visit(SDoubleValue object) throws IOException
    {
        append(Double.toString(object.getValue()));
    }
    
    public void visit(SEquation object) throws IOException
    {
        append(object.getExpression() + ";", TL);
    }
    
    public void visit(SExpressionValue object) throws IOException
    {
        append(object.getValue());
    }
    
    public void visit(SInput object) throws IOException
    {
        append(object.getName() + "=", T);
        String unit = object.getUnit();
        SDataValue value = object.getValue();
        
        if (unit != null)
        {
            append("{");
            valueOrZero(value);
            append(",'" + unit + "'}");
        }
        else
            valueOrZero(value);
        
        append(";");
        if (object.getLocation() == SLocation.RIGHT)
            append("% :right", L);
        else
            append("% :left", L);
    }
    
    private void valueOrZero(SDataValue svalue) throws IOException
    {
        if (svalue == null)
            append("0");
        else
            svalue.accept(getDispatcher());
    }
    
    public void visit(SMemberAssignment object) throws IOException
    {
        int sz = object.getMemberPath().size();
        boolean isparam = sz > 0 ? object.getMemberPath().get(sz - 1) instanceof SParameter : false;
        append("", T);
        for (int i = 0; i < sz; i++)
        {
            SMember smember = object.getMemberPath().get(i);
            if (i != 0)
                append(".");
            append(smember.getName());
        }
        append("=");
        if (object.getAssignedReference().size() != 0)
            for (int i = 0; i < object.getAssignedReference().size(); i++)
            {
                SMember smember = object.getAssignedReference().get(i);
                if (i != 0)
                    append(".");
                append(smember.getName());
            }
        else
        {
            // assignments of variables are always high priority, that's the
            // point of having them
            if (!isparam)
                append("{value=");
            String unit = null;
            SMember lastmember = object.getMemberPath().get(object.getMemberPath().size() - 1);
            if (lastmember instanceof SVariable)
                unit = ((SVariable) lastmember).getUnit();
            else if (lastmember instanceof SParameter)
                unit = ((SParameter) lastmember).getUnit();
            if (unit != null)
                append("{");
            object.getAssignedValue().accept(getDispatcher());
            if (unit != null)
                append(",'" + unit + "'}");
            if (!isparam)
                append(", priority=priority.high}");
        }
        append(";", L);
    }
    
    public void visit(SNode object) throws IOException
    {
        append(object.getName() + "=" + object.getDomain().getQualifiedName() + ";", T);
        if (object.getLocation() == SLocation.RIGHT)
            append("% :right", L);
        else
            append("% :left", L);
    }
    
    public void visit(SOutput object) throws IOException
    {
        append(object.getName() + "=", T);
        String unit = object.getUnit();
        SDataValue value = object.getValue();
        
        if (unit != null)
        {
            append("{");
            valueOrZero(value);
            append(",'" + unit + "'}");
        }
        else
            valueOrZero(value);
        
        append(";");
        if (object.getLocation() == SLocation.RIGHT)
            append("% :right", L);
        else
            append("% :left", L);
    }
    
    public void visit(SParameter object) throws IOException
    {
        append(object.getName() + "=", T);
        String unit = object.getUnit();
        SDataValue value = object.getValue();
        
        if (unit != null)
        {
            append("{");
            valueOrZero(value);
            append(",'" + unit + "'}");
        }
        else
            valueOrZero(value);
        
        append(";", L);
    }
    
    public void visit(SVariable object) throws IOException
    {
        append(object.getName() + "=", T);
        String unit = object.getUnit();
        SDataValue value = object.getValue();
        
        // values here are never in high priority.
        // assignments should override them with priority
        // if (value != null)
        // {
        // append("{value=");
        // }
        if (unit != null)
        {
            append("{");
            valueOrZero(value);
            append(",'" + unit + "'}");
        }
        else
            valueOrZero(value);
        // if (value != null)
        // {
        // append(", priority=priority.high}");
        // }
        append(";", L);
    }
    
}
