package com.engisis.sysphs.deserialization.simulink;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Hashtable;

import org.apache.log4j.Logger;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.stateflow.SChart;
import com.engisis.sysphs.language.stateflow.SConnectNode;
import com.engisis.sysphs.language.stateflow.SInstance;
import com.engisis.sysphs.language.stateflow.SMachine;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.language.stateflow.STransition;
import com.engisis.sysphs.language.stateflow.StateflowFactory;
import com.engisis.sysphs.util.Util;
import com.engisis.sysphs.util.XMLReader;

/**
 * Second-pass parser of a Stateflow XML file.
 * 
 * @author barbau
 *
 */
public class StateflowXMLReaderPass2 extends XMLReader implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(StateflowXMLReaderPass2.class);
    
    /**
     * Collection of Simulink references
     */
    private Collection<SimulinkReference> references;
    /**
     * Map with Integer as keys, and Simulink elements as values
     */
    private Hashtable<Integer, SElement> ids;
    /**
     * Input stream of the XML file
     */
    private InputStream is;
    
    /**
     * Constructs a second-pass Stateflow XML parser
     * 
     * @param sxmlr
     *            first-pass reader
     * @param inputStream
     *            input XML stream
     */
    public StateflowXMLReaderPass2(StateflowXMLReaderPass1 sxmlr, InputStream inputStream)
    {
        references = sxmlr.getReferences();
        ids = sxmlr.getIDs();
        this.is = inputStream;
    }
    
    @Override
    public void process()
    {
        // open xml
        try
        {
            XmlPullParserFactory xmlppf = XmlPullParserFactory.newInstance();
            xmlpp = xmlppf.newPullParser();
            xmlpp.setInput(is, null);
            lookForElement("ModelInformation");
            processModelInformation();
        }
        catch (XmlPullParserException e)
        {
            log.error(e);
        }
        catch (IOException e)
        {
            log.error(e);
        }
        finally
        {
            try
            {
                is.close();
            }
            catch (IOException e)
            {
            }
        }
    }
    
    /**
     * Processes a ModelInformation element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processModelInformation() throws XmlPullParserException, IOException
    {
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        String ln2;
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("Stateflow"))
                {
                    processStateflow();
                }
                else
                    skipToEnd();
            }
            
        }
    }
    
    /**
     * Processes a Stateflow Element
     * 
     * @return a Steteflow object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SStateflow processStateflow() throws XmlPullParserException, IOException
    {
        SStateflow sstateflow = StateflowFactory.eINSTANCE.createSStateflow();
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                // we don't care about the hierarchy
                String name = xmlpp.getName();
                if (name.equals("instance"))
                {
                    processInstance();
                }
                else if (name.equals("machine"))
                {
                    processMachine();
                }
                else
                    skipToEnd();
            }
        }
        return sstateflow;
    }
    
    /**
     * Processes an Instance element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processInstance() throws XmlPullParserException, IOException
    {
        log.info("Resolving Instance");
        String sid = xmlpp.getAttributeValue(null, "ID");
        SElement selement = getElement(sid);
        if (selement instanceof SInstance)
        {
            SInstance sinstance = (SInstance) selement;
            
            int depth = xmlpp.getDepth();
            int et = xmlpp.getEventType();
            
            while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
            {
                if (et == XmlPullParser.START_TAG)
                {
                    if (xmlpp.getName().equals("P"))
                    {
                        String name = xmlpp.getAttributeValue(null, "Name");
                        String val = xmlpp.nextText();
                        if ("machine".equals(name))
                        {
                            selement = getElement(val);
                            if (selement instanceof SMachine)
                                sinstance.setMachine((SMachine) selement);
                        }
                        else if ("chart".equals(name))
                        {
                            selement = getElement(val);
                            if (selement instanceof SChart)
                                sinstance.setChart((SChart) selement);
                        }
                    }
                    else
                        skipToEnd();
                }
            }
        }
    }
    
    /**
     * Processes a Machine element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processMachine() throws XmlPullParserException, IOException
    {
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("Children"))
                {
                    int depth2 = xmlpp.getDepth();
                    while (et != XmlPullParser.END_TAG || depth2 != xmlpp.getDepth())
                    {
                        et = xmlpp.next();
                        if (et == XmlPullParser.START_TAG)
                        {
                            name = xmlpp.getName();
                            if (name.equals("chart"))
                            {
                                processChart();
                            }
                            else
                                skipToEnd();
                        }
                    }
                }
                else
                    skipToEnd();
            }
        }
    }
    
    /**
     * Processes a Chart element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processChart() throws XmlPullParserException, IOException
    {
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("Children"))
                {
                    int depth2 = xmlpp.getDepth();
                    while (et != XmlPullParser.END_TAG || depth2 != xmlpp.getDepth())
                    {
                        et = xmlpp.next();
                        if (et == XmlPullParser.START_TAG)
                        {
                            String name2 = xmlpp.getName();
                            if (name2.equals("transition"))
                            {
                                processTransition();
                            }
                            else
                                skipToEnd();
                        }
                    }
                }
                else
                    skipToEnd();
            }
        }
    }
    
    /**
     * Processes a Transition element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processTransition() throws XmlPullParserException, IOException
    {
        String sid = xmlpp.getAttributeValue(null, "SSID");
        SElement selement = getElement(sid);
        if (selement instanceof STransition)
        {
            log.info("Resolving Transition");
            STransition stransition = (STransition) selement;
            
            int depth = xmlpp.getDepth();
            int et = xmlpp.getEventType();
            
            while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
            {
                et = xmlpp.next();
                if (et == XmlPullParser.START_TAG)
                {
                    String name = xmlpp.getName();
                    if (name.equals("src"))
                    {
                        int depth2 = xmlpp.getDepth();
                        while (et != XmlPullParser.END_TAG || xmlpp.getDepth() != depth2)
                        {
                            et = xmlpp.next();
                            if (et == XmlPullParser.START_TAG && xmlpp.getName().equals("P"))
                            {
                                name = xmlpp.getAttributeValue(null, "Name");
                                if ("SSID".equals(name))
                                {
                                    selement = getElement(xmlpp.nextText());
                                    if (selement instanceof SConnectNode)
                                    {
                                        log.info("Setting transition source");
                                        stransition.setSrc((SConnectNode) selement);
                                    }
                                    else
                                        log.error("Unable to set transition source");
                                }
                            }
                        }
                    }
                    else if (name.equals("dst"))
                    {
                        int depth2 = xmlpp.getDepth();
                        while (et != XmlPullParser.END_TAG || xmlpp.getDepth() != depth2)
                        {
                            et = xmlpp.next();
                            if (et == XmlPullParser.START_TAG && xmlpp.getName().equals("P"))
                            {
                                name = xmlpp.getAttributeValue(null, "Name");
                                if ("SSID".equals(name))
                                {
                                    selement = getElement(xmlpp.nextText());
                                    if (selement instanceof SConnectNode)
                                    {
                                        log.info("Setting transition target");
                                        stransition.setDst((SConnectNode) selement);
                                    }
                                    else
                                        log.error("Unable to set transition target");
                                }
                            }
                        }
                    }
                    else
                        skipToEnd();
                }
            }
        }
    }
    
    /**
     * Returns the Simulink element identified by the given ID
     * 
     * @param sid
     *            ID of the needed element
     * @return Simulink element
     */
    @SuppressWarnings("boxing")
    private SElement getElement(String sid)
    {
        return ids.get(Util.toInt(sid, 0));
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    /**
     * Sets the map with ID as keys, and Simulink elements as values
     * 
     * @param ids
     *            map with ID as keys, and Simulink elements as values
     */
    public void setIDs(Hashtable<Integer, SElement> ids)
    {
        this.ids = ids;
    }
    
    @Override
    public int getPriority()
    {
        return 15;
    }
    
    @Override
    public SElement getRootElement()
    {
        return null;
    }
    
}
