from refl1d.names import Parameter, SLD, Slab, Experiment, FitProblem, load4

## === Probes/data files ===
probe_dipa = load4('ch127_tiox_dipa.refl', back_reflectivity=True, name='d-isopropanol')

# Probe parameters
# Backgrounds (one for each model)
probe_dipa.background.range(-1e-7, 1e-5)

# Intensity (common to all models)
probe_dipa.intensity.range(0.95,1.05)

# Sample broadening (common to all models)
probe_dipa.sample_broadening.range(-0.003, 0.02)

# Theta offset (common to all models)
probe_dipa.theta_offset.range(-0.02, 0.02)

## === Custom parameters ===
tiox_roughness = Parameter(name='tiox roughness', value=5.0).range(2, 15.0)
tiox_thickness = Parameter(name='tiox thickness', value=140.0).range(80, 200)
siox_thickness = Parameter(name='siox thickness', value=20.0).range(2.0, 30.0)
silicon_roughness = Parameter(name='si roughness', value=2.0).range(2, 15.0)

## === Samples ===

# Material definitions
d_ipa = SLD(name='d-isopropanol', rho=6.3000, irho=0.0000)
tiox = SLD(name='tiox', rho=2.1630, irho=0.0000)
siox = SLD(name='siox', rho=4.1000, irho=0.0000)
silicon = SLD(name='silicon', rho=2.0690, irho=0.0000)

# Material SLD parameters
d_ipa.rho.range(5.3000, 6.36)
tiox.rho.range(1.2, 3.2)
siox.rho.range(2.8, 4.8)

# Sample stack definition

# Layer 3: semi-infinite layer filled with buffer
layer_water = Slab(material=d_ipa, thickness=0.0000, interface=0.0000)

# Layer 2: sputtered TiO2, nominal thickness 140 Ang
layer_tiox = Slab(material=tiox, thickness=tiox_thickness, interface=tiox_roughness)

# Layer 1: native silicon oxide
layer_siox = Slab(material=siox, thickness=siox_thickness, interface=silicon_roughness)

# Layer 0: semi-infinite silicon substrate
layer_silicon = Slab(material=silicon, thickness=0.0000, interface=silicon_roughness)

sample_dipa = layer_silicon | layer_siox | layer_tiox | layer_water

## === Problem definition ===
## a model object consists of a sample and a probe,
## zed is the step size in Angstroms to be used for rendering the profile
## increase zed to speed up the calculation
zed = 1    

## step = True corresponds to a calculation of the reflectivity from an actual profile
## with microslabbed interfaces.  When step = False, the Nevot-Croce
## approximation is used to account for roughness.  This approximation speeds up
## the calculation tremendously, and is reasonably accuarate as long as the
## roughness is much less than the layer thickness
step = False

model_dipa = Experiment(sample=sample_dipa, probe=probe_dipa, dz=zed, step_interfaces = step)

problem = FitProblem([model_dipa])

problem.name = "tiox_dipa"
