from refl1d.names import Parameter, SLD, Slab, Experiment, FitProblem, load4

## === Probes/data files ===
probe_d2o = load4('ch061_d2o_ph7.refl', back_reflectivity=True, name='D2O')
probe_h2o = load4('ch060_h2o_ph7.refl', back_reflectivity=True, name='H2O')

# Probe parameters
# Backgrounds (one for each model)
probe_d2o.background.range(-1e-7, 1e-5)
probe_h2o.background.range(-1e-7, 1e-5)

# Intensity (common to all models)
probe_d2o.intensity.range(0.95,1.05)
probe_h2o.intensity = probe_d2o.intensity

# Sample broadening (common to all models)
probe_d2o.sample_broadening.range(-0.003, 0.02)
probe_h2o.sample_broadening = probe_d2o.sample_broadening

# Theta offset (common to all models)
probe_d2o.theta_offset.range(-0.02, 0.02)
probe_h2o.theta_offset = probe_d2o.theta_offset


## === Custom parameters ===
vf_tails = Parameter(name='volume fraction bilayer', value=0.9).range(0.0, 1.0)
vf_ihg = Parameter(name='volume fraction inner headgroups', value=0.4).range(0.0, 1.0)
vf_ohg = Parameter(name='volume fraction outer headgroups', value=0.4).range(0.0, 1.0)
l_hg = Parameter(name='headgroup thickness', value=10.0)
bilayer_roughness = Parameter(name='bilayer roughness', value=5.0).range(2, 15.0)
bilayer_thickness = Parameter(name='bilayer thickness', value=30.0).range(20, 40.0)
submembrane_thickness = Parameter(name = 'submembrane thickness', value=10.0).range(2, 50)
tiox_roughness = Parameter(name='tiox roughness', value=5.0).range(2, 15.0)
tiox_thickness = Parameter(name='tiox thickness', value=140.0).range(80, 200)
siox_thickness = Parameter(name='siox thickness', value=20.0).range(2.0, 30.0)
silicon_roughness = Parameter(name='si roughness', value=2.0).range(2, 15.0)

## === Samples ===

# Material definitions
d2o = SLD(name='d2o', rho=6.3000, irho=0.0000)
h2o = SLD(name='h2o', rho=-0.56, irho=0.0000)
hg = SLD(name='headgroups', rho=1.8131, irho=0.000)
tails = SLD(name='lipid tails', rho=-0.2145, irho=0.000)
tiox = SLD(name='tiox', rho=2.1630, irho=0.0000)
siox = SLD(name='siox', rho=4.1000, irho=0.0000)
silicon = SLD(name='silicon', rho=2.0690, irho=0.0000)

# Material SLD parameters
d2o.rho.range(5.3000, 6.36)
h2o.rho.range(-0.56, 0.6)
tiox.rho.range(1.2, 3.2)
siox.rho.range(2.8, 4.8)

# Sample stack definition

def make_sample(buffer, sub_thick=submembrane_thickness, bilayer_rough=bilayer_roughness):
    """_summary_

    Args:
        buffer: buffer material
        sub_thick: submembrane thickness parameter. Default submembrane_thickness
        bilayer_rough: bilayer roughness parameter. Default bilayer_roughness

    Returns:
        sample stack
    """
        
    # Layer 7: semi-infinite layer filled with buffer
    layer_water = Slab(material=buffer, thickness=0.0000, interface=0.0000)
    
    # Layer 6: outer headgroups. Typically about 50% buffer, 50% headgroup material
    layer_ohg = Slab(material=SLD(name='ohg + ' + buffer.name, rho=vf_ohg*hg.rho + (1 - vf_ohg)*buffer.rho, irho=0.0), thickness=l_hg, interface=bilayer_rough)
    
    # Layer 5: hydrocarbon tails from both leaflets. Typically > 90% hydrocarbons and < 10% buffer.
    layer_tails = Slab(material=SLD(name='tails + ' + buffer.name, rho=vf_tails*tails.rho + (1 - vf_tails)*buffer.rho, irho=0.0), thickness=bilayer_thickness, interface=bilayer_roughness)
    
    # Layer 4: inner headgroups. Typically about 50% buffer, 50% headgroup material
    layer_ihg = Slab(material=SLD(name='ihg + ' + buffer.name, rho=vf_ihg*hg.rho + (1 - vf_ihg)*buffer.rho, irho=0.0), thickness=l_hg, interface=bilayer_rough)
    
    # Layer 3: buffer-filled submembrane space
    layer_subwater = Slab(material=buffer, thickness=sub_thick, interface=bilayer_rough)
    
    # Layer 2: sputtered TiO2, nominal thickness 140 Ang
    layer_tiox = Slab(material=tiox, thickness=tiox_thickness, interface=tiox_roughness)
    
    # Layer 1: native silicon oxide
    layer_siox = Slab(material=siox, thickness=siox_thickness, interface=silicon_roughness)
    
    # Layer 0: semi-infinite silicon substrate
    layer_silicon = Slab(material=silicon, thickness=0.0000, interface=silicon_roughness)

    sample = layer_silicon | layer_siox | layer_tiox | layer_subwater | layer_ihg | layer_tails | layer_ohg | layer_water

    return sample

sample_d2o = make_sample(d2o, submembrane_thickness, bilayer_roughness)
sample_h2o = make_sample(h2o, submembrane_thickness, bilayer_roughness)

## === Problem definition ===
## a model object consists of a sample and a probe,
## zed is the step size in Angstroms to be used for rendering the profile
## increase zed to speed up the calculation
zed = 1    

## step = True corresponds to a calculation of the reflectivity from an actual profile
## with microslabbed interfaces.  When step = False, the Nevot-Croce
## approximation is used to account for roughness.  This approximation speeds up
## the calculation tremendously, and is reasonably accuarate as long as the
## roughness is much less than the layer thickness
step = False

model_d2o = Experiment(sample=sample_d2o, probe=probe_d2o, dz=zed, step_interfaces = step)
model_h2o = Experiment(sample=sample_h2o, probe=probe_h2o, dz=zed, step_interfaces = step)

problem = FitProblem([model_d2o, model_h2o])

problem.name = "tiox_dopc_d2o_h2o"
