from refl1d.names import *
from copy import copy
from numpy import *

# "probe" refers to the measurement we will analyze, essentially telling Refl1D to 
# load a 4-column dataset with Q_Z, Reflectivity, dReflectivity, and dQ_Z. We can also
# add in some sample broadening and a variety of other parameters.
probe = load4('MAFO_Saturated.refl', sample_broadening=0.026, back_reflectivity=False)
# It should be noted that sample broadening is an extremely important parameter for determining the
# shape of the critical edge, and it most often be determined through a bit of trial and error.  In
# theory, the broadening is the amount that the curvature of the film and/or substrate surface broadens
# the beam beyond the width one would expect based purely on the instrument resolution for a given
# slit configuration.

# This is telling Refl1D that we don't have a spin-flip data (or "cross sections" i.e. ".xs") The
# spin flip data is known as cross sections 1 and 2 to Refl1D.
probe.xs[1],probe.xs[2] = None, None
probe.xs[1],probe.xs[2] = None, None


#=========================================================================================================
# Define various parameters we are going to use later for better organization. 
#=========================================================================================================
# We will group these parameters by the material or layer we use them in. Generally "XX_rho" refers to the
# nuclear scattering length density (SLD) of material XX. Similarly, "XX_t" refers to layer thickness and
# "XX_sig" refers to interface roughness (both in Å). "XX_rhoM" will be magnetic SLD. Of course, these names are
# arbitrary and you can use what's best for you.

# Define Initial Beam Parameters
I0 = 1.0# normalization factor
th = 0.0 # theta offset in degrees
bg = 0.0

# MgAl2O4
MAO_rho = 5.377
MAO_sig = 5
# Here, MAO is our substrate, and we assume it's "semi-infinite" so that in continues down forever. Thus,
# we don't need to define a substrate layer thickness

# (Mg,Al,Fe)3O4
MAFO_rho = 5
MAFO_rhoM = 0.1
MAFO_t = 160
MAFO_sig = 5

# Remember, the above values are just our initial guesses - they don't have to be perfect. However, we
# recommend that you fix some values that you know, such as the nuclear SLD of the substrate.


#=========================================================================================================
# Define the actual materials which will make up our layers
#=========================================================================================================
MAO= SLD(name="MgAl2O4",rho=MAO_rho)
MAFO= SLD(name="(Mg,Al,Fe)3O4",rho=MAFO_rho)





#=========================================================================================================
# Build the sample for a variety of material layers
#=========================================================================================================
sample = (MAO(100,MAO_sig)|MAFO(MAFO_t,MAFO_sig, Magnetism(rhoM=MAFO_rhoM))|air)

# The sample is  the stack of MgAl2O4/(Mg,Al,Fe)3O4/air. Note that the for may is (LayerA(LayerA_thickness,LayerA_interface)|LayerB(LayerB_thickness,LayerB_interface)|air
# If the layer is magnetic, you can instead use (LayerA(LayerA_thickness,LayerA_interface,Magnetism(rhoM = XXXX, SomeOtherParameter = XXX))|LayerB...|air)
# Please note that there are many magnetic parameters, including dead_above, dead_below, interface_above, interface_below, etc...


#=========================================================================================================
# Constraints 
#=========================================================================================================
# Here I like to put in any constraint that will help limit the model fitting. For example, we know that the neutron beam must have the
# same intensity, background, and angle offset for all scattering cross sections. The ".mm" and ".pp" refers to the "minus-minus" and
# "plus-plus" cross sections. Of course, the spin flip would then be ".pm" and ".mp". probe.mm.theta_offset = probe.pp.theta_offset.
# We can also accomplish this be including a line which says "probe.shared_beam()". 
probe.mm.intensity = probe.pp.intensity
probe.mm.theta_offset = probe.pp.theta_offset
probe.mm.background = probe.pp.background

# If we wanted to enforce a constant interface roughness for every layer, we might include statements like:
# "sample[MAO].interface = sample[MAFO].interface"

# Here we initialize the starting values for the incoming neutron beam, background, and sample misalignment.
probe.pp.theta_offset.value = th
probe.pp.intensity.value = I0
probe.pp.background.value = bg




#=========================================================================================================
# Fit parameters 
#=========================================================================================================
# Now we will begin selecting which parameters ought to be fitted, and what range they should vary over. To do this
# we just say sample[XX][LayerXX].parameterXX.range(bottomRange,topRange). We could also say ".pm(XX)" or ".pmp(XX)" instead
# of ".range(XX,YY)" if we want to have the parameter vary by ±XX or ±XX%, respectively.

# Here we fit the beam intensity, in case our normalization is a bit off (often not necessary) and the theta offset, in case our
# alignment is a tiny bit off. One can also bit a constant background, but this is generally discouraged since we have explicitly
# measured and subtracted off the background.
probe.pp.theta_offset.range(-0.02,0.02)
probe.pp.intensity.range(0.98,1.02)
#    probe[0].pp.background.range(-2e-6,2e-6)


# Here we fit the thicknesses
sample[MAFO].thickness.range(60,180)

# Here we fit the interlayer roughnesses. It's important to note that the roughness parameter captures  long-range roughness
# contributions such as thickness variation and conformal roughness as well as chemical intermixing.
sample[MAO].interface.range(1,15)
sample[MAFO].interface.range(1,15)

# Here we fit the nuclear SLDs. Note that I am not fitting the MgAl2O4 SLD since I know the crystal is high quality so that I
# can input theoretical density and composition values into the SLD calculator at https://www.ncnr.nist.gov/resources/activation/
# So we don't fit things we know if it's not necessary to describe the data. For the MAFO, it's a film and I don't know the actual
# composition, so we fit the SLD.
# MAO.rho.range(2.0,6.0)
MAFO.rho.range(2.0,7.0)


# Here we fit the magnetic SLD of MAFO
sample[MAFO].magnetism.rhoM.range(0,2)

 

#=========================================================================================================
# Fitting Problem - technical definition 
#=========================================================================================================
zed = 0.5 # microslabbing bin size, in Å
alpha = 0.0 # integration factor - leave this at zero
step = False
# This "step" or "step_interfaces" setting is really important. Effectively, there are two ways to calculate the reflectivity. 
# One way treats the entire system as a series of microslabe ("step_interfaces = True") while the other uses something
# called the Nevot-Croce approximation ("step_interfaces = False"). The difference between the two is beyond the scope of these
# notes, but generally you should know that Nevot-Croce is much, much faster but fails in some very specific cases where
# the sample contains layers for which the roughness on one side is very different from the other AND the roughness is comparable
# to or larger than the thickness. Be careful here , and always use "step_interfaces=False" as a sanity check.

# Define a single model object for Refl1D to fit. We give it the sample structure (sample), the dataset to fit (probe), etc.
model = Experiment(sample=sample, probe=probe, dz=zed,dA=alpha,step_interfaces=step)

#Now tell Refl1D to actually fit the data:
problem = FitProblem(model)
problem.name = "MAO/MAFO"
