################################################################################
# 
# This script is part of the NIST Source Code Analysis Tool Assessment Test Suite
#
# This software was developed at the National Institute of Standards and
# Technology by employees of the Federal Government in the course of their
# official duties. Pursuant to title 17 Section 105 of the United States
# Code this software is not subject to copyright protection and is in the
# public domain. NIST assumes no responsibility whatsoever for its use by
# other parties, and makes no guarantees, expressed or implied, about its
# quality, reliability, or any other characteristic.
#
################################################################################
#
# This Windows powershell script is generic in design, and should work with any
# type of command-line source code analysis/build tool, such as compilers, 
# security analyzers and build utilities.  It is designed and makes assumptions
# on file locations based upon the directory structure of the NIST Source Code 
# Analyzer Tool Test Suite.  It will not work on other directory structures without 
# modification.
#
# Powershell is Microsoft's advanced batch scripting language that eases the 
# development of complex scripting by providing over 100 utilities as part of 
# its basic package of commands. 
#
# Powershell runs in the Microsoft Visual Studio command shell.
#
# You can download Microsoft Powershell 1.0 at:
#
# http://www.microsoft.com/windowsserver2003/technologies/management/powershell/default.mspx
# 
#
# To compile all the c tests using the Microsoft CL compiler, the command is:
# powershell run-tool.ps1 -t cl -o '/DDEBUG -DCLCOMPILER -Wall /EHsc' -l c
#
# To compile all the c tests using the Gnu gcc compiler, the command is:
# powershell run-tool.ps1 -t gcc -o '-DDEBUG -DCLCOMPILER -Wall -ldl' -l c
#
# To compile all the c++ tests using the Microsoft CL compiler, the command is:
# powershell run-tool.ps1 -t cl -o '-DDEBUG -DCLCOMPILER -Wall /EHsc' -l cpp 
#
# To compile all the c++ tests using the Gnu g++ compiler, the command is:
# powershell run-tool.ps1 -t g++ -o '-DDEBUG -DGNUCOMPILER -ldl -Wall' -l cpp 
#
#
# To compile all the java tests using the Sun Micosystems javac compiler, the command is:
# powershell run-tool.ps1 -t javac -o  -l java
#
# Although this powershell script was built as a QA tool to verify that NIST 
# source code anlysis tool test code will compile, it can be used to run other 
# tools besides compilers. 
#
# To run a tool that requires a compiler name and defines the DEBUG macro and uses gnu libraries for c++:
# powershell run-tool.ps1 -t my_analysis_tool -o 'g++ -ldl -DDEBUG -DGNUCOMPILER'  -l cpp

# To compile test that use nmake, ant or other build utily, one could 
# compile them  by including the -b true option when invoking this script as follows (for c code tests):
# powershell run-tool.ps1 -t make -o 'clean' -b true -l c
#
# NOTE: This script must be run out of the tool_tests directory, because it makes assumptions on the directory
# structure of the test suite
#
################################################################################
#
#initially set variables to "null" 
$tool=$tool_options=$directory=$language=$searchPath=$null

function usage () {
"Usage: powershell run-tool.ps1 -t tool_command -l language  [-o 'tool_options'] -d [directory_path] [-b]
Where: 
tool_command argument = gcc, or cl, or make, or ant, or a source scanner...etc... (this is a required field)
language =  c, cpp or java only  (this is a required field)
tool_options = any of the tool's option flags and arguments... e.g.  '-Wall -c -DDEBUG'  (enclosed in single quotes!!)
directory_path = starting directory path (optional field, default is current working directory)
-b = build flag (indicates the tool is a command-line build tool.. like make, nmake or ant)

Some examples:

(compile all the c tests using the gcc compiler)
powershell run-tool.ps1 -t gcc -o '-DDEBUG -DGNUCOMPILER' -l c
(compile all the c tests using the Microsoft CL compiler)
powershell run-tool.ps1 -t cl -o '-DDEBUG -DCL_COMPILER' -l c
(compile all the c++ tests using the gcc compiler**if installed on your Windows machine**)
powershell run-tool.ps1 -t g++ -o '-DDEBUG -DGNUCOMPILER' -l cpp 
(compile all the c++ tests using the Microsoft CL compiler)
powershell run-tool.ps1 -t cl -o '-DDEBUG -DCL_COMPILER' -l cpp 
(compile all the java tests using the Sun Micosystems javac compiler)
powershell run-tool.ps1 -t javac -o  -l java
(compile all the java tests for comment convention tests only)
powershell run-tool.ps1 -t javac -o  -l java -d comment_conventions
(run a code analysis tool that needs a compiler name as an argument)
powershell run-tool.ps1 -t my_analysis_tool -o 'gcc -DDEBUG -DGNUCOMPILER'  -l c
compile tests using make, nmake, ant or other build utily, include the -b flag (i.e. don't compile, run nmake)
powershell run-tool.ps1 -t make -o 'clean' -b -l c
"
exit
}

# if any arguments are present on command line

if ($args)
{

# parse the options into variables...

switch ($args)
{
'-t' {[void] $switch.movenext(); $tool= $switch.current; continue}
'-o' {[void] $switch.movenext(); $tool_options= $switch.current; continue}
'-d' {[void] $switch.movenext(); $directory= $switch.current; continue}
'-l' {[void] $switch.movenext(); $language= $switch.current; continue}
'-b' {$build_tool= $true; continue}
}
}

# otherwise print out usage message

else
{
usage
}

# debug
"tool=$tool tool_options=$tool_options directory=$directory 
language=$language build_tool=$build_tool"


# if minimal user input is supplied, start processing tests

if (-not ($tool) -and ($language))
{
usage
}

#set starting directory if none is provided
if (-not $directory) 
{
$directory=get-location
}

#either way, make sure we have the absolute starting directory path
cd $directory
$directory=get-location

# if user is running "make" or "nmake" or "ant", 
# don't include file names in the execution command
if ($build_tool -eq $true)
{
$files = ""
}
else
{
$files = "*.$language"
"files are $files"
}

#split the tool options up, otherwise javac will hiccup on them
if ($tool_options) 
{
$tool_opt_array=$tool_options.split()
}

$searchPath = `dir -name "$language" -recurse -path "."`

#if no test directories found, assume we are running a single test
if (-not $searchPath)
{
& $tool $tool_opt_array $files 
exit
}
else
# assume that we must traverse some part of the test tree to find tests
{
foreach ($startDir in $searchPath)
{
cd $startDir
#debug
$subDirectory=get-location 
$subDirectory
foreach ($testDirectory in Get-ChildItem)
{
cd $testDirectory
get-location
& $tool $tool_opt_array $files
cd ..
}
cd $directory
}
}




