/*
 * $Header: /cvsroot/mvnforum/mvnforum/src/com/mvnforum/common/PostChecker.java,v 1.5 2007/10/09 11:09:16 lexuanttkhtn Exp $
 * $Author: lexuanttkhtn $
 * $Revision: 1.5 $
 * $Date: 2007/10/09 11:09:16 $
 *
 * ====================================================================
 *
 * Copyright (C) 2002-2007 by MyVietnam.net
 *
 * All copyright notices regarding mvnForum MUST remain
 * intact in the scripts and in the outputted HTML.
 * The "powered by" text/logo with a link back to
 * http://www.mvnForum.com and http://www.MyVietnam.net in
 * the footer of the pages MUST remain visible when the pages
 * are viewed on the internet or intranet.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Support can be obtained from support forums at:
 * http://www.mvnForum.com/mvnforum/index
 *
 * Correspondence and Marketing Questions can be sent to:
 * info at MyVietnam net
 *
 * @author: Phong Ta Quoc
 */
package com.mvnforum.common;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Iterator;

import net.myvietnam.mvncore.exception.BadInputException;
import net.myvietnam.mvncore.exception.DatabaseException;
import net.myvietnam.mvncore.util.DateUtil;

import com.mvnforum.MVNForumConfig;
import com.mvnforum.MVNForumResourceBundle;
import com.mvnforum.auth.*;
import com.mvnforum.db.DAOFactory;
import com.mvnforum.db.PostBean;

public class PostChecker {

    public static final int ROOT_POST = 0;

    public static void checkEditPost(OnlineUser user, PostBean post) throws AuthenticationException, BadInputException {

        MVNForumPermission permission = user.getPermission();
        int forumId = post.getForumID();

        // check constraint
        if (permission.canEditPost(forumId)) {
            // have permission, just do nothing, that is dont check the max day contraint
        } else if (isAuthorOfPost(user, post)) {
            // make sure user have permission to edit his own post
            permission.ensureCanEditOwnPost(forumId);

            // post should not be too old, gt 7 days by default
            checkTooOld(user, post);

            // post should not be disabled
            checkDisable(user, post);
        } else {//not an author, so this user must have Edit Permission
            permission.ensureCanEditPost(forumId);// this method ALWAYS throws AuthenticationException
        }

    }

    public static boolean isAuthorOfPost(OnlineUser user, PostBean post) {
        return (user.getMemberID() == post.getMemberID() && user.isMember());
    }

    public static boolean isDisabled(PostBean post) {
        return (post.getPostStatus() == PostBean.POST_STATUS_DISABLED);
    }

    public static void checkTooOld(OnlineUser user, PostBean post) throws BadInputException {
        Timestamp now = DateUtil.getCurrentGMTTimestamp();
        int maxDays = MVNForumConfig.getMaxEditDays();
        if ( (now.getTime() - post.getPostCreationDate().getTime()) > (DateUtil.DAY * maxDays) ) {
            /** @todo choose a better Exception here */
            String localizedMessage = MVNForumResourceBundle.getString(user.getLocale(), "mvncore.exception.BadInputException.cannot_edit.post_is_too_old", new Object[] {new Integer(maxDays)});
            throw new BadInputException(localizedMessage);
            //throw new BadInputException("You cannot edit a post which is older than " + maxDays + " days.");
        }
    }

    public static void checkDisable(OnlineUser user, PostBean post) throws BadInputException {
        if (isDisabled(post)) {
            String localizedMessage = MVNForumResourceBundle.getString(user.getLocale(), "mvncore.exception.BadInputException.cannot_edit_your_post.which_is_disabled");
            throw new BadInputException(localizedMessage);
        }
    }

    public static void checkNoReply(OnlineUser user, PostBean post) throws IllegalArgumentException, DatabaseException, BadInputException {
        Collection  posts = DAOFactory.getPostDAO().getEnablePosts_inThread_limit(post.getThreadID(), 0, 10000);
        boolean foundReply = false;
        for (Iterator ite = posts.iterator(); ite.hasNext(); ) {
            PostBean tPost = (PostBean) ite.next();
            if (tPost.getParentPostID() == post.getPostID()) {
                foundReply = true;
                break;
            }
        }

        if (foundReply) {
            String localizedMessage = MVNForumResourceBundle.getString(user.getLocale(), "mvncore.exception.BadInputException.cannot_delete_post.post_has_reply");
            throw new BadInputException(localizedMessage);
            //throw new BadInputException("Cannot delete a post that has reply!");
        }
    }

    public static void checkRootOfThread(OnlineUser user, PostBean post) throws BadInputException {
        if (post.getParentPostID() == ROOT_POST) {
            String localizedMessage = MVNForumResourceBundle.getString(user.getLocale(), "mvncore.exception.BadInputException.cannot_delete_root_post");
            throw new BadInputException(localizedMessage);
            //throw new BadInputException("Cannot delete a root post. Use delete thread instead.");
        }

    }

}
