"""
Graph Plot

"""
__author__ = """Brian Cloteaux (brian.cloteaux@nist.gov)"""
#  Mathematical and Computational Sciences Division
#  National Institute of Standards and Technology,
#  Gaithersburg, MD USA
# 
#  This software was developed at the National Institute of Standards and
#  Technology (NIST) by employees of the Federal Government in the course
#  of their official duties. Pursuant to title 17 Section 105 of the
#  United States Code, this software is not subject to copyright protection
#  and is in the public domain. NIST assumes no responsibility whatsoever for
#  its use by other parties, and makes no guarantees, expressed or implied,
#  about its quality, reliability, or any other characteristic.

import Tix
import networkx as nx

import NMTKModel

import matplotlib
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg, NavigationToolbar2TkAgg
from matplotlib.figure import Figure
import matplotlib.pyplot as plt

class GraphPlot(Tix.Frame): 
    def __init__(self, model, parent=None):
        Tix.Frame.__init__(self, parent, width=10, height=10)
        self.pack(fill=Tix.BOTH, expand=Tix.YES)
        self.parent = parent
        self.graphplot = None
        self.model = model
        self.v = Tix.IntVar()
        self.v.set(1)
        rbf = Tix.Frame(self)
        Tix.Radiobutton(rbf,text="Properties",variable=self.v,value=1, \
                command=self.updateGraph).pack(anchor=Tix.W,side=Tix.LEFT)
        Tix.Radiobutton(rbf,text="Spectrum",variable=self.v,value=2, \
                command=self.updateGraph).pack(anchor=Tix.W,side=Tix.LEFT)
        #Tix.Radiobutton(rbf,text="Structure",variable=self.v,value=3, \
        #        command=self.updateGraph).pack(anchor=Tix.W,side=Tix.LEFT)
        rbf.pack(side=Tix.BOTTOM)
        self.updateGraph()

    def updateGraph(self, model=None):
        if model != None:
            self.model = model
        graph = self.model.getGraph()
        if self.graphplot != None:
            self.graphplot.pack_forget()
        if self.v.get()==2:
            self.graphplot = GraphSpectrumPlot(graph,self) 
        elif self.v.get()==3:
            self.graphplot = GraphStructurePlot(graph,self) 
        else:
            self.graphplot = GraphProperties(graph,self) 
        self.graphplot.pack(side=Tix.TOP)

class GraphProperties(Tix.Frame):
    def __init__(self, graph, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        info = nx.classes.function.info(graph) + "\n"
        info = info + "s-metric: " + str(nx.s_metric(graph)) + "\n"
        if  not graph.is_directed():
            if nx.is_connected(graph):
                info = info + "Diameter: " + str(nx.diameter(graph)) + "\n"
            else:
                info = info + "Diameter: inf\n"
            info = info + "Clustering coefficient: " + \
                    str(nx.average_clustering(graph)) + "\n"
        pl = Tix.Label(self, text=info);
        pl.pack(anchor=Tix.W)

class GraphStructurePlot(Tix.Frame):
    def __init__(self, graph, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        self.fig = Figure(figsize=(5,4), dpi=100)
        self.axis = self.fig.add_subplot(111)
        nx.draw(graph, ax=self.axis)

        self.canvas = FigureCanvasTkAgg(self.fig,master=self)
        self.canvas.show()
        self.canvas.get_tk_widget().pack(side=Tix.TOP, fill=Tix.BOTH,\
                expand=1)
        self.toolbar = NavigationToolbar2TkAgg( self.canvas, self )
        self.toolbar.update()
        self.canvas._tkcanvas.pack(side=Tix.TOP, fill=Tix.BOTH, expand=1)

class GraphSpectrumPlot(Tix.Frame):
    def __init__(self, graph, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        self.fig = Figure(figsize=(5,4), dpi=100)
        a = self.fig.add_subplot(111, title='Graph Spectrum',\
                xlabel='Number', ylabel='Eigenvalues')
        t = nx.linalg.spectrum.adjacency_spectrum(graph)
        a.plot(t,'ro')

        self.canvas = FigureCanvasTkAgg(self.fig,master=self)
        self.canvas.show()
        self.canvas.get_tk_widget().pack(side=Tix.TOP, fill=Tix.BOTH,\
                expand=1)
        self.toolbar = NavigationToolbar2TkAgg( self.canvas, self )
        self.toolbar.update()
        self.canvas._tkcanvas.pack(side=Tix.TOP, fill=Tix.BOTH, expand=1)
