"""
Affiliation Graph Generation
"""
__author__ = 'Brian Cloteaux (brian.cloteaux@nist.gov)'

# Mathematical and Computational Sciences Division
# National Institute of Standards and Technology,
# Gaithersburg, MD USA
#
# This software was developed at the National Institute of Standards and
# Technology (NIST) by employees of the Federal Government in the course
# of their official duties. Pursuant to title 17 Section 105 of the
# United States Code, this software is not subject to copyright protection
# and is in the public domain. NIST assumes no responsibility whatsoever for
# its use by other parties, and makes no guarantees, expressed or implied,
# about its quality, reliability, or any other characteristic.

import networkx as nx

## Returns a two-mode graph whose projection is a given graph. 
#  
#  The algorithm is based on:
#  J.L. Guillaume & M. Latapy. <a
#  href="http:dx.doi.org/10.1016/j.ipl.2004.03.007">Bipartite structure of all
#  complex networks</a>,  Information Processing Letters, 90(2004), pg. 215-221.
#  @param graph Graph to extract affiliations from
#  @return Two-mode (or affiliation) graph 
def guillaume_latapy(graph):
    affil = nx.Graph(name='Affiliation Graph (Guillaume-Latapy)')
    affil.add_nodes_from(graph)
    affil.node_type={}
    for n in affil.nodes():
        affil.node_type[n] = "Bottom" 

    cliq=list(nx.find_cliques(graph))
    top = set([]) 
    for e in graph.edges():
        max = []
        n1, n2 = e
        for c in cliq:
            if n1 in c and n2 in c:
                if len(c) > len(max):
                    max = c
        top.add( frozenset(max) )

    a = -1
    for t in top:
        affil.add_node(a)
        affil.node_type[a] = "Top" 
        for n in t:
            affil.add_edge(n,a)
        a -= 1
    return affil
