#!/usr/bin/env python
"""
NMTK

"""
__author__ = """Brian Cloteaux (brian.cloteaux@nist.gov)"""
#  Mathematical and Computational Sciences Division
#  National Institute of Standards and Technology,
#  Gaithersburg, MD USA
# 
#  This software was developed at the National Institute of Standards and
#  Technology (NIST) by employees of the Federal Government in the course
#  of their official duties. Pursuant to title 17 Section 105 of the
#  United States Code, this software is not subject to copyright protection
#  and is in the public domain. NIST assumes no responsibility whatsoever for
#  its use by other parties, and makes no guarantees, expressed or implied,
#  about its quality, reliability, or any other characteristic.

import Tix
import tkMessageBox as tkm

import NMTKFrame

class ExceptionHandler:
    def __init__(self, function):
        self.function = function

    def __call__(self, *args):
        try:
            return self.function(*args)
        except Exception, e:
            st = "Error: %s" % (e)
            tkm.showerror("NMTK",st)

class TabContainer:
    def __init__(self, root):
        self.win_num = 1
        self.tabs = Tix.NoteBook(root, ipadx=6, ipady=6, bg='gray')
        self.tabs.pack(expand=1, fill=Tix.BOTH, padx=5, pady=5, side=Tix.TOP)

    def getCurrNMTKWidget(self):
        currname = self.tabs.raised()
        currpage = self.tabs.page(currname)
        currwidgets = currpage.pack_slaves()
        return currwidgets[0]

    def createTab(self):
        tabName = 'win_'+str(self.win_num)
        tabLabel = 'Sequence ' + str(self.win_num)
        self.win_num += 1
        tab = self.tabs.add(tabName, label=tabLabel)
        nmtk = NMTKFrame.NMTKFrame(tab)
        nmtk.pack(side=Tix.LEFT, padx=2, pady=2, fill=Tix.BOTH, expand=1)
        self.tabs.raise_page(tabName)
        return tab

    def renameCurrTab(name):
        currname = self.tabs.raised()
        currpage = self.tabs.page(currname)
        self.tabstk.call(tabs._w,'pageconfigure','tab','-label',name)

def createRootWindow(root):
    root.title('Network Model Toolkit')
    #root.iconbitmap('NMTKicon.xbm')
    root.minsize(300, 200)
    menubar = Tix.Menu(root) 
    tabs = TabContainer(root)

    filemenu = Tix.Menu(menubar)
    filemenu.add_command(label="New Tab", command=lambda: tabs.createTab()) 
    
    clonesubmenu = Tix.Menu(filemenu,tearoff=0)
    clonesubmenu.add_command(label="Sequence", \
            command=lambda: clone(tabs,"sequence"))
    clonesubmenu.add_command(label="Graph", \
            command=lambda: clone(tabs,"graph"))
    clonesubmenu.add_command(label="Affiliation", \
            command=lambda: clone(tabs,"affil"))
    filemenu.add_cascade(label="Clone", menu=clonesubmenu)
    loadsubmenu = Tix.Menu(filemenu,tearoff=0)
    loadsubmenu.add_command(label="Sequence", \
            command=lambda: readSequence(tabs))
    loadsubmenu.add_command(label="Graph", command=lambda: readGraph(tabs))
    loadsubmenu.add_command(label="Affiliation", \
            command=lambda: readAffiliation(tabs))
    filemenu.add_cascade(label="Load", menu=loadsubmenu)
    savesubmenu = Tix.Menu(filemenu,tearoff=0)
    savesubmenu.add_command(label="Sequence", \
            command=lambda: writeSequence(tabs))
    savesubmenu.add_command(label="Graph", \
            command=lambda: writeGraph(tabs,"graph"))
    savesubmenu.add_command(label="Affiliation", \
            command=lambda: writeGraph(tabs,"affil"))
    filemenu.add_cascade(label="Save", menu=savesubmenu)
    filemenu.add_separator()
    filemenu.add_command(label="Quit", command=quit)
    menubar.add_cascade(label="File", menu=filemenu) 

    seqmenu = Tix.Menu(menubar)
    seqgensubmenu = Tix.Menu(filemenu,tearoff=0)
    seqgensubmenu.add_command(label="Power law", \
            command=lambda: generateSequence(tabs,"power"))
    seqgensubmenu.add_command(label="Pareto", \
            command=lambda: generateSequence(tabs,"pareto"))
    seqgensubmenu.add_command(label="Uniform", \
            command=lambda: generateSequence(tabs,"uniform"))
    seqmenu.add_cascade(label="Generate", menu=seqgensubmenu)
    seqmenu.add_command(label="Nearest Graphical", \
            command=lambda: nearestGraphical(tabs))
    seqmenu.add_separator()
    seqtographsubmenu = Tix.Menu(filemenu,tearoff=0)
    seqtographsubmenu.add_command(label="Markov", \
            command=lambda: createGraphFromSequence(tabs,"markov"))
    seqtographsubmenu.add_command(label="Connected Markov", \
            command=lambda: createGraphFromSequence(tabs,"con_markov"))
    seqtographsubmenu.add_command(label="Blitzstein-Diaconis", \
            command=lambda: createGraphFromSequence(tabs,"blitzstein"))
    seqmenu.add_cascade(label="Generate Graph", menu=seqtographsubmenu)
    menubar.add_cascade(label="Sequence", menu=seqmenu) 

    graphmenu = Tix.Menu(menubar)
    graphgensubmenu = Tix.Menu(filemenu,tearoff=0)
    graphgensubmenu.add_command(label="Erdos-Renyi", \
            command=lambda: generateGraph(tabs,"erdos"))
    graphgensubmenu.add_command(label="Newman-Watts-Strogatz", \
            command=lambda: generateGraph(tabs,"strogatz"))
    graphgensubmenu.add_command(label="Barabasi-Albert", \
            command=lambda: generateGraph(tabs,"barabasi"))
    #graphgensubmenu.add_command(label="Powerlaw Cluster", command=callback)
    graphmenu.add_cascade(label="Generate", menu=graphgensubmenu)
    graphmenu.add_separator()
    graphtoaffilsubmenu = Tix.Menu(filemenu,tearoff=0)
    graphtoaffilsubmenu.add_command(label="Guillaume-Latapy", \
            command=lambda: generateAffil(tabs,"gl"))
    graphtoaffilsubmenu.add_command(label="Maximal Clique", \
            command=lambda: generateAffil(tabs,"max_clique"))
    graphmenu.add_cascade(label="Generate Affiliation", \
            menu=graphtoaffilsubmenu)
    menubar.add_cascade(label="Graph", menu=graphmenu) 

    affilmenu = Tix.Menu(menubar)
    affilmenu.add_command(label="Flip Sets", command=lambda: flipAffil(tabs))
    menubar.add_cascade(label="Affiliation", menu=affilmenu) 

    helpmenu = Tix.Menu(menubar)
    #helpmenu.add_command(label="Documentation", command=doc)
    helpmenu.add_command(label="About", command=infoBox)
    menubar.add_cascade(label="Help", menu=helpmenu) 

    root.config(menu=menubar) 
    tabs.createTab()

@ExceptionHandler
def readSequence(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.readSequence()

@ExceptionHandler
def writeSequence(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.writeSequence()

@ExceptionHandler
def generateSequence(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.generateSequence(type)

@ExceptionHandler
def readGraph(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.readGraph()

@ExceptionHandler
def writeGraph(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.writeGraph(type)

@ExceptionHandler
def nearestGraphical(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.nearestGraphical()

@ExceptionHandler
def clone(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    newtab = tabs.createTab()
    newnmtk = newtab.pack_slaves()[0]
    newmodel = nmtk.getModelCopy()
    if type == 'graph':
        newmodel.updateAffiliation()
    if type == 'sequence':
        newmodel.updateGraph()
        newmodel.updateAffiliation()
    newnmtk.cloneNMTK(newmodel)

@ExceptionHandler
def generateGraph(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.generateGraph(type)

@ExceptionHandler
def createGraphFromSequence(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.createGraphFromSequence(type)

@ExceptionHandler
def readAffiliation(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.readAffiliation()

@ExceptionHandler
def generateAffil(tabs,type):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.generateAffil(type)

@ExceptionHandler
def flipAffil(tabs):
    nmtk = tabs.getCurrNMTKWidget()
    nmtk.flipAffil()

def infoBox():
    str = "Network Model Toolkit\n\n"
    str = str + "This software was developed at the National Institute "
    str = str + "of Standards and Technology (NIST) by employees of "
    str = str + "the Federal Government in the course of their official "
    str = str + "duties.  Pursuant to title 17 Section 105 of the "
    str = str + "United States Code, this software is not subject to "
    str = str + "copyright protection and is in the public domain.  "
    str = str + "NIST assumes no responsibility whatsoever for its use by "
    str = str + "other parties, and makes no guarantees, expressed or "
    str = str + "implied, about its quality, reliability, or any other "
    str = str + "characteristic."
    tkm.showinfo("NMTK",str)

if __name__ == '__main__':
    root = Tix.Tk()
    createRootWindow(root)
    root.mainloop()
