"""
Sequential Graph Generation
"""
__author__ = 'Brian Cloteaux (brian.cloteaux@nist.gov)'

# Mathematical and Computational Sciences Division
# National Institute of Standards and Technology,
# Gaithersburg, MD USA
#
# This software was developed at the National Institute of Standards and
# Technology (NIST) by employees of the Federal Government in the course
# of their official duties. Pursuant to title 17 Section 105 of the
# United States Code, this software is not subject to copyright protection
# and is in the public domain. NIST assumes no responsibility whatsoever for
# its use by other parties, and makes no guarantees, expressed or implied,
# about its quality, reliability, or any other characteristic.

import networkx as nx
import numpy as np
import copy
import random
import degree_sequence

def blitzstein_diaconis(sequence):
    sequence = copy.deepcopy(sequence)
    degs = list(sequence.sequence)
    degs.sort(reverse=True)
    graph = nx.Graph(name='Random Graph (Blitzstein-Diaconis)')
    graph.add_nodes_from(range(sequence.get_length()))

    while  sequence.get_largest_val()>0:
        # Choose the least i with  d_i a minimal positive entry
        minVal = len(degs)
        indx = -1
        for i in reversed(xrange(len(degs))):
            if degs[i]>0 and degs[i] < minVal:
                minVal = degs[i]
                indx = i

        while degs[indx] > 0:
            candList = []

            # Compute candidate list
            for j in xrange(len(degs)):
                if degs[j]>0 and j!=indx and not graph.has_edge(indx,j):
                    val1 = degs[indx]
                    val2 = degs[j]
                    sequence.decrement( val1 )
                    sequence.decrement( val2 )
                    if sequence.is_graphical():
                        candList.append(j)
                    sequence.increment( val1-1 )
                    sequence.increment( val2-1 )

            # Pick candidate edge with probability proportional to its degree
            weights = []
            indices = [] 
            counter = 0
            for val in candList:
                indices.append(val)
                weights.append(degs[val])
                counter += 1
            s = sum(weights)
            weights = np.cumsum(weights)
            rv =  random.randint(1,s)
            x = np.searchsorted(weights,rv) 
            n = indices[x]

            # Add edge {i,j} to E and update d
            graph.add_edge(indx,n)
            sequence.decrement(degs[indx])
            sequence.decrement(degs[n])
            degs[indx] -= 1
            degs[n] -= 1

    return graph

def connected_blitzstein_diaconis(sequence):
    pass

def markov(sequence):
    graph = None
    if sequence.is_graphical():
        graph = nx.havel_hakimi_graph(sequence.sequence)
        graph.name = 'Random Graph (Markov)'
        for i in xrange(3*nx.number_of_edges(graph)):
            nx.double_edge_swap(graph)
    return graph

def connected_markov(sequence):
    graph = None
    if sequence.is_connected_graphical():
        graph = nx.havel_hakimi_graph(sequence.sequence)
        graph = component_merge(graph)
        graph.name = 'Random Connected Graph (Markov)'
        for i in xrange(3*nx.number_of_edges(graph)):
            nx.connected_double_edge_swap(graph)
    return graph


def component_merge(graph):
    gr = None
    if nx.number_of_edges(graph) >= nx.number_of_nodes(graph)-1: # connectable
        comps = nx.connected_component_subgraphs(graph)
        cg = 0
        while nx.number_of_edges(graph) == nx.number_of_nodes(graph)-1: # tree
            cg += 1
        gr = comps[cg]
        for i in xrange(len(comps)):
            if i != cg: 
                cycles = nx.cycle_basis(gr)
                cycle = cycles[0]
                u = cycle[0]
                v = cycle[1]
                ed = (comps[i].edges())[0]
                x = ed[0]
                y = ed[1]
                gr = nx.union(gr,comps[i])
                gr.remove_edge(u,v)
                gr.remove_edge(x,y)
                gr.add_edge(u,x)
                gr.add_edge(v,y)
    return gr



