"""
Degree Sequence

"""
__author__ = """Brian Cloteaux (brian.cloteaux@nist.gov)"""
#  Mathematical and Computational Sciences Division
#  National Institute of Standards and Technology,
#  Gaithersburg, MD USA
# 
#  This software was developed at the National Institute of Standards and
#  Technology (NIST) by employees of the Federal Government in the course
#  of their official duties. Pursuant to title 17 Section 105 of the
#  United States Code, this software is not subject to copyright protection
#  and is in the public domain. NIST assumes no responsibility whatsoever for
#  its use by other parties, and makes no guarantees, expressed or implied,
#  about its quality, reliability, or any other characteristic.

import bisect
import math

## A degree sequence \f$d\f$ is a sequence of positive integers
#  \f$d=(d_1, d_2, ..., d_n)\f$ such that \f$d_1 \geq d_2 \geq ... \geq
#  d_n\f$.  
class degree_sequence:
    def __init__(self, vals=[]):
        self.sequence = list(vals)
        self.sequence.sort()
        self.deg_sum = sum(self.sequence)

    def __str__(self):
        str = '';
        for i in xrange(self.get_length()-1,-1,-1):
                str = str + '%s ' % self.sequence[i]
        return str

    ## Returns the length of the sequence
    #  @return Number of values in sequence
    def get_length(self):
        return len(self.sequence)

    ## Returns the sum of the sequence
    #  @return Sum of all the values in the sequence
    def get_degree_sum(self):
        return self.deg_sum

    ## Returns the number of times a given value appears in the sequence.
    #  In other words, for the value \f$v\f$, the routine returns
    #  \f$| \{ i | d_i = v\}|\f$.
    #  @param val Value to count in sequence
    #  @return Number of times the value appears
    def get_num_degs(self,val):
        if self.in_sequence(val):
            ri = bisect.bisect_right(self.sequence,val)
            li = bisect.bisect_left(self.sequence,val)
            return ri-li 
        return 0

    ## Returns the smallest value in the sequence.
    #  @return The smallest value in the sequence
    def get_smallest_val(self):
        return self.sequence[0]

    ## Returns the largest value in the sequence.
    #  @return The largest value in the sequence
    def get_largest_val(self):
        return self.sequence[-1]

    ## Returns the next smaller value than a given value in the sequence.
    #  @param val Value to find next smaller against
    #  @return The next smaller value in the sequence
    def get_next_smaller(self,val):
        idx = bisect.bisect_left(self.sequence,val)
        if idx > 0:
            return self.sequence[idx-1]
        return self.sequence[0]

    ## Adds the given values to the sequence.
    #  @param val Value to add to sequence
    #  @param num Number of values to add (default is 1)
    def add_deg(self,val,num=1):
        for i in range(num):
            bisect.insort(self.sequence, val)
        self.deg_sum = self.deg_sum + (val * num)

    ## Checks if a value is in the sequence.
    #  @param val Value to check
    #  @return True if value is in sequence, False otherwise
    def in_sequence(self, val):
        idx = bisect.bisect_left(self.sequence, val)
        if idx != len(self.sequence) and self.sequence[idx]==val:
            return True
        return False
        
    ## For a value \f$v\f$ in the sequence, it is incremented by one.
    #  For the sequence \f$d = (d_1, d_2, ..., d_i, ...,d_n)\f$ where
    #  \f$ d_i = v\f$, we get the new sequence
    #  \f$d' = (d_1, d_2, ..., d_i+1, ...,d_n)\f$.
    #  @param v Value to increment in the sequence
    def increment(self, v):
        idx = bisect.bisect_right(self.sequence, v)
        if self.sequence[idx-1] == v:
            self.sequence[idx-1] = v+1
            self.deg_sum = self.deg_sum+1

    ## For a value \f$v\f$ in the sequence, it is decremented by one.
    #  For the sequence \f$d = (d_1, d_2, ..., d_i, ...,d_n)\f$ where
    #  \f$ d_i = v\f$, we get the new sequence
    #  \f$d' = (d_1, d_2, ..., d_i-1, ...,d_n)\f$.
    #  @param v Value to decrement in the sequence
    def decrement(self, v):
        idx = bisect.bisect_left(self.sequence, v)
        if idx != len(self.sequence) and self.sequence[idx] == v:
            self.sequence[idx] = v-1
            self.deg_sum = self.deg_sum-1

    ## For the values \f$v_1\f$ and \f$v_2\f$ in sequence,
    #  if \f$v_1 > v_2+1\f$ then a new sequence is created with the
    #  values \f$v_1 -1\f$ and \f$v_2+1\f$ and the rest of the values
    #  remaining the same. 
    #  @param v1 First value
    #  @param v2 Second value
    def unit_transform(self, v1, v2):
        if self.in_sequence(v1) and self.in_sequence(v2) and v1>v2+1:
            self.decrement(v1)
            self.increment(v2)

    ## Tests if the degree sequence majorizes a given sequence.
    #  A sequence p majorizes q (denoted as \f$p \succeq q\f$) if
    #  \f$\forall 1 \leq k \leq n\f$ then
    #  \f$\sum_{i=1}^{k} p_i \geq \sum_{i=1}^{k} q_i\f$ and
    #  \f$p_n = q_n\f$.
    #  @return True if sequence majorizes the given sequence, False otherwise.
    def majorizes(self, seq):
        if (seq.get_length() != self.get_length()):
            return False
        sum = 0
        sum_seq = 0
        for i in xrange(self.get_length()):
            sum = sum + self.sequence[self.get_length()-i-1]
            sum_seq = sum_seq + seq.sequence[seq.get_length()-i-1]
            if (sum < sum_seq):
                return False
        if (sum != sum_seq):
            return False
        return True

    ## Checks if the sequence is graphical.
    #
    #  The implementation uses a theorem that states a sequence
    #  \f$d\f$ is graphical if and only if the sum of the sequence is even
    #  and for all strong indices \f$k\f$ in the sequence the following
    #  inequality holds
    #  \f$\sum_{i=1}^{k} d_i + \left( k \cdot \sum_{j=0}^{k-1} n_j - 
    #  \sum_{j=0}^{k-1} j\cdot n_j \right) \leq k \cdot (n-1) \f$.
    #  A strong index \f$k\f$ is any index where \f$d_k \geq k\f$
    #  and the value \f$n_j\f$ is the number of occurrences of \f$j\f$ in
    #  \f$d\f$.
    #  This formulation comes from Theorem 3 in
    #  I.E. Zverovich & V.E. Zverovich.  <a
    #  href="http:dx.doi.org/10.1016/0012-365X(92)90152-6">Contributions to
    #  the theory of graphic sequences</a>, Discrete Mathematics, 105(1992),
    #  pp. 292-303.
    #  @returns True if graphical, false otherwise.
    #
    def is_graphical(self):
        if self.get_degree_sum() == 0:
            return True
        if self.get_degree_sum() % 2 == 1:
            return False
        n = self.get_length()
        k = 1
        sum_deg = 0 
        sum_nj = 0 
        sum_jnj = 0 
        strong_index = True 
        while (strong_index): 
            sum_deg = sum_deg + self.sequence[n-k]
            nk1 = self.get_num_degs(k-1)
            sum_nj = sum_nj + nk1
            sum_jnj = sum_jnj + (k-1)*nk1
            lhs = sum_deg + (k*sum_nj - sum_jnj)
            rhs = k * (n-1)
            if lhs > rhs:
                return False
            k = k+1
            strong_index = (self.sequence[n-k] >= k)
        return True

    ## Checks if the sequence is graphical and is connectable.
    #
    #  @returns True if graphical and connectable, false otherwise.
    def is_connected_graphical(self):
        if self.get_degree_sum() > 2*self.get_length()-2 and \
                self.is_graphical():
            return True
        return False

    ## Modifies the sequence to the closest graphical sequence in a
    #  \f$L_1\f$ measure.
    #  From B. Cloteaux, Graphical approximation of degree sequences.
    def closest_graphical(self):
        if self.get_degree_sum() % 2 != 0:
            self.increment(self.get_largest_val()) 
        nc = self.get_length() * (self.get_length()-1)
        if self.get_degree_sum() > nc:
            n = int(math.ceil((1 + math.sqrt(1+4*self.get_degree_sum()))/2))
            self.add_deg(0,n-nc)
        while not self.is_graphical():
            self.unit_transform(self.get_largest_val(),
                    self.get_smallest_val())
