import Tix
import math
import degree_sequence

import NMTKModel

import matplotlib
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg, NavigationToolbar2TkAgg
from matplotlib.figure import Figure
import matplotlib.pyplot as plt

class SequencePlot(Tix.Frame): 
    def __init__(self, model, parent=None):
        Tix.Frame.__init__(self, parent, width=10, height=10)
        self.pack(fill=Tix.BOTH, expand=Tix.YES)
        self.parent = parent
        self.sequenceplot = None
        self.model = model
        self.v = Tix.IntVar()
        self.v.set(1)
        rbf = Tix.Frame(self)
        Tix.Radiobutton(rbf,text="Properties",variable=self.v,value=1, \
                command=self.updateSequence).pack(anchor=Tix.W,side=Tix.LEFT)
        Tix.Radiobutton(rbf,text="Plot",variable=self.v,value=2, \
                command=self.updateSequence).pack(anchor=Tix.W,side=Tix.LEFT)
        rbf.pack(side=Tix.BOTTOM)
        self.updateSequence()

    def updateSequence(self, model=None):
        if model != None:
            self.model = model
        sequence = self.model.getSequence()
        if self.sequenceplot != None:
            self.sequenceplot.pack_forget()
        if self.v.get()==2:
            self.sequenceplot = SequenceDegreePlot(sequence,self) 
        else:
            self.sequenceplot = SequenceProperties(sequence,self) 
        self.sequenceplot.pack(side=Tix.TOP)

class SequenceProperties(Tix.Frame):
    def __init__(self, sequence, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        info = "Number nodes: " + str(sequence.get_length()) + "\n"
        info = info + "Degree sum: " + str(sequence.get_degree_sum()) + "\n"
        avgDeg = float(sequence.get_degree_sum()) / float(sequence.get_length())
        info = info + "Average degree: %.3f" % avgDeg + "\n"
        info = info + "Minimum degree: " + str(sequence.get_smallest_val()) + "\n"
        info = info + "Maximum degree: " + str(sequence.get_largest_val()) + "\n"
        alpha, beta = getACLModel(sequence)
        info = info + "ACL Alpha: %.3f" % alpha + "\n"
        info = info + "ACL Beta: %.3f" % beta + "\n"
        pl = Tix.Label(self, text=info);
        pl.pack(anchor=Tix.W)

class SequenceDegreePlot(Tix.Frame):
    def __init__(self, sequence, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        self.fig = Figure(figsize=(5,4), dpi=100)
        a = self.fig.add_subplot(111, title='Degree Distribution',\
                xlabel='Node Degree', ylabel='Number of Nodes')
        t = range(sequence.get_smallest_val(), sequence.get_largest_val()+1)
        s = []
        for i in t:
            s.append(sequence.get_num_degs(i))
        a.plot(t,s,'ro')

        self.canvas = FigureCanvasTkAgg(self.fig,master=self)
        self.canvas.show()
        self.canvas.get_tk_widget().pack(side=Tix.TOP, fill=Tix.BOTH,\
                expand=1)
        self.toolbar = NavigationToolbar2TkAgg( self.canvas, self )
        self.toolbar.update()
        self.canvas._tkcanvas.pack(side=Tix.TOP, fill=Tix.BOTH, expand=1)

## Returns an alpha and beta using least squares
#  where log y = alpha - beta * log x in the equation on page 55
#  from Aiello, Chung, and Lu. 
#  Using http://mathworld.wolfram.com/LeastSquaresFitting.html 
#  Equations (12) and (14) 
#  @author Brian Cloteaux
#  @see "Aiello, Chung, and Lu. A Random Graph Model for Power Law Graphs.
#  Experimental Mathematics, 10(1). pg 53-66 (2001)."
def getACLModel(sequence): 
    sx  = 0.0 
    sy  = 0.0 
    sx2 = 0.0
    sxy = 0.0
    n = 0.0
    i = sequence.get_largest_val()+1
    while i != sequence.get_smallest_val():
        i = sequence.get_next_smaller(i)
        if i>0:
            n += 1.0
            x = math.log(float(i))
            y = math.log(float(sequence.get_num_degs(i)))
            sx += x
            sy += y
            sx2 += x * x
            sxy += x * y
    sxsx = sx * sx
    d = n*sx2 - sxsx
    alpha = (sy*sx2 - sx*sxy) / d  
    beta = (n*sxy - sx*sy) / d  
    return (alpha,-1.0 * beta)
