"""
NMTK Frame

"""
__author__ = """Brian Cloteaux (brian.cloteaux@nist.gov)"""
#  Mathematical and Computational Sciences Division
#  National Institute of Standards and Technology,
#  Gaithersburg, MD USA
# 
#  This software was developed at the National Institute of Standards and
#  Technology (NIST) by employees of the Federal Government in the course
#  of their official duties. Pursuant to title 17 Section 105 of the
#  United States Code, this software is not subject to copyright protection
#  and is in the public domain. NIST assumes no responsibility whatsoever for
#  its use by other parties, and makes no guarantees, expressed or implied,
#  about its quality, reliability, or any other characteristic.

import Tix
import tkFileDialog
import os.path
import networkx as nx
import copy

import NMTKModel
import SequenceDisplay
import GraphDisplay
import AffiliationDisplay
import SequenceGeneratorDialogs
import GraphGeneratorDialogs
import DegreeSequenceReader
import SequentialGraphGeneration as sgg
import AffiliationGeneration 

import degree_sequence

class NMTKFrame(Tix.Frame):
    def __init__(self, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(fill=Tix.BOTH, expand=Tix.YES)
        self.parent = parent
        self.panes = Tix.PanedWindow(self, orientation='vertical')
        self.model = NMTKModel.NMTKModel()

        p1 = self.panes.add('affil')
        self.affildisplay = AffiliationDisplay.AffiliationDisplay(self.model,p1)
        self.affildisplay.pack(side=Tix.TOP, expand=Tix.YES, fill=Tix.BOTH) 
        p2 = self.panes.add('graph')
        self.graphdisplay =  GraphDisplay.GraphDisplay(self.model, p2)
        self.graphdisplay.pack(side=Tix.TOP, expand=Tix.YES, fill=Tix.BOTH) 
        p3 = self.panes.add('seq',min=100)
        self.seqdisplay =  SequenceDisplay.SequenceDisplay(self.model, p3)
        self.seqdisplay.pack(side=Tix.TOP, expand=Tix.YES, fill=Tix.BOTH) 

        self.panes.pack(side=Tix.TOP,fill=Tix.BOTH,expand=1)

    def updateDisplay(self):
        if self.model.getAffiliation() != None:
            self.panes.paneconfigure('affil',min=200)
        else:
            self.panes.paneconfigure('affil',min=2)
        if self.model.getGraph() != None:
            self.panes.paneconfigure('graph',min=200)
        else:
            self.panes.paneconfigure('graph',min=2)
        self.affildisplay.updateAffiliation(self.model)
        self.graphdisplay.updateGraph(self.model)
        self.seqdisplay.updateSequence(self.model)

    def readSequence(self):
        filename = tkFileDialog.askopenfilename(title='Choose a file', \
                filetypes = [('Sequence file', '.deg')] )
        if filename != None:
            seq = DegreeSequenceReader.getSequence(filename)
            self.model.updateSequence(seq)
            self.updateDisplay()

    def writeSequence(self):
        filename = tkFileDialog.asksaveasfilename(title='Choose a file', \
                filetypes = [('Sequence file', '.deg')] )
        if filename != None:
            DegreeSequenceReader.writeSequence(self.model.getSequence(),\
                    filename)

    def generateSequence(self, type):
        seq = None
        if type=="pareto":
            vals = SequenceGeneratorDialogs.ParetoDialog(self)
            if vals.number != None and vals.exponent != None:
                kwds = { 'exponent':vals.exponent } 
                ds = nx.create_degree_sequence(100,nx.utils.pareto_sequence, \
                                50,**kwds)
                seq = degree_sequence.degree_sequence(ds)
        elif type=="power":
            vals = SequenceGeneratorDialogs.PowerLawDialog(self)
            if vals.number != None and vals.exponent != None:
                kwds = { 'exponent':vals.exponent } 
                ds = nx.create_degree_sequence(100,nx.utils.powerlaw_sequence,\
                        50,**kwds)
                seq = degree_sequence.degree_sequence(ds)
        elif type=="uniform":
            vals = SequenceGeneratorDialogs.UniformDialog(self)
            if vals.number != None:
                ds = nx.create_degree_sequence(100,nx.utils.uniform_sequence)
                seq = degree_sequence.degree_sequence(ds)
        if seq != None:
            self.model.updateSequence(seq)
            self.updateDisplay()

        

    def readGraph(self):
        filename = tkFileDialog.askopenfilename(title='Choose a file', \
                filetypes = [('GML', '.gml'), ('GraphML', '.graphml')] )
        if filename is not None:
            extension = os.path.splitext(filename)[1]
            graph = None
            if extension == ".gml":
                graph = nx.read_gml(filename)
            elif extension == ".graphml":
                graph = nx.read_graphml(filename)
            if graph is not None:
                graph.name = filename
                self.model.updateGraph(graph)
                self.updateDisplay()

    def writeGraph(self,type):
        if type=='affil':
            graph = self.model.getAffiliation()
        else:
            graph = self.model.getGraph()
        filename = tkFileDialog.asksaveasfilename(title='Choose a file', \
                filetypes = [('GML', '.gml'), ('GraphML', '.graphml'), \
                ('GraphViz','.dot')] )
        if filename is not None:
            extension = os.path.splitext(filename)[1]
            if extension == ".gml":
                nx.write_gml(graph,filename)
            elif extension == ".graphml":
                nx.write_graphml(graph,filename)
            elif extension == ".dot":
                nx.write_dot(graph,filename)

    def getModelCopy(self):
        return copy.deepcopy(self.model)

    def cloneNMTK(self, nmtkModel):
        self.model = nmtkModel
        self.updateDisplay()

    def nearestGraphical(self):
        seq = self.model.getSequence()
        seq.closest_graphical() 
        self.updateDisplay()

    def createGraphFromSequence(self,type):
        graph = None
        seq = self.model.getSequence()
        if seq.is_graphical():
            if type=="markov":
                graph = sgg.markov(seq)
            elif type=="con_markov":
                graph = sgg.connected_markov(seq)
            elif type=="blitzstein":
                graph = sgg.blitzstein_diaconis(seq)
        if graph is not None:
            self.model.updateGraph(graph)
            self.updateDisplay()

    def generateGraph(self,type):
        graph = None
        if type=="erdos":
            vals = GraphGeneratorDialogs.ErdosRenyiDialog(self)
            if vals.number != None and vals.probability != None:
                graph = nx.erdos_renyi_graph(vals.number, vals.probability)
        elif type=="strogatz":
            vals = GraphGeneratorDialogs.WattsStrogatzDialog(self)
            if vals.number != None and vals.neighbors != None:
                graph = nx.newman_watts_strogatz_graph(vals.number, \
                        vals.neighbors, vals.probability)
        elif type=="barabasi":
            vals = GraphGeneratorDialogs.BarabasiAlbertDialog(self)
            if vals.number != None and vals.edges != None:
                graph = nx.barabasi_albert_graph(vals.number, vals.edges)
        if graph != None:
            self.model.updateGraph(graph)
            self.updateDisplay()

    def generateAffil(self,type):
        affil = None
        graph = self.model.getGraph()
        if graph != None:
            if type == 'max_clique':
                affil = nx.make_clique_bipartite(graph)
            else:
                affil = AffiliationGeneration.guillaume_latapy(graph)
        if affil != None:
            self.model.updateAffiliation(affil)
            self.updateDisplay()

    def readAffiliation(self):
        filename = tkFileDialog.askopenfilename(title='Choose a file', \
                filetypes = [('GML', '.gml'), ('GraphML', '.graphml')] )
        if filename is not None:
            extension = os.path.splitext(filename)[1]
            affil = None
            if extension == ".gml":
                affil = nx.read_gml(filename)
            elif extension == ".graphml":
                affil = nx.read_graphml(filename)
            if affil is not None:
                if nx.is_bipartite(affil):
                    affil.name = filename
                    affil.node_type={}
                    coloring = nx.bipartite_color(affil)
                    if coloring['0'] == 0:
                        v1 = "Bottom"
                        v2 = "Top" 
                    else:
                        v1 = "Top" 
                        v2 = "Bottom" 
                    for n,c in coloring.iteritems():
                        if c==0:
                            affil.node_type[n] = v1
                        else:
                            affil.node_type[n] = v2
                    self.model.updateAffiliation(affil)
                    self.updateDisplay()

    def flipAffil(self):
        affil = self.model.getAffiliation()
        if affil != None:
            for n in affil.nodes():
                if affil.node_type[n] == "Top": 
                    affil.node_type[n] = "Bottom"
                else:
                    affil.node_type[n] = "Top"
            self.model.updateAffiliation(affil)
            self.updateDisplay()
