"""
Affiliation Plot

"""
__author__ = """Brian Cloteaux (brian.cloteaux@nist.gov)"""
#  Mathematical and Computational Sciences Division
#  National Institute of Standards and Technology,
#  Gaithersburg, MD USA
# 
#  This software was developed at the National Institute of Standards and
#  Technology (NIST) by employees of the Federal Government in the course
#  of their official duties. Pursuant to title 17 Section 105 of the
#  United States Code, this software is not subject to copyright protection
#  and is in the public domain. NIST assumes no responsibility whatsoever for
#  its use by other parties, and makes no guarantees, expressed or implied,
#  about its quality, reliability, or any other characteristic.

import Tix
import math
import degree_sequence

import NMTKModel

import matplotlib
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg, NavigationToolbar2TkAgg
from matplotlib.figure import Figure
import matplotlib.pyplot as plt

class AffiliationPlot(Tix.Frame): 
    def __init__(self, model, parent=None):
        Tix.Frame.__init__(self, parent, width=10, height=10)
        self.pack(fill=Tix.BOTH, expand=Tix.YES)
        self.parent = parent
        self.affiliationplot = None
        self.model = model
        top = []
        bot = []
        if self.model != None:
            affil = self.model.getAffiliation()
            if affil != None:
                for n in affil.nodes():
                    if affil.node_type[n] == "Top":
                        top.append(affil.degree(n))
                    elif affil.node_type[n] == "Bottom":
                        bot.append(affil.degree(n))
        self.topsequence = degree_sequence.degree_sequence(top)
        self.bottomsequence = degree_sequence.degree_sequence(bot)

        self.v = Tix.IntVar()
        self.v.set(1)
        rbf = Tix.Frame(self)
        Tix.Radiobutton(rbf,text="Properties",variable=self.v,value=1, \
                command=self.updateSequence).pack(anchor=Tix.W,side=Tix.LEFT)
        Tix.Radiobutton(rbf,text="Bottom Plot",variable=self.v,value=2, \
                command=self.updateSequence).pack(anchor=Tix.W,side=Tix.LEFT)
        Tix.Radiobutton(rbf,text="Top Plot",variable=self.v,value=3, \
                command=self.updateSequence).pack(anchor=Tix.W,side=Tix.LEFT)
        rbf.pack(side=Tix.BOTTOM)
        self.updateSequence()

    def updateSequence(self, model=None):
        if self.affiliationplot != None:
            self.affiliationplot.pack_forget()
        if self.v.get()==2:
            self.affiliationplot = \
            AffiliationDegreePlot(self.bottomsequence, False, self) 
        elif self.v.get()==3:
            self.affiliationplot = \
            AffiliationDegreePlot(self.topsequence, True, self) 
        else:
            self.affiliationplot = AffiliationProperties(self.bottomsequence,\
                    self.topsequence, self)
        self.affiliationplot.pack(side=Tix.TOP)

class AffiliationProperties(Tix.Frame):
    def __init__(self, bottom, top, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        info = "Bottom Sequence\n"
        info = info + "Number nodes: " + str(bottom.get_length()) + "\n"
        info = info + "Degree sum: " + str(bottom.get_degree_sum()) + "\n"
        avgDeg = float(bottom.get_degree_sum()) / float(bottom.get_length())
        info = info + "Average degree: %.3f" % avgDeg + "\n"
        info = info + "Minimum degree: " + str(bottom.get_smallest_val()) + "\n"
        info = info + "Maximum degree: " + str(bottom.get_largest_val()) + "\n"
        info = info + "\nTop Sequence\n"
        info = info + "Number nodes: " + str(top.get_length()) + "\n"
        info = info + "Degree sum: " + str(top.get_degree_sum()) + "\n"
        avgDeg = float(top.get_degree_sum()) / float(top.get_length())
        info = info + "Average degree: %.3f" % avgDeg + "\n"
        info = info + "Minimum degree: " + str(top.get_smallest_val()) + "\n"
        info = info + "Maximum degree: " + str(top.get_largest_val()) + "\n"
        pl = Tix.Label(self, text=info);
        pl.pack(anchor=Tix.W)

class AffiliationDegreePlot(Tix.Frame):
    def __init__(self, sequence, isTop=True, parent=None):
        Tix.Frame.__init__(self, parent)
        self.pack(expand=Tix.YES, fill=Tix.BOTH)
        self.fig = Figure(figsize=(5,4), dpi=100)
        if isTop:
            t = 'Top Degree Distribution'
        else:
            t = 'Bottom Degree Distribution'
        a = self.fig.add_subplot(111, title=t, xlabel='Node Degree', \
                ylabel='Number of Nodes')
        t = range(sequence.get_smallest_val(), sequence.get_largest_val()+1)
        s = []
        for i in t:
            s.append(sequence.get_num_degs(i))
        a.plot(t,s,'ro')

        self.canvas = FigureCanvasTkAgg(self.fig,master=self)
        self.canvas.show()
        self.canvas.get_tk_widget().pack(side=Tix.TOP, fill=Tix.BOTH,\
                expand=1)
        self.toolbar = NavigationToolbar2TkAgg( self.canvas, self )
        self.toolbar.update()
        self.canvas._tkcanvas.pack(side=Tix.TOP, fill=Tix.BOTH, expand=1)

