%{

/************************************************************************
  DISCLAIMER:
  This software was produced by the National Institute of Standards
  and Technology (NIST), an agency of the U.S. government, and by statute
  is not subject to copyright in the United States.  Recipients of this
  software assume all responsibility associated with its operation,
  modification, maintenance, and subsequent redistribution.

  See NIST Administration Manual 4.09.07 b and Appendix I.
************************************************************************/

#include <string.h>  // for strlen, strcpy, strcat
#include <stdio.h>   // for fopen, etc.
#include <stdlib.h>  // for exit
#include "dmis.hh"

#define YYERROR_VERBOSE
#define YYDEBUG 1
#define PRENAME "PrEpRoCeSsDmIs"

/********************************************************************/

class stringListCell
{
 public:
  stringListCell(){theString = 0; next = 0;}
  stringListCell(char * theStringIn)
    {theString = theStringIn; next = 0;}
  ~stringListCell(){}

  int find(char * findIt)
  { // assumes list is in alphabetical order
    stringListCell * aCell;
    int compVal;

    if (theString == 0)
      return 0;
    for (aCell = this; aCell; aCell = aCell->next)
      {
	compVal = strcmp(findIt, aCell->theString);
	if (compVal == 0)
	  return 1;
	else if (compVal < 0)
	  return 0;
      }
    return 0;
  }

  void record(char * theStringIn)
  { // keeps list in alphabetical order, ignores duplicates
    stringListCell * aCell;
    stringListCell * newCell;
    stringListCell * backCell;
    int compVal;

    if (theString == 0)
      { // happens when first entry is made
	theString = theStringIn;
	return;
      }
    compVal = strcmp(theStringIn, theString);
    if (compVal < 0)
      { // theStringIn comes first, so need to shift
	newCell = new stringListCell(theString);
	theString = theStringIn;
	newCell->next = next;
	next = newCell;
	return;
      }
    else if (compVal == 0)
      { // ignore duplicate
	return;
      }
    backCell = this;
    for (aCell = next; aCell; aCell = aCell->next)
      {
	compVal = strcmp(theStringIn, aCell->theString);
	if (compVal < 0)
	  { // new cell goes immediately before aCell
	    newCell = new stringListCell(theStringIn);
	    backCell->next = newCell;
	    newCell->next = aCell;
	    return;
	  }
	else if (compVal == 0)
	  { // ignore duplicate
	    return;
	  }
	backCell = aCell;
      }
    // new cell must go at the end, since we haven't returned yet
    newCell = new stringListCell(theStringIn);
    backCell->next = newCell;
  }
  
  char * theString;
  stringListCell * next;
};

class stringList
{
 public:
  stringList(){first = 0; last = 0;}
  ~stringList(){}
  stringListCell * first;
  stringListCell *last;
  void pushBack(char * text)
  {
    if (last == 0)
      {
	last = new stringListCell(text);
	first = last;
      }
    else
      {
	last->next = new stringListCell(text);
	last = last->next;
      }
  }
};


/********************************************************************/

class macroRep
{
 public:
  macroRep(){text = 0; arguments = 0; name = 0;}
  macroRep(stringList * textIn, stringList * argumentsIn, char * nameIn)
    {
      text = textIn;
      arguments = argumentsIn;
      name = nameIn;
    }
  ~macroRep(){}
  stringList * text;
  stringList * arguments;
  char * name;
};

class macroListCell
{
 public:
  macroListCell(){theMacro = 0; next = 0;}
  macroListCell(macroRep * theMacroIn){theMacro = theMacroIn; next = 0;}
  ~macroListCell(){}
  macroRep * theMacro;
  macroListCell * next;
};

class macroList
{
 public:
  macroList(){first = 0; last = 0;}
  ~macroList(){}
  macroListCell * first;
  macroListCell * last;
  void pushBack(macroRep * aMacro)
  {
    if (last == 0)
      {
	last = new macroListCell(aMacro);
	first = last;
      }
    else
      {
	last->next = new macroListCell(aMacro);
	last = last->next;
      }
  }
};

/********************************************************************/

/*

The lineText array is used for saving everything on a line up to the end
of the line, 4096 characters, or an error, whichever comes first. The
lineText is used (printed by yyerror) only if there is an error. This 
lets the user see the point in the DMIS file at which an error occurred.
Lines longer than 4096 characters will not overflow the lineText array.

*/

char *                     aLabel;
int                        aLabelFound = 1;
int                        aLabelType;
std::list<dmisStatement *> dmisStms;
int                        getCallArgs = 0;
int                        inDecl = 0;
stringListCell             labels[1000];
char                       lexMessage[80];
char                       lexWarning[80];
int                        lineNo;
char                       lineText[4096];
int                        macroIsReal = 1;
macroList                  macros;
int                        numErrors = 0;
int                        numMacros = 0;
int                        numWarnings = 0;
int                        resetLex = 0;
char *                     setLabel;
int                        setLabelType;
inputFile *                tree;
char                       warningMessage[256];
extern FILE *              yyin;
extern int                 yylex();

/********************************************************************/

macroRep * findMacro(
 char * name)
{
  macroListCell * macroCell;

  for (macroCell = macros.first; macroCell; macroCell = macroCell->next)
    {
      if (strcmp(name, macroCell->theMacro->name) == 0)
	return macroCell->theMacro;
    }
  return 0;
}

/********************************************************************/

/* isMacro

This returns 1 if the beginning of the text is "M(label)=MACRO", where
there may be white space before the first M, before and after the
parentheses, and before and after the equal sign, and the letters in
MACRO may be upper or lower case.

Otherwise, it returns 0.

This stores the label in the name argument, changing any lower case
letters in the label to upper case.

*/

int isMacro(
 char * text,
 char ** name)
{
  int n;
  int k;
  char c;
  static char buffer[256];

  for (n = 0; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c != 'm') && (c != 'M'))
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != '(')
    return 0;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c == 0)
    return 0;
  for (k = 0; (((c = text[n]) == '.') || (c == '-') || (c == '_') ||
	       ((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'z')) ||
	       ((c >= 'A') && (c <= 'Z'))); n++)
    buffer[k++] = (((c >= 'a') && (c <= 'z')) ? (c + ('A' - 'a')) : c);
  for (; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != ')')
    return 0;
  buffer[k] = 0;
  *name = strdup(buffer);
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != '=')
    return 0;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c != 'm') && (c != 'M'))
    return 0;
  if (((c = text[++n]) != 'a') && (c != 'A'))
    return 0;
  if (((c = text[++n]) != 'c') && (c != 'C'))
    return 0;
  if (((c = text[++n]) != 'r') && (c != 'R'))
    return 0;
  if (((c = text[++n]) != 'o') && (c != 'O'))
    return 0;
  return 1;
}

/********************************************************************/

/* isCall

This returns 1 if the beginning of the text is "CALL/M(label)",
where there may be white space before the C, before and after the
slash, and before and after the parentheses, and the letters
in CALL may be upper or lower case.

Otherwise, this returns 2 if the beginning of the text is
"CALL/EXTERN,DMIS,M(label)", with similar allowance for white space.

Otherwise, this returns 3 if the beginning of the text is
"CALL/EXTERN"

Otherwise, this returns 0.

If this returns 1 or 2, it also stores the label in the name argument,
changing any lower case letters in the label to upper case. Otherwise,
it makes name the empty string.

*/

int isCall(
 char * text,
 char * name)
{
  int n;
  int k;
  char c;
  static char buffer[256];

  name[0] = 0;
  for (n = 0; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c != 'c') && (c != 'C'))
    return 0;
  if (((c = text[++n]) != 'a') && (c != 'A'))
    return 0;
  if (((c = text[++n]) != 'l') && (c != 'L'))
    return 0;
  if (((c = text[++n]) != 'l') && (c != 'L'))
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != '/')
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c == 'm') || (c == 'M'))
    {
      for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c != '(')
	return 0;
      for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
      for (k = 0; (((c = text[n]) == '.') || (c == '-') || (c == '_') ||
		   ((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'z')) ||
		   ((c >= 'A') && (c <= 'Z'))); n++)
	buffer[k++] = (((c >= 'a') && (c <= 'z')) ? (c + ('A' - 'a')) : c);
      for (; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c != ')')
	return 0;
      buffer[k] = 0;
      strcpy(name, buffer);
      return 1;
    }
  if ((c != 'e') && (c != 'E'))
    return 0;
  if (((c = text[++n]) != 'x') && (c != 'X'))
    return 0;
  if (((c = text[++n]) != 't') && (c != 'T'))
    return 0;
  if (((c = text[++n]) != 'e') && (c != 'E'))
    return 0;
  if (((c = text[++n]) != 'r') && (c != 'R'))
    return 0;
  if (((c = text[++n]) != 'n') && (c != 'N'))
    return 0;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != ',')
    return 0;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c != 'd') && (c != 'D'))
    return 3;
  if (((c = text[++n]) != 'm') && (c != 'M'))
    return 3;
  if (((c = text[++n]) != 'i') && (c != 'I'))
    return 3;
  if (((c = text[++n]) != 's') && (c != 'S'))
    return 3;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != ',')
    return 0;
  for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c == 'm') || (c == 'M'))
       {
      for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c != '(')
	return 0;
      for (n ++; (((c = text[n]) == ' ') || (c == '\t')); n++);
      for (k = 0; (((c = text[n]) == '.') || (c == '-') || (c == '_') ||
		   ((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'z')) ||
		   ((c >= 'A') && (c <= 'Z'))); n++)
	buffer[k++] = (((c >= 'a') && (c <= 'z')) ? (c + ('A' - 'a')) : c);
      for (; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c != ')')
	return 0;
      buffer[k] = 0;
      strcpy(name, buffer);
      return 2;
    }
  return 3;
}

/********************************************************************/

int isEndmac(
 char * text)
{
  int n;
  char c;

  for (n = 0; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if ((c != 'e') && (c != 'E'))
    return 0;
  if (((c = text[++n]) != 'n') && (c != 'N'))
    return 0;
  if (((c = text[++n]) != 'd') && (c != 'D'))
    return 0;
  if (((c = text[++n]) != 'm') && (c != 'M'))
    return 0;
  if (((c = text[++n]) != 'a') && (c != 'A'))
    return 0;
  if (((c = text[++n]) != 'c') && (c != 'C'))
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c != 13)
    return 0;
  return 1;
}

/********************************************************************/

/* findMacroArgs

This returns 1 if it is able to read and store all the macro arguments
given in the text. Otherwise, it returns 0.

The text is known to start with "M(label)=MACRO", so this skips to the
equal sign and from there to the O at the end of MACRO.

Lower case letters in the names of the macro arguments are changed
to upper case.

Arguments are required to have the format of variable names. That is,
they must start with a letter and contain only letters, digits, and
underscores. Quoted variable names (which represent labels) are stored
without the quotes.

*/

int findMacroArgs(
 char * text,
 stringList * arguments)
{
  int n;
  int k;
  char c;
  static char buffer[256];

  for (n = 0; text[n] != '='; n++); // find equal sign
  for (n++; (((c = text[n]) != 'o') && (c != 'O')); n++); //find o in MACRO
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++); // skip space
  if (c == 13) // found macro with no arguments
    {
      return 1;
    }
  else if (c != '/')
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++); // skip space
  for (; ((c = text[n]) != 13) ; )
    {
      if (c == '\'')
	{ // copy quoted argument (for label) without the quotes
	  c = text[++n];
	  if (!(((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))))
	    return 0;
	  for (k = 0;
	       (((c = text[n]) == '_') || ((c >= '0') && (c <= '9')) ||
		((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z')));
	       n++)
	    buffer[k++] = (((c >= 'a') && (c <= 'z'))?(c + ('A' - 'a')):c);
	  if (c == '\'')
	    n++;
	  else
	    return 0;
	}
      else if (((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z')))
	{ // copy argument starting with a letter
	  for (k = 0;
	       (((c = text[n]) == '_') || ((c >= '0') && (c <= '9')) ||
		((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z')));
	       n++)
	    buffer[k++] = (((c >= 'a') && (c <= 'z'))?(c + ('A' - 'a')):c);
	}
      else
	return 0;
      buffer[k] = 0;
      arguments->pushBack(strdup(buffer));
      for (; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c == ',')
	{
	  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
	}
      else if (c != 13)
	return 0;
    }
  return 1;
}

/********************************************************************/

/* doMacro

A. If buffer is 0, just reset the static inMacro local variable to 0.
The inMacro variable should be zero when parsing a DMIS input file
starts. If a number of files are being parsed and there
is an error in a macro in a file, inMacro may have a non-zero
value when parsing that file is finished. To do the resetting,
resetParser calls doMacro(0,0) before parsing a new file.

B. Otherwise, if inMacro is 0, meaning we are not already inside a
MACRO, this reads the buffer and checks whether a macro is starting.
If so, it does B1 and B2. Otherwise, it does nothing more.

B1. A new macroRep is constructed for the macro. If no macro by the
same name is already recorded, the new macro is recorded (the parser
will signal an error later if a macro by the same name is already
recorded).

B2. Next, this (1) tries to read the arguments of the macro from the
macroLine and store them in the macroArgs [if reading arguments fails,
this prints an error message and quits], and (2) sets inMacro to 1.

C. Otherwise, (inMacro must be 1) we are on a line of a macro after
the the first line. This does C1, C2, or C3.

C1. If the statement in the buffer defines a MACRO (so that one macro
definition is inside another), this prints an error message and quits.

C2. If the statement in the buffer defines a CALL, this prints an
error message and quits.

C3. This adds the macroLine to the macroText, and if this is an ENDMAC
line, this sets inMacro to 0.


Note: The macroText does not include line numbers.

*/

void doMacro(
 char * buffer,
 int lineNumber)
{
  macroRep * aMacro;              // macro to deal with
  static char callName[256];      // name of macro in CALL
  static char * macroName;        // name of macro in MACRO
  static stringList * macroText;  // text of macro
  static stringList * macroArgs;  // arguments of macro
  static int inMacro = 0;         // 0=not in macro, 1=in macro body

  if (buffer == 0)
    { // for resetting inMacro, call doMacro with buffer = 0
      inMacro = 0;
    }
  else if (inMacro == 0)
    {
      if (isMacro(buffer, &macroName))
	{
	  macroText = new stringList;
	  macroArgs = new stringList;
	  aMacro = findMacro(macroName);
	  if (!aMacro)
	    {
	      aMacro = new macroRep(macroText, macroArgs, macroName);
	      macros.pushBack(aMacro);
	    }
	  if (!findMacroArgs(buffer, macroArgs))
	    {
	      fprintf(stderr, "%d: bad macro arguments. Quitting.\n",
		      lineNumber);
	      exit(1);
	    }
	  macroText->pushBack(strdup(buffer));
	  inMacro = 1;
	}
    } // below here, (inMacro == 1)
  else if (isMacro(buffer, &macroName))
    {
      fprintf(stderr, "%d: macro defined inside macro. Quitting.\n",
	      lineNumber);
      exit(1);
    }
  else if (isCall(buffer, callName))
    {
      fprintf(stderr, "%d: cannot handle call inside macro. Quitting.\n",
	      lineNumber);
      exit(1);
    }
  else
    { // inside a macro
      macroText->pushBack(strdup(buffer));
      if (isEndmac(buffer))
	inMacro = 0;
    }
}

/********************************************************************/

/* findCallArgs

This returns 1 if it is able to read and store all the call arguments
given in the text. Otherwise, it returns 0.

This is called only if text is known to start with "CALL/M(label)" or
"CALL/EXTERN,DMIS,M(label)", so this skips to the first right
parenthesis. This should be followed by a comma and then any
arguments.

Arguments enclosed in parentheses are expected to be label names.

This sets numberArgs to the number of arguments found.

*/

int findCallArgs(
 char * text,
 stringList * callArgs,
 int * numberArgs)
{
  int n;
  int k;
  char c;
  static char buffer[256];

  *numberArgs = 0;
  for (n = 0; ((c = text[n]) != ')'); n++); //skip to right parenthesis
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  if (c == 13) // found call with no arguments
    {
      return 1;
    }
  else if (c != ',')
    return 0;
  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
  for (; ((c = text[n]) != 13) ; )
    {
      if (c == '(')
	{ // copy parenthesized argument (for label) without the parentheses
	  for (k = 0, n++; (((c = text[n]) != 13) && (c != ')')); n++)
	    buffer[k++] = c;
	  if (c == ')')
	    n++;
	  else
	    return 0;
	}
      else if (c == '\'')
	{ // copy quoted string with the quotes
	  buffer[0] = c;
	  for (k = 1, n++; (((c = text[n]) != 13) && (c != '\'')); n++)
	    buffer[k++] = c;
	  if (c == '\'')
	    {
	      buffer[k++] = c;
	      n++;
	    }
	  else
	    return 0;
	}
      else // not parenthesized or quoted
	{ // copy any other characters that are not comma, space, or endline.
	  for (k = 0;
	       (((c = text[n]) != 13) && (c != ' ') &&
		(c != '\t') && (c != ','));
	       n++)
	    buffer[k++] = c;
	}
      buffer[k] = 0;
      callArgs->pushBack(strdup(buffer));
      (*numberArgs)++;
      for (; (((c = text[n]) == ' ') || (c == '\t')); n++);
      if (c == ',')
	{
	  for (n++; (((c = text[n]) == ' ') || (c == '\t')); n++);
	}
      else if (c != 13)
	return 0;
    }
  return 1;
}

/********************************************************************/

/* insertCalledMacro

If the number of arguments to theMacro differs from numberArgs
(the number of arguments to the call), this records an error
message, prints an ENDMAC line, and returns.

If there are no arguments, this prints the text of theMacro, starting
with the second line.

Otherwise, this proceeds line by line through the text of theMacro. It
copies the line into the callLine, making changes as follows. On each
line, it looks for maximum length text strings whose characters could
be a variable name. Each time it finds such a string, it checks if the
string is identical to one of the arguments of the macro. If so, the
the corresponding CALL argument is copied into the callLine instead of
the string. When the entire MACRO text line has been checked, the
callLine is printed in the preprocessed file.

For comparison with MACRO arguments, letters in the text string are
changed to upper case.

The line number of the CALL is printed at the beginning of every line.

*/

void insertCalledMacro( /* ARGUMENTS                       */
 stringList * callArgs, /* arguments of the CALL           */
 macroRep * theMacro,   /* called macro                    */
 int numberArgs,        /* number of arguments to the CALL */
 int lineNumber,        /* line number of the CALL         */
 FILE * out)            /* preprocessor output file        */
{
  int n;                         // counter for arguments
  stringListCell * textCell;     // list cell for text of macro
  stringListCell * macroArgCell; // list cell for arguments of macro
  stringListCell * callArgCell;  // list cell for arguments of call
  int k;                         // counter for line of macro
  char * text;                   // line of macro text
  char word[256];                // word extracted from line of macro text
  int m;                         // counter for word
  char c;                        // current character

  for(macroArgCell = theMacro->arguments->first, n=0;
      macroArgCell;
      macroArgCell = macroArgCell->next, n++);
  if (n != numberArgs)
    {
      fprintf(out, "%d|ENDMAC%c%c", lineNumber, 13, 10);
      printf(
      "%d: Error - Numbers of CALL (%d) and MACRO (%d) arguments differ\n\n",
	      lineNumber, numberArgs, n);
      numErrors++;
      return;
    }
  else if (numberArgs == 0)
    {
      for (textCell = theMacro->text->first->next;
	   textCell;
	   textCell = textCell->next)
	{
	  fprintf(out, "%d|%s", lineNumber, textCell->theString);
	}
      return;
    }
  for (textCell = theMacro->text->first->next;
       textCell;
       textCell = textCell->next)
    {
      fprintf(out, "%d|", lineNumber);
      text = textCell->theString;
      for (k = 0; ((c = text[k]) != 0); k++)
	{
	  if (((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z')))
	    { // found the start of a word that could be a macro argument
	      for (m = 0;
		   (((c = text[k+m]) == '_') || ((c >= '0') && (c <= '9')) ||
		    ((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z')));
		     m++)
		word[m] = (((c >= 'a') && (c <= 'z')) ? (c + ('A' - 'a')) : c);
	      word[m] = 0;
	      // found a replaceable word; replace if it is a macro arg
	      for (macroArgCell = theMacro->arguments->first,
		     callArgCell = callArgs->first;
		   macroArgCell;
		   macroArgCell = macroArgCell->next,
		     callArgCell = callArgCell->next)
		{
		  if (strcmp(macroArgCell->theString, word) == 0)
		    break;
		}
	      if (macroArgCell)
		fprintf(out, "%s", callArgCell->theString);
	      else
		fprintf(out, "%s", word);
	      k = (k + m - 1);
	    }
	  else if (c == '\'')
	    { // copy quoted strings without looking for macro arguments
	      fputc(c, out);
	      for (k++; ((c = text[k]) != '\''); k++)
		fputc(c, out);
	      fputc(c, out);
	    }
	  else
	    fputc(c, out);
	}
    }
}

/********************************************************************/

/* doCall

This returns 1 if the line being read is a CALL line. Otherwise, it
returns 0.

If a call starts with CALL/M(label), then callType is 1. In this case
a MACRO is being called that should already have been defined in the
file being read, so a check is made of whether the CALLed MACRO (whose
name has been placed in macroName) has been defined. If not, an error
message is recorded.

If a call starts with CALL/EXTERN,DMIS,M(label), then callType is 2.
In this case, a MACRO is being called that is probably not defined in
the file being read. If the same macro was called previously, however,
a dummy macro by that name has been defined, so a check is made of
whether the CALLed MACRO (whose name has been placed in macroName) has
been defined.

If a call starts with CALL/ followed by anything else, then callType
is 3. In this case, only the CALL line and a following ENDMAC line
are printed.

If callType is 1 or 2:

   If the called macro has not yet been defined, then a dummy macro is
   is put into the preprocessed file just before the CALL. The dummy MACRO
   has the name given in the CALL and has the same number of arguments as
   the call. Having the same number of arguments is necessary in case
   another call is made to the same macro, since then the number of
   arguments will be checked. The dummy MACRO has only two lines (a MACRO
   line and an ENDMAC line). Inserting a dummy MACRO will prevent the
   parser from reporting an undefined label when it reads the M(label)
   in the CALL.

   The CALL line is printed.

   The modified text of the CALLed macro (without the MACRO line but
   with the ENDMAC line) is inserted after the CALL statement. If
   the CALLed macro is a dummy, this consists of only an ENDMAC line.
   The text of the macro is modified by replacing appearances of macro
   arguments with the corresponding arguments from the CALL.

When the parser parses a CALL in the preprocessed file, it will then
be parsing the macro after the CALL substitutions are made (so there
is no need to deal with untyped macro arguments). Thus, the parser will
deal gracefully with all macros, including undefined EXTERN macros.
At the point in the file where a macro is defined, the macro is parsed
as though it is unintelligible text (so there is no need to deal
with untyped macro arguments).

The parser considers that a CALL block starts with CALL and ends with
ENDMAC. The EBNF has been modified to define a CALL block that way.
The purpose of defining a CALL block is to establish the scope of any
variables defined in a CALLed MACRO.

This is all part of the "text substitution" approach to implementing
DMIS macros, which is like the C and C++ approach and satisfies the
requirements on macros given in the DMIS spec.

*/

int doCall(       /* ARGUMENTS                          */
 char * buffer,   /* buffer containing a DMIS statement */
 int lineNumber,  /* number of line starting statement  */
 FILE * out)      /* file to write preprocessed DMIS to */
{
  static char macroName[256];   // name of CALLed macro
  macroRep * theMacro;          // the CALLed macro
  static char text[1024];       // text of a line of dummy macro
  int m;                        // counter for characters of text
  char argName[20];             // name of an argument to dummy macro
  stringList * macroArgs;       // arguments of dummy macro
  stringList * macroText;       // text of dummy macro
  static int callType;          // 0=not, 1=M, 2=EXTERN,DMIS,M, 3=other EXTERN
  static stringList * callArgs; // arguments of call
  stringListCell * argCell;     // cell of callArgs
  int numberArgs;               // the number of arguments to the CALL
  int n;                        // counter for arguments

  callType = isCall(buffer, macroName);
  if ((callType == 1) || (callType == 2))
    {
      callArgs = new stringList;
      if (!findCallArgs(buffer, callArgs, &numberArgs))
	{ // quit if arguments to the call are unreadable
	  fprintf(stderr, "%d: bad call arguments. Quitting.\n",
		  lineNumber);
	  exit(1);
	}
      if ((theMacro = findMacro(macroName)) == 0)
	{ // macro not yet defined in file, so print and store dummy macro
	  if (callType == 1)
	    { // macro was expected to be defined but was not found
	      printf("%d: Error - undefined MACRO called.\n\n",
		      lineNumber);
	      numErrors++;
	    }
	  macroArgs = new stringList;
	  macroText = new stringList;
	  m = sprintf(text, "M(%s)=MACRO", macroName);
	  if (numberArgs > 0)
	    {
	      m = (m + sprintf((text+m), "/"));
	      for (argCell = callArgs->first, n = 0;
		   argCell;
		   argCell = argCell->next, n++)
		{
		  sprintf(argName, "arg%d", n);
		  m = (m + sprintf((text+m), "%s", argName));
		  if (argCell->next)
		    m = (m + sprintf((text+m), ","));
		  macroArgs->pushBack(strdup(argName));
		}
	      
	    }
	  sprintf((text+m), "%c%c", 13, 10);
	  macroText->pushBack(strdup(text));
	  fprintf(out, "%d|$$Next macro is a dummy - qwerty%c%c",
	          lineNumber, 13, 10);
	  fprintf(out, "%d|%s", lineNumber, text);	  
	  sprintf(text, "ENDMAC%c%c", 13, 10);
	  macroText->pushBack(strdup(text));
	  fprintf(out, "%d|%s", lineNumber, text);
	  theMacro = new macroRep(macroText, macroArgs, macroName);
	  macros.pushBack(theMacro);
	}
      fprintf(out, "%d|%s", lineNumber, buffer);
      insertCalledMacro(callArgs, theMacro, numberArgs, lineNumber, out);
      return 1;
    }
  else if (callType == 3)
    {
      fprintf(out, "%d|%s", lineNumber, buffer);
      fprintf(out, "%d|ENDMAC%c%c", lineNumber, 13, 10);
      return 1;
    }
  return 0;
}

/********************************************************************/

/* getStatement

This gets one statement (which may cover several lines) from the
original file and stores it in bigBuf as a single line. If a statement
is obtained, this returns true. If a statement is not obtained (which
happens only when the end of the file has been reached and all
characters before the end have been handled) this returns false.

The inner "for" loop reads characters from a line of the original
input file into the lilBuf and fixes the end of the line if it is
messed up. If the end of the line is messed up, a warning is given and
the end of the line is fixed so it ends correctly with <CR> followed
by <LF>. The end of the line is assumed to be intended and the end
is messed up if any of the following is seen.
1. <CR> appears without <LF> following.
2. <LF> appears without <CR> preceding.
3. EOF appears following a character that is neither <CR> nor <LF>.

The outer "for" loop analyzes the line in lilBuf, and then may copy
some or all of lilBuf onto the end of bigBuf. The outer "for" loop
loops back only if the line in lilBuf is continued and is not followed
by a comment or a blank line.

If a blank line follows a continued line, it ends the continuation.

If a comment line follows a continued line, an error is recorded, and
the continuation is ended.

This is allowing spaces and tabs before the $$ that starts a comment.

This is allowing spaces and tabs after the $ that signifies a
continued line.

If a line is continued, when lilBuf is copied into bigBuf, the $ that
signifies the continuation and anything following it are replaced by
<CR> followed by <LF>. If the next line is a valid continuation (not
a comment or a blank line), the <CR><LF> is overwritten by the first
two characters of the lilBuf.

The lineNumber argument is updated to the line number in the original
file of the last line that was read.

The extras arguments is set to the number of times the statement
stored in bigBuf was continued.

Whenever this returns true, bigBuf ends with <CR><LF>0 .

This checks whether either bigBuf or lilBuf is about to overflow and
exits if so.

*/

bool getStatement( /* ARGUMENTS                            */
 FILE * original,  /* DMIS file being preprocessed         */
 char * bigBuf,    /* statement buffer filled here         */
 int * lineNumber, /* input file line number, updated here */
 int * extras)     /* number of continuations, set here    */
{
  static char lilBuf[65540];    // input line file buffer
  int comment;                  // 1=line is a comment, 0=not
  int continued;                // 1=line continued, 0=not
  int n;                        // counter for characters in lilBuf
  int m;                        // counter for characters in bigBuf
  int k;                        // counter for spaces and tabs
  int aChar;                    // character from input file
  int blank;                    // 1=buffer is all blanks, 0=not

  continued = 0;
  for (m = 0; ; )
    {
      blank = 1;
      for (n = 0; n < 65536; n++)
	{ // read a line of original into lilBuf and check/fix line ending
	  aChar = getc(original);
	  lilBuf[n] = aChar;
	  if (aChar == EOF)
	    {
	      if (n == 0)
		{
		  if (continued)
		    {
		      printf("%d: Error - last line of file "
			     "is continued\n\n", *lineNumber);
		      numErrors++;
		      return true; 
		    }
		  return false; // done preprocessing
		}
	      else
		{
		  printf(
			  "%d: Error - file ends with no <LF><CR>\n\n",
			  (1 + *lineNumber));
		  numErrors++;
		  lilBuf[n++] = 13;
		  lilBuf[n] = 10;
		  break;
		}
	    }
	  else if (aChar == 13)
	    {
	      aChar = getc(original);
	      if (aChar != 10)
		{
		  ungetc(aChar, original); // no harm if aChar is EOF
		  printf(
			  "%d: Error - <LF> is missing after <CR>\n\n",
			  (1 + *lineNumber));
		  numErrors++;
		}
	      n++;
	      lilBuf[n] = 10;
	      break;
	    }
	  else if (aChar == 10)
	    {
	      printf(
		      "%d: Error - <CR> is missing before <LF>\n\n",
		      (1 + *lineNumber));
	      numErrors++;
	      lilBuf[n++] = 13;
	      lilBuf[n] = 10;
	      break;
	    }
	  else if ((aChar != ' ') && (aChar != '\t'))
	    blank = 0;
	}
      // line now ends with <CR><LF>
      // n is now set at the <LF>
      (*lineNumber)++;
      if (n > 65535)
	{
	  fprintf(stderr, "Quitting: line %d too long to handle\n",
		  *lineNumber);
	  exit(1);
	}
      else if (blank)
	{ // lilBuf is blank; bigBuf is ready to go if continued last time
	  if (continued)
	    {
	      return true;
	    }
	}
      else // if (!blank)
	{
	  for (k = 0; ((lilBuf[k] == ' ') || (lilBuf[k] == '\t')); k++);
	  comment = (((lilBuf[k] == '$') && (lilBuf[k+1] == '$')) ? 1 : 0);
	  if (comment)
	    { // lilBuf is comment
	      if (continued)
		{ // if lilBuf continued last time, error & bigBuf ready
		  printf(
		  "%d: Error - comment line follows continued line\n\n",
			  *lineNumber);
		  numErrors++;
		  return true;
		}
	    }
	  else // if (!comment)
	    { // remove <CR><LF> and spaces and tabs at end; reset continued
	      for (n = (n-2); 
		   ((lilBuf[n] == ' ') || (lilBuf[n] == '\t'));
		   n--);
	      if (lilBuf[n] == '$') // line is continued; remove $ at end
		{
		  continued = 1;
		  (*extras)++;
		  n--; 
		}
	      else
		continued = 0;
	      if ((n+m+3) > 65535)
		{
		  fprintf(stderr,
			  "Quitting: line %d too long to handle\n",
			  *lineNumber);
		  exit(1);
		}
	      for (k = 0; k <= n; k++, m++)
		{ // now copy lilBuf onto end of bigBuf
		  bigBuf[m] = lilBuf[k];
		}
	      bigBuf[m] = 13;
	      bigBuf[m+1] = 10;
	      bigBuf[m+2] = 0;
	      if (!continued)
		{ // bigBuf is ready to go if lilBuf not continued
		  return true;
		}
	    }
	}
    }
  return true; // just to make compiler happy; never get here
}

/********************************************************************/

/* preprocess

This reads the input file and prints a preprocessed DMIS input file
in which:
1. Blank lines are removed
2. Continued lines are joined together.
3. The line number is inserted at the beginning.

This also saves a representation of each macro that is used if the
macro is called.

This also changes calls to macros by inserting the text of the macro after
the call, with the macro arguments replaced by the call arguments.

This checks that the last line of the output file starts with ENDFIL
(case insensitive), possibly preceded by spaces and/or tabs. If not,
this prints an error message, adds 1 to numErrors, and adds an ENDFIL
line at the end of the file. This is necessary so that the parser
does not need to look for an unexpected file end everywhere one might
occur.

If any of the following errors occurs, this prints an error message
and quits.
1. The named input file cannot be opened.
2. The output file cannot be opened.

The getStatement function (1) sets lineNumber to the number of the line
from the original file last read and (2) sets extras to the number of
times the statement it reads was continued. In the output file, the
statement is assigned the number of the line on which it started, which
is (lineNumber - extras). The getStatement function returns false only
after all characters in the original file have been processed.

*/

void preprocess(
 char * fileNameIn)
{
  FILE * original;            // input file pointer
  FILE * out;                 // output file pointer
  static char buffer[65536];  // input file buffer
  int lineNumber;             // line number of input file
  int extras;                 // number of continuations
  int n;                      // index for buffer

  original = fopen(fileNameIn, "rb");
  if (original == 0)
    {
      fprintf(stderr,
	      "could not open file %s for reading\n", fileNameIn);
      exit(1);
    }
  out = fopen(PRENAME, "wb");
  if (out == 0)
    {
      fprintf(stderr, "could not open file %s for writing\n", PRENAME);
      exit(1);
    }
  lineNumber = 0;
  buffer[0] = 0;
  for (extras = 0; getStatement(original, buffer, &lineNumber, &extras); )
    {
      if (doCall(buffer, (lineNumber - extras), out));
      else
	{
	  fprintf(out, "%d|%s", (lineNumber - extras), buffer);
	  doMacro(buffer, (lineNumber - extras));
	}
      extras = 0;
    }
  for (n = 0; ((buffer[n] == ' ') || (buffer[n] == '\t')); n++);
  if ((strlen(buffer) < 7) ||
      ((buffer[n  ] != 'E') && (buffer[n  ] != 'e')) ||
      ((buffer[n+1] != 'N') && (buffer[n+1] != 'n')) ||
      ((buffer[n+2] != 'D') && (buffer[n+2] != 'd')) ||
      ((buffer[n+3] != 'F') && (buffer[n+3] != 'f')) ||
      ((buffer[n+4] != 'I') && (buffer[n+4] != 'i')) ||
      ((buffer[n+5] != 'L') && (buffer[n+5] != 'l')) ||
      ((buffer[n+6] >= 'A') && (buffer[n+6] <= 'Z')) ||
      ((buffer[n+6] >= 'a') && (buffer[n+6] <= 'z')) ||
      ((buffer[n+6] >= '0') && (buffer[n+6] <= '9')))
    {
      printf("Error - file does not end with ENDFIL\n\n");
      numErrors++;
      fprintf(out, "%d|ENDFIL%c%c", (lineNumber+1), 13, 10);
    } 
  fclose(original);
  fclose(out);
}

/********************************************************************/

void warn(
 char * s)
{
  int n;

  printf("%d Warning: %s\n", lineNo, s);
  printf("%s\n", lineText);
  for (n = 0; lineText[n] != 0; n++);
  if (lineText[n-1] != '\n')
    printf("\n");
  numWarnings++;
}

/********************************************************************/

/* yyerror

If lexMessage contains a message, lexMessage[0] will not be 0,
and a syntax error will have occurred, since BAD will have been
returned by the lexer. In this case, only the lexMessage is
printed, since reporting the syntax error (s) only adds confusion.

*/

int yyerror(char * s)
{
  int n;

  if (lexMessage[0])
    {
      printf("%d: %s\n", lineNo, lexMessage);
      lexMessage[0] = 0;
    }
  else
    printf("%d: %s\n", lineNo, s);
  printf("%s\n", lineText);
  for (n = 0; lineText[n] != 0; n++);
  if (lineText[n-1] != '\n')
    printf("\n");
  return 0;
}

/********************************************************************/

int yyparse();


/********************************************************************/

/* doLabel

This is called by the actions for productions that define labels. The
general idea is that when a label has been read, we need to determine
if the label has already been defined. In some cases it is an error
if the label is not defined. In other cases it is an error if the
label is defined. The type of the label is in the labelType argument,
and the name of the label is in the labelName argument.

If a label error has already occurred while reading a DMIS statement,
that is indicated by aLabelFound being 0. In this case we want to
remember the type and name of the label that caused the error so it
can be reported after reading the statement is finished.

If the global variable aLabelFound (0 or 1) has been set and the
labelName argument is not NULL, then the global variable aLabelType is
set to the labelType argument and the global variable aLabel is set to
the labelName argument.  Then aLabelFound is reset to to indicate
whether a label of the given labelType and labelName has already been
recorded.

*/

void doLabel(
  int labelType,
  char * labelName)
{
  if (aLabelFound && labelName)
    {
      aLabelType = labelType;
      aLabel = labelName;
      aLabelFound = labels[labelType].find(labelName);
    }
}

/********************************************************************/

/* handleLabel

When the DMIS EXIST function is called, its argument is a label. Call
that label L.  We do not want want to signal an error when L is read,
regardless of whether or not L exists. However, reading L resets
aLabel, aLabelType, and aLabelFound.  The job of the handleLabel is to
save the current values of those global variables when the left
parenthesis before L is read (the save argument is 1 in this case) and
to restore their values when the right parenthesis following L is read
(the save argument is 0 in this case). Static local variables are used
to do the saving.

*/

void handleLabel(
 int save)
{
  static int saveLabelFound;
  static char * saveLabel;
  static int saveLabelType;

  if (save)
    {
      saveLabelFound = aLabelFound;
      saveLabel = aLabel;
      saveLabelType = aLabelType;
    }
  else
    {
      aLabelFound = saveLabelFound;
      aLabel = saveLabel;
      aLabelType = saveLabelType;
    }
}

/********************************************************************/

/* resetParser

This resets the parser by clearing out stored labels and by
resetting several variables used by the parser.
It also clears out saved macros.

*/

void resetParser()
{
  int n;
  stringListCell * aCell;
  stringListCell * lastCell;
  macroListCell * macroCell;
  macroListCell * prevCell;

  numErrors = 0;
  numWarnings = 0;
  for (n = 0; n < 1000; n++)
    {
      if (labels[n].theString)
	{
	  delete labels[n].theString;
	  labels[n].theString = 0;
	  for (aCell = labels[n].next; aCell; )
	    {
	      delete aCell->theString;
	      lastCell = aCell;
	      aCell = aCell->next;
	      delete lastCell;
	    }
	  labels[n].next = 0;
	}
    }
  for (macroCell = macros.first; macroCell; )
    {
      delete macroCell->theMacro;
      prevCell = macroCell;
      macroCell = macroCell->next;
      delete prevCell;
    }
  macros.first = 0;
  macros.last = 0;
  doMacro(0,0);
  resetLex = 1;
}

/********************************************************************/

/* parseDmis

This:
preprocesses the input file,
opens the preprocessed input file,
parses the preprocessed input file,
closes the preprocessed input file,
removes the preprocessed input file.

*/

void parseDmis(
 char * fileName)
{
  lexMessage[0] = 0;
  lexWarning[0] = 0;
  preprocess(fileName);
  yyin = fopen(PRENAME, "rb");
  if (yyin == 0)
    {
      fprintf(stderr, "unable to open file %s for reading\n", PRENAME);
      exit(1);
    }
  yyparse();
  fclose(yyin);
  remove(PRENAME);
}

/********************************************************************/

%}

%union {
  inputFile                        * val1;
  dmisItem                         * val2;
  std::list<dmisItem *>            * val3;
  dmisFirstStatement               * val4;
  dmisBlock                        * val5;
  calibMasterBlock                 * val6;
  calibRtabBlock                   * val7;
  calibSensBlock                   * val8;
  std::list<calibSensBlockItem *>  * val9;
  calibSensBlockStatement          * val10;
  calibSensBlockItem               * val11;
  calibSensDoBlock                 * val12;
  calibSensIfBlock                 * val13;
  calibSensSelectBlock             * val14;
  std::list<calibSensCaseBlock *>  * val15;
  calibSensCaseBlock               * val16;
  calibSensDefaultCaseBlock        * val17;
  callBlock                        * val18;
  caseBlock                        * val19;
  std::list<caseStm *>             * val20;
  defaultCaseBlock                 * val21;
  dmisOffBlock                     * val22;
  doBlock                          * val23;
  gotargBlock                      * val24;
  std::list<gotoStm *>             * val25;
  ifBlock                          * val26;
  macroBlock                       * val27;
  measBlock                        * val28;
  std::list<measBlockItem *>       * val29;
  measBlockStatement               * val30;
  measBlockItem                    * val31;
  measDoBlock                      * val32;
  measIfBlock                      * val33;
  measSelectBlock                  * val34;
  std::list<measCaseBlock *>       * val35;
  measCaseBlock                    * val36;
  measDefaultCaseBlock             * val37;
  selectBlock                      * val38;
  std::list<caseBlock *>           * val39;
  simreqtBlock                     * val40;
  std::list<evalOrOutput *>        * val41;
  evalOrOutput                     * val42;
  xternBlock                       * val43;
  std::list<extfilStm *>           * val44;
  dmisFreeStatement                * val45;
  anyVal                           * val46;
  stringVal                        * val47;
  boolVal                          * val48;
  intVal                           * val49;
  realVal                          * val50;
  rentVal                          * val51;
  vectorVal                        * val52;
  stringConst                      * val53;
  boolConst                        * val54;
  intConst                         * val55;
  realConst                        * val56;
  stringFunc                       * val57;
  boolFunc                         * val58;
  intFunc                          * val59;
  realFunc                         * val60;
  vectorFunc                       * val61;
  boolExpr                         * val62;
  stringRel                        * val63;
  boolRel                          * val64;
  intRel                           * val65;
  intExpr                          * val66;
  realRel                          * val67;
  realRealRel                      * val68;
  intRealRel                       * val69;
  realIntRel                       * val70;
  vectorRel                        * val71;
  realExpr                         * val72;
  realExprParen                    * val73;
  realExprNegat                    * val74;
  realExprPosit                    * val75;
  realRealExpr                     * val76;
  realIntExpr                      * val77;
  intRealExpr                      * val78;
  vectorExpr                       * val79;
  boolVar                          * val80;
  intVar                           * val81;
  realVar                          * val82;
  stringVar                        * val83;
  vectorVar                        * val84;
  arrayIndex                       * val85;
  stringFuncChr                    * val86;
  stringFuncConcat                 * val87;
  stringFuncElemnt                 * val88;
  stringFuncScfeat                 * val89;
  stringFuncScsns                  * val90;
  stringFuncSdate                  * val91;
  stringFuncSerror                 * val92;
  stringFuncSmode                  * val93;
  stringFuncStime                  * val94;
  stringFuncSdatetime              * val95;
  stringFuncRpt                    * val96;
  stringFuncSelapsetime            * val97;
  longShort                        * val98;
  stringFuncStr                    * val99;
  strVar7                          * val100;
  stringFuncLwc                    * val101;
  stringFuncTrim                   * val102;
  stringFuncUpc                    * val103;
  stringFuncSubstr                 * val104;
  boolFuncBadgt                    * val105;
  boolFuncExist                    * val106;
  boolFuncBadpt                    * val107;
  boolFuncEof                      * val108;
  boolFuncEoln                     * val109;
  intFuncAbs                       * val110;
  intFuncIndx                      * val111;
  intFuncInt                       * val112;
  intFuncLen                       * val113;
  intFuncMod                       * val114;
  intFuncNint                      * val115;
  intFuncOrd                       * val116;
  intFuncPtdata                    * val117;
  intFuncSensnotouch               * val118;
  intFuncSign                      * val119;
  intFuncSiltch                    * val120;
  realFuncAbs                      * val121;
  realFuncCos                      * val122;
  realFuncDble                     * val123;
  realFuncDtor                     * val124;
  realFuncExp                      * val125;
  realFuncLn                       * val126;
  realFuncLog                      * val127;
  realFuncMn                       * val128;
  realFuncMod                      * val129;
  realFuncMx                       * val130;
  realFuncQtemp                    * val131;
  realFuncRand                     * val132;
  realFuncRl                       * val133;
  realFuncRtod                     * val134;
  realFuncSign                     * val135;
  realFuncSin                      * val136;
  realFuncSqrt                     * val137;
  realFuncTan                      * val138;
  realFuncVal                      * val139;
  realFuncVdot                     * val140;
  realFuncVecX                     * val141;
  realFuncVecY                     * val142;
  realFuncVecZ                     * val143;
  realFuncVmag                     * val144;
  realFuncAcos                     * val145;
  realFuncAsin                     * val146;
  realFuncAtan                     * val147;
  realFuncAtan2                    * val148;
  vectorFuncVcart                  * val149;
  vectorFuncVcross                 * val150;
  vectorFuncVmcs                   * val151;
  vectorFuncVpcs                   * val152;
  vectorFuncVpol                   * val153;
  vectorFuncVunit                  * val154;
  labelNameConst                   * val155;
  labelName                        * val156;
  datumLabel                       * val157;
  datumLabel1                      * val158;
  datumLabel2                      * val159;
  featureLabel                     * val160;
  featureLabel1                    * val161;
  qisLabel                         * val162;
  sensorLabel1                     * val163;
  sensorLabel2                     * val164;
  toleranceLabel                   * val165;
  ccLabel                          * val166;
  ciLabel                          * val167;
  crLabel                          * val168;
  crLabelConst                     * val169;
  csLabel                          * val170;
  czLabel                          * val171;
  dLabel                           * val172;
  daLabel                          * val173;
  datLabel                         * val174;
  dattrgLabel                      * val175;
  diLabel                          * val176;
  didLabel                         * val177;
  drLabel                          * val178;
  dsLabel                          * val179;
  dvLabel                          * val180;
  fLabel                           * val181;
  faLabel                          * val182;
  fiLabel                          * val183;
  fsLabel                          * val184;
  gLabel                           * val185;
  gsaLabel                         * val186;
  kcLabel                          * val187;
  kcaLabel                         * val188;
  liLabel                          * val189;
  mLabel                           * val190;
  mLabelConst                      * val191;
  maLabel                          * val192;
  mdLabel                          * val193;
  opLabel                          * val194;
  pLabel                           * val195;
  pcLabel                          * val196;
  plLabel                          * val197;
  pnLabel                          * val198;
  prLabel                          * val199;
  psLabel                          * val200;
  pvLabel                          * val201;
  qLabel                           * val202;
  rLabel                           * val203;
  rmLabel                          * val204;
  rtLabel                          * val205;
  sLabel                           * val206;
  saLabel                          * val207;
  seLabel                          * val208;
  sgLabel                          * val209;
  sgsLabel                         * val210;
  srLabel                          * val211;
  sraLabel                         * val212;
  ssLabel                          * val213;
  stLabel                          * val214;
  swLabel                          * val215;
  sxLabel                          * val216;
  tLabel                           * val217;
  taLabel                          * val218;
  thLabel                          * val219;
  tlLabel                          * val220;
  uLabel                           * val221;
  vLabel                           * val222;
  vaLabel                          * val223;
  vfLabel                          * val224;
  vlLabel                          * val225;
  vwLabel                          * val226;
  jumpLabel                        * val227;
  param                            * val228;
  std::list<param *>               * val229;
  std::list<stringVal *>           * val230;
  std::list<intVal *>              * val231;
  leftRight                        * val232;
  std::list<rentVal *>             * val233;
  std::list<featureLabel *>        * val234;
  std::list<fLabel *>              * val235;
  std::list<faLabel *>             * val236;
  std::list<noParseStm *>          * val237;
  std::list<dattrgLabel *>         * val238;
  std::list<saLabel *>             * val239;
  std::list<stackElement *>        * val240;
  std::list<tLabel *>              * val241;
  std::list<taLabel *>             * val242;
  pointRange                       * val243;
  indexedFeature                   * val244;
  std::list<indexedFeature *>      * val245;
  versionTag                       * val246;
  angle                            * val247;
  angleDms                         * val248;
  noParseStm                       * val249;
  secondsString                    * val250;
  vector                           * val251;
  matrix                           * val252;
  rwVar                            * val253;
  rwFormat                         * val254;
  device                           * val255;
  state                            * val256;
  coordType                        * val257;
  cartPoint                        * val258;
  impCartPoint                     * val259;
  std::list<impCartPoint *>        * val260;
  polPoint                         * val261;
  impPolPoint                      * val262;
  std::list<impPolPoint *>         * val263;
  typePoint                        * val264;
  pointVecCart                     * val265;
  std::list<pointVecCart *>        * val266;
  pointVecPol                      * val267;
  std::list<pointVecPol *>         * val268;
  axis                             * val269;
  orig                             * val270;
  dir                              * val271;
  posDir                           * val272;
  negDir                           * val273;
  plan                             * val274;
  matDir                           * val275;
  flatRoundOpen                    * val276;
  radiusSpec                       * val277;
  stackElement                     * val278;
  rotType                          * val279;
  rotDir                           * val280;
  rotOrient                        * val281;
  rotAbs                           * val282;
  rotIncr                          * val283;
  jumpStm                          * val284;
  aclratStm                        * val285;
  aclratMinor                      * val286;
  aclratMeas                       * val287;
  aclratPos                        * val288;
  aclratRot                        * val289;
  aclratScan                       * val290;
  aclratHedRot                     * val291;
  aclratHedMeas                    * val292;
  aclratHedScan                    * val293;
  aclratAngSpec                    * val294;
  aclratLinSpec                    * val295;
  aclratLinear                     * val296;
  aclratAngular                    * val297;
  aclratDef                        * val298;
  algdefStm                        * val299;
  algdefMinor                      * val300;
  assignStm                        * val301;
  assignString                     * val302;
  assignBool                       * val303;
  assignInt                        * val304;
  assignReal                       * val305;
  assignVector                     * val306;
  badtstStm                        * val307;
  boundStm                         * val308;
  boundMinor                       * val309;
  boundFeat                        * val310;
  boundTol                         * val311;
  calibSensStm                     * val312;
  calibRtabStm                     * val313;
  calibMasterStm                   * val314;
  calibSensMinor                   * val315;
  calibSensSpec                    * val316;
  calibRtabMinor                   * val317;
  calibRtabSpec                    * val318;
  calibMasterMinor                 * val319;
  callStm                          * val320;
  callMinor                        * val321;
  callMacro                        * val322;
  callModule                       * val323;
  callRoutine                      * val324;
  callProgram                      * val325;
  callType                         * val326;
  caseStm                          * val327;
  plusSign                         * val328;
  minusSign                        * val329;
  sign                             * val330;
  clmpidStm                        * val331;
  clmpsnStm                        * val332;
  closeStm                         * val333;
  closeMinor                       * val334;
  cmpntgrpStm                      * val335;
  std::list<cmpntgrpSpec *>        * val336;
  cmpntgrpSpec                     * val337;
  cnfrmrulStm                      * val338;
  cnfrmrulMinor                    * val339;
  constStm                         * val340;
  constMinor                       * val341;
  constArc                         * val342;
  constCircle                      * val343;
  constCompound                    * val344;
  constCone                        * val345;
  constCparln                      * val346;
  constCylndr                      * val347;
  constEllips                      * val348;
  constEdgept                      * val349;
  constGeom                        * val350;
  constGcurve                      * val351;
  constGsurf                       * val352;
  constLine                        * val353;
  constParpln                      * val354;
  constPatern                      * val355;
  constPlane                       * val356;
  constPoint                       * val357;
  retrieve1                        * val358;
  retrieve2                        * val359;
  retrieve4                        * val360;
  retrieve2b                       * val361;
  constRctngl                      * val362;
  constSgage                       * val363;
  constSpart                       * val364;
  constSphere                      * val365;
  constSympln                      * val366;
  constTorus                       * val367;
  bfConst                          * val368;
  cogConst                         * val369;
  coneConst                        * val370;
  curveConst                       * val371;
  extremConst                      * val372;
  extremConstDir                   * val373;
  extremConstAxial                 * val374;
  extremConstVectorial             * val375;
  extremConstFeature               * val376;
  extremConstRadial                * val377;
  intofConst                       * val378;
  midliConst                       * val379;
  midplConst                       * val380;
  midptConst                       * val381;
  moveptConst                      * val382;
  nearptConst                      * val383;
  offsetConst                      * val384;
  partoConst                       * val385;
  perptoConst                      * val386;
  pierceConst                      * val387;
  projctConst                      * val388;
  projliConst                      * val389;
  projptConst                      * val390;
  sgageConst                       * val391;
  spartConst                       * val392;
  tantoConst                       * val393;
  tantoConstPlane                  * val394;
  trConst                          * val395;
  vertexConst                      * val396;
  crgdefStm                        * val397;
  crgdefMinor                      * val398;
  crmodeStm                        * val399;
  crmodeMinor                      * val400;
  crosclStm                        * val401;
  crslctStm                        * val402;
  crslctMinor                      * val403;
  cutcomStm                        * val404;
  cutcomMinor                      * val405;
  cutcomAdjust                     * val406;
  cutcomParam                      * val407;
  cutcomMatrix                     * val408;
  cutcomMatx                       * val409;
  cutcomUserdf                     * val410;
  czoneStm                         * val411;
  czslctStm                        * val412;
  datdefStm                        * val413;
  datdefMinor                      * val414;
  datsetStm                        * val415;
  datsetMinor                      * val416;
  datsetMcs                        * val417;
  datsetMatrix                     * val418;
  datsetDats                       * val419;
  datsetDrf                        * val420;
  tier                             * val421;
  datsetSpec                       * val422;
  dattrgdefStm                     * val423;
  dattrgMinor                      * val424;
  declStm                          * val425;
  declMinor                        * val426;
  std::list<declVar *>             * val427;
  declScope                        * val428;
  declType                         * val429;
  declVar                          * val430;
  std::list<intConst *>            * val431;
  decplStm                         * val432;
  decplMinor                       * val433;
  decplAll                         * val434;
  std::list<decplSelection *>      * val435;
  decplNdigits                     * val436;
  decplSelection                   * val437;
  deleteStm                        * val438;
  deleteMinor                      * val439;
  deleteDatum                      * val440;
  deleteSensor                     * val441;
  deleteFeature                    * val442;
  deleteRotaryTable                * val443;
  deleteAllSensors                 * val444;
  deviceStm                        * val445;
  deviceMinor                      * val446;
  dftcasStm                        * val447;
  displyStm                        * val448;
  displyOff                        * val449;
  std::list<displySpecItem *>      * val450;
  dmehwStm                         * val451;
  dmehwMinor                       * val452;
  jointConf                        * val453;
  rightyLefty                      * val454;
  aboveBelow                       * val455;
  flipNoFlip                       * val456;
  dmeidStm                         * val457;
  dmeswStm                         * val458;
  dmeswMinor                       * val459;
  dmeswComand                      * val460;
  dmeswDelay                       * val461;
  dmeswSwitch                      * val462;
  dmeswiStm                        * val463;
  dmeswvStm                        * val464;
  dmisOffStm                       * val465;
  dmisOnStm                        * val466;
  dmismdStm                        * val467;
  dmismnStm                        * val468;
  std::list<conformItem *>         * val469;
  conformItem                      * val470;
  conformType                      * val471;
  doStm                            * val472;
  elseStm                          * val473;
  endcasStm                        * val474;
  enddoStm                         * val475;
  endfilStm                        * val476;
  endgoStm                         * val477;
  endifStm                         * val478;
  endmacStm                        * val479;
  endmesStm                        * val480;
  endselStm                        * val481;
  endsimreqtStm                    * val482;
  endxtnStm                        * val483;
  equateStm                        * val484;
  equateMinor                      * val485;
  equateCadcs                      * val486;
  errorStm                         * val487;
  errorMinor                       * val488;
  errorCondition                   * val489;
  evalStm                          * val490;
  evalMinor                        * val491;
  evalFeat                         * val492;
  evalDat                          * val493;
  evalKeychar                      * val494;
  extensStm                        * val495;
  extensMinor                      * val496;
  extfilStm                        * val497;
  extfilMinor                      * val498;
  extfilDmis                       * val499;
  extfilDme                        * val500;
  featArc1Stm                      * val501;
  featArc2Stm                      * val502;
  featCircleStm                    * val503;
  featCompoundStm                  * val504;
  featConeStm                      * val505;
  featConradsegStm                 * val506;
  featCparlnStm                    * val507;
  featCylndrStm                    * val508;
  featCylradsegStm                 * val509;
  featEdgeptStm                    * val510;
  featEllipsStm                    * val511;
  featElongcylStm                  * val512;
  featGcurveStm                    * val513;
  featGeomStm                      * val514;
  featGsurfStm                     * val515;
  featLineStm                      * val516;
  featObjectStm                    * val517;
  featParplnStm                    * val518;
  featPaternStm                    * val519;
  featPlaneStm                     * val520;
  featPointStm                     * val521;
  featRctnglStm                    * val522;
  featRevsurfStm                   * val523;
  featSphereStm                    * val524;
  featSphradsegStm                 * val525;
  featSymplnStm                    * val526;
  featTorradsegStm                 * val527;
  featTorusStm                     * val528;
  featMinorArc1                    * val529;
  featMinorArc2                    * val530;
  featMinorCircle                  * val531;
  featMinorCompound                * val532;
  featMinorCone                    * val533;
  featMinorConradseg               * val534;
  featMinorCparln                  * val535;
  featMinorCylndr                  * val536;
  featMinorCylradseg               * val537;
  featMinorEdgept                  * val538;
  featMinorEllips                  * val539;
  featMinorElongcyl                * val540;
  featMinorGeom                    * val541;
  featMinorGcurve                  * val542;
  featMinorGsurf                   * val543;
  featMinorLine                    * val544;
  featMinorObject                  * val545;
  featMinorParpln                  * val546;
  featMinorPatern                  * val547;
  featMinorPlane                   * val548;
  featMinorPoint                   * val549;
  featMinorRctngl                  * val550;
  featMinorRevsurf                 * val551;
  featMinorSphere                  * val552;
  featMinorSphradseg               * val553;
  featMinorSympln                  * val554;
  featMinorTorradseg               * val555;
  featMinorTorus                   * val556;
  fedratStm                        * val557;
  fedratMinor                      * val558;
  fedratMeas                       * val559;
  fedratPos                        * val560;
  fedratRot                        * val561;
  fedratScan                       * val562;
  fedratHedRot                     * val563;
  fedratHedMeas                    * val564;
  fedratHedScan                    * val565;
  fedratAngSpec                    * val566;
  fedratLinSpec                    * val567;
  fedratLinear                     * val568;
  fedratAngular                    * val569;
  fedratDef                        * val570;
  fildefStm                        * val571;
  filnamStm                        * val572;
  finposStm                        * val573;
  fixtidStm                        * val574;
  fixtsnStm                        * val575;
  flyStm                           * val576;
  flyMinor                         * val577;
  fromStm                          * val578;
  fromMinor                        * val579;
  geoalgStm                        * val580;
  geoalgMinor                      * val581;
  geoalgArc                        * val582;
  geoalgCircle                     * val583;
  geoalgCone                       * val584;
  geoalgConradsegmnt               * val585;
  geoalgCparln                     * val586;
  geoalgCylndr                     * val587;
  geoalgCylradsegmnt               * val588;
  geoalgEllips                     * val589;
  geoalgElongcyl                   * val590;
  geoalgGcurve                     * val591;
  geoalgGsurf                      * val592;
  geoalgLine                       * val593;
  geoalgObject                     * val594;
  geoalgParpln                     * val595;
  geoalgPlane                      * val596;
  geoalgRctngl                     * val597;
  geoalgRevsurf                    * val598;
  geoalgSphere                     * val599;
  geoalgSphradsegmnt               * val600;
  geoalgSympln                     * val601;
  geoalgTorus                      * val602;
  geoalgTorradsegmnt               * val603;
  geoalgSpec1                      * val604;
  geoalgSpec2                      * val605;
  geoalgSpec3                      * val606;
  geoalgSpec4                      * val607;
  geoalgSpec5                      * val608;
  geoalgSpec6                      * val609;
  geoalgExternFunc                 * val610;
  geoalgFilterSettings             * val611;
  geoalgEliminate                  * val612;
  geoalgFilter                     * val613;
  geoalgFilterType                 * val614;
  geoalgFilterCurve                * val615;
  geomStm                          * val616;
  geomMinor                        * val617;
  std::list<geomEntityItem *>      * val618;
  gohomeStm                        * val619;
  gotargStm                        * val620;
  gotoStm                          * val621;
  gotoMinor                        * val622;
  gotoAxis                         * val623;
  gotoAxisSpec                     * val624;
  gotoAbs                          * val625;
  sensorMove                       * val626;
  gotoRel                          * val627;
  gotoArc                          * val628;
  probeOrient                      * val629;
  std::list<gotoWristItem *>       * val630;
  std::list<gotoWristAngleItem *>  * val631;
  gotoWristAngleItem               * val632;
  groupStm                         * val633;
  ifStm                            * val634;
  includStm                        * val635;
  includMinor                      * val636;
  includDmis                       * val637;
  includDme                        * val638;
  iteratStm                        * val639;
  iteratMinor                      * val640;
  iteratConvergence                * val641;
  std::list<iteratCriterionItem *> * val642;
  iteratCriterionStart             * val643;
  jumptoStm                        * val644;
  keycharStm                       * val645;
  keycharMinor                     * val646;
  keycharOneFeature                * val647;
  keycharTwoFeatures               * val648;
  criticality                      * val649;
  litdefStm                        * val650;
  litdefMinor                      * val651;
  litdefLight                      * val652;
  litdefLightType                  * val653;
  litdefStrobe                     * val654;
  litdefStrobeType                 * val655;
  locateStm                        * val656;
  locateTransAllowed               * val657;
  locateRotAllowed                 * val658;
  locateLabel                      * val659;
  std::list<locateLabel *>         * val660;
  lotidStm                         * val661;
  macroStm                         * val662;
  std::list<macroPar *>            * val663;
  macroPar                         * val664;
  matdefStm                        * val665;
  matdefMinor                      * val666;
  matdefFeat                       * val667;
  matdefGeom                       * val668;
  matdefSpec                       * val669;
  std::list<faLabel *>             * val670;
  matdefType                       * val671;
  matdefMating                     * val672;
  matdefMethod                     * val673;
  matdefMat                        * val674;
  measStm                          * val675;
  measMinor                        * val676;
  measGeotype                      * val677;
  measVar2                         * val678;
  measVar2Aux                      * val679;
  mfgdevStm                        * val680;
  modeStm                          * val681;
  obtainStm                        * val682;
  obtainLabeled                    * val683;
  obtainPoint                      * val684;
  openStm                          * val685;
  openMinor                        * val686;
  openDevice                       * val687;
  inputAccess                      * val688;
  outputAccess                     * val689;
  openFdata                        * val690;
  openCadfile                      * val691;
  cadfileType                      * val692;
  std::list<storageSpec *>         * val693;
  storageSpec                      * val694;
  openDMLfile                      * val695;
  operidStm                        * val696;
  outputStm                        * val697;
  outputMinor                      * val698;
  outputFeatData                   * val699;
  outputFeat                       * val700;
  outputDat                        * val701;
  outputReportData                 * val702;
  outputConstData                  * val703;
  outputKeycharData                * val704;
  outputToleranceData              * val705;
  outputSensorData                 * val706;
  outputSensorDesc                 * val707;
  std::list<outputSensorWristItem *> * val708;
  std::list<outputSensorWristAngle *> * val709;
  outputSensorWristAngle           * val710;
  pameasStm                        * val711;
  pameasMinor                      * val712;
  pameasDetail                     * val713;
  std::list<pameasVar2ListItem *>  * val714;
  std::list<pameasVar3ListItem *>  * val715;
  pameasRemove                     * val716;
  pameasFedrat                     * val717;
  partidStm                        * val718;
  partrvStm                        * val719;
  partsnStm                        * val720;
  pathStm                          * val721;
  pathMinor                        * val722;
  euler                            * val723;
  pathPoint                        * val724;
  pathArc                          * val725;
  pathCurve                        * val726;
  pathCurvePtdata                  * val727;
  curvePtdataPcs                   * val728;
  std::list<curvePtdataPcsListItem *> * val729;
  curvePtdataHeadcs                * val730;
  std::list<curvePtdataHeadcsListItem *> * val731;
  pathCurvePoints                  * val732;
  pathHelical                      * val733;
  pathLine                         * val734;
  csSpec                           * val735;
  pathSurface                      * val736;
  std::list<surfPtdataListItem *>  * val737;
  surfPtdataListItem               * val738;
  std::list<surfPtdataPcsListItem *> * val739;
  std::list<surfPtdataHeadcsListItem *> * val740;
  pathUnknown                      * val741;
  planidStm                        * val742;
  popStm                           * val743;
  prcompStm                        * val744;
  prevopStm                        * val745;
  procidStm                        * val746;
  promptStm                        * val747;
  promptIntEnd                     * val748;
  std::list<promptItem *>          * val749;
  promptVar                        * val750;
  psthruStm                        * val751;
  psthruMinor                      * val752;
  ptbuffStm                        * val753;
  ptmeasStm                        * val754;
  ptmeasEnd                        * val755;
  touchSpec                        * val756;
  pushStm                          * val757;
  qisdefStm                        * val758;
  rapidStm                         * val759;
  readStm                          * val760;
  std::list<readSpec *>            * val761;
  readSpec                         * val762;
  recallDatumStm                   * val763;
  recallSensorStm                  * val764;
  recallFeatureStm                 * val765;
  recallRotaryTableStm             * val766;
  recallDMLStm                     * val767;
  refmntStm                        * val768;
  reportStm                        * val769;
  std::list<reportItem *>          * val770;
  reportItem                       * val771;
  resumeStm                        * val772;
  resumeMinor                      * val773;
  rmeasStm                         * val774;
  rmeasMinor                       * val775;
  rmeasArc                         * val776;
  rmeasCircle                      * val777;
  rmeasCone                        * val778;
  rmeasCparln                      * val779;
  rmeasCylndr                      * val780;
  rmeasEdgept                      * val781;
  rmeasEllips                      * val782;
  rmeasGcurve                      * val783;
  rmeasGsurf                       * val784;
  rmeasLine                        * val785;
  rmeasObject                      * val786;
  rmeasParpln                      * val787;
  rmeasPlane                       * val788;
  rmeasPoint                       * val789;
  rmeasRctngl                      * val790;
  rmeasSphere                      * val791;
  rmeasSympln                      * val792;
  rmeasTorus                       * val793;
  rmeasSpecFa                      * val794;
  rmeasSpecFaAxis                  * val795;
  rmeasSpecFaOrient                * val796;
  rmeasSpecFeatAxis                * val797;
  rmeasSpecVecbld                  * val798;
  rmeasSpecVecbldOrient            * val799;
  rmeasSpecVecbldEdgept            * val800;
  rotabStm                         * val801;
  rotabMinor                       * val802;
  rotateStm                        * val803;
  rotateMinor                      * val804;
  rotateValue                      * val805;
  rotateFeature                    * val806;
  rotateDatum                      * val807;
  rotdefStm                        * val808;
  rotsetStm                        * val809;
  saveDatumStm                     * val810;
  saveSensorStm                    * val811;
  saveFeatureStm                   * val812;
  saveRotaryTableStm               * val813;
  saveDMLStm                       * val814;
  scnmodStm                        * val815;
  scnsetStm                        * val816;
  scnsetMinor                      * val817;
  scnsetPeck                       * val818;
  scnsetDrag                       * val819;
  scnsetNoncon                     * val820;
  scnsetStop                       * val821;
  scnsetStopPlane                  * val822;
  scnsetStopSphere                 * val823;
  scnsetVendor                     * val824;
  scnsetSample                     * val825;
  scnsetSampleDrag                 * val826;
  forceOrDeflection                * val827;
  selectStm                        * val828;
  sensorStm                        * val829;
  sensorMinor                      * val830;
  sensorProbe                      * val831;
  sensorMltprb                     * val832;
  std::list<sensorMltprbItem *>    * val833;
  sensorVideo                      * val834;
  sensorLaser                      * val835;
  sensorInfred                     * val836;
  sensorNoncon                     * val837;
  sensorPoint                      * val838;
  sensorLine                       * val839;
  sensorArea                       * val840;
  sensorProbeGeometry              * val841;
  sensorProbeForm                  * val842;
  sensorItem                       * val843;
  sensorMltprbItem                 * val844;
  sensorGeometry                   * val845;
  simreqtStm                       * val846;
  simreqtMinor                     * val847;
  snsdefStm                        * val848;
  snsdefMinor                      * val849;
  snsdefProbe                      * val850;
  probeLocation                    * val851;
  snsdefProbeForm                  * val852;
  snsdefVideo                      * val853;
  snsdefLaser                      * val854;
  snsdefInfred                     * val855;
  snsdefNoncon                     * val856;
  snsdefLocation                   * val857;
  snsdefType                       * val858;
  snsdefBuildSensor                * val859;
  snsdefBuild                      * val860;
  std::list<snsdefBuildItem *>     * val861;
  std::list<snsdefWristAngleItem *> * val862;
  snsdefWristAngleItem             * val863;
  snsetStm                         * val864;
  std::list<snsetItem *>           * val865;
  snsetTypeAndValue                * val866;
  snsetType                        * val867;
  snsetToggle                      * val868;
  snsetLabelAndValue               * val869;
  snsetLabel                       * val870;
  snsetFeat                        * val871;
  snsgrpStm                        * val872;
  snsgrpMinor                      * val873;
  std::list<snsgrpItem *>          * val874;
  snsgrpItem                       * val875;
  snslctStm                        * val876;
  snslctGroup                      * val877;
  snslctSensor                     * val878;
  snslctTipData                    * val879;
  std::list<sensorLabel2 *>        * val880;
  snslctData                       * val881;
  std::list<snslctWristItem *>     * val882;
  std::list<snslctWristAngleItem *> * val883;
  snslctWristAngleItem             * val884;
  snsmntStm                        * val885;
  snsmntMinor                      * val886;
  tecompStm                        * val887;
  tecompMinor                      * val888;
  tecompVar2                       * val889;
  textStm                          * val890;
  textMinor                        * val891;
  textMan                          * val892;
  textOper                         * val893;
  textOutfil                       * val894;
  textQuery                        * val895;
  textQueryFormat                  * val896;
  textType                         * val897;
  thldefStm                        * val898;
  std::list<thldefPocket *>        * val899;
  thldefPocket                     * val900;
  tolAnglStm                       * val901;
  tolAnglbStm                      * val902;
  tolZoneDir2                      * val903;
  tolAnglwrtStm                    * val904;
  tolAnglrStm                      * val905;
  tolCirltyStm                     * val906;
  tolCompos1                       * val907;
  tolCompos2                       * val908;
  tolComposStm                     * val909;
  tolFeatureMat                    * val910;
  tolConcenStm                     * val911;
  tolCortolStm                     * val912;
  tolCortolMinor                   * val913;
  axial                            * val914;
  tolCproflStm                     * val915;
  tolCprofsStm                     * val916;
  tolCprofSpec                     * val917;
  tolCprofSpecC                    * val918;
  tolCrnoutStm                     * val919;
  tolCylctyStm                     * val920;
  tolDiamStm                       * val921;
  tolDiamMinor                     * val922;
  avgMaxMin                        * val923;
  tolDistbStm                      * val924;
  tolDistbDir                      * val925;
  tolDistbTol                      * val926;
  tolDistbLimit                    * val927;
  tolDistbNominl                   * val928;
  tolDistwrtStm                    * val929;
  tolFlatStm                       * val930;
  tolFlatMinor                     * val931;
  tolGtolStm                       * val932;
  tolGtolTrans                     * val933;
  tolGtolRot                       * val934;
  tolGtolSpec                      * val935;
  tolAppData                       * val936;
  tolFeatMaxMat                    * val937;
  tolMaxMatCond                    * val938;
  tolZoneDir                       * val939;
  tolParlelStm                     * val940;
  tolPerpStm                       * val941;
  tolPosStm                        * val942;
  tolPosStart                      * val943;
  tolPosEnd                        * val944;
  tolPosMethod                     * val945;
  tolProflStm                      * val946;
  tolProfpStm                      * val947;
  tolProfsStm                      * val948;
  tolRadStm                        * val949;
  tolRadSpec                       * val950;
  tolZoneDir3                      * val951;
  tolStrghtStm                     * val952;
  tolStrghtMinor                   * val953;
  tolSymStm                        * val954;
  tolTrnoutStm                     * val955;
  tolUsetolStm                     * val956;
  tolWidthStm                      * val957;
  tolWidthDir                      * val958;
  tolFeature                       * val959;
  tolMatCond                       * val960;
  tolZoneType                      * val961;
  tooldfStm                        * val962;
  transStm                         * val963;
  transMinor                       * val964;
  transValue                       * val965;
  transFeature                     * val966;
  transDatum                       * val967;
  transProbe                       * val968;
  uncertalgStm                     * val969;
  uncertMinor                      * val970;
  uncertVar3                       * val971;
  uncertsetStm                     * val972;
  uncertsetMinor                   * val973;
  uncertsetVar2                    * val974;
  unitsStm                         * val975;
  lengthUnit                       * val976;
  angleUnit                        * val977;
  tempUnit                         * val978;
  valueStm                         * val979;
  valueMinorBool                   * val980;
  valueMinorString                 * val981;
  valueMinorInt                    * val982;
  valueMinorReal                   * val983;
  valueMinorVector                 * val984;
  valueAclratReal                  * val985;
  valueAclratString                * val986;
  valueBadtst                      * val987;
  valueBoundString                 * val988;
  valueBoundInt                    * val989;
  valueCrmode                      * val990;
  valueCroscl                      * val991;
  valueCrslct                      * val992;
  valueCzslct                      * val993;
  valueDatset                      * val994;
  valueDeflection                  * val995;
  valueDmismd                      * val996;
  valueDmismn                      * val997;
  valueError                       * val998;
  valueFedratReal                  * val999;
  valueFedratString                * val1000;
  valueFilnam                      * val1001;
  valueFinpos                      * val1002;
  valueGeoalg                      * val1003;
  valueGotoReal                    * val1004;
  valueGotoVector                  * val1005;
  valueKeychar                     * val1006;
  valueMode                        * val1007;
  valuePrcomp                      * val1008;
  valuePtbuff                      * val1009;
  valuePtmeasReal                  * val1010;
  valuePtmeasVector                * val1011;
  valueScnmod                      * val1012;
  valueSnsetReal                   * val1013;
  valueSnsetString                 * val1014;
  valueSnslct                      * val1015;
  valueSnsmnt                      * val1016;
  valueTecomp                      * val1017;
  valueUnits                       * val1018;
  valueWkplan                      * val1019;
  valueFaInt                       * val1020;
  valueFaReal                      * val1021;
  valueRt                          * val1022;
  valueSa                          * val1023;
  valueSw                          * val1024;
  valueTaBool                      * val1025;
  valueTaReal                      * val1026;
  vformStm                         * val1027;
  std::list<vformItem *>           * val1028;
  windefStm                        * val1029;
  windefMinor                      * val1030;
  windefEdgeln                     * val1031;
  windefBox                        * val1032;
  wkplanStm                        * val1033;
  wristStm                         * val1034;
  std::list<wristDef *>            * val1035;
  wristDef                         * val1036;
  wristMountDef                    * val1037;
  wristRotDef                      * val1038;
  wristRotLimit                    * val1039;
  wristRotStep                     * val1040;
  wristMountOffset                 * val1041;
  wristDataDesc                    * val1042;
  writeStm                         * val1043;
  std::list<writeSpec *>           * val1044;
  writeSpec                        * val1045;
  xternStm                         * val1046;
  xtractStm                        * val1047;
  char                             * sval;
  int                                ival;
  double                             rval;
}

%type <val1> inputFile
%type <val2> dmisItem
%type <val3> dmisItemList
%type <val4> dmisFirstStatement
%type <val5> dmisBlock
%type <val6> calibMasterBlock
%type <val7> calibRtabBlock
%type <val8> calibSensBlock
%type <val9> calibSensBlockItemList
%type <val10> calibSensBlockStatement
%type <val11> calibSensBlockItem
%type <val12> calibSensDoBlock
%type <val13> calibSensIfBlock
%type <val14> calibSensSelectBlock
%type <val15> calibSensCaseBlockList
%type <val16> calibSensCaseBlock
%type <val17> calibSensDefaultCaseBlock
%type <val18> callBlock
%type <val19> caseBlock
%type <val20> caseList
%type <val21> defaultCaseBlock
%type <val22> dmisOffBlock
%type <val23> doBlock
%type <val24> gotargBlock
%type <val25> gotoList
%type <val26> ifBlock
%type <val27> macroBlock
%type <val28> measBlock
%type <val29> measBlockItemList
%type <val30> measBlockStatement
%type <val31> measBlockItem
%type <val32> measDoBlock
%type <val33> measIfBlock
%type <val34> measSelectBlock
%type <val35> measCaseBlockList
%type <val36> measCaseBlock
%type <val37> measDefaultCaseBlock
%type <val38> selectBlock
%type <val39> caseBlockList
%type <val40> simreqtBlock
%type <val41> evalOrOutputList
%type <val42> evalOrOutput
%type <val43> xternBlock
%type <val44> extfilList
%type <val45> dmisFreeStatement
%type <val46> anyVal
%type <val47> stringVal
%type <val48> boolVal
%type <val49> intVal
%type <val50> realVal
%type <val51> rentVal
%type <val52> vectorVal
%type <val53> stringConst
%type <val54> boolConst
%type <val55> intConst
%type <val56> realConst
%type <val57> stringFunc
%type <val58> boolFunc
%type <val59> intFunc
%type <val60> realFunc
%type <val61> vectorFunc
%type <val62> boolExpr
%type <val63> stringRel
%type <val64> boolRel
%type <val65> intRel
%type <val66> intExpr
%type <val67> realRel
%type <val68> realRealRel
%type <val69> intRealRel
%type <val70> realIntRel
%type <val71> vectorRel
%type <val72> realExpr
%type <val73> realExprParen
%type <val74> realExprNegat
%type <val75> realExprPosit
%type <val76> realRealExpr
%type <val77> realIntExpr
%type <val78> intRealExpr
%type <val79> vectorExpr
%type <val80> boolVar
%type <val81> intVar
%type <val82> realVar
%type <val83> stringVar
%type <val84> vectorVar
%type <val85> arrayIndex
%type <val86> stringFuncChr
%type <val87> stringFuncConcat
%type <val88> stringFuncElemnt
%type <val89> stringFuncScfeat
%type <val90> stringFuncScsns
%type <val91> stringFuncSdate
%type <val92> stringFuncSerror
%type <val93> stringFuncSmode
%type <val94> stringFuncStime
%type <val95> stringFuncSdatetime
%type <val96> stringFuncRpt
%type <val97> stringFuncSelapsetime
%type <val98> longShort
%type <val99> stringFuncStr
%type <val100> strVar7
%type <val101> stringFuncLwc
%type <val102> stringFuncTrim
%type <val103> stringFuncUpc
%type <val104> stringFuncSubstr
%type <val105> boolFuncBadgt
%type <val106> boolFuncExist
%type <val107> boolFuncBadpt
%type <val108> boolFuncEof
%type <val109> boolFuncEoln
%type <val110> intFuncAbs
%type <val111> intFuncIndx
%type <val112> intFuncInt
%type <val113> intFuncLen
%type <val114> intFuncMod
%type <val115> intFuncNint
%type <val116> intFuncOrd
%type <val117> intFuncPtdata
%type <val118> intFuncSensnotouch
%type <val119> intFuncSign
%type <val120> intFuncSiltch
%type <val121> realFuncAbs
%type <val122> realFuncCos
%type <val123> realFuncDble
%type <val124> realFuncDtor
%type <val125> realFuncExp
%type <val126> realFuncLn
%type <val127> realFuncLog
%type <val128> realFuncMn
%type <val129> realFuncMod
%type <val130> realFuncMx
%type <val131> realFuncQtemp
%type <val132> realFuncRand
%type <val133> realFuncRl
%type <val134> realFuncRtod
%type <val135> realFuncSign
%type <val136> realFuncSin
%type <val137> realFuncSqrt
%type <val138> realFuncTan
%type <val139> realFuncVal
%type <val140> realFuncVdot
%type <val141> realFuncVecX
%type <val142> realFuncVecY
%type <val143> realFuncVecZ
%type <val144> realFuncVmag
%type <val145> realFuncAcos
%type <val146> realFuncAsin
%type <val147> realFuncAtan
%type <val148> realFuncAtan2
%type <val149> vectorFuncVcart
%type <val150> vectorFuncVcross
%type <val151> vectorFuncVmcs
%type <val152> vectorFuncVpcs
%type <val153> vectorFuncVpol
%type <val154> vectorFuncVunit
%type <val155> labelNameConst
%type <val156> labelName
%type <val157> datumLabel
%type <val158> datumLabel1
%type <val159> datumLabel2
%type <val160> featureLabel
%type <val161> featureLabel1
%type <val162> qisLabel
%type <val163> sensorLabel1
%type <val164> sensorLabel2
%type <val165> toleranceLabel
%type <val166> ccLabel
%type <val167> ciLabel
%type <val168> crLabel
%type <val169> crLabelConst
%type <val170> csLabel
%type <val171> czLabel
%type <val172> dLabel
%type <val173> daLabel
%type <val174> datLabel
%type <val175> dattrgLabel
%type <val176> diLabel
%type <val177> didLabel
%type <val178> drLabel
%type <val179> dsLabel
%type <val180> dvLabel
%type <val181> fLabel
%type <val182> faLabel
%type <val183> fiLabel
%type <val184> fsLabel
%type <val185> gLabel
%type <val186> gsaLabel
%type <val187> kcLabel
%type <val188> kcaLabel
%type <val189> liLabel
%type <val190> mLabel
%type <val191> mLabelConst
%type <val192> maLabel
%type <val193> mdLabel
%type <val194> opLabel
%type <val195> pLabel
%type <val196> pcLabel
%type <val197> plLabel
%type <val198> pnLabel
%type <val199> prLabel
%type <val200> psLabel
%type <val201> pvLabel
%type <val202> qLabel
%type <val203> rLabel
%type <val204> rmLabel
%type <val205> rtLabel
%type <val206> sLabel
%type <val207> saLabel
%type <val208> seLabel
%type <val209> sgLabel
%type <val210> sgsLabel
%type <val211> srLabel
%type <val212> sraLabel
%type <val213> ssLabel
%type <val214> stLabel
%type <val215> swLabel
%type <val216> sxLabel
%type <val217> tLabel
%type <val218> taLabel
%type <val219> thLabel
%type <val220> tlLabel
%type <val221> uLabel
%type <val222> vLabel
%type <val223> vaLabel
%type <val224> vfLabel
%type <val225> vlLabel
%type <val226> vwLabel
%type <val227> jumpLabel
%type <val228> param
%type <val229> paramList
%type <val230> stringList
%type <val231> intList
%type <val232> leftRight
%type <val233> rentList
%type <val234> featureList
%type <val235> featureNominalList
%type <val236> featureActualList
%type <val237> noParseStmList
%type <val238> dattrgList
%type <val239> saLabelList
%type <val240> stackElementList
%type <val241> tLabelList
%type <val242> taLabelList
%type <val243> pointRange
%type <val244> indexedFeature
%type <val245> indexedFeatureList
%type <val246> versionTag
%type <val247> angle
%type <val248> angleDms
%type <val249> noParseStm
%type <val250> secondsString
%type <val251> vector
%type <val252> matrix
%type <val253> rwVar
%type <val254> rwFormat
%type <val255> device
%type <val256> state
%type <val257> coordType
%type <val258> cartPoint
%type <val259> impCartPoint
%type <val260> impCartPointList
%type <val261> polPoint
%type <val262> impPolPoint
%type <val263> impPolPointList
%type <val264> typePoint
%type <val265> pointVecCart
%type <val266> pointVecCartList
%type <val267> pointVecPol
%type <val268> pointVecPolList
%type <val269> axis
%type <val270> orig
%type <val271> dir
%type <val272> posDir
%type <val273> negDir
%type <val274> plan
%type <val275> matDir
%type <val276> flatRoundOpen
%type <val277> radiusSpec
%type <val278> stackElement
%type <val279> rotType
%type <val280> rotDir
%type <val281> rotOrient
%type <val282> rotAbs
%type <val283> rotIncr
%type <val284> jumpStm
%type <val285> aclratStm
%type <val286> aclratMinor
%type <val287> aclratMeas
%type <val288> aclratPos
%type <val289> aclratRot
%type <val290> aclratScan
%type <val291> aclratHedRot
%type <val292> aclratHedMeas
%type <val293> aclratHedScan
%type <val294> aclratAngSpec
%type <val295> aclratLinSpec
%type <val296> aclratLinear
%type <val297> aclratAngular
%type <val298> aclratDef
%type <val299> algdefStm
%type <val300> algdefMinor
%type <val301> assignStm
%type <val302> assignString
%type <val303> assignBool
%type <val304> assignInt
%type <val305> assignReal
%type <val306> assignVector
%type <val307> badtstStm
%type <val308> boundStm
%type <val309> boundMinor
%type <val310> boundFeat
%type <val311> boundTol
%type <val312> calibSensStm
%type <val313> calibRtabStm
%type <val314> calibMasterStm
%type <val315> calibSensMinor
%type <val316> calibSensSpec
%type <val317> calibRtabMinor
%type <val318> calibRtabSpec
%type <val319> calibMasterMinor
%type <val320> callStm
%type <val321> callMinor
%type <val322> callMacro
%type <val323> callModule
%type <val324> callRoutine
%type <val325> callProgram
%type <val326> callType
%type <val327> caseStm
%type <val328> plusSign
%type <val329> minusSign
%type <val330> sign
%type <val331> clmpidStm
%type <val332> clmpsnStm
%type <val333> closeStm
%type <val334> closeMinor
%type <val335> cmpntgrpStm
%type <val336> cmpntgrpSpecList
%type <val337> cmpntgrpSpec
%type <val338> cnfrmrulStm
%type <val339> cnfrmrulMinor
%type <val340> constStm
%type <val341> constMinor
%type <val342> constArc
%type <val343> constCircle
%type <val344> constCompound
%type <val345> constCone
%type <val346> constCparln
%type <val347> constCylndr
%type <val348> constEllips
%type <val349> constEdgept
%type <val350> constGeom
%type <val351> constGcurve
%type <val352> constGsurf
%type <val353> constLine
%type <val354> constParpln
%type <val355> constPatern
%type <val356> constPlane
%type <val357> constPoint
%type <val358> retrieve1
%type <val359> retrieve2
%type <val360> retrieve4
%type <val361> retrieve2b
%type <val362> constRctngl
%type <val363> constSgage
%type <val364> constSpart
%type <val365> constSphere
%type <val366> constSympln
%type <val367> constTorus
%type <val368> bfConst
%type <val369> cogConst
%type <val370> coneConst
%type <val371> curveConst
%type <val372> extremConst
%type <val373> extremConstDir
%type <val374> extremConstAxial
%type <val375> extremConstVectorial
%type <val376> extremConstFeature
%type <val377> extremConstRadial
%type <val378> intofConst
%type <val379> midliConst
%type <val380> midplConst
%type <val381> midptConst
%type <val382> moveptConst
%type <val383> nearptConst
%type <val384> offsetConst
%type <val385> partoConst
%type <val386> perptoConst
%type <val387> pierceConst
%type <val388> projctConst
%type <val389> projliConst
%type <val390> projptConst
%type <val391> sgageConst
%type <val392> spartConst
%type <val393> tantoConst
%type <val394> tantoConstPlane
%type <val395> trConst
%type <val396> vertexConst
%type <val397> crgdefStm
%type <val398> crgdefMinor
%type <val399> crmodeStm
%type <val400> crmodeMinor
%type <val401> crosclStm
%type <val402> crslctStm
%type <val403> crslctMinor
%type <val404> cutcomStm
%type <val405> cutcomMinor
%type <val406> cutcomAdjust
%type <val407> cutcomParam
%type <val408> cutcomMatrix
%type <val409> cutcomMatx
%type <val410> cutcomUserdf
%type <val411> czoneStm
%type <val412> czslctStm
%type <val413> datdefStm
%type <val414> datdefMinor
%type <val415> datsetStm
%type <val416> datsetMinor
%type <val417> datsetMcs
%type <val418> datsetMatrix
%type <val419> datsetDats
%type <val420> datsetDrf
%type <val421> tier
%type <val422> datsetSpec
%type <val422> datsetSpecC
%type <val423> dattrgdefStm
%type <val424> dattrgMinor
%type <val425> declStm
%type <val426> declMinor
%type <val427> declVarList
%type <val428> declScope
%type <val429> declType
%type <val430> declVar
%type <val431> declIndicesList
%type <val432> decplStm
%type <val433> decplMinor
%type <val434> decplAll
%type <val435> decplList
%type <val436> decplNdigits
%type <val437> decplSelection
%type <val438> deleteStm
%type <val439> deleteMinor
%type <val440> deleteDatum
%type <val441> deleteSensor
%type <val442> deleteFeature
%type <val443> deleteRotaryTable
%type <val444> deleteAllSensors
%type <val445> deviceStm
%type <val446> deviceMinor
%type <val447> dftcasStm
%type <val448> displyStm
%type <val449> displyOff
%type <val450> displySpecList
%type <val451> dmehwStm
%type <val452> dmehwMinor
%type <val453> jointConf
%type <val454> rightyLefty
%type <val455> aboveBelow
%type <val456> flipNoFlip
%type <val457> dmeidStm
%type <val458> dmeswStm
%type <val459> dmeswMinor
%type <val460> dmeswComand
%type <val461> dmeswDelay
%type <val462> dmeswSwitch
%type <val463> dmeswiStm
%type <val464> dmeswvStm
%type <val465> dmisOffStm
%type <val466> dmisOnStm
%type <val467> dmismdStm
%type <val468> dmismnStm
%type <val469> conformItemList
%type <val470> conformItem
%type <val471> conformType
%type <val472> doStm
%type <val473> elseStm
%type <val474> endcasStm
%type <val475> enddoStm
%type <val476> endfilStm
%type <val477> endgoStm
%type <val478> endifStm
%type <val479> endmacStm
%type <val480> endmesStm
%type <val481> endselStm
%type <val482> endsimreqtStm
%type <val483> endxtnStm
%type <val484> equateStm
%type <val485> equateMinor
%type <val486> equateCadcs
%type <val487> errorStm
%type <val488> errorMinor
%type <val489> errorCondition
%type <val490> evalStm
%type <val491> evalMinor
%type <val492> evalFeat
%type <val493> evalDat
%type <val494> evalKeychar
%type <val495> extensStm
%type <val496> extensMinor
%type <val497> extfilStm
%type <val498> extfilMinor
%type <val499> extfilDmis
%type <val500> extfilDme
%type <val501> featArc1Stm
%type <val502> featArc2Stm
%type <val503> featCircleStm
%type <val504> featCompoundStm
%type <val505> featConeStm
%type <val506> featConradsegStm
%type <val507> featCparlnStm
%type <val508> featCylndrStm
%type <val509> featCylradsegStm
%type <val510> featEdgeptStm
%type <val511> featEllipsStm
%type <val512> featElongcylStm
%type <val513> featGcurveStm
%type <val514> featGeomStm
%type <val515> featGsurfStm
%type <val516> featLineStm
%type <val517> featObjectStm
%type <val518> featParplnStm
%type <val519> featPaternStm
%type <val520> featPlaneStm
%type <val521> featPointStm
%type <val522> featRctnglStm
%type <val523> featRevsurfStm
%type <val524> featSphereStm
%type <val525> featSphradsegStm
%type <val526> featSymplnStm
%type <val527> featTorradsegStm
%type <val528> featTorusStm
%type <val529> featMinorArc1
%type <val530> featMinorArc2
%type <val531> featMinorCircle
%type <val532> featMinorCompound
%type <val533> featMinorCone
%type <val534> featMinorConradseg
%type <val535> featMinorCparln
%type <val536> featMinorCylndr
%type <val537> featMinorCylradseg
%type <val538> featMinorEdgept
%type <val539> featMinorEllips
%type <val540> featMinorElongcyl
%type <val541> featMinorGeom
%type <val542> featMinorGcurve
%type <val543> featMinorGsurf
%type <val544> featMinorLine
%type <val545> featMinorObject
%type <val546> featMinorParpln
%type <val547> featMinorPatern
%type <val548> featMinorPlane
%type <val549> featMinorPoint
%type <val550> featMinorRctngl
%type <val551> featMinorRevsurf
%type <val552> featMinorSphere
%type <val553> featMinorSphradseg
%type <val554> featMinorSympln
%type <val555> featMinorTorradseg
%type <val556> featMinorTorus
%type <val557> fedratStm
%type <val558> fedratMinor
%type <val559> fedratMeas
%type <val560> fedratPos
%type <val561> fedratRot
%type <val562> fedratScan
%type <val563> fedratHedRot
%type <val564> fedratHedMeas
%type <val565> fedratHedScan
%type <val566> fedratAngSpec
%type <val567> fedratLinSpec
%type <val568> fedratLinear
%type <val569> fedratAngular
%type <val570> fedratDef
%type <val571> fildefStm
%type <val572> filnamStm
%type <val573> finposStm
%type <val574> fixtidStm
%type <val575> fixtsnStm
%type <val576> flyStm
%type <val577> flyMinor
%type <val578> fromStm
%type <val579> fromMinor
%type <val580> geoalgStm
%type <val581> geoalgMinor
%type <val582> geoalgArc
%type <val583> geoalgCircle
%type <val584> geoalgCone
%type <val585> geoalgConradsegmnt
%type <val586> geoalgCparln
%type <val587> geoalgCylndr
%type <val588> geoalgCylradsegmnt
%type <val589> geoalgEllips
%type <val590> geoalgElongcyl
%type <val591> geoalgGcurve
%type <val592> geoalgGsurf
%type <val593> geoalgLine
%type <val594> geoalgObject
%type <val595> geoalgParpln
%type <val596> geoalgPlane
%type <val597> geoalgRctngl
%type <val598> geoalgRevsurf
%type <val599> geoalgSphere
%type <val600> geoalgSphradsegmnt
%type <val601> geoalgSympln
%type <val602> geoalgTorus
%type <val603> geoalgTorradsegmnt
%type <val604> geoalgSpec1
%type <val605> geoalgSpec2
%type <val606> geoalgSpec3
%type <val607> geoalgSpec4
%type <val608> geoalgSpec5
%type <val609> geoalgSpec6
%type <val610> geoalgExternFunc
%type <val611> geoalgFilterSettings
%type <val612> geoalgEliminate
%type <val613> geoalgFilter
%type <val614> geoalgFilterType
%type <val615> geoalgFilterCurve
%type <val616> geomStm
%type <val617> geomMinor
%type <val618> geomEntityList
%type <val619> gohomeStm
%type <val620> gotargStm
%type <val621> gotoStm
%type <val622> gotoMinor
%type <val623> gotoAxis
%type <val624> gotoAxisSpec
%type <val625> gotoAbs
%type <val626> sensorMove
%type <val627> gotoRel
%type <val628> gotoArc
%type <val629> probeOrient
%type <val630> gotoWristList
%type <val631> gotoWristAngleList
%type <val632> gotoWristAngleItem
%type <val633> groupStm
%type <val634> ifStm
%type <val635> includStm
%type <val636> includMinor
%type <val637> includDmis
%type <val638> includDme
%type <val639> iteratStm
%type <val640> iteratMinor
%type <val641> iteratConvergence
%type <val642> iteratCriterionList
%type <val643> iteratCriterionStart
%type <val644> jumptoStm
%type <val645> keycharStm
%type <val646> keycharMinor
%type <val647> keycharOneFeature
%type <val648> keycharTwoFeatures
%type <val649> criticality
%type <val650> litdefStm
%type <val651> litdefMinor
%type <val652> litdefLight
%type <val653> litdefLightType
%type <val654> litdefStrobe
%type <val655> litdefStrobeType
%type <val656> locateStm
%type <val657> locateTransAllowed
%type <val658> locateRotAllowed
%type <val659> locateLabel
%type <val660> locateLabelList
%type <val661> lotidStm
%type <val662> macroStm
%type <val663> macroParList
%type <val664> macroPar
%type <val665> matdefStm
%type <val666> matdefMinor
%type <val667> matdefFeat
%type <val668> matdefGeom
%type <val669> matdefSpec
%type <val670> faLabelList
%type <val671> matdefType
%type <val672> matdefMating
%type <val673> matdefMethod
%type <val674> matdefMat
%type <val675> measStm
%type <val676> measMinor
%type <val677> measGeotype
%type <val678> measVar2
%type <val679> measVar2Aux
%type <val680> mfgdevStm
%type <val681> modeStm
%type <val682> obtainStm
%type <val683> obtainLabeled
%type <val684> obtainPoint
%type <val685> openStm
%type <val686> openMinor
%type <val687> openDevice
%type <val688> inputAccess
%type <val689> outputAccess
%type <val690> openFdata
%type <val691> openCadfile
%type <val692> cadfileType
%type <val693> storageSpecList
%type <val694> storageSpec
%type <val695> openDMLfile
%type <val696> operidStm
%type <val697> outputStm
%type <val698> outputMinor
%type <val699> outputFeatData
%type <val700> outputFeat
%type <val701> outputDat
%type <val702> outputReportData
%type <val703> outputConstData
%type <val704> outputKeycharData
%type <val705> outputToleranceData
%type <val706> outputSensorData
%type <val707> outputSensorDesc
%type <val708> outputSensorWristList
%type <val709> outputSensorWristAngleList
%type <val710> outputSensorWristAngle
%type <val711> pameasStm
%type <val712> pameasMinor
%type <val713> pameasDetail
%type <val714> pameasVar2List
%type <val715> pameasVar3List
%type <val716> pameasRemove
%type <val717> pameasFedrat
%type <val718> partidStm
%type <val719> partrvStm
%type <val720> partsnStm
%type <val721> pathStm
%type <val722> pathMinor
%type <val723> euler
%type <val724> pathPoint
%type <val725> pathArc
%type <val726> pathCurve
%type <val727> pathCurvePtdata
%type <val728> curvePtdataPcs
%type <val729> curvePtdataPcsList
%type <val730> curvePtdataHeadcs
%type <val731> curvePtdataHeadcsList
%type <val732> pathCurvePoints
%type <val733> pathHelical
%type <val734> pathLine
%type <val735> csSpec
%type <val735> csSpecC
%type <val736> pathSurface
%type <val737> surfPtdataList
%type <val738> surfPtdataListItem
%type <val739> surfPtdataPcsList
%type <val740> surfPtdataHeadcsList
%type <val741> pathUnknown
%type <val742> planidStm
%type <val743> popStm
%type <val744> prcompStm
%type <val745> prevopStm
%type <val746> procidStm
%type <val747> promptStm
%type <val748> promptIntEnd
%type <val749> promptItemList
%type <val750> promptVar
%type <val751> psthruStm
%type <val752> psthruMinor
%type <val753> ptbuffStm
%type <val754> ptmeasStm
%type <val755> ptmeasEnd
%type <val756> touchSpec
%type <val757> pushStm
%type <val758> qisdefStm
%type <val759> rapidStm
%type <val760> readStm
%type <val761> readSpecList
%type <val762> readSpec
%type <val763> recallDatumStm
%type <val764> recallSensorStm
%type <val765> recallFeatureStm
%type <val766> recallRotaryTableStm
%type <val767> recallDMLStm
%type <val768> refmntStm
%type <val769> reportStm
%type <val770> reportItemList
%type <val771> reportItem
%type <val772> resumeStm
%type <val773> resumeMinor
%type <val774> rmeasStm
%type <val775> rmeasMinor
%type <val776> rmeasArc
%type <val777> rmeasCircle
%type <val778> rmeasCone
%type <val779> rmeasCparln
%type <val780> rmeasCylndr
%type <val781> rmeasEdgept
%type <val782> rmeasEllips
%type <val783> rmeasGcurve
%type <val784> rmeasGsurf
%type <val785> rmeasLine
%type <val786> rmeasObject
%type <val787> rmeasParpln
%type <val788> rmeasPlane
%type <val789> rmeasPoint
%type <val790> rmeasRctngl
%type <val791> rmeasSphere
%type <val792> rmeasSympln
%type <val793> rmeasTorus
%type <val794> rmeasSpecFa
%type <val795> rmeasSpecFaAxis
%type <val796> rmeasSpecFaOrient
%type <val797> rmeasSpecFeatAxis
%type <val798> rmeasSpecVecbld
%type <val799> rmeasSpecVecbldOrient
%type <val800> rmeasSpecVecbldEdgept
%type <val801> rotabStm
%type <val802> rotabMinor
%type <val803> rotateStm
%type <val804> rotateMinor
%type <val805> rotateValue
%type <val806> rotateFeature
%type <val807> rotateDatum
%type <val808> rotdefStm
%type <val809> rotsetStm
%type <val810> saveDatumStm
%type <val811> saveSensorStm
%type <val812> saveFeatureStm
%type <val813> saveRotaryTableStm
%type <val814> saveDMLStm
%type <val815> scnmodStm
%type <val816> scnsetStm
%type <val817> scnsetMinor
%type <val818> scnsetPeck
%type <val819> scnsetDrag
%type <val820> scnsetNoncon
%type <val821> scnsetStop
%type <val822> scnsetStopPlane
%type <val823> scnsetStopSphere
%type <val824> scnsetVendor
%type <val825> scnsetSample
%type <val826> scnsetSampleDrag
%type <val827> forceOrDeflection
%type <val828> selectStm
%type <val829> sensorStm
%type <val830> sensorMinor
%type <val831> sensorProbe
%type <val832> sensorMltprb
%type <val833> sensorMltprbItemList
%type <val834> sensorVideo
%type <val835> sensorLaser
%type <val836> sensorInfred
%type <val837> sensorNoncon
%type <val838> sensorPoint
%type <val839> sensorLine
%type <val840> sensorArea
%type <val841> sensorProbeGeometry
%type <val842> sensorProbeForm
%type <val843> sensorItem
%type <val844> sensorMltprbItem
%type <val845> sensorGeometry
%type <val846> simreqtStm
%type <val847> simreqtMinor
%type <val848> snsdefStm
%type <val849> snsdefMinor
%type <val850> snsdefProbe
%type <val851> probeLocation
%type <val852> snsdefProbeForm
%type <val853> snsdefVideo
%type <val854> snsdefLaser
%type <val855> snsdefInfred
%type <val856> snsdefNoncon
%type <val857> snsdefLocation
%type <val858> snsdefType
%type <val859> snsdefBuildSensor
%type <val860> snsdefBuild
%type <val861> snsdefBuildItemList
%type <val862> snsdefWristAngleList
%type <val863> snsdefWristAngleItem
%type <val864> snsetStm
%type <val865> snsetItemList
%type <val866> snsetTypeAndValue
%type <val867> snsetType
%type <val868> snsetToggle
%type <val869> snsetLabelAndValue
%type <val870> snsetLabel
%type <val871> snsetFeat
%type <val872> snsgrpStm
%type <val873> snsgrpMinor
%type <val874> snsgrpItemList
%type <val875> snsgrpItem
%type <val876> snslctStm
%type <val877> snslctGroup
%type <val878> snslctSensor
%type <val879> snslctTipData
%type <val880> sensorList
%type <val881> snslctData
%type <val882> snslctWristList
%type <val883> snslctWristAngleList
%type <val884> snslctWristAngleItem
%type <val885> snsmntStm
%type <val886> snsmntMinor
%type <val887> tecompStm
%type <val888> tecompMinor
%type <val889> tecompVar2
%type <val890> textStm
%type <val891> textMinor
%type <val892> textMan
%type <val893> textOper
%type <val894> textOutfil
%type <val895> textQuery
%type <val896> textQueryFormat
%type <val897> textType
%type <val898> thldefStm
%type <val899> thldefPocketList
%type <val900> thldefPocket
%type <val901> tolAnglStm
%type <val902> tolAnglbStm
%type <val903> tolZoneDir2
%type <val904> tolAnglwrtStm
%type <val905> tolAnglrStm
%type <val906> tolCirltyStm
%type <val907> tolCompos1
%type <val908> tolCompos2
%type <val909> tolComposStm
%type <val910> tolFeatureMat
%type <val910> tolFeatureMatC
%type <val911> tolConcenStm
%type <val912> tolCortolStm
%type <val913> tolCortolMinor
%type <val914> axial
%type <val915> tolCproflStm
%type <val916> tolCprofsStm
%type <val917> tolCprofSpec
%type <val918> tolCprofSpecC
%type <val919> tolCrnoutStm
%type <val920> tolCylctyStm
%type <val921> tolDiamStm
%type <val922> tolDiamMinor
%type <val923> avgMaxMin
%type <val924> tolDistbStm
%type <val925> tolDistbDir
%type <val926> tolDistbTol
%type <val927> tolDistbLimit
%type <val928> tolDistbNominl
%type <val929> tolDistwrtStm
%type <val930> tolFlatStm
%type <val931> tolFlatMinor
%type <val932> tolGtolStm
%type <val933> tolGtolTrans
%type <val934> tolGtolRot
%type <val935> tolGtolSpec
%type <val936> tolAppData
%type <val937> tolFeatMaxMat
%type <val937> tolFeatMaxMatC
%type <val938> tolMaxMatCond
%type <val938> tolMaxMatCondC
%type <val939> tolZoneDir
%type <val940> tolParlelStm
%type <val941> tolPerpStm
%type <val942> tolPosStm
%type <val943> tolPosStart
%type <val944> tolPosEnd
%type <val945> tolPosMethod
%type <val946> tolProflStm
%type <val947> tolProfpStm
%type <val948> tolProfsStm
%type <val949> tolRadStm
%type <val950> tolRadSpec
%type <val951> tolZoneDir3
%type <val952> tolStrghtStm
%type <val953> tolStrghtMinor
%type <val954> tolSymStm
%type <val955> tolTrnoutStm
%type <val956> tolUsetolStm
%type <val957> tolWidthStm
%type <val958> tolWidthDir
%type <val959> tolFeature
%type <val960> tolMatCond
%type <val961> tolZoneType
%type <val962> tooldfStm
%type <val963> transStm
%type <val964> transMinor
%type <val965> transValue
%type <val966> transFeature
%type <val967> transDatum
%type <val968> transProbe
%type <val969> uncertalgStm
%type <val970> uncertMinor
%type <val971> uncertVar3
%type <val972> uncertsetStm
%type <val973> uncertsetMinor
%type <val974> uncertsetVar2
%type <val975> unitsStm
%type <val976> lengthUnit
%type <val977> angleUnit
%type <val978> tempUnit
%type <val979> valueStm
%type <val980> valueMinorBool
%type <val981> valueMinorString
%type <val982> valueMinorInt
%type <val983> valueMinorReal
%type <val984> valueMinorVector
%type <val985> valueAclratReal
%type <val986> valueAclratString
%type <val987> valueBadtst
%type <val988> valueBoundString
%type <val989> valueBoundInt
%type <val990> valueCrmode
%type <val991> valueCroscl
%type <val992> valueCrslct
%type <val993> valueCzslct
%type <val994> valueDatset
%type <val995> valueDeflection
%type <val996> valueDmismd
%type <val997> valueDmismn
%type <val998> valueError
%type <val999> valueFedratReal
%type <val1000> valueFedratString
%type <val1001> valueFilnam
%type <val1002> valueFinpos
%type <val1003> valueGeoalg
%type <val1004> valueGotoReal
%type <val1005> valueGotoVector
%type <val1006> valueKeychar
%type <val1007> valueMode
%type <val1008> valuePrcomp
%type <val1009> valuePtbuff
%type <val1010> valuePtmeasReal
%type <val1011> valuePtmeasVector
%type <val1012> valueScnmod
%type <val1013> valueSnsetReal
%type <val1014> valueSnsetString
%type <val1015> valueSnslct
%type <val1016> valueSnsmnt
%type <val1017> valueTecomp
%type <val1018> valueUnits
%type <val1019> valueWkplan
%type <val1020> valueFaInt
%type <val1021> valueFaReal
%type <val1022> valueRt
%type <val1023> valueSa
%type <val1024> valueSw
%type <val1025> valueTaBool
%type <val1026> valueTaReal
%type <val1027> vformStm
%type <val1028> vformItemList
%type <val1029> windefStm
%type <val1030> windefMinor
%type <val1031> windefEdgeln
%type <val1032> windefBox
%type <val1033> wkplanStm
%type <val1034> wristStm
%type <val1035> wristList
%type <val1036> wristDef
%type <val1037> wristMountDef
%type <val1038> wristRotDef
%type <val1039> wristRotLimit
%type <val1040> wristRotStep
%type <val1041> wristMountOffset
%type <val1042> wristDataDesc
%type <val1043> writeStm
%type <val1044> writeSpecList
%type <val1045> writeSpec
%type <val1046> xternStm
%type <val1047> xtractStm

%token ABOVE
%token ABS
%token ABSL
%token ACEL
%token ACLRAT
%token ACOS
%token ACT
%token ADJUST
%token ALGDEF
%token ALGOR
%token ALL
%token ALLAXESTOUCH
%token ALLSA
%token ALPHA
%token AMT
%token ANGDEC
%token ANGDMS
%token ANGL
%token ANGLB
%token ANGLE
%token ANGLR
%token ANGLWRT
%token ANGRAD
%token APPEND
%token APPRCH
%token ARC
%token AREA
%token ASIN
%token ASSIGN
%token ATAN
%token ATAN2
%token ATTACH
%token AUTO
%token AVG
%token AVGDEV
%token AXDIR
%token AXIAL
%token BACK
%token BADGT
%token BADPT
%token BADTST
%token BANDPASS
%token BELOW
%token BEZIER
%token BF
%token BND
%token BOOL
%token BOUND
%token BOX
%token BSPLIN
%token BUILD
%token BUTTON
%token C
%token CAD
%token CADCS
%token CALIB
%token CALL
%token CART
%token CASE
%token CC
%token CCW
%token CHAR
%token CHECK
%token CHORD
%token CHR
%token CI
%token CIRCLE
%token CIRCULAR
%token CIRLTY
%token CLMPID
%token CLMPSN
%token CLOSE
%token CLRSRF
%token CM
%token CMPNTGRP
%token CNFRMRUL
%token CODE
%token COG
%token COMAND
%token COMM
%token COMMON
%token COMP
%token COMPOS
%token COMPOUND
%token CONCAT
%token CONCEN
%token CONE
%token CONRADSEGMNT
%token CONST
%token CONT
%token CONTIN
%token CORTOL
%token COS
%token COUNT
%token CPARLN
%token CPROFL
%token CPROFS
%token CR
%token CRAD
%token CRGDEF
%token CRITICAL
%token CRMODE
%token CRNOUT
%token CROSCL
%token CRSLCT
%token CS
%token CT
%token CURENT
%token CURVE
%token CUTCOM
%token CW
%token CYCLE
%token CYLCTY
%token CYLNDR
%token CYLRADSEGMNT
%token CZ
%token CZONE
%token CZSLCT
%token D
%token DA
%token DAT
%token DATDEF
%token DATE
%token DATSET
%token DATTRG
%token DATTRGDEF
%token DBLE
%token DECL
%token DECPL
%token DEFALT
%token DEFLECTION
%token DELAY
%token DELETE
%token DEPTH
%token DEV
%token DEVICE
%token DFTCAS
%token DI
%token DIAM
%token DID
%token DIRECT
%token DISK
%token DISPLY
%token DIST
%token DISTANCE
%token DISTB
%token DISTWRT
%token DME
%token DMEHW
%token DMEID
%token DMESW
%token DMESWI
%token DMESWV
%token DMIS
%token DMISMD
%token DMISMN
%token DML
%token DO
%token DOUBLE
%token DR
%token DRAG
%token DRF
%token DS
%token DTOR
%token DV
%token EDGELN
%token EDGEPT
%token EDIT
%token ELEMNT
%token ELIMINATE
%token ELLIPS
%token ELONGCYL
%token ELSE
%token END
%token ENDCAS
%token ENDDO
%token ENDFIL
%token ENDGO
%token ENDIF
%token ENDMAC
%token ENDMES
%token ENDSEL
%token ENDSIMREQT
%token ENDXTN
%token ENTITY
%token EOLN
%token EQUATE
%token ERR
%token ERRMODE
%token ERROR
%token EVAL
%token EXCEPT
%token EXIST
%token EXP
%token EXTENS
%token EXTERN
%token EXTFIL
%token EXTREM
%token F
%token FA
%token FALSE
%token FDATA
%token FEAT
%token FEATUR
%token FEDRAT
%token FEED
%token FEET
%token FI
%token FILDEF
%token FILNAM
%token FILTER
%token FINPOS
%token FIRST
%token FIXED
%token FIXTID
%token FIXTSN
%token FLAT
%token FLIP
%token FLY
%token FOCUSN
%token FOCUSY
%token FORCE
%token FORM
%token FOURPOINT
%token FROM
%token FS
%token FX
%token FZ
%token G
%token GAUSS
%token GCURVE
%token GEOALG
%token GEOM
%token GLOBAL
%token GOHOME
%token GOTARG
%token GOTO
%token GRID
%token GROUP
%token GSA
%token GSURF
%token GTOL
%token HEADCS
%token HEADTOUCH
%token HEDMESACL
%token HEDMESVEL
%token HEDROTACL
%token HEDROTVEL
%token HEDSCNACL
%token HEDSCNVEL
%token HELICAL
%token HIGH
%token HIGHPASS
%token HIST
%token HUMID
%token IF
%token IGES
%token ILLEGALTOUCH
%token INCH
%token INCLUD
%token INCR
%token INDEX
%token INDX
%token INFRED
%token INNER
%token INPUT
%token INT
%token INTFPT
%token INTGR
%token INTOF
%token INTOL
%token IP
%token IPM
%token IPMM
%token IPS
%token IPSS
%token ITERAT
%token JOINTCONFIG
%token JUMPTO
%token KC
%token KCA
%token KEEP
%token KEYCHAR
%token LAMBDAC
%token LASER
%token LEFT
%token LEFTY
%token LEN
%token LI
%token LIMIT
%token LINE
%token LIST
%token LITDEF
%token LMC
%token LN
%token LN2LN
%token LOCAL
%token LOCATE
%token LOG
%token LONG
%token LOTID
%token LOTIER
%token LOW
%token LOWPASS
%token LSTSQR
%token LWC
%token M
%token MA
%token MACH
%token MACRO
%token MAJOR
%token MAN
%token MASTER
%token MATDEF
%token MATRIX
%token MAX
%token MAXINS
%token MC
%token MCS
%token MD
%token MEAS
%token MESACL
%token MESVEL
%token METER
%token MFGDEV
%token MIDLI
%token MIDPL
%token MIDPT
%token MIN
%token MINCIR
%token MINCON
%token MINMAX
%token MINOR
%token MINUSPRBRAD
%token MINUSXDIR
%token MINUSYDIR
%token MINUSZDIR
%token MLTPRB
%token MM
%token MMC
%token MMPS
%token MMPSS
%token MN
%token MNTLEN
%token MOD
%token MODE
%token MOVEPT
%token MPM
%token MPMM
%token MU
%token MX
%token NAME
%token NEARPT
%token NEXT
%token NINT
%token NODATA
%token NOFLIP
%token NOM
%token NOMINL
%token NONCON
%token NONE
%token NOROT
%token NOTOUCH
%token NOTRAN
%token NUMERIC
%token NURBS
%token OBJECT
%token OBLQ
%token OBTAIN
%token OFF
%token OFFSET
%token ON
%token OP
%token OPEN
%token OPER
%token OPERID
%token OPTIMAL
%token ORD
%token ORIENT
%token OUTER
%token OUTFIL
%token OUTOL
%token OUTPUT
%token OVERWR
%token P
%token PAMEAS
%token PARAM
%token PARLEL
%token PARPLN
%token PART
%token PARTID
%token PARTO
%token PARTRV
%token PARTSN
%token PATERN
%token PATH
%token PAUSE
%token PC
%token PCENT
%token PCS
%token PECK
%token PERCNT
%token PERP
%token PERPTO
%token PICTURE
%token PIERCE
%token PITCH
%token PIXBTN
%token PL
%token PLANE
%token PLANID
%token PLOT
%token PM
%token PN
%token POINT
%token POL
%token POP
%token POS
%token POSACL
%token POSVEL
%token PR
%token PRBRAD
%token PRCOMP
%token PREVOP
%token PRINT
%token PRNTCHAR
%token PROBE
%token PROCID
%token PROFL
%token PROFP
%token PROFS
%token PROG
%token PROJCT
%token PROJLI
%token PROJPT
%token PROMPT
%token PS
%token PSTHRU
%token PT2LN
%token PT2PL
%token PT2PT
%token PTBUFF
%token PTDATA
%token PTMEAS
%token PUSH
%token PV
%token Q
%token QI
%token QISDEF
%token QTEMP
%token QUERY
%token R
%token RAD
%token RADIAL
%token RADIUS
%token RAM
%token RAND
%token RAPID
%token RCTNGL
%token READ
%token REAL
%token RECALIB
%token RECALL
%token RECFILT
%token REFMNT
%token REMOVE
%token REPORT
%token RESUME
%token RETRCT
%token RETRIEVE
%token REVSURF
%token RFS
%token RIGHT
%token RIGHTY
%token RL
%token RM
%token RMEAS
%token ROTAB
%token ROTACL
%token ROTARY
%token ROTATE
%token ROTCEN
%token ROTDEF
%token ROTNUL
%token ROTORG
%token ROTSET
%token ROTTOT
%token ROTVEL
%token ROUND
%token RPM
%token RPMM
%token RPT
%token RT
%token RTAB
%token RTOD
%token RULE
%token RY
%token S
%token SA
%token SAVE
%token SCALE
%token SCALEX
%token SCALEY
%token SCFEAT
%token SCNACL
%token SCNMOD
%token SCNSET
%token SCNVEL
%token SCSNS
%token SDATE
%token SDATETIME
%token SE
%token SEARCH
%token SELAPSETIME
%token SELECT
%token SELFCENTER
%token SENS
%token SENSNOTOUCH
%token SENSOR
%token SEQNTL
%token SERROR
%token SF
%token SG
%token SGAGE
%token SGS
%token SHORT
%token SIGN
%token SILTCH
%token SIMREQT
%token SIMUL
%token SIN
%token SINGLE
%token SIZE
%token SMODE
%token SNS
%token SNSDEF
%token SNSET
%token SNSGRP
%token SNSLCT
%token SNSMNT
%token SOUND
%token SPART
%token SPH
%token SPHERE
%token SPHRADSEGMNT
%token SPLINE
%token SQRT
%token SR
%token SRA
%token SS
%token ST
%token START
%token STAT
%token STDDEVLIMIT
%token STEP
%token STIME
%token STOP
%token STOR
%token STR
%token STRGHT
%token STROBE
%token SUBSTR
%token SURF
%token SURFACE
%token SW
%token SX
%token SYM
%token SYMPLN
%token SYNC
%token SYS
%token T
%token TA
%token TAN
%token TANGPL
%token TANTO
%token TECOMP
%token TEMP
%token TEMPC
%token TEMPF
%token TEMPWC
%token TEMPWF
%token TERM
%token TEXT
%token TH
%token THLDEF
%token THREED
%token THRU
%token TIME
%token TITLE
%token TL
%token TOL
%token TOOLDF
%token TORRADSEGMNT
%token TORUS
%token TR
%token TRANS
%token TRIGER
%token TRIM
%token TRMATX
%token TRNOUT
%token TRUE
%token TW
%token TWOD
%token TWORC
%token U
%token UNBND
%token UNCERTALG
%token UNCERTSET
%token UNITS
%token UNKNOWN
%token UPC
%token UPTIER
%token USERDF
%token USETOL
%token V
%token VA
%token VAL
%token VALUE
%token VCART
%token VCROSS
%token VDOT
%token VEC
%token VECBLD
%token VECTOR
%token VECX
%token VECY
%token VECZ
%token VENDOR
%token VERSION
%token VERTEX
%token VF
%token VFORM
%token VIDEO
%token VL
%token VMAG
%token VMCS
%token VPCS
%token VPOL
%token VUNIT
%token VW
%token WAIT
%token WIDTH
%token WINDEF
%token WKPLAN
%token WRIST
%token WRITE
%token WWEOF
%token XAXIS
%token XDIR
%token XORIG
%token XTERN
%token XTRACT
%token XVEC
%token XYAXIS
%token XYDIR
%token XYPLAN
%token XYZAXI
%token XYZDIR
%token YAXIS
%token YDIR
%token YORIG
%token YZAXIS
%token YZDIR
%token YZPLAN
%token ZAXIS
%token ZDIR
%token ZORIG
%token ZVEC
%token ZXAXIS
%token ZXDIR
%token ZXPLAN
%token <sval> BOOLVARNAME
%token <sval> CHARSTRING
%token <sval> DECLVARNAME
%token <ival> INTSTRING
%token <sval> INTVARNAME
%token <sval> LBLNAME
%token <sval> MACROVARNAME
%token <rval> REALSTRING
%token <sval> REALVARNAME
%token <sval> STRINGVARNAME
%token <sval> VECTORVARNAME

%token AT
%token BAD
%token COLON
%token ENDLINE
%token EQUALS
%token LBOX
%token LPAREN
%token RBOX
%token RPAREN

%left AND OR
%left EQ NE LT LE GT GE
%left PLUS MINUS
%left TIMES SLASH
%left POWER
%left NOT

%start inputFile

%%

inputFile :
	  dmisFirstStatement endfilStm
	    { $$ = new inputFile($1, 0, $2);
	      tree = $$;
	    }
	| dmisFirstStatement dmisItemList endfilStm
	    { $$ = new inputFile($1, $2, $3);
	      tree = $$;
	    }
	;

endOfLine :
          ENDLINE
          {
            if (setLabelType)
              {
                labels[setLabelType].record(setLabel);
                setLabelType = 0;
              }
            if (aLabelFound == 0)
              {
                sprintf(warningMessage,
                         "undefined label used?: %s", aLabel);
                warn(warningMessage);
                aLabelFound = 1;
              }
          }
        ;

equalSign :
          EQUALS
          {
            if (aLabelType)
              {
                if (aLabelFound == 1)
                  {
                    if ((aLabelType != F) && (aLabelType != FA))
                      {
                        sprintf(warningMessage,
                                "label doubly defined?: %s", aLabel);
                        warn(warningMessage);
                      }
                  }
                else
                  {
                    aLabelFound = 1;
                    setLabel = aLabel;
                    setLabelType = aLabelType;
                  }
              }
          }
        ;

defCheckComma :
          C
          {
            if (aLabelFound == 1)
              {
                sprintf(warningMessage,
			"label doubly defined?: %s", aLabel);
                warn(warningMessage);
              }
            else
              {
                labels[aLabelType].record(aLabel);
                aLabelFound = 1;
              }
          }
	;

undefCheckComma :
	  C
	  {
            if (aLabelFound == 0)
              {
                sprintf(warningMessage,
                        "undefined label used?: %s", aLabel);
                warn(warningMessage);
                aLabelFound = 1;
              }
          }
	;

existLParen :
	  LPAREN
	  { handleLabel(1); }
	;

existRParen :
	  RPAREN
	  { handleLabel(0); }
	;

dmisItem :
	  dmisFreeStatement
	    { $$ = $1; }
	| dmisBlock
	    { $$ = $1; }
	;

dmisItemList :
	  dmisItem
	    { $$ = new std::list<dmisItem *>;
	      $$->push_back($1); }
	| dmisItemList dmisItem
	    { $$ = $1;
	      $$->push_back($2); }
	;

dmisFirstStatement :
	  dmismdStm
	    { $$ = $1; }
	| dmismnStm
	    { $$ = $1; }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

dmisBlock :
	  calibMasterBlock
	    { $$ = $1; }
	| calibRtabBlock
	    { $$ = $1; }
	| calibSensBlock
	    { $$ = $1; }
	| callBlock
	    { $$ = $1; }
	| dmisOffBlock
	    { $$ = $1; }
	| doBlock
	    { $$ = $1; }
	| gotargBlock
	    { $$ = $1; }
	| ifBlock
	    { $$ = $1; }
	| macroBlock
	    { $$ = $1; }
	| measBlock
	    { $$ = $1; }
	| selectBlock
	    { $$ = $1; }
	| simreqtBlock
	    { $$ = $1; }
	| xternBlock
	    { $$ = $1; }
	;

calibMasterBlock :
	  calibMasterStm endmesStm
	    { $$ = new calibMasterBlock($1, 0, $2); }
	| calibMasterStm calibSensBlockItemList endmesStm
	    { $$ = new calibMasterBlock($1, $2, $3); }
	;

calibRtabBlock :
	  calibRtabStm endmesStm
	    { $$ = new calibRtabBlock($1, 0, $2); }
	| calibRtabStm measBlockItemList endmesStm
	    { $$ = new calibRtabBlock($1, $2, $3); }
	;

calibSensBlock :
	  calibSensStm endmesStm
	    { $$ = new calibSensBlock($1, 0, $2); }
	| calibSensStm calibSensBlockItemList endmesStm
	    { $$ = new calibSensBlock($1, $2, $3); }
	;

calibSensBlockItemList :
	  calibSensBlockItem
	    { $$ = new std::list<calibSensBlockItem *>;
	      $$->push_back($1); }
	| calibSensBlockItemList calibSensBlockItem
	    { $$ = $1;
	      $$->push_back($2); }
	;

calibSensBlockStatement :
	  jumpStm
	    { $$ = $1; }
	| aclratStm
	    { $$ = $1; }
	| assignStm
	    { $$ = $1; }
	| badtstStm
	    { $$ = $1; }
	| czslctStm
	    { $$ = $1; }
	| dmehwStm
	    { $$ = $1; }
	| dmeswStm
	    { $$ = $1; }
	| errorStm
	    { $$ = $1; }
	| fedratStm
	    { $$ = $1; }
	| finposStm
	    { $$ = $1; }
	| fromStm
	    { $$ = $1; }
	| gohomeStm
	    { $$ = $1; }
	| gotoStm
	    { $$ = $1; }
	| includStm
	    { $$ = $1; }
	| jumptoStm
	    { $$ = $1; }
	| obtainStm
	    { $$ = $1; }
	| pameasStm
	    { $$ = $1; }
	| ptmeasStm
	    { $$ = $1; }
	| rapidStm
	    { $$ = $1; }
	| rotabStm
	    { $$ = $1; }
	| rotsetStm
	    { $$ = $1; }
	| snsetStm
	    { $$ = $1; }
	| textStm
	    { $$ = $1; }
	| valueStm
	    { $$ = $1; }
	| wkplanStm
	    { $$ = $1; }
	;

calibSensBlockItem :
	  calibSensBlockStatement
	    { $$ = $1; }
	| calibSensDoBlock
	    { $$ = $1; }
	| calibSensIfBlock
	    { $$ = $1; }
	| calibSensSelectBlock
	    { $$ = $1; }
	| gotargBlock
	    { $$ = $1; }
	;

calibSensDoBlock :
	  doStm enddoStm
	    { $$ = new calibSensDoBlock($1, 0, $2); }
	| doStm calibSensBlockItemList enddoStm
	    { $$ = new calibSensDoBlock($1, $2, $3); }
	;

calibSensIfBlock :
	  ifStm endifStm
	    { $$ = new calibSensIfBlock($1, 0, 0, 0, $2); }
	| ifStm elseStm endifStm
	    { $$ = new calibSensIfBlock($1, 0, $2, 0, $3); }
	| ifStm elseStm calibSensBlockItemList endifStm
	    { $$ = new calibSensIfBlock($1, 0, $2, $3, $4); }
	| ifStm calibSensBlockItemList endifStm
	    { $$ = new calibSensIfBlock($1, $2, 0, 0, $3); }
	| ifStm calibSensBlockItemList elseStm endifStm
	    { $$ = new calibSensIfBlock($1, $2, $3, 0, $4); }
	| ifStm calibSensBlockItemList elseStm calibSensBlockItemList endifStm
	    { $$ = new calibSensIfBlock($1, $2, $3, $4, $5); }
	;

calibSensSelectBlock :
	  selectStm calibSensCaseBlockList endselStm
	    { $$ = new calibSensSelectBlock($1, $2, 0, $3); }
	| selectStm calibSensCaseBlockList calibSensDefaultCaseBlock endselStm
	    { $$ = new calibSensSelectBlock($1, $2, $3, $4); }
	;

calibSensCaseBlockList :
	  calibSensCaseBlock
	    { $$ = new std::list<calibSensCaseBlock *>;
	      $$->push_back($1); }
	| calibSensCaseBlockList calibSensCaseBlock
	    { $$ = $1;
	      $$->push_back($2); }
	;

calibSensCaseBlock :
	  caseList endcasStm
	    { $$ = new calibSensCaseBlock($1, 0, $2); }
	| caseList calibSensBlockItemList endcasStm
	    { $$ = new calibSensCaseBlock($1, $2, $3); }
	;

calibSensDefaultCaseBlock :
	  dftcasStm endcasStm
	    { $$ = new calibSensDefaultCaseBlock($1, 0, $2); }
	| dftcasStm calibSensBlockItemList endcasStm
	    { $$ = new calibSensDefaultCaseBlock($1, $2, $3); }
	;

callBlock :
	  callStm endmacStm
	    { $$ = new callBlock($1, 0, $2); }
	| callStm dmisItemList endmacStm
	    { $$ = new callBlock($1, $2, $3); }
	;

caseBlock :
	  caseList endcasStm
	    { $$ = new caseBlock($1, 0, $2); }
	| caseList dmisItemList endcasStm
	    { $$ = new caseBlock($1, $2, $3); }
	;

caseList :
	  caseStm
	    { $$ = new std::list<caseStm *>;
	      $$->push_back($1); }
	| caseList caseStm
	    { $$ = $1;
	      $$->push_back($2); }
	;

defaultCaseBlock :
	  dftcasStm endcasStm
	    { $$ = new defaultCaseBlock($1, 0, $2); }
	| dftcasStm dmisItemList endcasStm
	    { $$ = new defaultCaseBlock($1, $2, $3); }
	;

dmisOffBlock :
	  dmisOffStm dmisOnStm
	    { $$ = new dmisOffBlock($1, 0, $2); }
	| dmisOffStm noParseStmList dmisOnStm
	    { $$ = new dmisOffBlock($1, $2, $3); }
	;

doBlock :
	  doStm enddoStm
	    { $$ = new doBlock($1, 0, $2); }
	| doStm dmisItemList enddoStm
	    { $$ = new doBlock($1, $2, $3); }
	;

gotargBlock :
	  gotargStm gotoStm gotoList endgoStm
	    { $$ = new gotargBlock($1, $2, $3, $4); }
	;

gotoList :
	  gotoStm
	    { $$ = new std::list<gotoStm *>;
	      $$->push_back($1); }
	| gotoList gotoStm
	    { $$ = $1;
	      $$->push_back($2); }
	;

ifBlock :
	  ifStm endifStm
	    { $$ = new ifBlock($1, 0, 0, 0, $2); }
	| ifStm elseStm endifStm
	    { $$ = new ifBlock($1, 0, $2, 0, $3); }
	| ifStm elseStm dmisItemList endifStm
	    { $$ = new ifBlock($1, 0, $2, $3, $4); }
	| ifStm dmisItemList endifStm
	    { $$ = new ifBlock($1, $2, 0, 0, $3); }
	| ifStm dmisItemList elseStm endifStm
	    { $$ = new ifBlock($1, $2, $3, 0, $4); }
	| ifStm dmisItemList elseStm dmisItemList endifStm
	    { $$ = new ifBlock($1, $2, $3, $4, $5); }
	;

macroBlock :
	  macroStm endmacStm
	    { $$ = new macroBlock($1, 0, $2, macroIsReal);
	      macroIsReal = 1; }
	| macroStm noParseStmList endmacStm
	    { $$ = new macroBlock($1, $2, $3, macroIsReal);
	      macroIsReal = 1; }
	;

measBlock :
	  measStm endmesStm
	    { $$ = new measBlock_measStm($1, 0, $2); }
	| measStm measBlockItemList endmesStm
	    { $$ = new measBlock_measStm($1, $2, $3); }
	| rmeasStm endmesStm
	    { $$ = new measBlock_rmeasStm($1, 0, $2); }
	| rmeasStm measBlockItemList endmesStm
	    { $$ = new measBlock_rmeasStm($1, $2, $3); }
	;

measBlockItemList :
	  measBlockItem
	    { $$ = new std::list<measBlockItem *>;
	      $$->push_back($1); }
	| measBlockItemList measBlockItem
	    { $$ = $1;
	      $$->push_back($2); }
	;

measBlockStatement :
	  calibSensBlockStatement
	    { $$ = $1; }
	| crslctStm
	    { $$ = $1; }
	| recallSensorStm
	    { $$ = $1; }
	| saveSensorStm
	    { $$ = $1; }
	| snslctStm
	    { $$ = $1; }
	;

measBlockItem :
	  measBlockStatement
	    { $$ = $1; }
	| measDoBlock
	    { $$ = $1; }
	| measIfBlock
	    { $$ = $1; }
	| measSelectBlock
	    { $$ = $1; }
	;

measDoBlock :
	  doStm enddoStm
	    { $$ = new measDoBlock($1, 0, $2); }
	| doStm measBlockItemList enddoStm
	    { $$ = new measDoBlock($1, $2, $3); }
	;

measIfBlock :
	  ifStm endifStm
	    { $$ = new measIfBlock($1, 0, 0, 0, $2); }
	| ifStm elseStm endifStm
	    { $$ = new measIfBlock($1, 0, $2, 0, $3); }
	| ifStm elseStm measBlockItemList endifStm
	    { $$ = new measIfBlock($1, 0, $2, $3, $4); }
	| ifStm measBlockItemList endifStm
	    { $$ = new measIfBlock($1, $2, 0, 0, $3); }
	| ifStm measBlockItemList elseStm endifStm
	    { $$ = new measIfBlock($1, $2, $3, 0, $4); }
	| ifStm measBlockItemList elseStm measBlockItemList endifStm
	    { $$ = new measIfBlock($1, $2, $3, $4, $5); }
	;

measSelectBlock :
	  selectStm measCaseBlockList endselStm
	    { $$ = new measSelectBlock($1, $2, 0, $3); }
	| selectStm measCaseBlockList measDefaultCaseBlock endselStm
	    { $$ = new measSelectBlock($1, $2, $3, $4); }
	;

measCaseBlockList :
	  measCaseBlock
	    { $$ = new std::list<measCaseBlock *>;
	      $$->push_back($1); }
	| measCaseBlockList measCaseBlock
	    { $$ = $1;
	      $$->push_back($2); }
	;

measCaseBlock :
	  caseStm endcasStm
	    { $$ = new measCaseBlock($1, 0, $2); }
	| caseStm measBlockItemList endcasStm
	    { $$ = new measCaseBlock($1, $2, $3); }
	;

measDefaultCaseBlock :
	  dftcasStm endcasStm
	    { $$ = new measDefaultCaseBlock($1, 0, $2); }
	| dftcasStm measBlockItemList endcasStm
	    { $$ = new measDefaultCaseBlock($1, $2, $3); }
	;

selectBlock :
	  selectStm caseBlockList endselStm
	    { $$ = new selectBlock($1, $2, 0, $3); }
	| selectStm caseBlockList defaultCaseBlock endselStm
	    { $$ = new selectBlock($1, $2, $3, $4); }
	;

caseBlockList :
	  caseBlock
	    { $$ = new std::list<caseBlock *>;
	      $$->push_back($1); }
	| caseBlockList caseBlock
	    { $$ = $1;
	      $$->push_back($2); }
	;

simreqtBlock :
	  simreqtStm endsimreqtStm
	    { $$ = new simreqtBlock($1, 0, $2); }
	| simreqtStm evalOrOutputList endsimreqtStm
	    { $$ = new simreqtBlock($1, $2, $3); }
	;

evalOrOutputList :
	  evalOrOutput
	    { $$ = new std::list<evalOrOutput *>;
	      $$->push_back($1); }
	| evalOrOutputList evalOrOutput
	    { $$ = $1;
	      $$->push_back($2); }
	;

evalOrOutput :
	  evalStm
	    { $$ = $1; }
	| outputStm
	    { $$ = $1; }
	;

xternBlock :
	  xternStm endxtnStm
	    { $$ = new xternBlock($1, 0, $2); }
	| xternStm extfilList endxtnStm
	    { $$ = new xternBlock($1, $2, $3); }
	;

extfilList :
	  extfilStm
	    { $$ = new std::list<extfilStm *>;
	      $$->push_back($1); }
	| extfilList extfilStm
	    { $$ = $1;
	      $$->push_back($2); }
	;

dmisFreeStatement :
	  jumpStm
	    { $$ = $1; }
	| aclratStm
	    { $$ = $1; }
	| algdefStm
	    { $$ = $1; }
	| assignStm
	    { $$ = $1; }
	| badtstStm
	    { $$ = $1; }
	| boundStm
	    { $$ = $1; }
	| clmpidStm
	    { $$ = $1; }
	| clmpsnStm
	    { $$ = $1; }
	| closeStm
	    { $$ = $1; }
	| cmpntgrpStm
	    { $$ = $1; }
	| cnfrmrulStm
	    { $$ = $1; }
	| constStm
	    { $$ = $1; }
	| crgdefStm
	    { $$ = $1; }
	| crmodeStm
	    { $$ = $1; }
	| crosclStm
	    { $$ = $1; }
	| crslctStm
	    { $$ = $1; }
	| cutcomStm
	    { $$ = $1; }
	| czoneStm
	    { $$ = $1; }
	| czslctStm
	    { $$ = $1; }
	| datdefStm
	    { $$ = $1; }
	| datsetStm
	    { $$ = $1; }
	| dattrgdefStm
	    { $$ = $1; }
	| declStm
	    { $$ = $1; }
	| decplStm
	    { $$ = $1; }
	| deleteStm
	    { $$ = $1; }
	| deviceStm
	    { $$ = $1; }
	| displyStm
	    { $$ = $1; }
	| dmehwStm
	    { $$ = $1; }
	| dmeidStm
	    { $$ = $1; }
	| dmeswStm
	    { $$ = $1; }
	| dmeswiStm
	    { $$ = $1; }
	| dmeswvStm
	    { $$ = $1; }
	| dmisOnStm
	    { $$ = $1; }
	| equateStm
	    { $$ = $1; }
	| errorStm
	    { $$ = $1; }
	| evalStm
	    { $$ = $1; }
	| extensStm
	    { $$ = $1; }
	| featArc1Stm
	    { $$ = $1; }
	| featArc2Stm
	    { $$ = $1; }
	| featCircleStm
	    { $$ = $1; }
	| featCompoundStm
	    { $$ = $1; }
	| featConeStm
	    { $$ = $1; }
	| featConradsegStm
	    { $$ = $1; }
	| featCparlnStm
	    { $$ = $1; }
	| featCylndrStm
	    { $$ = $1; }
	| featCylradsegStm
	    { $$ = $1; }
	| featEdgeptStm
	    { $$ = $1; }
	| featEllipsStm
	    { $$ = $1; }
	| featElongcylStm
	    { $$ = $1; }
	| featGcurveStm
	    { $$ = $1; }
	| featGeomStm
	    { $$ = $1; }
	| featGsurfStm
	    { $$ = $1; }
	| featLineStm
	    { $$ = $1; }
	| featObjectStm
	    { $$ = $1; }
	| featParplnStm
	    { $$ = $1; }
	| featPaternStm
	    { $$ = $1; }
	| featPlaneStm
	    { $$ = $1; }
	| featPointStm
	    { $$ = $1; }
	| featRctnglStm
	    { $$ = $1; }
	| featRevsurfStm
	    { $$ = $1; }
	| featSphereStm
	    { $$ = $1; }
	| featSphradsegStm
	    { $$ = $1; }
	| featSymplnStm
	    { $$ = $1; }
	| featTorradsegStm
	    { $$ = $1; }
	| featTorusStm
	    { $$ = $1; }
	| fedratStm
	    { $$ = $1; }
	| fildefStm
	    { $$ = $1; }
	| filnamStm
	    { $$ = $1; }
	| finposStm
	    { $$ = $1; }
	| fixtidStm
	    { $$ = $1; }
	| fixtsnStm
	    { $$ = $1; }
	| flyStm
	    { $$ = $1; }
	| fromStm
	    { $$ = $1; }
	| geoalgStm
	    { $$ = $1; }
	| geomStm
	    { $$ = $1; }
	| gohomeStm
	    { $$ = $1; }
	| gotoStm
	    { $$ = $1; }
	| groupStm
	    { $$ = $1; }
	| includStm
	    { $$ = $1; }
	| iteratStm
	    { $$ = $1; }
	| jumptoStm
	    { $$ = $1; }
	| keycharStm
	    { $$ = $1; }
	| litdefStm
	    { $$ = $1; }
	| locateStm
	    { $$ = $1; }
	| lotidStm
	    { $$ = $1; }
	| matdefStm
	    { $$ = $1; }
	| mfgdevStm
	    { $$ = $1; }
	| modeStm
	    { $$ = $1; }
	| obtainStm
	    { $$ = $1; }
	| openStm
	    { $$ = $1; }
	| operidStm
	    { $$ = $1; }
	| outputStm
	    { $$ = $1; }
	| pameasStm
	    { $$ = $1; }
	| partidStm
	    { $$ = $1; }
	| partrvStm
	    { $$ = $1; }
	| partsnStm
	    { $$ = $1; }
	| pathStm
	    { $$ = $1; }
	| planidStm
	    { $$ = $1; }
	| popStm
	    { $$ = $1; }
	| prcompStm
	    { $$ = $1; }
	| prevopStm
	    { $$ = $1; }
	| procidStm
	    { $$ = $1; }
	| promptStm
	    { $$ = $1; }
	| psthruStm
	    { $$ = $1; }
	| ptbuffStm
	    { $$ = $1; }
	| ptmeasStm
	    { $$ = $1; }
	| pushStm
	    { $$ = $1; }
	| qisdefStm
	    { $$ = $1; }
	| rapidStm
	    { $$ = $1; }
	| readStm
	    { $$ = $1; }
	| recallDatumStm
	    { $$ = $1; }
	| recallDMLStm
	    { $$ = $1; }
	| recallFeatureStm
	    { $$ = $1; }
	| recallRotaryTableStm
	    { $$ = $1; }
	| recallSensorStm
	    { $$ = $1; }
	| refmntStm
	    { $$ = $1; }
	| reportStm
	    { $$ = $1; }
	| resumeStm
	    { $$ = $1; }
	| rotabStm
	    { $$ = $1; }
	| rotateStm
	    { $$ = $1; }
	| rotdefStm
	    { $$ = $1; }
	| rotsetStm
	    { $$ = $1; }
	| saveDatumStm
	    { $$ = $1; }
	| saveDMLStm
	    { $$ = $1; }
	| saveFeatureStm
	    { $$ = $1; }
	| saveRotaryTableStm
	    { $$ = $1; }
	| saveSensorStm
	    { $$ = $1; }
	| scnmodStm
	    { $$ = $1; }
	| scnsetStm
	    { $$ = $1; }
	| sensorStm
	    { $$ = $1; }
	| snsdefStm
	    { $$ = $1; }
	| snsetStm
	    { $$ = $1; }
	| snsgrpStm
	    { $$ = $1; }
	| snslctStm
	    { $$ = $1; }
	| snsmntStm
	    { $$ = $1; }
	| tecompStm
	    { $$ = $1; }
	| textStm
	    { $$ = $1; }
	| thldefStm
	    { $$ = $1; }
	| tolAnglStm
	    { $$ = $1; }
	| tolAnglbStm
	    { $$ = $1; }
	| tolAnglrStm
	    { $$ = $1; }
	| tolAnglwrtStm
	    { $$ = $1; }
	| tolCirltyStm
	    { $$ = $1; }
	| tolComposStm
	    { $$ = $1; }
	| tolConcenStm
	    { $$ = $1; }
	| tolCortolStm
	    { $$ = $1; }
	| tolCproflStm
	    { $$ = $1; }
	| tolCprofsStm
	    { $$ = $1; }
	| tolCrnoutStm
	    { $$ = $1; }
	| tolCylctyStm
	    { $$ = $1; }
	| tolDiamStm
	    { $$ = $1; }
	| tolDistbStm
	    { $$ = $1; }
	| tolDistwrtStm
	    { $$ = $1; }
	| tolFlatStm
	    { $$ = $1; }
	| tolGtolStm
	    { $$ = $1; }
	| tolParlelStm
	    { $$ = $1; }
	| tolPerpStm
	    { $$ = $1; }
	| tolPosStm
	    { $$ = $1; }
	| tolProflStm
	    { $$ = $1; }
	| tolProfpStm
	    { $$ = $1; }
	| tolProfsStm
	    { $$ = $1; }
	| tolRadStm
	    { $$ = $1; }
	| tolStrghtStm
	    { $$ = $1; }
	| tolSymStm
	    { $$ = $1; }
	| tolTrnoutStm
	    { $$ = $1; }
	| tolUsetolStm
	    { $$ = $1; }
	| tolWidthStm
	    { $$ = $1; }
	| tooldfStm
	    { $$ = $1; }
	| transStm
	    { $$ = $1; }
	| uncertalgStm
	    { $$ = $1; }
	| uncertsetStm
	    { $$ = $1; }
	| unitsStm
	    { $$ = $1; }
	| valueStm
	    { $$ = $1; }
	| vformStm
	    { $$ = $1; }
	| windefStm
	    { $$ = $1; }
	| wkplanStm
	    { $$ = $1; }
	| wristStm
	    { $$ = $1; }
	| writeStm
	    { $$ = $1; }
	| xtractStm
	    { $$ = $1; }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

anyVal :
	  stringVal
	    { $$ = $1; }
	| boolVal
	    { $$ = $1; }
	| rentVal
	    { $$ = $1; }
	| vectorVal
	    { $$ = $1; }
	;

stringVal :
	  stringConst
	    { $$ = $1; }
	| stringFunc
	    { $$ = $1; }
	| stringVar
	    { $$ = $1; }
	;

boolVal :
	  boolConst
	    { $$ = $1; }
	| boolExpr
	    { $$ = $1; }
	| boolFunc
	    { $$ = $1; }
	| boolVar
	    { $$ = $1; }
	;

intVal :
	  intConst
	    { $$ = $1; }
	| intExpr
	    { $$ = $1; }
	| intFunc
	    { $$ = $1; }
	| intVar
	    { $$ = $1; }
	;

realVal :
	  realConst
	    { $$ = $1; }
	| realExpr
	    { $$ = $1; }
	| realFunc
	    { $$ = $1; }
	| realVar
	    { $$ = $1; }
	;

rentVal :
	  realVal
	    { $$ = $1; }
	| intVal
	    { $$ = $1; }
	;

vectorVal :
	  vectorFunc
	    { $$ = $1; }
	| vectorExpr
	    { $$ = $1; }
	| vectorVar
	    { $$ = $1; }
	;

stringConst :
	  CHARSTRING
	    { $$ = new stringConst($1); }
	;

boolConst :
	  TRUE
	    { $$ = new boolConst_TRUE(); }
	| FALSE
	    { $$ = new boolConst_FALSE(); }
	;

intConst :
	  INTSTRING
	    { $$ = new intConst($1); }
	;

realConst :
	  REALSTRING
	    { $$ = new realConst($1); }
	;

stringFunc :
	  stringFuncChr
	    { $$ = $1; }
	| stringFuncConcat
	    { $$ = $1; }
	| stringFuncElemnt
	    { $$ = $1; }
	| stringFuncLwc
	    { $$ = $1; }
	| stringFuncRpt
	    { $$ = $1; }
	| stringFuncScfeat
	    { $$ = $1; }
	| stringFuncScsns
	    { $$ = $1; }
	| stringFuncSdate
	    { $$ = $1; }
	| stringFuncSdatetime
	    { $$ = $1; }
	| stringFuncSelapsetime
	    { $$ = $1; }
	| stringFuncSerror
	    { $$ = $1; }
	| stringFuncSmode
	    { $$ = $1; }
	| stringFuncStime
	    { $$ = $1; }
	| stringFuncStr
	    { $$ = $1; }
	| stringFuncSubstr
	    { $$ = $1; }
	| stringFuncTrim
	    { $$ = $1; }
	| stringFuncUpc
	    { $$ = $1; }
	;

boolFunc :
	  boolFuncBadgt
	    { $$ = $1; }
	| boolFuncBadpt
	    { $$ = $1; }
	| boolFuncEof
	    { $$ = $1; }
	| boolFuncEoln
	    { $$ = $1; }
	| boolFuncExist
	    { $$ = $1; }
	;

intFunc :
	  intFuncAbs
	    { $$ = $1; }
	| intFuncIndx
	    { $$ = $1; }
	| intFuncInt
	    { $$ = $1; }
	| intFuncLen
	    { $$ = $1; }
	| intFuncMod
	    { $$ = $1; }
	| intFuncNint
	    { $$ = $1; }
	| intFuncOrd
	    { $$ = $1; }
	| intFuncPtdata
	    { $$ = $1; }
	| intFuncSensnotouch
	    { $$ = $1; }
	| intFuncSign
	    { $$ = $1; }
	| intFuncSiltch
	    { $$ = $1; }
	;

realFunc :
	  realFuncAbs
	    { $$ = $1; }
	| realFuncAcos
	    { $$ = $1; }
	| realFuncAsin
	    { $$ = $1; }
	| realFuncAtan
	    { $$ = $1; }
	| realFuncAtan2
	    { $$ = $1; }
	| realFuncCos
	    { $$ = $1; }
	| realFuncDble
	    { $$ = $1; }
	| realFuncDtor
	    { $$ = $1; }
	| realFuncExp
	    { $$ = $1; }
	| realFuncLn
	    { $$ = $1; }
	| realFuncLog
	    { $$ = $1; }
	| realFuncMn
	    { $$ = $1; }
	| realFuncMod
	    { $$ = $1; }
	| realFuncMx
	    { $$ = $1; }
	| realFuncQtemp
	    { $$ = $1; }
	| realFuncRand
	    { $$ = $1; }
	| realFuncRl
	    { $$ = $1; }
	| realFuncRtod
	    { $$ = $1; }
	| realFuncSign
	    { $$ = $1; }
	| realFuncSin
	    { $$ = $1; }
	| realFuncSqrt
	    { $$ = $1; }
	| realFuncTan
	    { $$ = $1; }
	| realFuncVal
	    { $$ = $1; }
	| realFuncVdot
	    { $$ = $1; }
	| realFuncVecX
	    { $$ = $1; }
	| realFuncVecY
	    { $$ = $1; }
	| realFuncVecZ
	    { $$ = $1; }
	| realFuncVmag
	    { $$ = $1; }
	;

vectorFunc :
	  vectorFuncVcart
	    { $$ = $1; }
	| vectorFuncVcross
	    { $$ = $1; }
	| vectorFuncVmcs
	    { $$ = $1; }
	| vectorFuncVpcs
	    { $$ = $1; }
	| vectorFuncVpol
	    { $$ = $1; }
	| vectorFuncVunit
	    { $$ = $1; }
	;

boolExpr :
	  LPAREN boolVal RPAREN
	    { $$ = new boolExpr_PAREN($2); }
	| stringRel
	    { $$ = new boolExpr_stringRel($1); }
	| boolRel
	    { $$ = new boolExpr_boolRel($1); }
	| intRel
	    { $$ = new boolExpr_intRel($1); }
	| realRel
	    { $$ = new boolExpr_realRel($1); }
	| vectorRel
	    { $$ = new boolExpr_vectorRel($1); }
	;

stringRel :
	  stringVal EQ stringVal
	    { $$ = new stringRel_EQ($1, $3); }
	| stringVal NE stringVal
	    { $$ = new stringRel_NE($1, $3); }
	| stringVal LT stringVal
	    { $$ = new stringRel_LT($1, $3); }
	| stringVal LE stringVal
	    { $$ = new stringRel_LE($1, $3); }
	| stringVal GT stringVal
	    { $$ = new stringRel_GT($1, $3); }
	| stringVal GE stringVal
	    { $$ = new stringRel_GE($1, $3); }
	;

boolRel :
	  NOT boolVal
	    { $$ = new boolRel_NOT($2); }
	| boolVal AND boolVal
	    { $$ = new boolRel_AND($1, $3); }
	| boolVal OR boolVal
	    { $$ = new boolRel_OR($1, $3); }
	| boolVal EQ boolVal
	    { $$ = new boolRel_EQ($1, $3); }
	| boolVal NE boolVal
	    { $$ = new boolRel_NE($1, $3); }
	;

intRel :
	  intVal EQ intVal
	    { $$ = new intRel_EQ($1, $3); }
	| intVal NE intVal
	    { $$ = new intRel_NE($1, $3); }
	| intVal LT intVal
	    { $$ = new intRel_LT($1, $3); }
	| intVal LE intVal
	    { $$ = new intRel_LE($1, $3); }
	| intVal GT intVal
	    { $$ = new intRel_GT($1, $3); }
	| intVal GE intVal
	    { $$ = new intRel_GE($1, $3); }
	;

intExpr :
	  LPAREN intVal RPAREN
	    { $$ = new intExprParen($2); }
	| intVal PLUS intVal
	    { $$ = new intExprPlus($1, $3); }
	| intVal MINUS intVal
	    { $$ = new intExprMinus($1, $3); }
	| intVal TIMES intVal
	    { $$ = new intExprTimes($1, $3); }
	| intVal SLASH intVal
	    { $$ = new intExprDivid($1, $3); }
	| intVal POWER intVal
	    { $$ = new intExprPower($1, $3); }
	| MINUS intVal
	    { $$ = new intExprNegat($2); }
	| PLUS intVal
	    { $$ = new intExprPosit($2); }
	;

realRel :
	  realRealRel
	    { $$ = $1; }
	| realIntRel
	    { $$ = $1; }
	| intRealRel
	    { $$ = $1; }
	;

realRealRel :
	  realVal EQ realVal
	    { $$ = new realRealRel_EQ($1, $3); }
	| realVal NE realVal
	    { $$ = new realRealRel_NE($1, $3); }
	| realVal LT realVal
	    { $$ = new realRealRel_LT($1, $3); }
	| realVal LE realVal
	    { $$ = new realRealRel_LE($1, $3); }
	| realVal GT realVal
	    { $$ = new realRealRel_GT($1, $3); }
	| realVal GE realVal
	    { $$ = new realRealRel_GE($1, $3); }
	;

intRealRel :
	  intVal EQ realVal
	    { $$ = new intRealRel_EQ($1, $3); }
	| intVal NE realVal
	    { $$ = new intRealRel_NE($1, $3); }
	| intVal LT realVal
	    { $$ = new intRealRel_LT($1, $3); }
	| intVal LE realVal
	    { $$ = new intRealRel_LE($1, $3); }
	| intVal GT realVal
	    { $$ = new intRealRel_GT($1, $3); }
	| intVal GE realVal
	    { $$ = new intRealRel_GE($1, $3); }
	;

realIntRel :
	  realVal EQ intVal
	    { $$ = new realIntRel_EQ($1, $3); }
	| realVal NE intVal
	    { $$ = new realIntRel_NE($1, $3); }
	| realVal LT intVal
	    { $$ = new realIntRel_LT($1, $3); }
	| realVal LE intVal
	    { $$ = new realIntRel_LE($1, $3); }
	| realVal GT intVal
	    { $$ = new realIntRel_GT($1, $3); }
	| realVal GE intVal
	    { $$ = new realIntRel_GE($1, $3); }
	;

vectorRel :
	  vectorVal EQ vectorVal
	    { $$ = new vectorRel_EQ($1, $3); }
	| vectorVal NE vectorVal
	    { $$ = new vectorRel_NE($1, $3); }
	;

realExpr :
	  realExprParen
	    { $$ = $1; }
	| realRealExpr
	    { $$ = $1; }
	| realIntExpr
	    { $$ = $1; }
	| intRealExpr
	    { $$ = $1; }
	| realExprNegat
	    { $$ = $1; }
	| realExprPosit
	    { $$ = $1; }
	;

realExprParen :
	  LPAREN realVal RPAREN
	    { $$ = new realExprParen($2); }
	;

realExprNegat :
	  MINUS realVal
	    { $$ = new realExprNegat($2); }
	;

realExprPosit :
	  PLUS realVal
	    { $$ = new realExprPosit($2); }
	;

realRealExpr :
	  realVal PLUS realVal
	    { $$ = new realRealExpr_PLUS($1, $3); }
	| realVal MINUS realVal
	    { $$ = new realRealExpr_MINUS($1, $3); }
	| realVal TIMES realVal
	    { $$ = new realRealExpr_TIMES($1, $3); }
	| realVal SLASH realVal
	    { $$ = new realRealExpr_DIVID($1, $3); }
	| realVal POWER realVal
	    { $$ = new realRealExpr_POWER($1, $3); }
	;

realIntExpr :
	  realVal PLUS intVal
	    { $$ = new realIntExpr_PLUS($1, $3); }
	| realVal MINUS intVal
	    { $$ = new realIntExpr_MINUS($1, $3); }
	| realVal TIMES intVal
	    { $$ = new realIntExpr_TIMES($1, $3); }
	| realVal SLASH intVal
	    { $$ = new realIntExpr_DIVID($1, $3); }
	| realVal POWER intVal
	    { $$ = new realIntExpr_POWER($1, $3); }
	;

intRealExpr :
	  intVal PLUS realVal
	    { $$ = new intRealExpr_PLUS($1, $3); }
	| intVal MINUS realVal
	    { $$ = new intRealExpr_MINUS($1, $3); }
	| intVal TIMES realVal
	    { $$ = new intRealExpr_TIMES($1, $3); }
	| intVal SLASH realVal
	    { $$ = new intRealExpr_DIVID($1, $3); }
	| intVal POWER realVal
	    { $$ = new intRealExpr_POWER($1, $3); }
	;

vectorExpr :
	  LPAREN vectorVal RPAREN
	    { $$ = new vecExpr_PAREN($2); }
	| vectorVal PLUS vectorVal
	    { $$ = new vecExpr_PLUS($1, $3); }
	| vectorVal MINUS vectorVal
	    { $$ = new vecExpr_MINUS($1, $3); }
	| vectorVal SLASH realVal
	    { $$ = new vecRealExpr_DIVID($1, $3); }
	| vectorVal SLASH intVal
	    { $$ = new vecIntExpr_DIVID($1, $3); }
	| vectorVal TIMES realVal
	    { $$ = new vecRealExpr_TIMES($1, $3); }
	| realVal TIMES vectorVal
	    { $$ = new realVecExpr_TIMES($1, $3); }
	| vectorVal TIMES intVal
	    { $$ = new vecIntExpr_TIMES($1, $3); }
	| intVal TIMES vectorVal
	    { $$ = new intVecExpr_TIMES($1, $3); }
	;

boolVar :
	  BOOLVARNAME
	    { $$ = new boolVar($1, 0); }
	| BOOLVARNAME arrayIndex
	    { $$ = new boolVar($1, $2); }
	;

intVar :
	  INTVARNAME
	    { $$ = new intVar($1, 0); }
	| INTVARNAME arrayIndex
	    { $$ = new intVar($1, $2); }
	;

realVar :
	  REALVARNAME
	    { $$ = new realVar($1, 0); }
	| REALVARNAME arrayIndex
	    { $$ = new realVar($1, $2); }
	;

stringVar :
	  STRINGVARNAME
	    { $$ = new stringVar($1, 0); }
	| STRINGVARNAME arrayIndex
	    { $$ = new stringVar($1, $2); }
	;

vectorVar :
	  VECTORVARNAME
	    { $$ = new vectorVar($1, 0); }
	| VECTORVARNAME arrayIndex
	    { $$ = new vectorVar($1, $2); }
	;

arrayIndex :
	  LBOX intList RBOX
	    { $$ = new arrayIndex($2); }
	;

stringFuncChr :
	  CHR LPAREN intVal RPAREN
	    { $$ = new stringFuncChr($3); }
	;

stringFuncConcat :
	  CONCAT LPAREN stringVal C stringList RPAREN
	    { $$ = new stringFuncConcat($3, $5); }
	;

stringFuncElemnt :
	  ELEMNT LPAREN intVal C stringVal C stringVal RPAREN
	    { $$ = new stringFuncElemnt($3, $5, $7); }
	;

stringFuncScfeat :
	  SCFEAT LPAREN RPAREN
	    { $$ = new stringFuncScfeat(); }
	;

stringFuncScsns :
	  SCSNS LPAREN RPAREN
	    { $$ = new stringFuncScsns(); }
	;

stringFuncSdate :
	  SDATE LPAREN RPAREN
	    { $$ = new stringFuncSdate(); }
	;

stringFuncSerror :
	  SERROR LPAREN RPAREN
	    { $$ = new stringFuncSerror(); }
	;

stringFuncSmode :
	  SMODE LPAREN RPAREN
	    { $$ = new stringFuncSmode(); }
	;

stringFuncStime :
	  STIME LPAREN RPAREN
	    { $$ = new stringFuncStime(); }
	;

stringFuncSdatetime :
	  SDATETIME LPAREN RPAREN
	    { $$ = new stringFuncSdatetime(); }
	;

stringFuncRpt :
	  RPT LPAREN stringVal C intVal RPAREN
	    { $$ = new stringFuncRpt($3, $5); }
	;

stringFuncSelapsetime :
	  SELAPSETIME LPAREN stringVal C stringVal C longShort RPAREN
	    { $$ = new stringFuncSelapsetime($3, $5, $7); }
	;

longShort :
	  LONG
	    { $$ = new longShort_LONG(); }
	| SHORT
	    { $$ = new longShort_SHORT(); }
	;

stringFuncStr :
	  STR LPAREN rentVal RPAREN
	    { $$ = new stringFuncStr($3, 0); }
	| STR LPAREN rentVal strVar7 RPAREN
	    { $$ = new stringFuncStr($3, $4); }
	;

strVar7 :
	  C intVal
	    { $$ = new strVar7($2, 0); }
	| C intVal COLON intVal
	    { $$ = new strVar7($2, $4); }
	;

stringFuncLwc :
	  LWC LPAREN stringVal RPAREN
	    { $$ = new stringFuncLwc($3); }
	;

stringFuncTrim :
	  TRIM LPAREN stringVal RPAREN
	    { $$ = new stringFuncTrim($3); }
	;

stringFuncUpc :
	  UPC LPAREN stringVal RPAREN
	    { $$ = new stringFuncUpc($3); }
	;

stringFuncSubstr :
	  SUBSTR LPAREN stringVal C intVal RPAREN
	    { $$ = new stringFuncSubstr($3, $5, 0); }
	| SUBSTR LPAREN stringVal C intVal C intVal RPAREN
	    { $$ = new stringFuncSubstr($3, $5, $7); }
	;

boolFuncBadgt :
	  BADGT LPAREN RPAREN
	    { $$ = new boolFuncBadgt(); }
	;

boolFuncExist :
	  EXIST existLParen sLabel existRParen
	    { $$ = new boolFuncExist_sLabel($3); }
	| EXIST existLParen saLabel existRParen
	    { $$ = new boolFuncExist_saLabel($3); }
	| EXIST existLParen dLabel existRParen
	    { $$ = new boolFuncExist_dLabel($3); }
	| EXIST existLParen daLabel existRParen
	    { $$ = new boolFuncExist_daLabel($3); }
	| EXIST existLParen didLabel existRParen
	    { $$ = new boolFuncExist_didLabel($3); }
	| EXIST existLParen fLabel existRParen
	    { $$ = new boolFuncExist_fLabel($3); }
	| EXIST existLParen faLabel existRParen
	    { $$ = new boolFuncExist_faLabel($3); }
	;

boolFuncBadpt :
	  BADPT LPAREN RPAREN
	    { $$ = new boolFuncBadpt(); }
	;

boolFuncEof :
	  WWEOF LPAREN didLabel RPAREN
	    { $$ = new boolFuncEof($3); }
	;

boolFuncEoln :
	  EOLN LPAREN didLabel RPAREN
	    { $$ = new boolFuncEoln($3); }
	;

intFuncAbs :
	  ABS LPAREN intVal RPAREN
	    { $$ = new intFuncAbs($3); }
	;

intFuncIndx :
	  INDX LPAREN stringVal C stringVal RPAREN
	    { $$ = new intFuncIndx($3, $5); }
	;

intFuncInt :
	  INT LPAREN realVal RPAREN
	    { $$ = new intFuncInt($3); }
	;

intFuncLen :
	  LEN LPAREN stringVal RPAREN
	    { $$ = new intFuncLen($3); }
	;

intFuncMod :
	  MOD LPAREN intVal C intVal RPAREN
	    { $$ = new intFuncMod($3, $5); }
	;

intFuncNint :
	  NINT LPAREN realVal RPAREN
	    { $$ = new intFuncNint($3); }
	;

intFuncOrd :
	  ORD LPAREN stringVal RPAREN
	    { $$ = new intFuncOrd_stringVal($3); }
	| ORD LPAREN boolVal RPAREN
	    { $$ = new intFuncOrd_boolVal($3); }
	;

intFuncPtdata :
	  PTDATA LPAREN faLabel RPAREN
	    { $$ = new intFuncPtdata($3); }
	;

intFuncSensnotouch :
	  SENSNOTOUCH LPAREN RPAREN
	    { $$ = new intFuncSensnotouch(); }
	;

intFuncSign :
	  SIGN LPAREN intVal C intVal RPAREN
	    { $$ = new intFuncSign($3, $5); }
	;

intFuncSiltch :
	  SILTCH LPAREN RPAREN
	    { $$ = new intFuncSiltch(); }
	;

realFuncAbs :
	  ABS LPAREN realVal RPAREN
	    { $$ = new realFuncAbs($3); }
	;

realFuncCos :
	  COS LPAREN rentVal RPAREN
	    { $$ = new realFuncCos($3); }
	;

realFuncDble :
	  DBLE LPAREN rentVal RPAREN
	    { $$ = new realFuncDble($3); }
	;

realFuncDtor :
	  DTOR LPAREN rentVal RPAREN
	    { $$ = new realFuncDtor($3); }
	;

realFuncExp :
	  EXP LPAREN rentVal RPAREN
	    { $$ = new realFuncExp($3); }
	;

realFuncLn :
	  LN LPAREN rentVal RPAREN
	    { $$ = new realFuncLn($3); }
	;

realFuncLog :
	  LOG LPAREN rentVal RPAREN
	    { $$ = new realFuncLog($3); }
	;

realFuncMn :
	  MN LPAREN rentVal C rentList RPAREN
	    { $$ = new realFuncMn($3, $5); }
	;

realFuncMod :
	  MOD LPAREN realVal C realVal RPAREN
	    { $$ = new realFuncMod($3, $5); }
	;

realFuncMx :
	  MX LPAREN rentVal C rentList RPAREN
	    { $$ = new realFuncMx($3, $5); }
	;

realFuncQtemp :
	  QTEMP LPAREN stringVal RPAREN
	    { $$ = new realFuncQtemp($3); }
	;

realFuncRand :
	  RAND LPAREN RPAREN
	    { $$ = new realFuncRand(0); }
	| RAND LPAREN rentVal RPAREN
	    { $$ = new realFuncRand($3); }
	;

realFuncRl :
	  RL LPAREN rentVal RPAREN
	    { $$ = new realFuncRl($3); }
	;

realFuncRtod :
	  RTOD LPAREN rentVal RPAREN
	    { $$ = new realFuncRtod($3); }
	;

realFuncSign :
	  SIGN LPAREN realVal C realVal RPAREN
	    { $$ = new realFuncSign($3, $5); }
	;

realFuncSin :
	  SIN LPAREN rentVal RPAREN
	    { $$ = new realFuncSin($3); }
	;

realFuncSqrt :
	  SQRT LPAREN rentVal RPAREN
	    { $$ = new realFuncSqrt($3); }
	;

realFuncTan :
	  TAN LPAREN rentVal RPAREN
	    { $$ = new realFuncTan($3); }
	;

realFuncVal :
	  VAL LPAREN stringVal RPAREN
	    { $$ = new realFuncVal($3); }
	;

realFuncVdot :
	  VDOT LPAREN vectorVal C vectorVal RPAREN
	    { $$ = new realFuncVdot($3, $5); }
	;

realFuncVecX :
	  VECX LPAREN vectorVal RPAREN
	    { $$ = new realFuncVecX($3); }
	;

realFuncVecY :
	  VECY LPAREN vectorVal RPAREN
	    { $$ = new realFuncVecY($3); }
	;

realFuncVecZ :
	  VECZ LPAREN vectorVal RPAREN
	    { $$ = new realFuncVecZ($3); }
	;

realFuncVmag :
	  VMAG LPAREN vectorVal RPAREN
	    { $$ = new realFuncVmag($3); }
	;

realFuncAcos :
	  ACOS LPAREN rentVal RPAREN
	    { $$ = new realFuncAcos($3); }
	;

realFuncAsin :
	  ASIN LPAREN rentVal RPAREN
	    { $$ = new realFuncAsin($3); }
	;

realFuncAtan :
	  ATAN LPAREN rentVal RPAREN
	    { $$ = new realFuncAtan($3); }
	;

realFuncAtan2 :
	  ATAN2 LPAREN rentVal C rentVal RPAREN
	    { $$ = new realFuncAtan2($3, $5); }
	;

vectorFuncVcart :
	  VCART LPAREN rentVal C rentVal C rentVal RPAREN
	    { $$ = new vectorFuncVcart($3, $5, $7); }
	;

vectorFuncVcross :
	  VCROSS LPAREN vectorVal C vectorVal RPAREN
	    { $$ = new vectorFuncVcross($3, $5); }
	;

vectorFuncVmcs :
	  VMCS LPAREN vectorVal RPAREN
	    { $$ = new vectorFuncVmcs($3); }
	;

vectorFuncVpcs :
	  VPCS LPAREN vectorVal RPAREN
	    { $$ = new vectorFuncVpcs($3); }
	;

vectorFuncVpol :
	  VPOL LPAREN rentVal C angle C rentVal RPAREN
	    { $$ = new vectorFuncVpol($3, $5, $7); }
	;

vectorFuncVunit :
	  VUNIT LPAREN vectorVal RPAREN
	    { $$ = new vectorFuncVunit($3); }
	;

labelNameConst :
	  LPAREN LBLNAME RPAREN
	    { $$ = new labelNameConst($2); }
	;

labelName :
	  labelNameConst
	    { $$ = new labelNameCon($1); }
	| LPAREN AT stringVar RPAREN
	    { $$ = new labelNameAt($3);
	      aLabelType = 0; }
	;

datumLabel :
	  dLabel
	    { $$ = $1; }
	| daLabel
	    { $$ = $1; }
	;

datumLabel1 :
	  dLabel
	    { $$ = $1; }
	| daLabel
	    { $$ = $1; }
	;

datumLabel2 :
	  dLabel
	    { $$ = $1; }
	| daLabel
	    { $$ = $1; }
	;

featureLabel :
	  fLabel
	    { $$ = $1; }
	| faLabel
	    { $$ = $1; }
	;

featureLabel1 :
	  fLabel
	    { $$ = $1; }
	| faLabel
	    { $$ = $1; }
	;

qisLabel :
	  ccLabel
	    { $$ = $1; }
	| ciLabel
	    { $$ = $1; }
	| csLabel
	    { $$ = $1; }
	| diLabel
	    { $$ = $1; }
	| dsLabel
	    { $$ = $1; }
	| dvLabel
	    { $$ = $1; }
	| fiLabel
	    { $$ = $1; }
	| fsLabel
	    { $$ = $1; }
	| liLabel
	    { $$ = $1; }
	| mdLabel
	    { $$ = $1; }
	| opLabel
	    { $$ = $1; }
	| pcLabel
	    { $$ = $1; }
	| plLabel
	    { $$ = $1; }
	| pnLabel
	    { $$ = $1; }
	| prLabel
	    { $$ = $1; }
	| psLabel
	    { $$ = $1; }
	| pvLabel
	    { $$ = $1; }
	| qLabel
	    { $$ = $1; }
	| tlLabel
	    { $$ = $1; }
	;

sensorLabel1 :
	  sLabel
	    { $$ = $1; }
	| saLabel
	    { $$ = $1; }
	;

sensorLabel2 :
	  sLabel
	    { $$ = $1; }
	| saLabel
	    { $$ = $1; }
	;

toleranceLabel :
	  tLabel
	    { $$ = $1; }
	| taLabel
	    { $$ = $1; }
	;

ccLabel :
	  CC labelName
	  { $$ = new ccLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(CC, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

ciLabel :
	  CI labelName
	  { $$ = new ciLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(CI, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

crLabel :
	  CR labelName
	  { $$ = new crLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(CR, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

crLabelConst :
	  CR labelNameConst
	  { $$ = new crLabelConst($2);
	    doLabel(CR, $2->get_string()); }
	;

csLabel :
	  CS labelName
	  { $$ = new csLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(CS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

czLabel :
	  CZ labelName
	  { $$ = new czLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(CZ, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

dLabel :
	  D labelName
	  { $$ = new dLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(D, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

daLabel :
	  DA labelName
	  { $$ = new daLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(D, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

datLabel :
	  DAT labelName
	  { $$ = new datLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DAT, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

dattrgLabel :
	  DATTRG labelName
	  { $$ = new dattrgLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DATTRG, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

diLabel :
	  DI labelName
	  { $$ = new diLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DI, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

didLabel :
	  DID labelName
	  { $$ = new didLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DID, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

drLabel :
	  DR labelName
	  { $$ = new drLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DR, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

dsLabel :
	  DS labelName
	  { $$ = new dsLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

dvLabel :
	  DV labelName
	  { $$ = new dvLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(DV, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

fLabel :
	  F labelName
	  { $$ = new fLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(F, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

faLabel :
	  FA labelName
	  { $$ = new faLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(F, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

fiLabel :
	  FI labelName
	  { $$ = new fiLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(FI, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

fsLabel :
	  FS labelName
	  { $$ = new fsLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(FS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

gLabel :
	  G labelName
	  { $$ = new gLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(G, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

gsaLabel :
	  GSA labelName
	  { $$ = new gsaLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(GSA, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

kcLabel :
	  KC labelName
	  { $$ = new kcLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(KC, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

kcaLabel :
	  KCA labelName
	  { $$ = new kcaLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(KC, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

liLabel :
	  LI labelName
	  { $$ = new liLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(LI, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

mLabel :
	  M labelName
	  { $$ = new mLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(M, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

mLabelConst :
	  M labelNameConst
	  { $$ = new mLabelConst($2);
	    doLabel(M, $2->get_string()); }
	;

maLabel :
	  MA labelName
	  { $$ = new maLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(MA, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

mdLabel :
	  MD labelName
	  { $$ = new mdLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(MD, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

opLabel :
	  OP labelName
	  { $$ = new opLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(OP, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

pLabel :
	  P labelName
	  { $$ = new pLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(P, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

pcLabel :
	  PC labelName
	  { $$ = new pcLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PC, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

plLabel :
	  PL labelName
	  { $$ = new plLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PL, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

pnLabel :
	  PN labelName
	  { $$ = new pnLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PN, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

prLabel :
	  PR labelName
	  { $$ = new prLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PR, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

psLabel :
	  PS labelName
	  { $$ = new psLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

pvLabel :
	  PV labelName
	  { $$ = new pvLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(PV, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

qLabel :
	  Q labelName
	  { $$ = new qLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(Q, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

rLabel :
	  R labelName
	  { $$ = new rLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(R, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

rmLabel :
	  RM labelName
	  { $$ = new rmLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(RM, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

rtLabel :
	  RT labelName
	  { $$ = new rtLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(RT, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

sLabel :
	  S labelName
	  { $$ = new sLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(S, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

saLabel :
	  SA labelName
	  { $$ = new saLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(S, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

seLabel :
	  SE labelName
	  { $$ = new seLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SE, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

sgLabel :
	  SG labelName
	  { $$ = new sgLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SG, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

sgsLabel :
	  SGS labelName
	  { $$ = new sgsLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SGS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

srLabel :
	  SR labelName
	  { $$ = new srLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SR, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

sraLabel :
	  SRA labelName
	  { $$ = new sraLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SRA, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

ssLabel :
	  SS labelName
	  { $$ = new ssLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SS, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

stLabel :
	  ST labelName
	  { $$ = new stLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(ST, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

swLabel :
	  SW labelName
	  { $$ = new swLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SW, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

sxLabel :
	  SX labelName
	  { $$ = new sxLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(SX, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

tLabel :
	  T labelName
	  { $$ = new tLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(T, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

taLabel :
	  TA labelName
	  { $$ = new taLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(T, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

thLabel :
	  TH labelName
	  { $$ = new thLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(TH, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

tlLabel :
	  TL labelName
	  { $$ = new tlLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(TL, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

uLabel :
	  U labelName
	  { $$ = new uLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(U, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

vLabel :
	  V labelName
	  { $$ = new vLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(V, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

vaLabel :
	  VA labelName
	  { $$ = new vaLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(VA, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

vfLabel :
	  VF labelName
	  { $$ = new vfLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(VF, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

vlLabel :
	  VL labelName
	  { $$ = new vlLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(VL, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

vwLabel :
	  VW labelName
	  { $$ = new vwLabel($2);
	    if (isA($2, labelNameCon))
	      doLabel(VW, (dynamic_cast<labelNameCon *>($2))->
	                    get_labelNameConst()->get_string());
	  }
	;

jumpLabel :
	  labelName
	    { $$ = $1; }
	;

param :
	  anyVal
	    { $$ = $1; }
	;

paramList :
	  param
	    { $$ = new std::list<param *>;
	      $$->push_back($1); }
	| paramList C param
	    { $$ = $1;
	      $$->push_back($3); }
	;

stringList :
	  stringVal
	    { $$ = new std::list<stringVal *>;
	      $$->push_back($1); }
	| stringList C stringVal
	    { $$ = $1;
	      $$->push_back($3); }
	;

intList :
	  intVal
	    { $$ = new std::list<intVal *>;
	      $$->push_back($1); }
	| intList C intVal
	    { $$ = $1;
	      $$->push_back($3); }
	;

leftRight :
	  LEFT
	    { $$ = new leftRight_LEFT(); }
	| RIGHT
	    { $$ = new leftRight_RIGHT(); }
	;

rentList :
	  rentVal
	    { $$ = new std::list<rentVal *>;
	      $$->push_back($1); }
	| rentList C rentVal
	    { $$ = $1;
	      $$->push_back($3); }
	;

featureList :
	  featureLabel
	    { $$ = new std::list<featureLabel *>;
	      $$->push_back($1); }
	| featureList C featureLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

featureNominalList :
	  fLabel
	    { $$ = new std::list<fLabel *>;
	      $$->push_back($1); }
	| featureNominalList C fLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

featureActualList :
	  faLabel
	    { $$ = new std::list<faLabel *>;
	      $$->push_back($1); }
	| featureActualList C faLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

noParseStmList :
	  noParseStm
	    { $$ = new std::list<noParseStm *>;
	      $$->push_back($1); }
	| noParseStmList noParseStm
	    { $$ = $1;
	      $$->push_back($2); }
	;

dattrgList :
	  dattrgLabel
	    { $$ = new std::list<dattrgLabel *>;
	      $$->push_back($1); }
	| dattrgList C dattrgLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

saLabelList :
	  saLabel
	    { $$ = new std::list<saLabel *>;
	      $$->push_back($1); }
	| saLabelList C saLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

stackElementList :
	  stackElement
	    { $$ = new std::list<stackElement *>;
	      $$->push_back($1); }
	| stackElementList C stackElement
	    { $$ = $1;
	      $$->push_back($3); }
	;

tLabelList :
	  tLabel
	    { $$ = new std::list<tLabel *>;
	      $$->push_back($1); }
	| tLabelList C tLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

taLabelList :
	  taLabel
	    { $$ = new std::list<taLabel *>;
	      $$->push_back($1); }
	| taLabelList C taLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

pointRange :
	  LBOX intVal RBOX
	    { $$ = new pointRange($2, 0); }
	| LBOX intVal C intVal RBOX
	    { $$ = new pointRange($2, $4); }
	;

indexedFeature :
	  featureLabel pointRange
	    { $$ = new indexedFeature($1, $2); }
	;

indexedFeatureList :
	  indexedFeature
	    { $$ = new std::list<indexedFeature *>;
	      $$->push_back($1); }
	| indexedFeatureList C indexedFeature
	    { $$ = $1;
	      $$->push_back($3); }
	;

versionTag :
	  REALSTRING
	    { $$ = new versionTag($1); }
	;

angle :
	  rentVal
	    { $$ = $1; }
	| angleDms
	    { $$ = $1; }
	;

angleDms :
	  INTSTRING COLON INTSTRING
	    { $$ = new angleDms($1, $3, 0); }
	| INTSTRING COLON INTSTRING COLON secondsString
	    { $$ = new angleDms($1, $3, $5); }
	;

noParseStm :
	  CHARSTRING endOfLine
	    { $$ = new noParseStm($1); }
	;

secondsString :
	  REALSTRING
	    { $$ = new secondsString_REALSTRING($1); }
	| INTSTRING
	    { $$ = new secondsString_INTSTRING($1); }
	;

vector :
	  rentVal C rentVal C rentVal
	    { $$ = new vector($1, $3, $5); }
	;

matrix :
	  rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal
	    { $$ = new matrix($1, $3, $5, $7, $9, $11, $13, $15, $17, $19, $21, $23); }
	;

rwVar :
	  stringVar
	    { $$ = $1; }
	| boolVar
	    { $$ = $1; }
	| intVar
	    { $$ = $1; }
	| realVar
	    { $$ = $1; }
	| vectorVar
	    { $$ = $1; }
	;

rwFormat :
	  COLON intVal
	    { $$ = new rwFormat($2, 0); }
	| COLON intVal COLON intVal
	    { $$ = new rwFormat($2, $4); }
	;

device :
	  PRINT
	    { $$ = new device_PRINT(); }
	| TERM
	    { $$ = new device_TERM(); }
	| COMM
	    { $$ = new device_COMM(); }
	| STOR
	    { $$ = new device_STOR(); }
	;

state :
	  ON
	    { $$ = new state_ON(); }
	| OFF
	    { $$ = new state_OFF(); }
	;

coordType :
	  CART
	    { $$ = new coordType_CART(); }
	| POL
	    { $$ = new coordType_POL(); }
	;

cartPoint :
	  CART C rentVal C rentVal C rentVal
	    { $$ = new cartPoint($3, $5, $7); }
	;

impCartPoint :
	  rentVal C rentVal C rentVal
	    { $$ = new impCartPoint($1, $3, $5); }
	;

impCartPointList :
	  impCartPoint
	    { $$ = new std::list<impCartPoint *>;
	      $$->push_back($1); }
	| impCartPointList C impCartPoint
	    { $$ = $1;
	      $$->push_back($3); }
	;

polPoint :
	  POL C rentVal C rentVal C rentVal
	    { $$ = new polPoint($3, $5, $7); }
	;

impPolPoint :
	  rentVal C rentVal C rentVal
	    { $$ = new impPolPoint($1, $3, $5); }
	;

impPolPointList :
	  impPolPoint
	    { $$ = new std::list<impPolPoint *>;
	      $$->push_back($1); }
	| impPolPointList C impPolPoint
	    { $$ = $1;
	      $$->push_back($3); }
	;

typePoint :
	  cartPoint
	    { $$ = $1; }
	| polPoint
	    { $$ = $1; }
	;

pointVecCart :
	  rentVal C rentVal C rentVal C rentVal C rentVal C rentVal
	    { $$ = new pointVecCart($1, $3, $5, $7, $9, $11); }
	;

pointVecCartList :
	  pointVecCart
	    { $$ = new std::list<pointVecCart *>;
	      $$->push_back($1); }
	| pointVecCartList C pointVecCart
	    { $$ = $1;
	      $$->push_back($3); }
	;

pointVecPol :
	  rentVal C rentVal C rentVal C rentVal C rentVal C rentVal
	    { $$ = new pointVecPol($1, $3, $5, $7, $9, $11); }
	;

pointVecPolList :
	  pointVecPol
	    { $$ = new std::list<pointVecPol *>;
	      $$->push_back($1); }
	| pointVecPolList C pointVecPol
	    { $$ = $1;
	      $$->push_back($3); }
	;

axis :
	  XAXIS
	    { $$ = new axis_XAXIS(); }
	| YAXIS
	    { $$ = new axis_YAXIS(); }
	| ZAXIS
	    { $$ = new axis_ZAXIS(); }
	;

orig :
	  XORIG
	    { $$ = new orig_XORIG(); }
	| YORIG
	    { $$ = new orig_YORIG(); }
	| ZORIG
	    { $$ = new orig_ZORIG(); }
	;

dir :
	  posDir
	    { $$ = $1; }
	| negDir
	    { $$ = $1; }
	;

posDir :
	  XDIR
	    { $$ = new posDir_XDIR(); }
	| YDIR
	    { $$ = new posDir_YDIR(); }
	| ZDIR
	    { $$ = new posDir_ZDIR(); }
	;

negDir :
	  MINUSXDIR
	    { $$ = new negDir_MINUSXDIR(); }
	| MINUSYDIR
	    { $$ = new negDir_MINUSYDIR(); }
	| MINUSZDIR
	    { $$ = new negDir_MINUSZDIR(); }
	;

plan :
	  XYPLAN
	    { $$ = new plan_XYPLAN(); }
	| YZPLAN
	    { $$ = new plan_YZPLAN(); }
	| ZXPLAN
	    { $$ = new plan_ZXPLAN(); }
	;

matDir :
	  INNER
	    { $$ = new matDir_INNER(); }
	| OUTER
	    { $$ = new matDir_OUTER(); }
	;

flatRoundOpen :
	  FLAT
	    { $$ = new flatRoundOpen_FLAT(); }
	| ROUND
	    { $$ = new flatRoundOpen_ROUND(); }
	| OPEN
	    { $$ = new flatRoundOpen_OPEN(); }
	;

radiusSpec :
	  MAJOR
	    { $$ = new radiusSpec_MAJOR(); }
	| MINOR
	    { $$ = new radiusSpec_MINOR(); }
	;

stackElement :
	  ALGOR
	    { $$ = new stackElement_ALGOR(); }
	| DME
	    { $$ = new stackElement_DME(); }
	| DATSET
	    { $$ = new stackElement_DATSET(); }
	| REPORT
	    { $$ = new stackElement_REPORT(); }
	;

rotType :
	  ROTTOT
	    { $$ = new rotType_ROTTOT(); }
	| ROTORG
	    { $$ = new rotType_ROTORG(); }
	| ROTNUL
	    { $$ = new rotType_ROTNUL(); }
	;

rotDir :
	  rotOrient
	    { $$ = new rotDir_rotOrient($1); }
	| SHORT
	    { $$ = new rotDir_SHORT(); }
	;

rotOrient :
	  CW
	    { $$ = new rotOrient_CW(); }
	| CCW
	    { $$ = new rotOrient_CCW(); }
	;

rotAbs :
	  ABSL C rotDir
	    { $$ = new rotAbs($3); }
	;

rotIncr :
	  INCR C rotOrient
	    { $$ = new rotIncr($3); }
	;

jumpStm :
	  labelNameConst endOfLine
	    { $$ = new jumpStm($1);
	      dmisStms.push_back($$);
	    }
	;

aclratStm :
	  ACLRAT SLASH aclratMinor endOfLine
	    { $$ = new aclratStm($3);
	      dmisStms.push_back($$);
	    }
	;

aclratMinor :
	  aclratMeas
	    { $$ = $1; }
	| aclratPos
	    { $$ = $1; }
	| aclratRot
	    { $$ = $1; }
	| aclratScan
	    { $$ = $1; }
	| aclratHedRot
	    { $$ = $1; }
	| aclratHedMeas
	    { $$ = $1; }
	| aclratHedScan
	    { $$ = $1; }
	;

aclratMeas :
	  MESACL C aclratLinSpec
	    { $$ = new aclratMeas($3); }
	;

aclratPos :
	  POSACL C aclratLinSpec
	    { $$ = new aclratPos($3); }
	;

aclratRot :
	  ROTACL C aclratAngSpec
	    { $$ = new aclratRot($3); }
	;

aclratScan :
	  SCNACL C aclratLinSpec
	    { $$ = new aclratScan($3); }
	;

aclratHedRot :
	  HEDROTACL C aclratLinSpec
	    { $$ = new aclratHedRot($3); }
	;

aclratHedMeas :
	  HEDMESACL C aclratLinSpec
	    { $$ = new aclratHedMeas($3); }
	;

aclratHedScan :
	  HEDSCNACL C aclratLinSpec
	    { $$ = new aclratHedScan($3); }
	;

aclratAngSpec :
	  aclratAngular
	    { $$ = $1; }
	| aclratDef
	    { $$ = $1; }
	;

aclratLinSpec :
	  aclratLinear
	    { $$ = $1; }
	| aclratDef
	    { $$ = $1; }
	;

aclratLinear :
	  MPMM C rentVal
	    { $$ = new aclratLinear_MPMM($3); }
	| MMPSS C rentVal
	    { $$ = new aclratLinear_MMPSS($3); }
	| IPMM C rentVal
	    { $$ = new aclratLinear_IPMM($3); }
	| IPSS C rentVal
	    { $$ = new aclratLinear_IPSS($3); }
	;

aclratAngular :
	  RPMM C rentVal
	    { $$ = new aclratAngular($3); }
	;

aclratDef :
	  PCENT C rentVal
	    { $$ = new aclratDef_PCENT($3); }
	| HIGH
	    { $$ = new aclratDef_HIGH(); }
	| LOW
	    { $$ = new aclratDef_LOW(); }
	| DEFALT
	    { $$ = new aclratDef_DEFALT(); }
	;

algdefStm :
	  vaLabel equalSign ALGDEF SLASH algdefMinor endOfLine
	    { $$ = new algdefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

algdefMinor :
	  CODE C intVal
	    { $$ = new algdefMinor_CODE($3); }
	| stringVal
	    { $$ = new algdefMinor_stringVal($1, 0); }
	| stringVal C paramList
	    { $$ = new algdefMinor_stringVal($1, $3); }
	;

assignStm :
	  assignString endOfLine
	    { $$ = new assignStm_assignString($1);
	      dmisStms.push_back($$);
	    }
	| assignBool endOfLine
	    { $$ = new assignStm_assignBool($1);
	      dmisStms.push_back($$);
	    }
	| assignInt endOfLine
	    { $$ = new assignStm_assignInt($1);
	      dmisStms.push_back($$);
	    }
	| assignReal endOfLine
	    { $$ = new assignStm_assignReal($1);
	      dmisStms.push_back($$);
	    }
	| assignVector endOfLine
	    { $$ = new assignStm_assignVector($1);
	      dmisStms.push_back($$);
	    }
	;

assignString :
	  stringVar EQUALS ASSIGN SLASH stringVal
	    { $$ = new assignString($1, $5); }
	;

assignBool :
	  boolVar EQUALS ASSIGN SLASH boolVal
	    { $$ = new assignBool($1, $5); }
	;

assignInt :
	  intVar EQUALS ASSIGN SLASH rentVal
	    { $$ = new assignInt($1, $5); }
	;

assignReal :
	  realVar EQUALS ASSIGN SLASH rentVal
	    { $$ = new assignReal($1, $5); }
	;

assignVector :
	  vectorVar EQUALS ASSIGN SLASH vectorVal
	    { $$ = new assignVector($1, $5); }
	;

badtstStm :
	  BADTST SLASH state endOfLine
	    { $$ = new badtstStm($3);
	      dmisStms.push_back($$);
	    }
	;

boundStm :
	  BOUND SLASH boundMinor endOfLine
	    { $$ = new boundStm($3);
	      dmisStms.push_back($$);
	    }
	;

boundMinor :
	  boundFeat
	    { $$ = $1; }
	| boundTol
	    { $$ = $1; }
	;

boundFeat :
	  fLabel C featureList
	    { $$ = new boundFeat($1, $3); }
	;

boundTol :
	  tLabel C featureList
	    { $$ = new boundTol($1, $3); }
	;

calibSensStm :
	  CALIB SLASH calibSensMinor endOfLine
	    { $$ = new calibSensStm($3);
	      dmisStms.push_back($$);
	    }
	;

calibRtabStm :
	  CALIB SLASH calibRtabMinor endOfLine
	    { $$ = new calibRtabStm($3);
	      dmisStms.push_back($$);
	    }
	;

calibMasterStm :
	  CALIB SLASH calibMasterMinor endOfLine
	    { $$ = new calibMasterStm($3);
	      dmisStms.push_back($$);
	    }
	;

calibSensMinor :
	  SENS C sLabel C calibSensSpec
	    { $$ = new calibSensMinorSpec($3, $5); }
	| SENS C RECALIB
	    { $$ = new calibSensMinorRecalib(); }
	;

calibSensSpec :
	  featureLabel1 C intVal
	    { $$ = new calibSensSpecInt($1, $3); }
	| featureLabel1 C stringVal
	    { $$ = new calibSensSpecString($1, $3, 0); }
	| featureLabel1 C stringVal C intVal
	    { $$ = new calibSensSpecString($1, $3, $5); }
	;

calibRtabMinor :
	  RTAB C rtLabel C calibRtabSpec
	    { $$ = new calibRtabMinorSpec($3, $5); }
	| RTAB C RECALIB
	    { $$ = new calibRtabMinorRecalib(); }
	;

calibRtabSpec :
	  fLabel C intVal
	    { $$ = new calibRtabSpec_fLabel($1, $3); }
	| faLabel C faLabel
	    { $$ = new calibRtabSpec_faLabel($1, $3); }
	;

calibMasterMinor :
	  MASTER C stringVal
	    { $$ = new calibMasterMinor($3); }
	;

callStm :
	  CALL SLASH callMinor endOfLine
	    { $$ = new callStm($3);
	      dmisStms.push_back($$);
	    }
	;

callMinor :
	  callMacro
	    { $$ = $1; }
	| callModule
	    { $$ = $1; }
	| callRoutine
	    { $$ = $1; }
	| callProgram
	    { $$ = $1; }
	;

callMacro :
	  mLabel
	    { $$ = new callMacro(false, $1, 0); }
	| mLabel C {getCallArgs = 1;} CHARSTRING
	    { $$ = new callMacro(false, $1, $4); }
	| EXTERN C DMIS C mLabel
	    { $$ = new callMacro(true, $5, 0); }
	| EXTERN C DMIS C mLabel C {getCallArgs = 1;} CHARSTRING
	    { $$ = new callMacro(true, $5, $8); }
	;

callModule :
	  EXTERN C DMIS C stringVal
	    { $$ = new callModule($5); }
	;

callRoutine :
	  EXTERN C DME C stringVal
	    { $$ = new callRoutine($5, 0, 0); }
	| EXTERN C DME C stringVal C paramList
	    { $$ = new callRoutine($5, 0, $7); }
	| EXTERN C DME C stringVal C callType
	    { $$ = new callRoutine($5, $7, 0); }
	| EXTERN C DME C stringVal C callType C paramList
	    { $$ = new callRoutine($5, $7, $9); }
	;

callProgram :
	  EXTERN C SYS C stringVal
	    { $$ = new callProgram($5, 0, 0); }
	| EXTERN C SYS C stringVal C paramList
	    { $$ = new callProgram($5, 0, $7); }
	| EXTERN C SYS C stringVal C callType
	    { $$ = new callProgram($5, $7, 0); }
	| EXTERN C SYS C stringVal C callType C paramList
	    { $$ = new callProgram($5, $7, $9); }
	;

callType :
	  WAIT
	    { $$ = new callType_WAIT(); }
	| CONT
	    { $$ = new callType_CONT(); }
	| ATTACH
	    { $$ = new callType_ATTACH(); }
	;

caseStm :
	  CASE SLASH intConst endOfLine
	    { $$ = new caseStmInt(0, $3);
	      dmisStms.push_back($$);
	    }
	| CASE SLASH sign intConst endOfLine
	    { $$ = new caseStmInt($3, $4);
	      dmisStms.push_back($$);
	    }
	| CASE SLASH stringConst endOfLine
	    { $$ = new caseStmString($3);
	      dmisStms.push_back($$);
	    }
	;

plusSign :
	  PLUS
	    { $$ = new plusSign(); }
	;

minusSign :
	  MINUS
	    { $$ = new minusSign(); }
	;

sign :
	  plusSign
	    { $$ = $1; }
	| minusSign
	    { $$ = $1; }
	;

clmpidStm :
	  ciLabel equalSign CLMPID SLASH stringVal endOfLine
	    { $$ = new clmpidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

clmpsnStm :
	  csLabel equalSign CLMPSN SLASH stringVal endOfLine
	    { $$ = new clmpsnStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

closeStm :
	  CLOSE SLASH closeMinor endOfLine
	    { $$ = new closeStm($3);
	      dmisStms.push_back($$);
	    }
	;

closeMinor :
	  didLabel
	    { $$ = new closePlain($1); }
	| didLabel C KEEP
	    { $$ = new closeKeep($1); }
	| didLabel C DELETE
	    { $$ = new closeDelete($1); }
	| didLabel C END
	    { $$ = new closeEnd($1); }
	;

cmpntgrpStm :
	  sgLabel equalSign CMPNTGRP SLASH BUILD C cmpntgrpSpecList endOfLine
	    { $$ = new cmpntgrpStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

cmpntgrpSpecList :
	  cmpntgrpSpec
	    { $$ = new std::list<cmpntgrpSpec *>;
	      $$->push_back($1); }
	| cmpntgrpSpecList C cmpntgrpSpec
	    { $$ = $1;
	      $$->push_back($3); }
	;

cmpntgrpSpec :
	  sgLabel
	    { $$ = $1; }
	| swLabel
	    { $$ = $1; }
	| sxLabel
	    { $$ = $1; }
	| rmLabel
	    { $$ = $1; }
	;

cnfrmrulStm :
	  drLabel equalSign CNFRMRUL SLASH cnfrmrulMinor endOfLine
	    { $$ = new cnfrmrulStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

cnfrmrulMinor :
	  RULE C intVal
	    { $$ = new cnfrmrulCode($3); }
	| stringVal
	    { $$ = new cnfrmrulName($1, 0); }
	| stringVal C paramList
	    { $$ = new cnfrmrulName($1, $3); }
	;

constStm :
	  CONST SLASH constMinor endOfLine
	    { $$ = new constStm($3);
	      dmisStms.push_back($$);
	    }
	;

constMinor :
	  constArc
	    { $$ = $1; }
	| constCircle
	    { $$ = $1; }
	| constCompound
	    { $$ = $1; }
	| constCone
	    { $$ = $1; }
	| constCylndr
	    { $$ = $1; }
	| constCparln
	    { $$ = $1; }
	| constEllips
	    { $$ = $1; }
	| constEdgept
	    { $$ = $1; }
	| constGeom
	    { $$ = $1; }
	| constGcurve
	    { $$ = $1; }
	| constGsurf
	    { $$ = $1; }
	| constLine
	    { $$ = $1; }
	| constParpln
	    { $$ = $1; }
	| constPatern
	    { $$ = $1; }
	| constPlane
	    { $$ = $1; }
	| constPoint
	    { $$ = $1; }
	| constRctngl
	    { $$ = $1; }
	| constSgage
	    { $$ = $1; }
	| constSpart
	    { $$ = $1; }
	| constSphere
	    { $$ = $1; }
	| constSympln
	    { $$ = $1; }
	| constTorus
	    { $$ = $1; }
	;

constArc :
	  ARC C fLabel C bfConst
	    { $$ = new constArc_bfConst($3, $5); }
	| ARC C fLabel C projctConst
	    { $$ = new constArc_projctConst($3, $5); }
	| ARC C fLabel C trConst
	    { $$ = new constArc_trConst($3, $5); }
	;

constCircle :
	  CIRCLE C fLabel C bfConst
	    { $$ = new constCircle_bfConst($3, $5); }
	| CIRCLE C fLabel C coneConst
	    { $$ = new constCircle_coneConst($3, $5); }
	| CIRCLE C fLabel C intofConst
	    { $$ = new constCircle_intofConst($3, $5); }
	| CIRCLE C fLabel C projctConst
	    { $$ = new constCircle_projctConst($3, $5); }
	| CIRCLE C fLabel C tantoConst
	    { $$ = new constCircle_tantoConst($3, $5); }
	| CIRCLE C fLabel C trConst
	    { $$ = new constCircle_trConst($3, $5); }
	| CIRCLE C fLabel C retrieve2
	    { $$ = new constCircle_retrieve2($3, $5); }
	;

constCompound :
	  COMPOUND C fLabel C BUILD
	    { $$ = new constCompound($3); }
	;

constCone :
	  CONE C fLabel C bfConst
	    { $$ = new constCone_bfConst($3, $5); }
	| CONE C fLabel C trConst
	    { $$ = new constCone_trConst($3, $5); }
	;

constCparln :
	  CPARLN C fLabel C bfConst
	    { $$ = new constCparln_bfConst($3, $5); }
	| CPARLN C fLabel C projctConst
	    { $$ = new constCparln_projctConst($3, $5); }
	| CPARLN C fLabel C trConst
	    { $$ = new constCparln_trConst($3, $5); }
	| CPARLN C fLabel C retrieve2
	    { $$ = new constCparln_retrieve2($3, $5); }
	;

constCylndr :
	  CYLNDR C fLabel C bfConst
	    { $$ = new constCylndr_bfConst($3, $5); }
	| CYLNDR C fLabel C trConst
	    { $$ = new constCylndr_trConst($3, $5); }
	| CYLNDR C fLabel C retrieve1
	    { $$ = new constCylndr_retrieve1($3, $5); }
	;

constEllips :
	  ELLIPS C fLabel C bfConst
	    { $$ = new constEllips_bfConst($3, $5); }
	| ELLIPS C fLabel C intofConst
	    { $$ = new constEllips_intofConst($3, $5); }
	| ELLIPS C fLabel C projctConst
	    { $$ = new constEllips_projctConst($3, $5); }
	| ELLIPS C fLabel C trConst
	    { $$ = new constEllips_trConst($3, $5); }
	;

constEdgept :
	  EDGEPT C fLabel C retrieve4
	    { $$ = new constEdgept($3, $5); }
	;

constGeom :
	  GEOM C fLabel C nearptConst
	    { $$ = new constGeom($3, $5); }
	;

constGcurve :
	  GCURVE C fLabel C bfConst
	    { $$ = new constGcurve_bfConst($3, $5); }
	| GCURVE C fLabel C projctConst
	    { $$ = new constGcurve_projctConst($3, $5); }
	| GCURVE C fLabel C trConst
	    { $$ = new constGcurve_trConst($3, $5); }
	;

constGsurf :
	  GSURF C fLabel C bfConst
	    { $$ = new constGsurf_bfConst($3, $5); }
	| GSURF C fLabel C trConst
	    { $$ = new constGsurf_trConst($3, $5); }
	;

constLine :
	  LINE C fLabel C bfConst
	    { $$ = new constLine_bfConst($3, $5); }
	| LINE C fLabel C intofConst
	    { $$ = new constLine_intofConst($3, $5); }
	| LINE C fLabel C midliConst
	    { $$ = new constLine_midliConst($3, $5); }
	| LINE C fLabel C offsetConst
	    { $$ = new constLine_offsetConst($3, $5); }
	| LINE C fLabel C partoConst
	    { $$ = new constLine_partoConst($3, $5); }
	| LINE C fLabel C perptoConst
	    { $$ = new constLine_perptoConst($3, $5); }
	| LINE C fLabel C projliConst
	    { $$ = new constLine_projliConst($3, $5); }
	| LINE C fLabel C tantoConst
	    { $$ = new constLine_tantoConst($3, $5); }
	| LINE C fLabel C trConst
	    { $$ = new constLine_trConst($3, $5); }
	;

constParpln :
	  PARPLN C fLabel C bfConst
	    { $$ = new constParpln($3, $5); }
	;

constPatern :
	  PATERN C fLabel C trConst
	    { $$ = new constPatern_trConst($3, $5); }
	| PATERN C fLabel C BUILD
	    { $$ = new constPatern_BUILD($3); }
	;

constPlane :
	  PLANE C fLabel C bfConst
	    { $$ = new constPlane_bfConst($3, $5); }
	| PLANE C fLabel C midplConst
	    { $$ = new constPlane_midplConst($3, $5); }
	| PLANE C fLabel C offsetConst
	    { $$ = new constPlane_offsetConst($3, $5); }
	| PLANE C fLabel C partoConst
	    { $$ = new constPlane_partoConst($3, $5); }
	| PLANE C fLabel C perptoConst
	    { $$ = new constPlane_perptoConst($3, $5); }
	| PLANE C fLabel C tantoConstPlane
	    { $$ = new constPlane_tantoConstPlane($3, $5); }
	| PLANE C fLabel C trConst
	    { $$ = new constPlane_trConst($3, $5); }
	;

constPoint :
	  POINT C fLabel C cogConst
	    { $$ = new constPoint_cogConst($3, $5); }
	| POINT C fLabel C curveConst
	    { $$ = new constPoint_curveConst($3, $5); }
	| POINT C fLabel C extremConst
	    { $$ = new constPoint_extremConst($3, $5); }
	| POINT C fLabel C intofConst
	    { $$ = new constPoint_intofConst($3, $5); }
	| POINT C fLabel C midptConst
	    { $$ = new constPoint_midptConst($3, $5); }
	| POINT C fLabel C moveptConst
	    { $$ = new constPoint_moveptConst($3, $5); }
	| POINT C fLabel C pierceConst
	    { $$ = new constPoint_pierceConst($3, $5); }
	| POINT C fLabel C projptConst
	    { $$ = new constPoint_projptConst($3, $5); }
	| POINT C fLabel C trConst
	    { $$ = new constPoint_trConst($3, $5); }
	| POINT C fLabel C vertexConst
	    { $$ = new constPoint_vertexConst($3, $5); }
	| POINT C fLabel C retrieve1
	    { $$ = new constPoint_retrieve1($3, $5); }
	;

retrieve1 :
	  RETRIEVE C rentVal C featureActualList
	    { $$ = new retrieve1($3, $5); }
	;

retrieve2 :
	  RETRIEVE C rentVal C rentVal C featureActualList
	    { $$ = new retrieve2($3, $5, $7); }
	;

retrieve4 :
	  RETRIEVE C rentVal C rentVal C rentVal C rentVal C featureActualList
	    { $$ = new retrieve4($3, $5, $7, $9, $11); }
	;

retrieve2b :
	  RETRIEVE C rentVal C featureActualList
	    { $$ = new retrieve2b($3, 0, $5); }
	| RETRIEVE C rentVal C vector C featureActualList
	    { $$ = new retrieve2b($3, $5, $7); }
	;

constRctngl :
	  RCTNGL C fLabel C bfConst
	    { $$ = new constRctngl_bfConst($3, $5); }
	| RCTNGL C fLabel C trConst
	    { $$ = new constRctngl_trConst($3, $5); }
	;

constSgage :
	  SGAGE C seLabel defCheckComma sgageConst
	    { $$ = new constSgage($3, $5); }
	;

constSpart :
	  SPART C stLabel defCheckComma spartConst
	    { $$ = new constSpart($3, $5); }
	;

constSphere :
	  SPHERE C fLabel C bfConst
	    { $$ = new constSphere_bfConst($3, $5); }
	| SPHERE C fLabel C trConst
	    { $$ = new constSphere_trConst($3, $5); }
	| SPHERE C fLabel C retrieve2b
	    { $$ = new constSphere_retrieve2b($3, $5); }
	;

constSympln :
	  SYMPLN C fLabel C bfConst
	    { $$ = new constSympln($3, $5); }
	;

constTorus :
	  TORUS C fLabel C bfConst
	    { $$ = new constTorus_bfConst($3, $5); }
	| TORUS C fLabel C trConst
	    { $$ = new constTorus_trConst($3, $5); }
	;

bfConst :
	  BF C featureList
	    { $$ = new bfConst_featureList($3); }
	| BF C indexedFeatureList
	    { $$ = new bfConst_indexedFeatureList($3); }
	;

cogConst :
	  COG C featureList
	    { $$ = new cogConst($3); }
	;

coneConst :
	  CONE C DIAM C rentVal C faLabel
	    { $$ = new coneConst_DIAM($5, $7); }
	| CONE C DIST C rentVal C faLabel
	    { $$ = new coneConst_DIST($5, $7); }
	;

curveConst :
	  CURVE C faLabel C featureLabel
	    { $$ = new curveConst($3, $5); }
	;

extremConst :
	  EXTREM C MIN C faLabel C extremConstDir
	    { $$ = new extremConst_MIN($5, $7); }
	| EXTREM C MAX C faLabel C extremConstDir
	    { $$ = new extremConst_MAX($5, $7); }
	;

extremConstDir :
	  extremConstAxial
	    { $$ = $1; }
	| extremConstVectorial
	    { $$ = $1; }
	| extremConstFeature
	    { $$ = $1; }
	| extremConstRadial
	    { $$ = $1; }
	;

extremConstAxial :
	  posDir
	    { $$ = $1; }
	;

extremConstVectorial :
	  VEC C vector
	    { $$ = new extremConstVectorial($3); }
	;

extremConstFeature :
	  featureLabel
	    { $$ = $1; }
	;

extremConstRadial :
	  RADIAL
	    { $$ = new extremConstRadial(); }
	;

intofConst :
	  INTOF C faLabel C featureLabel
	    { $$ = new intofConst($3, $5); }
	;

midliConst :
	  MIDLI C faLabel C featureLabel
	    { $$ = new midliConst($3, $5); }
	;

midplConst :
	  MIDPL C faLabel C featureLabel
	    { $$ = new midplConst($3, $5); }
	;

midptConst :
	  MIDPT C faLabel C featureLabel
	    { $$ = new midptConst($3, $5); }
	;

moveptConst :
	  MOVEPT C faLabel C vector
	    { $$ = new moveptConstVec($3, $5); }
	| MOVEPT C faLabel C featureLabel C rentVal
	    { $$ = new moveptConstAxis($3, $5, $7); }
	;

nearptConst :
	  NEARPT C faLabel
	    { $$ = new nearptConst($3); }
	;

offsetConst :
	  OFFSET C featureList
	    { $$ = new offsetConst($3); }
	;

partoConst :
	  PARTO C faLabel C THRU C featureLabel
	    { $$ = new partoConstAct($3, $7); }
	| PARTO C fLabel C THRU C faLabel
	    { $$ = new partoConstNom($3, $7); }
	;

perptoConst :
	  PERPTO C faLabel C THRU C featureLabel
	    { $$ = new perptoConstAct($3, $7); }
	| PERPTO C fLabel C THRU C faLabel
	    { $$ = new perptoConstNom($3, $7); }
	;

pierceConst :
	  PIERCE C faLabel C featureLabel
	    { $$ = new pierceConst($3, $5); }
	;

projctConst :
	  PROJCT C faLabel
	    { $$ = new projctConst($3, 0); }
	| PROJCT C faLabel C featureLabel
	    { $$ = new projctConst($3, $5); }
	;

projliConst :
	  PROJLI C faLabel
	    { $$ = new projliConst($3, 0); }
	| PROJLI C faLabel C featureLabel
	    { $$ = new projliConst($3, $5); }
	;

projptConst :
	  PROJPT C faLabel
	    { $$ = new projptConst($3, 0); }
	| PROJPT C faLabel C featureLabel
	    { $$ = new projptConst($3, $5); }
	;

sgageConst :
	  featureNominalList
	    { $$ = new sgageConst($1); }
	;

spartConst :
	  featureActualList
	    { $$ = new spartConst($1); }
	;

tantoConst :
	  TANTO C faLabel C featureLabel
	    { $$ = new tantoConstFa($3, false, $5); }
	| TANTO C faLabel C THRU C featureLabel
	    { $$ = new tantoConstFa($3, true, $7); }
	| TANTO C fLabel C THRU C faLabel
	    { $$ = new tantoConstF($3, $7); }
	;

tantoConstPlane :
	  TANTO C faLabel C THRU C featureLabel
	    { $$ = new tantoConstPlaneFa($3, $7); }
	| TANTO C fLabel C THRU C faLabel
	    { $$ = new tantoConstPlaneF($3, $7); }
	;

trConst :
	  TR C faLabel
	    { $$ = new trConst($3, 0); }
	| TR C faLabel C datumLabel
	    { $$ = new trConst($3, $5); }
	;

vertexConst :
	  VERTEX C faLabel
	    { $$ = new vertexConst($3); }
	;

crgdefStm :
	  crLabelConst equalSign crgdefMinor endOfLine
	    { $$ = new crgdefStm($1, $3);
	      dmisStms.push_back($$);
	    }
	;

crgdefMinor :
	  CRGDEF
	    { $$ = new crgdefNoData(); }
	| CRGDEF SLASH impCartPoint C impCartPoint C vector C vector C vector C vector
	    { $$ = new crgdefData($3, $5, $7, $9, $11, $13); }
	;

crmodeStm :
	  CRMODE SLASH crmodeMinor endOfLine
	    { $$ = new crmodeStm($3);
	      dmisStms.push_back($$);
	    }
	;

crmodeMinor :
	  SEQNTL
	    { $$ = new crmodeMinor_SEQNTL(); }
	| SIMUL
	    { $$ = new crmodeMinor_SIMUL(); }
	| SYNC
	    { $$ = new crmodeMinor_SYNC(); }
	;

crosclStm :
	  CROSCL SLASH state endOfLine
	    { $$ = new crosclStm($3);
	      dmisStms.push_back($$);
	    }
	;

crslctStm :
	  CRSLCT SLASH crslctMinor endOfLine
	    { $$ = new crslctStm($3);
	      dmisStms.push_back($$);
	    }
	;

crslctMinor :
	  crLabelConst
	    { $$ = new crslctMinor_crLabelConst($1); }
	| ALL
	    { $$ = new crslctMinor_ALL(); }
	;

cutcomStm :
	  ccLabel equalSign CUTCOM SLASH cutcomMinor endOfLine
	    { $$ = new cutcomStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

cutcomMinor :
	  cutcomAdjust
	    { $$ = $1; }
	| cutcomParam
	    { $$ = $1; }
	| cutcomMatrix
	    { $$ = $1; }
	| cutcomUserdf
	    { $$ = $1; }
	;

cutcomAdjust :
	  mdLabel C ADJUST C tlLabel C leftRight C plan C rentVal
	    { $$ = new cutcomAdjust($1, $5, $7, $9, $11); }
	;

cutcomParam :
	  mdLabel C PARAM C pointVecCart
	    { $$ = new cutcomParam($1, $5); }
	;

cutcomMatrix :
	  mdLabel C MATRIX C cutcomMatx
	    { $$ = new cutcomMatrix($1, $5); }
	;

cutcomMatx :
	  rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal C rentVal
	    { $$ = new cutcomMatx($1, $3, $5, $7, $9, $11, $13, $15, $17, $19, $21, $23); }
	;

cutcomUserdf :
	  mdLabel C USERDF C stringVal
	    { $$ = new cutcomUserdf($1, $5); }
	;

czoneStm :
	  czLabel equalSign CZONE endOfLine
	    { $$ = new czoneStm($1);
	      dmisStms.push_back($$);
	    }
	;

czslctStm :
	  CZSLCT SLASH czLabel C state endOfLine
	    { $$ = new czslctStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

datdefStm :
	  DATDEF SLASH datdefMinor endOfLine
	    { $$ = new datdefStm($3);
	      dmisStms.push_back($$);
	    }
	;

datdefMinor :
	   featureLabel1 undefCheckComma datLabel
          {
            if (aLabelFound == 1)
              {
                sprintf(warningMessage,
			"label doubly defined?: %s", aLabel);
                warn(warningMessage);
              }
            else
              {
                labels[DAT].record(aLabel);
                aLabelFound = 1;
	        $$ = new datdefMinor_featureLabel1($1, $3);
              }
          }
	|  dattrgList C fLabel undefCheckComma datLabel
          {
            if (aLabelFound == 1)
              {
                sprintf(warningMessage,
			"label doubly defined?: %s", aLabel);
                warn(warningMessage);
              }
            else
              {
                labels[DAT].record(aLabel);
                aLabelFound = 1;
	        $$ = new datdefMinor_dattrgList($1, $3, $5);
              }
          }
	;

datsetStm :
	  dLabel equalSign DATSET SLASH datsetMinor endOfLine
	    { $$ = new datsetStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

datsetMinor :
	  datsetMcs
	    { $$ = $1; }
	| datsetDats
	    { $$ = $1; }
	| datsetMatrix
	    { $$ = $1; }
	| datsetDrf
	    { $$ = $1; }
	;

datsetMcs :
	  MCS
	    { $$ = new datsetMcs(); }
	;

datsetMatrix :
	  TRMATX C matrix
	    { $$ = new datsetMatrix($3); }
	;

datsetDats :
	  datsetSpec
	    { $$ = new datsetDats($1, 0, 0); }
	| datsetSpecC datsetSpec
	    { $$ = new datsetDats($1, $2, 0); }
	| datsetSpecC datsetSpecC datsetSpec
	    { $$ = new datsetDats($1, $2, $3); }
	;

datsetDrf :
	  DRF C taLabel C dir C dir
	    { $$ = new datsetDrf($3, 0, $5, $7); }
	| DRF C taLabel C tier C dir C dir
	    { $$ = new datsetDrf($3, $5, $7, $9); }
	;

tier :
	  UPTIER
	    { $$ = new tier_UPTIER(); }
	| LOTIER
	    { $$ = new tier_LOTIER(); }
	;

datsetSpec :
	  datLabel C orig
	    { $$ = new datsetSpecNoDir($1, $3, 0, 0); }
	| datLabel C orig C orig
	    { $$ = new datsetSpecNoDir($1, $3, $5, 0); }
	| datLabel C orig C orig C orig
	    { $$ = new datsetSpecNoDir($1, $3, $5, $7); }
	| datLabel C dir
	    { $$ = new datsetSpecDir($1, $3, 0, 0, 0); }
	| datLabel C dir C orig
	    { $$ = new datsetSpecDir($1, $3, $5, 0, 0); }
	| datLabel C dir C orig C orig
	    { $$ = new datsetSpecDir($1, $3, $5, $7, 0); }
	| datLabel C dir C orig C orig C orig
	    { $$ = new datsetSpecDir($1, $3, $5, $7, $9); }
	;

datsetSpecC :
	  datLabel C orig C
	    { $$ = new datsetSpecNoDir($1, $3, 0, 0); }
	| datLabel C orig C orig C
	    { $$ = new datsetSpecNoDir($1, $3, $5, 0); }
	| datLabel C orig C orig C orig C
	    { $$ = new datsetSpecNoDir($1, $3, $5, $7); }
	| datLabel C dir C
	    { $$ = new datsetSpecDir($1, $3, 0, 0, 0); }
	| datLabel C dir C orig C
	    { $$ = new datsetSpecDir($1, $3, $5, 0, 0); }
	| datLabel C dir C orig C orig C
	    { $$ = new datsetSpecDir($1, $3, $5, $7, 0); }
	| datLabel C dir C orig C orig C orig C
	    { $$ = new datsetSpecDir($1, $3, $5, $7, $9); }
	;

dattrgdefStm :
	  DATTRGDEF SLASH dattrgMinor endOfLine
	    { $$ = new dattrgdefStm($3);
	      dmisStms.push_back($$);
	    }
	;

dattrgMinor :
	   featureList undefCheckComma dattrgLabel
          {
            if (aLabelFound == 1)
              {
                sprintf(warningMessage,
			"label doubly defined?: %s", aLabel);
                warn(warningMessage);
              }
            else
              {
                labels[DATTRG].record(aLabel);
                aLabelFound = 1;
	        $$ = new dattrgMinor($1, $3);
              }
          }
	;

declStm :
	  DECL {inDecl = 1;} SLASH declMinor ENDLINE
	    { $$ = new declStm($4);
	      dmisStms.push_back($$);
	      inDecl = 0;
	      if (lexWarning[0])
		{
		  warn(lexWarning);
		  lexWarning[0] = 0;
		}
	    }
	;

declMinor :
	  declType C declVarList
	    { $$ = new declMinor(0, $1, $3); }
	| declScope C declType C declVarList
	    { $$ = new declMinor($1, $3, $5); }
	;

declVarList :
	  declVar
	    { $$ = new std::list<declVar *>;
	      $$->push_back($1); }
	| declVarList C declVar
	    { $$ = $1;
	      $$->push_back($3); }
	;

declScope :
	  COMMON
	    { $$ = new declScope_COMMON(); }
	| GLOBAL
	    { $$ = new declScope_GLOBAL(); }
	| LOCAL
	    { $$ = new declScope_LOCAL(); }
	;

declType :
	  BOOL
	    { $$ = new declType_BOOL(); }
	| INTGR
	    { $$ = new declType_INTGR(); }
	| LONG
	    { $$ = new declType_LONG(); }
	| REAL
	    { $$ = new declType_REAL(); }
	| DOUBLE
	    { $$ = new declType_DOUBLE(); }
	| CHAR C intVal
	    { $$ = new declType_CHAR($3); }
	| VECTOR
	    { $$ = new declType_VECTOR(); }
	;

declVar :
	  DECLVARNAME
	    { $$ = new declVar($1, 0); }
	| DECLVARNAME LBOX declIndicesList RBOX
	    { $$ = new declVar($1, $3); }
	;

declIndicesList :
	  intConst
	    { $$ = new std::list<intConst *>;
	      $$->push_back($1); }
	| declIndicesList C intConst
	    { $$ = $1;
	      $$->push_back($3); }
	;

decplStm :
	  DECPL SLASH decplMinor endOfLine
	    { $$ = new decplStm($3);
	      dmisStms.push_back($$);
	    }
	;

decplMinor :
	  decplAll
	    { $$ = new decplMinor_decplAll($1); }
	| decplList
	    { $$ = new decplMinor_decplList($1); }
	;

decplAll :
	  ALL C decplNdigits
	    { $$ = new decplAll($3); }
	;

decplList :
	  decplSelection
	    { $$ = new std::list<decplSelection *>;
	      $$->push_back($1); }
	| decplList C decplSelection
	    { $$ = $1;
	      $$->push_back($3); }
	;

decplNdigits :
	  DEFALT
	    { $$ = new decplNdigits_DEFALT(); }
	| intVal
	    { $$ = new decplNdigits_intVal($1); }
	;

decplSelection :
	  ANGLE C decplNdigits
	    { $$ = new decplSelection_ANGLE($3); }
	| DIST C decplNdigits
	    { $$ = new decplSelection_DIST($3); }
	| HUMID C decplNdigits
	    { $$ = new decplSelection_HUMID($3); }
	| DEV C decplNdigits
	    { $$ = new decplSelection_DEV($3); }
	| TEMP C decplNdigits
	    { $$ = new decplSelection_TEMP($3); }
	| VEC C decplNdigits
	    { $$ = new decplSelection_VEC($3); }
	;

deleteStm :
	  DELETE SLASH deleteMinor endOfLine
	    { $$ = new deleteStm($3);
	      dmisStms.push_back($$);
	    }
	;

deleteMinor :
	  deleteDatum
	    { $$ = $1; }
	| deleteSensor
	    { $$ = $1; }
	| deleteFeature
	    { $$ = $1; }
	| deleteRotaryTable
	    { $$ = $1; }
	| deleteAllSensors
	    { $$ = $1; }
	;

deleteDatum :
	  datumLabel1
	    { $$ = new deleteDatum($1, 0); }
	| datumLabel1 C didLabel
	    { $$ = new deleteDatum($1, $3); }
	;

deleteSensor :
	  sensorLabel1
	    { $$ = new deleteSensor($1, 0); }
	| sensorLabel1 C didLabel
	    { $$ = new deleteSensor($1, $3); }
	;

deleteFeature :
	  faLabel
	    { $$ = new deleteFeature($1, 0); }
	| faLabel C didLabel
	    { $$ = new deleteFeature($1, $3); }
	;

deleteRotaryTable :
	  rtLabel
	    { $$ = new deleteRotaryTable($1, 0); }
	| rtLabel C didLabel
	    { $$ = new deleteRotaryTable($1, $3); }
	;

deleteAllSensors :
	  ALLSA
	    { $$ = new deleteAllSensors(0, 0); }
	| ALLSA C didLabel
	    { $$ = new deleteAllSensors(0, $3); }
	| ALLSA C EXCEPT C saLabelList
	    { $$ = new deleteAllSensors($5, 0); }
	| ALLSA C EXCEPT C saLabelList C didLabel
	    { $$ = new deleteAllSensors($5, $7); }
	;

deviceStm :
	  didLabel equalSign DEVICE SLASH deviceMinor endOfLine
	    { $$ = new deviceStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

deviceMinor :
	  device C stringVal
	    { $$ = new deviceMinor_device($1, $3); }
	| INCR C stringVal
	    { $$ = new deviceMinor_INCR($3); }
	;

dftcasStm :
	  DFTCAS endOfLine
	    { $$ = new dftcasStm();
	      dmisStms.push_back($$);
	    }
	;

displyStm :
	  DISPLY SLASH displyOff endOfLine
	    { $$ = new displyStmOff($3);
	      dmisStms.push_back($$);
	    }
	| DISPLY SLASH displySpecList endOfLine
	    { $$ = new displyStmSpec($3);
	      dmisStms.push_back($$);
	    }
	;

displyOff :
	  OFF
	    { $$ = new displyOff(); }
	;

displySpecList :
	  device C DMIS
	    { $$ = new std::list<displySpecItem *>;
	      $$->push_front(new displyItemDMIS($1, 0)); }
	| device C DMIS C displySpecList
	    { $$ = $5;
	      $$->push_front(new displyItemDMIS($1, 0)); }
	| device C DMIS C vLabel
	    { $$ = new std::list<displySpecItem *>;
	      $$->push_front(new displyItemDMIS($1, $5)); }
	| device C DMIS C vLabel C displySpecList
	    { $$ = $7;
	      $$->push_front(new displyItemDMIS($1, $5)); }
	| device C vLabel
	    { $$ = new std::list<displySpecItem *>;
	      $$->push_front(new displyItemNoDMIS($1, $3)); }
	| device C vLabel C displySpecList
	    { $$ = $5;
	      $$->push_front(new displyItemNoDMIS($1, $3)); }
	;

dmehwStm :
	  DMEHW SLASH dmehwMinor endOfLine
	    { $$ = new dmehwStm($3);
	      dmisStms.push_back($$);
	    }
	;

dmehwMinor :
	  CONTIN
	    { $$ = new dmehwMinor_CONTIN(); }
	| PAUSE
	    { $$ = new dmehwMinor_PAUSE(); }
	| SINGLE
	    { $$ = new dmehwMinor_SINGLE(); }
	| AUTO
	    { $$ = new dmehwMinor_AUTO(); }
	| jointConf
	    { $$ = new dmehwMinor_jointConf($1); }
	;

jointConf :
	  JOINTCONFIG
	    { $$ = new jointConf(0, 0, 0, 0); }
	| JOINTCONFIG C stringList
	    { $$ = new jointConf(0, 0, 0, $3); }
	| JOINTCONFIG C flipNoFlip
	    { $$ = new jointConf(0, 0, $3, 0); }
	| JOINTCONFIG C flipNoFlip C stringList
	    { $$ = new jointConf(0, 0, $3, $5); }
	| JOINTCONFIG C aboveBelow
	    { $$ = new jointConf(0, $3, 0, 0); }
	| JOINTCONFIG C aboveBelow C stringList
	    { $$ = new jointConf(0, $3, 0, $5); }
	| JOINTCONFIG C aboveBelow C flipNoFlip
	    { $$ = new jointConf(0, $3, $5, 0); }
	| JOINTCONFIG C aboveBelow C flipNoFlip C stringList
	    { $$ = new jointConf(0, $3, $5, $7); }
	| JOINTCONFIG C rightyLefty
	    { $$ = new jointConf($3, 0, 0, 0); }
	| JOINTCONFIG C rightyLefty C stringList
	    { $$ = new jointConf($3, 0, 0, $5); }
	| JOINTCONFIG C rightyLefty C flipNoFlip
	    { $$ = new jointConf($3, 0, $5, 0); }
	| JOINTCONFIG C rightyLefty C flipNoFlip C stringList
	    { $$ = new jointConf($3, 0, $5, $7); }
	| JOINTCONFIG C rightyLefty C aboveBelow
	    { $$ = new jointConf($3, $5, 0, 0); }
	| JOINTCONFIG C rightyLefty C aboveBelow C stringList
	    { $$ = new jointConf($3, $5, 0, $7); }
	| JOINTCONFIG C rightyLefty C aboveBelow C flipNoFlip
	    { $$ = new jointConf($3, $5, $7, 0); }
	| JOINTCONFIG C rightyLefty C aboveBelow C flipNoFlip C stringList
	    { $$ = new jointConf($3, $5, $7, $9); }
	;

rightyLefty :
	  RIGHTY
	    { $$ = new rightyLefty_RIGHTY(); }
	| LEFTY
	    { $$ = new rightyLefty_LEFTY(); }
	;

aboveBelow :
	  ABOVE
	    { $$ = new aboveBelow_ABOVE(); }
	| BELOW
	    { $$ = new aboveBelow_BELOW(); }
	;

flipNoFlip :
	  FLIP
	    { $$ = new flipNoFlip_FLIP(); }
	| NOFLIP
	    { $$ = new flipNoFlip_NOFLIP(); }
	;

dmeidStm :
	  diLabel equalSign DMEID SLASH stringVal endOfLine
	    { $$ = new dmeidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

dmeswStm :
	  DMESW SLASH dmeswMinor endOfLine
	    { $$ = new dmeswStm($3);
	      dmisStms.push_back($$);
	    }
	;

dmeswMinor :
	  dmeswComand
	    { $$ = $1; }
	| dmeswDelay
	    { $$ = $1; }
	| dmeswSwitch
	    { $$ = $1; }
	;

dmeswComand :
	  COMAND C stringVal
	    { $$ = new dmeswComand($3); }
	;

dmeswDelay :
	  DELAY C intVal
	    { $$ = new dmeswDelay($3); }
	;

dmeswSwitch :
	  CONTIN
	    { $$ = new dmeswSwitch_CONTIN(); }
	| PAUSE
	    { $$ = new dmeswSwitch_PAUSE(); }
	;

dmeswiStm :
	  dsLabel equalSign DMESWI SLASH stringVal endOfLine
	    { $$ = new dmeswiStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

dmeswvStm :
	  dvLabel equalSign DMESWV SLASH stringVal endOfLine
	    { $$ = new dmeswvStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

dmisOffStm :
	  DMIS SLASH OFF endOfLine
	    { $$ = new dmisOffStm();
	      dmisStms.push_back($$);
	    }
	;

dmisOnStm :
	  DMIS SLASH ON endOfLine
	    { $$ = new dmisOnStm();
	      dmisStms.push_back($$);
	    }
	;

dmismdStm :
	  DMISMD SLASH stringConst C versionTag endOfLine
	    { $$ = new dmismdStm($3, $5, 0);
	      dmisStms.push_back($$);
	    }
	| DMISMD SLASH stringConst C versionTag C conformItemList endOfLine
	    { $$ = new dmismdStm($3, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

dmismnStm :
	  DMISMN SLASH stringConst C versionTag endOfLine
	    { $$ = new dmismnStm($3, $5, 0);
	      dmisStms.push_back($$);
	    }
	| DMISMN SLASH stringConst C versionTag C conformItemList endOfLine
	    { $$ = new dmismnStm($3, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

conformItemList :
	  conformItem
	    { $$ = new std::list<conformItem *>;
	      $$->push_back($1); }
	| conformItemList C conformItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

conformItem :
	  conformType C INTSTRING
	    { $$ = new conformItem($1, $3); }
	;

conformType :
	  CT
	    { $$ = new conformType_CT(); }
	| FX
	    { $$ = new conformType_FX(); }
	| IP
	    { $$ = new conformType_IP(); }
	| MC
	    { $$ = new conformType_MC(); }
	| MU
	    { $$ = new conformType_MU(); }
	| PM
	    { $$ = new conformType_PM(); }
	| QI
	    { $$ = new conformType_QI(); }
	| RY
	    { $$ = new conformType_RY(); }
	| SF
	    { $$ = new conformType_SF(); }
	| TW
	    { $$ = new conformType_TW(); }
	;

doStm :
	  DO SLASH intVar C intVal C intVal endOfLine
	    { $$ = new doStm($3, $5, $7, 0);
	      dmisStms.push_back($$);
	    }
	| DO SLASH intVar C intVal C intVal C intVal endOfLine
	    { $$ = new doStm($3, $5, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

elseStm :
	  ELSE endOfLine
	    { $$ = new elseStm();
	      dmisStms.push_back($$);
	    }
	;

endcasStm :
	  ENDCAS endOfLine
	    { $$ = new endcasStm();
	      dmisStms.push_back($$);
	    }
	;

enddoStm :
	  ENDDO endOfLine
	    { $$ = new enddoStm();
	      dmisStms.push_back($$);
	    }
	;

endfilStm :
	  ENDFIL endOfLine
	    { $$ = new endfilStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

endgoStm :
	  ENDGO endOfLine
	    { $$ = new endgoStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

endifStm :
	  ENDIF endOfLine
	    { $$ = new endifStm();
	      dmisStms.push_back($$);
	    }
	;

endmacStm :
	  ENDMAC endOfLine
	    { $$ = new endmacStm();
	      dmisStms.push_back($$);
	    }
	;

endmesStm :
	  ENDMES endOfLine
	    { $$ = new endmesStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

endselStm :
	  ENDSEL endOfLine
	    { $$ = new endselStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

endsimreqtStm :
	  ENDSIMREQT endOfLine
	    { $$ = new endsimreqtStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

endxtnStm :
	  ENDXTN endOfLine
	    { $$ = new endxtnStm();
	      dmisStms.push_back($$);
	    }
	| error ENDLINE
	  {
	    numErrors++;
	    yyerrok;
	    aLabelFound = 1;
	    setLabelType = 0;
	  }
	;

equateStm :
	  EQUATE SLASH equateMinor endOfLine
	    { $$ = new equateStm($3);
	      dmisStms.push_back($$);
	    }
	;

equateMinor :
	  daLabel C daLabel
	    { $$ = new equateDatum($1, $3); }
	| daLabel C CADCS C equateCadcs
	    { $$ = new equateCad($1, $5); }
	;

equateCadcs :
	  didLabel C matrix
	    { $$ = new equateCadcs_matrix($1, $3); }
	| didLabel C stringVal
	    { $$ = new equateCadcs_stringVal($1, $3); }
	;

errorStm :
	  ERROR SLASH errorMinor endOfLine
	    { $$ = new errorStm($3);
	      dmisStms.push_back($$);
	    }
	;

errorMinor :
	  jumpLabel C errorCondition
	    { $$ = new errorMinor_jumpLabel($1, $3); }
	| AUTO C errorCondition
	    { $$ = new errorMinor_AUTO($3); }
	| OFF
	    { $$ = new errorMinor_OFF(); }
	;

errorCondition :
	  ALL
	    { $$ = new errorCondition_ALL(); }
	| ILLEGALTOUCH
	    { $$ = new errorCondition_ILLEGALTOUCH(); }
	| NOTOUCH
	    { $$ = new errorCondition_NOTOUCH(); }
	| intVal
	    { $$ = new errorCondition_intVal($1); }
	;

evalStm :
	  EVAL SLASH evalMinor endOfLine
	    { $$ = new evalStm($3);
	      dmisStms.push_back($$);
	    }
	;

evalMinor :
	  evalFeat
	    { $$ = $1; }
	| evalDat
	    { $$ = $1; }
	| evalKeychar
	    { $$ = $1; }
	;

evalFeat :
	  faLabel C tLabelList
	    { $$ = new evalFeatFaT($1, $3); }
	| faLabel C fLabel C tLabel
	    { $$ = new evalFeatFaFT($1, $3, $5); }
	| fLabel C faLabel C tLabel
	    { $$ = new evalFeatFFaT($1, $3, $5); }
	| faLabel C faLabel C tLabel
	    { $$ = new evalFeatFaFaT($1, $3, $5); }
	;

evalDat :
	  datLabel C faLabel C tLabel
	    { $$ = new evalDatDatFa($1, $3, $5); }
	| faLabel C datLabel C tLabel
	    { $$ = new evalDatFaDat($1, $3, $5); }
	;

evalKeychar :
	  kcLabel
	    { $$ = $1; }
	;

extensStm :
	  sxLabel equalSign EXTENS SLASH extensMinor endOfLine
	    { $$ = new extensStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

extensMinor :
	  vector
	    { $$ = new extensMinor_vector($1); }
	| VEC C vector C rentVal
	    { $$ = new extensMinor_VEC($3, $5); }
	;

extfilStm :
	  EXTFIL SLASH extfilMinor endOfLine
	    { $$ = new extfilStm($3);
	      dmisStms.push_back($$);
	    }
	;

extfilMinor :
	  extfilDmis
	    { $$ = $1; }
	| extfilDme
	    { $$ = $1; }
	;

extfilDmis :
	  DMIS C stringVal
	    { $$ = new extfilDmis($3); }
	;

extfilDme :
	  DME C stringVal
	    { $$ = new extfilDme($3); }
	;

featArc1Stm :
	  fLabel equalSign FEAT SLASH featMinorArc1 endOfLine
	    { $$ = new featArc1Nom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorArc1 endOfLine
	    { $$ = new featArc1Act($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featArc2Stm :
	  fLabel equalSign FEAT SLASH featMinorArc2 endOfLine
	    { $$ = new featArc2Nom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorArc2 endOfLine
	    { $$ = new featArc2Act($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featCircleStm :
	  fLabel equalSign FEAT SLASH featMinorCircle endOfLine
	    { $$ = new featCircleNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorCircle endOfLine
	    { $$ = new featCircleAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featCompoundStm :
	  fLabel equalSign FEAT SLASH featMinorCompound endOfLine
	    { $$ = new featCompoundStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featConeStm :
	  fLabel equalSign FEAT SLASH featMinorCone endOfLine
	    { $$ = new featConeNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorCone endOfLine
	    { $$ = new featConeAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featConradsegStm :
	  fLabel equalSign FEAT SLASH featMinorConradseg endOfLine
	    { $$ = new featConradsegNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorConradseg endOfLine
	    { $$ = new featConradsegAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featCparlnStm :
	  fLabel equalSign FEAT SLASH featMinorCparln endOfLine
	    { $$ = new featCparlnNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorCparln endOfLine
	    { $$ = new featCparlnAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featCylndrStm :
	  fLabel equalSign FEAT SLASH featMinorCylndr endOfLine
	    { $$ = new featCylndrNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorCylndr endOfLine
	    { $$ = new featCylndrAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featCylradsegStm :
	  fLabel equalSign FEAT SLASH featMinorCylradseg endOfLine
	    { $$ = new featCylradsegNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorCylradseg endOfLine
	    { $$ = new featCylradsegAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featEdgeptStm :
	  fLabel equalSign FEAT SLASH featMinorEdgept endOfLine
	    { $$ = new featEdgeptNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorEdgept endOfLine
	    { $$ = new featEdgeptAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featEllipsStm :
	  fLabel equalSign FEAT SLASH featMinorEllips endOfLine
	    { $$ = new featEllipsNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorEllips endOfLine
	    { $$ = new featEllipsAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featElongcylStm :
	  fLabel equalSign FEAT SLASH featMinorElongcyl endOfLine
	    { $$ = new featElongcylNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorElongcyl endOfLine
	    { $$ = new featElongcylAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featGcurveStm :
	  fLabel equalSign FEAT SLASH featMinorGcurve endOfLine
	    { $$ = new featGcurveStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featGeomStm :
	  fLabel equalSign FEAT SLASH featMinorGeom endOfLine
	    { $$ = new featGeomStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featGsurfStm :
	  fLabel equalSign FEAT SLASH featMinorGsurf endOfLine
	    { $$ = new featGsurfStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featLineStm :
	  fLabel equalSign FEAT SLASH featMinorLine endOfLine
	    { $$ = new featLineNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorLine endOfLine
	    { $$ = new featLineAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featObjectStm :
	  fLabel equalSign FEAT SLASH featMinorObject endOfLine
	    { $$ = new featObjectNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorObject endOfLine
	    { $$ = new featObjectAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featParplnStm :
	  fLabel equalSign FEAT SLASH featMinorParpln endOfLine
	    { $$ = new featParplnNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorParpln endOfLine
	    { $$ = new featParplnAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featPaternStm :
	  fLabel equalSign FEAT SLASH featMinorPatern endOfLine
	    { $$ = new featPaternStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featPlaneStm :
	  fLabel equalSign FEAT SLASH featMinorPlane endOfLine
	    { $$ = new featPlaneNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorPlane endOfLine
	    { $$ = new featPlaneAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featPointStm :
	  fLabel equalSign FEAT SLASH featMinorPoint endOfLine
	    { $$ = new featPointNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorPoint endOfLine
	    { $$ = new featPointAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featRctnglStm :
	  fLabel equalSign FEAT SLASH featMinorRctngl endOfLine
	    { $$ = new featRctnglNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorRctngl endOfLine
	    { $$ = new featRctnglAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featRevsurfStm :
	  fLabel equalSign FEAT SLASH featMinorRevsurf endOfLine
	    { $$ = new featRevsurfNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorRevsurf endOfLine
	    { $$ = new featRevsurfAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featSphereStm :
	  fLabel equalSign FEAT SLASH featMinorSphere endOfLine
	    { $$ = new featSphereNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorSphere endOfLine
	    { $$ = new featSphereAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featSphradsegStm :
	  fLabel equalSign FEAT SLASH featMinorSphradseg endOfLine
	    { $$ = new featSphradsegNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorSphradseg endOfLine
	    { $$ = new featSphradsegAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featSymplnStm :
	  fLabel equalSign FEAT SLASH featMinorSympln endOfLine
	    { $$ = new featSymplnNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorSympln endOfLine
	    { $$ = new featSymplnAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featTorradsegStm :
	  fLabel equalSign FEAT SLASH featMinorTorradseg endOfLine
	    { $$ = new featTorradsegNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorTorradseg endOfLine
	    { $$ = new featTorradsegAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featTorusStm :
	  fLabel equalSign FEAT SLASH featMinorTorus endOfLine
	    { $$ = new featTorusNom($1, $5);
	      dmisStms.push_back($$);
	    }
	| faLabel equalSign FEAT SLASH featMinorTorus endOfLine
	    { $$ = new featTorusAct($1, $5);
	      dmisStms.push_back($$);
	    }
	;

featMinorArc1 :
	  ARC C matDir C typePoint C vector C rentVal C angle C angle
	    { $$ = new featMinorArc1($3, $5, $7, $9, $11, $13, 0); }
	| ARC C matDir C typePoint C vector C rentVal C angle C angle C vector
	    { $$ = new featMinorArc1($3, $5, $7, $9, $11, $13, $15); }
	;

featMinorArc2 :
	  ARC C FOURPOINT C matDir C impCartPoint C impCartPoint C impCartPoint C impCartPoint
	    { $$ = new featMinorArc2($5, $7, $9, $11, $13); }
	;

featMinorCircle :
	  CIRCLE C matDir C typePoint C vector C rentVal
	    { $$ = new featMinorCircle($3, $5, $7, $9); }
	;

featMinorCompound :
	  COMPOUND C AXIAL C typePoint C vector C fLabel C featureNominalList
	    { $$ = new featCompoundAxial($5, $7, $9, $11); }
	| COMPOUND C PLANE C typePoint C vector C fLabel C featureNominalList
	    { $$ = new featCompoundPlane($5, $7, $9, $11); }
	| COMPOUND C SPHERE C typePoint C fLabel C featureNominalList
	    { $$ = new featCompoundSphere($5, $7, $9); }
	;

featMinorCone :
	  CONE C matDir C typePoint C vector C angle
	    { $$ = new featMinorCone($3, $5, $7, $9); }
	;

featMinorConradseg :
	  CONRADSEGMNT C matDir C cartPoint C impCartPoint C rentVal C rentVal C vector C vector
	    { $$ = new featConradsegCart($3, $5, $7, $9, $11, $13, $15); }
	| CONRADSEGMNT C matDir C polPoint C impPolPoint C rentVal C rentVal C vector C vector
	    { $$ = new featConradsegPol($3, $5, $7, $9, $11, $13, $15); }
	;

featMinorCparln :
	  CPARLN C matDir C flatRoundOpen C typePoint C vector C vector C rentVal C rentVal
	    { $$ = new featMinorCparln($3, $5, $7, $9, $11, $13, $15); }
	;

featMinorCylndr :
	  CYLNDR C matDir C typePoint C vector C rentVal
	    { $$ = new featMinorCylndr($3, $5, $7, $9, 0); }
	| CYLNDR C matDir C typePoint C vector C rentVal C rentVal
	    { $$ = new featMinorCylndr($3, $5, $7, $9, $11); }
	;

featMinorCylradseg :
	  CYLRADSEGMNT C matDir C cartPoint C impCartPoint C rentVal C vector C vector
	    { $$ = new featCylradsegCart($3, $5, $7, $9, $11, $13); }
	| CYLRADSEGMNT C matDir C polPoint C impPolPoint C rentVal C vector C vector
	    { $$ = new featCylradsegPol($3, $5, $7, $9, $11, $13); }
	;

featMinorEdgept :
	  EDGEPT C typePoint C vector C vector
	    { $$ = new featMinorEdgept($3, $5, $7); }
	;

featMinorEllips :
	  ELLIPS C matDir C cartPoint C impCartPoint C radiusSpec C vector C rentVal
	    { $$ = new featEllipsCart($3, $5, $7, $9, $11, $13); }
	| ELLIPS C matDir C polPoint C impPolPoint C radiusSpec C vector C rentVal
	    { $$ = new featEllipsPol($3, $5, $7, $9, $11, $13); }
	;

featMinorElongcyl :
	  ELONGCYL C matDir C typePoint C vector C vector C rentVal C rentVal
	    { $$ = new featMinorElongcyl($3, $5, $7, $9, $11, $13, 0); }
	| ELONGCYL C matDir C typePoint C vector C vector C rentVal C rentVal C rentVal
	    { $$ = new featMinorElongcyl($3, $5, $7, $9, $11, $13, $15); }
	;

featMinorGeom :
	  GEOM C gLabel C coordType
	    { $$ = new featMinorGeom($3, $5); }
	;

featMinorGcurve :
	  GCURVE C CART C pointVecCart
	    { $$ = new featGcurveCart($5, 0, 0); }
	| GCURVE C CART C pointVecCart C PTDATA C pointVecCart C pointVecCartList
	    { $$ = new featGcurveCart($5, $9, $11); }
	| GCURVE C POL C pointVecPol
	    { $$ = new featGcurvePol($5, 0, 0); }
	| GCURVE C POL C pointVecPol C PTDATA C pointVecPol C pointVecPolList
	    { $$ = new featGcurvePol($5, $9, $11); }
	;

featMinorGsurf :
	  GSURF
	    { $$ = new featGsurfPlain(); }
	| GSURF C CART
	    { $$ = new featGsurfCart(0, 0, 0); }
	| GSURF C CART C PTDATA C pointVecCart C pointVecCart C pointVecCartList
	    { $$ = new featGsurfCart($7, $9, $11); }
	| GSURF C POL
	    { $$ = new featGsurfPol(0, 0, 0); }
	| GSURF C POL C PTDATA C pointVecPol C pointVecPol C pointVecPolList
	    { $$ = new featGsurfPol($7, $9, $11); }
	;

featMinorLine :
	  LINE C BND C cartPoint C impCartPoint C vector
	    { $$ = new featLineBndCart($5, $7, $9); }
	| LINE C BND C polPoint C impPolPoint C vector
	    { $$ = new featLineBndPol($5, $7, $9); }
	| LINE C UNBND C cartPoint C vector C vector
	    { $$ = new featLineUnbndCart($5, $7, $9); }
	| LINE C UNBND C polPoint C vector C vector
	    { $$ = new featLineUnbndPol($5, $7, $9); }
	;

featMinorObject :
	  OBJECT C paramList
	    { $$ = new featMinorObject($3); }
	;

featMinorParpln :
	  PARPLN C matDir C cartPoint C pointVecCart C pointVecCart C rentVal
	    { $$ = new featParplnPointsCart($3, $5, $7, $9, $11); }
	| PARPLN C matDir C polPoint C pointVecPol C pointVecPol C rentVal
	    { $$ = new featParplnPointsPol($3, $5, $7, $9, $11); }
	| PARPLN C matDir C MIDPL C cartPoint C vector C rentVal
	    { $$ = new featParplnMidplCart($3, $7, $9, $11); }
	| PARPLN C matDir C MIDPL C polPoint C vector C rentVal
	    { $$ = new featParplnMidplPol($3, $7, $9, $11); }
	;

featMinorPatern :
	  PATERN C fLabel C featureNominalList
	    { $$ = new featMinorPatern($3, $5); }
	;

featMinorPlane :
	  PLANE C typePoint C vector
	    { $$ = new featMinorPlane($3, $5); }
	;

featMinorPoint :
	  POINT C typePoint C vector
	    { $$ = new featMinorPoint($3, $5); }
	;

featMinorRctngl :
	  RCTNGL C matDir C typePoint C vector C rentVal C vector C rentVal C vector C rentVal
	    { $$ = new featMinorRctngl($3, $5, $7, $9, $11, $13, $15, $17); }
	;

featMinorRevsurf :
	  REVSURF C matDir C typePoint C vector C cartPoint C impCartPointList
	    { $$ = new featRevsurfCart($3, $5, $7, $9, $11); }
	| REVSURF C matDir C typePoint C vector C polPoint C impPolPointList
	    { $$ = new featRevsurfPol($3, $5, $7, $9, $11); }
	;

featMinorSphere :
	  SPHERE C matDir C typePoint C rentVal
	    { $$ = new featMinorSphere($3, $5, $7, 0, 0); }
	| SPHERE C matDir C typePoint C rentVal C vector
	    { $$ = new featMinorSphere($3, $5, $7, $9, 0); }
	| SPHERE C matDir C typePoint C rentVal C vector C angle
	    { $$ = new featMinorSphere($3, $5, $7, $9, $11); }
	;

featMinorSphradseg :
	  SPHRADSEGMNT C matDir C typePoint C rentVal C vector C rentVal C rentVal C vector C rentVal C rentVal
	    { $$ = new featMinorSphradseg($3, $5, $7, $9, $11, $13, $15, $17, $19); }
	;

featMinorSympln :
	  SYMPLN C matDir C cartPoint C pointVecCart C pointVecCart C rentVal
	    { $$ = new featSymplnCart($3, $5, $7, $9, $11); }
	| SYMPLN C matDir C polPoint C pointVecPol C pointVecPol C rentVal
	    { $$ = new featSymplnPol($3, $5, $7, $9, $11); }
	;

featMinorTorradseg :
	  TORRADSEGMNT C matDir C typePoint C rentVal C rentVal C vector C rentVal C rentVal C vector C rentVal C rentVal
	    { $$ = new featMinorTorradseg($3, $5, $7, $9, $11, $13, $15, $17, $19, $21); }
	;

featMinorTorus :
	  TORUS C matDir C typePoint C vector C rentVal C rentVal
	    { $$ = new featMinorTorus($3, $5, $7, $9, $11); }
	;

fedratStm :
	  FEDRAT SLASH fedratMinor endOfLine
	    { $$ = new fedratStm($3);
	      dmisStms.push_back($$);
	    }
	;

fedratMinor :
	  fedratMeas
	    { $$ = $1; }
	| fedratPos
	    { $$ = $1; }
	| fedratRot
	    { $$ = $1; }
	| fedratScan
	    { $$ = $1; }
	| fedratHedRot
	    { $$ = $1; }
	| fedratHedMeas
	    { $$ = $1; }
	| fedratHedScan
	    { $$ = $1; }
	;

fedratMeas :
	  MESVEL C fedratLinSpec
	    { $$ = new fedratMeas($3); }
	;

fedratPos :
	  POSVEL C fedratLinSpec
	    { $$ = new fedratPos($3); }
	;

fedratRot :
	  ROTVEL C fedratAngSpec
	    { $$ = new fedratRot($3); }
	;

fedratScan :
	  SCNVEL C fedratLinSpec
	    { $$ = new fedratScan($3); }
	;

fedratHedRot :
	  HEDROTVEL C fedratLinSpec
	    { $$ = new fedratHedRot($3); }
	;

fedratHedMeas :
	  HEDMESVEL C fedratLinSpec
	    { $$ = new fedratHedMeas($3); }
	;

fedratHedScan :
	  HEDSCNVEL C fedratLinSpec
	    { $$ = new fedratHedScan($3); }
	;

fedratAngSpec :
	  fedratAngular
	    { $$ = $1; }
	| fedratDef
	    { $$ = $1; }
	;

fedratLinSpec :
	  fedratLinear
	    { $$ = $1; }
	| fedratDef
	    { $$ = $1; }
	;

fedratLinear :
	  MPM C rentVal
	    { $$ = new fedratLinear_MPM($3); }
	| MMPS C rentVal
	    { $$ = new fedratLinear_MMPS($3); }
	| IPM C rentVal
	    { $$ = new fedratLinear_IPM($3); }
	| IPS C rentVal
	    { $$ = new fedratLinear_IPS($3); }
	;

fedratAngular :
	  RPM C rentVal
	    { $$ = new fedratAngular($3); }
	;

fedratDef :
	  PCENT C rentVal
	    { $$ = new fedratDef_PCENT($3); }
	| HIGH
	    { $$ = new fedratDef_HIGH(); }
	| LOW
	    { $$ = new fedratDef_LOW(); }
	| DEFALT
	    { $$ = new fedratDef_DEFALT(); }
	;

fildefStm :
	  vfLabel equalSign FILDEF SLASH CODE C intVal endOfLine
	    { $$ = new fildefStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

filnamStm :
	  FILNAM SLASH stringConst C versionTag endOfLine
	    { $$ = new filnamStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

finposStm :
	  FINPOS SLASH state endOfLine
	    { $$ = new finposStm($3);
	      dmisStms.push_back($$);
	    }
	;

fixtidStm :
	  fiLabel equalSign FIXTID SLASH stringVal endOfLine
	    { $$ = new fixtidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

fixtsnStm :
	  fsLabel equalSign FIXTSN SLASH stringVal endOfLine
	    { $$ = new fixtsnStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

flyStm :
	  FLY SLASH flyMinor endOfLine
	    { $$ = new flyStm($3);
	      dmisStms.push_back($$);
	    }
	;

flyMinor :
	  OFF
	    { $$ = new flyMinor_OFF(); }
	| rentVal
	    { $$ = new flyMinor_rentVal($1); }
	;

fromStm :
	  FROM SLASH fromMinor endOfLine
	    { $$ = new fromStm($3);
	      dmisStms.push_back($$);
	    }
	;

fromMinor :
	  cartPoint
	    { $$ = new fromMinor_cartPoint($1, false); }
	| cartPoint C RAM
	    { $$ = new fromMinor_cartPoint($1, true); }
	| DME
	    { $$ = new fromMinor_DME(); }
	| impCartPoint
	    { $$ = new fromMinor_impCartPoint($1, false); }
	| impCartPoint C RAM
	    { $$ = new fromMinor_impCartPoint($1, true); }
	| polPoint
	    { $$ = new fromMinor_polPoint($1, false); }
	| polPoint C RAM
	    { $$ = new fromMinor_polPoint($1, true); }
	| SCALE
	    { $$ = new fromMinor_SCALE(); }
	;

geoalgStm :
	  GEOALG SLASH geoalgMinor endOfLine
	    { $$ = new geoalgStm($3);
	      dmisStms.push_back($$);
	    }
	;

geoalgMinor :
	  geoalgArc
	    { $$ = $1; }
	| geoalgCircle
	    { $$ = $1; }
	| geoalgCone
	    { $$ = $1; }
	| geoalgConradsegmnt
	    { $$ = $1; }
	| geoalgCparln
	    { $$ = $1; }
	| geoalgCylndr
	    { $$ = $1; }
	| geoalgCylradsegmnt
	    { $$ = $1; }
	| geoalgEllips
	    { $$ = $1; }
	| geoalgElongcyl
	    { $$ = $1; }
	| geoalgGcurve
	    { $$ = $1; }
	| geoalgGsurf
	    { $$ = $1; }
	| geoalgLine
	    { $$ = $1; }
	| geoalgObject
	    { $$ = $1; }
	| geoalgParpln
	    { $$ = $1; }
	| geoalgPlane
	    { $$ = $1; }
	| geoalgRctngl
	    { $$ = $1; }
	| geoalgRevsurf
	    { $$ = $1; }
	| geoalgSphere
	    { $$ = $1; }
	| geoalgSphradsegmnt
	    { $$ = $1; }
	| geoalgSympln
	    { $$ = $1; }
	| geoalgTorus
	    { $$ = $1; }
	| geoalgTorradsegmnt
	    { $$ = $1; }
	;

geoalgArc :
	  ARC C geoalgSpec1
	    { $$ = new geoalgArc($3); }
	;

geoalgCircle :
	  CIRCLE C geoalgSpec2
	    { $$ = new geoalgCircle($3); }
	;

geoalgCone :
	  CONE C geoalgSpec2
	    { $$ = new geoalgCone($3); }
	;

geoalgConradsegmnt :
	  CONRADSEGMNT C geoalgSpec2
	    { $$ = new geoalgConradsegmnt($3); }
	;

geoalgCparln :
	  CPARLN C geoalgSpec3
	    { $$ = new geoalgCparln($3); }
	;

geoalgCylndr :
	  CYLNDR C geoalgSpec2
	    { $$ = new geoalgCylndr($3); }
	;

geoalgCylradsegmnt :
	  CYLRADSEGMNT C geoalgSpec2
	    { $$ = new geoalgCylradsegmnt($3); }
	;

geoalgEllips :
	  ELLIPS C geoalgSpec1
	    { $$ = new geoalgEllips($3); }
	;

geoalgElongcyl :
	  ELONGCYL C geoalgSpec2
	    { $$ = new geoalgElongcyl($3); }
	;

geoalgGcurve :
	  GCURVE C geoalgSpec4
	    { $$ = new geoalgGcurve($3); }
	;

geoalgGsurf :
	  GSURF C geoalgSpec5
	    { $$ = new geoalgGsurf($3); }
	;

geoalgLine :
	  LINE C geoalgSpec1
	    { $$ = new geoalgLine($3); }
	;

geoalgObject :
	  OBJECT C geoalgSpec3
	    { $$ = new geoalgObject($3); }
	;

geoalgParpln :
	  PARPLN C geoalgSpec2
	    { $$ = new geoalgParpln($3); }
	;

geoalgPlane :
	  PLANE C geoalgSpec1
	    { $$ = new geoalgPlane($3); }
	;

geoalgRctngl :
	  RCTNGL C geoalgSpec1
	    { $$ = new geoalgRctngl($3); }
	;

geoalgRevsurf :
	  REVSURF C geoalgSpec6
	    { $$ = new geoalgRevsurf($3); }
	;

geoalgSphere :
	  SPHERE C geoalgSpec2
	    { $$ = new geoalgSphere($3); }
	;

geoalgSphradsegmnt :
	  SPHRADSEGMNT C geoalgSpec2
	    { $$ = new geoalgSphradsegmnt($3); }
	;

geoalgSympln :
	  SYMPLN C geoalgSpec2
	    { $$ = new geoalgSympln($3); }
	;

geoalgTorus :
	  TORUS C geoalgSpec2
	    { $$ = new geoalgTorus($3); }
	;

geoalgTorradsegmnt :
	  TORRADSEGMNT C geoalgSpec2
	    { $$ = new geoalgTorradsegmnt($3); }
	;

geoalgSpec1 :
	  LSTSQR
	    { $$ = new geoalgSpec1_LSTSQR(0); }
	| LSTSQR C geoalgFilterSettings
	    { $$ = new geoalgSpec1_LSTSQR($3); }
	| MINMAX
	    { $$ = new geoalgSpec1_MINMAX(0); }
	| MINMAX C geoalgFilterSettings
	    { $$ = new geoalgSpec1_MINMAX($3); }
	| DEFALT
	    { $$ = new geoalgSpec1_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec1_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec1_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec1_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec1_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec1_EXTERN($3, $5, $7); }
	;

geoalgSpec2 :
	  LSTSQR
	    { $$ = new geoalgSpec2_LSTSQR(0); }
	| LSTSQR C geoalgFilterSettings
	    { $$ = new geoalgSpec2_LSTSQR($3); }
	| MINMAX
	    { $$ = new geoalgSpec2_MINMAX(0); }
	| MINMAX C geoalgFilterSettings
	    { $$ = new geoalgSpec2_MINMAX($3); }
	| MAXINS
	    { $$ = new geoalgSpec2_MAXINS(0); }
	| MAXINS C geoalgFilterSettings
	    { $$ = new geoalgSpec2_MAXINS($3); }
	| MINCIR
	    { $$ = new geoalgSpec2_MINCIR(0); }
	| MINCIR C geoalgFilterSettings
	    { $$ = new geoalgSpec2_MINCIR($3); }
	| DEFALT
	    { $$ = new geoalgSpec2_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec2_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec2_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec2_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec2_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec2_EXTERN($3, $5, $7); }
	;

geoalgSpec3 :
	  DEFALT
	    { $$ = new geoalgSpec3_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec3_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec3_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec3_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec3_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec3_EXTERN($3, $5, $7); }
	;

geoalgSpec4 :
	  LSTSQR
	    { $$ = new geoalgSpec4_LSTSQR(0); }
	| LSTSQR C geoalgFilterSettings
	    { $$ = new geoalgSpec4_LSTSQR($3); }
	| MINMAX
	    { $$ = new geoalgSpec4_MINMAX(0); }
	| MINMAX C geoalgFilterSettings
	    { $$ = new geoalgSpec4_MINMAX($3); }
	| BSPLIN
	    { $$ = new geoalgSpec4_BSPLIN(0); }
	| BSPLIN C geoalgFilterSettings
	    { $$ = new geoalgSpec4_BSPLIN($3); }
	| DEFALT
	    { $$ = new geoalgSpec4_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec4_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec4_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec4_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec4_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec4_EXTERN($3, $5, $7); }
	;

geoalgSpec5 :
	  LSTSQR
	    { $$ = new geoalgSpec5_LSTSQR(0); }
	| LSTSQR C geoalgFilterSettings
	    { $$ = new geoalgSpec5_LSTSQR($3); }
	| MINMAX
	    { $$ = new geoalgSpec5_MINMAX(0); }
	| MINMAX C geoalgFilterSettings
	    { $$ = new geoalgSpec5_MINMAX($3); }
	| BEZIER
	    { $$ = new geoalgSpec5_BEZIER(0); }
	| BEZIER C geoalgFilterSettings
	    { $$ = new geoalgSpec5_BEZIER($3); }
	| NURBS
	    { $$ = new geoalgSpec5_NURBS(0); }
	| NURBS C geoalgFilterSettings
	    { $$ = new geoalgSpec5_NURBS($3); }
	| DEFALT
	    { $$ = new geoalgSpec5_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec5_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec5_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec5_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec5_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec5_EXTERN($3, $5, $7); }
	;

geoalgSpec6 :
	  LSTSQR
	    { $$ = new geoalgSpec6_LSTSQR(0); }
	| LSTSQR C geoalgFilterSettings
	    { $$ = new geoalgSpec6_LSTSQR($3); }
	| BSPLIN
	    { $$ = new geoalgSpec6_BSPLIN(0); }
	| BSPLIN C geoalgFilterSettings
	    { $$ = new geoalgSpec6_BSPLIN($3); }
	| DEFALT
	    { $$ = new geoalgSpec6_DEFALT(0); }
	| DEFALT C geoalgFilterSettings
	    { $$ = new geoalgSpec6_DEFALT($3); }
	| EXTERN C geoalgExternFunc
	    { $$ = new geoalgSpec6_EXTERN($3, 0, 0); }
	| EXTERN C geoalgExternFunc C geoalgFilterSettings
	    { $$ = new geoalgSpec6_EXTERN($3, 0, $5); }
	| EXTERN C geoalgExternFunc C paramList
	    { $$ = new geoalgSpec6_EXTERN($3, $5, 0); }
	| EXTERN C geoalgExternFunc C paramList C geoalgFilterSettings
	    { $$ = new geoalgSpec6_EXTERN($3, $5, $7); }
	;

geoalgExternFunc :
	  DMIS C mLabel
	    { $$ = new geoalgExternFunc_DMIS($3);
	      aLabelFound = 1; }
	| DME C stringVal
	    { $$ = new geoalgExternFunc_DME($3); }
	| SYS C stringVal
	    { $$ = new geoalgExternFunc_SYS($3); }
	;

geoalgFilterSettings :
	  geoalgFilter
	    { $$ = new geoalgFiltSetFilt(0, $1); }
	| geoalgEliminate C geoalgFilter
	    { $$ = new geoalgFiltSetFilt($1, $3); }
	| geoalgEliminate
	    { $$ = new geoalgFiltSetElim($1); }
	;

geoalgEliminate :
	  ELIMINATE C STDDEVLIMIT C rentVal
	    { $$ = new geoalgEliminateStd($5); }
	| ELIMINATE C OFF
	    { $$ = new geoalgEliminateOff(); }
	;

geoalgFilter :
	  FILTER C LAMBDAC C geoalgFilterType
	    { $$ = new geoalgFilterLambda($5); }
	| FILTER C CIRCULAR C geoalgFilterType
	    { $$ = new geoalgFilterCirc($5); }
	| FILTER C OFF
	    { $$ = new geoalgFilterOff(); }
	;

geoalgFilterType :
	  LOWPASS C rentVal C geoalgFilterCurve
	    { $$ = new geoalgFilterType_LOWPASS($3, $5); }
	| HIGHPASS C rentVal C geoalgFilterCurve
	    { $$ = new geoalgFilterType_HIGHPASS($3, $5); }
	| BANDPASS C rentVal C rentVal C geoalgFilterCurve
	    { $$ = new geoalgFilterType_BANDPASS($3, $5, $7); }
	;

geoalgFilterCurve :
	  GAUSS
	    { $$ = new geoalgFilterCurve_GAUSS(); }
	| TWORC
	    { $$ = new geoalgFilterCurve_TWORC(); }
	| SPLINE
	    { $$ = new geoalgFilterCurve_SPLINE(); }
	| RECFILT
	    { $$ = new geoalgFilterCurve_RECFILT(); }
	;

geomStm :
	  gLabel equalSign GEOM SLASH geomMinor endOfLine
	    { $$ = new geomStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

geomMinor :
	  NONE
	    { $$ = new geomStmNone(); }
	| didLabel
	    { $$ = new geomStmDid($1); }
	| gLabel
	    { $$ = new geomStmOffset($1, 0); }
	| gLabel C OFFSET C rentVal
	    { $$ = new geomStmOffset($1, $5); }
	| gLabel C ENTITY C geomEntityList
	    { $$ = new geomStmEntity($1, $5); }
	;

geomEntityList :
	  stringVal
	    { $$ = new std::list<geomEntityItem *>;
	      $$->push_front(new geomEntityItem($1, 0)); }
	| stringVal C geomEntityList
	    { $$ = $3;
	      $$->push_front(new geomEntityItem($1, 0)); }
	| stringVal C OFFSET C rentVal
	    { $$ = new std::list<geomEntityItem *>;
	      $$->push_front(new geomEntityItem($1, $5)); }
	| stringVal C OFFSET C rentVal C geomEntityList
	    { $$ = $7;
	      $$->push_front(new geomEntityItem($1, $5)); }
	;

gohomeStm :
	  GOHOME endOfLine
	    { $$ = new gohomeStm();
	      dmisStms.push_back($$);
	    }
	;

gotargStm :
	  GOTARG SLASH impCartPoint endOfLine
	    { $$ = new gotargXyz($3);
	      dmisStms.push_back($$);
	    }
	| GOTARG SLASH cartPoint endOfLine
	    { $$ = new gotargCart($3);
	      dmisStms.push_back($$);
	    }
	| GOTARG SLASH polPoint endOfLine
	    { $$ = new gotargPol($3);
	      dmisStms.push_back($$);
	    }
	;

gotoStm :
	  GOTO SLASH gotoMinor endOfLine
	    { $$ = new gotoStm($3);
	      dmisStms.push_back($$);
	    }
	;

gotoMinor :
	  gotoAbs
	    { $$ = $1; }
	| gotoRel
	    { $$ = $1; }
	| gotoArc
	    { $$ = $1; }
	| gotoAxis
	    { $$ = $1; }
	| probeOrient
	    { $$ = $1; }
	;

gotoAxis :
	  gotoAxisSpec
	    { $$ = new gotoAxis($1, 0, 0); }
	| gotoAxisSpec C gotoAxisSpec
	    { $$ = new gotoAxis($1, $3, 0); }
	| gotoAxisSpec C gotoAxisSpec C gotoAxisSpec
	    { $$ = new gotoAxis($1, $3, $5); }
	;

gotoAxisSpec :
	  axis C rentVal
	    { $$ = new gotoAxisSpec($1, $3); }
	;

gotoAbs :
	  impCartPoint
	    { $$ = new gotoAbsXyz($1); }
	| impCartPoint C sensorMove
	    { $$ = new gotoAbsXyzSens($1, $3); }
	| cartPoint
	    { $$ = new gotoAbsCart($1); }
	| cartPoint C sensorMove
	    { $$ = new gotoAbsCartSens($1, $3); }
	| polPoint
	    { $$ = new gotoAbsPol($1); }
	| polPoint C sensorMove
	    { $$ = new gotoAbsPolSens($1, $3); }
	;

sensorMove :
	  saLabel
	    { $$ = new sensorMove_saLabel($1); }
	| gotoWristList
	    { $$ = new sensorMove_gotoWristList($1); }
	| probeOrient
	    { $$ = new sensorMove_probeOrient($1); }
	;

gotoRel :
	  INCR C rentVal C vector
	    { $$ = new gotoRel($3, $5); }
	;

gotoArc :
	  ARC C impCartPoint C impCartPoint
	    { $$ = new gotoArc($3, $5, 0); }
	| ARC C impCartPoint C impCartPoint C probeOrient
	    { $$ = new gotoArc($3, $5, $7); }
	;

probeOrient :
	  featureLabel
	    { $$ = new probeOrient_featureLabel($1, 0); }
	| featureLabel C FZ C angle
	    { $$ = new probeOrient_featureLabel($1, $5); }
	| VEC C vector
	    { $$ = new probeOrient_VEC($3, 0); }
	| VEC C vector C FZ C angle
	    { $$ = new probeOrient_VEC($3, $7); }
	| PCS C euler
	    { $$ = new probeOrient_PCS($3); }
	| HEADCS C rentVal C rentVal
	    { $$ = new probeOrient_HEADCS($3, $5, 0); }
	| HEADCS C rentVal C rentVal C rentVal
	    { $$ = new probeOrient_HEADCS($3, $5, $7); }
	;

gotoWristList :
	  swLabel C gotoWristAngleList
	    { $$ = new std::list<gotoWristItem *>;
	      $$->push_front(new gotoWristItem($1, $3)); }
	| swLabel C gotoWristAngleList C gotoWristList
	    { $$ = $5;
	      $$->push_front(new gotoWristItem($1, $3)); }
	;

gotoWristAngleList :
	  gotoWristAngleItem
	    { $$ = new std::list<gotoWristAngleItem *>;
	      $$->push_back($1); }
	| gotoWristAngleList C gotoWristAngleItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

gotoWristAngleItem :
	  stringVal C angle
	    { $$ = new gotoWristAngleItem($1, $3); }
	;

groupStm :
	  gsaLabel equalSign GROUP SLASH saLabel C saLabelList endOfLine
	    { $$ = new groupStm($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

ifStm :
	  IF SLASH LPAREN boolVal RPAREN endOfLine
	    { $$ = new ifStm($4);
	      dmisStms.push_back($$);
	    }
	;

includStm :
	  INCLUD SLASH includMinor endOfLine
	    { $$ = new includStm($3);
	      dmisStms.push_back($$);
	    }
	;

includMinor :
	  includDmis
	    { $$ = $1; }
	| includDme
	    { $$ = $1; }
	;

includDmis :
	  DMIS C stringConst
	    { $$ = new includDmis($3); }
	;

includDme :
	  DME C stringConst
	    { $$ = new includDme($3); }
	;

iteratStm :
	  realVar EQUALS ITERAT SLASH iteratMinor endOfLine
	    { $$ = new iteratStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

iteratMinor :
	  jumpLabel C jumpLabel C iteratConvergence C intVal C iteratCriterionList
	    { $$ = new iteratMinor($1, $3, $5, $7, $9); }
	;

iteratConvergence :
	  rentVal C ABSL
	    { $$ = new iteratConvergence_ABSL($1); }
	| rentVal C INCR
	    { $$ = new iteratConvergence_INCR($1); }
	;

iteratCriterionList :
	  iteratCriterionStart C featureActualList
	    { $$ = new std::list<iteratCriterionItem *>;
	      $$->push_front(new iteratCriterionItem($1, $3)); }
	| iteratCriterionStart C featureActualList C iteratCriterionList
	    { $$ = $5;
	      $$->push_front(new iteratCriterionItem($1, $3)); }
	;

iteratCriterionStart :
	  axis
	    { $$ = new iteratCriterionStart_axis($1); }
	| vector
	    { $$ = new iteratCriterionStart_vector($1); }
	| NOM
	    { $$ = new iteratCriterionStart_NOM(); }
	;

jumptoStm :
	  JUMPTO SLASH jumpLabel endOfLine
	    { $$ = new jumptoStm($3);
	      dmisStms.push_back($$);
	    }
	;

keycharStm :
	  kcLabel equalSign KEYCHAR SLASH keycharMinor endOfLine
	    { $$ = new keycharStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

keycharMinor :
	  keycharOneFeature
	    { $$ = $1; }
	| keycharTwoFeatures
	    { $$ = $1; }
	;

keycharOneFeature :
	  fLabel C tLabelList
	    { $$ = new keycharOneFeature($1, $3, 0); }
	| fLabel C tLabelList C criticality
	    { $$ = new keycharOneFeature($1, $3, $5); }
	;

keycharTwoFeatures :
	  fLabel C fLabel C tLabel
	    { $$ = new keycharTwoFeatures($1, $3, $5, 0); }
	| fLabel C fLabel C tLabel C criticality
	    { $$ = new keycharTwoFeatures($1, $3, $5, $7); }
	;

criticality :
	  CRITICAL
	    { $$ = new criticality_CRITICAL(); }
	| MAJOR
	    { $$ = new criticality_MAJOR(); }
	| MINOR
	    { $$ = new criticality_MINOR(); }
	;

litdefStm :
	  vlLabel equalSign LITDEF SLASH litdefMinor endOfLine
	    { $$ = new litdefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

litdefMinor :
	  litdefLight
	    { $$ = $1; }
	| litdefStrobe
	    { $$ = $1; }
	;

litdefLight :
	  litdefLightType C vector
	    { $$ = new litdefLight($1, $3); }
	;

litdefLightType :
	  SURF
	    { $$ = new litdefLightType_SURF(); }
	| BACK
	    { $$ = new litdefLightType_BACK(); }
	| GRID
	    { $$ = new litdefLightType_GRID(); }
	| OBLQ
	    { $$ = new litdefLightType_OBLQ(); }
	;

litdefStrobe :
	  STROBE C litdefStrobeType C rentVal C vector
	    { $$ = new litdefStrobe($3, $5, $7); }
	;

litdefStrobeType :
	  CYCLE C rentVal
	    { $$ = new litdefStrobeType_CYCLE($3); }
	| TRIGER
	    { $$ = new litdefStrobeType_TRIGER(); }
	;

locateStm :
	  dLabel equalSign LOCATE SLASH locateLabelList endOfLine
	    { $$ = new locateStm($1, 0, 0, $5);
	      dmisStms.push_back($$);
	    }
	| dLabel equalSign LOCATE SLASH locateRotAllowed C locateLabelList endOfLine
	    { $$ = new locateStm($1, 0, $5, $7);
	      dmisStms.push_back($$);
	    }
	| dLabel equalSign LOCATE SLASH locateTransAllowed C locateLabelList endOfLine
	    { $$ = new locateStm($1, $5, 0, $7);
	      dmisStms.push_back($$);
	    }
	| dLabel equalSign LOCATE SLASH locateTransAllowed C locateRotAllowed C locateLabelList endOfLine
	    { $$ = new locateStm($1, $5, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

locateTransAllowed :
	  XDIR
	    { $$ = new locateTransAllowed_XDIR(); }
	| YDIR
	    { $$ = new locateTransAllowed_YDIR(); }
	| ZDIR
	    { $$ = new locateTransAllowed_ZDIR(); }
	| XYDIR
	    { $$ = new locateTransAllowed_XYDIR(); }
	| YZDIR
	    { $$ = new locateTransAllowed_YZDIR(); }
	| ZXDIR
	    { $$ = new locateTransAllowed_ZXDIR(); }
	| XYZDIR
	    { $$ = new locateTransAllowed_XYZDIR(); }
	| NOTRAN
	    { $$ = new locateTransAllowed_NOTRAN(); }
	;

locateRotAllowed :
	  XAXIS
	    { $$ = new locateRotAllowed_XAXIS(); }
	| YAXIS
	    { $$ = new locateRotAllowed_YAXIS(); }
	| ZAXIS
	    { $$ = new locateRotAllowed_ZAXIS(); }
	| XYAXIS
	    { $$ = new locateRotAllowed_XYAXIS(); }
	| YZAXIS
	    { $$ = new locateRotAllowed_YZAXIS(); }
	| ZXAXIS
	    { $$ = new locateRotAllowed_ZXAXIS(); }
	| XYZAXI
	    { $$ = new locateRotAllowed_XYZAXI(); }
	| NOROT
	    { $$ = new locateRotAllowed_NOROT(); }
	;

locateLabel :
	  faLabel
	    { $$ = $1; }
	| maLabel
	    { $$ = $1; }
	| datLabel
	    { $$ = $1; }
	;

locateLabelList :
	  locateLabel
	    { $$ = new std::list<locateLabel *>;
	      $$->push_back($1); }
	| locateLabelList C locateLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

lotidStm :
	  liLabel equalSign LOTID SLASH stringVal endOfLine
	    { $$ = new lotidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

macroStm :
	  mLabelConst equalSign MACRO endOfLine
	    { $$ = new macroStm($1, 0);
	      dmisStms.push_back($$);
	    }
	| mLabelConst equalSign MACRO SLASH macroParList endOfLine
	    { $$ = new macroStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

macroParList :
	  macroPar
	    { $$ = new std::list<macroPar *>;
	      $$->push_back($1); }
	| macroParList C macroPar
	    { $$ = $1;
	      $$->push_back($3); }
	;

macroPar :
	  MACROVARNAME
	    { $$ = new macroPar_MACROVARNAME($1); }
	| stringConst
	    { $$ = new macroPar_stringConst($1); }
	;

matdefStm :
	  maLabel equalSign MATDEF SLASH matdefMinor endOfLine
	    { $$ = new matdefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

matdefMinor :
	  matdefFeat
	    { $$ = $1; }
	| matdefGeom
	    { $$ = $1; }
	;

matdefFeat :
	  fLabel C matdefSpec
	    { $$ = new matdefFeat($1, $3); }
	;

matdefGeom :
	  gLabel C matdefSpec
	    { $$ = new matdefGeom($1, $3); }
	;

matdefSpec :
	  faLabelList
	    { $$ = new matdefSpec($1, 0); }
	| faLabelList C matdefType
	    { $$ = new matdefSpec($1, $3); }
	;

faLabelList :
	  faLabel
	    { $$ = new std::list<faLabel *>;
	      $$->push_back($1); }
	| faLabelList C faLabel
	    { $$ = $1;
	      $$->push_back($3); }
	;

matdefType :
	  matdefMating C matdefMethod C rentVal
	    { $$ = new matdefTypeZone($1, $3, $5, 0); }
	| matdefMating C matdefMethod C rentVal C matdefMat
	    { $$ = new matdefTypeZone($1, $3, $5, $7); }
	| matdefMating C matdefMethod C rentVal C rentVal
	    { $$ = new matdefTypeBand($1, $3, $5, $7, 0); }
	| matdefMating C matdefMethod C rentVal C rentVal C matdefMat
	    { $$ = new matdefTypeBand($1, $3, $5, $7, $9); }
	;

matdefMating :
	  PT2PT
	    { $$ = new matdefMating_PT2PT(); }
	| PT2LN
	    { $$ = new matdefMating_PT2LN(); }
	| PT2PL
	    { $$ = new matdefMating_PT2PL(); }
	| LN2LN
	    { $$ = new matdefMating_LN2LN(); }
	;

matdefMethod :
	  BF
	    { $$ = new matdefMethod_BF(); }
	| FZ
	    { $$ = new matdefMethod_FZ(); }
	;

matdefMat :
	  MMC C tLabel
	    { $$ = new matdefMat_MMC($3); }
	| LMC C tLabel
	    { $$ = new matdefMat_LMC($3); }
	;

measStm :
	  MEAS SLASH measMinor fLabel C intVal endOfLine
	    { $$ = new measStm($3, $4, $6);
	      dmisStms.push_back($$);
	    }
	;

measMinor :
	  measGeotype C
	    { $$ = new measMinorOther($1); }
	| POINT C
	    { $$ = new measMinorPoint(); }
	| POINT C COMP C measVar2 C
	    { $$ = new measMinorComp($5); }
	;

measGeotype :
	  ARC
	    { $$ = new measGeotype_ARC(); }
	| CIRCLE
	    { $$ = new measGeotype_CIRCLE(); }
	| CONE
	    { $$ = new measGeotype_CONE(); }
	| CONRADSEGMNT
	    { $$ = new measGeotype_CONRADSEGMNT(); }
	| CPARLN
	    { $$ = new measGeotype_CPARLN(); }
	| CYLNDR
	    { $$ = new measGeotype_CYLNDR(); }
	| CYLRADSEGMNT
	    { $$ = new measGeotype_CYLRADSEGMNT(); }
	| EDGEPT
	    { $$ = new measGeotype_EDGEPT(); }
	| ELLIPS
	    { $$ = new measGeotype_ELLIPS(); }
	| ELONGCYL
	    { $$ = new measGeotype_ELONGCYL(); }
	| GCURVE
	    { $$ = new measGeotype_GCURVE(); }
	| GSURF
	    { $$ = new measGeotype_GSURF(); }
	| LINE
	    { $$ = new measGeotype_LINE(); }
	| OBJECT
	    { $$ = new measGeotype_OBJECT(); }
	| PARPLN
	    { $$ = new measGeotype_PARPLN(); }
	| PLANE
	    { $$ = new measGeotype_PLANE(); }
	| REVSURF
	    { $$ = new measGeotype_REVSURF(); }
	| RCTNGL
	    { $$ = new measGeotype_RCTNGL(); }
	| SPHERE
	    { $$ = new measGeotype_SPHERE(); }
	| SPHRADSEGMNT
	    { $$ = new measGeotype_SPHRADSEGMNT(); }
	| SYMPLN
	    { $$ = new measGeotype_SYMPLN(); }
	| TORUS
	    { $$ = new measGeotype_TORUS(); }
	| TORRADSEGMNT
	    { $$ = new measGeotype_TORRADSEGMNT(); }
	;

measVar2 :
	  AXDIR
	    { $$ = new measVar2_AXDIR(); }
	| DME
	    { $$ = new measVar2_DME(); }
	| FEAT C measVar2Aux
	    { $$ = new measVar2_FEAT($3); }
	| POL
	    { $$ = new measVar2_POL(); }
	| SPH
	    { $$ = new measVar2_SPH(); }
	| VEC C vector
	    { $$ = new measVar2_VEC($3); }
	;

measVar2Aux :
	  faLabel
	    { $$ = $1; }
	| fLabel
	    { $$ = $1; }
	| gLabel
	    { $$ = $1; }
	;

mfgdevStm :
	  mdLabel equalSign MFGDEV SLASH stringVal endOfLine
	    { $$ = new mfgdevStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

modeStm :
	  MODE SLASH MAN endOfLine
	    { $$ = new modeStm(false, false);
	      dmisStms.push_back($$);
	    }
	| MODE SLASH PROG C MAN endOfLine
	    { $$ = new modeStm(false, true);
	      dmisStms.push_back($$);
	    }
	| MODE SLASH AUTO C MAN endOfLine
	    { $$ = new modeStm(true, false);
	      dmisStms.push_back($$);
	    }
	| MODE SLASH AUTO C PROG C MAN endOfLine
	    { $$ = new modeStm(true, true);
	      dmisStms.push_back($$);
	    }
	;

obtainStm :
	  boolVar EQUALS OBTAIN SLASH obtainLabeled endOfLine
	    { $$ = new obtainBool($1, $5);
	      dmisStms.push_back($$);
	    }
	| stringVar EQUALS OBTAIN SLASH obtainLabeled endOfLine
	    { $$ = new obtainString($1, $5);
	      dmisStms.push_back($$);
	    }
	| intVar EQUALS OBTAIN SLASH obtainLabeled endOfLine
	    { $$ = new obtainInt($1, $5);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS OBTAIN SLASH obtainLabeled endOfLine
	    { $$ = new obtainRealLabel($1, $5);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS OBTAIN SLASH obtainPoint endOfLine
	    { $$ = new obtainRealPoint($1, $5);
	      dmisStms.push_back($$);
	    }
	;

obtainLabeled :
	  datumLabel1 C intVal
	    { $$ = new obtainLabeled_datumLabel1($1, $3); }
	| featureLabel C intVal
	    { $$ = new obtainLabeled_featureLabel($1, $3); }
	| sensorLabel1 C intVal
	    { $$ = new obtainLabeled_sensorLabel1($1, $3); }
	| toleranceLabel C intVal
	    { $$ = new obtainLabeled_toleranceLabel($1, $3); }
	| didLabel C intVal
	    { $$ = new obtainLabeled_didLabel($1, $3); }
	| qisLabel C intVal
	    { $$ = new obtainLabeled_qisLabel($1, $3); }
	| crLabel C intVal
	    { $$ = new obtainLabeled_crLabel($1, $3); }
	| gLabel C intVal
	    { $$ = new obtainLabeled_gLabel($1, $3); }
	| gsaLabel C intVal
	    { $$ = new obtainLabeled_gsaLabel($1, $3); }
	| kcLabel C intVal
	    { $$ = new obtainLabeled_kcLabel($1, $3); }
	| maLabel C intVal
	    { $$ = new obtainLabeled_maLabel($1, $3); }
	| pLabel C intVal
	    { $$ = new obtainLabeled_pLabel($1, $3); }
	| rLabel C intVal
	    { $$ = new obtainLabeled_rLabel($1, $3); }
	| rmLabel C intVal
	    { $$ = new obtainLabeled_rmLabel($1, $3); }
	| rtLabel C intVal
	    { $$ = new obtainLabeled_rtLabel($1, $3); }
	| sgLabel C intVal
	    { $$ = new obtainLabeled_sgLabel($1, $3); }
	| sgsLabel C intVal
	    { $$ = new obtainLabeled_sgsLabel($1, $3); }
	| sraLabel C intVal
	    { $$ = new obtainLabeled_sraLabel($1, $3); }
	| ssLabel C intVal
	    { $$ = new obtainLabeled_ssLabel($1, $3); }
	| swLabel C intVal
	    { $$ = new obtainLabeled_swLabel($1, $3); }
	| sxLabel C intVal
	    { $$ = new obtainLabeled_sxLabel($1, $3); }
	| thLabel C intVal
	    { $$ = new obtainLabeled_thLabel($1, $3); }
	| vLabel C intVal
	    { $$ = new obtainLabeled_vLabel($1, $3); }
	| vaLabel C intVal
	    { $$ = new obtainLabeled_vaLabel($1, $3); }
	| vfLabel C intVal
	    { $$ = new obtainLabeled_vfLabel($1, $3); }
	| vlLabel C intVal
	    { $$ = new obtainLabeled_vlLabel($1, $3); }
	| vwLabel C intVal
	    { $$ = new obtainLabeled_vwLabel($1, $3); }
	;

obtainPoint :
	  featureLabel LBOX intVal RBOX C intVal
	    { $$ = new obtainPoint($1, $3, $6); }
	;

openStm :
	  OPEN SLASH openMinor endOfLine
	    { $$ = new openStm($3);
	      dmisStms.push_back($$);
	    }
	;

openMinor :
	  didLabel C openDevice
	    { $$ = new openMinor_openDevice($1, $3); }
	| didLabel C openFdata
	    { $$ = new openMinor_openFdata($1, $3); }
	| didLabel C openCadfile
	    { $$ = new openMinor_openCadfile($1, $3); }
	| didLabel C storageSpecList
	    { $$ = new openMinor_storageSpecList($1, $3); }
	| didLabel C openDMLfile
	    { $$ = new openMinor_openDMLfile($1, $3); }
	;

openDevice :
	  DIRECT C inputAccess
	    { $$ = new openDevice_inputAccess($3); }
	| DIRECT C outputAccess
	    { $$ = new openDevice_outputAccess($3); }
	;

inputAccess :
	  INPUT
	    { $$ = new inputAccess(); }
	;

outputAccess :
	  OUTPUT
	    { $$ = new outputAccessUnspec(); }
	| OUTPUT C APPEND
	    { $$ = new outputAccessAppend(); }
	| OUTPUT C OVERWR
	    { $$ = new outputAccessOverwr(); }
	;

openFdata :
	  FDATA C vLabel C outputAccess
	    { $$ = new openFdata_vLabel($3, $5); }
	| FDATA C DMIS C outputAccess
	    { $$ = new openFdata_DMIS($5); }
	;

openCadfile :
	  CAD
	    { $$ = new openCadfile(0); }
	| CAD C cadfileType
	    { $$ = new openCadfile($3); }
	;

cadfileType :
	  STEP
	    { $$ = new cadfileType_STEP(); }
	| IGES
	    { $$ = new cadfileType_IGES(); }
	| VENDOR C stringVal
	    { $$ = new cadfileType_VENDOR($3); }
	;

storageSpecList :
	  storageSpec
	    { $$ = new std::list<storageSpec *>;
	      $$->push_back($1); }
	| storageSpecList C storageSpec
	    { $$ = $1;
	      $$->push_back($3); }
	;

storageSpec :
	  SNS
	    { $$ = new storageSpec_SNS(); }
	| PCS
	    { $$ = new storageSpec_PCS(); }
	| FEATUR
	    { $$ = new storageSpec_FEATUR(); }
	| RTAB
	    { $$ = new storageSpec_RTAB(); }
	;

openDMLfile :
	  DML C stringVal C INPUT
	    { $$ = new openDmlInput($3); }
	| DML C stringVal C OUTPUT C PTDATA C state
	    { $$ = new openDmlOutput($3, $9); }
	;

operidStm :
	  opLabel equalSign OPERID SLASH stringVal endOfLine
	    { $$ = new operidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

outputStm :
	  OUTPUT SLASH outputMinor endOfLine
	    { $$ = new outputStm($3);
	      dmisStms.push_back($$);
	    }
	;

outputMinor :
	  outputFeatData
	    { $$ = $1; }
	| outputConstData
	    { $$ = $1; }
	| outputKeycharData
	    { $$ = $1; }
	| outputSensorData
	    { $$ = $1; }
	| outputToleranceData
	    { $$ = $1; }
	| outputReportData
	    { $$ = $1; }
	;

outputFeatData :
	  outputFeat
	    { $$ = $1; }
	| outputDat
	    { $$ = $1; }
	;

outputFeat :
	  fLabel
	    { $$ = new outputFeat1F($1, 0, 0); }
	| fLabel C rLabel
	    { $$ = new outputFeat1F($1, 0, $3); }
	| fLabel pointRange
	    { $$ = new outputFeat1F($1, $2, 0); }
	| fLabel pointRange C rLabel
	    { $$ = new outputFeat1F($1, $2, $4); }
	| faLabel
	    { $$ = new outputFeat1Fa($1, 0, 0); }
	| faLabel C rLabel
	    { $$ = new outputFeat1Fa($1, 0, $3); }
	| faLabel pointRange
	    { $$ = new outputFeat1Fa($1, $2, 0); }
	| faLabel pointRange C rLabel
	    { $$ = new outputFeat1Fa($1, $2, $4); }
	| fLabel C faLabel C taLabel
	    { $$ = new outputFeat2FFaTa($1, $3, $5, 0); }
	| fLabel C faLabel C taLabel C rLabel
	    { $$ = new outputFeat2FFaTa($1, $3, $5, $7); }
	| faLabel C faLabel C taLabel
	    { $$ = new outputFeat2FaFaTa($1, $3, $5, 0); }
	| faLabel C faLabel C taLabel C rLabel
	    { $$ = new outputFeat2FaFaTa($1, $3, $5, $7); }
	| fLabel C fLabel C tLabel
	    { $$ = new outputFeat2FFT($1, $3, $5, 0); }
	| fLabel C fLabel C tLabel C rLabel
	    { $$ = new outputFeat2FFT($1, $3, $5, $7); }
	| faLabel C fLabel C taLabel
	    { $$ = new outputFeat2FaFTa($1, $3, $5, 0); }
	| faLabel C fLabel C taLabel C rLabel
	    { $$ = new outputFeat2FaFTa($1, $3, $5, $7); }
	| fLabel C tLabelList
	    { $$ = new outputFeat1FT($1, 0, $3, 0); }
	| fLabel C tLabelList C rLabel
	    { $$ = new outputFeat1FT($1, 0, $3, $5); }
	| fLabel pointRange C tLabelList
	    { $$ = new outputFeat1FT($1, $2, $4, 0); }
	| fLabel pointRange C tLabelList C rLabel
	    { $$ = new outputFeat1FT($1, $2, $4, $6); }
	| faLabel C taLabelList
	    { $$ = new outputFeat1FaTa($1, 0, $3, 0); }
	| faLabel C taLabelList C rLabel
	    { $$ = new outputFeat1FaTa($1, 0, $3, $5); }
	| faLabel pointRange C taLabelList
	    { $$ = new outputFeat1FaTa($1, $2, $4, 0); }
	| faLabel pointRange C taLabelList C rLabel
	    { $$ = new outputFeat1FaTa($1, $2, $4, $6); }
	;

outputDat :
	  datLabel C faLabel C taLabel
	    { $$ = new outputDat_datLabel($1, $3, $5, 0); }
	| datLabel C faLabel C taLabel C rLabel
	    { $$ = new outputDat_datLabel($1, $3, $5, $7); }
	| faLabel C datLabel C taLabel
	    { $$ = new outputDat_faLabel($1, $3, $5, 0); }
	| faLabel C datLabel C taLabel C rLabel
	    { $$ = new outputDat_faLabel($1, $3, $5, $7); }
	;

outputReportData :
	  rLabel
	    { $$ = $1; }
	;

outputConstData :
	  seLabel
	    { $$ = new outputConstGaugeData($1); }
	| stLabel
	    { $$ = new outputConstPartData($1); }
	| stLabel C seLabel C toleranceLabel
	    { $$ = new outputConstBothData($1, $3, $5); }
	;

outputKeycharData :
	  kcLabel
	    { $$ = $1; }
	| kcaLabel
	    { $$ = $1; }
	;

outputToleranceData :
	  tLabel
	    { $$ = $1; }
	| taLabel
	    { $$ = $1; }
	;

outputSensorData :
	  sLabel
	    { $$ = new outputSensorData_sLabel($1); }
	| saLabel
	    { $$ = new outputSensorData_saLabel($1, 0); }
	| saLabel C outputSensorDesc
	    { $$ = new outputSensorData_saLabel($1, $3); }
	;

outputSensorDesc :
	  stringVal
	    { $$ = new outputSensorDesc_stringVal($1, 0); }
	| stringVal C outputSensorWristList
	    { $$ = new outputSensorDesc_stringVal($1, $3); }
	| intVal
	    { $$ = new outputSensorDesc_intVal($1, 0); }
	| intVal C outputSensorWristList
	    { $$ = new outputSensorDesc_intVal($1, $3); }
	| outputSensorWristList
	    { $$ = new outputSensorDesc_outputSensorWristList($1); }
	| CURENT
	    { $$ = new outputSensorDesc_CURENT(); }
	;

outputSensorWristList :
	  swLabel C outputSensorWristAngleList
	    { $$ = new std::list<outputSensorWristItem *>;
	      $$->push_front(new outputSensorWristItem($1, $3)); }
	| swLabel C outputSensorWristAngleList C outputSensorWristList
	    { $$ = $5;
	      $$->push_front(new outputSensorWristItem($1, $3)); }
	;

outputSensorWristAngleList :
	  outputSensorWristAngle
	    { $$ = new std::list<outputSensorWristAngle *>;
	      $$->push_back($1); }
	| outputSensorWristAngleList C outputSensorWristAngle
	    { $$ = $1;
	      $$->push_back($3); }
	;

outputSensorWristAngle :
	  stringVal C angle
	    { $$ = new outputSensorWristAngle($1, $3); }
	;

pameasStm :
	  PAMEAS SLASH pameasMinor endOfLine
	    { $$ = new pameasStm($3);
	      dmisStms.push_back($$);
	    }
	;

pameasMinor :
	  pameasVar2List
	    { $$ = new pameasMinor_pameasVar2List($1); }
	| pameasVar3List
	    { $$ = new pameasMinor_pameasVar3List($1); }
	;

pameasDetail :
	  DISTANCE C rentVal C
	    { $$ = new pameasDetailDistance($3, 0, 0); }
	| DISTANCE C rentVal C PITCH C rentVal C
	    { $$ = new pameasDetailDistance($3, 0, $7); }
	| DISTANCE C rentVal C SCNVEL C pameasFedrat C
	    { $$ = new pameasDetailDistance($3, $7, 0); }
	| DISTANCE C rentVal C SCNVEL C pameasFedrat C PITCH C rentVal C
	    { $$ = new pameasDetailDistance($3, $7, $11); }
	| SCNVEL C pameasFedrat C
	    { $$ = new pameasDetailScnvelData($3, 0); }
	| SCNVEL C pameasFedrat C PITCH C rentVal C
	    { $$ = new pameasDetailScnvelData($3, $7); }
	| PITCH C rentVal C
	    { $$ = new pameasDetailPitch($3); }
	| NODATA C
	    { $$ = new pameasDetailNoData(); }
	| SCNVEL C pameasFedrat C NODATA C
	    { $$ = new pameasDetailScnvelNoData($3); }
	;

pameasVar2List :
	  pameasDetail pLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, 0, 0)); }
	| pameasDetail pLabel C pameasVar2List
	    { $$ = $4;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, 0, 0)); }
	| pameasDetail pLabel C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, 0, $6)); }
	| pameasDetail pLabel C ROTARY C rtLabel C pameasVar2List
	    { $$ = $8;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, 0, $6)); }
	| pameasDetail pLabel C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, $4, 0)); }
	| pameasDetail pLabel C pameasRemove C pameasVar2List
	    { $$ = $6;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, $4, 0)); }
	| pameasDetail pLabel C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, $4, $8)); }
	| pameasDetail pLabel C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $10;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, 0, 0, $4, $8)); }
	| pameasDetail pLabel C SELFCENTER C vector
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, 0, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C pameasVar2List
	    { $$ = $8;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, 0, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, 0, $10)); }
	| pameasDetail pLabel C SELFCENTER C vector C ROTARY C rtLabel C pameasVar2List
	    { $$ = $12;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, 0, $10)); }
	| pameasDetail pLabel C SELFCENTER C vector C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, $8, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C pameasRemove C pameasVar2List
	    { $$ = $10;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, $8, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, $8, $12)); }
	| pameasDetail pLabel C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $14;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, 0, $8, $12)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, 0, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C pameasVar2List
	    { $$ = $10;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, 0, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, 0, $12)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel C pameasVar2List
	    { $$ = $14;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, 0, $12)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, $10, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C pameasVar2List
	    { $$ = $12;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, $10, 0)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, $10, $14)); }
	| pameasDetail pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $16;
	      $$->push_front(new pameasVar2ListItem($1, $2, 0, $6, $8, $10, $14)); }
	| pameasDetail pLabel C vector
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, 0, 0)); }
	| pameasDetail pLabel C vector C pameasVar2List
	    { $$ = $6;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, 0, 0)); }
	| pameasDetail pLabel C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, 0, $8)); }
	| pameasDetail pLabel C vector C ROTARY C rtLabel C pameasVar2List
	    { $$ = $10;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, 0, $8)); }
	| pameasDetail pLabel C vector C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, $6, 0)); }
	| pameasDetail pLabel C vector C pameasRemove C pameasVar2List
	    { $$ = $8;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, $6, 0)); }
	| pameasDetail pLabel C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, $6, $10)); }
	| pameasDetail pLabel C vector C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $12;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, 0, 0, $6, $10)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, 0, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C pameasVar2List
	    { $$ = $10;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, 0, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, 0, $12)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C ROTARY C rtLabel C pameasVar2List
	    { $$ = $14;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, 0, $12)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, $10, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C pameasRemove C pameasVar2List
	    { $$ = $12;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, $10, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, $10, $14)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $16;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, 0, $10, $14)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, 0, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasVar2List
	    { $$ = $12;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, 0, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, 0, $14)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel C pameasVar2List
	    { $$ = $16;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, 0, $14)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, $12, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C pameasVar2List
	    { $$ = $14;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, $12, 0)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar2ListItem *>;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, $12, $16)); }
	| pameasDetail pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel C pameasVar2List
	    { $$ = $18;
	      $$->push_front(new pameasVar2ListItem($1, $2, $4, $8, $10, $12, $16)); }
	;

pameasVar3List :
	  pLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, 0, 0)); }
	| pLabel C pameasVar3List
	    { $$ = $3;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, 0, 0)); }
	| pLabel C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, 0, $5)); }
	| pLabel C ROTARY C rtLabel C pameasVar3List
	    { $$ = $7;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, 0, $5)); }
	| pLabel C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, $3, 0)); }
	| pLabel C pameasRemove C pameasVar3List
	    { $$ = $5;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, $3, 0)); }
	| pLabel C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, $3, $7)); }
	| pLabel C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $9;
	      $$->push_front(new pameasVar3ListItem($1, 0, 0, 0, $3, $7)); }
	| pLabel C SELFCENTER C vector
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, 0, 0)); }
	| pLabel C SELFCENTER C vector C pameasVar3List
	    { $$ = $7;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, 0, 0)); }
	| pLabel C SELFCENTER C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, 0, $9)); }
	| pLabel C SELFCENTER C vector C ROTARY C rtLabel C pameasVar3List
	    { $$ = $11;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, 0, $9)); }
	| pLabel C SELFCENTER C vector C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, $7, 0)); }
	| pLabel C SELFCENTER C vector C pameasRemove C pameasVar3List
	    { $$ = $9;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, $7, 0)); }
	| pLabel C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, $7, $11)); }
	| pLabel C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $13;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, 0, $7, $11)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, 0, 0)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C pameasVar3List
	    { $$ = $9;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, 0, 0)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, 0, $11)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel C pameasVar3List
	    { $$ = $13;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, 0, $11)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, $9, 0)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C pameasVar3List
	    { $$ = $11;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, $9, 0)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, $9, $13)); }
	| pLabel C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $15;
	      $$->push_front(new pameasVar3ListItem($1, 0, $5, $7, $9, $13)); }
	| pLabel C vector
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, 0, 0)); }
	| pLabel C vector C pameasVar3List
	    { $$ = $5;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, 0, 0)); }
	| pLabel C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, 0, $7)); }
	| pLabel C vector C ROTARY C rtLabel C pameasVar3List
	    { $$ = $9;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, 0, $7)); }
	| pLabel C vector C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, $5, 0)); }
	| pLabel C vector C pameasRemove C pameasVar3List
	    { $$ = $7;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, $5, 0)); }
	| pLabel C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, $5, $9)); }
	| pLabel C vector C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $11;
	      $$->push_front(new pameasVar3ListItem($1, $3, 0, 0, $5, $9)); }
	| pLabel C vector C SELFCENTER C vector
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, 0, 0)); }
	| pLabel C vector C SELFCENTER C vector C pameasVar3List
	    { $$ = $9;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, 0, 0)); }
	| pLabel C vector C SELFCENTER C vector C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, 0, $11)); }
	| pLabel C vector C SELFCENTER C vector C ROTARY C rtLabel C pameasVar3List
	    { $$ = $13;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, 0, $11)); }
	| pLabel C vector C SELFCENTER C vector C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, $9, 0)); }
	| pLabel C vector C SELFCENTER C vector C pameasRemove C pameasVar3List
	    { $$ = $11;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, $9, 0)); }
	| pLabel C vector C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, $9, $13)); }
	| pLabel C vector C SELFCENTER C vector C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $15;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, 0, $9, $13)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, 0, 0)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasVar3List
	    { $$ = $11;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, 0, 0)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, 0, $13)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C ROTARY C rtLabel C pameasVar3List
	    { $$ = $15;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, 0, $13)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, $11, 0)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C pameasVar3List
	    { $$ = $13;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, $11, 0)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel
	    { $$ = new std::list<pameasVar3ListItem *>;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, $11, $15)); }
	| pLabel C vector C SELFCENTER C vector C forceOrDeflection C pameasRemove C ROTARY C rtLabel C pameasVar3List
	    { $$ = $17;
	      $$->push_front(new pameasVar3ListItem($1, $3, $7, $9, $11, $15)); }
	;

pameasRemove :
	  REMOVE C COUNT C intVal C intVal
	    { $$ = new pameasRemoveCount($5, $7); }
	| REMOVE C DIST C rentVal C rentVal
	    { $$ = new pameasRemoveDist($5, $7); }
	| REMOVE C ALL
	    { $$ = new pameasRemoveAll(); }
	;

pameasFedrat :
	  MPM C rentVal
	    { $$ = new pameasFedrat_MPM($3); }
	| MMPS C rentVal
	    { $$ = new pameasFedrat_MMPS($3); }
	| IPM C rentVal
	    { $$ = new pameasFedrat_IPM($3); }
	| IPS C rentVal
	    { $$ = new pameasFedrat_IPS($3); }
	;

partidStm :
	  pnLabel equalSign PARTID SLASH stringVal endOfLine
	    { $$ = new partidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

partrvStm :
	  prLabel equalSign PARTRV SLASH stringVal endOfLine
	    { $$ = new partrvStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

partsnStm :
	  psLabel equalSign PARTSN SLASH stringVal endOfLine
	    { $$ = new partsnStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

pathStm :
	  pLabel equalSign PATH SLASH pathMinor endOfLine
	    { $$ = new pathStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

pathMinor :
	  pathPoint
	    { $$ = $1; }
	| pathArc
	    { $$ = $1; }
	| pathCurve
	    { $$ = $1; }
	| pathHelical
	    { $$ = $1; }
	| pathLine
	    { $$ = $1; }
	| pathSurface
	    { $$ = $1; }
	| pathUnknown
	    { $$ = $1; }
	;

euler :
	  rentVal C rentVal C rentVal
	    { $$ = new euler($1, $3, $5); }
	;

pathPoint :
	  POINT C typePoint C vector
	    { $$ = new pathPoint($3, $5); }
	;

pathArc :
	  ARC C typePoint C vector C rentVal C angle C angle
	    { $$ = new pathArc($3, $5, $7, $9, $11, 0); }
	| ARC C typePoint C vector C rentVal C angle C angle C vector
	    { $$ = new pathArc($3, $5, $7, $9, $11, $13); }
	;

pathCurve :
	  CURVE C pathCurvePtdata
	    { $$ = new pathCurve_pathCurvePtdata($3); }
	| CURVE C pathCurvePoints
	    { $$ = new pathCurve_pathCurvePoints($3); }
	;

pathCurvePtdata :
	  curvePtdataPcs
	    { $$ = $1; }
	| curvePtdataHeadcs
	    { $$ = $1; }
	;

curvePtdataPcs :
	  PTDATA C pointVecCart C curvePtdataPcsList
	    { $$ = new curvePtdataPcs($3, 0, $5); }
	| PTDATA C pointVecCart C PCS C euler C curvePtdataPcsList
	    { $$ = new curvePtdataPcs($3, $7, $9); }
	;

curvePtdataPcsList :
	  PTDATA C pointVecCart
	    { $$ = new std::list<curvePtdataPcsListItem *>;
	      $$->push_front(new curvePtdataPcsListItem($3, 0)); }
	| PTDATA C pointVecCart C curvePtdataPcsList
	    { $$ = $5;
	      $$->push_front(new curvePtdataPcsListItem($3, 0)); }
	| PTDATA C pointVecCart C PCS C euler
	    { $$ = new std::list<curvePtdataPcsListItem *>;
	      $$->push_front(new curvePtdataPcsListItem($3, $7)); }
	| PTDATA C pointVecCart C PCS C euler C curvePtdataPcsList
	    { $$ = $9;
	      $$->push_front(new curvePtdataPcsListItem($3, $7)); }
	;

curvePtdataHeadcs :
	  PTDATA C pointVecCart C HEADCS C rentVal C rentVal C curvePtdataHeadcsList
	    { $$ = new curvePtdataHeadcs($3, $7, $9, 0, $11); }
	| PTDATA C pointVecCart C HEADCS C rentVal C rentVal C rentVal C curvePtdataHeadcsList
	    { $$ = new curvePtdataHeadcs($3, $7, $9, $11, $13); }
	;

curvePtdataHeadcsList :
	  PTDATA C pointVecCart
	    { $$ = new std::list<curvePtdataHeadcsListItem *>;
	      $$->push_front(new curvePtdataHeadcsListItem($3, 0, 0, 0)); }
	| PTDATA C pointVecCart C curvePtdataHeadcsList
	    { $$ = $5;
	      $$->push_front(new curvePtdataHeadcsListItem($3, 0, 0, 0)); }
	| PTDATA C pointVecCart C HEADCS C rentVal C rentVal
	    { $$ = new std::list<curvePtdataHeadcsListItem *>;
	      $$->push_front(new curvePtdataHeadcsListItem($3, $7, $9, 0)); }
	| PTDATA C pointVecCart C HEADCS C rentVal C rentVal C curvePtdataHeadcsList
	    { $$ = $11;
	      $$->push_front(new curvePtdataHeadcsListItem($3, $7, $9, 0)); }
	| PTDATA C pointVecCart C HEADCS C rentVal C rentVal C rentVal
	    { $$ = new std::list<curvePtdataHeadcsListItem *>;
	      $$->push_front(new curvePtdataHeadcsListItem($3, $7, $9, $11)); }
	| PTDATA C pointVecCart C HEADCS C rentVal C rentVal C rentVal C curvePtdataHeadcsList
	    { $$ = $13;
	      $$->push_front(new curvePtdataHeadcsListItem($3, $7, $9, $11)); }
	;

pathCurvePoints :
	  pointVecCart C pointVecCartList
	    { $$ = new pathCurvePoints($1, $3); }
	;

pathHelical :
	  HELICAL C typePoint C vector C rentVal C angle C angle C rentVal
	    { $$ = new pathHelical($3, $5, $7, $9, $11, 0, $13, 0); }
	| HELICAL C typePoint C vector C rentVal C angle C angle C rentVal C angle
	    { $$ = new pathHelical($3, $5, $7, $9, $11, 0, $13, $15); }
	| HELICAL C typePoint C vector C rentVal C angle C angle C vector C rentVal
	    { $$ = new pathHelical($3, $5, $7, $9, $11, $13, $15, 0); }
	| HELICAL C typePoint C vector C rentVal C angle C angle C vector C rentVal C angle
	    { $$ = new pathHelical($3, $5, $7, $9, $11, $13, $15, $17); }
	;

pathLine :
	  LINE C BND C CART C START C impCartPoint C END C impCartPoint C VEC C vector
	    { $$ = new pathLineCart($9, 0, $13, 0, $17); }
	| LINE C BND C CART C START C impCartPoint C END C impCartPoint C csSpecC VEC C vector
	    { $$ = new pathLineCart($9, 0, $13, $15, $18); }
	| LINE C BND C CART C START C impCartPoint C csSpecC END C impCartPoint C VEC C vector
	    { $$ = new pathLineCart($9, $11, $14, 0, $18); }
	| LINE C BND C CART C START C impCartPoint C csSpecC END C impCartPoint C csSpecC VEC C vector
	    { $$ = new pathLineCart($9, $11, $14, $16, $19); }
	| LINE C BND C POL C impPolPoint C impPolPoint C VEC C vector
	    { $$ = new pathLinePol($7, $9, $13); }
	;

csSpec :
	  PCS C euler
	    { $$ = new csSpec_PCS($3); }
	| HEADCS C rentVal C rentVal
	    { $$ = new csSpec_HEADCS($3, $5, 0); }
	| HEADCS C rentVal C rentVal C rentVal
	    { $$ = new csSpec_HEADCS($3, $5, $7); }
	;

csSpecC :
	  PCS C euler C
	    { $$ = new csSpec_PCS($3); }
	| HEADCS C rentVal C rentVal C
	    { $$ = new csSpec_HEADCS($3, $5, 0); }
	| HEADCS C rentVal C rentVal C rentVal C
	    { $$ = new csSpec_HEADCS($3, $5, $7); }
	;

pathSurface :
	  SURFACE C surfPtdataList
	    { $$ = new pathSurfacePlain($3); }
	| SURFACE C surfPtdataList C PCS C euler
	    { $$ = new pathSurfacePcs($3, $7); }
	| SURFACE C surfPtdataList C PCS C euler C surfPtdataPcsList
	    { $$ = new pathSurfacePcsData($3, $7, $9); }
	| SURFACE C surfPtdataList C HEADCS C rentVal C rentVal
	    { $$ = new pathSurfaceHeadcs($3, $7, $9, 0); }
	| SURFACE C surfPtdataList C HEADCS C rentVal C rentVal C rentVal
	    { $$ = new pathSurfaceHeadcs($3, $7, $9, $11); }
	| SURFACE C surfPtdataList C HEADCS C rentVal C rentVal C surfPtdataHeadcsList
	    { $$ = new pathSurfaceHeadcsData($3, $7, $9, 0, $11); }
	| SURFACE C surfPtdataList C HEADCS C rentVal C rentVal C rentVal C surfPtdataHeadcsList
	    { $$ = new pathSurfaceHeadcsData($3, $7, $9, $11, $13); }
	;

surfPtdataList :
	  surfPtdataListItem
	    { $$ = new std::list<surfPtdataListItem *>;
	      $$->push_back($1); }
	| surfPtdataList C surfPtdataListItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

surfPtdataListItem :
	  PTDATA C pointVecCart C rentVal C rentVal
	    { $$ = new surfPtdataListItem($3, $5, $7); }
	;

surfPtdataPcsList :
	  PTDATA C pointVecCart C rentVal C rentVal
	    { $$ = new std::list<surfPtdataPcsListItem *>;
	      $$->push_front(new surfPtdataPcsListItem($3, $5, $7, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C surfPtdataPcsList
	    { $$ = $9;
	      $$->push_front(new surfPtdataPcsListItem($3, $5, $7, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C PCS C euler
	    { $$ = new std::list<surfPtdataPcsListItem *>;
	      $$->push_front(new surfPtdataPcsListItem($3, $5, $7, $11)); }
	| PTDATA C pointVecCart C rentVal C rentVal C PCS C euler C surfPtdataPcsList
	    { $$ = $13;
	      $$->push_front(new surfPtdataPcsListItem($3, $5, $7, $11)); }
	;

surfPtdataHeadcsList :
	  PTDATA C pointVecCart C rentVal C rentVal
	    { $$ = new std::list<surfPtdataHeadcsListItem *>;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, 0, 0, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C surfPtdataHeadcsList
	    { $$ = $9;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, 0, 0, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C HEADCS C rentVal C rentVal
	    { $$ = new std::list<surfPtdataHeadcsListItem *>;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, $11, $13, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C HEADCS C rentVal C rentVal C surfPtdataHeadcsList
	    { $$ = $15;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, $11, $13, 0)); }
	| PTDATA C pointVecCart C rentVal C rentVal C HEADCS C rentVal C rentVal C rentVal
	    { $$ = new std::list<surfPtdataHeadcsListItem *>;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, $11, $13, $15)); }
	| PTDATA C pointVecCart C rentVal C rentVal C HEADCS C rentVal C rentVal C rentVal C surfPtdataHeadcsList
	    { $$ = $17;
	      $$->push_front(new surfPtdataHeadcsListItem($3, $5, $7, $11, $13, $15)); }
	;

pathUnknown :
	  UNKNOWN C impCartPoint C impCartPoint C impCartPoint
	    { $$ = new pathUnknown($3, $5, $7, 0); }
	| UNKNOWN C impCartPoint C impCartPoint C impCartPoint C vector
	    { $$ = new pathUnknown($3, $5, $7, $9); }
	;

planidStm :
	  plLabel equalSign PLANID SLASH stringVal endOfLine
	    { $$ = new planidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

popStm :
	  POP SLASH stackElementList endOfLine
	    { $$ = new popStm($3);
	      dmisStms.push_back($$);
	    }
	;

prcompStm :
	  PRCOMP SLASH state endOfLine
	    { $$ = new prcompStm($3);
	      dmisStms.push_back($$);
	    }
	;

prevopStm :
	  pvLabel equalSign PREVOP SLASH stringVal endOfLine
	    { $$ = new prevopStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

procidStm :
	  pcLabel equalSign PROCID SLASH stringVal endOfLine
	    { $$ = new procidStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

promptStm :
	  boolVar EQUALS PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_boolVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| stringVar EQUALS PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_stringVar($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| stringVar EQUALS PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_stringVar($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| intVar EQUALS PROMPT SLASH promptIntEnd endOfLine
	    { $$ = new promptStm_intVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_realVar($1, $5, 0, 0);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS PROMPT SLASH stringVal C rentVal endOfLine
	    { $$ = new promptStm_realVar($1, $5, $7, 0);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS PROMPT SLASH stringVal C rentVal C rentVal endOfLine
	    { $$ = new promptStm_realVar($1, $5, $7, $9);
	      dmisStms.push_back($$);
	    }
	| ccLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_ccLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| ccLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_ccLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| ciLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_ciLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| ciLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_ciLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| csLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_csLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| csLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_csLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| diLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_diLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| diLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_diLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| dsLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_dsLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| dsLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_dsLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| dvLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_dvLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| dvLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_dvLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| fiLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_fiLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| fiLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_fiLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| fsLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_fsLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| fsLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_fsLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| liLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_liLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| liLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_liLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| mdLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_mdLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| mdLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_mdLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| opLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_opLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| opLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_opLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| pcLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_pcLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| pcLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_pcLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| plLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_plLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| plLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_plLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| pnLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_pnLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| pnLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_pnLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| prLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_prLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| prLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_prLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| psLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_psLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| psLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_psLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| pvLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_pvLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| pvLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_pvLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| qLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_qLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| qLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_qLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	| tlLabel equalSign PROMPT SLASH stringVal endOfLine
	    { $$ = new promptStm_tlLabel($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| tlLabel equalSign PROMPT SLASH stringVal C intVal endOfLine
	    { $$ = new promptStm_tlLabel($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

promptIntEnd :
	  stringVal
	    { $$ = new promptIntEnd_stringVal($1, 0, 0); }
	| stringVal C intVal
	    { $$ = new promptIntEnd_stringVal($1, $3, 0); }
	| stringVal C intVal C intVal
	    { $$ = new promptIntEnd_stringVal($1, $3, $5); }
	| promptItemList
	    { $$ = new promptIntEnd_promptItemList($1); }
	;

promptItemList :
	  BUTTON C stringVal C intVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_BUTTON($3, $5)); }
	| BUTTON C stringVal C intVal C promptItemList
	    { $$ = $7;
	      $$->push_front(new promptItem_BUTTON($3, $5)); }
	| CHECK C stringVal C boolVar
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_CHECK($3, $5)); }
	| CHECK C stringVal C boolVar C promptItemList
	    { $$ = $7;
	      $$->push_front(new promptItem_CHECK($3, $5)); }
	| EDIT C promptVar
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_EDIT($3, 0, 0)); }
	| EDIT C promptVar C promptItemList
	    { $$ = $5;
	      $$->push_front(new promptItem_EDIT($3, 0, 0)); }
	| EDIT C promptVar C rentVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_EDIT($3, $5, 0)); }
	| EDIT C promptVar C rentVal C promptItemList
	    { $$ = $7;
	      $$->push_front(new promptItem_EDIT($3, $5, 0)); }
	| EDIT C promptVar C rentVal C rentVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_EDIT($3, $5, $7)); }
	| EDIT C promptVar C rentVal C rentVal C promptItemList
	    { $$ = $9;
	      $$->push_front(new promptItem_EDIT($3, $5, $7)); }
	| GROUP C stringVal C promptVar C stringVal C stringList
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_GROUP($3, $5, $7, $9)); }
	| GROUP C stringVal C promptVar C stringVal C stringList C promptItemList
	    { $$ = $11;
	      $$->push_front(new promptItem_GROUP($3, $5, $7, $9)); }
	| LIST C promptVar C stringVal C stringList
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_LIST($3, $5, $7)); }
	| LIST C promptVar C stringVal C stringList C promptItemList
	    { $$ = $9;
	      $$->push_front(new promptItem_LIST($3, $5, $7)); }
	| PICTURE C stringVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_PICTURE($3, 0)); }
	| PICTURE C stringVal C promptItemList
	    { $$ = $5;
	      $$->push_front(new promptItem_PICTURE($3, 0)); }
	| PICTURE C stringVal C intVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_PICTURE($3, $5)); }
	| PICTURE C stringVal C intVal C promptItemList
	    { $$ = $7;
	      $$->push_front(new promptItem_PICTURE($3, $5)); }
	| PIXBTN C stringVal C intVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_PIXBTN($3, 0, $5)); }
	| PIXBTN C stringVal C intVal C promptItemList
	    { $$ = $7;
	      $$->push_front(new promptItem_PIXBTN($3, 0, $5)); }
	| PIXBTN C stringVal C stringVal C intVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_PIXBTN($3, $5, $7)); }
	| PIXBTN C stringVal C stringVal C intVal C promptItemList
	    { $$ = $9;
	      $$->push_front(new promptItem_PIXBTN($3, $5, $7)); }
	| SOUND C stringVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_SOUND($3)); }
	| SOUND C stringVal C promptItemList
	    { $$ = $5;
	      $$->push_front(new promptItem_SOUND($3)); }
	| TEXT C stringVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_TEXT($3)); }
	| TEXT C stringVal C promptItemList
	    { $$ = $5;
	      $$->push_front(new promptItem_TEXT($3)); }
	| TITLE C stringVal
	    { $$ = new std::list<promptItem *>;
	      $$->push_front(new promptItem_TITLE($3)); }
	| TITLE C stringVal C promptItemList
	    { $$ = $5;
	      $$->push_front(new promptItem_TITLE($3)); }
	;

promptVar :
	  stringVar
	    { $$ = $1; }
	| boolVar
	    { $$ = $1; }
	| intVar
	    { $$ = $1; }
	| realVar
	    { $$ = $1; }
	| qisLabel
	    { $$ = $1; }
	;

psthruStm :
	  PSTHRU SLASH psthruMinor endOfLine
	    { $$ = new psthruStm($3);
	      dmisStms.push_back($$);
	    }
	;

psthruMinor :
	  COMAND C stringVal
	    { $$ = new psthruMinor_COMAND($3); }
	| CONTIN
	    { $$ = new psthruMinor_CONTIN(); }
	| PAUSE
	    { $$ = new psthruMinor_PAUSE(); }
	| START
	    { $$ = new psthruMinor_START(); }
	| STOP
	    { $$ = new psthruMinor_STOP(); }
	| TRMATX C matrix
	    { $$ = new psthruMinor_TRMATX($3); }
	;

ptbuffStm :
	  PTBUFF SLASH state endOfLine
	    { $$ = new ptbuffStm($3);
	      dmisStms.push_back($$);
	    }
	;

ptmeasStm :
	  PTMEAS SLASH typePoint endOfLine
	    { $$ = new ptmeasStm($3, 0, 0);
	      dmisStms.push_back($$);
	    }
	| PTMEAS SLASH typePoint C ptmeasEnd endOfLine
	    { $$ = new ptmeasStm($3, 0, $5);
	      dmisStms.push_back($$);
	    }
	| PTMEAS SLASH typePoint C vector endOfLine
	    { $$ = new ptmeasStm($3, $5, 0);
	      dmisStms.push_back($$);
	    }
	| PTMEAS SLASH typePoint C vector C ptmeasEnd endOfLine
	    { $$ = new ptmeasStm($3, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

ptmeasEnd :
	  csSpec
	    { $$ = new ptmeasEnd_csSpec($1, 0); }
	| csSpecC touchSpec
	    { $$ = new ptmeasEnd_csSpec($1, $2); }
	| touchSpec
	    { $$ = new ptmeasEnd_touchSpec($1); }
	;

touchSpec :
	  HEADTOUCH
	    { $$ = new touchSpec_HEADTOUCH(); }
	| ALLAXESTOUCH
	    { $$ = new touchSpec_ALLAXESTOUCH(); }
	;

pushStm :
	  PUSH SLASH stackElementList endOfLine
	    { $$ = new pushStm($3);
	      dmisStms.push_back($$);
	    }
	;

qisdefStm :
	  qLabel equalSign QISDEF SLASH stringVal endOfLine
	    { $$ = new qisdefStm($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| qLabel equalSign QISDEF SLASH stringVal C stringVal endOfLine
	    { $$ = new qisdefStm($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

rapidStm :
	  RAPID SLASH rentVal endOfLine
	    { $$ = new rapidStm($3);
	      dmisStms.push_back($$);
	    }
	;

readStm :
	  READ SLASH didLabel C readSpecList endOfLine
	    { $$ = new readStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

readSpecList :
	  readSpec
	    { $$ = new std::list<readSpec *>;
	      $$->push_back($1); }
	| readSpecList C readSpec
	    { $$ = $1;
	      $$->push_back($3); }
	;

readSpec :
	  rwVar
	    { $$ = new readSpec($1, 0); }
	| rwVar rwFormat
	    { $$ = new readSpec($1, $2); }
	;

recallDatumStm :
	  RECALL SLASH datumLabel2 endOfLine
	    { $$ = new recallDatumStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| RECALL SLASH datumLabel2 C didLabel endOfLine
	    { $$ = new recallDatumStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

recallSensorStm :
	  RECALL SLASH sensorLabel1 endOfLine
	    { $$ = new recallSensorStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| RECALL SLASH sensorLabel1 C didLabel endOfLine
	    { $$ = new recallSensorStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

recallFeatureStm :
	  RECALL SLASH faLabel endOfLine
	    { $$ = new recallFeatureStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| RECALL SLASH faLabel C didLabel endOfLine
	    { $$ = new recallFeatureStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

recallRotaryTableStm :
	  RECALL SLASH rtLabel endOfLine
	    { $$ = new recallRotaryTableStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| RECALL SLASH rtLabel C didLabel endOfLine
	    { $$ = new recallRotaryTableStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

recallDMLStm :
	  RECALL SLASH DML C didLabel C daLabel endOfLine
	    { $$ = new recallDMLStm($5, $7);
	      dmisStms.push_back($$);
	    }
	;

refmntStm :
	  rmLabel equalSign REFMNT SLASH XVEC C vector C ZVEC C vector C MNTLEN C vector endOfLine
	    { $$ = new refmntStm($1, $7, $11, $15);
	      dmisStms.push_back($$);
	    }
	;

reportStm :
	  rLabel equalSign REPORT SLASH reportItemList endOfLine
	    { $$ = new reportStm($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| rLabel equalSign REPORT SLASH reportItemList C stringVal endOfLine
	    { $$ = new reportStm($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

reportItemList :
	  reportItem
	    { $$ = new std::list<reportItem *>;
	      $$->push_back($1); }
	| reportItemList C reportItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

reportItem :
	  ALGOR
	    { $$ = new reportItem_ALGOR(); }
	| DATE
	    { $$ = new reportItem_DATE(); }
	| HUMID
	    { $$ = new reportItem_HUMID(); }
	| MODE
	    { $$ = new reportItem_MODE(); }
	| TEMPC
	    { $$ = new reportItem_TEMPC(); }
	| TEMPF
	    { $$ = new reportItem_TEMPF(); }
	| TEMPWC
	    { $$ = new reportItem_TEMPWC(); }
	| TEMPWF
	    { $$ = new reportItem_TEMPWF(); }
	| TIME
	    { $$ = new reportItem_TIME(); }
	| qisLabel
	    { $$ = new reportItem_qisLabel($1); }
	;

resumeStm :
	  RESUME SLASH resumeMinor endOfLine
	    { $$ = new resumeStm($3);
	      dmisStms.push_back($$);
	    }
	;

resumeMinor :
	  jumpLabel
	    { $$ = new resumeMinor_jumpLabel($1); }
	| CURENT
	    { $$ = new resumeMinor_CURENT(); }
	| END
	    { $$ = new resumeMinor_END(); }
	| NEXT
	    { $$ = new resumeMinor_NEXT(); }
	| START
	    { $$ = new resumeMinor_START(); }
	| STOP
	    { $$ = new resumeMinor_STOP(); }
	;

rmeasStm :
	  RMEAS SLASH rmeasMinor endOfLine
	    { $$ = new rmeasStm($3);
	      dmisStms.push_back($$);
	    }
	;

rmeasMinor :
	  rmeasArc
	    { $$ = $1; }
	| rmeasCircle
	    { $$ = $1; }
	| rmeasCone
	    { $$ = $1; }
	| rmeasCparln
	    { $$ = $1; }
	| rmeasCylndr
	    { $$ = $1; }
	| rmeasEdgept
	    { $$ = $1; }
	| rmeasEllips
	    { $$ = $1; }
	| rmeasGcurve
	    { $$ = $1; }
	| rmeasGsurf
	    { $$ = $1; }
	| rmeasLine
	    { $$ = $1; }
	| rmeasObject
	    { $$ = $1; }
	| rmeasParpln
	    { $$ = $1; }
	| rmeasPlane
	    { $$ = $1; }
	| rmeasPoint
	    { $$ = $1; }
	| rmeasRctngl
	    { $$ = $1; }
	| rmeasSphere
	    { $$ = $1; }
	| rmeasSympln
	    { $$ = $1; }
	| rmeasTorus
	    { $$ = $1; }
	;

rmeasArc :
	  ARC C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasArc_rmeasSpecFa($3, $5, $7); }
	| ARC C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasArc_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasCircle :
	  CIRCLE C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasCircle_rmeasSpecFa($3, $5, $7); }
	| CIRCLE C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasCircle_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasCone :
	  CONE C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasCone($3, $5, $7); }
	;

rmeasCparln :
	  CPARLN C fLabel C intVal C rmeasSpecFaOrient
	    { $$ = new rmeasCparln_rmeasSpecFaOrient($3, $5, $7); }
	| CPARLN C fLabel C intVal C rmeasSpecVecbldOrient
	    { $$ = new rmeasCparln_rmeasSpecVecbldOrient($3, $5, $7); }
	;

rmeasCylndr :
	  CYLNDR C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasCylndr($3, $5, $7); }
	;

rmeasEdgept :
	  EDGEPT C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasEdgept_rmeasSpecFaAxis($3, $5, $7); }
	| EDGEPT C fLabel C intVal C rmeasSpecVecbldEdgept
	    { $$ = new rmeasEdgept_rmeasSpecVecbldEdgept($3, $5, $7); }
	;

rmeasEllips :
	  ELLIPS C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasEllips_rmeasSpecFa($3, $5, $7); }
	| ELLIPS C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasEllips_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasGcurve :
	  GCURVE C fLabel C intVal C rmeasSpecFeatAxis
	    { $$ = new rmeasGcurve_rmeasSpecFeatAxis($3, $5, $7); }
	| GCURVE C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasGcurve_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasGsurf :
	  GSURF C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasGsurf($3, $5, $7); }
	;

rmeasLine :
	  LINE C fLabel C intVal C rmeasSpecFeatAxis
	    { $$ = new rmeasLine_rmeasSpecFeatAxis($3, $5, $7); }
	| LINE C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasLine_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasObject :
	  OBJECT C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasObject_rmeasSpecFa($3, $5, $7); }
	| OBJECT C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasObject_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasParpln :
	  PARPLN C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasParpln($3, $5, $7); }
	;

rmeasPlane :
	  PLANE C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasPlane($3, $5, $7); }
	;

rmeasPoint :
	  POINT C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasPoint_rmeasSpecFaAxis($3, $5, $7); }
	| POINT C fLabel C intVal C rmeasSpecVecbld
	    { $$ = new rmeasPoint_rmeasSpecVecbld($3, $5, $7); }
	;

rmeasRctngl :
	  RCTNGL C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasRctngl($3, $5, $7); }
	;

rmeasSphere :
	  SPHERE C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasSphere($3, $5, $7); }
	;

rmeasSympln :
	  SYMPLN C fLabel C intVal C rmeasSpecFa
	    { $$ = new rmeasSympln($3, $5, $7); }
	;

rmeasTorus :
	  TORUS C fLabel C intVal C rmeasSpecFaAxis
	    { $$ = new rmeasTorus($3, $5, $7); }
	;

rmeasSpecFa :
	  faLabel
	    { $$ = $1; }
	;

rmeasSpecFaAxis :
	  faLabel
	    { $$ = new rmeasSpecFaAxis_faLabel($1, 0); }
	| faLabel C axis
	    { $$ = new rmeasSpecFaAxis_faLabel($1, $3); }
	| axis
	    { $$ = new rmeasSpecFaAxis_axis($1); }
	;

rmeasSpecFaOrient :
	  faLabel
	    { $$ = new rmeasSpecFaOrient($1, false); }
	| faLabel C ORIENT
	    { $$ = new rmeasSpecFaOrient($1, true); }
	;

rmeasSpecFeatAxis :
	  featureLabel1
	    { $$ = new rmeasSpecFeatAxis_featureLabel1($1, 0); }
	| featureLabel1 C axis
	    { $$ = new rmeasSpecFeatAxis_featureLabel1($1, $3); }
	| axis
	    { $$ = new rmeasSpecFeatAxis_axis($1); }
	;

rmeasSpecVecbld :
	  VECBLD C rentVal C intVal
	    { $$ = new rmeasSpecVecbld($3, $5); }
	;

rmeasSpecVecbldOrient :
	  rmeasSpecVecbld
	    { $$ = new rmeasSpecVecbldOrient($1, false); }
	| rmeasSpecVecbld C ORIENT
	    { $$ = new rmeasSpecVecbldOrient($1, true); }
	;

rmeasSpecVecbldEdgept :
	  rmeasSpecVecbld C rentVal
	    { $$ = new rmeasSpecVecbldEdgept($1, $3, 0, 0); }
	| rmeasSpecVecbld C rentVal C axis
	    { $$ = new rmeasSpecVecbldEdgept($1, $3, 0, $5); }
	| rmeasSpecVecbld C rentVal C posDir
	    { $$ = new rmeasSpecVecbldEdgept($1, $3, $5, 0); }
	| rmeasSpecVecbld C rentVal C posDir C axis
	    { $$ = new rmeasSpecVecbldEdgept($1, $3, $5, $7); }
	;

rotabStm :
	  ROTAB SLASH rotabMinor endOfLine
	    { $$ = new rotabStm($3);
	      dmisStms.push_back($$);
	    }
	;

rotabMinor :
	  rtLabel C rotAbs C rotType C angle
	    { $$ = new rotabAbs($1, $3, $5, $7, 0); }
	| rtLabel C rotAbs C rotType C angle C FZ C angle
	    { $$ = new rotabAbs($1, $3, $5, $7, $11); }
	| rtLabel C rotIncr C rotType C angle
	    { $$ = new rotabIncr($1, $3, $5, $7, 0); }
	| rtLabel C rotIncr C rotType C angle C FZ C angle
	    { $$ = new rotabIncr($1, $3, $5, $7, $11); }
	| rtLabel C featureLabel C rotType C rotDir
	    { $$ = new rotabFeat($1, $3, $5, $7); }
	;

rotateStm :
	  dLabel equalSign ROTATE SLASH rotateMinor endOfLine
	    { $$ = new rotateStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

rotateMinor :
	  rotateValue
	    { $$ = $1; }
	| rotateFeature
	    { $$ = $1; }
	| rotateDatum
	    { $$ = $1; }
	;

rotateValue :
	  axis C angle
	    { $$ = new rotateValue($1, $3); }
	;

rotateFeature :
	  axis C featureLabel C dir
	    { $$ = new rotateFeature($1, $3, $5); }
	;

rotateDatum :
	  axis C datLabel C dir
	    { $$ = new rotateDatum($1, $3, $5); }
	;

rotdefStm :
	  rtLabel equalSign ROTDEF SLASH pointVecCart endOfLine
	    { $$ = new rotdefStm($1, $5, 0);
	      dmisStms.push_back($$);
	    }
	| rtLabel equalSign ROTDEF SLASH pointVecCart C rtLabel endOfLine
	    { $$ = new rotdefStm($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

rotsetStm :
	  ROTSET SLASH rtLabel C angle endOfLine
	    { $$ = new rotsetStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

saveDatumStm :
	  SAVE SLASH datumLabel1 endOfLine
	    { $$ = new saveDatumStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| SAVE SLASH datumLabel1 C didLabel endOfLine
	    { $$ = new saveDatumStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

saveSensorStm :
	  SAVE SLASH sensorLabel1 endOfLine
	    { $$ = new saveSensorStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| SAVE SLASH sensorLabel1 C didLabel endOfLine
	    { $$ = new saveSensorStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

saveFeatureStm :
	  SAVE SLASH faLabel endOfLine
	    { $$ = new saveFeatureStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| SAVE SLASH faLabel C didLabel endOfLine
	    { $$ = new saveFeatureStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

saveRotaryTableStm :
	  SAVE SLASH rtLabel endOfLine
	    { $$ = new saveRotaryTableStm($3, 0);
	      dmisStms.push_back($$);
	    }
	| SAVE SLASH rtLabel C didLabel endOfLine
	    { $$ = new saveRotaryTableStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

saveDMLStm :
	  SAVE SLASH DML C didLabel C daLabel endOfLine
	    { $$ = new saveDMLStm($5, $7);
	      dmisStms.push_back($$);
	    }
	;

scnmodStm :
	  SCNMOD SLASH state endOfLine
	    { $$ = new scnmodStm($3);
	      dmisStms.push_back($$);
	    }
	;

scnsetStm :
	  SCNSET SLASH scnsetMinor endOfLine
	    { $$ = new scnsetStm($3);
	      dmisStms.push_back($$);
	    }
	;

scnsetMinor :
	  scnsetPeck
	    { $$ = $1; }
	| scnsetDrag
	    { $$ = $1; }
	| scnsetNoncon
	    { $$ = $1; }
	| scnsetStop
	    { $$ = $1; }
	| scnsetVendor
	    { $$ = $1; }
	;

scnsetPeck :
	  PECK C scnsetSample
	    { $$ = new scnsetPeck($3); }
	;

scnsetDrag :
	  DRAG C scnsetSampleDrag
	    { $$ = new scnsetDrag($3); }
	;

scnsetNoncon :
	  NONCON C scnsetSample
	    { $$ = new scnsetNoncon($3); }
	;

scnsetStop :
	  scnsetStopPlane
	    { $$ = $1; }
	| scnsetStopSphere
	    { $$ = $1; }
	;

scnsetStopPlane :
	  STOP C PLANE C vector
	    { $$ = new scnsetStopPlane($5, 0, 0); }
	| STOP C PLANE C vector C COUNT C intVal
	    { $$ = new scnsetStopPlane($5, 0, $9); }
	| STOP C PLANE C vector C RADIUS C rentVal
	    { $$ = new scnsetStopPlane($5, $9, 0); }
	| STOP C PLANE C vector C RADIUS C rentVal C COUNT C intVal
	    { $$ = new scnsetStopPlane($5, $9, $13); }
	;

scnsetStopSphere :
	  STOP C SPHERE C rentVal
	    { $$ = new scnsetStopSphere($5); }
	;

scnsetVendor :
	  VENDOR C FORM
	    { $$ = new scnsetVendor_FORM(); }
	| VENDOR C POS
	    { $$ = new scnsetVendor_POS(); }
	| VENDOR C SIZE
	    { $$ = new scnsetVendor_SIZE(); }
	;

scnsetSample :
	  DIST C rentVal
	    { $$ = new scnsetSample_DIST($3, 0); }
	| DIST C rentVal C axis
	    { $$ = new scnsetSample_DIST($3, $5); }
	| CHORD C rentVal
	    { $$ = new scnsetSample_CHORD($3, 0); }
	| CHORD C rentVal C rentVal
	    { $$ = new scnsetSample_CHORD($3, $5); }
	| TIME C rentVal
	    { $$ = new scnsetSample_TIME($3); }
	| ANGLE C angle
	    { $$ = new scnsetSample_ANGLE($3); }
	| DEFALT
	    { $$ = new scnsetSample_DEFALT(); }
	;

scnsetSampleDrag :
	  DIST C rentVal
	    { $$ = new scnsetSampleDrag_DIST($3, 0, 0); }
	| DIST C rentVal C forceOrDeflection
	    { $$ = new scnsetSampleDrag_DIST($3, 0, $5); }
	| DIST C rentVal C axis
	    { $$ = new scnsetSampleDrag_DIST($3, $5, 0); }
	| DIST C rentVal C axis C forceOrDeflection
	    { $$ = new scnsetSampleDrag_DIST($3, $5, $7); }
	| CHORD C rentVal
	    { $$ = new scnsetSampleDrag_CHORD($3, 0, 0); }
	| CHORD C rentVal C forceOrDeflection
	    { $$ = new scnsetSampleDrag_CHORD($3, 0, $5); }
	| CHORD C rentVal C rentVal
	    { $$ = new scnsetSampleDrag_CHORD($3, $5, 0); }
	| CHORD C rentVal C rentVal C forceOrDeflection
	    { $$ = new scnsetSampleDrag_CHORD($3, $5, $7); }
	| TIME C rentVal
	    { $$ = new scnsetSampleDrag_TIME($3, 0); }
	| TIME C rentVal C forceOrDeflection
	    { $$ = new scnsetSampleDrag_TIME($3, $5); }
	| ANGLE C angle
	    { $$ = new scnsetSampleDrag_ANGLE($3, 0); }
	| ANGLE C angle C forceOrDeflection
	    { $$ = new scnsetSampleDrag_ANGLE($3, $5); }
	| DEFALT
	    { $$ = new scnsetSampleDrag_DEFALT(0); }
	| DEFALT C forceOrDeflection
	    { $$ = new scnsetSampleDrag_DEFALT($3); }
	;

forceOrDeflection :
	  FORCE C rentVal
	    { $$ = new forceOrDeflection_FORCE($3); }
	| DEFLECTION C rentVal
	    { $$ = new forceOrDeflection_DEFLECTION($3); }
	;

selectStm :
	  SELECT SLASH intVal endOfLine
	    { $$ = new selectStmInt($3);
	      dmisStms.push_back($$);
	    }
	| SELECT SLASH stringVal endOfLine
	    { $$ = new selectStmString($3);
	      dmisStms.push_back($$);
	    }
	;

sensorStm :
	  ssLabel equalSign SENSOR SLASH sensorMinor endOfLine
	    { $$ = new sensorStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

sensorMinor :
	  sensorProbe
	    { $$ = $1; }
	| sensorMltprb
	    { $$ = $1; }
	| sensorVideo
	    { $$ = $1; }
	| sensorLaser
	    { $$ = $1; }
	| sensorInfred
	    { $$ = $1; }
	| sensorNoncon
	    { $$ = $1; }
	| sensorPoint
	    { $$ = $1; }
	| sensorLine
	    { $$ = $1; }
	| sensorArea
	    { $$ = $1; }
	;

sensorProbe :
	  PROBE C sensorProbeGeometry
	    { $$ = new sensorProbe($3, 0, 0); }
	| PROBE C sensorProbeGeometry C sensorItem
	    { $$ = new sensorProbe($3, 0, $5); }
	| PROBE C sensorProbeGeometry C sensorProbeForm
	    { $$ = new sensorProbe($3, $5, 0); }
	| PROBE C sensorProbeGeometry C sensorProbeForm C sensorItem
	    { $$ = new sensorProbe($3, $5, $7); }
	;

sensorMltprb :
	  MLTPRB C intVal C sensorMltprbItemList
	    { $$ = new sensorMltprb($3, $5, 0); }
	| MLTPRB C intVal C sensorMltprbItemList C sensorItem
	    { $$ = new sensorMltprb($3, $5, $7); }
	;

sensorMltprbItemList :
	  sensorMltprbItem
	    { $$ = new std::list<sensorMltprbItem *>;
	      $$->push_back($1); }
	| sensorMltprbItemList C sensorMltprbItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

sensorVideo :
	  VIDEO C sensorGeometry C rentVal C rentVal C rentVal
	    { $$ = new sensorVideo($3, $5, $7, $9, 0); }
	| VIDEO C sensorGeometry C rentVal C rentVal C rentVal C sensorItem
	    { $$ = new sensorVideo($3, $5, $7, $9, $11); }
	;

sensorLaser :
	  LASER C sensorGeometry C rentVal C rentVal
	    { $$ = new sensorLaser($3, $5, $7, 0); }
	| LASER C sensorGeometry C rentVal C rentVal C sensorItem
	    { $$ = new sensorLaser($3, $5, $7, $9); }
	;

sensorInfred :
	  INFRED C sensorGeometry C rentVal C rentVal C rentVal C rentVal
	    { $$ = new sensorInfred($3, $5, $7, $9, $11, 0); }
	| INFRED C sensorGeometry C rentVal C rentVal C rentVal C rentVal C sensorItem
	    { $$ = new sensorInfred($3, $5, $7, $9, $11, $13); }
	;

sensorNoncon :
	  NONCON C sensorGeometry C intVal
	    { $$ = new sensorNoncon($3, $5, 0); }
	| NONCON C sensorGeometry C intVal C sensorItem
	    { $$ = new sensorNoncon($3, $5, $7); }
	;

sensorPoint :
	  POINT C sensorGeometry C rentVal
	    { $$ = new sensorPoint($3, $5, 0); }
	| POINT C sensorGeometry C rentVal C sensorItem
	    { $$ = new sensorPoint($3, $5, $7); }
	;

sensorLine :
	  LINE C sensorGeometry C rentVal C rentVal
	    { $$ = new sensorLine($3, $5, $7, 0); }
	| LINE C sensorGeometry C rentVal C rentVal C sensorItem
	    { $$ = new sensorLine($3, $5, $7, $9); }
	;

sensorArea :
	  AREA C sensorGeometry C rentVal C rentVal C rentVal
	    { $$ = new sensorArea($3, $5, $7, $9, 0); }
	| AREA C sensorGeometry C rentVal C rentVal C rentVal C sensorItem
	    { $$ = new sensorArea($3, $5, $7, $9, $11); }
	;

sensorProbeGeometry :
	  vector C vector C rentVal
	    { $$ = new sensorProbeGeometry($1, $3, $5); }
	;

sensorProbeForm :
	  SPHERE
	    { $$ = new sensorProbeForm_SPHERE(); }
	| CYLNDR C rentVal
	    { $$ = new sensorProbeForm_CYLNDR($3); }
	| DISK C rentVal
	    { $$ = new sensorProbeForm_DISK($3); }
	;

sensorItem :
	  stringVal C stringVal C stringVal
	    { $$ = new sensorItem_stringVal($1, $3, $5); }
	| stringVal C stringVal C intVal
	    { $$ = new sensorItem_intVal($1, $3, $5); }
	;

sensorMltprbItem :
	  stringVal C sensorProbeGeometry
	    { $$ = new sensorMltprbItem_stringVal($1, $3); }
	| intVal C sensorProbeGeometry
	    { $$ = new sensorMltprbItem_intVal($1, $3); }
	;

sensorGeometry :
	  vector C vector C vector
	    { $$ = new sensorGeometry($1, $3, $5); }
	;

simreqtStm :
	  srLabel equalSign SIMREQT SLASH simreqtMinor endOfLine
	    { $$ = new simreqtStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

simreqtMinor :
	  FIRST
	    { $$ = new simreqtMinor_FIRST(); }
	| OPTIMAL
	    { $$ = new simreqtMinor_OPTIMAL(); }
	;

snsdefStm :
	  sLabel equalSign SNSDEF SLASH snsdefMinor endOfLine
	    { $$ = new snsdefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

snsdefMinor :
	  snsdefProbe
	    { $$ = $1; }
	| snsdefVideo
	    { $$ = $1; }
	| snsdefLaser
	    { $$ = $1; }
	| snsdefInfred
	    { $$ = $1; }
	| snsdefNoncon
	    { $$ = $1; }
	| snsdefBuild
	    { $$ = $1; }
	;

snsdefProbe :
	  PROBE C snsdefType C probeLocation C rentVal
	    { $$ = new snsdefProbe($3, $5, $7, 0); }
	| PROBE C snsdefType C probeLocation C rentVal C snsdefProbeForm
	    { $$ = new snsdefProbe($3, $5, $7, $9); }
	;

probeLocation :
	  CART C vector C vector
	    { $$ = new probeLocCart($3, $5); }
	| POL C angle C angle C vector C rentVal
	    { $$ = new probeLocPol($3, $5, $7, $9); }
	| VEC C vector C vector C rentVal
	    { $$ = new probeLocVec($3, $5, $7); }
	| sensorLabel2 C CART C vector
	    { $$ = new probeLocSnsCart($1, $5); }
	| sensorLabel2 C VEC C vector C rentVal
	    { $$ = new probeLocSnsVec($1, $5, $7); }
	;

snsdefProbeForm :
	  SPHERE
	    { $$ = new snsdefProbeForm_SPHERE(); }
	| CYLNDR C rentVal
	    { $$ = new snsdefProbeForm_CYLNDR($3); }
	| DISK C rentVal
	    { $$ = new snsdefProbeForm_DISK($3); }
	;

snsdefVideo :
	  VIDEO C snsdefType C snsdefLocation C rentVal C rentVal C rentVal
	    { $$ = new snsdefVideo($3, $5, $7, $9, $11); }
	;

snsdefLaser :
	  LASER C snsdefType C snsdefLocation C rentVal C rentVal
	    { $$ = new snsdefLaser($3, $5, $7, $9); }
	;

snsdefInfred :
	  INFRED C snsdefType C snsdefLocation C rentVal C rentVal C rentVal C rentVal
	    { $$ = new snsdefInfred($3, $5, $7, $9, $11, $13); }
	;

snsdefNoncon :
	  NONCON C snsdefType C snsdefLocation C intVal
	    { $$ = new snsdefNoncon($3, $5, $7); }
	;

snsdefLocation :
	  CART C vector C vector
	    { $$ = new snsdefLocation_CART($3, $5); }
	| POL C angle C angle C vector
	    { $$ = new snsdefLocation_POL($3, $5, $7); }
	| VEC C vector C vector
	    { $$ = new snsdefLocation_VEC($3, $5); }
	;

snsdefType :
	  FIXED
	    { $$ = new snsdefType_FIXED(); }
	| INDEX
	    { $$ = new snsdefType_INDEX(); }
	;

snsdefBuildSensor :
	  ssLabel
	    { $$ = $1; }
	| sgsLabel
	    { $$ = $1; }
	;

snsdefBuild :
	  BUILD C snsdefBuildItemList snsdefBuildSensor
	    { $$ = new snsdefBuild($3, $4); }
	;

snsdefBuildItemList :
	  sgLabel C
	    { $$ = new std::list<snsdefBuildItem *>;
	      $$->push_front(new snsdefBuildItem_sgLabel($1)); }
	| sgLabel C snsdefBuildItemList
	    { $$ = $3;
	      $$->push_front(new snsdefBuildItem_sgLabel($1)); }
	| swLabel C
	    { $$ = new std::list<snsdefBuildItem *>;
	      $$->push_front(new snsdefBuildItem_swLabel($1, 0)); }
	| swLabel C snsdefBuildItemList
	    { $$ = $3;
	      $$->push_front(new snsdefBuildItem_swLabel($1, 0)); }
	| swLabel C snsdefWristAngleList C
	    { $$ = new std::list<snsdefBuildItem *>;
	      $$->push_front(new snsdefBuildItem_swLabel($1, $3)); }
	| swLabel C snsdefWristAngleList C snsdefBuildItemList
	    { $$ = $5;
	      $$->push_front(new snsdefBuildItem_swLabel($1, $3)); }
	| sxLabel C
	    { $$ = new std::list<snsdefBuildItem *>;
	      $$->push_front(new snsdefBuildItem_sxLabel($1)); }
	| sxLabel C snsdefBuildItemList
	    { $$ = $3;
	      $$->push_front(new snsdefBuildItem_sxLabel($1)); }
	| rmLabel C
	    { $$ = new std::list<snsdefBuildItem *>;
	      $$->push_front(new snsdefBuildItem_rmLabel($1)); }
	| rmLabel C snsdefBuildItemList
	    { $$ = $3;
	      $$->push_front(new snsdefBuildItem_rmLabel($1)); }
	;

snsdefWristAngleList :
	  snsdefWristAngleItem
	    { $$ = new std::list<snsdefWristAngleItem *>;
	      $$->push_back($1); }
	| snsdefWristAngleList C snsdefWristAngleItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

snsdefWristAngleItem :
	  stringVal C angle
	    { $$ = new snsdefWristAngleItem($1, $3); }
	;

snsetStm :
	  SNSET SLASH snsetItemList endOfLine
	    { $$ = new snsetStm($3);
	      dmisStms.push_back($$);
	    }
	;

snsetItemList :
	  snsetTypeAndValue
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_snsetTypeAndValue($1)); }
	| snsetTypeAndValue C snsetItemList
	    { $$ = $3;
	      $$->push_front(new snsetItem_snsetTypeAndValue($1)); }
	| snsetType
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_snsetType($1)); }
	| snsetType C snsetItemList
	    { $$ = $3;
	      $$->push_front(new snsetItem_snsetType($1)); }
	| snsetLabelAndValue
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_snsetLabelAndValue($1)); }
	| snsetLabelAndValue C snsetItemList
	    { $$ = $3;
	      $$->push_front(new snsetItem_snsetLabelAndValue($1)); }
	| snsetLabel
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_snsetLabel($1)); }
	| snsetLabel C snsetItemList
	    { $$ = $3;
	      $$->push_front(new snsetItem_snsetLabel($1)); }
	| snsetToggle
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_snsetToggle($1)); }
	| snsetToggle C snsetItemList
	    { $$ = $3;
	      $$->push_front(new snsetItem_snsetToggle($1)); }
	| CLRSRF C snsetFeat
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_CLRSRF($3, 0)); }
	| CLRSRF C snsetFeat C snsetItemList
	    { $$ = $5;
	      $$->push_front(new snsetItem_CLRSRF($3, 0)); }
	| CLRSRF C snsetFeat C rentVal
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_CLRSRF($3, $5)); }
	| CLRSRF C snsetFeat C rentVal C snsetItemList
	    { $$ = $7;
	      $$->push_front(new snsetItem_CLRSRF($3, $5)); }
	| DEPTH C snsetFeat
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_DEPTH($3, 0)); }
	| DEPTH C snsetFeat C snsetItemList
	    { $$ = $5;
	      $$->push_front(new snsetItem_DEPTH($3, 0)); }
	| DEPTH C snsetFeat C rentVal
	    { $$ = new std::list<snsetItem *>;
	      $$->push_front(new snsetItem_DEPTH($3, $5)); }
	| DEPTH C snsetFeat C rentVal C snsetItemList
	    { $$ = $7;
	      $$->push_front(new snsetItem_DEPTH($3, $5)); }
	;

snsetTypeAndValue :
	  APPRCH C rentVal
	    { $$ = new snsetTypeAndValue_APPRCH($3); }
	| CLRSRF C rentVal
	    { $$ = new snsetTypeAndValue_CLRSRF($3); }
	| DEPTH C rentVal
	    { $$ = new snsetTypeAndValue_DEPTH($3); }
	| MINCON C rentVal
	    { $$ = new snsetTypeAndValue_MINCON($3); }
	| RETRCT C rentVal
	    { $$ = new snsetTypeAndValue_RETRCT($3); }
	| SEARCH C rentVal
	    { $$ = new snsetTypeAndValue_SEARCH($3); }
	| SCALEX C rentVal
	    { $$ = new snsetTypeAndValue_SCALEX($3); }
	| SCALEY C rentVal
	    { $$ = new snsetTypeAndValue_SCALEY($3); }
	;

snsetType :
	  FOCUSY
	    { $$ = new snsetType_FOCUSY(); }
	| FOCUSN
	    { $$ = new snsetType_FOCUSN(); }
	;

snsetToggle :
	  CLRSRF C OFF
	    { $$ = new snsetToggle_CLRSRF(); }
	| DEPTH C OFF
	    { $$ = new snsetToggle_DEPTH(); }
	;

snsetLabelAndValue :
	  vlLabel C rentVal
	    { $$ = new snsetLabelAndValue($1, $3); }
	;

snsetLabel :
	  vaLabel
	    { $$ = $1; }
	| vfLabel
	    { $$ = $1; }
	| vwLabel
	    { $$ = $1; }
	;

snsetFeat :
	  fLabel
	    { $$ = $1; }
	| faLabel
	    { $$ = $1; }
	| datLabel
	    { $$ = $1; }
	;

snsgrpStm :
	  sgsLabel equalSign SNSGRP SLASH snsgrpMinor endOfLine
	    { $$ = new snsgrpStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

snsgrpMinor :
	  BUILD C snsgrpItemList C ssLabel
	    { $$ = new snsgrpMinor_ssLabel($3, $5); }
	| BUILD C snsgrpItemList C sgsLabel
	    { $$ = new snsgrpMinor_sgsLabel($3, $5); }
	;

snsgrpItemList :
	  snsgrpItem
	    { $$ = new std::list<snsgrpItem *>;
	      $$->push_back($1); }
	| snsgrpItemList C snsgrpItem
	    { $$ = $1;
	      $$->push_back($3); }
	;

snsgrpItem :
	  sgLabel
	    { $$ = $1; }
	| swLabel
	    { $$ = $1; }
	| sxLabel
	    { $$ = $1; }
	| rmLabel
	    { $$ = $1; }
	;

snslctStm :
	  SNSLCT SLASH snslctSensor endOfLine
	    { $$ = new snslctStmSensor($3);
	      dmisStms.push_back($$);
	    }
	| SNSLCT SLASH snslctGroup endOfLine
	    { $$ = new snslctStmGroup($3);
	      dmisStms.push_back($$);
	    }
	;

snslctGroup :
	  gsaLabel C featureLabel
	    { $$ = new snslctGroupFeat($1, $3, 0); }
	| gsaLabel C featureLabel C FZ C angle
	    { $$ = new snslctGroupFeat($1, $3, $7); }
	| gsaLabel C VEC C vector
	    { $$ = new snslctGroupVec($1, $5, 0); }
	| gsaLabel C VEC C vector C FZ C angle
	    { $$ = new snslctGroupVec($1, $5, $9); }
	;

snslctSensor :
	  sensorLabel2
	    { $$ = new snslctSensorTip($1, 0, 0); }
	| sensorLabel2 C snslctData
	    { $$ = new snslctSensorTip($1, 0, $3); }
	| sensorLabel2 C snslctTipData
	    { $$ = new snslctSensorTip($1, $3, 0); }
	| sensorLabel2 C snslctTipData C snslctData
	    { $$ = new snslctSensorTip($1, $3, $5); }
	| sensorLabel2 C sensorList
	    { $$ = new snslctSensorList($1, $3); }
	;

snslctTipData :
	  stringVal
	    { $$ = $1; }
	| intVal
	    { $$ = $1; }
	;

sensorList :
	  sensorLabel2
	    { $$ = new std::list<sensorLabel2 *>;
	      $$->push_back($1); }
	| sensorList C sensorLabel2
	    { $$ = $1;
	      $$->push_back($3); }
	;

snslctData :
	  snslctWristList
	    { $$ = new snslctData_snslctWristList($1); }
	| probeOrient
	    { $$ = new snslctData_probeOrient($1); }
	;

snslctWristList :
	  swLabel C snslctWristAngleItem
            { $$ = new std::list<snslctWristItem *>;
              $$->push_front(new snslctWristItem
                    ($1, new std::list<snslctWristAngleItem *>(1, $3))); }
	| swLabel C snslctWristAngleList snslctWristAngleItem
            { $$ = new std::list<snslctWristItem *>;
              $3->push_back($4);
              $$->push_front(new snslctWristItem($1, $3)); }
	| swLabel C snslctWristAngleList snslctWristList
            { $$ = $4;
              $$->push_front(new snslctWristItem($1, $3)); }
	;

snslctWristAngleList :
	  stringVal C angle C
	    { $$ = new std::list<snslctWristAngleItem *>;
	      $$->push_back(new snslctWristAngleAngle($1, $3)); }
	| snslctWristAngleList stringVal C angle C
	    { $$ = $1;
	      $$->push_back(new snslctWristAngleAngle($2, $4)); }
	| stringVal C featureLabel C
	    { $$ = new std::list<snslctWristAngleItem *>;
	      $$->push_back(new snslctWristAngleFeat($1, $3, 0)); }
	| stringVal C featureLabel C FZ C angle C
	    { $$ = new std::list<snslctWristAngleItem *>;
	      $$->push_back(new snslctWristAngleFeat($1, $3, $7)); }
	| snslctWristAngleList stringVal C featureLabel C
	    { $$ = $1;
	      $$->push_back(new snslctWristAngleFeat($2, $4, 0)); }
	| snslctWristAngleList stringVal C featureLabel C FZ C angle C
	    { $$ = $1;
	      $$->push_back(new snslctWristAngleFeat($2, $4, $8)); }
	| stringVal C VEC C vector C
	    { $$ = new std::list<snslctWristAngleItem *>;
	      $$->push_back(new snslctWristAngleVec($1, $5, 0)); }
	| stringVal C VEC C vector C FZ C angle C
	    { $$ = new std::list<snslctWristAngleItem *>;
	      $$->push_back(new snslctWristAngleVec($1, $5, $9)); }
	| snslctWristAngleList stringVal C VEC C vector C
	    { $$ = $1;
	      $$->push_back(new snslctWristAngleVec($2, $6, 0)); }
	| snslctWristAngleList stringVal C VEC C vector C FZ C angle C
	    { $$ = $1;
	      $$->push_back(new snslctWristAngleVec($2, $6, $10)); }
	;

snslctWristAngleItem :
	  stringVal C angle
	    { $$ = new snslctWristAngleAngle($1, $3); }
	| stringVal C featureLabel
	    { $$ = new snslctWristAngleFeat($1, $3, 0); }
	| stringVal C featureLabel C FZ C angle
	    { $$ = new snslctWristAngleFeat($1, $3, $7); }
	| stringVal C VEC C vector
	    { $$ = new snslctWristAngleVec($1, $5, 0); }
	| stringVal C VEC C vector C FZ C angle
	    { $$ = new snslctWristAngleVec($1, $5, $9); }
	;

snsmntStm :
	  SNSMNT SLASH snsmntMinor endOfLine
	    { $$ = new snsmntStm($3);
	      dmisStms.push_back($$);
	    }
	;

snsmntMinor :
	  XVEC C vector C ZVEC C vector C MNTLEN C vector
	    { $$ = new snsmntMinor($3, $7, $11); }
	;

tecompStm :
	  TECOMP SLASH tecompMinor endOfLine
	    { $$ = new tecompStm($3);
	      dmisStms.push_back($$);
	    }
	;

tecompMinor :
	  MACH C state
	    { $$ = new tecompMinorMach($3); }
	| PART C ON C rentVal C stringList
	    { $$ = new tecompMinorOnList(0, $5, 0, $7); }
	| PART C ON C rentVal C rentVal C stringList
	    { $$ = new tecompMinorOnList(0, $5, $7, $9); }
	| PART C ON tecompVar2 C rentVal C stringList
	    { $$ = new tecompMinorOnList($4, $6, 0, $8); }
	| PART C ON tecompVar2 C rentVal C rentVal C stringList
	    { $$ = new tecompMinorOnList($4, $6, $8, $10); }
	| PART C ON C rentVal C ALL
	    { $$ = new tecompMinorOnAll(0, $5, 0); }
	| PART C ON C rentVal C rentVal C ALL
	    { $$ = new tecompMinorOnAll(0, $5, $7); }
	| PART C ON tecompVar2 C rentVal C ALL
	    { $$ = new tecompMinorOnAll($4, $6, 0); }
	| PART C ON tecompVar2 C rentVal C rentVal C ALL
	    { $$ = new tecompMinorOnAll($4, $6, $8); }
	| PART C OFF
	    { $$ = new tecompMinorOff(); }
	;

tecompVar2 :
	  C daLabel
	    { $$ = new tecompDatum($2); }
	| C OFFSET C rentVal C rentVal C rentVal
	    { $$ = new tecompOffset($4, $6, $8); }
	;

textStm :
	  TEXT SLASH textMinor endOfLine
	    { $$ = new textStm($3);
	      dmisStms.push_back($$);
	    }
	;

textMinor :
	  textMan
	    { $$ = $1; }
	| textOper
	    { $$ = $1; }
	| textOutfil
	    { $$ = $1; }
	| textQuery
	    { $$ = $1; }
	;

textMan :
	  MAN C stringVal
	    { $$ = new textMan($3); }
	;

textOper :
	  OPER C stringVal
	    { $$ = new textOper($3); }
	;

textOutfil :
	  OUTFIL C stringVal
	    { $$ = new textOutfil($3); }
	;

textQuery :
	  QUERY C textQueryFormat C stringVal
	    { $$ = new textQuery($3, $5); }
	;

textQueryFormat :
	  labelName C intVal C textType C leftRight
	    { $$ = new textQueryFormat($1, $3, $5, $7); }
	;

textType :
	  ALPHA
	    { $$ = new textType_ALPHA(); }
	| NUMERIC
	    { $$ = new textType_NUMERIC(); }
	| PRNTCHAR
	    { $$ = new textType_PRNTCHAR(); }
	;

thldefStm :
	  thLabel equalSign THLDEF SLASH thldefPocketList endOfLine
	    { $$ = new thldefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

thldefPocketList :
	  thldefPocket
	    { $$ = new std::list<thldefPocket *>;
	      $$->push_back($1); }
	| thldefPocketList C thldefPocket
	    { $$ = $1;
	      $$->push_back($3); }
	;

thldefPocket :
	  sLabel C intVal
	    { $$ = new thldefPocket_sLabel($1, $3); }
	| ssLabel C intVal
	    { $$ = new thldefPocket_ssLabel($1, $3); }
	| sgsLabel C intVal
	    { $$ = new thldefPocket_sgsLabel($1, $3); }
	| sgLabel C intVal
	    { $$ = new thldefPocket_sgLabel($1, $3); }
	| swLabel C intVal
	    { $$ = new thldefPocket_swLabel($1, $3); }
	| sxLabel C intVal
	    { $$ = new thldefPocket_sxLabel($1, $3); }
	| rmLabel C intVal
	    { $$ = new thldefPocket_rmLabel($1, $3); }
	;

tolAnglStm :
	  tLabel equalSign TOL SLASH ANGL C angle C angle endOfLine
	    { $$ = new tolAnglStm($1, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

tolAnglbStm :
	  tLabel equalSign TOL SLASH ANGLB C angle C angle C angle endOfLine
	    { $$ = new tolAnglbStm($1, $7, $9, $11, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH ANGLB C angle C angle C angle C tolZoneDir2 endOfLine
	    { $$ = new tolAnglbStm($1, $7, $9, $11, $13);
	      dmisStms.push_back($$);
	    }
	;

tolZoneDir2 :
	  XYPLAN
	    { $$ = new tolZoneDir2_XYPLAN(); }
	| YZPLAN
	    { $$ = new tolZoneDir2_YZPLAN(); }
	| ZXPLAN
	    { $$ = new tolZoneDir2_ZXPLAN(); }
	| VEC C vector
	    { $$ = new tolZoneDir2_VEC($3); }
	;

tolAnglwrtStm :
	  tLabel equalSign TOL SLASH ANGLWRT C angle C angle C angle C tolFeature endOfLine
	    { $$ = new tolAnglwrtStm($1, $7, $9, $11, $13, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH ANGLWRT C angle C angle C angle C tolFeature C tolZoneDir2 endOfLine
	    { $$ = new tolAnglwrtStm($1, $7, $9, $11, $13, $15);
	      dmisStms.push_back($$);
	    }
	;

tolAnglrStm :
	  tLabel equalSign TOL SLASH ANGLR C angle C tolAppData endOfLine
	    { $$ = new tolAnglrStm($1, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

tolCirltyStm :
	  tLabel equalSign TOL SLASH CIRLTY C rentVal endOfLine
	    { $$ = new tolCirltyStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolCompos1 :
	  rentVal C tolFeatureMatC
	    { $$ = new tolCompos1($1, 0, $3, 0, 0); }
	| rentVal C tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCompos1($1, 0, $3, $4, 0); }
	| rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCompos1($1, 0, $3, $4, $5); }
	| rentVal C tolMatCond C tolFeatureMatC
	    { $$ = new tolCompos1($1, $3, $5, 0, 0); }
	| rentVal C tolMatCond C tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCompos1($1, $3, $5, $6, 0); }
	| rentVal C tolMatCond C tolFeatureMatC tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCompos1($1, $3, $5, $6, $7); }
	;

tolCompos2 :
	  rentVal
	    { $$ = new tolCompos2($1, 0, 0, 0, 0); }
	| rentVal C tolFeatureMat
	    { $$ = new tolCompos2($1, 0, $3, 0, 0); }
	| rentVal C tolFeatureMatC tolFeatureMat
	    { $$ = new tolCompos2($1, 0, $3, $4, 0); }
	| rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMat
	    { $$ = new tolCompos2($1, 0, $3, $4, $5); }
	| rentVal C tolMatCond
	    { $$ = new tolCompos2($1, $3, 0, 0, 0); }
	| rentVal C tolMatCond C tolFeatureMat
	    { $$ = new tolCompos2($1, $3, $5, 0, 0); }
	| rentVal C tolMatCond C tolFeatureMatC tolFeatureMat
	    { $$ = new tolCompos2($1, $3, $5, $6, 0); }
	| rentVal C tolMatCond C tolFeatureMatC tolFeatureMatC tolFeatureMat
	    { $$ = new tolCompos2($1, $3, $5, $6, $7); }
	;

tolComposStm :
	  tLabel equalSign TOL SLASH COMPOS C PATERN C tolCompos1 FEATUR C tolCompos2 endOfLine
	    { $$ = new tolComposStm($1, $9, $12);
	      dmisStms.push_back($$);
	    }
	;

tolFeatureMat :
	  datLabel
	    { $$ = new tolFeatureMat_datLabel($1, 0); }
	| datLabel C tolMatCond
	    { $$ = new tolFeatureMat_datLabel($1, $3); }
	| faLabel
	    { $$ = new tolFeatureMat_faLabel($1, 0); }
	| faLabel C tolMatCond
	    { $$ = new tolFeatureMat_faLabel($1, $3); }
	| fLabel
	    { $$ = new tolFeatureMat_fLabel($1); }
	;

tolFeatureMatC :
	  datLabel C
	    { $$ = new tolFeatureMat_datLabel($1, 0); }
	| datLabel C tolMatCond C
	    { $$ = new tolFeatureMat_datLabel($1, $3); }
	| faLabel C
	    { $$ = new tolFeatureMat_faLabel($1, 0); }
	| faLabel C tolMatCond C
	    { $$ = new tolFeatureMat_faLabel($1, $3); }
	| fLabel C
	    { $$ = new tolFeatureMat_fLabel($1); }
	;

tolConcenStm :
	  tLabel equalSign TOL SLASH CONCEN C rentVal C tolFeature endOfLine
	    { $$ = new tolConcenStm($1, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

tolCortolStm :
	  tLabel equalSign TOL SLASH CORTOL C tolCortolMinor endOfLine
	    { $$ = new tolCortolStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolCortolMinor :
	  axis C rentVal C rentVal
	    { $$ = new tolCortolAxis($1, $3, $5, 0); }
	| axis C rentVal C rentVal C axial
	    { $$ = new tolCortolAxis($1, $3, $5, $7); }
	| RADIAL C rentVal C rentVal
	    { $$ = new tolCortolRadial($3, $5); }
	| ANGLE C rentVal C rentVal
	    { $$ = new tolCortolAngle($3, $5); }
	;

axial :
	  AXIAL C featureLabel
	    { $$ = new axial($3, 0); }
	| AXIAL C featureLabel C rentVal
	    { $$ = new axial($3, $5); }
	;

tolCproflStm :
	  tLabel equalSign TOL SLASH CPROFL C tolCprofSpecC tolCprofSpec endOfLine
	    { $$ = new tolCproflStm($1, $7, $8);
	      dmisStms.push_back($$);
	    }
	;

tolCprofsStm :
	  tLabel equalSign TOL SLASH CPROFS C tolCprofSpecC tolCprofSpec endOfLine
	    { $$ = new tolCprofsStm($1, $7, $8);
	      dmisStms.push_back($$);
	    }
	;

tolCprofSpec :
	  rentVal C rentVal
	    { $$ = new tolCprofSpec($1, $3, 0, 0, 0, false); }
	| rentVal C rentVal C AVGDEV
	    { $$ = new tolCprofSpec($1, $3, 0, 0, 0, true); }
	| rentVal C rentVal C tolFeatureMat
	    { $$ = new tolCprofSpec($1, $3, $5, 0, 0, false); }
	| rentVal C rentVal C tolFeatureMatC AVGDEV
	    { $$ = new tolCprofSpec($1, $3, $5, 0, 0, true); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMat
	    { $$ = new tolCprofSpec($1, $3, $5, $6, 0, false); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC AVGDEV
	    { $$ = new tolCprofSpec($1, $3, $5, $6, 0, true); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMat
	    { $$ = new tolCprofSpec($1, $3, $5, $6, $7, false); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC AVGDEV
	    { $$ = new tolCprofSpec($1, $3, $5, $6, $7, true); }
	;

tolCprofSpecC :
	  rentVal C rentVal C
	    { $$ = new tolCprofSpecC($1, $3, 0, 0, 0, false); }
	| rentVal C rentVal C AVGDEV C
	    { $$ = new tolCprofSpecC($1, $3, 0, 0, 0, true); }
	| rentVal C rentVal C tolFeatureMatC
	    { $$ = new tolCprofSpecC($1, $3, $5, 0, 0, false); }
	| rentVal C rentVal C tolFeatureMatC AVGDEV C
	    { $$ = new tolCprofSpecC($1, $3, $5, 0, 0, true); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCprofSpecC($1, $3, $5, $6, 0, false); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC AVGDEV C
	    { $$ = new tolCprofSpecC($1, $3, $5, $6, 0, true); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC
	    { $$ = new tolCprofSpecC($1, $3, $5, $6, $7, false); }
	| rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC AVGDEV C
	    { $$ = new tolCprofSpecC($1, $3, $5, $6, $7, true); }
	;

tolCrnoutStm :
	  tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, 0, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel C VEC C vector endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, 0, 0, $13);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel C tolFeature endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, $11, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel C tolFeature C VEC C vector endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, $11, 0, $15);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel C tolFeature C tolFeature endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, $11, $13, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH CRNOUT C rentVal C datLabel C tolFeature C tolFeature C VEC C vector endOfLine
	    { $$ = new tolCrnoutStm($1, $7, $9, $11, $13, $17);
	      dmisStms.push_back($$);
	    }
	;

tolCylctyStm :
	  tLabel equalSign TOL SLASH CYLCTY C rentVal endOfLine
	    { $$ = new tolCylctyStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolDiamStm :
	  tLabel equalSign TOL SLASH DIAM C tolDiamMinor endOfLine
	    { $$ = new tolDiamStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolDiamMinor :
	  rentVal C rentVal
	    { $$ = new tolDiamPlain($1, $3, 0); }
	| rentVal C rentVal C radiusSpec
	    { $$ = new tolDiamPlain($1, $3, $5); }
	| rentVal C rentVal C AVG
	    { $$ = new tolDiamAvg($1, $3, 0); }
	| rentVal C rentVal C radiusSpec C AVG
	    { $$ = new tolDiamAvg($1, $3, $5); }
	| rentVal C rentVal C MINMAX
	    { $$ = new tolDiamMinmax($1, $3, 0); }
	| rentVal C rentVal C radiusSpec C MINMAX
	    { $$ = new tolDiamMinmax($1, $3, $5); }
	;

avgMaxMin :
	  C AVG
	    { $$ = new avgMaxMin_AVG(); }
	| C MAX
	    { $$ = new avgMaxMin_MAX(); }
	| C MIN
	    { $$ = new avgMaxMin_MIN(); }
	;

tolDistbStm :
	  tLabel equalSign TOL SLASH DISTB C tolDistbTol C tolDistbDir endOfLine
	    { $$ = new tolDistbStm($1, $7, $9, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH DISTB C tolDistbTol C tolDistbDir avgMaxMin endOfLine
	    { $$ = new tolDistbStm($1, $7, $9, $10);
	      dmisStms.push_back($$);
	    }
	;

tolDistbDir :
	  XAXIS
	    { $$ = new tolDistbDir_XAXIS(); }
	| YAXIS
	    { $$ = new tolDistbDir_YAXIS(); }
	| ZAXIS
	    { $$ = new tolDistbDir_ZAXIS(); }
	| PT2PT
	    { $$ = new tolDistbDir_PT2PT(); }
	;

tolDistbTol :
	  tolDistbLimit
	    { $$ = $1; }
	| tolDistbNominl
	    { $$ = $1; }
	;

tolDistbLimit :
	  LIMIT C rentVal C rentVal
	    { $$ = new tolDistbLimit($3, $5); }
	;

tolDistbNominl :
	  NOMINL C rentVal C rentVal C rentVal
	    { $$ = new tolDistbNominl($3, $5, $7); }
	;

tolDistwrtStm :
	  tLabel equalSign TOL SLASH DISTWRT C tolDistbTol C tolFeature C tolDistbDir endOfLine
	    { $$ = new tolDistwrtStm($1, $7, $9, $11, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH DISTWRT C tolDistbTol C tolFeature C tolDistbDir avgMaxMin endOfLine
	    { $$ = new tolDistwrtStm($1, $7, $9, $11, $12);
	      dmisStms.push_back($$);
	    }
	;

tolFlatStm :
	  tLabel equalSign TOL SLASH FLAT C tolFlatMinor endOfLine
	    { $$ = new tolFlatStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolFlatMinor :
	  rentVal
	    { $$ = new tolFlatZon($1); }
	| rentVal C rentVal C rentVal
	    { $$ = new tolFlatPer($1, $3, $5); }
	| rentVal C rentVal C rentVal C rentVal
	    { $$ = new tolFlatZonPer($1, $3, $5, $7); }
	;

tolGtolStm :
	  tLabel equalSign TOL SLASH GTOL C tolGtolTrans C tolGtolRot C intVal C rentVal C angle endOfLine
	    { $$ = new tolGtolStm($1, $7, $9, $11, $13, $15, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH GTOL C tolGtolTrans C tolGtolRot C intVal C rentVal C angle C tolGtolSpec endOfLine
	    { $$ = new tolGtolStm($1, $7, $9, $11, $13, $15, $17);
	      dmisStms.push_back($$);
	    }
	;

tolGtolTrans :
	  XDIR
	    { $$ = new tolGtolTrans_XDIR(); }
	| YDIR
	    { $$ = new tolGtolTrans_YDIR(); }
	| ZDIR
	    { $$ = new tolGtolTrans_ZDIR(); }
	| XYDIR
	    { $$ = new tolGtolTrans_XYDIR(); }
	| YZDIR
	    { $$ = new tolGtolTrans_YZDIR(); }
	| ZXDIR
	    { $$ = new tolGtolTrans_ZXDIR(); }
	| XYZDIR
	    { $$ = new tolGtolTrans_XYZDIR(); }
	| NOTRAN
	    { $$ = new tolGtolTrans_NOTRAN(); }
	;

tolGtolRot :
	  XAXIS
	    { $$ = new tolGtolRot_XAXIS(); }
	| YAXIS
	    { $$ = new tolGtolRot_YAXIS(); }
	| ZAXIS
	    { $$ = new tolGtolRot_ZAXIS(); }
	| XYAXIS
	    { $$ = new tolGtolRot_XYAXIS(); }
	| YZAXIS
	    { $$ = new tolGtolRot_YZAXIS(); }
	| ZXAXIS
	    { $$ = new tolGtolRot_ZXAXIS(); }
	| XYZAXI
	    { $$ = new tolGtolRot_XYZAXI(); }
	| NOROT
	    { $$ = new tolGtolRot_NOROT(); }
	;

tolGtolSpec :
	  PERCNT
	    { $$ = new tolGtolSpec_PERCNT(); }
	| INTFPT
	    { $$ = new tolGtolSpec_INTFPT(); }
	;

tolAppData :
	  rentVal C tolFeatMaxMat
	    { $$ = new tolAppData($1, 0, $3, 0, 0, 0); }
	| rentVal C tolFeatMaxMatC tolZoneDir
	    { $$ = new tolAppData($1, 0, $3, 0, 0, $4); }
	| rentVal C tolFeatMaxMatC tolZoneType
	    { $$ = new tolAppData($1, 0, $3, 0, $4, 0); }
	| rentVal C tolFeatMaxMatC tolZoneType C tolZoneDir
	    { $$ = new tolAppData($1, 0, $3, 0, $4, $6); }
	| rentVal C tolFeatMaxMatC tolFeatMaxMat
	    { $$ = new tolAppData($1, 0, $3, $4, 0, 0); }
	| rentVal C tolFeatMaxMatC tolFeatMaxMatC tolZoneDir
	    { $$ = new tolAppData($1, 0, $3, $4, 0, $5); }
	| rentVal C tolFeatMaxMatC tolFeatMaxMatC tolZoneType
	    { $$ = new tolAppData($1, 0, $3, $4, $5, 0); }
	| rentVal C tolFeatMaxMatC tolFeatMaxMatC tolZoneType C tolZoneDir
	    { $$ = new tolAppData($1, 0, $3, $4, $5, $7); }
	| rentVal C tolMaxMatCondC tolFeatMaxMat
	    { $$ = new tolAppData($1, $3, $4, 0, 0, 0); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolZoneDir
	    { $$ = new tolAppData($1, $3, $4, 0, 0, $5); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolZoneType
	    { $$ = new tolAppData($1, $3, $4, 0, $5, 0); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolZoneType C tolZoneDir
	    { $$ = new tolAppData($1, $3, $4, 0, $5, $7); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolFeatMaxMat
	    { $$ = new tolAppData($1, $3, $4, $5, 0, 0); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolFeatMaxMatC tolZoneDir
	    { $$ = new tolAppData($1, $3, $4, $5, 0, $6); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolFeatMaxMatC tolZoneType
	    { $$ = new tolAppData($1, $3, $4, $5, $6, 0); }
	| rentVal C tolMaxMatCondC tolFeatMaxMatC tolFeatMaxMatC tolZoneType C tolZoneDir
	    { $$ = new tolAppData($1, $3, $4, $5, $6, $8); }
	;

tolFeatMaxMat :
	  datLabel
	    { $$ = new tolFeatMaxMat_datLabel($1, 0); }
	| datLabel C tolMaxMatCond
	    { $$ = new tolFeatMaxMat_datLabel($1, $3); }
	| faLabel
	    { $$ = new tolFeatMaxMat_faLabel($1, 0); }
	| faLabel C tolMaxMatCond
	    { $$ = new tolFeatMaxMat_faLabel($1, $3); }
	| fLabel
	    { $$ = new tolFeatMaxMat_fLabel($1); }
	;

tolFeatMaxMatC :
	  datLabel C
	    { $$ = new tolFeatMaxMat_datLabel($1, 0); }
	| datLabel C tolMaxMatCondC
	    { $$ = new tolFeatMaxMat_datLabel($1, $3); }
	| faLabel C
	    { $$ = new tolFeatMaxMat_faLabel($1, 0); }
	| faLabel C tolMaxMatCondC
	    { $$ = new tolFeatMaxMat_faLabel($1, $3); }
	| fLabel C
	    { $$ = new tolFeatMaxMat_fLabel($1); }
	;

tolMaxMatCond :
	  LMC
	    { $$ = new tolMaxMatCond_LMC(0); }
	| LMC C MAX C rentVal
	    { $$ = new tolMaxMatCond_LMC($5); }
	| RFS
	    { $$ = new tolMaxMatCond_RFS(); }
	| MMC
	    { $$ = new tolMaxMatCond_MMC(0); }
	| MMC C MAX C rentVal
	    { $$ = new tolMaxMatCond_MMC($5); }
	;

tolMaxMatCondC :
	  LMC C
	    { $$ = new tolMaxMatCond_LMC(0); }
	| LMC C MAX C rentVal C
	    { $$ = new tolMaxMatCond_LMC($5); }
	| RFS C
	    { $$ = new tolMaxMatCond_RFS(); }
	| MMC C
	    { $$ = new tolMaxMatCond_MMC(0); }
	| MMC C MAX C rentVal C
	    { $$ = new tolMaxMatCond_MMC($5); }
	;

tolZoneDir :
	  XAXIS
	    { $$ = new tolZoneDir_XAXIS(); }
	| YAXIS
	    { $$ = new tolZoneDir_YAXIS(); }
	| ZAXIS
	    { $$ = new tolZoneDir_ZAXIS(); }
	| VEC C vector
	    { $$ = new tolZoneDir_VEC($3); }
	;

tolParlelStm :
	  tLabel equalSign TOL SLASH PARLEL C tolAppData endOfLine
	    { $$ = new tolParlelStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolPerpStm :
	  tLabel equalSign TOL SLASH PERP C tolAppData endOfLine
	    { $$ = new tolPerpStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolPosStm :
	  tLabel equalSign TOL SLASH POS C tolPosStart tolPosEnd endOfLine
	    { $$ = new tolPosStm($1, $7, $8);
	      dmisStms.push_back($$);
	    }
	;

tolPosStart :
	  TWOD C rentVal
	    { $$ = new tolPosStart_TWOD($3); }
	| THREED C rentVal
	    { $$ = new tolPosStart_THREED($3); }
	;

tolPosEnd :
	 
	    { $$ = new tolPosEnd(0, 0, 0, 0, 0); }
	| C tolPosMethod
	    { $$ = new tolPosEnd(0, 0, 0, 0, $2); }
	| C tolFeatureMat
	    { $$ = new tolPosEnd(0, $2, 0, 0, 0); }
	| C tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd(0, $2, 0, 0, $3); }
	| C tolFeatureMatC tolFeatureMat
	    { $$ = new tolPosEnd(0, $2, $3, 0, 0); }
	| C tolFeatureMatC tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd(0, $2, $3, 0, $4); }
	| C tolFeatureMatC tolFeatureMatC tolFeatureMat
	    { $$ = new tolPosEnd(0, $2, $3, $4, 0); }
	| C tolFeatureMatC tolFeatureMatC tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd(0, $2, $3, $4, $5); }
	| C tolMatCond
	    { $$ = new tolPosEnd($2, 0, 0, 0, 0); }
	| C tolMatCond C tolPosMethod
	    { $$ = new tolPosEnd($2, 0, 0, 0, $4); }
	| C tolMatCond C tolFeatureMat
	    { $$ = new tolPosEnd($2, $4, 0, 0, 0); }
	| C tolMatCond C tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd($2, $4, 0, 0, $5); }
	| C tolMatCond C tolFeatureMatC tolFeatureMat
	    { $$ = new tolPosEnd($2, $4, $5, 0, 0); }
	| C tolMatCond C tolFeatureMatC tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd($2, $4, $5, 0, $6); }
	| C tolMatCond C tolFeatureMatC tolFeatureMatC tolFeatureMat
	    { $$ = new tolPosEnd($2, $4, $5, $6, 0); }
	| C tolMatCond C tolFeatureMatC tolFeatureMatC tolFeatureMatC tolPosMethod
	    { $$ = new tolPosEnd($2, $4, $5, $6, $7); }
	;

tolPosMethod :
	  XAXIS
	    { $$ = new tolPosMethod_XAXIS(); }
	| YAXIS
	    { $$ = new tolPosMethod_YAXIS(); }
	| ZAXIS
	    { $$ = new tolPosMethod_ZAXIS(); }
	| RADIAL
	    { $$ = new tolPosMethod_RADIAL(); }
	| ANGLE
	    { $$ = new tolPosMethod_ANGLE(); }
	| VEC C vector
	    { $$ = new tolPosMethod_VEC($3); }
	;

tolProflStm :
	  tLabel equalSign TOL SLASH PROFL C rentVal C rentVal endOfLine
	    { $$ = new tolProflStm($1, $7, $9, 0, 0, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolZoneDir2 endOfLine
	    { $$ = new tolProflStm($1, $7, $9, 0, 0, 0, $11);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMat endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, 0, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMatC tolZoneDir2 endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, 0, 0, $12);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, $12, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolZoneDir2 endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, $12, 0, $13);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, $12, $13, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFL C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC tolZoneDir2 endOfLine
	    { $$ = new tolProflStm($1, $7, $9, $11, $12, $13, $14);
	      dmisStms.push_back($$);
	    }
	;

tolProfpStm :
	  tLabel equalSign TOL SLASH PROFP C rentVal C rentVal endOfLine
	    { $$ = new tolProfpStm($1, $7, $9, 0, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFP C rentVal C rentVal C tolFeatureMat endOfLine
	    { $$ = new tolProfpStm($1, $7, $9, $11, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFP C rentVal C rentVal C tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProfpStm($1, $7, $9, $11, $12, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFP C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProfpStm($1, $7, $9, $11, $12, $13);
	      dmisStms.push_back($$);
	    }
	;

tolProfsStm :
	  tLabel equalSign TOL SLASH PROFS C rentVal C rentVal endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, 0, 0, 0, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C AVGDEV endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, 0, 0, 0, true);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMat endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, 0, 0, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMatC AVGDEV endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, 0, 0, true);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, $12, 0, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMatC tolFeatureMatC AVGDEV endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, $12, 0, true);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMat endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, $12, $13, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH PROFS C rentVal C rentVal C tolFeatureMatC tolFeatureMatC tolFeatureMatC AVGDEV endOfLine
	    { $$ = new tolProfsStm($1, $7, $9, $11, $12, $13, true);
	      dmisStms.push_back($$);
	    }
	;

tolRadStm :
	  tLabel equalSign TOL SLASH RAD C rentVal C rentVal endOfLine
	    { $$ = new tolRadStm($1, $7, $9, 0, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH RAD C rentVal C rentVal C MINMAX endOfLine
	    { $$ = new tolRadStm($1, $7, $9, 0, true);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH RAD C rentVal C rentVal C tolRadSpec endOfLine
	    { $$ = new tolRadStm($1, $7, $9, $11, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH RAD C rentVal C rentVal C tolRadSpec C MINMAX endOfLine
	    { $$ = new tolRadStm($1, $7, $9, $11, true);
	      dmisStms.push_back($$);
	    }
	;

tolRadSpec :
	  AVG
	    { $$ = new tolRadSpec_AVG(); }
	| CRAD
	    { $$ = new tolRadSpec_CRAD(); }
	;

tolZoneDir3 :
	  XDIR
	    { $$ = new tolZoneDir3_XDIR(); }
	| YDIR
	    { $$ = new tolZoneDir3_YDIR(); }
	| ZDIR
	    { $$ = new tolZoneDir3_ZDIR(); }
	| VEC C vector
	    { $$ = new tolZoneDir3_VEC($3); }
	;

tolStrghtStm :
	  tLabel equalSign TOL SLASH STRGHT C tolStrghtMinor endOfLine
	    { $$ = new tolStrghtStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolStrghtMinor :
	  rentVal
	    { $$ = new tolStrghtPlain($1, 0, 0); }
	| rentVal C tolZoneDir3
	    { $$ = new tolStrghtPlain($1, 0, $3); }
	| rentVal C axis
	    { $$ = new tolStrghtPlain($1, $3, 0); }
	| rentVal C axis C tolZoneDir3
	    { $$ = new tolStrghtPlain($1, $3, $5); }
	| rentVal C tolMatCond
	    { $$ = new tolStrghtMat($1, $3, 0, 0); }
	| rentVal C tolMatCond C tolZoneDir3
	    { $$ = new tolStrghtMat($1, $3, 0, $5); }
	| rentVal C tolMatCond C axis
	    { $$ = new tolStrghtMat($1, $3, $5, 0); }
	| rentVal C tolMatCond C axis C tolZoneDir3
	    { $$ = new tolStrghtMat($1, $3, $5, $7); }
	| rentVal C rentVal
	    { $$ = new tolStrghtPer1($1, $3, 0, 0); }
	| rentVal C rentVal C tolZoneDir3
	    { $$ = new tolStrghtPer1($1, $3, 0, $5); }
	| rentVal C rentVal C axis
	    { $$ = new tolStrghtPer1($1, $3, $5, 0); }
	| rentVal C rentVal C axis C tolZoneDir3
	    { $$ = new tolStrghtPer1($1, $3, $5, $7); }
	| rentVal C rentVal C rentVal
	    { $$ = new tolStrghtPer2($1, $3, $5, 0, 0); }
	| rentVal C rentVal C rentVal C tolZoneDir3
	    { $$ = new tolStrghtPer2($1, $3, $5, 0, $7); }
	| rentVal C rentVal C rentVal C axis
	    { $$ = new tolStrghtPer2($1, $3, $5, $7, 0); }
	| rentVal C rentVal C rentVal C axis C tolZoneDir3
	    { $$ = new tolStrghtPer2($1, $3, $5, $7, $9); }
	;

tolSymStm :
	  tLabel equalSign TOL SLASH SYM C rentVal C tolFeatureMat endOfLine
	    { $$ = new tolSymStm($1, $7, $9, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH SYM C rentVal C tolFeatureMatC tolZoneDir3 endOfLine
	    { $$ = new tolSymStm($1, $7, $9, $10);
	      dmisStms.push_back($$);
	    }
	;

tolTrnoutStm :
	  tLabel equalSign TOL SLASH TRNOUT C rentVal C datLabel endOfLine
	    { $$ = new tolTrnoutStm($1, $7, $9, 0, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH TRNOUT C rentVal C datLabel C tolFeature endOfLine
	    { $$ = new tolTrnoutStm($1, $7, $9, $11, 0);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH TRNOUT C rentVal C datLabel C tolFeature C tolFeature endOfLine
	    { $$ = new tolTrnoutStm($1, $7, $9, $11, $13);
	      dmisStms.push_back($$);
	    }
	;

tolUsetolStm :
	  tLabel equalSign TOL SLASH USETOL C stringVal endOfLine
	    { $$ = new tolUsetolStm($1, $7);
	      dmisStms.push_back($$);
	    }
	;

tolWidthStm :
	  tLabel equalSign TOL SLASH WIDTH C rentVal C rentVal endOfLine
	    { $$ = new tolWidthStm($1, $7, $9, 0, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH WIDTH C rentVal C rentVal C MINMAX endOfLine
	    { $$ = new tolWidthStm($1, $7, $9, 0, true);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH WIDTH C rentVal C rentVal C tolWidthDir endOfLine
	    { $$ = new tolWidthStm($1, $7, $9, $11, false);
	      dmisStms.push_back($$);
	    }
	| tLabel equalSign TOL SLASH WIDTH C rentVal C rentVal C tolWidthDir C MINMAX endOfLine
	    { $$ = new tolWidthStm($1, $7, $9, $11, true);
	      dmisStms.push_back($$);
	    }
	;

tolWidthDir :
	  vector
	    { $$ = new tolWidthDir_vector($1); }
	| SHORT
	    { $$ = new tolWidthDir_SHORT(); }
	| LONG
	    { $$ = new tolWidthDir_LONG(); }
	;

tolFeature :
	  fLabel
	    { $$ = $1; }
	| faLabel
	    { $$ = $1; }
	| datLabel
	    { $$ = $1; }
	;

tolMatCond :
	  MMC
	    { $$ = new tolMatCond_MMC(); }
	| LMC
	    { $$ = new tolMatCond_LMC(); }
	| RFS
	    { $$ = new tolMatCond_RFS(); }
	;

tolZoneType :
	  TANGPL
	    { $$ = new tolZoneType_TANGPL(); }
	| PARPLN
	    { $$ = new tolZoneType_PARPLN(); }
	;

tooldfStm :
	  tlLabel equalSign TOOLDF SLASH mdLabel C stringVal endOfLine
	    { $$ = new tooldfStm($1, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

transStm :
	  dLabel equalSign TRANS SLASH transMinor endOfLine
	    { $$ = new transStm($1, $5, 0, 0);
	      dmisStms.push_back($$);
	    }
	| dLabel equalSign TRANS SLASH transMinor C transMinor endOfLine
	    { $$ = new transStm($1, $5, $7, 0);
	      dmisStms.push_back($$);
	    }
	| dLabel equalSign TRANS SLASH transMinor C transMinor C transMinor endOfLine
	    { $$ = new transStm($1, $5, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

transMinor :
	  transValue
	    { $$ = $1; }
	| transFeature
	    { $$ = $1; }
	| transDatum
	    { $$ = $1; }
	| transProbe
	    { $$ = $1; }
	;

transValue :
	  orig C rentVal
	    { $$ = new transValue($1, $3); }
	;

transFeature :
	  orig C featureLabel
	    { $$ = new transFeature($1, $3); }
	;

transDatum :
	  orig C datLabel
	    { $$ = new transDatum($1, $3); }
	;

transProbe :
	  orig C MINUSPRBRAD
	    { $$ = new transProbe_MINUSPRBRAD($1); }
	| orig C PRBRAD
	    { $$ = new transProbe_PRBRAD($1); }
	;

uncertalgStm :
	  uLabel equalSign UNCERTALG SLASH uncertMinor endOfLine
	    { $$ = new uncertalgStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

uncertMinor :
	  ALGOR C intVal
	    { $$ = new uncertMinor_ALGOR($3); }
	| stringVal C uncertVar3
	    { $$ = new uncertMinor_stringVal($1, $3); }
	;

uncertVar3 :
	  DME
	    { $$ = new uncertVar3_DME(0); }
	| DME C paramList
	    { $$ = new uncertVar3_DME($3); }
	| SENS
	    { $$ = new uncertVar3_SENS(0); }
	| SENS C paramList
	    { $$ = new uncertVar3_SENS($3); }
	| paramList
	    { $$ = new uncertVar3_paramList($1); }
	;

uncertsetStm :
	  UNCERTSET SLASH uncertsetMinor endOfLine
	    { $$ = new uncertsetStm($3);
	      dmisStms.push_back($$);
	    }
	;

uncertsetMinor :
	  ON C uLabel C uncertsetVar2
	    { $$ = new uncertsetMinor_ON($3, $5); }
	| OFF
	    { $$ = new uncertsetMinor_OFF(); }
	;

uncertsetVar2 :
	  drLabel
	    { $$ = new uncertsetVar2_drLabel($1); }
	| NONE
	    { $$ = new uncertsetVar2_NONE(); }
	;

unitsStm :
	  UNITS SLASH lengthUnit C angleUnit endOfLine
	    { $$ = new unitsStm($3, $5, 0);
	      dmisStms.push_back($$);
	    }
	| UNITS SLASH lengthUnit C angleUnit C tempUnit endOfLine
	    { $$ = new unitsStm($3, $5, $7);
	      dmisStms.push_back($$);
	    }
	;

lengthUnit :
	  MM
	    { $$ = new lengthUnit_MM(); }
	| CM
	    { $$ = new lengthUnit_CM(); }
	| METER
	    { $$ = new lengthUnit_METER(); }
	| INCH
	    { $$ = new lengthUnit_INCH(); }
	| FEET
	    { $$ = new lengthUnit_FEET(); }
	;

angleUnit :
	  ANGDEC
	    { $$ = new angleUnit_ANGDEC(); }
	| ANGDMS
	    { $$ = new angleUnit_ANGDMS(); }
	| ANGRAD
	    { $$ = new angleUnit_ANGRAD(); }
	;

tempUnit :
	  TEMPF
	    { $$ = new tempUnit_TEMPF(); }
	| TEMPC
	    { $$ = new tempUnit_TEMPC(); }
	;

valueStm :
	  boolVar EQUALS VALUE SLASH valueMinorBool endOfLine
	    { $$ = new valueStm_boolVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| stringVar EQUALS VALUE SLASH valueMinorString endOfLine
	    { $$ = new valueStm_stringVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| intVar EQUALS VALUE SLASH valueMinorInt endOfLine
	    { $$ = new valueStm_intVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| realVar EQUALS VALUE SLASH valueMinorReal endOfLine
	    { $$ = new valueStm_realVar($1, $5);
	      dmisStms.push_back($$);
	    }
	| vectorVar EQUALS VALUE SLASH valueMinorVector endOfLine
	    { $$ = new valueStm_vectorVar($1, $5);
	      dmisStms.push_back($$);
	    }
	;

valueMinorBool :
	  valueBadtst
	    { $$ = $1; }
	| valueCroscl
	    { $$ = $1; }
	| valueCzslct
	    { $$ = $1; }
	| valueFinpos
	    { $$ = $1; }
	| valuePrcomp
	    { $$ = $1; }
	| valuePtbuff
	    { $$ = $1; }
	| valueScnmod
	    { $$ = $1; }
	| valueTaBool
	    { $$ = $1; }
	| valueTecomp
	    { $$ = $1; }
	;

valueMinorString :
	  valueAclratString
	    { $$ = $1; }
	| valueBoundString
	    { $$ = $1; }
	| valueCrmode
	    { $$ = $1; }
	| valueCrslct
	    { $$ = $1; }
	| valueDatset
	    { $$ = $1; }
	| valueDmismd
	    { $$ = $1; }
	| valueDmismn
	    { $$ = $1; }
	| valueError
	    { $$ = $1; }
	| valueFedratString
	    { $$ = $1; }
	| valueFilnam
	    { $$ = $1; }
	| valueGeoalg
	    { $$ = $1; }
	| valueKeychar
	    { $$ = $1; }
	| valueMode
	    { $$ = $1; }
	| valueSnsetString
	    { $$ = $1; }
	| valueSnslct
	    { $$ = $1; }
	| valueUnits
	    { $$ = $1; }
	| valueWkplan
	    { $$ = $1; }
	;

valueMinorInt :
	  valueBoundInt
	    { $$ = $1; }
	| valueFaInt
	    { $$ = $1; }
	;

valueMinorReal :
	  valueAclratReal
	    { $$ = $1; }
	| valueDeflection
	    { $$ = $1; }
	| valueFedratReal
	    { $$ = $1; }
	| valueGotoReal
	    { $$ = $1; }
	| valuePtmeasReal
	    { $$ = $1; }
	| valueSnsetReal
	    { $$ = $1; }
	| valueFaReal
	    { $$ = $1; }
	| valueRt
	    { $$ = $1; }
	| valueSa
	    { $$ = $1; }
	| valueSw
	    { $$ = $1; }
	| valueTaReal
	    { $$ = $1; }
	;

valueMinorVector :
	  valueGotoVector
	    { $$ = $1; }
	| valuePtmeasVector
	    { $$ = $1; }
	| valueSnsmnt
	    { $$ = $1; }
	;

valueAclratReal :
	  ACLRAT C MESACL
	    { $$ = new valueAclratReal_MESACL(); }
	| ACLRAT C POSACL
	    { $$ = new valueAclratReal_POSACL(); }
	| ACLRAT C ROTACL
	    { $$ = new valueAclratReal_ROTACL(); }
	;

valueAclratString :
	  valueAclratReal C ACEL
	    { $$ = new valueAclratString($1); }
	;

valueBadtst :
	  BADTST
	    { $$ = new valueBadtst(); }
	;

valueBoundString :
	  BOUND C featureLabel C intVal
	    { $$ = new valueBoundString_featureLabel($3, $5); }
	| BOUND C toleranceLabel C intVal
	    { $$ = new valueBoundString_toleranceLabel($3, $5); }
	;

valueBoundInt :
	  BOUND C featureLabel C COUNT
	    { $$ = new valueBoundInt_featureLabel($3); }
	| BOUND C toleranceLabel C COUNT
	    { $$ = new valueBoundInt_toleranceLabel($3); }
	;

valueCrmode :
	  CRMODE
	    { $$ = new valueCrmode(); }
	;

valueCroscl :
	  CROSCL
	    { $$ = new valueCroscl(); }
	;

valueCrslct :
	  CRSLCT
	    { $$ = new valueCrslct(); }
	;

valueCzslct :
	  CZSLCT C czLabel
	    { $$ = new valueCzslct($3); }
	;

valueDatset :
	  DATSET
	    { $$ = new valueDatset(); }
	;

valueDeflection :
	  DEFLECTION
	    { $$ = new valueDeflection(); }
	;

valueDmismd :
	  DMISMD C NAME
	    { $$ = new valueDmismd_NAME(); }
	| DMISMD C VERSION
	    { $$ = new valueDmismd_VERSION(); }
	;

valueDmismn :
	  DMISMN C NAME
	    { $$ = new valueDmismn_NAME(); }
	| DMISMN C VERSION
	    { $$ = new valueDmismn_VERSION(); }
	;

valueError :
	  ERROR C ERR
	    { $$ = new valueError_ERR(); }
	| ERROR C ERRMODE
	    { $$ = new valueError_ERRMODE(); }
	;

valueFedratReal :
	  FEDRAT C MESVEL
	    { $$ = new valueFedratReal_MESVEL(); }
	| FEDRAT C POSVEL
	    { $$ = new valueFedratReal_POSVEL(); }
	| FEDRAT C ROTVEL
	    { $$ = new valueFedratReal_ROTVEL(); }
	| FEDRAT C SCNVEL
	    { $$ = new valueFedratReal_SCNVEL(); }
	;

valueFedratString :
	  valueFedratReal C FEED
	    { $$ = new valueFedratString($1); }
	;

valueFilnam :
	  FILNAM C NAME
	    { $$ = new valueFilnam_NAME(); }
	| FILNAM C VERSION
	    { $$ = new valueFilnam_VERSION(); }
	;

valueFinpos :
	  FINPOS
	    { $$ = new valueFinpos(); }
	;

valueGeoalg :
	  GEOALG C measGeotype
	    { $$ = new valueGeoalg($3); }
	;

valueGotoReal :
	  GOTO C axis
	    { $$ = new valueGotoReal($3); }
	;

valueGotoVector :
	  GOTO C POS
	    { $$ = new valueGotoVector(); }
	;

valueKeychar :
	  kcLabel
	    { $$ = $1; }
	;

valueMode :
	  MODE
	    { $$ = new valueMode(); }
	;

valuePrcomp :
	  PRCOMP
	    { $$ = new valuePrcomp(); }
	;

valuePtbuff :
	  PTBUFF
	    { $$ = new valuePtbuff(); }
	;

valuePtmeasReal :
	  PTMEAS C axis
	    { $$ = new valuePtmeasReal($3); }
	;

valuePtmeasVector :
	  PTMEAS C POS
	    { $$ = new valuePtmeasVector(); }
	;

valueScnmod :
	  SCNMOD
	    { $$ = new valueScnmod(); }
	;

valueSnsetReal :
	  SNSET C APPRCH
	    { $$ = new valueSnsetRealApprch(); }
	| SNSET C RETRCT
	    { $$ = new valueSnsetRealRetrct(); }
	| SNSET C SEARCH
	    { $$ = new valueSnsetRealSearch(); }
	| SNSET C CLRSRF C DIST
	    { $$ = new valueSnsetRealClrsrf(); }
	| SNSET C DEPTH C DIST
	    { $$ = new valueSnsetRealDepth(); }
	;

valueSnsetString :
	  SNSET C CLRSRF
	    { $$ = new valueSnsetString_CLRSRF(); }
	| SNSET C DEPTH
	    { $$ = new valueSnsetString_DEPTH(); }
	;

valueSnslct :
	  SNSLCT
	    { $$ = new valueSnslct(); }
	;

valueSnsmnt :
	  SNSMNT C XVEC
	    { $$ = new valueSnsmnt_XVEC(); }
	| SNSMNT C ZVEC
	    { $$ = new valueSnsmnt_ZVEC(); }
	| SNSMNT C MNTLEN
	    { $$ = new valueSnsmnt_MNTLEN(); }
	;

valueTecomp :
	  TECOMP
	    { $$ = new valueTecomp(); }
	;

valueUnits :
	  UNITS C DIST
	    { $$ = new valueUnits_DIST(); }
	| UNITS C ANGL
	    { $$ = new valueUnits_ANGL(); }
	| UNITS C TEMP
	    { $$ = new valueUnits_TEMP(); }
	;

valueWkplan :
	  WKPLAN
	    { $$ = new valueWkplan(); }
	;

valueFaInt :
	  faLabel C PTDATA
	    { $$ = new valueFaInt($1); }
	;

valueFaReal :
	  faLabel C SIZE
	    { $$ = new valueFaReal($1, 0); }
	| faLabel C SIZE C intVal
	    { $$ = new valueFaReal($1, $5); }
	;

valueRt :
	  rtLabel C ANGL C CW
	    { $$ = new valueRt_CW($1); }
	| rtLabel C ANGL C CCW
	    { $$ = new valueRt_CCW($1); }
	;

valueSa :
	  saLabel
	    { $$ = new valueSaPlain($1); }
	| saLabel C stringVal
	    { $$ = new valueSaDesc($1, $3); }
	| saLabel C intVal
	    { $$ = new valueSaTipnum($1, $3); }
	;

valueSw :
	  swLabel C ANGLE C stringVal
	    { $$ = new valueSw($1, $5); }
	;

valueTaBool :
	  taLabel C INTOL
	    { $$ = new valueTaBoolIntol($1, 0); }
	| taLabel C INTOL C intVal
	    { $$ = new valueTaBoolIntol($1, $5); }
	| taLabel C OUTOL
	    { $$ = new valueTaBoolOutol($1, 0); }
	| taLabel C OUTOL C intVal
	    { $$ = new valueTaBoolOutol($1, $5); }
	;

valueTaReal :
	  taLabel C ACT
	    { $$ = new valueTaRealAct($1, 0); }
	| taLabel C ACT C intVal
	    { $$ = new valueTaRealAct($1, $5); }
	| taLabel C DEV
	    { $$ = new valueTaRealDev($1, 0); }
	| taLabel C DEV C intVal
	    { $$ = new valueTaRealDev($1, $5); }
	| taLabel C AMT
	    { $$ = new valueTaRealAmt($1, 0); }
	| taLabel C AMT C intVal
	    { $$ = new valueTaRealAmt($1, $5); }
	;

vformStm :
	  vLabel equalSign VFORM SLASH vformItemList endOfLine
	    { $$ = new vformStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

vformItemList :
	  NOM
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_NOM(0)); }
	| NOM C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_NOM(0)); }
	| NOM C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_NOM($3)); }
	| NOM C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_NOM($3)); }
	| ACT
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_ACT(0)); }
	| ACT C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_ACT(0)); }
	| ACT C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_ACT($3)); }
	| ACT C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_ACT($3)); }
	| DEV
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_DEV(0)); }
	| DEV C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_DEV(0)); }
	| DEV C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_DEV($3)); }
	| DEV C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_DEV($3)); }
	| AMT
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_AMT(0)); }
	| AMT C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_AMT(0)); }
	| AMT C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_AMT($3)); }
	| AMT C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_AMT($3)); }
	| HIST
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_HIST(0)); }
	| HIST C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_HIST(0)); }
	| HIST C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_HIST($3)); }
	| HIST C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_HIST($3)); }
	| PLOT
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_PLOT(0)); }
	| PLOT C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_PLOT(0)); }
	| PLOT C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_PLOT($3)); }
	| PLOT C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_PLOT($3)); }
	| STAT
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_STAT(0)); }
	| STAT C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_STAT(0)); }
	| STAT C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_STAT($3)); }
	| STAT C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_STAT($3)); }
	| ALL
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_ALL()); }
	| ALL C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_ALL()); }
	| DME
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_DME(0)); }
	| DME C vformItemList
	    { $$ = $3;
	      $$->push_front(new vformItem_DME(0)); }
	| DME C stringVal
	    { $$ = new std::list<vformItem *>;
	      $$->push_front(new vformItem_DME($3)); }
	| DME C stringVal C vformItemList
	    { $$ = $5;
	      $$->push_front(new vformItem_DME($3)); }
	;

windefStm :
	  vwLabel equalSign WINDEF SLASH windefMinor endOfLine
	    { $$ = new windefStm($1, $5);
	      dmisStms.push_back($$);
	    }
	;

windefMinor :
	  windefEdgeln
	    { $$ = $1; }
	| windefBox
	    { $$ = $1; }
	;

windefEdgeln :
	  EDGELN C impCartPoint C angle C rentVal
	    { $$ = new windefEdgeln($3, $5, $7); }
	;

windefBox :
	  BOX C impCartPoint C rentVal C rentVal C angle
	    { $$ = new windefBox($3, $5, $7, $9); }
	;

wkplanStm :
	  WKPLAN SLASH plan endOfLine
	    { $$ = new wkplanStm($3);
	      dmisStms.push_back($$);
	    }
	;

wristStm :
	  swLabel equalSign WRIST SLASH wristList C wristMountOffset endOfLine
	    { $$ = new wristStm($1, $5, $7, 0);
	      dmisStms.push_back($$);
	    }
	| swLabel equalSign WRIST SLASH wristList C wristMountOffset C wristDataDesc endOfLine
	    { $$ = new wristStm($1, $5, $7, $9);
	      dmisStms.push_back($$);
	    }
	;

wristList :
	  wristDef
	    { $$ = new std::list<wristDef *>;
	      $$->push_back($1); }
	| wristList C wristDef
	    { $$ = $1;
	      $$->push_back($3); }
	;

wristDef :
	  wristMountDef C wristRotDef
	    { $$ = new wristDef($1, $3); }
	;

wristMountDef :
	  ROTCEN C vector C vector C vector
	    { $$ = new wristMountDef($3, $5, $7); }
	;

wristRotDef :
	  ANGLE C stringVal C wristRotLimit C wristRotStep
	    { $$ = new wristRotDef($3, $5, $7); }
	;

wristRotLimit :
	  angle C angle
	    { $$ = new wristRotLimit_angle($1, $3); }
	| THRU
	    { $$ = new wristRotLimit_THRU(); }
	;

wristRotStep :
	  angle
	    { $$ = new wristRotStep_angle($1); }
	| CONTIN
	    { $$ = new wristRotStep_CONTIN(); }
	;

wristMountOffset :
	  MNTLEN C vector
	    { $$ = new wristMountOffset($3); }
	;

wristDataDesc :
	  stringVal C stringVal C stringVal
	    { $$ = new wristDataDesc_stringVal($1, $3, $5); }
	| stringVal C stringVal C intVal
	    { $$ = new wristDataDesc_intVal($1, $3, $5); }
	;

writeStm :
	  WRITE SLASH didLabel C writeSpecList endOfLine
	    { $$ = new writeStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

writeSpecList :
	  writeSpec
	    { $$ = new std::list<writeSpec *>;
	      $$->push_back($1); }
	| writeSpecList C writeSpec
	    { $$ = $1;
	      $$->push_back($3); }
	;

writeSpec :
	  stringConst
	    { $$ = new writeSpec_stringConst($1); }
	| stringFunc
	    { $$ = new writeSpec_stringFunc($1); }
	| rwVar
	    { $$ = new writeSpec_rwVar($1, 0); }
	| rwVar rwFormat
	    { $$ = new writeSpec_rwVar($1, $2); }
	;

xternStm :
	  XTERN endOfLine
	    { $$ = new xternStm();
	      dmisStms.push_back($$);
	    }
	;

xtractStm :
	  XTRACT SLASH fLabel C faLabel endOfLine
	    { $$ = new xtractStm($3, $5);
	      dmisStms.push_back($$);
	    }
	;

%%
