/* Process model C form file: mac_802_15_4.pr.c */
/* Portions of this file copyright 1992-2006 by OPNET Technologies, Inc. */



/* This variable carries the header into the object file */
const char mac_802_15_4_pr_c [] = "MIL_3_Tfile_Hdr_ 120A 30A op_runsim 7 45228ADA 45228ADA 1 P614876 nchevrol 0 0 none none 0 0 none 0 0 0 0 0 0 0 0 fec 0                                                                                                                                                                                                                                                                                                                                                                                                        ";
#include <string.h>



/* OPNET system definitions */
#include <opnet.h>



/* Header Block */

/*
** $File : LR-WPAN Mac layer header
**
** EPON model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      O. Rebala
** Secondary Author:	N. Chevrollier
** Module description:  header file for Low Rate WPAN model
** Last Modification:   October, 1st 2003
**
*/

/* header include */
#include "lr_wpan_support.h"
#include "channel_buffer.h"
#include "lr_wpan_stat_write.h"
#include <math.h>
/* include for the TCP/IP stack */
#include "oms_pr.h"
#include "oms_auto_addr_support.h"

/* Constant definitions */
#define HIGHER_LAYER_DOWN_STRM 0
#define HIGHER_LAYER_UP_STRM 0
#define LOWER_LAYER_UP_STRM 1
#define MAC_SUBQ	0
#define POWER_TRANS 0.001

/* macro definition */
#ifndef min
#define min(a,b)	(a < b)?a:b
#endif

/* debugging Macro */
#define ENTER_STATE_ODB_PRINTING(a) if (STATE_PRINT_STATUS)\
	printf ("%s MAC layer (%d): %.9f s: %s\n",\
	lr_wpan_node_name, my_parameters->mac_address, op_sim_time (), a)

#define FLAG_ODB_PRINTING		if (FLAG_PRINT_STATUS)\
	printf ("%s MAC layer (%d): flag status:\n\t%s\n\t%s\n\t%s\n\t%s\n",\
	lr_wpan_node_name, my_parameters->mac_address,\
	(channel_flag.tx_idle)?"TX is idle":"TX is busy",\
	(channel_flag.rx_idle)?"RX is idle":"RX is busy",\
	(Rx_Ack_Expected)?"Acknowledge expected":"No expected Ack",\
	(Tx_Ack_Require)?"Ack has to be sent":"No Ack pending")

/* structure definition */

/* structure to create the MAC subqueues system */
typedef struct t_Wpan_Queue {
	List * subqueue;
	int mac_address;
	int size;
	int nb_element;
	struct t_Wpan_Queue * next;
} Wpan_Queue;

/* define the codes for the interruptions */
typedef enum {
	Backoff_Time_Out,
	Data_Pkt_To_Send,
	Ack_Time_Out,
	Ack_Received,
	Wpan_Ack_Tx_Required,
	End_Of_Transmission,
	CCA_Pkt_Detect,
	Wpan_CCA_Time_Out,
	Wpan_Rcv_Pkt,
	Wpan_Rx_TAT,
	Wpan_Tx_TAT,
	Wpan_LIFS_Over,
	Wpan_Beacon_To_Send,
	Wpan_Data_Req_To_Send
} Wpan_Intrpt_Code;

/* define some code for TX and RX */
typedef enum {
	RX_CODE,
	TX_CODE,
	TXRX_CODE
} TxRx_Management;

/* define a statistic vector */
typedef struct {
	int Data_Pkt_Rcv;
	int Segment_generated;
	int Data_Pkt_Sent;
	int Data_Req_Send;
	int Ack_Pkt_Rcv;
	int Ack_Pkt_Sent;
	int Pkt_Lost;
	int Pkt_Collided;
	int Pkt_Noise;
	int Pkt_Dropped;
	int Segment_Dropped;
	int Retransmission;
	int Nb_Time_Out;
	double total_backoff_time;
	int	total_backoff;
	double Total_MAC_Delay;
	int tcp_HL_pkt_sent;
	int tcp_HL_bit_sent;
	int tcp_HL_pkt_rcvd;
	int tcp_HL_bit_rcvd;
} Wpan_Stat_Vector;

/* define the channel flag */
typedef struct {
	Boolean tx_idle;
	Boolean rx_idle;
	Boolean LIFS_on;
	Boolean data_is_sending;
} Wpan_Chan_Flag;

/* define the Data Rx Seqn */
typedef struct {
	int mac_address;
	int seqn;
} Mac_Data_Seqn;

/* define the backoff parameters */
typedef struct {
	int max_backoff; // maximum number of Backoff
	int nb_backoff; // current number of backoff
	int min_BE; // minimum backoff exponent
	int max_BE; // maximum backoff exponent
	int Be; // Backoff exponent
} Mac_CSMA_Param;


/* state machine conditions */
#define		SLOTTED_CONDITION		((my_parameters->master_enable && lr_wpan_beacon_param.beacon_rcvd) ||\
									 (!my_parameters->master_enable && lr_wpan_beacon_param.data_requested))
#define		UNSLOTTED_CONDITION		(my_parameters->slotted_enable == OPC_FALSE)
#define		READY_TO_SEND			(!channel_flag.LIFS_on && wpan_transmission_required)
#define		PACKET_TO_SEND			(INTRPT_SELF && (READY_TO_SEND || intrpt_code == Wpan_Data_Req_To_Send))
#define 	BACKOFF_EXPIRED			(INTRPT_SELF && intrpt_code == Backoff_Time_Out)
#define		CCA_EXPIRED				(INTRPT_SELF && intrpt_code == Wpan_CCA_Time_Out)
#define		TRANSMISSION_OK			(channel_is_idle && !SECOND_CCA_REQ)
#define		BACK_TO_BACKOFF			(CCA_EXPIRED && !channel_is_idle && csma_parameters.nb_backoff <= csma_parameters.max_backoff)
#define		END_OF_CCA				(CCA_EXPIRED && (channel_is_idle || csma_parameters.nb_backoff > csma_parameters.max_backoff))
#define		SECOND_CCA_REQ			(my_parameters->slotted_enable && lr_wpan_beacon_param.second_cca_req)
#define		BACK_TO_CCA				(channel_is_idle && SECOND_CCA_REQ)
#define		BACK_TO_IDLE			(INTRPT_SELF && (intrpt_code == Ack_Time_Out || intrpt_code == Ack_Received))
#define		END_OF_TRANSMISSION		(channel_flag.data_is_sending && INTRPT_SELF && intrpt_code == End_Of_Transmission)
#define		ACKNOWLEDGEMENT			(END_OF_TRANSMISSION && Rx_Ack_Expected)
#define		NO_ACK					(END_OF_TRANSMISSION && !Rx_Ack_Expected)
#define		TIME_TO_TRANSMIT		(!Tx_Ack_Require && !Rx_Ack_Expected && channel_flag.tx_idle && !END_OF_TRANSMISSION)
#define		BOUNDARY_REACHED		(INTRPT_SELF && intrpt_code == Wpan_Tx_TAT)


/* function prototypes */
static void 		lr_wpan_mac_init (void);
static void			lr_wpan_model_wide_register (void);
static void 		lr_wpan_mac_address_resolve (void);
static void			lr_wpan_schedule_TAT (TxRx_Management code);
static void			lr_wpan_cancel_TAT (TxRx_Management code);
static void			lr_wpan_cca_init (void);
static void 		lr_wpan_mac_backoff (void);
static void 		lr_wpan_intrpt_check (const char * state_name);
static void			lr_wpan_generate_beacon (void);
static void			lr_wpan_generate_data_request (void);
static void 		higher_layer_pkt_recv (void);
static void 		lower_layer_pkt_rcv (void);
static void			unacknowledgement_reset (void);
static void 		lr_wpan_packet_to_transmit (void);
static void			lr_wpan_ack_to_transmit (void);
static void 		lr_wpan_debug_state_msg (const char * state_name, Boolean exit);
static void			lr_wpan_drop_higher_layer_pkt (void);
static int			lr_wpan_destroy_retransmission (void);
static void 		lr_wpan_print_queue (void);
static int			lr_wpan_get_beacon_size (Addressing_Field * address);
static void 		lr_wpan_enqueue_packet (int mac_address, Packet * pkptr);
static Wpan_Queue * lr_wpan_allocate_subqueue (int mac_address, Packet * pkptr);
//static void		lr_wpan_subq_allocation (void);
static void			lr_wpan_subq_checking (void);
static int 			lr_wpan_get_rx_seqn (int mac_address);
static void 		lr_wpan_set_rx_seqn (int mac_address, int seqn);
static void 		lr_wpan_collect_stat (void);

/* End of Header Block */

#if !defined (VOSD_NO_FIN)
#undef	BIN
#undef	BOUT
#define	BIN		FIN_LOCAL_FIELD(_op_last_line_passed) = __LINE__ - _op_block_origin;
#define	BOUT	BIN
#define	BINIT	FIN_LOCAL_FIELD(_op_last_line_passed) = 0; _op_block_origin = __LINE__;
#else
#define	BINIT
#endif /* #if !defined (VOSD_NO_FIN) */



/* State variable definitions */
typedef struct
	{
	/* Internal state tracking for FSM */
	FSM_SYS_STATE
	/* State Variables */
	int	                    		intrpt_type                                     ;	/* type of current interruption */
	int	                    		intrpt_stream                                   ;	/* stream of current interruption */
	int	                    		intrpt_code                                     ;	/* code of current interruption */
	Mac_CSMA_Param	         		csma_parameters                                 ;	/* CSMA CA parameters */
	Boolean	                		channel_is_idle                                 ;	/* flag to perform the CCA */
	Wpan_Node_Param *	      		my_parameters                                   ;	/* Parameters of the node */
	double	                 		wpan_frequency_center                           ;	/* Frequency used by the device */
	char *	                 		lr_wpan_node_name                               ;	/* Name of the WPAN node */
	Wpan_Queue *	           		wpan_queue                                      ;	/* MAC queue to store the packet from the higher layer */
	Stathandle	             		nb_pkt_subq                                     ;	/* Number of packet in the subqueue */
	Stathandle	             		nb_pkt_recv                                     ;	/* Number of packet received */
	Stathandle	             		nb_pkt_send                                     ;	/* Number of packet send */
	Packet *	               		retransmission_ptr                              ;	/* Copy of the current packet to transmit for retransmission */
	List *	                 		Mac_Data_Rx_Seqn                                ;	/* Data sequence number (DSN) defined in the IEEE 802.15.4 */
	Boolean	                		Rx_Ack_Expected                                 ;	/* flag to know whether we are waiting for an acknowledgment */
	Boolean	                		Tx_Ack_Require                                  ;	/* flag to know whether we have to send an acknowledgment */
	Evhandle	               		Ack_TimeOut_Evhandle                            ;	/* Handle for the interruption of the acknowledgment time out */
	int	                    		Mac_Data_Tx_Seqn                                ;	/* Data sequence number (DSN) defined in the IEEE 802.15.4 */
	int	                    		Ack_Dest_Addr                                   ;	/* Destination address of the acknowledgment */
	Wpan_Chan_Flag	         		channel_flag                                    ;	/* Flag for the TX/RX channel */
	Wpan_Stat_Vector	       		statistic                                       ;	/* Vector of all the collected statistics */
	Boolean	                		bulk_data_source                                ;	/* Flag to know whether the source of packet is a simple source or not */
	Evhandle	               		TAT_TX_Evhandle                                 ;	/* Handle for the interruption of the turn around time of the TX */
	Evhandle	               		TAT_RX_Evhandle                                 ;	/* Handle for the interruption of the turn around time of the RX */
	Boolean	                		wpan_transmission_required                      ;	/* Indicate that some packets need to be transmitted */
	Wpan_Queue *	           		current_queue                                   ;	/* mark the queue using to send the packets */
	int	                    		max_frame_retries                               ;	/* Number of maximum transmission retries */
	int	                    		nb_transmission_retries                         ;	/* Number of current transmission retries */
	OmsT_Pr_Handle	         		own_process_record_handle                       ;	/* Used to record the MAC process in the model wide registry */
	OmsT_Aa_Address_Handle	 		oms_aa_handle                                   ;	/* Used to resolve the MAC address */
	Boolean	                		Acknowledgement_Enable                          ;	/* Used to make the acknowledgement or not in the transmission */
	Wpan_Beacon_Param	      		lr_wpan_beacon_param                            ;	/* Beacon Parameters */
	int	                    		Mac_Beacon_Tx_Seqn                              ;	/* Beacon sequence number (BSN) defined in the IEEE 802.15.4 */
	} mac_802_15_4_state;

#define intrpt_type             		op_sv_ptr->intrpt_type
#define intrpt_stream           		op_sv_ptr->intrpt_stream
#define intrpt_code             		op_sv_ptr->intrpt_code
#define csma_parameters         		op_sv_ptr->csma_parameters
#define channel_is_idle         		op_sv_ptr->channel_is_idle
#define my_parameters           		op_sv_ptr->my_parameters
#define wpan_frequency_center   		op_sv_ptr->wpan_frequency_center
#define lr_wpan_node_name       		op_sv_ptr->lr_wpan_node_name
#define wpan_queue              		op_sv_ptr->wpan_queue
#define nb_pkt_subq             		op_sv_ptr->nb_pkt_subq
#define nb_pkt_recv             		op_sv_ptr->nb_pkt_recv
#define nb_pkt_send             		op_sv_ptr->nb_pkt_send
#define retransmission_ptr      		op_sv_ptr->retransmission_ptr
#define Mac_Data_Rx_Seqn        		op_sv_ptr->Mac_Data_Rx_Seqn
#define Rx_Ack_Expected         		op_sv_ptr->Rx_Ack_Expected
#define Tx_Ack_Require          		op_sv_ptr->Tx_Ack_Require
#define Ack_TimeOut_Evhandle    		op_sv_ptr->Ack_TimeOut_Evhandle
#define Mac_Data_Tx_Seqn        		op_sv_ptr->Mac_Data_Tx_Seqn
#define Ack_Dest_Addr           		op_sv_ptr->Ack_Dest_Addr
#define channel_flag            		op_sv_ptr->channel_flag
#define statistic               		op_sv_ptr->statistic
#define bulk_data_source        		op_sv_ptr->bulk_data_source
#define TAT_TX_Evhandle         		op_sv_ptr->TAT_TX_Evhandle
#define TAT_RX_Evhandle         		op_sv_ptr->TAT_RX_Evhandle
#define wpan_transmission_required		op_sv_ptr->wpan_transmission_required
#define current_queue           		op_sv_ptr->current_queue
#define max_frame_retries       		op_sv_ptr->max_frame_retries
#define nb_transmission_retries 		op_sv_ptr->nb_transmission_retries
#define own_process_record_handle		op_sv_ptr->own_process_record_handle
#define oms_aa_handle           		op_sv_ptr->oms_aa_handle
#define Acknowledgement_Enable  		op_sv_ptr->Acknowledgement_Enable
#define lr_wpan_beacon_param    		op_sv_ptr->lr_wpan_beacon_param
#define Mac_Beacon_Tx_Seqn      		op_sv_ptr->Mac_Beacon_Tx_Seqn

/* These macro definitions will define a local variable called	*/
/* "op_sv_ptr" in each function containing a FIN statement.	*/
/* This variable points to the state variable data structure,	*/
/* and can be used from a C debugger to display their values.	*/
#undef FIN_PREAMBLE_DEC
#undef FIN_PREAMBLE_CODE
#define FIN_PREAMBLE_DEC	mac_802_15_4_state *op_sv_ptr;
#define FIN_PREAMBLE_CODE	\
		op_sv_ptr = ((mac_802_15_4_state *)(OP_SIM_CONTEXT_PTR->_op_mod_state_ptr));


/* Function Block */

#if !defined (VOSD_NO_FIN)
enum { _op_block_origin = __LINE__ + 2};
#endif

/*
 * Function:	lr_wpan_init
 *
 * Description:	initialize the process
 *				read the attributes and set the global variables
 *
 * No parameters
 */

static void lr_wpan_mac_init ()
{
	Objid csma_parameters_comp_id;
	Objid csma_parameters_id;
	int mac_address, index_wpan;
	int channel_number;
	char buffer[64];
	char proc_model_name[64];
	
	char error_message[256];
	Wpan_Node_Param * element;
	
	
	FIN (lr_wpan_mac_init ());

	/* memory allocation */
	my_parameters	= (Wpan_Node_Param *) op_prg_mem_alloc (sizeof (Wpan_Node_Param));
	
	/* initialize the wpan queue to NULL */
	wpan_queue = NULL;
	current_queue = NULL;
	
	/* set the type of the node */
	strcpy (my_parameters->type, "wpan");
	
	/* get the ID of this module */
	my_parameters->objid = op_id_self ();

	/* get the ID of the node */
	my_parameters->parent_id = op_topo_parent (my_parameters->objid);

	/* get the channel number for the 2.4 GHz band */
	op_ima_obj_attr_get (my_parameters->objid, "Channel", &channel_number);
	
	/* get the power of the transmission */
	op_ima_obj_attr_get (my_parameters->objid, "Power", &my_parameters->power);
	
	/* compute the center frequency for the channel selected */
	wpan_frequency_center = WPAN_FREQUENCY_CENTER (channel_number);
	my_parameters->frequency_wpan = wpan_frequency_center;
	
	/* get the type of source */
	op_ima_obj_attr_get (my_parameters->objid, "Source Type", &bulk_data_source);
	
	/* get the Acknowledgemnent attribute */
	op_ima_obj_attr_get (my_parameters->objid, "Acknowledgement", &Acknowledgement_Enable);
	
	/* get the Channel Access attribute */
	op_ima_obj_attr_get (my_parameters->objid, "Channel Access", &my_parameters->slotted_enable);
	op_ima_obj_attr_get (my_parameters->objid, "Mode", &my_parameters->master_enable);
	
	/* get the CSMA-CA settings */
	op_ima_obj_attr_get (my_parameters->objid, "CSMA-CA Parameters", &csma_parameters_id);
	csma_parameters_comp_id = op_topo_child (csma_parameters_id, OPC_OBJTYPE_GENERIC, 0);
	
	op_ima_obj_attr_get (csma_parameters_comp_id, "Maximum backoff number", &csma_parameters.max_backoff);
	op_ima_obj_attr_get (csma_parameters_comp_id, "Minimum backoff exponent", &csma_parameters.min_BE);
	op_ima_obj_attr_get (csma_parameters_comp_id, "Maximum backoff exponent", &csma_parameters.max_BE);	
	
	/* get the maximum transmission retries */
	op_ima_obj_attr_get (my_parameters->objid, "Nb of transmission retries", &max_frame_retries);
	
	/* get the channel number for the 2.4 GHz band */
	op_ima_obj_attr_get (my_parameters->objid, "Group number", &my_parameters->group);
	
	/* get the position of the node */
	op_ima_obj_attr_get (my_parameters->parent_id, "x position", &my_parameters->x);
	op_ima_obj_attr_get (my_parameters->parent_id, "y position", &my_parameters->y);

	/* get the name of the node */
	op_ima_obj_attr_get_str (my_parameters->parent_id, "name", 64, buffer);
	
	lr_wpan_node_name = (char *) op_prg_mem_alloc ((strlen (buffer) + 1) * sizeof (char));
	strcpy (lr_wpan_node_name, buffer);
		
	/* get the name of the process model */
	op_ima_obj_attr_get (my_parameters->objid, "process model", proc_model_name);
	
	/* get the MAC address of the node */
	op_ima_obj_attr_get (my_parameters->objid, "Address", &mac_address);
	
	if (bulk_data_source)
		{
		/* check the MAC address */
		if (mac_address == -2)
			{
			/* if the mac address is "auto assigned" */
			mac_address = (int) my_parameters->objid;
			}
		else if (wpan_node_param_list != OPC_NIL)
			{
			/* 
			* if the MAC address was set by the user, we have to
			* check whether this address doesn't exist
			*/
			
			if ((index_wpan = wpan_search_mac_address (mac_address)) != -1)
				{
				/* get the element with the same MAC address */
				element = op_prg_list_access (wpan_node_param_list, index_wpan);
				
				/* get the name of this node */
				op_ima_obj_attr_get_str (op_topo_parent (element->objid), "name", 64, buffer);
				
				/* edit the error message */
				sprintf (error_message, "Check the MAC addresses of the nodes %s and %s", lr_wpan_node_name, buffer);
				lr_wpan_mac_error ("lr_wpan_init:", "The MAC address already exist.", error_message);
				}
			}
		}
	
	/* set the MAC address */
	my_parameters->mac_address = mac_address;
	
	/* initialize the CCA requirement */
	my_parameters->cca_requirement = OPC_FALSE;
	
	/* Set the WLAN field to NULL */
	my_parameters->wlan_info = NULL;
		
	/* initialize the node list if it is not done */
	if (wpan_node_param_list == OPC_NIL)
		wpan_node_param_list = op_prg_list_create ();
	
	/* register the node into the list */
	op_prg_list_insert (wpan_node_param_list, my_parameters, OPC_LISTPOS_TAIL);
	
	/* set the backoff parameters */
	csma_parameters.Be = csma_parameters.min_BE;
	csma_parameters.nb_backoff = 0;
	
	
	/* statistic initializations */
	nb_pkt_subq = op_stat_reg ("subqueues.Number of packets", OPC_STAT_INDEX_NONE, OPC_STAT_LOCAL);
	nb_pkt_send = op_stat_reg ("mac.Packet sent", OPC_STAT_INDEX_NONE, OPC_STAT_LOCAL);
	nb_pkt_recv = op_stat_reg ("mac.Packet received", OPC_STAT_INDEX_NONE, OPC_STAT_LOCAL);
	
	/* initialize variables */
	retransmission_ptr 				= OPC_NIL;
	Rx_Ack_Expected 				= OPC_FALSE;
	Tx_Ack_Require 					= OPC_FALSE;
	Mac_Data_Tx_Seqn 				= rand () % MAX_MAC_SEQN;
	Mac_Beacon_Tx_Seqn				= 0;
	Mac_Data_Rx_Seqn 				= op_prg_list_create ();
	channel_flag.tx_idle 			= OPC_TRUE;
	channel_flag.rx_idle 			= OPC_TRUE;
	channel_flag.LIFS_on 			= OPC_FALSE;
	channel_flag.data_is_sending	= OPC_FALSE;
	wpan_transmission_required 		= OPC_FALSE;
	nb_transmission_retries 		= 0;
	
	/* beacon variables */
	lr_wpan_beacon_param.stop_time		= 0.0;
	lr_wpan_beacon_param.nb_addresses 	= 0;
	lr_wpan_beacon_param.addr_table 	= NULL;
	lr_wpan_beacon_param.beacon_rcvd	= OPC_FALSE;
	lr_wpan_beacon_param.data_req_rcvd	= OPC_FALSE;
	lr_wpan_beacon_param.second_cca_req = OPC_FALSE;
	lr_wpan_beacon_param.master_addr 	= -1;
	lr_wpan_beacon_param.data_request 	= OPC_NIL;
	
	/* initialize the statistics */
	statistic.Data_Pkt_Rcv 		= 0;
	statistic.Segment_generated = 0;
	statistic.Data_Pkt_Sent 	= 0;
	statistic.Ack_Pkt_Rcv 		= 0;
	statistic.Ack_Pkt_Sent 		= 0;
	statistic.Pkt_Lost 			= 0;
	statistic.Pkt_Collided		= 0;
	statistic.Pkt_Noise			= 0;
	statistic.Pkt_Dropped		= 0;
	statistic.Segment_Dropped	= 0;
	statistic.Retransmission 	= 0;
	statistic.Nb_Time_Out 		= 0;
	statistic.Total_MAC_Delay 	= 0.0;
	statistic.tcp_HL_pkt_sent	= 0;
	statistic.tcp_HL_bit_sent	= 0;
	statistic.tcp_HL_pkt_rcvd	= 0;
	statistic.tcp_HL_bit_rcvd	= 0;
	statistic.Data_Req_Send 	= 0;
	statistic.total_backoff		= 0;
	statistic.total_backoff_time= 0.0;
	
	/* print information about the node */
	printf ("+-----------------------------------------------\n");
	printf ("| MAC layer of the node %s:\n", lr_wpan_node_name);
	printf ("| \tposition: (%.1f, %.1f)\n", my_parameters->x, my_parameters->y);
	printf ("| \tGroup: %d; MAC address: %s%d\n", my_parameters->group, (my_parameters->mac_address == -2)?"Auto Assigned, ":"", my_parameters->mac_address);
	printf ("| \tChannel Access: %s (%s)\n", (my_parameters->slotted_enable) ? "Slotted" : "Unslotted", (my_parameters->master_enable) ? "Master" : "Slave");
	printf ("| \tFirst TX Seqn: %d\n", Mac_Data_Tx_Seqn);
	printf ("| \tAcknowledgement: %s\n", Acknowledgement_Enable ? "On" : "Off");
	printf ("| Status:\n");
	printf ("| \tMaximum number of backoff:\t%d\n", csma_parameters.max_backoff);
	printf ("| \tMinimum backoff exponent:\t%d\n", csma_parameters.min_BE);
	printf ("| \tMaximum backoff exponent:\t%d\n", csma_parameters.max_BE);
	printf ("+-----------------------------------------------\n");

	
	FOUT;
}


/*
 * Function:	lr_wpan_model_wide_register
 *
 * Description:	initialize the MAC layer with the TCP/IP
 *				stack if any
 */

static void	lr_wpan_model_wide_register (void)
{
	Prohandle own_prohandle;
	char proc_model_name[64];
	
	FIN (lr_wpan_tcp_ip_init ());
	
	/* Obtain the process prohandle */
	own_prohandle = op_pro_self ();
	
	/* Obatin the name of the process */
	op_ima_obj_attr_get (my_parameters->objid, "process model", proc_model_name);
	
	/* Register this MAC process in the model wide registry */
	own_process_record_handle = (OmsT_Pr_Handle)
		oms_pr_process_register (my_parameters->parent_id, my_parameters->objid, own_prohandle, proc_model_name);
	
	/*
	 * obtain  the address handle assigned to this node. The string
	 * "MAC addresses" rendez-vous with other MACs to guarantee
	 * unique addresses across all MAC types. The OMS_AA package
	 * handles this feature.
	 */
	oms_aa_handle = oms_aa_address_handle_get ("MAC Addresses", "Address");
	
	
	FOUT;
}


/*
 * Function:
 */

static void lr_wpan_mac_address_resolve (void)
{
	Objid subnet_objid;
	int mac_address;
	
	FIN (lr_wpan_mac_address_resolve ());
	
	mac_address = my_parameters->mac_address;
	
	/* Obtain the subnet identifier */
	subnet_objid = op_topo_parent (my_parameters->parent_id);
	
	/*
	 * Perform auto-addressing for the MAC address. Appart
	 * from dynamically addresses, if atau-assigned, the
	 * address resolution function also detects duplicate
	 * static assignments. By default, each address is
	 * considered as being a valid destination unless it is
	 * explicitly set.
	 */
	oms_aa_address_resolve (oms_aa_handle, my_parameters->objid, &mac_address);
	
	/* update the parameter of the MAC layer */
	my_parameters->mac_address = mac_address;
	
	/*
	 * Register station's MAC address into model-wide
	 * registery, since the address assignment is final now.
	 */
	oms_pr_attr_set (own_process_record_handle,
		"protocol",				OMSC_PR_STRING,			"mac",
		"mac_type",				OMSC_PR_STRING,			"wireless_pan",
		"subprotocol", 			OMSC_PR_NUMBER,			0.0,
		"subnetid",				OMSC_PR_OBJID,			subnet_objid,
		"address", 				OMSC_PR_NUMBER, 		(double) mac_address,
		"auto address handle",	OMSC_PR_ADDRESS,		oms_aa_handle,
		OPC_NIL);
	
	FOUT;
}

/*
 * Function:	lr_wpan_schedule_TAT
 *
 * Description:	schedule the next turn around time
 *
 * ParamIn:		
 */

static void lr_wpan_schedule_TAT (TxRx_Management code)
{
	FIN (lr_wpan_schedule_TAT (code));
	
	/* check if a Turn Around time was already schedule */
	lr_wpan_cancel_TAT (code);
	
	if (code == TX_CODE)
		TAT_TX_Evhandle = op_intrpt_schedule_self (op_sim_time () + WPAN_TURN_AROUND_TIME, Wpan_Tx_TAT);
	
	if (code == RX_CODE)
		TAT_RX_Evhandle = op_intrpt_schedule_self (op_sim_time () + WPAN_TURN_AROUND_TIME, Wpan_Rx_TAT);
	
	FOUT;
}

/*
 */
static void lr_wpan_cancel_TAT (TxRx_Management code)
{
	FIN (lr_wpan_cancel_TAT (code));
	
	/* check the channel according to the Tx RX management code */
	if ((code == RX_CODE || code == TXRX_CODE) && op_ev_valid (TAT_RX_Evhandle) == OPC_TRUE &&
		op_ev_id (TAT_RX_Evhandle) != op_ev_id (op_ev_current ()))
		op_ev_cancel (TAT_RX_Evhandle);
	
	if ((code == TX_CODE || code == TXRX_CODE) && op_ev_valid (TAT_TX_Evhandle) == OPC_TRUE &&
		op_ev_id (TAT_TX_Evhandle) != op_ev_id (op_ev_current ()))
		op_ev_cancel (TAT_TX_Evhandle);
	
	FOUT;
}


/*
 * Function:	lr_wpan_cca_init
 */

static void lr_wpan_cca_init (void)
{
	Buff_Info_Packet * info_packet;
	Boolean 			temp;
	
	FIN (lr_wpan_cca_init ());
	
	/* Update the number of backoff */
	csma_parameters.nb_backoff ++;

	/* update the backoff exponent */
	csma_parameters.Be = min (csma_parameters.Be + 1, csma_parameters.max_BE);
	
	/* perform the CCA */
	my_parameters->cca_requirement = OPC_TRUE;
	
	/* set the time out for the CCA */
	op_intrpt_schedule_self (op_sim_time () + WPAN_CCA_PERIOD, Wpan_CCA_Time_Out);
	
	/* set TX to busy */
	channel_flag.tx_idle = OPC_FALSE;
	
	/* cancel any Turn Around time scheduled for the TX */
	lr_wpan_cancel_TAT (TX_CODE);
	
	/* check if the channel is idle */
	if (wpan_cca_all_pkt_types)
		channel_is_idle = (channel_buffer_ptr == NULL);
	else
		{
		/*
		 * in the case we only check the WPAN packets, we search for a WPAN packet
		 * in the channel buffer and if we find one, the channel is busy
		 */
		temp = OPC_TRUE;
		
		info_packet = channel_buffer_ptr;
				
		while (info_packet != NULL)
			{
			if (info_packet->packet_type == WPAN_PKT_TYPE && info_packet->frequency == my_parameters->frequency_wpan )
				{
				temp = OPC_FALSE;
				info_packet = NULL;		
				}
			else
				{
				info_packet = info_packet->next_packet;
				}
			}
		/* check the channel: if the info_packet pointer is null, the channel will be idle */
		channel_is_idle = temp;	
		}
		
	FOUT;
}


/*
 * Function:	lr_wpan_mac_backoff
 *
 * description:	
 *
 * No parameters
 */

static void lr_wpan_mac_backoff ()
{
	int		cw;
	int		backoff_unit;
	double 	backoff_time;
	
	
	FIN (lr_wpan_mac_backoff ());
		
	/* compute the backoff time period */
	cw = (int) (01<<csma_parameters.Be) - 1; // compute 2^be - 1
	backoff_unit = floor (op_dist_uniform (cw));
	
	backoff_time = (double) backoff_unit * LR_WPAN_BACKOFF_UNIT; // multiply by the unit time for the backoff
	// compute average backoff per packet
	if (csma_parameters.nb_backoff == 0)
		{
		statistic.total_backoff ++;
		}
	
	statistic.total_backoff_time += backoff_time;

	/* set the backoff timer */
	if (my_parameters->slotted_enable)
		{
		/* if we are in a slotted mode, we start the backoff at the boundary */
		op_intrpt_schedule_self (lr_wpan_slotted_boundary_time () + backoff_time, Backoff_Time_Out);
		}
	else
		{
		/* if we are in unslotted mode, we start the backoff at the curent time */
		op_intrpt_schedule_self (op_sim_time () + backoff_time, Backoff_Time_Out);
		}
	 
	
	FOUT;
}


static void lr_wpan_intrpt_check (const char * state_name)
{
	char odb_msg[128]; // debugging message
	
	
	FIN (lr_wpan_intrpt_check ());
		
	/* compute the debugging message */
	sprintf (odb_msg, "exit the \"%s\" state: ", state_name);
	FLAG_ODB_PRINTING;
	
	/* get the type of the interuption */
	switch (intrpt_type = op_intrpt_type ())
		{
		case OPC_INTRPT_SELF:
		case OPC_INTRPT_REMOTE:
		switch (intrpt_code = op_intrpt_code ())
			{
			case Backoff_Time_Out:
			strcat (odb_msg, "backoff expired");
			break;
				
			case Data_Pkt_To_Send:
			strcat (odb_msg, "transmission required");
			
			/* set the flag */
			wpan_transmission_required = OPC_TRUE;
			break;
			
			case Ack_Time_Out:
			strcat (odb_msg, "Ack period expired");
			
			/* if the time is out, we reset the acknowledgment flag */
			Rx_Ack_Expected = OPC_FALSE;
			
			/* update the statistic */
			(statistic.Nb_Time_Out)++;
			break;
			
			case Ack_Received:
			strcat (odb_msg, "Received an Ack");
			
			/* turn on the LIFS period */
			channel_flag.LIFS_on = OPC_TRUE;
	
			/* schedule when the LIFS will be over */
			op_intrpt_schedule_self (op_sim_time () + WPAN_LIFS_DURATION, Wpan_LIFS_Over);
			
			/* canceled the Ack Time out if it is still schedule */
			if (op_ev_valid (Ack_TimeOut_Evhandle) == OPC_TRUE &&
				op_ev_id (Ack_TimeOut_Evhandle) != op_ev_id (op_ev_current()))
				op_ev_cancel (Ack_TimeOut_Evhandle);
			break;
			
			case Wpan_Ack_Tx_Required:
			strcat (odb_msg, "Transmit an Ack");
			
			
			break;
			
			case End_Of_Transmission:
			strcat (odb_msg, "End of the transmission");
			
			/* set the TX to idle */
			channel_flag.tx_idle = OPC_TRUE;
						
			/* schedule the release of RX (Turn Around Time) */
			lr_wpan_schedule_TAT (RX_CODE);
			break;
			
			case CCA_Pkt_Detect:
			strcat (odb_msg, "Packet detecting during the CCA");
			
			/* set the channel to busy */
			channel_is_idle = OPC_FALSE;
			break;
			
			case Wpan_CCA_Time_Out:
			strcat (odb_msg, "CCA is over");
			
			/* schedule the release of TX (Turn Around Time) */
			lr_wpan_schedule_TAT (TX_CODE);
			
			/* end the CCA */
			my_parameters->cca_requirement = OPC_FALSE;
			
			/*
			 * if we are in a slotted mode, we need to switch the flag 
			 * for the second cca
			 */
			if (my_parameters->slotted_enable)
				lr_wpan_beacon_param.second_cca_req = (lr_wpan_beacon_param.second_cca_req) ? OPC_FALSE : OPC_TRUE;
			break;
			
			case Wpan_Rcv_Pkt:
			strcat (odb_msg, "We're receiving a packet");
			
			/* turn off the TX */
			channel_flag.tx_idle = OPC_FALSE;
			
			/* if a turn around time is schedule, we cancel it */
			lr_wpan_cancel_TAT (TX_CODE);
			break;
			
			case Wpan_Rx_TAT:
			strcat (odb_msg, "End of the Turn Around Time, Rx is idle");
			
			/* set the RX to idle */
			channel_flag.rx_idle    = OPC_TRUE;
			break;
			
			case Wpan_Tx_TAT:
			strcat (odb_msg, "End of the Turn Around Time, Tx is idle");
			
			/* set the TX to idle */
			channel_flag.tx_idle = OPC_TRUE;
			
			/* we check whether we need to send an Ack */
			if (Tx_Ack_Require)
				{
				/* transmit an acknowledgment */
				lr_wpan_ack_to_transmit ();
				
				/* complete the ODB message */
				strcat (odb_msg, "; send an acknoledge");
				
				/* turn off the flag */
				Tx_Ack_Require = OPC_FALSE;
				}
			break;
			
			case Wpan_LIFS_Over:
			strcat (odb_msg, "End of the LIFS period, Tx is idle");
			
			/* turn off the LIFS Period */
			channel_flag.LIFS_on = OPC_FALSE;
			break;
			
			case Wpan_Beacon_To_Send:
			/* we generate and send a beacon on broadcast */\
			lr_wpan_generate_beacon ();
			break;
			
			case Wpan_Data_Req_To_Send:
			/* we generate a data request */
			lr_wpan_generate_data_request ();	
			break;
			
			default:
			strcat (odb_msg, "unknown code");
			break;
			}
		break;
			
		case OPC_INTRPT_STRM:
		switch (intrpt_stream = op_intrpt_strm ())
			{
			case HIGHER_LAYER_DOWN_STRM:
			/* we received a packet from the higher layer */
			higher_layer_pkt_recv ();
				
			strcat (odb_msg, "packet from the higher layer");
			break;
				
			case LOWER_LAYER_UP_STRM:
			strcat (odb_msg, "packet from the physical layer: reception completed");
			
			/* we received a packet from the channel */
			lower_layer_pkt_rcv ();
			
			/* schedule the release of TX (Turn Around Time) */
			lr_wpan_schedule_TAT (TX_CODE);
			break;
				
			default:
			strcat (odb_msg, "unknown stream");
			break;
			}
		break;
		
			
		case OPC_INTRPT_ENDSIM:
		lr_wpan_collect_stat ();
		break;
		
		default:
		strcat (odb_msg, "unknown intrpt");
		break;
		}
	
	/* print the debugging message */
	ENTER_STATE_ODB_PRINTING (odb_msg);
	FLAG_ODB_PRINTING;
	
	/* define a break point */
	op_prg_odb_bkpt ("wpan_state");
	FOUT;
}


/*
 * Function:	lr_wpan_generate_beacon
 *
 * Description:	create a Beacon packet and send it in
 *				Broadcast to the devices in the same
 *				group
 *
 * No parameters
 */

static void 
lr_wpan_generate_beacon (void)
{
	Packet * beacon_ptr;
	Addressing_Field * address;
	int i; // loop variables
	int beacon_size, rx_index;
	double txrx_distance;
	Wpan_Node_Param * rxptr;
	Ici * ici_ptr;
	
	
	FIN (lr_wpan_generate_beacon ());
	
	
	/* debugging */
	ENTER_STATE_ODB_PRINTING ("Generate the beacon packet.");
	
	/* Check if the MAC address table has been created */
	if (lr_wpan_beacon_param.nb_addresses == 0)
		lr_wpan_mac_error ("lr_wpan_generate_beacon:", "No connexion found.", OPC_NIL);
	
	/* for each device in the group, we create a beacon packet */
	for (i=0; i<lr_wpan_beacon_param.nb_addresses; i++)
		{
		/* init the pointer */
		address = NULL;
		beacon_ptr = OPC_NIL;
		
		/* create a Beacon packet */
		beacon_ptr = op_pk_create_fmt ("lr_wpan_beacon");
	
		/* create the address field and the frame Ctrl */
		address = (Addressing_Field *) op_prg_mem_alloc (sizeof (Addressing_Field));
		
		/* complete the address field */
		address->src_addr = my_parameters->mac_address;
		address->dest_addr = lr_wpan_beacon_param.addr_table[i];
		
		/* compute the beacon packet size */
		beacon_size = lr_wpan_get_beacon_size (address);

		/* set the fields of the Beacon packet */
		if (op_pk_nfd_set (beacon_ptr, "Addressing field", address, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Addressing_Field)) == OPC_COMPCODE_FAILURE ||
			op_pk_nfd_set_dbl (beacon_ptr, "Sequence number", Mac_Beacon_Tx_Seqn) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("lr_wpan_generate_beacon:", "Unable to set the Beacon packet.", OPC_NIL);
	
		/* set the size of the packet */
		op_pk_total_size_set (beacon_ptr, beacon_size);
		
		/* search the mac address in the list of node to get the parameters of the destination node */
		if ((rx_index = wpan_search_mac_address (address->dest_addr)) == -1)
			lr_wpan_mac_error ("lr_wpan_generate_beacon:", "The MAC destination address doesn't exist.", OPC_NIL);
	
		/* get the parameter of the destination device */
		rxptr = (Wpan_Node_Param *) op_prg_list_access (wpan_node_param_list, rx_index);

		/* compute the distance between the RX and TX */
		txrx_distance = sqrt ((rxptr->x - my_parameters->x)*(rxptr->x - my_parameters->x) + (rxptr->y - my_parameters->y)*(rxptr->y - my_parameters->y));
		
		/* initialize the ICI structure */
		ici_ptr = OPC_NIL;
		
		/* create ICI pointer */
		ici_ptr = op_ici_create ("ici_channel");
			
		/* set the ICI pointer */
		if (op_ici_attr_set_dbl (ici_ptr, "txrx distance", txrx_distance) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "bit rate", LR_WPAN_BIT_RATE) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "channel", wpan_frequency_center) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "power", my_parameters->power) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_int32 (ici_ptr, "packet type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_int32 (ici_ptr, "dest address", address->dest_addr) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Tx x", my_parameters->x) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Tx y", my_parameters->y) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Rx x", rxptr->x) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Rx y", rxptr->y) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("lr_wpan_generate_beacon:", "Unable to set the ICI pointer.", OPC_NIL);
		
		/* Associate the ICI pointer with the beacon packet */
		op_pk_ici_set (beacon_ptr, ici_ptr);
		
		/* send the packet to the channel */
		op_pk_deliver (beacon_ptr, wpan_channel_objid, CHANNEL_UP_STREAM);
		}
	
	/* update the sequence number */
	Mac_Beacon_Tx_Seqn++;
	Mac_Beacon_Tx_Seqn %= MAX_MAC_SEQN;
	
	/* schedule the next Beacon packet */
	//op_intrpt_schedule_self (op_sim_time () + LR_WPAN_BEACON_INTERVAL_TIME, Wpan_Beacon_To_Send);
	
	FOUT;
}
	

/*
 * Function:	lr_wpan_generate_data_request
 *
 * Description:	create a data request packet and send it
 *				to the master device in the same group
 *
 * ParamIn:		int dest_address
 *				MAC destination address of the master
 */

static void
lr_wpan_generate_data_request (void)
{
	Addressing_Field * address;
	Frame_Ctrl_Field * ctrl_field_ptr;
	int rx_index;
	double txrx_distance;
	Wpan_Node_Param * rxptr;
	Ici * ici_ptr;
	
	
	FIN (lr_wpan_generate_beacon ());
		
	/* debugging */
	ENTER_STATE_ODB_PRINTING ("Create a data request packet.");
	
	/* init the pointer */
	address = NULL;
	ctrl_field_ptr = NULL;
		
	/* create a data request packet */
	lr_wpan_beacon_param.data_request = op_pk_create_fmt ("lr_wpan_mac");
	
	/* create the address field and the frame Ctrl */
	address = (Addressing_Field *) op_prg_mem_alloc (sizeof (Addressing_Field));
	ctrl_field_ptr = (Frame_Ctrl_Field *) op_prg_mem_alloc (sizeof (Frame_Ctrl_Field));
		
	/* complete the address field */
	address->src_addr = my_parameters->mac_address;
	address->dest_addr = lr_wpan_beacon_param.master_addr;
	
	/* complete the frame control field */
	ctrl_field_ptr->ack_req = Acknowledgement_Enable;
	
	/* set the fields of the Beacon packet */
	if (op_pk_nfd_set (lr_wpan_beacon_param.data_request, "Addressing field", address, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Addressing_Field)) == OPC_COMPCODE_FAILURE ||
		op_pk_nfd_set (lr_wpan_beacon_param.data_request, "Frame Ctrl", ctrl_field_ptr, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Frame_Ctrl_Field)) == OPC_COMPCODE_FAILURE ||
		op_pk_nfd_set_int32 (lr_wpan_beacon_param.data_request, "Request", 1) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("lr_wpan_generate_data_request:", "Unable to set the data request packet.", OPC_NIL);
		
	/* set the size of the packet */
	op_pk_total_size_set (lr_wpan_beacon_param.data_request, WPAN_DATA_REQ_SIZE);
	
	/* search the mac address in the list of node to get the parameters of the destination node */
	if ((rx_index = wpan_search_mac_address (address->dest_addr)) == -1)
		lr_wpan_mac_error ("lr_wpan_generate_data_request:", "The MAC destination address doesn't exist.", OPC_NIL);
	
	/* get the parameter of the destination device */
	rxptr = (Wpan_Node_Param *) op_prg_list_access (wpan_node_param_list, rx_index);
	
	/* compute the distance between the RX and TX */
	txrx_distance = sqrt ((rxptr->x - my_parameters->x)*(rxptr->x - my_parameters->x) + (rxptr->y - my_parameters->y)*(rxptr->y - my_parameters->y));
	
	/* initialize the ICI structure */
	ici_ptr = OPC_NIL;
	
	/* create ICI pointer */
	ici_ptr = op_ici_create ("ici_channel");
			
	/* set the ICI pointer */
	if (op_ici_attr_set_dbl (ici_ptr, "txrx distance", txrx_distance) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "bit rate", LR_WPAN_BIT_RATE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "channel", wpan_frequency_center) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "power", my_parameters->power) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "packet type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "dest address", address->dest_addr) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx x", my_parameters->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx y", my_parameters->y) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx x", rxptr->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx y", rxptr->y) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("lr_wpan_generate_beacon:", "Unable to set the ICI pointer.", OPC_NIL);
		
	/* Associate the ICI pointer with the beacon packet */
	op_pk_ici_set (lr_wpan_beacon_param.data_request, ici_ptr);
		
	
	FOUT;
}


/*
 * Function:	higher_layer_pkt_recv
 *
 * Description:	receive the packet from the higher
 *				layer and encapsulated the packet
 *				into a MAC packet
 *
 * No parameter
 */

static void 
higher_layer_pkt_recv (void)
{
	Packet * pkptr;
	Packet * higher_layer_pkptr;
	int higher_layer_pksize;
	int last_pksize, pk_number;
	Addressing_Field * address, * address_copy;
	Frame_Ctrl_Field * ctrl_field_ptr, * ctrl_field_copy;
	Wpan_Node_Param * rxptr;
	Ici * ici_ptr;
	int rx_index;
	double txrx_distance;
	int dest_mac_address;
	int i; // loop variable
	
	
	FIN (higher_layer_pkt_recv ());
	
	/* get the packet from the stream */
	higher_layer_pkptr = op_pk_get (intrpt_stream);
	
	/* get the address of the packet */
	if (bulk_data_source)
		{
		/* if the packet source is a simple source, we just get the destination address from the packet */
		if (op_pk_nfd_get_int32 (higher_layer_pkptr, "Address", &dest_mac_address) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to read the packet address.", OPC_NIL);
		}
	else
		{
		/* if we have the TCP/IP stack, we get the address in the ICI attached with the packet */
		/* Read ICI parameters at the stream interrupt.					*/
		ici_ptr = op_intrpt_ici ();

		/* Retrieve destination address from the ici set by the interface layer.	*/
		if (ici_ptr == OPC_NIL || op_ici_attr_get (ici_ptr, "dest_addr", &dest_mac_address) == OPC_COMPCODE_FAILURE)
			{
			/* Generate error message.	*/
			lr_wpan_mac_error ("higher_layer_pkt_recv:", "Destination address in not valid.", OPC_NIL);
			}
		
		/* update the statistics */
		statistic.tcp_HL_pkt_sent ++;
		statistic.tcp_HL_bit_sent += op_pk_total_size_get (higher_layer_pkptr);
		}
	
	/* create address field */
	address = (Addressing_Field *) op_prg_mem_alloc (sizeof (Addressing_Field));
	
	/* complete the address field */
	address->src_addr = my_parameters->mac_address;
	address->dest_addr = dest_mac_address;
	
	/* Create a frame control field */
	ctrl_field_ptr = (Frame_Ctrl_Field *) op_prg_mem_alloc (sizeof (Frame_Ctrl_Field));
	
	/* complete the frame control field */
	ctrl_field_ptr->ack_req = Acknowledgement_Enable;
	
	if ((rx_index = wpan_search_mac_address (dest_mac_address)) == -1)
		lr_wpan_mac_error ("higher_layer_pkt_recv:", "The MAC destination address doesn't exist.", OPC_NIL);
	
	/* get the parameter of the destination device */
	rxptr = (Wpan_Node_Param *) op_prg_list_access (wpan_node_param_list, rx_index);

	/* compute the distance between the RX and TX */
	txrx_distance = sqrt ((rxptr->x - my_parameters->x)*(rxptr->x - my_parameters->x) + (rxptr->y - my_parameters->y)*(rxptr->y - my_parameters->y));

	/* compute the size of the higher layer packet */
	higher_layer_pksize = (int) op_pk_total_size_get (higher_layer_pkptr);
	/* start the segmentation */
	if (higher_layer_pksize > WPAN_PAYLOAD_MAX_SIZE)
		{
		/* dimension of the last packet */
		last_pksize = higher_layer_pksize%WPAN_PAYLOAD_MAX_SIZE;
		
		/* compute the number of packet */
		pk_number = (higher_layer_pksize - last_pksize)/WPAN_PAYLOAD_MAX_SIZE;
		}
	else
		{
		pk_number = 0;
		last_pksize = higher_layer_pksize;
		}
	
	/* 
	 * if the last packet size is null, we decrease the number of packets by one
	 * to be able to create the last packet with the address of the real packet
	 */
	if (last_pksize == 0)
		{
		/* decrease the loop number */
		pk_number--;
		
		/* the last packet size will be equal to WPAN_PAYLOAD_MAX_SIZE */
		last_pksize = WPAN_PAYLOAD_MAX_SIZE;
		}
	
	for (i=0; i<pk_number; i++)
		{
		/* initialize the ICI structure */
		ici_ptr = OPC_NIL;
		
		/* create ICI pointer */
		ici_ptr = op_ici_create ("ici_channel");
			
		/* set the ICI pointer */
		if (op_ici_attr_set_dbl (ici_ptr, "txrx distance", txrx_distance) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "bit rate", LR_WPAN_BIT_RATE) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "channel", wpan_frequency_center) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "power", my_parameters->power) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_int32 (ici_ptr, "packet type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_int32 (ici_ptr, "dest address", dest_mac_address) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Tx x", my_parameters->x) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Tx y", my_parameters->y) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Rx x", rxptr->x) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (ici_ptr, "Rx y", rxptr->y) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to set the ICI pointer.", OPC_NIL);
		
		/* create a MAC packet */
		pkptr = op_pk_create_fmt ("lr_wpan_mac");
		
		/* copy the address field and the frame Ctrl */
		address_copy = (Addressing_Field *) op_prg_mem_copy_create (address, sizeof (Addressing_Field));
		ctrl_field_copy = (Frame_Ctrl_Field *) op_prg_mem_copy_create (ctrl_field_ptr, sizeof (Frame_Ctrl_Field));
		
		/* set the fields of the MAC packet */
		if (op_pk_nfd_set (pkptr, "Frame Ctrl", ctrl_field_copy, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Frame_Ctrl_Field)) == OPC_COMPCODE_FAILURE ||
			op_pk_nfd_set (pkptr, "Addressing field", address_copy, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Addressing_Field)) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to set the MAC packet.", OPC_NIL);
		
		/* set the size of the packet */
		op_pk_total_size_set (pkptr, WPAN_MAC_MAX_SIZE);
		
		/* Associate the ICI pointer with this packet */
		op_pk_ici_set (pkptr, ici_ptr);
				
		/* enqueue the packet */
		lr_wpan_enqueue_packet (dest_mac_address, pkptr);
		}
	
	/* we create the last packet with a pointer to the real higher layer packet */
	/* initialize the ICI structure */
	ici_ptr = OPC_NIL;
		
	/* create ICI pointer */
	ici_ptr = op_ici_create ("ici_channel");
	
	/* set the ICI pointer */
	if (op_ici_attr_set_dbl (ici_ptr, "txrx distance", txrx_distance) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "bit rate", LR_WPAN_BIT_RATE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "channel", wpan_frequency_center) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "power", my_parameters->power) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "packet type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "dest address", address->dest_addr) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx x", my_parameters->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx y", my_parameters->y) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx x", rxptr->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx y", rxptr->y) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to set the ICI pointer.", OPC_NIL);
		
	/* create a MAC packet */
	pkptr = op_pk_create_fmt ("lr_wpan_mac");
	
	/* set the fields of the MAC packet */
	if (op_pk_nfd_set_int32 (pkptr, "Segmentation", 1) == OPC_COMPCODE_FAILURE ||
		op_pk_nfd_set_pkt (pkptr, "Higher Layer Pkt", higher_layer_pkptr) == OPC_COMPCODE_FAILURE ||
		op_pk_nfd_set (pkptr, "Frame Ctrl", ctrl_field_ptr, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Frame_Ctrl_Field)) == OPC_COMPCODE_FAILURE ||
		op_pk_nfd_set (pkptr, "Addressing field", address, op_prg_mem_copy_create, op_prg_mem_free, sizeof (Addressing_Field)) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to set the MAC packet.", OPC_NIL);
	
	/* set the size of the packet */
	op_pk_total_size_set (pkptr, last_pksize + WPAN_HEADER_SIZE);
	
	/* Associate the ICI pointer with this packet */
	op_pk_ici_set (pkptr, ici_ptr);
		
	/* enqueue the packet */
	lr_wpan_enqueue_packet (dest_mac_address, pkptr);
	
	
	FOUT;
}



/*
 * Function:	lower_layer_pkt_rcv
 *
 * description:	Received all the packet from the
 *				lower layer
 *
 * No Parameter
 */

static void
lower_layer_pkt_rcv (void)
{
	Packet * pkptr, *higher_layer_pkptr = NULL;
	char format[32];
	Addressing_Field * addr_field;
	Frame_Ctrl_Field * frame_ctrl_ptr;
	int last_packet;
	int seqn;
	int accept;
	int noise;
	Ici * ici_ptr;
	//int wpan_bulk_data_id;
	
	
	FIN (lower_layer_pkt_rcv ());
	
	/* get the packet from the stream */
	pkptr = op_pk_get (intrpt_stream);
		
	/* get the accept flag */
	op_pk_nfd_get_int32 (pkptr, "Accept", &accept);
	
	/* get the noise flag */
	op_pk_nfd_get_int32 (pkptr, "Noise", &noise);
	//printf("noise flag is %d\n", noise);
	
	/* get the format of the packet */
	op_pk_format (pkptr, format);
	
	/* 
	* if this is a WPAN MAC data packet, check if a higher layer
	* is in there
	*/
	if (!strcmp (format, "lr_wpan_mac"))
		{
		/* get the higher layer packet if the segmentation flag is true */
		op_pk_nfd_get_int32 (pkptr, "Segmentation", &last_packet);
		
		if (last_packet)
			{
			/*
			* if this is the last segment received,
			* we get the higher layer packet
			*/
			op_pk_nfd_get_pkt (pkptr, "Higher Layer Pkt", &higher_layer_pkptr);
			}
		}
	
	/* if the RX flag is not idle or the packet is not accepted, we discard the packet */
	if (!channel_flag.rx_idle || !accept || noise)
		{
		/* debugging */
		if (!channel_flag.rx_idle)
			{
			ENTER_STATE_ODB_PRINTING ("packet is  collided or bad");
			(statistic.Pkt_Collided)++;
			}
		else if (noise)
			{
			ENTER_STATE_ODB_PRINTING ("packet is considered as noise");
			(statistic.Pkt_Noise)++;
			}
		else if (!accept)
			{
			ENTER_STATE_ODB_PRINTING ("packet is lost due to interference");
			(statistic.Pkt_Lost)++;	
			}
		
		op_prg_odb_bkpt ("collision");
   		/* check if a higher layer packet is set */
		if (higher_layer_pkptr != NULL)
			{
			/* destroy the higher layer packet */
			op_pk_destroy (higher_layer_pkptr);
			}
		
		/* destroy the packet */
		op_pk_destroy (pkptr);
		FOUT;
		}
	
	/*
	 * Check the type of the received packet:
	 *		* if it's a beacon packet, we schedule 
	 */
	if (!strcmp (format, "lr_wpan_beacon"))
		{
		/* debugging */
		ENTER_STATE_ODB_PRINTING ("Beacon Frame received.");
			
		/* update the flag */
		lr_wpan_beacon_param.beacon_rcvd = OPC_TRUE;
		
		/* get the address field */
		if (op_pk_nfd_get_ptr (pkptr, "Addressing field", &addr_field) == OPC_COMPCODE_FAILURE)
			lr_wpan_mac_error ("lower_layer_pkt_rcv:", "Unable to get the address field of the Beacon frame.", OPC_NIL);
		
		/* get the MAC address of the master */
		if (lr_wpan_beacon_param.master_addr == -1)
			lr_wpan_beacon_param.master_addr = addr_field->src_addr;
		
		if (addr_field->packet_pending != 0)
			{
			/* schedule a data request to send if some packets are pending */
			op_intrpt_schedule_self (lr_wpan_slotted_boundary_time (), Wpan_Data_Req_To_Send);
			}
		else if (wpan_transmission_required)
			{
			/*
			 * otherwise, if no packets are pending and if we need to send a data,
			 * we schedule the transmission
			 */
			op_intrpt_schedule_self (lr_wpan_slotted_boundary_time (), Data_Pkt_To_Send);
			}
		
		/* destroy the packet */
		op_pk_destroy (pkptr);			
		}
	else if (!Rx_Ack_Expected && !strcmp (format, "lr_wpan_mac"))
		{
		/* get the sequence number of the Data packet */
		op_pk_nfd_get_int32 (pkptr, "Seqn", &seqn);
		
		/* look if the packet is a data request */
		op_pk_nfd_get_int32 (pkptr, "Request", &lr_wpan_beacon_param.data_req_rcvd);
		
		/* get the address field of the packet */
		op_pk_nfd_get_ptr (pkptr, "Addressing field", &addr_field);
		
		/* get the frame control field */
		op_pk_nfd_get_ptr (pkptr, "Frame Ctrl", &frame_ctrl_ptr);
		
		/* get the source MAC address of the packet for the acknowledgment */
		Ack_Dest_Addr = addr_field->src_addr;
		
		/* schedule the acknowledgment transmission */
		Tx_Ack_Require = frame_ctrl_ptr->ack_req;
		
		/* if the packet sequence number is correct */
		if (seqn != lr_wpan_get_rx_seqn (addr_field->src_addr))
			{
			/* update the sequence number */
			lr_wpan_set_rx_seqn (addr_field->src_addr, seqn);
			
			/* update the statistic */
			statistic.Data_Pkt_Rcv ++;
			statistic.Total_MAC_Delay += op_sim_time () - op_pk_stamp_time_get (pkptr);
			
			/*
			* if we have a higher layer packet in there,
			* we forward it to the higher layer
			*/
			if (higher_layer_pkptr != NULL)
				{
				/* check if we have a simple source */
				if (bulk_data_source)
					{
					/* forward the packet to the higher layer */
					op_pk_send (higher_layer_pkptr, HIGHER_LAYER_UP_STRM);
					}
				else
					{
					/* create an ICI for the LLC layer */
					ici_ptr = op_ici_create ("wlan_mac_ind");
					
					/* set the ICI */
					if (op_ici_attr_set_int32 (ici_ptr, "dest_addr", my_parameters->mac_address) == OPC_COMPCODE_FAILURE ||
						op_ici_attr_set_int32 (ici_ptr, "src_addr", addr_field->src_addr) == OPC_COMPCODE_FAILURE)
						lr_wpan_mac_error ("lower_layer_pkt_rcv:", "Unable to set the LLC ICI pointer.", OPC_NIL);
					
					/* update the statistics */
					statistic.tcp_HL_pkt_rcvd ++;
					statistic.tcp_HL_bit_rcvd += op_pk_total_size_get (higher_layer_pkptr);
					
					/* install the ici */
					op_ici_install (ici_ptr);
					
					/* forward the packet to the higher layer */
					op_pk_send (higher_layer_pkptr, HIGHER_LAYER_UP_STRM);
					
					/* de-install the ici */
					op_ici_install (OPC_NIL);
					}
				}
			}
		else
			{
			/* debugging */
			ENTER_STATE_ODB_PRINTING ("Bad Sequence number.");
		
			if (higher_layer_pkptr != NULL)
				{
				/* discard the higher layer packet */
				op_pk_destroy (higher_layer_pkptr);
				}
			}
		
		/* free the memory */
		op_prg_mem_free (addr_field);
		op_prg_mem_free (frame_ctrl_ptr);
		
		/* destroy the packet */
		op_pk_destroy (pkptr);
		}
	else if (Rx_Ack_Expected && !strcmp (format, "lr_wpan_ack"))
		{
		/* destroy the packet */
		op_pk_destroy (pkptr);
		
		/* update the flag */
		Rx_Ack_Expected = OPC_FALSE;
		
		/* destroy the retransmission packet */
		lr_wpan_destroy_retransmission ();
		
		/* update the sequence number */
		Mac_Data_Tx_Seqn++;
		Mac_Data_Tx_Seqn %= MAX_MAC_SEQN;
		
		/* schedule the exit of the state */
		op_intrpt_schedule_self (op_sim_time (), Ack_Received);
		
		/* update the statistic */
		statistic.Ack_Pkt_Rcv ++;
		}
	else
		{
		/* debugging */
		ENTER_STATE_ODB_PRINTING ("WPAN Packet not accepted by the MAC layer.");
		
		/* check if a higher layer packet is set */
		if (higher_layer_pkptr != NULL)
			{
			/* discard the higher layer packet */
			op_pk_destroy (higher_layer_pkptr);
			}
		
		/* discard the packet */
		op_pk_destroy (pkptr);
		}
	
	FOUT;
}

/*
 * Function:	unacknowlegement_reset
 */

static void
unacknowlegement_reset (void)
{
/* 
 * if we are in unacknowledgement mode, we turn the LIFS on 
 * and we destroy the retransmission packet 
 */
	FIN (unacknowlegement_reset ());
	
	/* The transmission of the data is finished */
	channel_flag.data_is_sending = OPC_FALSE;
	
	/* turn on the LIFS period */
	channel_flag.LIFS_on = OPC_TRUE;
	
	/* schedule when the LIFS will be over */
	op_intrpt_schedule_self (op_sim_time () + WPAN_LIFS_DURATION, Wpan_LIFS_Over);
	
	/* destroy the retransmission packet */
	lr_wpan_destroy_retransmission ();
	
	/* update the sequence number */
	Mac_Data_Tx_Seqn++;
	Mac_Data_Tx_Seqn %= MAX_MAC_SEQN;
		
	FOUT;
}

	
/*
 * Function:	lr_wpan_packet_to_transmit
 *
 * Description:	set and transmit the MAC packet to the channel
 *
 * No parameter
 */

static void lr_wpan_packet_to_transmit (void)
{
	Packet * pkptr;
	
	
	FIN (lr_wpan_packet_to_transmit ());
	

	if (lr_wpan_beacon_param.data_request != OPC_NIL)
		{
		/* set the packet to send  */
		pkptr = lr_wpan_beacon_param.data_request;
		
		/* reset the pointer without destroying the packet to send */
		lr_wpan_beacon_param.data_request = OPC_NIL;
		}
	else if (retransmission_ptr == OPC_NIL)
		{
		/* debugging */
		ENTER_STATE_ODB_PRINTING ("transmit a packet");
		
		/* check if the subqueue is not empty */
		if (!current_queue->size)
			lr_wpan_mac_error ("lr_wpan_packet_to_transmit:", "The higher layer subqueue is empty.", OPC_NIL);
		
		/* dequeue the higher layer packet to transmit */
		if ((pkptr = op_prg_list_remove (current_queue->subqueue, OPC_LISTPOS_HEAD)) == OPC_NIL)
			lr_wpan_mac_error ("lr_wpan_packet_to_transmit:", "Unable to get the packet from the subqueue.", OPC_NIL);
		
		/* update the size of the list */
		current_queue->size = op_prg_list_size (current_queue->subqueue);
		
		/* Set the sequence number of the packet */
		op_pk_nfd_set_int32 (pkptr, "Seqn", Mac_Data_Tx_Seqn);
		
		/* get a copy of the packet in case of retransmission */
		retransmission_ptr = op_pk_copy (pkptr);
		}
	else
		{
		/* debugging */
		ENTER_STATE_ODB_PRINTING ("retransmit a packet");

		/* compute the retransmission packet */
		(statistic.Retransmission)++;
		
		/* set a break point for the retransmission */
		op_prg_odb_bkpt ("retransmission");
		
		/* get the packet to retransmit */
		pkptr = op_pk_copy (retransmission_ptr);
		}
	
	
	/* send the packet to the channel */
	op_pk_deliver (pkptr, wpan_channel_objid, CHANNEL_UP_STREAM);
	
	/* update the statistics */
	statistic.Data_Pkt_Sent ++;
	
	/* schedule the end of the transmission */
	op_intrpt_schedule_self (op_sim_time () + (op_pk_total_size_get (pkptr) + (double) LR_WPAN_PHY_OVERHEAD)/LR_WPAN_BIT_RATE, End_Of_Transmission);
	
	/* set the TX and RX to busy */
	channel_flag.tx_idle = OPC_FALSE;
	channel_flag.rx_idle = OPC_FALSE;
	
	/* if a turn around time is schedule, we cancel it */
	lr_wpan_cancel_TAT (TXRX_CODE);
	
	/* we expected to received an acknowledgment */
	Rx_Ack_Expected = Acknowledgement_Enable;
	
	/* We are sending a data packet */
	channel_flag.data_is_sending = OPC_TRUE;
	
	FOUT;
}

 
/*
 * Function:	lr_wpan_ack_to_transmit
 *
 * Description:	Transmit an Ack packet
 *
 * No Parameters
 */

static void lr_wpan_ack_to_transmit ()
{
	Packet * Ack_Pkptr;
	Ici * ici_ptr;
	double txrx_distance;
	int rx_index, rx_seqn;
	Wpan_Node_Param * rxptr;
	
	
	FIN (lr_wpan_ack_to_transmit ());
	
	/* create an Ack packet */
	Ack_Pkptr = op_pk_create_fmt ("lr_wpan_ack");
	
	/* get the sequence number */
	if ((rx_seqn = lr_wpan_get_rx_seqn (Ack_Dest_Addr)) == -1)
		lr_wpan_mac_error ("lr_wpan_ack_to_transmit:", "Unable to get the sequence number.", OPC_NIL);
	
	/* fill out the field of the packet */
	if (op_pk_nfd_set_int32 (Ack_Pkptr, "Seqn", rx_seqn) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("lr_wpan_ack_to_transmit:", "Unable to fill out the packet's fields.", OPC_NIL);
		
	if ((rx_index = wpan_search_mac_address (Ack_Dest_Addr)) == -1)
		lr_wpan_mac_error ("lr_wpan_ack_to_transmit:", "The MAC destination address doesn't exist.", OPC_NIL);
	
	/* get the parameter of the destination device */
	rxptr = op_prg_list_access (wpan_node_param_list, rx_index);

	/* compute the distance between the RX and TX */
	txrx_distance = sqrt ((rxptr->x - my_parameters->x)*(rxptr->x - my_parameters->x) + (rxptr->y - my_parameters->y)*(rxptr->y - my_parameters->y));

	/* create a new ICI */
	ici_ptr = op_ici_create ("ici_channel");
	
	/* set the ICI pointer */
	if (op_ici_attr_set_dbl (ici_ptr, "txrx distance", txrx_distance) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "bit rate", LR_WPAN_BIT_RATE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "channel", wpan_frequency_center) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "power", my_parameters->power) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "packet type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_int32 (ici_ptr, "dest address", Ack_Dest_Addr) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx x", my_parameters->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Tx y", my_parameters->y) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx x", rxptr->x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_set_dbl (ici_ptr, "Rx y", rxptr->y) == OPC_COMPCODE_FAILURE)
		lr_wpan_mac_error ("higher_layer_pkt_recv:", "Unable to set the ICI pointer.", OPC_NIL);
	
	/* Associate the ICI pointer with this packet */
	op_pk_ici_set (Ack_Pkptr, ici_ptr);
	
	
	/* send the packet to the channel */
	op_pk_deliver (Ack_Pkptr, wpan_channel_objid, CHANNEL_UP_STREAM);
	
	/* update the statistics */
	statistic.Ack_Pkt_Sent ++;
		
	/* schedule the end of the transmission */
	op_intrpt_schedule_self (op_sim_time () + op_pk_total_size_get (Ack_Pkptr)/LR_WPAN_BIT_RATE, End_Of_Transmission);
	
	/* set the transmitter busy */
	channel_flag.tx_idle = OPC_FALSE;
	channel_flag.rx_idle = OPC_FALSE;

	/* if a turn around time is schedule, we cancel it */
	lr_wpan_cancel_TAT (TXRX_CODE);
	
	FOUT;
}


/*
 * Function:	lr_wpan_drop_higher_layer_pkt
 *
 * Description: 
 *
 * No parameters
 */

static void	lr_wpan_drop_higher_layer_pkt (void)
{
	Packet * pkptr;
	Packet * higher_layer_pkptr;
	int last_packet = 0;
	Ici * ici_ptr;
	char error_msg[256];
	
	
	FIN (lr_wpan_drop_higher_layer_pkt ());
	
	while (!last_packet)
		{
		/* dequeue the packets from the current queue */
		if ((pkptr = op_prg_list_remove (current_queue->subqueue, OPC_LISTPOS_HEAD)) == OPC_NIL)
			{
			sprintf (error_msg, "Queue size: %d; list size: %d\n", current_queue->size, op_prg_list_size (current_queue->subqueue));
			lr_wpan_mac_error ("lr_wpan_drop_higher_layer_pkt:", "Unable to get the packet from the subqueue.", error_msg);
			}
	
		/* get the ICI associate with the packet */
		if ((ici_ptr = op_pk_ici_get (pkptr)) == OPC_NIL)
			lr_wpan_mac_error ("lr_wpan_drop_higher_layer_pkt", "No ICI associates with the current packet", OPC_NIL);
		
		/* destroy the ICI */
		op_ici_destroy (ici_ptr);
				
		/* get the higher layer packet if the segmentation flag is true */
		op_pk_nfd_get_int32 (pkptr, "Segmentation", &last_packet);
		
		if (last_packet)
			{
			/*
			* if this is the last segment of the higher
			* layer packet, we get it and destroy it
			*/
			op_pk_nfd_get_pkt (pkptr, "Higher Layer Pkt", &higher_layer_pkptr);
			
			/* destroy the higher layer packet */
			op_pk_destroy (higher_layer_pkptr);
			}
		
		/* destroy the packet */
		op_pk_destroy (pkptr);
		
		/* update the statistic */
		statistic.Segment_Dropped ++;
		}
	
	/* update the size of the queue */
	current_queue->size = op_prg_list_size (current_queue->subqueue);
		
	FOUT;
}


/*
 * Function:	lr_wpan_destroy_retransmission
 *
 * Description:	
 *
 * ParamOut:	int last_packet
 *				return the value 1 if the retransmission packet
 *				was the last segment of the higher layer packet,
 *				otherwise, we return 0
 */

static int	lr_wpan_destroy_retransmission (void)
{
	Packet * higher_layer_pkptr;	
	Ici * ici_ptr;
	int last_packet;
	
	FIN (lr_wpan_destroy_retransmission ());
	
	/* get the ICI associates with the retransmission packet */
	if ((ici_ptr = op_pk_ici_get (retransmission_ptr)) == OPC_NIL)
		lr_wpan_mac_error ("lr_wpan_destroy_retransmission:", "Unable to get the ICI pointer.", OPC_NIL);
		
	/* destroy the ICI structure */
	op_ici_destroy (ici_ptr);
	
	/* get the segmentation flag */
	op_pk_nfd_get_int32 (retransmission_ptr, "Segmentation", &last_packet);
	
	if (last_packet)
		{
		/*
		* if this is the last segment of the higher
		* layer packet, we get it and destroy it
		*/
		op_pk_nfd_get_pkt (retransmission_ptr, "Higher Layer Pkt", &higher_layer_pkptr);
		
		/* destroy the higher layer packet */
		op_pk_destroy (higher_layer_pkptr);
		}
	
	/* destroy the retransmission packet */
	op_pk_destroy(retransmission_ptr);
	retransmission_ptr = OPC_NIL;
	
	/* reset the transmission retries counter */
	nb_transmission_retries = 0;
	
	FRET (last_packet);
}


/*
 * Function:	lr_wpan_enqueue_packet
 *
 * Description:	subqueue mechanism:
 *				each packet is enqueue according to its destination
 *				MAC address. A structure has been made to record
 *				these information: "Wpan_queue"
 *				this function enqueue a packet according to its
 *				destination MAC address. If no subqueue exists for
 *				the current MAC address, we create a new one.
 *
 * ParamIn:		int mac_address
 *				Destination MAC address of the packet
 *
 *				Packet * pkptr
 *				pointer to the packet to enqueue
 *				
 */

static void lr_wpan_enqueue_packet (int mac_address, Packet * pkptr)
{
	Wpan_Queue * queue;
	
	FIN (lr_wpan_enqueue_packet (mac_address, pkptr));
	
	queue = wpan_queue;
	
	/* update the statistic */
	statistic.Segment_generated ++;
	
	/* consider all the possibilities */
	if (wpan_queue == NULL)
		{
		/* if the structure is not yet created */
		wpan_queue = lr_wpan_allocate_subqueue (mac_address, pkptr);
		
		/* we make a loop */
		wpan_queue->next = wpan_queue;
		
		
		/* initialize the round robin if any */
		current_queue = wpan_queue;

		FOUT;
		}
	
	while (queue != NULL)
		{
		/* */
		if (queue->mac_address == mac_address)
			{
			op_prg_list_insert (queue->subqueue, pkptr, OPC_LISTPOS_TAIL);
			queue->size = op_prg_list_size (queue->subqueue);
			FOUT;
			}
		
		if (queue->next == wpan_queue)
			{
			/* we are at the end of the chain */
			queue->next = lr_wpan_allocate_subqueue (mac_address, pkptr);
			
			/* increase the number of elements */
			wpan_queue->nb_element ++;
			FOUT;
			}
		
		queue = queue->next;
		}
	
	/* we shouldn't be here, so we print out an error message */
	lr_wpan_mac_error ("lr_wpan_enqueue_packet:", "Unable to enqueue the packet", OPC_NIL);
}


/*
 * Function:	lr_wpan_allocate_subqueue
 *
 * Description:	allocate a new subqueue for a new destination
 *				MAC address
 *
 * ParamIn:		int mac_address
 *				Destination MAC address of the packets in the subqueue
 *
 *				Packet * pkptr
 *				pointer to the packet to enqueue
 *
 * ParamOut:	Wpan_queue * queue
 *				New queue structure with the first pascket enqueue
 */

static Wpan_Queue * lr_wpan_allocate_subqueue (int mac_address, Packet * pkptr)
{
	Wpan_Queue * queue;
	
	FIN (lr_wpan_allocate_subqueue ());
	
	/* allocate the memory for the cell */
	queue = (Wpan_Queue *) op_prg_mem_alloc (sizeof (Wpan_Queue));
	
	/* create a the list attribute */
	queue->subqueue = op_prg_list_create ();
	
	/* enqueue the packet */
	op_prg_list_insert (queue->subqueue, pkptr, OPC_LISTPOS_TAIL);
	
	/* initialize the other fields */
	queue->mac_address = mac_address;
	queue->size = 1;
	queue->nb_element = 1;
	queue->next = wpan_queue;
	
	FRET (queue);
}


/*
 * Function:	lr_wpan_print_queue
 *
 * Description:	print the whole queue of the node
 *
 * No parameter
 */

static void
lr_wpan_print_queue (void)
{
	Wpan_Queue * queue;
	int i;
	
	FIN (lr_wpan_print_queue ());
	
	queue = wpan_queue;
	
	/* */
	printf ("\n|-----------------------------------------------------\n");
	printf ("| Node %s:\n", lr_wpan_node_name);
	if (queue != NULL)
		{
		for (i=0; i<wpan_queue->nb_element; i++)
			{
			printf ("|\tMac address: %d\n", queue->mac_address);
			printf ("|\tsize: %d\n|\n", queue->size);
			queue = queue->next;
			}
		}
	else
		printf ("|\tNo subqueues\n");

	printf ("|-----------------------------------------------------\n");

	FOUT;
}


/*
 * Function:	lr_wpan_get_beacon_size
 *
 * Description:	compute the size of a Beacon packet
 *
 * ParamOut:	int beacon_size
 *				size of the beacon packet
 */

static int
lr_wpan_get_beacon_size (Addressing_Field * address)
{
	Wpan_Queue * queue;
	int i;
	int beacon_size, nbPacket = 0;
	
	const int address_size = 16; // size of the address field in bits
	const int maximum_nbpacket = 7; // maximum number of packet
	
	FIN (lr_wpan_get_beacon_size ());
	
	queue = wpan_queue;
	
	/* count the number of packets */
	if (queue != NULL)
		{
		for (i=0; i<wpan_queue->nb_element; i++)
			{
			/* increase the number of total packet */
			nbPacket += queue->size;
			
			/* if we have a packet for the particular source address */
			if (address->src_addr == queue->mac_address)
				address->packet_pending = queue->size;
			
			/* get the next queue */
			queue = queue->next;
			}
		}
	
	/* compute the Beacon packet size */
	if (nbPacket < maximum_nbpacket)
		beacon_size = 104 + nbPacket * address_size;
	else
		beacon_size = 104 + maximum_nbpacket * address_size;

	
	FRET (beacon_size);
}


/*
 * Function:	lr_wpan_subq_checking
 */

static void lr_wpan_subq_checking ()
{
	int i = 0;
	Wpan_Queue * queue;
	Boolean subq_empty;
	
	FIN (lr_wpan_subq_checking ());
	
	queue = current_queue;
	
	/*
 	 * if no transmission is already scheduled, we check
	 * the subqueues for the next transmission
	 */
	/* initialize the flag subqueue empty */
	if (queue != NULL)
		{
		queue = queue->next;
		subq_empty = (queue->size == 0);
		}
	else
		{
		FOUT;
		}
	
	/* check the subqueues */
	while (subq_empty && queue->next != current_queue)
		{
		/* get the next subqueue index */
		queue = queue->next;
		
		/* check if the subqueue is empty */
		subq_empty = (queue->size == 0);
		}
	
	if (!subq_empty)
		{
		/* schedule an interuption for the next transmission */
		op_intrpt_schedule_self (op_sim_time (), Data_Pkt_To_Send);
		
		/* get the subqueue */
		current_queue = queue;
		}
	
	FOUT;
}


/*
 * Function:	lr_wpan_get_rx_seqn
 */

static int lr_wpan_get_rx_seqn (int mac_address)
{
	Mac_Data_Seqn * element;
	int list_size;
	int i; // loop variable
	
	
	FIN (lr_wpan_get_rx_seqn (mac_address));
	
	list_size = op_prg_list_size (Mac_Data_Rx_Seqn);
	
	for (i=0; i<list_size; i++)
		{
		/* get the ith element */
		element = op_prg_list_access (Mac_Data_Rx_Seqn, i);
		
		if (element->mac_address == mac_address)
			FRET (element->seqn);
		}
	
	/* if the MAC address is not present, we initialize it */
	lr_wpan_set_rx_seqn (mac_address, -1);
	
	FRET (-1);
}


/*
 * Function:	lr_wpan_set_rx_seqn
 */

static void lr_wpan_set_rx_seqn (int mac_address, int seqn)
{
	Mac_Data_Seqn * element;
	int list_size;
	int i; // loop variable
	
	FIN (lr_wpan_set_rx_seqn (mac_address, seqn));
	
	 list_size = op_prg_list_size (Mac_Data_Rx_Seqn);
	
	/* search for an existing element */
	for (i=0; i<list_size; i++)
		{
		/* get the ith element */
		element = op_prg_list_access (Mac_Data_Rx_Seqn, i);
		
		if (element->mac_address == mac_address)
			{
			element->seqn = seqn;
			FOUT;
			}
		}
	
	/* if we are here, no sequence number exist yet */
	/* we create a new one */
	element = (Mac_Data_Seqn *) op_prg_mem_alloc (sizeof (Mac_Data_Seqn));
	
	element->mac_address = mac_address;
	element->seqn = seqn;
	
	/* insert this element into the list */
	op_prg_list_insert (Mac_Data_Rx_Seqn, element, OPC_LISTPOS_TAIL);
	
	FOUT;
}

/*
 * Function:	lr_wpan_collect_stat
 *
 * Description:	collect data at the end of the simulation
 *				4 files are created during this function:
 *	
 *				* {basename}_pkt_rcv.txt: record the number
 *				  of packets received and packets loss
 *
 *				* {basename}_pkt_sent.txt: record the number
 *				  of packets sent during the simulation
 *
 *				* {basename}_node_status.txt: record the number
 *				  of packets present in the subqueues
 *
 *				* {basename}_pkt_delay.txt: record the MAC delay
 *
 * No parameter
 */

static void lr_wpan_collect_stat ()
{
	FILE * fp;
	char report_name[128];
	Boolean exist;
	double packet_loss, mac_delay;
	
	const char * packet_rcv = "_pkt_rcv.txt";
	const char * packet_sent = "_pkt_sent.txt";
	const char * node_status = "_node_status.txt";
	const char * packet_delay = "_pkt_delay.txt";
	
	Wpan_Queue * queue;
	int i; // loop variable
	time_t date;
	char date_str[32];
	
	
	FIN (lr_wpan_collect_stat ());
	
	/* get the date of the day */
	time (&date);
	sprintf (date_str, "%s", ctime (&date));
	
	printf(" total backoff time %f\n", statistic.total_backoff_time);
	printf(" total backoff %d\n", statistic.total_backoff);
	
	/* start printing report */
	/* get the report name */
	sprintf (report_name, "%s%s", lr_wpan_node_name, packet_rcv);
	
	fp = lr_wpan_open_report_file (report_name, &exist);
	
	if (!exist)
		{
		/* write a header in the file */
		fprintf (fp, "\t*** Packet received at the node %s ***\n", lr_wpan_node_name);
		fprintf (fp, "\tCreation date: %s\n", date_str);
		fprintf (fp, "\tNb packet collided:\t\tnumber of packet collided, protocol issue\n");
		fprintf (fp, "\tNb packet lost:\t\tnumber of packet lost by collisions\n");
		fprintf (fp, "\tNb packet noise:\t\tnumber of packet lost by noise\n");
		fprintf (fp, "\tData packet rcv:\t\tnumber of MAC data packets received\n");
		fprintf (fp, "\tAck packet rcv:\t\tnumber of acknowledgment packets received\n");
		fprintf (fp, "\tPacket loss:\t\tratio between the number of data packet received and the total number of packet sent (except the Ack)\n");
		fprintf (fp, "\tRetransmit packet:\tnumber of retransmission packet sent\n");
		fprintf (fp, "\tNb Time Out:\t\tnumber of time out reached in the Acknowledgement state\n");
		fprintf (fp, "\tSimulation time:\t\ttotal simulation time in seconds\n\n");
		fprintf (fp, "Nb packet collided\t Nb packet lost\t Nb packet Noise\t Data packet rcv\t Ack packet rcv\t Packet loss\t Nb Time Out\t simulation time\t last modified date\n");
		}
	
	/* compute the packet loss */
	if (statistic.Data_Pkt_Rcv + statistic.Pkt_Lost + statistic.Pkt_Collided + statistic.Pkt_Noise  == 0)
		packet_loss = -1.0;
	else
		packet_loss = (double)( statistic.Pkt_Lost + statistic.Pkt_Collided + statistic.Pkt_Noise) / ((double) statistic.Data_Pkt_Rcv + (double) statistic.Pkt_Lost + (double) statistic.Pkt_Collided + (double)statistic.Pkt_Noise);
	
	fprintf (fp, "%s\t %s\t %s\t %s\t %s\t %s\t %s\t %s\t %s", integer_to_string (statistic.Pkt_Collided, 15),
		integer_to_string (statistic.Pkt_Lost, 14),
		integer_to_string (statistic.Pkt_Noise, 14),
		integer_to_string (statistic.Data_Pkt_Rcv, 15),
		integer_to_string (statistic.Ack_Pkt_Rcv, 14),
		double_to_string (packet_loss, 11),
		integer_to_string (statistic.Nb_Time_Out, 11),
		double_to_string (op_sim_time (), strlen ("simulation time")),
		date_str);
	
	fclose (fp);
	
	/* get the report name */
	sprintf (report_name, "%s%s", lr_wpan_node_name, packet_sent);
	
	fp = lr_wpan_open_report_file (report_name, &exist);
	
	if (!exist)
		{
		/* write a header in the file */
		fprintf (fp, "\t*** Packet sent from the node %s ***\n", lr_wpan_node_name);
		fprintf (fp, "\tCreation date: %s\n", date_str);
		fprintf (fp, "\tSegment generated:\tdata segment generated in the MAC layer\n");
		fprintf (fp, "\tData packet sent:\t\ttotal number of data packets sent\n");
		fprintf (fp, "\tAck packet sent:\t\ttotal number of acknowledgement packets sent\n");
		fprintf (fp, "\tPacket dropped:\t\tnumber of higher layer packets dropped by the CCA\n");
		fprintf (fp, "\tSegment dropped:\t\tnumber of segments dropped by the CCA\n");
		fprintf (fp, "\tSimulation time:\t\ttotal simulation time in seconds\n\n");
		fprintf (fp, "Segment generated\t Data packet sent\t Ack packet sent\t Packet dropped\t Segment dropped\t Retransmit packet\t Simulation time\t last modified date\n");
		}
	
	fprintf (fp, "%s\t %s\t %s\t %s\t %s\t %s\t %s\t %s", integer_to_string (statistic.Segment_generated, strlen ("Segment generated")),
		integer_to_string (statistic.Data_Pkt_Sent, strlen ("Data packet sent")),
		integer_to_string (statistic.Ack_Pkt_Sent, strlen ("Ack packet sent")),
		integer_to_string (statistic.Pkt_Dropped, strlen ("Packet dropped")),
		integer_to_string (statistic.Segment_Dropped, strlen ("Segment dropped")),
		integer_to_string (statistic.Retransmission, strlen ("Retransmit packet")),
		double_to_string (op_sim_time (), strlen ("simulation time")),
		date_str);
	
	fclose (fp);

	/* get the total report name */
	if (wpan_queue != NULL)
		{
		/* get the report name */
		sprintf (report_name, "%s%s", lr_wpan_node_name, node_status);
		
		fp = lr_wpan_open_report_file (report_name, &exist);
		
		/* Print the header of the file */
		if (!exist)
			{
			/* write a header in the file */
			fprintf (fp, "\t*** Status of the node queues for %s ***\n", lr_wpan_node_name);
			fprintf (fp, "\tCreation date: %s\n", date_str);
			fprintf (fp, "\tSubq n length:\tsize of the subqueue n\n");
			fprintf (fp, "\tSimulation time:\ttotal simulation time in seconds\n\n");
			
			for (i=0; i < wpan_queue->nb_element; i++)
				fprintf (fp, "Subq %d length, ", i);
			
			/* print the return char */
			fprintf (fp, "simulation time, last modified date\n");
			}
		
		/* print the results */
		for (i=0, queue = wpan_queue; i < wpan_queue->nb_element; i++)
			{
			fprintf (fp, "%s, ", integer_to_string (queue->size, 13));
			queue = queue->next;
			}
		
		/* print the return char */
		fprintf (fp, "%s, %s", double_to_string (op_sim_time (), strlen ("simulation time")), date_str);
		
		fclose (fp);
		}
	
	
	/* get the report name */
	sprintf (report_name, "%s%s", lr_wpan_node_name, packet_delay);
	
	fp = lr_wpan_open_report_file (report_name, &exist);
	
	/* Print the header of the file */
	if (!exist)
		{
		/* write a header in the file */
		fprintf (fp, "\t*** Packet delay at the node %s ***\n", lr_wpan_node_name);
		fprintf (fp, "\tCreation date: %s\n", date_str);
		fprintf (fp, "\tTotal MAC delay:\t\tsum of all the delay collected at the MAC level\n");
		fprintf (fp, "\tnb data packets:\t\ttotal number of data packets received\n");
		fprintf (fp, "\tMAC delay (everage):\teverage of the MAC delay for one packet\n");
		fprintf (fp, "\tSimulation time:\t\ttotal simulation time in seconds\n\n");
		fprintf (fp, "Total MAC dela\t nb data packets\t MAC delay (everage)\t simulation time\t last modified date\n");
		}
	
	/* compute the MAC everage delay */
	mac_delay = (statistic.Data_Pkt_Rcv) ? statistic.Total_MAC_Delay / (double) statistic.Data_Pkt_Rcv : -1.0;
	
	/* print the results */
	fprintf (fp, "%s\t %s\t %s\t %s\t %s", double_to_string (statistic.Total_MAC_Delay, 15), 
		integer_to_string (statistic.Data_Pkt_Rcv, 15),
		double_to_string (mac_delay, 19),
		double_to_string (op_sim_time (), strlen ("simulation time")),
		date_str);
	
	fclose (fp);
	
	if (!bulk_data_source)
		{
		/* 
		 * we don't have the simple source but the TCP/IP stack, so we must
		 * generate a report for the upper layer
		 */
		/* get the report name */
		sprintf (report_name, "%s_higher_layer.txt", lr_wpan_node_name);
	
		fp = lr_wpan_open_report_file (report_name, &exist);
	
		/* Print the header of the file */
		if (!exist)
			{
			/* write a header in the file */
			fprintf (fp, "\t*** Packet delay at the node %s ***\n", lr_wpan_node_name);
			fprintf (fp, "\tCreation date: %s\n", date_str);
			fprintf (fp, "\tnb packets rcvd:\t\ttotal number of higher layer packets received\n");
			fprintf (fp, "\tnb bits rcvd:\t\ttotal number of higher layer bits received\n");
			fprintf (fp, "\tnb packets sent:\t\ttotal number of higher layer packets sent\n");
			fprintf (fp, "\tnb bits sent:\t\ttotal number of higher layer bits sent\n");			
			fprintf (fp, "\tSimulation time:\t\ttotal simulation time in seconds\n\n");
			fprintf (fp, "nb packets rcvd\t nb bits rcvd\t nb packets sent\t nb bits sent\t simulation time\t last modified date\n");
			}
				
		/* print the results */
		fprintf (fp, "%s\t %s\t %s\t %s\t %s\t %s", integer_to_string (statistic.tcp_HL_pkt_rcvd, strlen ("nb packets rcvd")),
			integer_to_string (statistic.tcp_HL_bit_rcvd, strlen ("nb bits rcvd")),
			integer_to_string (statistic.tcp_HL_pkt_sent, strlen ("nb packets sent")),
			integer_to_string (statistic.tcp_HL_bit_sent, strlen ("nb bits sent")),
			double_to_string (op_sim_time (), strlen ("simulation time")),
			date_str);
		
		fclose (fp);
		}
	
	FOUT;
}

/* End of Function Block */

/* Undefine optional tracing in FIN/FOUT/FRET */
/* The FSM has its own tracing code and the other */
/* functions should not have any tracing.		  */
#undef FIN_TRACING
#define FIN_TRACING

#undef FOUTRET_TRACING
#define FOUTRET_TRACING

#if defined (__cplusplus)
extern "C" {
#endif
	void mac_802_15_4 (OP_SIM_CONTEXT_ARG_OPT);
	VosT_Obtype _op_mac_802_15_4_init (int * init_block_ptr);
	void _op_mac_802_15_4_diag (OP_SIM_CONTEXT_ARG_OPT);
	void _op_mac_802_15_4_terminate (OP_SIM_CONTEXT_ARG_OPT);
	VosT_Address _op_mac_802_15_4_alloc (VosT_Obtype, int);
	void _op_mac_802_15_4_svar (void *, const char *, void **);


	VosT_Obtype Vos_Define_Object_Prstate (const char * _op_name, size_t _op_size);
	VosT_Address Vos_Alloc_Object (VosT_Obtype _op_ob_hndl);
	VosT_Fun_Status Vos_Poolmem_Dealloc (VosT_Address _op_ob_ptr);
#if defined (__cplusplus)
} /* end of 'extern "C"' */
#endif




/* Process model interrupt handling procedure */


void
mac_802_15_4 (OP_SIM_CONTEXT_ARG_OPT)
	{
#if !defined (VOSD_NO_FIN)
	int _op_block_origin = 0;
#endif
	FIN_MT (mac_802_15_4 ());

		{


		FSM_ENTER ("mac_802_15_4")

		FSM_BLOCK_SWITCH
			{
			/*---------------------------------------------------------*/
			/** state (Init) enter executives **/
			FSM_STATE_ENTER_UNFORCED_NOLABEL (0, "Init", "mac_802_15_4 [Init enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Init enter execs]", state0_enter_exec)
				{
				/* initialize the process */
				lr_wpan_mac_init ();
				
				
				if (!bulk_data_source)
					{
					/*
					 * If the packet source is the tcp/ip stack,
					 * we need to register the MAC layer in the 
					 * model wide registry 
					 */
					lr_wpan_model_wide_register ();
					}
				/* create a self interupt */
				op_intrpt_schedule_self (op_sim_time (), 0);
				}
				FSM_PROFILE_SECTION_OUT (state0_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (1,"mac_802_15_4")


			/** state (Init) exit executives **/
			FSM_STATE_EXIT_UNFORCED (0, "Init", "mac_802_15_4 [Init exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Init exit execs]", state0_exit_exec)
				{
				if (!bulk_data_source)
					{
					/*
					 * if the TCP/IP stack is present, we need
					 * to resolve the MAC address of the node
					 */
					lr_wpan_mac_address_resolve ();
					}
				
				/*
				 * If we are in a Slotted mode and the current node is the master,
				 * we schedule the first Beacon packet transmission
				 */
				if (my_parameters->slotted_enable && my_parameters->master_enable)
					{
					/* check the consistency of the slotted mode parameters */
					lr_wpan_slotted_consistency (my_parameters);
				
					/* create the MAC address table to send the Beacons on broadcast */
					lr_wpan_slotted_generate_addr_table (my_parameters, &lr_wpan_beacon_param);
				
					/* schedule the first Beacon packet */
					op_intrpt_schedule_self (lr_wpan_slotted_boundary_time (), Wpan_Beacon_To_Send);
					}
				}
				FSM_PROFILE_SECTION_OUT (state0_exit_exec)


			/** state (Init) transition processing **/
			FSM_TRANSIT_FORCE (1, state1_enter_exec, ;, "default", "", "Init", "Idle", "mac_802_15_4 [Init -> Idle : default / ]")
				/*---------------------------------------------------------*/



			/** state (Idle) enter executives **/
			FSM_STATE_ENTER_UNFORCED (1, "Idle", state1_enter_exec, "mac_802_15_4 [Idle enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Idle enter execs]", state1_enter_exec)
				{
				/* print a debugging message */
				ENTER_STATE_ODB_PRINTING ("enter the \"Idle\" state");
				
				
				if (!wpan_transmission_required)
					{
					if (retransmission_ptr == OPC_NIL)
						{
						/* check the queues to know whether we have some packets to send */
						lr_wpan_subq_checking ();
						}
					else
						{
						/* update the number of transmission retries */
						nb_transmission_retries++;
					
						/* check the number */
						if (nb_transmission_retries <= max_frame_retries)
							{
							/* schedule the retransmission */
							op_intrpt_schedule_self (op_sim_time (), Data_Pkt_To_Send);
							}
						else
							{
							/* 
							 * the maximum number of retransmission is reach, we have
							 * to drop the packet
							 */
							if (!lr_wpan_destroy_retransmission ())
								{
								/*
								 * if this packet is not the last segment of the higher layer
								 * we have to destroy the left segments
								 */
								lr_wpan_drop_higher_layer_pkt ();
								}
							
							/* update the statistic */
							statistic.Pkt_Dropped ++;
							/*
							 * we have to count the segment dropped
							 * in the function "lr_wpan_destroy_retransmission"
							 */
							statistic.Segment_Dropped ++;
							}
						}
					}
				}
				FSM_PROFILE_SECTION_OUT (state1_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (3,"mac_802_15_4")


			/** state (Idle) exit executives **/
			FSM_STATE_EXIT_UNFORCED (1, "Idle", "mac_802_15_4 [Idle exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Idle exit execs]", state1_exit_exec)
				{
				/* check if we receive some packet */
				lr_wpan_intrpt_check ("Idle");
				}
				FSM_PROFILE_SECTION_OUT (state1_exit_exec)


			/** state (Idle) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Idle trans conditions]", state1_trans_conds)
			FSM_INIT_COND (PACKET_TO_SEND)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("Idle")
			FSM_PROFILE_SECTION_OUT (state1_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 6, state6_enter_exec, ;, "PACKET_TO_SEND", "", "Idle", "Init Backoff", "mac_802_15_4 [Idle -> Init Backoff : PACKET_TO_SEND / ]")
				FSM_CASE_TRANSIT (1, 1, state1_enter_exec, ;, "default", "", "Idle", "Idle", "mac_802_15_4 [Idle -> Idle : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (BackOff) enter executives **/
			FSM_STATE_ENTER_UNFORCED (2, "BackOff", state2_enter_exec, "mac_802_15_4 [BackOff enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [BackOff enter execs]", state2_enter_exec)
				{
				/*
				 * We have a packet arriving from the higher layer
				 * We need to send it. To do so, we have to wait for
				 * a backoff period and listen the channel (CCA state)
				 */
				
				/* print a debugging message */
				ENTER_STATE_ODB_PRINTING ("enter the \"Backoff\" state");
				}
				FSM_PROFILE_SECTION_OUT (state2_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (5,"mac_802_15_4")


			/** state (BackOff) exit executives **/
			FSM_STATE_EXIT_UNFORCED (2, "BackOff", "mac_802_15_4 [BackOff exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [BackOff exit execs]", state2_exit_exec)
				{
				/* check if we receive some packet */
				lr_wpan_intrpt_check ("Backoff");
				}
				FSM_PROFILE_SECTION_OUT (state2_exit_exec)


			/** state (BackOff) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [BackOff trans conditions]", state2_trans_conds)
			FSM_INIT_COND (BACKOFF_EXPIRED)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("BackOff")
			FSM_PROFILE_SECTION_OUT (state2_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 3, state3_enter_exec, lr_wpan_cca_init ();, "BACKOFF_EXPIRED", "lr_wpan_cca_init ()", "BackOff", "CCA", "mac_802_15_4 [BackOff -> CCA : BACKOFF_EXPIRED / lr_wpan_cca_init ()]")
				FSM_CASE_TRANSIT (1, 2, state2_enter_exec, ;, "default", "", "BackOff", "BackOff", "mac_802_15_4 [BackOff -> BackOff : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (CCA) enter executives **/
			FSM_STATE_ENTER_UNFORCED (3, "CCA", state3_enter_exec, "mac_802_15_4 [CCA enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [CCA enter execs]", state3_enter_exec)
				{
				/* debugging */
				ENTER_STATE_ODB_PRINTING ("Enter the \"CCA\" state");
				}
				FSM_PROFILE_SECTION_OUT (state3_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (7,"mac_802_15_4")


			/** state (CCA) exit executives **/
			FSM_STATE_EXIT_UNFORCED (3, "CCA", "mac_802_15_4 [CCA exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [CCA exit execs]", state3_exit_exec)
				{
				/* check the interruption */
				lr_wpan_intrpt_check ("CCA");
				}
				FSM_PROFILE_SECTION_OUT (state3_exit_exec)


			/** state (CCA) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [CCA trans conditions]", state3_trans_conds)
			FSM_INIT_COND (BACK_TO_BACKOFF)
			FSM_TEST_COND (END_OF_CCA)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("CCA")
			FSM_PROFILE_SECTION_OUT (state3_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 6, state6_enter_exec, ;, "BACK_TO_BACKOFF", "", "CCA", "Init Backoff", "mac_802_15_4 [CCA -> Init Backoff : BACK_TO_BACKOFF / ]")
				FSM_CASE_TRANSIT (1, 5, state5_enter_exec, ;, "END_OF_CCA", "", "CCA", "Reset Backoff", "mac_802_15_4 [CCA -> Reset Backoff : END_OF_CCA / ]")
				FSM_CASE_TRANSIT (2, 3, state3_enter_exec, ;, "default", "", "CCA", "CCA", "mac_802_15_4 [CCA -> CCA : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (Wait for Ack) enter executives **/
			FSM_STATE_ENTER_UNFORCED (4, "Wait for Ack", state4_enter_exec, "mac_802_15_4 [Wait for Ack enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Wait for Ack enter execs]", state4_enter_exec)
				{
				/* print a debugging message */
				ENTER_STATE_ODB_PRINTING ("enter the \"Wait for Ack\" state");
				}
				FSM_PROFILE_SECTION_OUT (state4_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (9,"mac_802_15_4")


			/** state (Wait for Ack) exit executives **/
			FSM_STATE_EXIT_UNFORCED (4, "Wait for Ack", "mac_802_15_4 [Wait for Ack exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Wait for Ack exit execs]", state4_exit_exec)
				{
				/* check if we receive some packet */
				lr_wpan_intrpt_check ("Wait for Ack");
				
				/*
				 * if the transmission is finished, we schedule Ack the time out
				 * Warning: This step should apply only after a Data transmission.
				 */
				if (channel_flag.tx_idle && op_ev_valid (Ack_TimeOut_Evhandle) == OPC_FALSE)
					{
					/* The data transmission is finished */
					channel_flag.data_is_sending = OPC_FALSE;
					
					/* schedule the time out for the acknowledgment */
					if (my_parameters->slotted_enable)
						{
						/* if we are in the slotted mode, we start the timer at the boundary */
						Ack_TimeOut_Evhandle = op_intrpt_schedule_self (lr_wpan_slotted_boundary_time () + MAC_ACK_WAIT_DURATION, Ack_Time_Out);
						}
					else
						Ack_TimeOut_Evhandle = op_intrpt_schedule_self (op_sim_time () + MAC_ACK_WAIT_DURATION, Ack_Time_Out);
					}		
				
				//if (INTRPT_SELF && intrpt_code == Ack_Received)
				//	{
					/* turn on the LIFS period */
				//	channel_flag.LIFS_on = OPC_TRUE;
					
					/* schedule when the LIFS will be over */
				//	op_intrpt_schedule_self (op_sim_time () + WPAN_LIFS_DURATION, Wpan_LIFS_Over);
					
					/* canceled the Ack Time out if it is still schedule */
				//	if (op_ev_valid (Ack_TimeOut_Evhandle) == OPC_TRUE &&
				//		op_ev_id (Ack_TimeOut_Evhandle) != op_ev_id (op_ev_current()))
				//		op_ev_cancel (Ack_TimeOut_Evhandle);
				//	}
				}
				FSM_PROFILE_SECTION_OUT (state4_exit_exec)


			/** state (Wait for Ack) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Wait for Ack trans conditions]", state4_trans_conds)
			FSM_INIT_COND (BACK_TO_IDLE)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("Wait for Ack")
			FSM_PROFILE_SECTION_OUT (state4_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 1, state1_enter_exec, ;, "BACK_TO_IDLE", "", "Wait for Ack", "Idle", "mac_802_15_4 [Wait for Ack -> Idle : BACK_TO_IDLE / ]")
				FSM_CASE_TRANSIT (1, 4, state4_enter_exec, ;, "default", "", "Wait for Ack", "Wait for Ack", "mac_802_15_4 [Wait for Ack -> Wait for Ack : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (Reset Backoff) enter executives **/
			FSM_STATE_ENTER_FORCED (5, "Reset Backoff", state5_enter_exec, "mac_802_15_4 [Reset Backoff enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Reset Backoff enter execs]", state5_enter_exec)
				{
				/* debugging */
				ENTER_STATE_ODB_PRINTING ("Enter the \"Reset Backoff\" state");
				
				/* set off the backoff parameters if we don't need a second CCA */
				if (lr_wpan_beacon_param.second_cca_req == OPC_FALSE)
					{
					csma_parameters.Be = csma_parameters.min_BE;
					csma_parameters.nb_backoff = 0;
					}
				
				if (channel_is_idle)
					{
					/* 
					 * after the CCA is completed, if the channel is idle, we need to turn off
					 * the RX flag before the transmission
					 */
					channel_flag.rx_idle = OPC_FALSE;
					}
				else
					{
					/* 
					 * Here, all the CCA attempts have failed, we must drop the current packet
					 * and all the segments associates with it if it is the case
					 */
					if (lr_wpan_beacon_param.data_request != OPC_NIL)
						{
						/* we attempt to send a data request, so we destroy it */
						op_pk_destroy (lr_wpan_beacon_param.data_request);
						
						/* we reset the pointer to NULL */
						lr_wpan_beacon_param.data_request = OPC_NIL;
						}
					else if (retransmission_ptr == OPC_NIL)
						{
						/* 
						 * The transmission is not a retransmission:
						 * the packet to transmit is still in the queue,
						 * we have to remove it and all the segment associate
						 * with it from the subqueue
						 */
						lr_wpan_drop_higher_layer_pkt ();
						}
					else if (!lr_wpan_destroy_retransmission ())
						{
						/*
						 * The packet to transmit is a retransmission and 
						 * is not the last segment of the higher layer packet,
						 * we have to destroy the left segments in the subqueue
						 */
						lr_wpan_drop_higher_layer_pkt ();
						
						/*
						 * update the statistic: we have to count the segment
						 * dropped in the function "lr_wpan_destroy_retransmission"
						 */
						statistic.Segment_Dropped ++;
						}
					else
						{
						/* 
						 * if it was a retransmission and the packet was
						 * the last one, just update the number of segment dropped
						 */
						statistic.Segment_Dropped ++;
						}
					/* update the statistic */
					statistic.Pkt_Dropped ++;
					}
				}
				FSM_PROFILE_SECTION_OUT (state5_enter_exec)

			/** state (Reset Backoff) exit executives **/
			FSM_STATE_EXIT_FORCED (5, "Reset Backoff", "mac_802_15_4 [Reset Backoff exit execs]")


			/** state (Reset Backoff) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Reset Backoff trans conditions]", state5_trans_conds)
			FSM_INIT_COND (TRANSMISSION_OK)
			FSM_TEST_COND (!TRANSMISSION_OK)
			FSM_TEST_COND (BACK_TO_CCA)
			FSM_TEST_LOGIC ("Reset Backoff")
			FSM_PROFILE_SECTION_OUT (state5_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 7, state7_enter_exec, ;, "TRANSMISSION_OK", "", "Reset Backoff", "TX", "mac_802_15_4 [Reset Backoff -> TX : TRANSMISSION_OK / ]")
				FSM_CASE_TRANSIT (1, 1, state1_enter_exec, ;, "!TRANSMISSION_OK", "", "Reset Backoff", "Idle", "mac_802_15_4 [Reset Backoff -> Idle : !TRANSMISSION_OK / ]")
				FSM_CASE_TRANSIT (2, 8, state8_enter_exec, ;, "BACK_TO_CCA", "", "Reset Backoff", "Wait boundary", "mac_802_15_4 [Reset Backoff -> Wait boundary : BACK_TO_CCA / ]")
				}
				/*---------------------------------------------------------*/



			/** state (Init Backoff) enter executives **/
			FSM_STATE_ENTER_FORCED (6, "Init Backoff", state6_enter_exec, "mac_802_15_4 [Init Backoff enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Init Backoff enter execs]", state6_enter_exec)
				{
				/* debugging */
				ENTER_STATE_ODB_PRINTING ("Enter the \"init backoff\" state");
				
				/* init the CCA */
				lr_wpan_mac_backoff ();
				
				/* reset the transmission flag if we are in unslotted mode */
				if (my_parameters->slotted_enable == OPC_FALSE)
					wpan_transmission_required = OPC_FALSE;
				}
				FSM_PROFILE_SECTION_OUT (state6_enter_exec)

			/** state (Init Backoff) exit executives **/
			FSM_STATE_EXIT_FORCED (6, "Init Backoff", "mac_802_15_4 [Init Backoff exit execs]")


			/** state (Init Backoff) transition processing **/
			FSM_TRANSIT_FORCE (2, state2_enter_exec, ;, "default", "", "Init Backoff", "BackOff", "mac_802_15_4 [Init Backoff -> BackOff : default / ]")
				/*---------------------------------------------------------*/



			/** state (TX) enter executives **/
			FSM_STATE_ENTER_UNFORCED (7, "TX", state7_enter_exec, "mac_802_15_4 [TX enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [TX enter execs]", state7_enter_exec)
				{
				/* debugging */
				ENTER_STATE_ODB_PRINTING ("Enter the \"TX\" state");
				
				}
				FSM_PROFILE_SECTION_OUT (state7_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (15,"mac_802_15_4")


			/** state (TX) exit executives **/
			FSM_STATE_EXIT_UNFORCED (7, "TX", "mac_802_15_4 [TX exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [TX exit execs]", state7_exit_exec)
				{
				/* check the interuption */
				lr_wpan_intrpt_check ("TX");
				}
				FSM_PROFILE_SECTION_OUT (state7_exit_exec)


			/** state (TX) transition processing **/
			FSM_PROFILE_SECTION_IN ("mac_802_15_4 [TX trans conditions]", state7_trans_conds)
			FSM_INIT_COND (TIME_TO_TRANSMIT)
			FSM_TEST_COND (NO_ACK)
			FSM_TEST_COND (ACKNOWLEDGEMENT)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("TX")
			FSM_PROFILE_SECTION_OUT (state7_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 7, state7_enter_exec, lr_wpan_packet_to_transmit ();, "TIME_TO_TRANSMIT", "lr_wpan_packet_to_transmit ()", "TX", "TX", "mac_802_15_4 [TX -> TX : TIME_TO_TRANSMIT / lr_wpan_packet_to_transmit ()]")
				FSM_CASE_TRANSIT (1, 1, state1_enter_exec, unacknowlegement_reset ();, "NO_ACK", "unacknowlegement_reset ()", "TX", "Idle", "mac_802_15_4 [TX -> Idle : NO_ACK / unacknowlegement_reset ()]")
				FSM_CASE_TRANSIT (2, 4, state4_enter_exec, ;, "ACKNOWLEDGEMENT", "", "TX", "Wait for Ack", "mac_802_15_4 [TX -> Wait for Ack : ACKNOWLEDGEMENT / ]")
				FSM_CASE_TRANSIT (3, 7, state7_enter_exec, ;, "default", "", "TX", "TX", "mac_802_15_4 [TX -> TX : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (Wait boundary) enter executives **/
			FSM_STATE_ENTER_UNFORCED (8, "Wait boundary", state8_enter_exec, "mac_802_15_4 [Wait boundary enter execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Wait boundary enter execs]", state8_enter_exec)
				{
				/*
				 * This state is needed to wait the next boundary 
				 * before performing the second CCA in case of
				 * slotted mode
				 */
				
				/* debugging */
				ENTER_STATE_ODB_PRINTING ("Enter the \"Wait boundary\" state");
				}
				FSM_PROFILE_SECTION_OUT (state8_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (17,"mac_802_15_4")


			/** state (Wait boundary) exit executives **/
			FSM_STATE_EXIT_UNFORCED (8, "Wait boundary", "mac_802_15_4 [Wait boundary exit execs]")
				FSM_PROFILE_SECTION_IN ("mac_802_15_4 [Wait boundary exit execs]", state8_exit_exec)
				{
				/* check the interruption */
				lr_wpan_intrpt_check ("CCA");
				}
				FSM_PROFILE_SECTION_OUT (state8_exit_exec)


			/** state (Wait boundary) transition processing **/
			FSM_TRANSIT_ONLY ((BOUNDARY_REACHED), 3, state3_enter_exec, lr_wpan_cca_init ();, Wait boundary, "BOUNDARY_REACHED", "lr_wpan_cca_init ()", "Wait boundary", "CCA", "mac_802_15_4 [Wait boundary -> CCA : BOUNDARY_REACHED / lr_wpan_cca_init ()]")
				/*---------------------------------------------------------*/



			}


		FSM_EXIT (0,"mac_802_15_4")
		}
	}




void
_op_mac_802_15_4_diag (OP_SIM_CONTEXT_ARG_OPT)
	{
	/* No Diagnostic Block */
	}




void
_op_mac_802_15_4_terminate (OP_SIM_CONTEXT_ARG_OPT)
	{

	FIN_MT (_op_mac_802_15_4_terminate ())


	/* No Termination Block */

	Vos_Poolmem_Dealloc (op_sv_ptr);

	FOUT
	}


/* Undefine shortcuts to state variables to avoid */
/* syntax error in direct access to fields of */
/* local variable prs_ptr in _op_mac_802_15_4_svar function. */
#undef intrpt_type
#undef intrpt_stream
#undef intrpt_code
#undef csma_parameters
#undef channel_is_idle
#undef my_parameters
#undef wpan_frequency_center
#undef lr_wpan_node_name
#undef wpan_queue
#undef nb_pkt_subq
#undef nb_pkt_recv
#undef nb_pkt_send
#undef retransmission_ptr
#undef Mac_Data_Rx_Seqn
#undef Rx_Ack_Expected
#undef Tx_Ack_Require
#undef Ack_TimeOut_Evhandle
#undef Mac_Data_Tx_Seqn
#undef Ack_Dest_Addr
#undef channel_flag
#undef statistic
#undef bulk_data_source
#undef TAT_TX_Evhandle
#undef TAT_RX_Evhandle
#undef wpan_transmission_required
#undef current_queue
#undef max_frame_retries
#undef nb_transmission_retries
#undef own_process_record_handle
#undef oms_aa_handle
#undef Acknowledgement_Enable
#undef lr_wpan_beacon_param
#undef Mac_Beacon_Tx_Seqn

#undef FIN_PREAMBLE_DEC
#undef FIN_PREAMBLE_CODE

#define FIN_PREAMBLE_DEC
#define FIN_PREAMBLE_CODE

VosT_Obtype
_op_mac_802_15_4_init (int * init_block_ptr)
	{
	VosT_Obtype obtype = OPC_NIL;
	FIN_MT (_op_mac_802_15_4_init (init_block_ptr))

	obtype = Vos_Define_Object_Prstate ("proc state vars (mac_802_15_4)",
		sizeof (mac_802_15_4_state));
	*init_block_ptr = 0;

	FRET (obtype)
	}

VosT_Address
_op_mac_802_15_4_alloc (VosT_Obtype obtype, int init_block)
	{
#if !defined (VOSD_NO_FIN)
	int _op_block_origin = 0;
#endif
	mac_802_15_4_state * ptr;
	FIN_MT (_op_mac_802_15_4_alloc (obtype))

	ptr = (mac_802_15_4_state *)Vos_Alloc_Object (obtype);
	if (ptr != OPC_NIL)
		{
		ptr->_op_current_block = init_block;
#if defined (OPD_ALLOW_ODB)
		ptr->_op_current_state = "mac_802_15_4 [Init enter execs]";
#endif
		}
	FRET ((VosT_Address)ptr)
	}



void
_op_mac_802_15_4_svar (void * gen_ptr, const char * var_name, void ** var_p_ptr)
	{
	mac_802_15_4_state		*prs_ptr;

	FIN_MT (_op_mac_802_15_4_svar (gen_ptr, var_name, var_p_ptr))

	if (var_name == OPC_NIL)
		{
		*var_p_ptr = (void *)OPC_NIL;
		FOUT
		}
	prs_ptr = (mac_802_15_4_state *)gen_ptr;

	if (strcmp ("intrpt_type" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_type);
		FOUT
		}
	if (strcmp ("intrpt_stream" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_stream);
		FOUT
		}
	if (strcmp ("intrpt_code" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_code);
		FOUT
		}
	if (strcmp ("csma_parameters" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->csma_parameters);
		FOUT
		}
	if (strcmp ("channel_is_idle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->channel_is_idle);
		FOUT
		}
	if (strcmp ("my_parameters" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->my_parameters);
		FOUT
		}
	if (strcmp ("wpan_frequency_center" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->wpan_frequency_center);
		FOUT
		}
	if (strcmp ("lr_wpan_node_name" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->lr_wpan_node_name);
		FOUT
		}
	if (strcmp ("wpan_queue" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->wpan_queue);
		FOUT
		}
	if (strcmp ("nb_pkt_subq" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->nb_pkt_subq);
		FOUT
		}
	if (strcmp ("nb_pkt_recv" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->nb_pkt_recv);
		FOUT
		}
	if (strcmp ("nb_pkt_send" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->nb_pkt_send);
		FOUT
		}
	if (strcmp ("retransmission_ptr" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->retransmission_ptr);
		FOUT
		}
	if (strcmp ("Mac_Data_Rx_Seqn" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Mac_Data_Rx_Seqn);
		FOUT
		}
	if (strcmp ("Rx_Ack_Expected" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Rx_Ack_Expected);
		FOUT
		}
	if (strcmp ("Tx_Ack_Require" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Tx_Ack_Require);
		FOUT
		}
	if (strcmp ("Ack_TimeOut_Evhandle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Ack_TimeOut_Evhandle);
		FOUT
		}
	if (strcmp ("Mac_Data_Tx_Seqn" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Mac_Data_Tx_Seqn);
		FOUT
		}
	if (strcmp ("Ack_Dest_Addr" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Ack_Dest_Addr);
		FOUT
		}
	if (strcmp ("channel_flag" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->channel_flag);
		FOUT
		}
	if (strcmp ("statistic" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->statistic);
		FOUT
		}
	if (strcmp ("bulk_data_source" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->bulk_data_source);
		FOUT
		}
	if (strcmp ("TAT_TX_Evhandle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->TAT_TX_Evhandle);
		FOUT
		}
	if (strcmp ("TAT_RX_Evhandle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->TAT_RX_Evhandle);
		FOUT
		}
	if (strcmp ("wpan_transmission_required" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->wpan_transmission_required);
		FOUT
		}
	if (strcmp ("current_queue" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->current_queue);
		FOUT
		}
	if (strcmp ("max_frame_retries" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->max_frame_retries);
		FOUT
		}
	if (strcmp ("nb_transmission_retries" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->nb_transmission_retries);
		FOUT
		}
	if (strcmp ("own_process_record_handle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->own_process_record_handle);
		FOUT
		}
	if (strcmp ("oms_aa_handle" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->oms_aa_handle);
		FOUT
		}
	if (strcmp ("Acknowledgement_Enable" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Acknowledgement_Enable);
		FOUT
		}
	if (strcmp ("lr_wpan_beacon_param" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->lr_wpan_beacon_param);
		FOUT
		}
	if (strcmp ("Mac_Beacon_Tx_Seqn" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->Mac_Beacon_Tx_Seqn);
		FOUT
		}
	*var_p_ptr = (void *)OPC_NIL;

	FOUT
	}

