/*
** $File : lr_wpan_support.h
**
** Low Rate WPAN model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      O. Rebala
** Module description:  support file header for Low Rate WPAN model
** Last Modification:   October, 2nd 2003
**
*/

#ifndef __LR_WPAN_SUPPORT__
#define __LR_WPAN_SUPPORT__

/* Standard includes */
#include <opnet.h>
#include <stdio.h>
#include <stdlib.h>


/* wlan support include */
#include "wlan_support.h"

/* define some constant */
#define BEGIN_SIM	(op_intrpt_type () == OPC_INTRPT_BEGSIM)
#define	END_SIM		(op_intrpt_type () == OPC_INTRPT_ENDSIM)
#define INTRPT_STRM	(op_intrpt_type () == OPC_INTRPT_STRM)
#define	INTRPT_SELF	(op_intrpt_type () == OPC_INTRPT_SELF)
#define INTRPT_REMOTE	(op_intrpt_type () == OPC_INTRPT_REMOTE)
#define INTRPT_PROCESS	(op_intrpt_type () == OPC_INTRPT_PROCESS)

#define CHANNEL_UP_STREAM	0

/* define some ODB trace */
#define ODB_CHANNEL_BUFFER	(op_prg_odb_ltrace_active("chan_buffer") == OPC_TRUE)
#define STATE_PRINT_STATUS	(op_prg_odb_ltrace_active("wpan_state") == OPC_TRUE)
#define FLAG_PRINT_STATUS	(op_prg_odb_ltrace_active("wpan_flag") == OPC_TRUE)
#define ODB_GENERATOR_ENABLED (op_prg_odb_ltrace_active("generator") == OPC_TRUE)
#define ODB_WLAN_TRANSMISSION (op_prg_odb_ltrace_active("wlan_transmission") == OPC_TRUE)

/* general constant */
/* the light speed in m/s */
#define LIGHT_SPEED		300000000.0

/* define the packet type */
typedef enum {
	WLAN_PKT_TYPE,
	WPAN_PKT_TYPE,
	BT_MAS_PKT_TYPE,
	BT_SLA_PKT_TYPE	
} Wpan_Pkt_Type;

/* Low Rate WPAN constant (in bytes/s) */
/* WPAN bit rate */
#define LR_WPAN_BIT_RATE	250000.0

/* define one symbol (in seconds) for a data rate of 250 Kb/s */
#define LR_WPAN_SYMBOL		0.000016

/* backoff time unit (in seconds) = 20 symbols */
#define LR_WPAN_BACKOFF_UNIT	20.0*LR_WPAN_SYMBOL

/* Maximum and minimum of the backoff exponent as defined in the specification */
//#define MIN_BACKOFF_EXPONENT	3
//#define MAX_BACKOFF_EXPONENT	5

/* Define the time to wait for an acknowledgment (in second) = 54 symbols */
#define MAC_ACK_WAIT_DURATION	54.0*LR_WPAN_SYMBOL

/* define the maximum sequence number (8 bits) = 2^8 */
#define MAX_MAC_SEQN	256

/* define the turn around time, period needs to switch the channel (in seconds) = 12 symbols */
#define WPAN_TURN_AROUND_TIME	12.0*LR_WPAN_SYMBOL

/* define the time spend in the CCA (in seconds) = 8 symbols */
#define	WPAN_CCA_PERIOD		8.0*LR_WPAN_SYMBOL

/* define the LIFS period (in second) = 40 symbols */
#define WPAN_LIFS_DURATION	40.0*LR_WPAN_SYMBOL

/* define the maximum bit in the payload (118 bytes) = 944 bits */
#define WPAN_PAYLOAD_MAX_SIZE	944

/* define the physical overhead in bit (6 bytes) = 48 bits */
#define LR_WPAN_PHY_OVERHEAD	48 

/* define the maximum bit for a MAC WPAN packet (127 bytes) = 1016 bits */
#define WPAN_MAC_MAX_SIZE	1016

/* define the size of the data request packet (10 bytes) = 80 bits */
#define WPAN_DATA_REQ_SIZE	80

/* define the MAC header size (9 bytes) = 72 bits */
#define 	WPAN_HEADER_SIZE	72

/* define the maximum devices per group */
#define		MAX_DEVICES_PER_GROUP	64

/* compute the center frequency */
#define WPAN_FREQUENCY_CENTER(a)	2405.0 + 5*((double) a - 11.0)


/* define structure */
/* structure for the packet definition in the MAC Layer */
typedef struct {
  int frame_type;
  Boolean security_enabled;
  Boolean frame_pending;
  Boolean ack_req;
  Boolean intra_pan;
  int dest_addr_mode;
  int src_addr_mode;
} Frame_Ctrl_Field;

typedef struct {
  int dest_pan_id;
  int dest_addr; // destination MAC address
  int src_pan_id;
  int src_addr; // source MAC address
  int packet_pending; // artificial flag to know how many packets are pending in the slotted model
} Addressing_Field;

typedef struct {
	Objid	parent_id; // ID of the node
	Objid	objid;	// ID of the module which received the packets
	int		mac_address; // mac address of the node
	int		group; // group of the node
	double 	frequency_wpan;
	Boolean slotted_enable; // establish a slotted or unslotted connexion
	Boolean master_enable; // In case of slotted access, we have to determine a master
	double	x; // X coordinate of the node 
	double	y; // Y coordinate of the node
	char 	type[5]; // type of node
	Boolean cca_requirement; // say if a CCA is required
	int 	share_statistic; // Higher layer packet well received
	double	power;			// Power of the transmission
	double  frequency;      // frequency center of WLAN
	WlanT_Rx_State_Info * wlan_info; // info on the WLAN
} Wpan_Node_Param;

/* Define the Beacon parameters */
typedef struct {
	double stop_time; // stop time of the beacon
	int * addr_table; // mac address table for the broadcast transmission of the Beacons
	int nb_addresses; // number of addresses in the table
	Boolean beacon_rcvd; // flag to know if a Beacon has been received
	Boolean data_req_rcvd; // flag to know if we received a data request
	Boolean second_cca_req; // flag to know if a second cca is pending
	int master_addr; // MAC address of the master of the group
	Packet * data_request; // data request packet
} Wpan_Beacon_Param;

/* variable to store the nodes parameters */
List * wpan_node_param_list;

/* objid of the channel in the topology */
Objid wpan_channel_objid;

/* center frequency according to the channel number */
double wpan_center_freq;

/* CCA mode flag: check of all packet types or just the WPAN/WLAN packets */
Boolean wpan_cca_all_pkt_types;
Boolean wlan_cca_all_pkt_types;

/* function prototypes */
void				lr_wpan_mac_error (const char * fcnt, const char * msg, const char * opt);
Boolean				is_wpan_packet (Packet * pkptr);
int					wpan_node_list_compare_proc (const Wpan_Node_Param * element1, const Wpan_Node_Param * element2);
int 				wpan_search_mac_address (int mac_address);
void 				wpan_store_stranger_node (int mac_address, const char type[5], WlanT_Rx_State_Info * wlan_info);
WlanT_Rx_State_Info * 	wlan_state_variable_get (Objid objid);
int					wpan_get_mac_address (Objid node_id);
Wpan_Node_Param *	wpan_search_wlan_node (int index);
int					wpan_list_size_get_wlan_node (void);
void 				lr_wpan_slotted_consistency (Wpan_Node_Param * master_elt);
void				lr_wpan_slotted_generate_addr_table (Wpan_Node_Param * master_elt, Wpan_Beacon_Param * beacon_ptr);
double				lr_wpan_slotted_boundary_time (void);

#endif // end of __LR_WPAN_SUPPORT__
