/* Process model C form file: lr_wpan_channel.pr.c */
/* Portions of this file copyright 1992-2006 by OPNET Technologies, Inc. */



/* This variable carries the header into the object file */
const char lr_wpan_channel_pr_c [] = "MIL_3_Tfile_Hdr_ 120A 30A op_runsim 7 45228AD4 45228AD4 1 P614876 nchevrol 0 0 none none 0 0 none 0 0 0 0 0 0 0 0 fec 0                                                                                                                                                                                                                                                                                                                                                                                                        ";
#include <string.h>



/* OPNET system definitions */
#include <opnet.h>



/* Header Block */

/*
** $File : WPAN channel header
**
** EPON model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      O. Rebala
** Secondary Author:	N. Chevrollier
** Module description:  channel for Low Rate WPAN model
** Last Modification:   October, 1st 2005
**
*/

/* include header */
#include "lr_wpan_support.h"
#include "channel_buffer.h"
#include "Physical_Layer.h"
#include <math.h>
#include "wlan_support.h"



/* define some constants */
#define PKT_DELIVERY_CODE	0
#define	WPAN_CCA_CODE 		6
#define WPAN_RX_START		8

/* State machine conditions */
#define PACKET_RECEIVED	(INTRPT_STRM)
#define PACKET_TO_SEND 	(INTRPT_SELF && intrpt_code == PKT_DELIVERY_CODE)

#define     MAX_ERROR_TO_CORRECT   35  /* Maximum of errors that we can correct */

double *bt_report_table[2][MAX_ERROR_TO_CORRECT];

/* function prototypes */
static void 		wpan_channel_init (void);
static void 		lr_wpan_chan_error (const char * fcnt, const char * msg, const char * opt);
static void 		lr_wpan_chan_buffer_insert (void);
static int			channel_queue_search_pkt (OpT_Packet_Id pkt_id);
static void 		lr_wpan_channel_pk_send (void);
static int			compute_accept_flag (Buff_Info_Packet * info_packet);
static void 		wlan_flag_transmission (const Buff_Info_Packet * info_packet);
static void 		wlan_flag_transmission_nico (const Buff_Info_Packet * info_packet);

static void			wpan_update_wlan_flag (const Buff_Info_Packet * info_packet);
static void			wpan_flag_cca (const Buff_Info_Packet * info_packet);

/* End of Header Block */

#if !defined (VOSD_NO_FIN)
#undef	BIN
#undef	BOUT
#define	BIN		FIN_LOCAL_FIELD(_op_last_line_passed) = __LINE__ - _op_block_origin;
#define	BOUT	BIN
#define	BINIT	FIN_LOCAL_FIELD(_op_last_line_passed) = 0; _op_block_origin = __LINE__;
#else
#define	BINIT
#endif /* #if !defined (VOSD_NO_FIN) */



/* State variable definitions */
typedef struct
	{
	/* Internal state tracking for FSM */
	FSM_SYS_STATE
	/* State Variables */
	Objid	                  		my_objid                                        ;	/* ID of the current module */
	Objid	                  		my_node_objid                                   ;	/* ID of the current node */
	int	                    		intrpt_type                                     ;	/* type of the current interruption */
	int	                    		intrpt_stream                                   ;	/* stream of the current interruption */
	int	                    		intrpt_code                                     ;	/* code of the current interruption */
	List *	                 		channel_pkt_queue                               ;	/* queue for the received packet */
	} lr_wpan_channel_state;

#define my_objid                		op_sv_ptr->my_objid
#define my_node_objid           		op_sv_ptr->my_node_objid
#define intrpt_type             		op_sv_ptr->intrpt_type
#define intrpt_stream           		op_sv_ptr->intrpt_stream
#define intrpt_code             		op_sv_ptr->intrpt_code
#define channel_pkt_queue       		op_sv_ptr->channel_pkt_queue

/* These macro definitions will define a local variable called	*/
/* "op_sv_ptr" in each function containing a FIN statement.	*/
/* This variable points to the state variable data structure,	*/
/* and can be used from a C debugger to display their values.	*/
#undef FIN_PREAMBLE_DEC
#undef FIN_PREAMBLE_CODE
#define FIN_PREAMBLE_DEC	lr_wpan_channel_state *op_sv_ptr;
#define FIN_PREAMBLE_CODE	\
		op_sv_ptr = ((lr_wpan_channel_state *)(OP_SIM_CONTEXT_PTR->_op_mod_state_ptr));


/* Function Block */

#if !defined (VOSD_NO_FIN)
enum { _op_block_origin = __LINE__ + 2};
#endif

/*
 * Function:	lr_wpan_chan_error
 *
 * Descritpion: print error message and end the simulation
 *
 * ParamIn:		const char * fcnt
 *				name of the function
 *
 *				const char * msg
 *				error message to print
 *
 *				const char * opt
 *				optional error message to print
 *
 */

static void 
lr_wpan_chan_error (const char * fcnt, const char * msg, const char * opt)
{
	FIN (static void lr_wpan_mac_error (fcnt, msg, opt));
	
	/* print error message and end the simulation */
	op_sim_end ("LR WPAN Channel:", fcnt, msg, opt);
	
	FOUT;
}


/*
 * Function:	wpan_channel_init
 *
 * Description:	initialization the channel node
 *
 * No parameter
 */

static void 
wpan_channel_init (void)
{
	Objid cca_param_comp_objid;
	Objid cca_param_objid;
	
	FIN (wpan_channel_init ());
	
	/* initialize the pointer to the channel buffer */
	channel_buffer_ptr = NULL;
	
	/* record the channel node id */
	wpan_channel_objid = my_objid;
	
	/* init the channel queue */
	channel_pkt_queue = op_prg_list_create ();
	
	/* get the WPAN Parameters Object ID */
	op_ima_obj_attr_get (my_objid, "CCA Parameters", &cca_param_comp_objid);
	cca_param_objid = op_topo_child (cca_param_comp_objid, OPC_OBJTYPE_GENERIC, 0);
	
	/* get the WPAN CCA mode */
	op_ima_obj_attr_get (cca_param_objid, "WPAN node", &wpan_cca_all_pkt_types);
	op_ima_obj_attr_get (cca_param_objid, "WLAN node", &wlan_cca_all_pkt_types);
	
	
	/* print informations */
	printf ("\n+--------------------------------------------------------------\n");
	printf ("| Channel Information:\n");
	printf ("| \tWPAN CCA mode: %s\n", wpan_cca_all_pkt_types ? "All packet types checked" : "Only WPAN packets checked");
	printf ("+--------------------------------------------------------------\n");
	
	FOUT;
}


/*
 * Function:	lr_wpan_chan_buffer_insert
 *
 * Description:	insert the received packet into the channel buffer.
 *				All the packet we receive are linked with an ICI structure
 *				which contain the infoamtion about the packet type and
 *				the position of the nodes.
 *				The computation of the transmission delay and the
 *				propagation delay are made here with the information
 *				collected. We record all this information in a structure
 *				Buff_Info_Packet.
 *				Finally, we schedules remote interuptions for the destination
 *				and source nodes to keep track of the different delays. 
 *
 * No parameter
 */

static void 
lr_wpan_chan_buffer_insert (void)
{
	Packet * rcv_pkptr;
	Ici *	ici_ptr;
	Buff_Info_Packet * info_packet;
	double bit_rate;
	int dest_address, pos_index;
	Wpan_Node_Param * element;
	char format[64];
   
	
	FIN (lr_wpan_chan_buffer_insert ());
	
	
	/* get the received packet */
	if ((rcv_pkptr = op_pk_get (intrpt_stream)) == OPC_NIL)
		lr_wpan_chan_error ("lr_wpan_chan_buffer_insert", "No received packet", OPC_NIL);
		
	/* get the received ICI */
	if ((ici_ptr = op_pk_ici_get (rcv_pkptr)) == OPC_NIL)
		lr_wpan_chan_error ("lr_wpan_chan_buffer_insert", "No ICI associates with the current packet", OPC_NIL);
	
	if (ici_ptr == NULL || ici_ptr == OPC_NIL)
		printf ("lr_wpan_chan_buffer_insert: Invalid ICI\n");
	
	/* create the info packet cell */
	info_packet = create_info_packet_cell ();
	
	/* get information from the ici */
	if (op_ici_attr_get (ici_ptr, "txrx distance", &info_packet->txrx_distance) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "bit rate", &bit_rate) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "channel", &info_packet->frequency) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "power", &info_packet->power) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "packet type", &info_packet->packet_type) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "dest address", &dest_address) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "Tx x", &info_packet->tx_x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "Tx y", &info_packet->tx_y) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "Rx x", &info_packet->rx_x) == OPC_COMPCODE_FAILURE ||
		op_ici_attr_get (ici_ptr, "Rx y", &info_packet->rx_y) == OPC_COMPCODE_FAILURE)
		lr_wpan_chan_error ("lr_wpan_chan_buffer_insert", "cannot read the ICI attributes", OPC_NIL);
	
		//op_ici_attr_get (ici_ptr, "piconet number", &piconet_number) == OPC_COMPCODE_FAILURE)
		//lr_wpan_chan_error ("lr_wpan_chan_buffer_insert", "cannot read the ICI attributes", OPC_NIL);
	
	/* get the packet format */
	op_pk_format (rcv_pkptr, format);
	
	/* 
	 * destroy the ICI pointer if the packet comes from a WLAN
	 * or is a WPAN acknolegement.
	 * If the packet is a WPAN data, the ICI will be destroyed
	 * with the copy of the packet used for the retransmission
	 */
	if (strcmp (format, "lr_wpan_mac"))
		op_ici_destroy (ici_ptr);
	
	/* a priori, the packet is valid until the opposite statment is made */
	if (info_packet->packet_type == WPAN_PKT_TYPE)
		{
		info_packet->noise = OPC_FALSE;
		}
	/* get the size of the packet */
	info_packet->packet_size = op_pk_total_size_get (rcv_pkptr);
	
	/* compute the propagation delay */
	info_packet->prop_delay =  info_packet->txrx_distance / LIGHT_SPEED;
	
	/* compute the transmission delay */
	if (info_packet->packet_type == WLAN_PKT_TYPE)
		info_packet->tx_delay = info_packet->packet_size / bit_rate;
	else if (info_packet->packet_type == WPAN_PKT_TYPE)
		info_packet->tx_delay = (info_packet->packet_size + (double) LR_WPAN_PHY_OVERHEAD) / bit_rate;
		
	/*search the information about the destination module in the list of nodes */

	pos_index = wpan_search_mac_address (dest_address);
	/* get the information */
	element = (Wpan_Node_Param *) op_prg_list_access (wpan_node_param_list, pos_index);
	
		
	/* complete the info packet pointer */
	info_packet->packet_id		= op_pk_id (rcv_pkptr);
	info_packet->tx_time		= op_sim_time ();
	info_packet->delivery_time	= op_sim_time () + info_packet->prop_delay + info_packet->tx_delay; 
	
	info_packet->dest_objid		= element->objid;
	info_packet->src_objid		= op_intrpt_source ();
	
	/* if the packet is a wlan packet, we update the WLAN flags */
	if (wlan_cca_all_pkt_types || info_packet->packet_type == WLAN_PKT_TYPE)
		{
		/* update the transmission flag */
		wlan_flag_transmission (info_packet);
		}
	
	if (wpan_cca_all_pkt_types || info_packet->packet_type == WPAN_PKT_TYPE)
		{
		/* update the transmission flag of the WPAN devices */
		wpan_flag_cca (info_packet);
		}
	 
	/* Send some remote interruption according to the packet type */
	if (info_packet->packet_type == WPAN_PKT_TYPE)
		{
		/* if the packet is a WPAN, send an interuption to the destination node to start the reception */
		op_intrpt_schedule_remote (op_sim_time () + info_packet->prop_delay, WPAN_RX_START, info_packet->dest_objid);
		}
	else if (info_packet->packet_type == WLAN_PKT_TYPE)
		{
		/* if the packet is a WLAN, send an interuption to the source module to idle the transmission */
		op_intrpt_schedule_remote (op_sim_time () + info_packet->tx_delay, 0, info_packet->src_objid);
		}
	
	/* schedule the interuption to send the packet to its destination */
	op_intrpt_schedule_self (info_packet->delivery_time, PKT_DELIVERY_CODE);

	/* insert the packet into the buffer */
	channel_buffer_register_packet (info_packet);
	
	/* insert the packet into the queue */
	op_prg_list_insert (channel_pkt_queue, rcv_pkptr, OPC_LISTPOS_TAIL);
	
	/* debug informations */
	if (ODB_CHANNEL_BUFFER)
		{
		printf ("\n+---------------------------------------------\n");
		printf ("| Channel buffer: %.12f s\n|\tpropagation delay: %.12f s\n", op_sim_time (), info_packet->prop_delay);
		printf ("|\ttransmission delay: %.9f s\n|\tdelivery time: %.12f s\n", info_packet->tx_delay, info_packet->delivery_time);
		printf ("+---------------------------------------------\n");
		}
	
	
	FOUT;
}


/*
 * Function:	channel_queue_search_pkt
 *
 * Description:	search a packet in the channel queue, according
 *				to the packet ID in parameter and return the
 *				index of this packet in the list if it exists,
 *				otherwise return -1
 *
 * ParamIn:		OpT_Packet_Id pkt_id
 *				ID of the packet to search
 *
 * ParamOut:	int pos_index
 *				position index of the element
 */

static int 
channel_queue_search_pkt (OpT_Packet_Id pkt_id)
{
	int i, pos_index = -1;
	Packet * element;
	int nbElement;
	
	
	FIN (channel_queue_search_pkt (pkt_id));
	
	nbElement = op_prg_list_size (channel_pkt_queue);
	
	/* chack if the list is not null */
	if (!nbElement)
		lr_wpan_chan_error ("channel_queue_search_pkt:", "No packet registered.", OPC_NIL);

	/* search the element in the list of nodes */
	for (i = 0; i < nbElement; i++)
		{
		/* access the element */
		element = op_prg_list_access (channel_pkt_queue, i);
		
		/* check the MAC address*/
		if (op_pk_id(element) == pkt_id)
			pos_index = i;
		}
		
	FRET (pos_index);
}


/*
 * Function:	lr_wpan_channel_pk_send
 */

static void 
lr_wpan_channel_pk_send (void)
{
	Buff_Info_Packet * info_packet;
	Packet * pkptr;
	int pos_index;
	int accept;
	FIN (lr_wpan_channel_pk_send ());
	
	/* get the information of the packet to send */
	info_packet = search_pkt_info_for_delivery ();
	
	/* get the packet according to the information cell */
	if ((pos_index = channel_queue_search_pkt (info_packet->packet_id)) == -1)
		lr_wpan_chan_error ("lr_wpan_channel_pk_send:", "No packet found in the queue.", OPC_NIL);
	
	/* remove the packet from the channel queue */
	if ((pkptr = op_prg_list_remove (channel_pkt_queue, pos_index)) == OPC_NIL)
		lr_wpan_chan_error ("lr_wpan_channel_pk_send:", "Cannot remove packet from the queue.", OPC_NIL);
		
	/* compute the accept flag */
	accept = compute_accept_flag (info_packet);
	
	/* set the accept flag */
	op_pk_nfd_set (pkptr, "Accept", accept);
	
	/* Set the noise flag */
	if (info_packet->packet_type == WPAN_PKT_TYPE)
		{
		op_pk_nfd_set (pkptr, "Noise", info_packet->noise);
		}
	
	if (info_packet->packet_type == WPAN_PKT_TYPE)
		{	
		/* send the packet to its destination */
		op_pk_deliver (pkptr, info_packet->dest_objid, 1);
		}
	else
		{
		 op_pk_deliver (pkptr, info_packet->dest_objid, 0);	
		 }
		
	/* remove the packet from the queue */
	channel_buffer_remove_packet (info_packet);
   
	FOUT;
}



/*
 * Function:	compute_accept_flag
 *
 * Description	
 */

static int
compute_accept_flag (Buff_Info_Packet * info_packet)
{
	int i; // loop variable
	double ber = 0.0, distance_interferer, distance_transmitter;
	double delta_x, delta_y; // difference beetween Receiver and Interferer coordinates
	Buff_Info_Packet * info_packet_interferer = info_packet->next_collision;
	int packet_size = (int) info_packet->packet_size;
	double power_packet_w, power_packet_interferer_w;
	
	FIN (compute_accept_flag (info_packet));
	

	/* check the number of collisions with the packet */
	if (info_packet->nb_collisions == 1)
		{
		/* the packet has no collision, so it is accepted */
		FRET (1);
		}
	
	
	/* We are in the case that that packet has at least 1 collisions */
	while (info_packet_interferer != NULL)
	   {
		//chan_buffer_print_info_pkt (info_packet);
		
	 // chan_buffer_print_info_pkt (info_packet_interferer);
		
		/* compute the distance beetween the transmitter and the receiver */
		delta_x = info_packet->rx_x - info_packet->tx_x;
		delta_y = info_packet->rx_y - info_packet->tx_y;
		distance_transmitter = sqrt (delta_x * delta_x + delta_y * delta_y);
		
		
		/* compute the distance beetween the interferer and the receiver */
		delta_x = info_packet->rx_x - info_packet_interferer->tx_x;
		delta_y = info_packet->rx_y - info_packet_interferer->tx_y;
		distance_interferer = sqrt (delta_x * delta_x + delta_y * delta_y);
		
		power_packet_w = power_packet_interferer_w = 0.0;
		/* if BT an WPAN conversion of the power in W */
		if (info_packet->packet_type == WLAN_PKT_TYPE)
			power_packet_w = info_packet->power;
		else
			power_packet_w = info_packet->power / 1000;
		
		if (info_packet_interferer->packet_type == WLAN_PKT_TYPE)
			power_packet_interferer_w = info_packet_interferer->power;
		else
			power_packet_interferer_w = info_packet_interferer->power / 1000;
		
		/* compute the BER */
		ber = coexistence_ri (info_packet->packet_type, info_packet->frequency, power_packet_w, distance_transmitter,
			info_packet_interferer->packet_type, info_packet_interferer->frequency, power_packet_interferer_w, distance_interferer);
		
		
		/* compute the error in the packet */
		for (i=0; i<packet_size; i++)
			{
			if (op_dist_uniform (1.0) < ber)
				{
				/* there is no error correction in the WLAN and WPAN devices */
				/* if one error occur in the packet, it is not accepted */
				op_prg_odb_bkpt ("collision");
				FRET (0);
				}
			}
		
		/* Get the next collision */
		info_packet_interferer = info_packet_interferer->next_collision;
		}	
	
	/* in all other case, the packet is accepted */
	FRET (1);
}


/*
 * Function:	wlan_flag_transmission
 *
 * Description:	
 */

static void 
wlan_flag_transmission (const Buff_Info_Packet * info_packet)
{	
	Ici * wlan_rx, *wlan_rx_beginning, *wlan_rx_end;
	int nb_wlan,i;
	Wpan_Node_Param * element;
	double prop_delay;
	double delivery_time;
	
	FIN (wlan_idle_transmission (info_packet));

	
	/* check the information cell */
	if (info_packet->delivery_time	== 0.0 ||
		info_packet->prop_delay		== 0.0 ||
		info_packet->tx_delay		== 0.0 ||
		info_packet->dest_objid		== 0 ||
		info_packet->src_objid		== 0)
		op_sim_message ("FUNCTION WARNING: \"wlan_flag_transmission\" of the LR WPAN Channel", "Some informations of the cell are set to the default value.");
   
	
	if (info_packet->packet_type == WLAN_PKT_TYPE)
		{
		/* check the information cell */
		if (info_packet == NULL)
			lr_wpan_chan_error ("wpan_update_wlan_flag:", "No packet information found.", OPC_NIL);
	
		/* get the number of WLAN in the topology */
		nb_wlan = wpan_list_size_get_wlan_node ();
		
		/* For each WLAN, we need to send an interuption */
		for (i = 0; i < nb_wlan; i++)
			{
			/* get the WLAN device */
			element = wpan_search_wlan_node (i);
			//printf(" device %d, packet %d,\n", element->objid, info_packet->dest_objid );
			
			if( element->objid == info_packet->dest_objid)
				{
				/* 
				* We create an ICI to send the packet delivery time to the wlan receiver
				* in order to update the "rx_end_time" field of the MAC layer 
				*/
				wlan_rx = (Ici *) op_ici_create ("ici_wlan_rx_end");
		
				if (op_ici_attr_set_int32 (wlan_rx, "Packet Type", WLAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
					op_ici_attr_set_dbl (wlan_rx, "End of Transmission", info_packet->delivery_time) == OPC_COMPCODE_FAILURE)
					lr_wpan_chan_error ("wlan_flag_transmission:", "Unable to set the ICI.", OPC_NIL);
		
				/* install the ici with the first interruption */
				op_ici_install (wlan_rx);
		
				/* send an interuption to the destination module to update the power transmission */
				op_intrpt_schedule_remote (op_sim_time () + info_packet->prop_delay, 0, info_packet->dest_objid);
	
				/* Deinstall the ici mechanism */
				op_ici_install (OPC_NIL);
				}
			else
				{
				if ((element->parent_id != op_topo_parent(info_packet->src_objid))&&(element->frequency == info_packet->frequency))
					{
					/* compute the propagation delay */
					prop_delay = sqrt((info_packet->tx_x - element->x)*(info_packet->tx_x - element->x) + 
						(info_packet->tx_y - element->y)*(info_packet->tx_y - element->y)) / LIGHT_SPEED;
		
					/* compute the delivery time */
					delivery_time = info_packet->delivery_time - info_packet->prop_delay + prop_delay;
					
					/* 
					* We create an ICI to send the packet delivery time to the wlan receiver
					* in order to update the "rx_end_time" field of the MAC layer 
					*/
					wlan_rx_beginning = (Ici *) op_ici_create ("ici_wlan_rx_end");
					
					if (op_ici_attr_set_int32 (wlan_rx_beginning, "Packet Type", WLAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
						op_ici_attr_set_dbl (wlan_rx_beginning, "End of Transmission", delivery_time) == OPC_COMPCODE_FAILURE)
						lr_wpan_chan_error ("wpan_update_wlan_flag:", "Unable to set the ICI.", OPC_NIL);
		
					/* install the ici with the first interruption */ 
					op_ici_install (wlan_rx_beginning);
		
					/* send an interuption to the destination module to update the power transmission */
					op_intrpt_schedule_remote (op_sim_time () + prop_delay, 0, element->objid);
					
					/* Deinstall the ici mechanism */
					op_ici_install (OPC_NIL);
					
					wlan_rx_end = (Ici *) op_ici_create ("ici_wlan_rx_end");
					
					if (op_ici_attr_set_int32 (wlan_rx_end, "Packet Type", WLAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
						op_ici_attr_set_dbl (wlan_rx_end, "End of Transmission", delivery_time) == OPC_COMPCODE_FAILURE ||
						op_ici_attr_set_dbl (wlan_rx_end, "Packet Size", info_packet->packet_size) == OPC_COMPCODE_FAILURE )
						lr_wpan_chan_error ("wpan_update_wlan_flag:", "Unable to set the ICI.", OPC_NIL);
		
					/* install the ici with the first interruption */ 
					op_ici_install (wlan_rx_end);
					
					op_intrpt_schedule_remote (delivery_time, 1, element->objid);
					
					op_ici_install (OPC_NIL);
					
					}
				}
			}	
		
		}
	else
		{
		/* if the packet is from a WPAN, we have to send a power transmission in each WLAN device */
		wpan_update_wlan_flag (info_packet);
		}
	
	FOUT;
}



/*
 * Function:	wpan_update_wlan_flag
 *
 * Description:	send a power transmission to all WLAN devices
 *				if a WPAN send a packet and the CCA of the WLAN
 *				detects all packet type
 */

static void
wpan_update_wlan_flag (const Buff_Info_Packet * info_packet)
{
	Ici * wlan_rx;
	int nb_wlan;
	int i; // loop variable
	Wpan_Node_Param * element;
	double prop_delay;
	double delivery_time;
	
	FIN (wpan_update_wlan_flag (info_packet));
	
	/* check the information cell */
	if (info_packet == NULL)
		lr_wpan_chan_error ("wpan_update_wlan_flag:", "No packet information found.", OPC_NIL);
	
	/* get the number of WLAN in the topology */
	nb_wlan = wpan_list_size_get_wlan_node ();
		
	/* For each WLAN, we need to send an interuption */
	for (i = 0; i < nb_wlan; i++)
		{
		/* get the WLAN device */
		element = wpan_search_wlan_node (i);
		
		/* compute the propagation delay */
		prop_delay = sqrt((info_packet->tx_x - element->x)*(info_packet->tx_x - element->x) + 
			(info_packet->tx_y - element->y)*(info_packet->tx_y - element->y)) / LIGHT_SPEED;
		
		/* compute the delivery time */
		delivery_time = info_packet->delivery_time - info_packet->prop_delay + prop_delay;
		
		/* 
		* We create an ICI to send the packet delivery time to the wlan receiver
		* in order to update the "rx_end_time" field of the MAC layer 
		*/
		wlan_rx = (Ici *) op_ici_create ("ici_wlan_rx_end");
		
		if (op_ici_attr_set_int32 (wlan_rx, "Packet Type", WPAN_PKT_TYPE) == OPC_COMPCODE_FAILURE ||
			op_ici_attr_set_dbl (wlan_rx, "End of Transmission", delivery_time) == OPC_COMPCODE_FAILURE)
			lr_wpan_chan_error ("wpan_update_wlan_flag:", "Unable to set the ICI.", OPC_NIL);
		
		/* install the ici with the first interruption */
		op_ici_install (wlan_rx);
		
		/* send an interuption to the destination module to update the power transmission */
		op_intrpt_schedule_remote (op_sim_time () + prop_delay, 0, element->objid);
			
		/* Deinstall the ici mechanism */
		op_ici_install (OPC_NIL);
		}
		
	FOUT;
}


/*
 * Function:	wpan_flag_transmission
 *
 * Description:	
 */

static void
wpan_flag_cca (const Buff_Info_Packet * info_packet)
{
	Wpan_Node_Param * element;
	int list_size = op_prg_list_size (wpan_node_param_list);
	int i; // loop variable
	double distance;
	double prop_delay;
	
	FIN (wpan_flag_cca (info_packet));
	
	/* check the information cell */
	if (info_packet->tx_x == 0.0 ||
		info_packet->tx_y == 0.0)
		op_sim_message ("FUNCTION WARNING: \"wpan_flag_cca\" of the LR WPAN Channel", "Some informations of the cell are set to the default value.");
	
	/*
	 * We check all the node list to know whether we need
	 * to send an interuption for the CCA
	 */
	for (i=0; i<list_size; i++)
		{
		/* get the element in the node list */
		element = op_prg_list_access (wpan_node_param_list, i);
		/* if the node asks for a CCA */
		// equality of frequencies depends on detection mechanism
		//if ((element->cca_requirement) && (element->frequency_wpan == info_packet->frequency))
		if ((element->cca_requirement))	
			{
			//printf("CCA updat\n");
			/* compute the distance between the two nodes */
			distance = sqrt((info_packet->tx_x - element->x)*(info_packet->tx_x - element->x) + (info_packet->tx_y - element->y)*(info_packet->tx_y - element->y));
			
			/* compute the propagation delay */
			prop_delay = distance/LIGHT_SPEED;
						
			/* schedule an event for the CCA period */
			op_intrpt_schedule_remote (op_sim_time () + prop_delay, WPAN_CCA_CODE, element->objid);
			}
		}
		
	
	FOUT
}

/* End of Function Block */

/* Undefine optional tracing in FIN/FOUT/FRET */
/* The FSM has its own tracing code and the other */
/* functions should not have any tracing.		  */
#undef FIN_TRACING
#define FIN_TRACING

#undef FOUTRET_TRACING
#define FOUTRET_TRACING

#if defined (__cplusplus)
extern "C" {
#endif
	void lr_wpan_channel (OP_SIM_CONTEXT_ARG_OPT);
	VosT_Obtype _op_lr_wpan_channel_init (int * init_block_ptr);
	void _op_lr_wpan_channel_diag (OP_SIM_CONTEXT_ARG_OPT);
	void _op_lr_wpan_channel_terminate (OP_SIM_CONTEXT_ARG_OPT);
	VosT_Address _op_lr_wpan_channel_alloc (VosT_Obtype, int);
	void _op_lr_wpan_channel_svar (void *, const char *, void **);


	VosT_Obtype Vos_Define_Object_Prstate (const char * _op_name, size_t _op_size);
	VosT_Address Vos_Alloc_Object (VosT_Obtype _op_ob_hndl);
	VosT_Fun_Status Vos_Poolmem_Dealloc (VosT_Address _op_ob_ptr);
#if defined (__cplusplus)
} /* end of 'extern "C"' */
#endif




/* Process model interrupt handling procedure */


void
lr_wpan_channel (OP_SIM_CONTEXT_ARG_OPT)
	{
#if !defined (VOSD_NO_FIN)
	int _op_block_origin = 0;
#endif
	FIN_MT (lr_wpan_channel ());

		{


		FSM_ENTER ("lr_wpan_channel")

		FSM_BLOCK_SWITCH
			{
			/*---------------------------------------------------------*/
			/** state (Init) enter executives **/
			FSM_STATE_ENTER_UNFORCED_NOLABEL (0, "Init", "lr_wpan_channel [Init enter execs]")
				FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Init enter execs]", state0_enter_exec)
				{
				/* get the ID of the current module */
				my_objid = op_id_self ();
				
				/* get the ID of the current node */
				my_node_objid = op_topo_parent (my_objid);
				
				/* init the process */
				wpan_channel_init ();
				
				/* create a self interuption to exit the initialize state */
				op_intrpt_schedule_self (op_sim_time (), 0);
				}
				FSM_PROFILE_SECTION_OUT (state0_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (1,"lr_wpan_channel")


			/** state (Init) exit executives **/
			FSM_STATE_EXIT_UNFORCED (0, "Init", "lr_wpan_channel [Init exit execs]")


			/** state (Init) transition processing **/
			FSM_TRANSIT_FORCE (1, state1_enter_exec, ;, "default", "", "Init", "Idle", "lr_wpan_channel [Init -> Idle : default / ]")
				/*---------------------------------------------------------*/



			/** state (Idle) enter executives **/
			FSM_STATE_ENTER_UNFORCED (1, "Idle", state1_enter_exec, "lr_wpan_channel [Idle enter execs]")
				FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Idle enter execs]", state1_enter_exec)
				{
				
				}
				FSM_PROFILE_SECTION_OUT (state1_enter_exec)

			/** blocking after enter executives of unforced state. **/
			FSM_EXIT (3,"lr_wpan_channel")


			/** state (Idle) exit executives **/
			FSM_STATE_EXIT_UNFORCED (1, "Idle", "lr_wpan_channel [Idle exit execs]")
				FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Idle exit execs]", state1_exit_exec)
				{
				/* get the type of the interruption */
				intrpt_type = op_intrpt_type ();
				
				switch (intrpt_type)
					{
					case OPC_INTRPT_STRM:
					intrpt_stream = op_intrpt_strm ();
					break;
					
					case OPC_INTRPT_SELF:
					intrpt_code = op_intrpt_code ();
					break;
					
					default:
					lr_wpan_chan_error ("Idle state:", "Unexpected interruption.", OPC_NIL);
					break;
					}
				}
				FSM_PROFILE_SECTION_OUT (state1_exit_exec)


			/** state (Idle) transition processing **/
			FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Idle trans conditions]", state1_trans_conds)
			FSM_INIT_COND (PACKET_RECEIVED)
			FSM_TEST_COND (PACKET_TO_SEND)
			FSM_DFLT_COND
			FSM_TEST_LOGIC ("Idle")
			FSM_PROFILE_SECTION_OUT (state1_trans_conds)

			FSM_TRANSIT_SWITCH
				{
				FSM_CASE_TRANSIT (0, 2, state2_enter_exec, ;, "PACKET_RECEIVED", "", "Idle", "Pkt Rcv", "lr_wpan_channel [Idle -> Pkt Rcv : PACKET_RECEIVED / ]")
				FSM_CASE_TRANSIT (1, 3, state3_enter_exec, ;, "PACKET_TO_SEND", "", "Idle", "Send", "lr_wpan_channel [Idle -> Send : PACKET_TO_SEND / ]")
				FSM_CASE_TRANSIT (2, 1, state1_enter_exec, ;, "default", "", "Idle", "Idle", "lr_wpan_channel [Idle -> Idle : default / ]")
				}
				/*---------------------------------------------------------*/



			/** state (Pkt Rcv) enter executives **/
			FSM_STATE_ENTER_FORCED (2, "Pkt Rcv", state2_enter_exec, "lr_wpan_channel [Pkt Rcv enter execs]")
				FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Pkt Rcv enter execs]", state2_enter_exec)
				{
				/* put the packet received in the channel buffer */
				lr_wpan_chan_buffer_insert ();
				}
				FSM_PROFILE_SECTION_OUT (state2_enter_exec)

			/** state (Pkt Rcv) exit executives **/
			FSM_STATE_EXIT_FORCED (2, "Pkt Rcv", "lr_wpan_channel [Pkt Rcv exit execs]")


			/** state (Pkt Rcv) transition processing **/
			FSM_TRANSIT_FORCE (1, state1_enter_exec, ;, "default", "", "Pkt Rcv", "Idle", "lr_wpan_channel [Pkt Rcv -> Idle : default / ]")
				/*---------------------------------------------------------*/



			/** state (Send) enter executives **/
			FSM_STATE_ENTER_FORCED (3, "Send", state3_enter_exec, "lr_wpan_channel [Send enter execs]")
				FSM_PROFILE_SECTION_IN ("lr_wpan_channel [Send enter execs]", state3_enter_exec)
				{
				/* forward the packet and call the physical layer */
				lr_wpan_channel_pk_send ();
				}
				FSM_PROFILE_SECTION_OUT (state3_enter_exec)

			/** state (Send) exit executives **/
			FSM_STATE_EXIT_FORCED (3, "Send", "lr_wpan_channel [Send exit execs]")


			/** state (Send) transition processing **/
			FSM_TRANSIT_FORCE (1, state1_enter_exec, ;, "default", "", "Send", "Idle", "lr_wpan_channel [Send -> Idle : default / ]")
				/*---------------------------------------------------------*/



			}


		FSM_EXIT (0,"lr_wpan_channel")
		}
	}




void
_op_lr_wpan_channel_diag (OP_SIM_CONTEXT_ARG_OPT)
	{
	/* No Diagnostic Block */
	}




void
_op_lr_wpan_channel_terminate (OP_SIM_CONTEXT_ARG_OPT)
	{

	FIN_MT (_op_lr_wpan_channel_terminate ())


	/* No Termination Block */

	Vos_Poolmem_Dealloc (op_sv_ptr);

	FOUT
	}


/* Undefine shortcuts to state variables to avoid */
/* syntax error in direct access to fields of */
/* local variable prs_ptr in _op_lr_wpan_channel_svar function. */
#undef my_objid
#undef my_node_objid
#undef intrpt_type
#undef intrpt_stream
#undef intrpt_code
#undef channel_pkt_queue

#undef FIN_PREAMBLE_DEC
#undef FIN_PREAMBLE_CODE

#define FIN_PREAMBLE_DEC
#define FIN_PREAMBLE_CODE

VosT_Obtype
_op_lr_wpan_channel_init (int * init_block_ptr)
	{
	VosT_Obtype obtype = OPC_NIL;
	FIN_MT (_op_lr_wpan_channel_init (init_block_ptr))

	obtype = Vos_Define_Object_Prstate ("proc state vars (lr_wpan_channel)",
		sizeof (lr_wpan_channel_state));
	*init_block_ptr = 0;

	FRET (obtype)
	}

VosT_Address
_op_lr_wpan_channel_alloc (VosT_Obtype obtype, int init_block)
	{
#if !defined (VOSD_NO_FIN)
	int _op_block_origin = 0;
#endif
	lr_wpan_channel_state * ptr;
	FIN_MT (_op_lr_wpan_channel_alloc (obtype))

	ptr = (lr_wpan_channel_state *)Vos_Alloc_Object (obtype);
	if (ptr != OPC_NIL)
		{
		ptr->_op_current_block = init_block;
#if defined (OPD_ALLOW_ODB)
		ptr->_op_current_state = "lr_wpan_channel [Init enter execs]";
#endif
		}
	FRET ((VosT_Address)ptr)
	}



void
_op_lr_wpan_channel_svar (void * gen_ptr, const char * var_name, void ** var_p_ptr)
	{
	lr_wpan_channel_state		*prs_ptr;

	FIN_MT (_op_lr_wpan_channel_svar (gen_ptr, var_name, var_p_ptr))

	if (var_name == OPC_NIL)
		{
		*var_p_ptr = (void *)OPC_NIL;
		FOUT
		}
	prs_ptr = (lr_wpan_channel_state *)gen_ptr;

	if (strcmp ("my_objid" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->my_objid);
		FOUT
		}
	if (strcmp ("my_node_objid" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->my_node_objid);
		FOUT
		}
	if (strcmp ("intrpt_type" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_type);
		FOUT
		}
	if (strcmp ("intrpt_stream" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_stream);
		FOUT
		}
	if (strcmp ("intrpt_code" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->intrpt_code);
		FOUT
		}
	if (strcmp ("channel_pkt_queue" , var_name) == 0)
		{
		*var_p_ptr = (void *) (&prs_ptr->channel_pkt_queue);
		FOUT
		}
	*var_p_ptr = (void *)OPC_NIL;

	FOUT
	}

