/* wlan_custom_power.ps.c */                                                       
/* Received power model for WLAN suite physical layer modeling.	*/

/****************************************/
/*  	  Copyright (c) 1993-2002		*/
/*      by OPNET Technologies, Inc.     */
/*		(A Delaware Corporation)		*/
/*	7255 Woodmont Av., Suite 250  		*/
/*     Bethesda, MD 20814, U.S.A.       */
/*			All Rights Reserved.		*/
/****************************************/

#include "opnet.h"
#include "wlan_support.h"	
#include <math.h>


/***** constants *****/

#define C					3.0E+08		/* speed of light (m/s) */
#define SIXTEEN_PI_SQ		157.91367	/* 16 times pi-squared 	*/

static const char*	PowI_Err_Hdr = "Error in radio power computation pipeline stage (wlan_power):";


/***** pipeline procedure *****/

#if defined (__cplusplus)
extern "C"
#endif
void
wlan_custom_power_mt (OP_SIM_CONTEXT_ARG_OPT_COMMA Packet* pkptr)
	{
	double		prop_distance, rcvd_power, path_loss;
	double		tx_power, tx_base_freq, tx_bandwidth, tx_center_freq;
	double		lambda, rx_ant_gain, tx_ant_gain;
	Objid		rx_ch_obid;
	double		in_band_tx_power, band_max, band_min;
	double		rx_base_freq, rx_bandwidth;
	double		pk_reception_end;
	int			rcvd_frame_type;
	int			chanmatch_status;
	WlanT_Rx_State_Info* rx_state_ptr;

	/** Compute the average received power in Watts of the		**/
	/** signal associated with a transmitted packet.			**/
	FIN_MT (wlan_custom_power (pkptr));

	/* Get power allotted to transmitter channel. 				*/
	tx_power = op_td_get_dbl (pkptr, OPC_TDA_RA_TX_POWER);

	/* Get transmission frequency in Hz. 						*/
	tx_base_freq = op_td_get_dbl (pkptr, OPC_TDA_RA_TX_FREQ);
	tx_bandwidth = op_td_get_dbl (pkptr, OPC_TDA_RA_TX_BW);
	tx_center_freq = tx_base_freq + (tx_bandwidth / 2.0);

	/* When using TMM, the TDA OPC_TDA_RA_RCVD_POWER will		*/
	/* already have a raw value for the path loss.				*/
	if (op_td_is_set (pkptr, OPC_TDA_RA_RCVD_POWER))
		{
		path_loss = op_td_get_dbl (pkptr, OPC_TDA_RA_RCVD_POWER);
		}
	else
		{
		/* Caclculate wavelength (in meters). 					*/
		lambda = C / tx_center_freq;

		/* Get distance between transmitter and receiver (in	*/
		/* meters). 											*/
		prop_distance = op_td_get_dbl (pkptr, OPC_TDA_RA_START_DIST);
		
		/* Compute the path loss for this distance and			*/
		/* wavelength. 											*/
		if (prop_distance > 0.0)
			{
			path_loss = (lambda * lambda) / 
				(SIXTEEN_PI_SQ * prop_distance * prop_distance);
			}
		else
			path_loss = 1.0;
		}

	/* Compute the in-band transmission power by multiplying	*/
	/* the transmission power with the overlap ratio between	*/
	/* the frequency bands of the transmitter and receiver.		*/
	chanmatch_status = op_td_get_int (pkptr, OPC_TDA_RA_MATCH_STATUS);
	if (chanmatch_status == OPC_TDA_RA_MATCH_VALID)
		{
		/* Since the packet status is not set to noise, there	*/
		/* must be a exact match between the bands (i.e.		*/
		/* overlap ratio = 1.0).								*/
		in_band_tx_power = tx_power;
		}
	else
		{
		/* Determine the receiver bandwidth and base frequency.	*/
		rx_base_freq = op_td_get_dbl (pkptr, OPC_TDA_RA_RX_FREQ);
		rx_bandwidth = op_td_get_dbl (pkptr, OPC_TDA_RA_RX_BW);

		/* Use these values to determine the band overlap with	*/
		/* the transmitter. Note that if there were no overlap	*/
		/* at all, the packet would already have been filtered	*/
		/* by the channel match stage.							*/

		/* The base of the overlap band is the highest base		*/
		/* frequency.											*/
		if (rx_base_freq > tx_base_freq) 
			band_min = rx_base_freq;
		else
			band_min = tx_base_freq;
	
		/* The top of the overlap band is the lowest end		*/
		/* frequency. 											*/
		if (rx_base_freq + rx_bandwidth > tx_base_freq + tx_bandwidth)
			band_max = tx_base_freq + tx_bandwidth;
		else
			band_max = rx_base_freq + rx_bandwidth;
	
		/* Compute the amount of in-band transmitter power. 	*/
		in_band_tx_power = tx_power * (band_max - band_min) / tx_bandwidth;
		}
	
	/* Get antenna gains (raw form, not in dB). 				*/
	tx_ant_gain = pow (10.0, op_td_get_dbl (pkptr, OPC_TDA_RA_TX_GAIN) / 10.0);
	rx_ant_gain = pow (10.0, op_td_get_dbl (pkptr, OPC_TDA_RA_RX_GAIN) / 10.0);

	/* Calculate received power level. 							*/
	rcvd_power = in_band_tx_power * tx_ant_gain * path_loss * rx_ant_gain;

	/* Assign the received power level (in Watts) to the packet	*/
	/* transmission data attribute.								*/
	op_td_set_dbl (pkptr, OPC_TDA_RA_RCVD_POWER, rcvd_power);

	/* Now we need to check whether the receiver will lock onto	*/
	/* this signal, which depends on the receiver's current		*/
	/* status and the reception power of the signal.			*/
	/* First check the status of the receiver node.				*/
	if (op_td_is_set (pkptr, OPC_TDA_RA_ND_FAIL))
		{
		/* The receiving node is disabled. Change the channel	*/
		/* match status to noise.								*/
		op_td_set_int (pkptr, OPC_TDA_RA_MATCH_STATUS, OPC_TDA_RA_MATCH_NOISE);
		}
	else
		{
		/* The receiving node is enabled. Retrieve the state	*/
		/* information of the receiver channel.					*/
		rx_ch_obid   = op_td_get_int (pkptr, OPC_TDA_RA_RX_CH_OBJID);
		rx_state_ptr = (WlanT_Rx_State_Info *) op_ima_obj_state_get (rx_ch_obid);
			
		/* If the receiver is already receiving another packet,	*/
		/* then this packet will be considered as noise. This	*/
		/* prevents simultaneous reception of multiple valid	*/
		/* packets.												*/
		if (rx_state_ptr->rx_end_time <= op_sim_time ())
			{
			/* The receiver is idle. Check the packet's			*/
			/* reception power value.							*/
			if (rcvd_power > rx_state_ptr->rx_power_thresh)
				{					
				/* Check the status of the packet. It could be	*/
				/* set to noise in channel match stage because	*/
				/* of imperfect match (interference from		*/
				/* another BSS). If this is the case, since it	*/
				/* is still a powerful packet and arriving the	*/
				/* receiver while it is idling, change its		*/
				/* status to "VALID" and set its "Accept"		*/
				/* field appropriately.							*/
				if (chanmatch_status == OPC_TDA_RA_MATCH_NOISE)
					{
					op_td_set_int (pkptr, OPC_TDA_RA_MATCH_STATUS, OPC_TDA_RA_MATCH_VALID);
					op_pk_nfd_set (pkptr, "Accept", WLANC_NEIGHBOR_BSS_PKT_REJECT);
					}
				}
			else
				{
				/* Packet's power is too weak to turn on the	*/
				/* signal lock and to remain a valid packet.	*/
				/* Change its status to noise.					*/
				op_td_set_int (pkptr, OPC_TDA_RA_MATCH_STATUS, OPC_TDA_RA_MATCH_NOISE);

				/* Beacons are monitored for evaluating			*/
				/* connectivity to the AP. One method to model	*/
				/* Beacon monitoring is to periodically check	*/
				/* for Beacon reception, by setting periodic	*/
				/* self-interrupts. However, this method is not	*/
				/* very efficient. Therefore, we will rig the	*/
				/* physical layer so that all Beacons are		*/
				/* processed for reliability. Those that can't	*/
				/* be correctly demodulated will be used to		*/
				/* model missing beacons without actually       */
				/* using an inefficient self-timer mechanism.	*/
				/* Only consider the Beacons that are			*/
				/* transmitted at the channel currently used by	*/
				/* the receiver (i.e. if the initial status is	*/
				/* VALID).										*/
				if (rx_state_ptr->roaming_info_ptr->enable_roaming && chanmatch_status == OPC_TDA_RA_MATCH_VALID)
					{
					op_pk_nfd_access (pkptr, "Type", &rcvd_frame_type) ;
					if (rcvd_frame_type == WlanC_Beac)
						{
						/* Serialize the execution of AP		*/
						/* reliability evaluation, since the	*/
						/* MAC of the receiver can be reading/	*/
						/* writing the same information.		*/
						op_prg_mt_mutex_lock (rx_state_ptr->roaming_info_ptr->roam_info_mutex, 0);
						wlan_ap_reliability_eval (pkptr, OPC_FALSE, rx_state_ptr->roaming_info_ptr);
						op_prg_mt_mutex_unlock (rx_state_ptr->roaming_info_ptr->roam_info_mutex);
						}
					}
				}
			}
		else
			{			
			/* The channel is already busy. We will treat the	*/
			/* current packet as noise.							*/
			op_td_set_int (pkptr, OPC_TDA_RA_MATCH_STATUS, OPC_TDA_RA_MATCH_NOISE);

			/* See comment in above that explains special		*/
			/* treatment of Beacons.							*/
			if (rx_state_ptr->roaming_info_ptr->enable_roaming && chanmatch_status == OPC_TDA_RA_MATCH_VALID)
				{
				op_pk_nfd_access (pkptr, "Type", &rcvd_frame_type) ;
				if (rcvd_frame_type == WlanC_Beac)
					{
					/* Serialize the execution of AP			*/
					/* reliability evaluation, since the MAC of	*/
					/* the receiver can be reading/writing the	*/
					/* same information.						*/
					op_prg_mt_mutex_lock (rx_state_ptr->roaming_info_ptr->roam_info_mutex, 0);
					wlan_ap_reliability_eval (pkptr, OPC_FALSE, rx_state_ptr->roaming_info_ptr);
					op_prg_mt_mutex_unlock (rx_state_ptr->roaming_info_ptr->roam_info_mutex);
					}
				}
			}

		/* Update the reception end time for the receiver based	*/
		/* on the new packet if its power is exceeding the		*/
		/* threshold.											*/
		if (rcvd_power > rx_state_ptr->rx_power_thresh)
			{
			pk_reception_end = op_td_get_dbl (pkptr, OPC_TDA_RA_END_RX);
			if (pk_reception_end > rx_state_ptr->rx_end_time)
				rx_state_ptr->rx_end_time = pk_reception_end;
			
			/* Modif. Oliv */
			op_prg_odb_bkpt ("pipeline_power");
			/***************/
			}
		}

	FOUT;
	}

