//========================= Official Notice ===============================
//
// "This software was developed at the National Institute of Standards
// and Technology by employees of the Federal Government in the course of
// their official duties. Pursuant to Title 17 Section 105 of the United
// States Code this software is not subject to copyright protection and
// is in the public domain.
//
// The NIST Data Flow System (NDFS) is an experimental system and is
// offered AS IS. NIST assumes no responsibility whatsoever for its use
// by other parties, and makes no guarantees and NO WARRANTIES, EXPRESS
// OR IMPLIED, about its quality, reliability, fitness for any purpose,
// or any other characteristic.
//
// We would appreciate acknowledgement if the software is used.
//
// This software can be redistributed and/or modified freely provided
// that any derivative works bear some notice that they are derived from
// it, and any modified versions bear some notice that they have been
// modified from the original."
//
//=========================================================================

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <getopt.h>

char *progname;
int verbose = 0;

enum {
  data_1K = 1024,
  data_1M = data_1K * data_1K,
  
  buffers_mult = 4,
  
  in_data_size  = 3,
  out_data_size = 4,

  // we read 12M, write 16M at a time
  buffer_in_toread   = in_data_size  * buffers_mult * data_1M,
  buffer_out_towrite = out_data_size * buffers_mult * data_1M,
};

unsigned char *buffer_in, *buffer_out;

char *dname = NULL;
int mulit = 1;

/********************/

void equit(char *msg)
{
  fprintf(stderr, "ERROR: %s\n", msg);
  exit(1);
}

/********************/

static void print_usage(FILE *stream)
{
  fprintf(stream, "\nUsage : %s [options] infile outfile\n"
          "  -h         --help          Gives this message\n"
	  "  -d file    --dump          Dump int values into \'file\'\n"
	  "  -m mulx    --multiply      Multiply each individual value by \'mulx\' (\'256\' gives you the equivalent of the 24 bits number in a 32 bits range (only positive ints allowed)\n"
	  "  -v         --verbose       Be a little more detailled\n"
	  " The program takes a RAW 24bits packed data file as given by the MkIII (Big Endian) and converts it to the Linux (Little Endian) 32bits value into the output file. Note that the number of channel is not important, since the data is all packed onto one big chunk.\n"
          ,progname);
}

/********************/

int options(int argc, char ***argv)
{
  static struct option optlist[] = {
    { "help",     0, 0, 'h' },
    { "dump",     1, 0, 'd' },
    { "multiply", 1, 0, 'm' },
    { "verbose",  0, 0, 'v' },
    { 0,          0, 0, 0   }
  };
  
  int usage = 0, finish = 0, error = 0;
  int ra = 1; // Among removed arguments is always argv[0]
  
  for(;;) {
    int opt = getopt_long(argc, *argv, "hd:m:v", optlist, 0);
    if(opt == EOF)
      break;
    switch(opt) {
    case 'h':
      usage = 1;
      finish = 1;
      error = 0;
      ra++;
      break;
    case 'd':
      dname = optarg;
      ra += 2;
      break;
    case 'm':
      mulit = atoi(optarg);
      ra += 2;
      break;
    case 'v':
      verbose++;
      ra += 1;
      break;
    case '?':
    case ':':
      usage = 1;
      finish = 1;
      error = 1;
      break;
    default:
      abort();
    }
    if (finish)
      break;
  }
  
  if (usage)
    print_usage(error ? stderr : stdout);

  if (finish)
    exit(error);

  *argv += optind;
  
  return ra;
}

/********************/

int main(int argc, char **argv)
{
  int ifile;
  int ofile;
  FILE *dfile = NULL;
  int totread = 0;
  int keepreading;
  
  char msg[1024];
  const char *iname, *oname;
  
  progname = argv[0];

  argc -= options(argc, &argv);
  if (argc < 2) {
    print_usage(stderr);
    exit(1);
  }
  
  iname = argv[0];
  oname = argv[1];
  
  sprintf(msg, "opening input file (%s)", iname);
  ifile = open(iname, O_RDONLY);
  if (ifile < 0) equit(msg);
  
  sprintf(msg, "opening output file (%s)", oname);
  ofile = open(oname, O_RDWR|O_CREAT|O_TRUNC, 0666);
  if (ofile < 0) equit(msg);
  
  if (dname != NULL) {
    sprintf(msg, "opening dump file (%s)", dname);
    dfile = fopen(dname, "w");
    if (dfile == NULL) equit(msg);
  }

  if (mulit < 1)
    equit("Error: no negative or zero value allowed for \'multiply\'");

  if (verbose) printf("Input file:  %s\nOutput file: %s\n", iname, oname);
  
  buffer_in = (unsigned char *) malloc(buffer_in_toread);
  if (buffer_in == NULL) equit("Could not allocate input memory");
  
  buffer_out = (unsigned char *) malloc(buffer_out_towrite);
  if (buffer_out == NULL) equit("Could not allocate output memory");
  
  keepreading = 1;
  if (verbose) printf("Processing: ");
  while (keepreading) {
    int act_read, act_wrote;
    int readin, towrite;
    int value;
    unsigned char *ptr = (unsigned char *) &value;
    
    act_read = read(ifile, buffer_in, buffer_in_toread);
    
    if (act_read != buffer_in_toread) {
      keepreading = 0;
    }
    totread += act_read;
    
    readin = 0;
    towrite = 0;
    while (act_read - readin >= 3) {
      int j;

      value = (buffer_in[readin+2]) | (buffer_in[readin+1] << 8) 
	| (((signed char) buffer_in[readin]) << 16);

      if (mulit > 1)
	value *= mulit;

      if (dname != NULL) 
	fprintf(dfile, "%d\n", value);

      for (j = 0; j < out_data_size; j++) 
	buffer_out[towrite + j] = ptr[j];

      readin  += in_data_size;
      towrite += out_data_size;
    }
    value = act_read - readin; // tmp use of the var
    if (value > 0) {
      fprintf(stderr, "Warning: the number of bytes remaining after current processing is not a multiple of %d (%d); are you sure the input file is proper ? We will stop processing further data\n", in_data_size, value);
      keepreading = 0;
      totread -= value;
    }
    
    act_wrote = write(ofile, buffer_out, towrite);
    if (act_wrote != towrite)
      equit("problem writing output data");
    
    if (verbose) printf(".");
  }
  if (verbose) printf(" done\n");
  
  close(ifile);
  close(ofile);
  if (dname != NULL) fclose(dfile);
  if (verbose) printf("Exiting after reading %d bytes\n", totread);

  return EXIT_SUCCESS;
}

      
