//========================= Official Notice ===============================
//
// "This software was developed at the National Institute of Standards
// and Technology by employees of the Federal Government in the course of
// their official duties. Pursuant to Title 17 Section 105 of the United
// States Code this software is not subject to copyright protection and
// is in the public domain.
//
// The NIST Data Flow System (NDFS) is an experimental system and is
// offered AS IS. NIST assumes no responsibility whatsoever for its use
// by other parties, and makes no guarantees and NO WARRANTIES, EXPRESS
// OR IMPLIED, about its quality, reliability, fitness for any purpose,
// or any other characteristic.
//
// We would appreciate acknowledgement if the software is used.
//
// This software can be redistributed and/or modified freely provided
// that any derivative works bear some notice that they are derived from
// it, and any modified versions bear some notice that they have been
// modified from the original."
//
//=========================================================================



// "Multi Channel Audio" (MCA) "Make It Happen" (MIH) program

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <getopt.h>

#define MAX_CHANNELS 256

int bytes;
int channels;
int extract;

int transform;
unsigned char transform_buf[4];

char outfile[1024];

int absmax[32];

int file_ready;
char *current_if;

int ofiles_ready;
int ofile_h[MAX_CHANNELS];
char *ofile[MAX_CHANNELS];

int channels_mem;
int channel_mem;
unsigned char *channel_mem_m[MAX_CHANNELS];
int channel_mem_c[MAX_CHANNELS];

int extract_channel[MAX_CHANNELS];

int nowrite;

int minMax;
int minMax_ready;
int file_min[MAX_CHANNELS], file_max[MAX_CHANNELS];
int global_min[MAX_CHANNELS], global_max[MAX_CHANNELS];
int File_Min, File_Max;
int Global_Min, Global_Max;

int usedbitstat;
long long int fbit[MAX_CHANNELS][32];
long long int tfbit;
long long int cbit[MAX_CHANNELS][32];
long long int tcbit;
int cumulres;

const char *progname, *shortname;


/****************************************/

int debug = 0;

void dprint(char *fmt, ...)
{
  va_list argp;

  if (debug == 0) return;

  fprintf(stderr, "[DEBUG] ");
  va_start(argp, fmt);
  vfprintf(stderr, fmt, argp);
  va_end(argp);
  fprintf(stderr, "\n");
}

/****************************************/

void eq(char *text)
{ // this one _really_ quits
  printf("%s", text);
  exit(1);
}

/****************************************/

char *get_shortname(char *longname)
{
  char *shortname;

  shortname = strrchr(longname, '/');
  if(!shortname)
    shortname = longname;
  else
    shortname++;

  return (shortname);
}

/****************************************/

static void print_usage(FILE *stream)
{
  fprintf(stream, "\nUsage : %s [options] file(s)\n"
          "  -h         --help          Gives this message\n"
          "  -b b       --bytes b       Process source file with raw data of b bytes size\n"
	  "  -c c       --channels c    Max number of channels in file (required for RAW files\n"
	  "  -e e       --extract e     Channel to extract (':' separated list / '0' for all). Since in some cases, the total number of microphone is not known until the input file is read, this will accept values out of range, but will not process them.\n"
	  "  -o o       --output o      File name base to extract channel(s)/data to. If one is specified, will use this name to extract all single channel data to; ie concatenate from input file(s) to output file. If all channels are being split (on a multi channel file), each channel file will have its number added to the prefix. If no file is specified, the program will use the input file(s) base, prefix them with 'raw_' and suffix them with '.raw', using the same channel format if multiple output channels are specified.\n"
	  "  -t t       --transform t   Transform data from 'b' input bytes to 't' output bytes\n"
	  "  -m         --minMax        Will display selected channel's min/Max as well as the global min/Max value including the number of bytes used\n"
	  "  -U         --Usedbitstat   Evaluate the bit usage repartition for each individual channel\n"
	  "  -C         --Cumulativeres Print cumulated results for 'usebitstat'\n"
	  "  -n         --nowrite       Ask that no output file be generated (usualy used in conjunction with options such as 'minMax')\n"
	  "  -d         --dump d        Specify the amount of memory (in MB) allocated for extracted data/channels before dumping them to disk (default: %d MB)\n"
	  "\n"
	  "The program will extract raw data from either a RAW file (no header and packed data) to easily extract multi channel audio to disk\n"
	  "\n"
	  "\n ! WARNING ! When performing a data conversion (-t) or any evaluation of the data (-m, -u): THIS PROGRAM HAS BEEN DESIGNED TO WORK WITH LINUX ENDIANNESS DATA FILES\n"
          ,progname, channels_mem);
}

int options(int argc, char ***argv)
{
  static struct option optlist[] = {
    { "help",     0, 0, 'h' },
    { "bytes",    1, 0, 'b' },
    { "channels", 1, 0, 'c' },
    { "extract",  1, 0, 'e' },
    { "output",   1, 0, 'o' },
    { "transform", 1, 0, 't' },
    { "minMax",   0, 0, 'm' },
    { "Usedbitstat",   0, 0, 'U' },
    { "Cumulativeres", 0, 0, 'C' },
    { "nowrite",  0, 0, 'n' },
    { "dump",     1, 0, 'd' },
    { 0,          0, 0, 0   }
  };
  
  int usage = 0, finish = 0, error = 0;
  int ra = 1; // Among removed arguments is always argv[0]

  dprint("options");
 
  for(;;) {
    int opt = getopt_long(argc, *argv, "hb:c:e:o:t:mUCnd:i:u", optlist, 0);
    if(opt == EOF)
      break;
    switch(opt) {
    case 'h':
      usage = 1;
      finish = 1;
      error = 0;
      ra++;
      break;
    case 'b':
      bytes = atoi(optarg);
      ra += 2;
      break;
    case 'c':
      channels = atoi(optarg);
      ra += 2;
      break;
    case 'e':
      {
	char tmp[2048];
	int kp, i, v;
	char *ptr1, *p1;
	char *ptr2, *p2;
	int a, b, c;

	strcpy(tmp, optarg);
	
	kp = 1;
	ptr1 = strtok_r(tmp, ":", &p1);
	while (kp) {
	  if (ptr1 == NULL) {
	    kp = 0;
	    continue;
	  }
	  ptr2 = strtok_r(ptr1, "-", &p2);
	  a = atoi(ptr2);
	  ptr2 = strtok_r(NULL, "-", &p2);
	  if (ptr2 == NULL)
	    b = a;
	  else
	    b = atoi(ptr2);
	  if (b < a) {
	    c = b;
	    b = a;
	    a = c;
	  }
	  for (v = a; v <= b; v++) {
	    dprint("v=%d", v);
	    if ((v < 0) || (v > MAX_CHANNELS)) {usage=1;finish=1;fprintf(stderr,"Error : wrong microphone number (%d)\n", v);break;}
	    if (v == 0) {
	      for (i = 0; i < MAX_CHANNELS; i++) extract_channel[i]++;
	    } else {
	      if (extract_channel[v-1] > 0) {usage=1;finish=1;fprintf(stderr,"Error : microphone number %d appears more than once\n", v);break;}
	      extract_channel[v-1]++;
	    }
	  }
	  ptr1 = strtok_r(NULL, ":", &p1);
	}
	extract = 1;
	ra += 2;
	break;
      }
    case 'o':
      strcpy(outfile, optarg);
      ra += 2;
      break;
    case 't':
      transform = atoi(optarg);
      ra += 2;
      break;
    case 'm':
      minMax = 1;
      ra += 1;
      break;
    case 'U':
      usedbitstat = 1;
      ra += 1;
      break;
    case 'C':
      cumulres = 1;
      ra += 1;
      break;
    case 'n':
      nowrite = 1;
      ra += 1;
      break;
    case 'd':
      channels_mem = atoi(optarg);
      ra += 2;
      break;
    case '?':
    case ':':
      usage = 1;
      finish = 1;
      error = 1;
      break;
    default:
      abort();
    }
    if (finish)
      break;
  }

  if (usage)
    print_usage(error ? stderr : stdout);

  if (finish)
    exit(error);

  *argv += optind;
  
  return ra;
}

/****************************************/

void init_absmax()
{
  int i;

  dprint("init_absmax");
  absmax[31] = 0x7FFFFFFF;
  for (i = 30; i > 0; i--)
    absmax[i] = absmax[i+1]/2;
}

/*****/

void init_ofilearray()
{
  int i;
  int cu;
  int tc;

  cu = 0;
  for (i = 0; i < channels; i++) if (extract_channel[i] > 0) cu++; 

  if (cu == 0)
    eq("No valid channels were selected for extraction, aborting\n");

  if (transform != -1)
    tc = transform;
  else
    tc = bytes;

  // Integer arithmetic (part 1)
  channel_mem = (channels_mem * 1024 * 1024);
  channel_mem /= cu;
  channel_mem /= cu * tc; // Number of elements

  dprint("init_ofilearray");
  for (i = 0; i < channels; i++) {
    if (extract_channel[i] > 0) {
      ofile[i] = malloc(2048);
      if (! ofile[i])
	eq("Error: Could not allocate memory for output fle name, aborting\n");
      channel_mem_m[i] = calloc(channel_mem, cu*tc);
      if (! channel_mem_m[i])
	eq("Error: Could not allocate memory for channel temporary buffer\n");
      channel_mem_c[i] = 0;
    }
  }

  // Integer arithmetic (part 2)
  channel_mem *= cu * tc; // Total size in bytes

}

/****************************************/

int maxabs(int a, int b)
{
  // This is required to try to avoid some buffer overflow issues
  if (a < 0) { a++; a = -a; }
  if (b < 0) { b++; b= - b; }

  if (a > b) return (a);
  return (b);
}

/*****/

int usedbits(int val)
{
  int i, used;
  used = 0;
  if (val < 0) { val++; val = -val; }
  for (i = 1; used == 0 && i < 32 ; i++) {
    if (val <= absmax[i]) used = i + 1;
  }

  return used;
}

/*****/

static inline int totalbits(int a) { return (8*a); }

/*****/

void display_channel_minMax(int channel)
{
  int min, max;

  if (minMax == 0) return;

  min = file_min[channel];
  max = file_max[channel];

  printf("Channel '%d' (out of %d) min/Max: %d/%d [%d bits used out of %d]\n", channel + 1, channels, min, max, usedbits(maxabs(min, max)), totalbits(bytes));
}

/*****/

void display_file_minMax(char *name)
{
  if (minMax == 0) return;

  printf("File '%s' min/Max: %d/%d [%d bits used out of %d]\n", name, File_Min, File_Max, usedbits(maxabs(File_Min, File_Max)), totalbits(bytes));
}  

/*****/

void display_global_channel_minMax(int channel)
{
  int min, max;

  if (minMax == 0) return;

  min = global_min[channel];
  max = global_max[channel];

  printf("Global Channel '%d' (out of %d) min/Max: %d/%d [%d bits used out of %d]\n", channel + 1, channels, min, max, usedbits(maxabs(min, max)), totalbits(bytes));
}
/*****/

void display_global_minMax()
{
  if (minMax == 0) return;

  printf("Global min/Max: %d/%d [%d bits used out of %d]\n", Global_Min, Global_Max, usedbits(maxabs(Global_Min, Global_Max)), totalbits(bytes));
}  

/************************************************/

void percentdisplay_distr(char *text, long long *tmp, long long divid)
{
  int i;
  float per[32];
  long long addinc;

  if (usedbitstat == 0) return;

  addinc = 0;
  for (i = totalbits(bytes); i > 0; i--) {
    per[i] = ((float) tmp[i] / (float) divid) * 100.0;
    if ((cumulres == 1) && (i != totalbits(bytes))) per[i] += per[i+1];
    addinc += tmp[i];
  }
  if (addinc != divid){
    printf("[%lld/%lld]\n", addinc, divid);
    eq("Error: total elements checksum does not match, aborting\n");
  }

  printf("%s", text);
  for (i = totalbits(bytes); i > 0; i--)
    if (tmp[i] > 0) printf("[%d:%05.2f%%]", i, per[i]);
  printf("\n");
}
 
/*****/

void display_file_channelsdistr()
{
  long long tmp[32];
  long long tmpfbit;
  int i, j;

  if (usedbitstat == 0) return;

  tmpfbit = 0;
  for (i = 0; i < 32; i++) tmp[i] = 0;
  for (i = 0; i < 32; i++) {
    for (j = 0; j < channels; j++) {
      tmp[i] += fbit[j][i];
      tmpfbit += fbit[j][i];
    }
  }

  percentdisplay_distr("File Channels Distribution : ", tmp, tmpfbit);
}

/*****/

void display_file_channeldistr(int c)
{
  long long tmp[32];
  int i;
  char tmpt[80];

  if (usedbitstat == 0) return;

  for (i = 0; i < 32; i++) tmp[i] = fbit[c][i];
  sprintf(tmpt, "Channel \'%d\' Distribution : ", c+1);

  percentdisplay_distr(tmpt, tmp, tfbit);
}

/*****/

void display_global_channelsdistr()
{
  long long tmp[32];
  long long tmpcbit;
  int i, j;

  if (usedbitstat == 0) return;
  
  tmpcbit = 0;
  for (i = 0; i < 32; i++) tmp[i] = 0;
  for (i = 0; i < 32; i++) {
    for (j = 0; j < channels; j++) {
      tmp[i] += cbit[j][i];
      tmpcbit += cbit[j][i];
    }
  }
  percentdisplay_distr("Global Channels Distribution : ", tmp, tmpcbit);
}

/*****/

void display_global_channeldistr(int c)
{
  long long tmp[32];
  int i;
  char tmpt[80];

  if (usedbitstat == 0) return;

  for (i = 0; i < 32; i++) tmp[i] = cbit[c][i];
  sprintf(tmpt, "Global Channel \'%d\' Distribution : ", c+1);

  percentdisplay_distr(tmpt, tmp, tcbit);
}

/************************************************/

unsigned char* transformit(unsigned char *buffer)
{
  // The transformation are "bit" ready, not "value" ready
  // ie in 24 bits the last 8 bits are set to 0 -> 'int' value is wrong
  char *otab = transform_buf;

  //  dprint("-> transformit");

  if (transform == -1)
    eq("Step into transform mechanism without a valid transformation !\n");
  
  if (transform == 4) { // X -> 32
    if (bytes == 2) {
      otab[0] = '\0';
      otab[1] = '\0';
      otab[2] = buffer[0];
      otab[3] = buffer[1];
    } else if (bytes == 3) {
      otab[0] = '\0';
      otab[1] = buffer[0];
      otab[2] = buffer[1];
      otab[3] = buffer[2];
    } else if (bytes == 4) {
      otab[0] = buffer[0];
      otab[1] = buffer[1];
      otab[2] = buffer[2];
      otab[3] = buffer[3];
    } else
      eq("We do not handle that many input bytes, sorry\n");
  } else if (transform == 3) { // X -> 24
    otab[3] = '\0';
    if (bytes == 2) {
      otab[0] = '\0';
      otab[1] = buffer[0];
      otab[2] = buffer[1];
    } else if (bytes == 3) {
      otab[0] = buffer[0];
      otab[1] = buffer[1];
      otab[2] = buffer[2];
    } else if (bytes == 4) {
      otab[0] = buffer[1];
      otab[1] = buffer[2];
      otab[2] = buffer[3];
    } else
      eq("We do not handle that many input bytes, sorry\n");
  } else if (transform == 2) { // X -> 16
    otab[2] = '\0';
    otab[3] = '\0';
    if (bytes == 2) {
      otab[0] = buffer[0];
      otab[1] = buffer[1];
    } else if (bytes == 3) {
      otab[0] = buffer[1];
      otab[1] = buffer[2];
    } else if (bytes == 4) {
      otab[0] = buffer[2];
      otab[1] = buffer[3];
    } else
      eq("We do not handle that many input bytes, sorry\n");
  } else    
    eq("We do not handle that many output bytes, sorry\n");
  
  return (otab);
}

/**********/

int evaluate_bytes(char *buffer)
{
  int value;

  if (bytes == 2) {
    char otab[2];
    short *p = (short *) &otab;
    otab[0] = buffer[0];
    otab[1] = buffer[1];
    value = (int) *p;
  } else if (bytes == 3) {
    char otab[4];
    int *p = (int *) &otab;
    otab[0] = '\0';
    otab[1] = buffer[0];
    otab[2] = buffer[1];
    otab[3] = buffer[2];
    *p = *p / 256;
    value = *p;
  } else if (bytes == 4) {
    char otab[4];
    int *p = (int *) &otab;
    otab[0] = buffer[0];
    otab[1] = buffer[1];
    otab[2] = buffer[2];
    otab[3] = buffer[3];
    value = *p;
  } else {
    printf("We do not handle that many bytes, sorry\n");
    exit(-1);
  }
  
  return value;
}

/****************************************/

void dump_channel_mem2file(int c)
{
  int tmp;

  dprint("dump_channel_mem2file (channel: %d / size: %d bytes)", c, channel_mem_c[c]);

  if (channel_mem_c[c] == 0) return; // (safeguard)

  tmp = write(ofile_h[c], (void *) channel_mem_m[c], channel_mem_c[c]);
  if (tmp != channel_mem_c[c])
    eq("Error: Could not write channel data to file, aborting\n");

  channel_mem_c[c] = 0;
}

/****************************************/

void close_ofiles()
{
  int i;

  if (nowrite == 1) return;

  for (i = 0; i < channels; i++) 
    if ((extract_channel[i] > 0) && (ofile_h[i] > 0)) {
      dump_channel_mem2file(i);
      close(ofile_h[i]);
    }
}

/*****/

void ready_minMax_UB()
{
  int i, j;

  if ((minMax == 0) && (usedbitstat == 0)) return;
  // minMax and UB are related so we initialize both at the same time
  // About minMax and UB, it is to be said that their "functions" are
  // "aware" of their usage (especialy display ones)

  if ((file_ready == 1) && (minMax_ready > 0)) return;

  if (minMax_ready == 0) {
    for (i = 0; i < channels; i++) {
      global_min[i] = 0;
      global_max[i] = 0;
      if (usedbitstat == 1) for (j = 0; j < 32; j++) cbit[i][j] = 0;
    }
    Global_Min = 0;
    Global_Max = 0;
    if (usedbitstat == 1) tcbit = 0;
  }

  for (i = 0; i < channels; i++) {
    file_min[i] = 0;
    file_max[i] = 0;
    if (usedbitstat == 1) for (j = 0; j < 32; j++) fbit[i][j] = 0;
  }
  File_Min = 0;
  File_Max = 0;
  if (usedbitstat == 1) tfbit = 0;

  minMax_ready++;
}

/*****/

void ready_ofiles()
{
  char raw_of[1024];
  char *tmpc;
  char dir[1024], prefix[1024], fname[1024], ext[1024];
  int i;

  if (nowrite == 1) return;

  if (ofiles_ready == 0) init_ofilearray();

  if ((file_ready == 1) && (ofiles_ready > 0)) return;

  // Concat mode (do not close the already opened output files)
  if ((outfile[0] != '\0') && (ofiles_ready > 0)) return;

  // Close all the previously opened files
  if (ofiles_ready > 0) close_ofiles();

  // First, strip the output file information
  strcpy(raw_of, (outfile[0] != '\0') ? outfile : current_if);

  // Get the file directory base
  tmpc = strrchr(raw_of, '/');
  if (! tmpc) {
    dir[0] = '\0';
  } else {
    *tmpc = '\0';
    strcpy(dir, raw_of);
    strcat(dir, "/");

    tmpc++;
    strcpy(raw_of, tmpc);
  }
  
  // Extract the filename and extension
  tmpc = strrchr(raw_of, '.');
  if (! tmpc) {
    strcpy(fname, raw_of);
    ext[0] = '\0';
  } else {
    *tmpc = '\0';
    strcpy(fname, raw_of);
    
    tmpc++;
    strcpy(ext, tmpc);
  }
  
  // Add its dot to the extension (if any)
  if ((ext[0] != '\0') && (ext[0] != '.')) {
    char tmp[1024];
    strcat(tmp, ".");
    strcat(tmp, ext);
    strcpy(ext, tmp);
  }

  // Make a default prefix
  prefix[0] = '\0';

  // We now have: dir, fname, ext
  // Let us open the files
  for (i = 0; i < channels; i++) {
    if (extract_channel[i] == 1) {
      if (outfile[0] == '\0') {// 1 output file per input file
	strcpy(ext, ".raw");
	if (channels > 1)  // Add the channel number to the prefix
	  sprintf(prefix, "raw_%03d_", i+1);
	else
	  strcpy(prefix, "raw_");
      } else { // 1 output file for all input files
	if (channels > 1) // Add the channel number to the prefix
	  sprintf(prefix, "%03d_", i+1);
      }
      // Now we have the base for the output file name, create it
      sprintf(ofile[i], "%s%s%s%s", dir, prefix, fname, ext);
      ofile_h[i] = open(ofile[i], O_RDWR|O_CREAT|O_TRUNC, 0666);
      if (ofile_h[i] < 0) 
	eq("Error: Could not create output file, aborting\n");
    }
  }
  
  ofiles_ready++;
}

/*****/

void ready_file()
{
  // The first common part after opening the current input file
  // and getting the first burst of data from it

  if (file_ready == 1) return; // (safeguard)
  
  ready_ofiles();
  ready_minMax_UB();

  file_ready = 1;
}

/****************************************/

void endfile_ofiles()
{
  if (nowrite == 1) return;

  // Closing/opening of files is taken care in the 'ready_ofiles()' function
}

/*****/

void endfile_minMax_UB()
{
  int i, j;

  if (minMax != 0) {
    for (i = 0; i < channels; i++) {
      if (file_min[i] < global_min[i]) global_min[i] = file_min[i];
      if (file_max[i] > global_max[i]) global_max[i] = file_max[i];
      
      if (file_min[i] < File_Min) File_Min = file_min[i];
      if (file_max[i] > File_Max) File_Max = file_max[i];
    }
    
    if (File_Min < Global_Min) Global_Min = File_Min;
    if (File_Max > Global_Max) Global_Max = File_Max;
  }

  display_file_minMax(current_if);
  if (usedbitstat == 1) {
    display_file_channelsdistr(current_if);
      tcbit += tfbit;
  }
  
  for (i = 0; i < channels; i++) {
    if (extract_channel[i] == 1) {
      display_channel_minMax(i);
      if (usedbitstat == 1) {
	display_file_channeldistr(i);
	for (j = 0; j < 32; j++) cbit[i][j] += fbit[i][j];
      }
    }
  }
}

/*****/

void end_infile(int infile)
{
  close(infile);

  endfile_ofiles();
  endfile_minMax_UB();

  file_ready = 0;
}

/****************************************/

void data_processor(unsigned char* data, int length)
{
  int i, j;
  unsigned char *buffer;

  dprint("-> data_processor (length: %d)", length);

  if (file_ready == 0) ready_file();
  
  for (i = 0; i < length; i += bytes*channels) {
    for (j = 0; j < channels; j++) {
      if (extract_channel[j] == 0) continue;
      
      buffer = data;
      buffer += i;
      buffer += j*bytes;

      if (nowrite == 0) { // We can write to file
	int k;
	unsigned char *ccurs;
	unsigned char *tmpc;
	int tmpv;

	ccurs = channel_mem_m[j];
	ccurs += channel_mem_c[j];
	
	if (transform != -1) {
	  tmpc = transformit(buffer);
	  tmpv = transform;
	} else {
	  tmpc = buffer;
	  tmpv = bytes;
	}

	for (k = 0; k < tmpv; k++) ccurs[k] = tmpc[k];
	channel_mem_c[j] += tmpv;
	if (channel_mem_c[j] >= channel_mem)
	  dump_channel_mem2file(j);
      }
      
      if ((minMax != 0) || (usedbitstat == 1)) { // Compute min/Max + UB
	int value;
	value = evaluate_bytes(buffer);
	if (usedbitstat == 1) fbit[j][usedbits(value)]++;
	if (minMax != 0) {
	  if (value > file_max[j]) file_max[j] = value;
	  if (value < file_min[j]) file_min[j] = value;
	}
      }
    }
    // we always treat as many bits from each channels...
    tfbit++;
  }
}

/****************************************/

void end_all(int file_count)
{
  // Close all open output files (will also finish writting data)
  close_ofiles();

  // Now if the number of input file is > 1, print Global min/Max + used bits
  if (file_count > 1) {
    int i;

    printf("\n");
    for (i = 0; i < channels; i++) {
      if (extract_channel[i] > 0) {
	display_global_channel_minMax(i);
	display_global_channeldistr(i);
      }
    }
    display_global_minMax();
    display_global_channelsdistr();
  }
}

/***********************************************************************/

int main(int argc, char **argv)
{
  int file_count;
  
  int infile;
  int i;

  char msg[1024]; 
  
  dprint("START");
  
  progname = argv[0];
  shortname = strrchr(progname, '/');
  if(!shortname)
    shortname = progname;
  else
    shortname++;

  bytes = -1;
  channels = -1;
  extract = 0;
  transform = -1;
  minMax = 0;
  for (i = 0; i < MAX_CHANNELS; i++) {
    extract_channel[i] = 0;
    ofile_h[i] = -1;
    ofile[i] = NULL;
  }
  outfile[0] = '\0';
  nowrite = 0;
  channels_mem = 64;
  usedbitstat = 0;
  cumulres = 0;

  argc -= options(argc, &argv);
  if (argc == 0) {
    printf("No input file(s) specified, aborting\n");
    print_usage(stderr);
    exit(1);
  }
  
  // Quick sanity check
  if ((bytes == 0) || (bytes > 4))
    eq("Sorry, we do not currently handle that many bytes, aborting\n");
  if ((channels == 0) || (channels > MAX_CHANNELS))
    eq("Sorry, we do not currently handle that many channels, aborting\n");
  if ((nowrite == 1) && (transform != -1))
    eq("Sorry, the 'nowrite' option is not authorized when 'transform' is selected\n");
  if ((nowrite == 1) && (minMax == 0))
    eq("Sorry, 'nowrite' still requires this program to 'do' something, aborting\n");
  if (channels_mem > 256) {
    printf("Warning, amount of 'dump' memory requested over the maximum authorized value, using default\n");
    channels_mem = 64;
  }
  if ((minMax == 1) && (extract == 0))
    eq("In order to extract some min/Max values, some channels have to be selected\n");
  if ((usedbitstat == 0) && (cumulres == 1))
    eq("Error: Cumulative results mode can only be selected when 'usebitstat' is selected, aborting\n");
  
  init_absmax();
  
  ofiles_ready = 0;
  minMax_ready = 0;
  file_ready = 0;
  for (file_count = 0; file_count < argc; file_count++) {

    current_if = argv[file_count];
    
    infile = open(current_if, O_RDONLY);
    if (infile < 0) {
      printf("Warning: Can't open input file %s, skipping\n", current_if);
      continue;
    }
    
    printf("Processing Input file (%d/%d): %s\n"
	   , file_count+1, argc, current_if);

    {
      unsigned char *buffer = NULL;
      int keepreading;
      int toread;
      
      dprint("RAW");
      // Quick sanity check
      if (bytes == -1)
	eq("Error: when working with RAW files, a byte length must be specified, aborting\n");
      if (channels == -1)
	eq("Error: when working with RAW files, the maximum number of channels must be specified, aborting\n");
      
      // Integer arithmetic (part 1) (we read the closest to 4MB at a time)
      toread = (4 * 1024 * 1024 );
      toread /= bytes * channels; // Number of elements
            
      buffer = (unsigned char *) calloc(toread, bytes*channels);
      if (buffer == NULL)
	eq("Error: Could not allocate memory for read buffer, aborting\n");
      
      // Integer arithmetic (part 2)
      toread *= bytes * channels; // Size in bytes

      keepreading = 1;
      while ( keepreading ) {
	int act_read;

	act_read = read(infile, buffer, toread);
	
	if (act_read != toread)
	  keepreading = 0;
	
	data_processor(buffer, act_read);
      }
    }
    
    end_infile(infile);
  }
  
  end_all(file_count);

  // I am not un-malloc-ing the memory, and it is bad, I know :)
  return (0);
}
