'
'  Multimodal Biometric Applicaiton Resource Kit (MBARK)
'
'  File author(s):
'       Ross J. Micheals (rossm@nist.gov)
'       Kayee Kwong (kayee@nist.gov)
'
' 
' | LICENSE & DISCLAIMER                                                                                |
' |                                                                                                     |
' | This software was developed at the National Institute of Standards and Technology (NIST) by         |
' | employees of the Federal Government in the course of their official duties. Pursuant to title 17    |
' | Section 105 of the United States Code. This software is not subject to copyright protection and     |
' | is in the public domain. NIST assumes no responsibility whatsoever for use by other parties of      |
' | its source code or open source server, and makes no guarantees, expressed or implied, about its     |
' | quality, reliability, or any other characteristic.                                                  |
' |                                                                                                     |
' | Specific hardware and software products identified in this open source project were used in order   |
' | to perform technology transfer and collaboration. In no case does such identification imply         |
' | recommendation or endorsement by the National Institute of Standards and Technology, nor            |
' | does it imply that the products and equipment identified are necessarily the best available for the |
' | purpose.                                                                                            |
' 

Option Strict On

Imports System
Imports System.Globalization
Imports System.Collections.Specialized
Imports System.Xml
Imports System.Xml.Serialization

Namespace Mbark.Sensors

    <XmlType("RuntimeSensorTaskFactoryCollection")> Public Class SensorTaskFactoryCollection
        Implements ICollection

        Private mActiveTask As SensorTask
        Private mFactories As New ArrayList
        Private mAllTasks As New SensorTaskCollection
        Private mTasksOfSensor As New SensorTaskCollectionDictionary
        Private mSensors As New SensorCollection

        Friend mCurrentConditions As New ConditionCollection
        Friend mCurrentInaccessibleBodyParts As New BodyParts

        Public Function SensorConfigurationTypes() As StringCollection
            Dim types As New StringCollection
            For i As Integer = 0 To mFactories.Count - 1
                Dim typeName As String = DirectCast(mFactories(i), SensorTaskFactory).SensorConfiguration.GetType().FullName
                If types.IndexOf(typeName) = -1 Then types.Add(typeName)
            Next
            Return types
        End Function

        Friend Function CreateDefinition() As SensorTaskFactoryCollectionDefinition
            Dim newDef As New SensorTaskFactoryCollectionDefinition

            ReDim newDef.SensorDefinitions(mSensors.Count - 1)
            For i As Integer = 0 To mSensors.Count - 1
                newDef.SensorDefinitions(i) = SensorTypeDefinition.CreateDefinition(mSensors(i))
            Next

            newDef.UpdateCapacity(mFactories.Count)
            For i As Integer = 0 To mFactories.Count - 1
                newDef.FactoryDefinitions.Add(DirectCast(mFactories(i), SensorTaskFactory).CreateDefinition())
            Next

            newDef.CurrentConditions = mCurrentConditions.CreateDefinition()
            newDef.CurrentInaccessibleBodyParts = mCurrentInaccessibleBodyParts.DeepCopy

            Return newDef
        End Function

        Public Shared Function LoadFromXmlFile(ByVal fileName As String) As SensorTaskFactoryCollection

            If fileName = String.Empty Then Return New SensorTaskFactoryCollection

            Dim xml As New XmlDocument
            xml.Load(fileName)


            Dim definitionNodes As XmlNodeList = xml.SelectNodes("//SensorDefinitions/SensorType")

            ' Get all the sensors with configuration classes, and manually create type definitions for them
            Dim configurationTypes As New ArrayList
            For i As Integer = 0 To definitionNodes.Count - 1
                Dim hasConfigurationClass As Boolean = CType(definitionNodes(i).SelectNodes("HasConfigurationClass")(0).InnerText, Boolean)
                If hasConfigurationClass Then
                    Dim configurationClassname As String = definitionNodes(i).SelectNodes("ConfigurationClassName")(0).InnerText
                    Dim libraryFileName As String = definitionNodes(i).SelectNodes("LibraryFileName")(0).InnerText

                    If configurationClassname.Length = 0 Then Throw New BadDefinitionException
                    If libraryFileName.Length = 0 Then Throw New BadDefinitionException


                    Dim definition As New TypeDefinition
                    definition.LibraryFileName = libraryFileName
                    definition.ClassName = configurationClassname
                    configurationTypes.Add(definition)
                End If
            Next


            ' Make an array of configuration class exemplars for the serializers to grok upon construction
            Dim additionalTypes(configurationTypes.Count - 1) As Type
            For i As Integer = 0 To configurationTypes.Count - 1
                Dim def As TypeDefinition = DirectCast(configurationTypes(i), TypeDefinition)
                Try
                    additionalTypes(i) = def.InstantiateObject().GetType()
                Catch ex As IO.FileNotFoundException
                    ' If there is no such file, then try the 'Sensors' subdirectory 
                    def.Subdirectory = "Sensors"
                    additionalTypes(i) = def.InstantiateObject().GetType()
                End Try
            Next


            Dim factories As SensorTaskFactoryCollection


            Dim definitionFile As New IO.FileStream(fileName, IO.FileMode.Open)
            Dim serializer As New XmlSerializer(GetType(SensorTaskFactoryCollectionDefinition), additionalTypes)
            Dim factoriesDefinition As SensorTaskFactoryCollectionDefinition = _
                DirectCast(serializer.Deserialize(definitionFile), SensorTaskFactoryCollectionDefinition)
            factories = factoriesDefinition.CreateSensorTaskFactories()

            ' Restore the active task (if there was one)
            For i As Integer = 0 To factories.Count - 1
                For j As Integer = 0 To factories(i).GeneratedTasks.Count - 1
                    Dim task As SensorTask = factories(i).GeneratedTasks(j)
                    If task.Status = SensorTaskStatus.Active Then factories.mActiveTask = task
                Next
            Next
            definitionFile.Close()

            Return factories
        End Function

        Public Sub SaveAsXmlFile(ByVal fileName As String)


            Dim uniqueConfigurationType As New ArrayList
            Dim exampleConfigurations As New ArrayList
            For i As Integer = 0 To Count - 1
                If Not Factory(i).SensorConfiguration Is Nothing Then
                    Dim typeName As String = Factory(i).SensorConfiguration.GetType().FullName
                    If uniqueConfigurationType.IndexOf(typeName) = -1 Then
                        uniqueConfigurationType.Add(typeName)
                        exampleConfigurations.Add(Factory(i).SensorConfiguration)
                    End If
                End If

            Next

            Dim typeDefs(uniqueConfigurationType.Count - 1) As TypeDefinition
            Dim types(uniqueConfigurationType.Count - 1) As Type

            For i As Integer = 0 To types.Length - 1
                typeDefs(i) = TypeDefinition.Create(exampleConfigurations(i))
                types(i) = exampleConfigurations(i).GetType
            Next

            Dim serializer As New XmlSerializer(GetType(SensorTaskFactoryCollectionDefinition), types)
            Dim factoryDefinitionFile As New IO.FileStream(fileName, IO.FileMode.OpenOrCreate)
            factoryDefinitionFile.SetLength(0)
            serializer.Serialize(factoryDefinitionFile, CreateDefinition)
            factoryDefinitionFile.Close()

        End Sub

        Public ReadOnly Property CurrentConditions() As ConditionCollection
            Get
                Return mCurrentConditions
            End Get
        End Property

        Public ReadOnly Property CurrentInaccessibleBodyParts() As BodyParts
            Get
                Return mCurrentInaccessibleBodyParts
            End Get
        End Property

        Public ReadOnly Property Sensors() As SensorCollection
            Get
                Return mSensors
            End Get
        End Property



        Public ReadOnly Property ActiveTask() As SensorTask
            Get
                Return mActiveTask
            End Get
        End Property

        Public Sub Add(ByVal factory As SensorTaskFactory)
            If factory Is Nothing Then Throw New ArgumentNullException("factory")
            mFactories.Add(factory)
            factory.WireParentSensorTaskFactories(Me)
        End Sub

        Public Sub PopulateTargets()
            For i As Integer = 0 To mFactories.Count - 1
                Factory(i).PopulateTargets(CurrentConditions, CurrentInaccessibleBodyParts)
            Next
            PopulateAllTasksList()
        End Sub

        Public Sub TrimExcessTasks()

            For i As Integer = 0 To mFactories.Count - 1
                Factory(i).TrimExcessTasks()
            Next
        End Sub

        Default Public Property Factory(ByVal index As Integer) As SensorTaskFactory
            Get
                Return DirectCast(mFactories(index), SensorTaskFactory)
            End Get
            Set(ByVal value As SensorTaskFactory)
                mFactories(index) = value
            End Set
        End Property



        Public Function IndexOf(ByVal factory As SensorTaskFactory) As Integer
            Return mFactories.IndexOf(factory)
        End Function

        Public ReadOnly Property AllTasks() As SensorTaskCollection
            Get
                Return mAllTasks
            End Get
        End Property

        Private Sub AutoSelectActiveTask(ByVal currentConditions As ConditionCollection)

            If Not ActiveTask Is Nothing AndAlso ActiveTask.IsActivatable AndAlso _
               currentConditions.EqualsSubset(ActiveTask.Conditions) Then
                ' If the active task is activatable, and the first of all activatable tasks, then stay here
                Return
            End If

            ' If we've skipped to this task, then stay here
            If Not ActiveTask Is Nothing AndAlso ActiveTask.SkipDestination AndAlso ActiveTask.IsActivatable Then Return

            Dim newActiveTask As SensorTask




            ' Seek the next activatable task *for this sensor* given the current conditions
            If newActiveTask Is Nothing AndAlso Not ActiveTask Is Nothing Then
                For i As Integer = 0 To mTasksOfSensor(ActiveTask.Sensor).Count - 1

                    Dim task As SensorTask = mTasksOfSensor(ActiveTask.Sensor)(i)
                    Dim conditionsOK As Boolean = currentConditions.EqualsSubset(task.Conditions)
                    If task.IsActivatable AndAlso conditionsOK Then
                        newActiveTask = task
                        Exit For
                    End If
                Next
            End If

            ' As a last resort, select a task for this sensor ignoring the current conditions
            If newActiveTask Is Nothing AndAlso Not ActiveTask Is Nothing Then
                For i As Integer = 0 To mTasksOfSensor(ActiveTask.Sensor).Count - 1
                    Dim task As SensorTask = mTasksOfSensor(ActiveTask.Sensor)(i)
                    Dim conditionsOK As Boolean = currentConditions.EqualsSubset(task.Conditions)
                    If task.IsActivatable AndAlso Not conditionsOK Then
                        newActiveTask = task
                        Exit For
                    End If
                Next
            End If

            ' If the very next task is activatable, then choose that one
            Dim currentTaskIndex As Integer = mAllTasks.IndexOf(ActiveTask)
            Dim notLastTask As Boolean = currentTaskIndex < (mAllTasks.Count - 1)
            Dim nextTaskOK As Boolean = notLastTask AndAlso mAllTasks(currentTaskIndex + 1).IsActivatable

            If newActiveTask Is Nothing AndAlso Not ActiveTask Is Nothing AndAlso notLastTask AndAlso nextTaskOK Then
                newActiveTask = mAllTasks(currentTaskIndex + 1)
            End If


            ' If we can't migrate within the same sensor, then pick the first activatable task
            ' among the remaining tasks
            If newActiveTask Is Nothing Then
                For i As Integer = 0 To mAllTasks.Count - 1
                    If mAllTasks(i).IsActivatable Then
                        newActiveTask = mAllTasks(i)
                        Exit For
                    End If
                Next
            End If

            ' If the task has changed, the task can no longer a skip destination
            If Not ActiveTask Is Nothing AndAlso Not newActiveTask Is ActiveTask Then
                mActiveTask.SkipDestination = False
            End If

            ' If we've picked a new active task, then activate it!
            If Not newActiveTask Is Nothing Then
                MigrateActiveTask(newActiveTask)
            Else
                mActiveTask = Nothing
            End If




        End Sub

        Public Sub MigrateActiveTask(ByVal newActiveTask As SensorTask)

            If newActiveTask Is Nothing Then Throw New ArgumentNullException("newActiveTask")

            ' Migrate the active task
            If Not ActiveTask Is Nothing AndAlso ActiveTask.Status = SensorTaskStatus.Active Then
                ActiveTask.Status = SensorTaskStatus.Suspended
            End If
            newActiveTask.Status = SensorTaskStatus.Active
            mActiveTask = newActiveTask
        End Sub

        Public Function AttemptIsMigratable(ByVal attempt As SensorTaskAttempt) As Boolean
            If attempt Is Nothing Then Throw New ArgumentNullException("attempt")
            For i As Integer = 0 To mFactories.Count - 1
                Dim factory As SensorTaskFactory = DirectCast(mFactories.Item(i), SensorTaskFactory)
                If attempt.Satisfies(factory) Then Return True
            Next
            Return False
        End Function

        'Private Function SeekCompatibleTask( _
        '    ByVal badTask As SensorTask, _
        '    ByVal currentConditions As ConditionCollection, _
        '    ByVal currentInaccessibleParts As BodyParts) _
        ' As SensorTask

        '    ' First, seek a compatible factory
        '    Dim targetFactory As SensorTaskFactory
        '    For i As Integer = 0 To mFactories.Count - 1
        '        Dim testConditions As ConditionCollection = Factory(i).ConditionFactories.CreateConditionCollection

        '        Dim categoryOK As Boolean = badTask.TargetCategory = Factory(i).Category
        '        Dim prereqOK As Boolean = Factory(i).Prerequisite.Evaluate
        '        Dim currConditionsOK As Boolean = currentConditions.EqualsSubset(testConditions)

        '        If categoryOK AndAlso prereqOK AndAlso currConditionsOK Then
        '            targetFactory = Factory(i)
        '            Exit For
        '        End If

        '    Next

        '    If targetFactory Is Nothing Then Return badTask

        '    ' Now, seek a task within that factory
        '    Dim targetTask As SensorTask
        '    For i As Integer = 0 To targetFactory.GeneratedTasks.Count - 1
        '        Dim task As SensorTask = targetFactory.GeneratedTasks(i)
        '        If task.IsActivatable Then
        '            targetTask = task
        '            Exit For
        '        End If
        '    Next

        '    ' If there is no task in that factory, then we'll have to make a new one
        '    If targetTask Is Nothing Then
        '        targetFactory.AddExtraTask(currentConditions, currentInaccessibleParts)
        '        Dim last As SensorTask = targetFactory.GeneratedTasks.Last
        '        If last.IsActivatable Then targetTask = last
        '    End If

        '    Return targetTask

        'End Function


        ' Returns the task migrated to
        Private Function AutoMigrateAttempt( _
            ByVal attempt As SensorTaskAttempt, _
            ByVal currentConditions As ConditionCollection, _
            ByVal currentInaccessibleParts As BodyParts, _
            ByVal style As ConditionsMatchStyle) As SensorTask


            ' First, seek a compatible factory
            Dim targetFactory As SensorTaskFactory
            For i As Integer = 0 To mFactories.Count - 1
                If attempt.Satisfies(Factory(i)) Then
                    targetFactory = Factory(i)
                    Exit For
                End If
            Next

            ' Without a satisfactory target factory, forget migrating the attempt
            If targetFactory Is Nothing Then Return Nothing

            ' Now, seek a target task to migrate the attempt onto
            Dim targetTask As SensorTask
            For i As Integer = 0 To targetFactory.GeneratedTasks.Count - 1
                Dim task As SensorTask = targetFactory.GeneratedTasks(i)
                If attempt.Satisfies(task, style) Then
                    targetTask = task
                    Exit For
                End If
            Next

            ' If there is still no target task, then see if we can generate up a new one
            If targetTask Is Nothing Then
                targetFactory.AddExtraTask(currentConditions, currentInaccessibleParts)
                If attempt.Satisfies(targetFactory.GeneratedTasks.Last, style) Then
                    targetTask = targetFactory.GeneratedTasks.Last
                Else
                    'DebuggingConsoleWriteLine(" Generated task doesn't satisfy!")
                End If
            End If

            ' If we still don't have a target task, then give up. We can't migrate this task
            If targetTask Is Nothing Then Return Nothing

            ' The migrated attempt should have the same conditions as the new task. We know that
            ' they are compatible, because the migrating tasks has either the same conditions as the 
            ' target task, or, it passed any equivalence class tests.

            attempt.CapturedConditions.AssignSubset(targetTask.Conditions)

            attempt.ParentTask.Attempts.Remove(attempt)
            attempt.ParentTask.RefreshStatus()

            targetTask.Attempts.Insert(0, attempt)
            targetTask.RefreshStatus()


            Return targetTask


        End Function


        Private Sub PopulateAllTasksList()

            ' Pass one of two
            RefreshAllTasksArray()

            ' Automigrate any attempt in conflict
            ' -----------------------------------
            '
            ' The purpose of this stage is to take any attempt that is in conflict, and try to figure 
            ' out where it should go. If there is no destination for an attempt, then the task will appear
            ' as in conflict to the user
            '
            '
            For i As Integer = 0 To mAllTasks.Count - 1
                Dim task As SensorTask = mAllTasks(i)

                ' Mark all of those tasks that need migration (deletion requires two passes). Migrating them
                ' in opposite order would also cause the tasks to appear done before the rejected attempts even
                ' had a chance to migrate
                '
                Dim toMigrate As New ArrayList
                For j As Integer = 0 To task.Attempts.Count - 1
                    Dim attempt As SensorTaskAttempt = task.Attempts(j)

                    If attempt.InConflict(CurrentConditions, CurrentInaccessibleBodyParts) Then
                        toMigrate.Add(attempt)
                    End If
                Next

                ' Migrate those attempts
                For j As Integer = 0 To toMigrate.Count - 1
                    Dim attempt As SensorTaskAttempt = DirectCast(toMigrate(j), SensorTaskAttempt)
                    AutoMigrateAttempt(attempt, CurrentConditions, CurrentInaccessibleBodyParts, ConditionsMatchStyle.All)
                Next

            Next

            ' Pass two of two
            RefreshAllTasksArray()

            ' AutoSelectActiveTask makes use of the task map, so we should update it
            UpdateTaskMap()

            AutoSelectActiveTask(CurrentConditions)

        End Sub

        Private Sub RefreshAllTasksArray()

            ' Remove any excess tasks 
            TrimExcessTasks()

            ' Add/remove (well, mostly *remove*) from the task side. Okay, well, this comment should
            ' be 'remove stale tasks' but then it doesn't match the next comment
            '
            For i As Integer = mAllTasks.Count - 1 To 0 Step -1
                If mAllTasks(i).OriginatingFactory.GeneratedTasks.IndexOf(mAllTasks(i)) = -1 Then
                    mAllTasks.Remove(mAllTasks(i))
                End If
            Next


            ' Add/remove tasks from the factory side
            For i As Integer = 0 To mFactories.Count - 1
                Dim factory As SensorTaskFactory = Me.Factory(i)

                For j As Integer = 0 To factory.GeneratedTasks.Count - 1
                    Dim task As SensorTask = factory.GeneratedTasks(j)

                    Dim hasIndex As Boolean = mAllTasks.IndexOf(task) <> -1
                    Dim noParent As Boolean = task.OriginatingFactory.GeneratedTasks.IndexOf(task) = -1
                    Dim prereqOK As Boolean = factory.Prerequisite.Evaluate
                    Dim hasParts As Boolean = Not task.TargetParts.Equals(task.TargetInaccessibleParts)
                    Dim inConflict As Boolean = task.HasAttemptsInConflict(CurrentConditions, CurrentInaccessibleBodyParts)
                    Dim sensorDisabled As Boolean = task.Sensor.Disabled

                    Dim insertTask As Boolean = _
                        Not hasIndex AndAlso _
                        (inConflict OrElse prereqOK) AndAlso _
                        hasParts AndAlso _
                        Not sensorDisabled

                    Dim removeTask As Boolean = _
                        Not hasParts AndAlso Not inConflict OrElse _
                        Not inConflict AndAlso hasIndex AndAlso Not prereqOK OrElse _
                        sensorDisabled OrElse _
                        noParent

                    If insertTask Then
                        mAllTasks.Add(task)
                    ElseIf removeTask Then
                        If task.Status = SensorTaskStatus.Active Then task.Status = SensorTaskStatus.Suspended
                        mAllTasks.Remove(task)
                    ElseIf mAllTasks.IndexOf(task) > -1 Then
                        ' Leaving in
                    Else
                        ' Leaving out
                    End If

                Next
            Next

            mAllTasks.Sort()

            UpdateTaskMap()

        End Sub

        Friend Sub UpdateTaskMap()
            Dim en As IEnumerator = Me.mTasksOfSensor.Keys.GetEnumerator

            While en.MoveNext
                Dim sensor As ISensor = DirectCast(en.Current, ISensor)
                Dim tasks As SensorTaskCollection = mTasksOfSensor(sensor)
                tasks.Clear()
            End While

            For i As Integer = 0 To mAllTasks.Count - 1
                Dim sensor As ISensor = mAllTasks(i).Sensor
                If mTasksOfSensor(sensor) Is Nothing Then mTasksOfSensor(sensor) = New SensorTaskCollection
                Dim tasks As SensorTaskCollection = mTasksOfSensor(sensor)
                If tasks.IndexOf(mAllTasks(i)) = -1 Then tasks.Add(mAllTasks(i))
            Next


        End Sub

        Public ReadOnly Property TasksNotDownloadable(ByVal sensor As ISensor) As Integer
            Get
                If mTasksOfSensor(sensor) Is Nothing Then Return 0
                For i As Integer = 0 To mTasksOfSensor(sensor).Count - 1
                    Dim task As SensorTask = mTasksOfSensor(sensor)(i)

                    Dim notPending As Boolean = task.Status <> SensorTaskStatus.Pending
                    Dim notDone As Boolean = task.Status <> SensorTaskStatus.Done
                    Dim notSkipped As Boolean = task.Status <> SensorTaskStatus.Skipped
                    Dim notDLSuspended As Boolean = Not (task.Status = SensorTaskStatus.Suspended AndAlso task.DownloadsSuspended)

                    If notPending AndAlso notDone AndAlso notDLSuspended AndAlso notSkipped Then TasksNotDownloadable += 1
                Next
            End Get
        End Property

        Public ReadOnly Property TasksOfSensor(ByVal sensor As ISensor) As SensorTaskCollection
            Get
                If sensor Is Nothing Then Throw New ArgumentNullException("sensor")
                Return mTasksOfSensor(sensor)
            End Get
        End Property

        Public Function FirstAttemptHavingOutstandingDownload(ByVal sensor As ISensor) As SensorTaskAttempt

            If sensor Is Nothing Then Throw New ArgumentNullException("sensor")
            If TasksNotDownloadable(sensor) > 0 Then Return Nothing

            If mTasksOfSensor(sensor) Is Nothing Then Return Nothing

            For i As Integer = 0 To mTasksOfSensor(sensor).Count - 1
                Dim task As SensorTask = mTasksOfSensor(sensor)(i)
                For j As Integer = 0 To task.Attempts.Count - 1
                    Dim attempt As SensorTaskAttempt = task.Attempts(j)
                    If attempt.NeedsDownload AndAlso Not attempt.ParentTask.DownloadsSuspended Then Return task.Attempts(j)
                Next
            Next

            Return Nothing
        End Function

        Public ReadOnly Property TasksNotDownloaded(ByVal sensor As ISensor) As Integer
            Get
                If sensor Is Nothing Then Throw New ArgumentNullException("sensor")
                If mTasksOfSensor(sensor) Is Nothing Then Return 0
                For i As Integer = 0 To mTasksOfSensor(sensor).Count - 1
                    Dim task As SensorTask = mTasksOfSensor(sensor)(i)
                    For j As Integer = 0 To task.Attempts.Count - 1
                        Dim attempt As SensorTaskAttempt = task.Attempts(j)
                        If attempt.NeedsDownload Then TasksNotDownloaded += 1
                    Next
                Next
            End Get
        End Property

        Public Sub Clear()
            For i As Integer = 0 To Me.mFactories.Count - 1
                Factory(i).Clear()
            Next

            mActiveTask = Nothing
            mAllTasks.Clear()
            mTasksOfSensor.Clear()

            PopulateAllTasksList()

        End Sub

#Region "ICollection Implementation"

        Public ReadOnly Property Count() As Integer Implements ICollection.Count
            Get
                Return mFactories.Count
            End Get
        End Property

        Public Sub CopyTo(ByVal array As System.Array, ByVal index As Integer) Implements ICollection.CopyTo
            mFactories.CopyTo(array, index)
        End Sub

        Public Sub CopyTo(ByVal factories As SensorTaskFactoryCollection, ByVal index As Integer)
            If factories Is Nothing Then Throw New ArgumentNullException("factories")
            mFactories.CopyTo(factories.mFactories.ToArray(), index)
        End Sub


        Public ReadOnly Property IsSynchronized() As Boolean Implements ICollection.IsSynchronized
            Get
                Return mFactories.IsSynchronized
            End Get
        End Property

        Public ReadOnly Property SyncRoot() As Object Implements ICollection.SyncRoot
            Get
                Return mFactories.SyncRoot
            End Get
        End Property

        Public Function GetEnumerator() As System.Collections.IEnumerator Implements IEnumerable.GetEnumerator
            Return mFactories.GetEnumerator
        End Function


#End Region
    End Class

End Namespace
