'
'  Multimodal Biometric Applicaiton Resource Kit (MBARK)
'
'  File author(s):
'       Ross J. Micheals (rossm@nist.gov)
'       Kayee Kwong (kayee@nist.gov)
'
' 
' | LICENSE & DISCLAIMER                                                                                |
' |                                                                                                     |
' | This software was developed at the National Institute of Standards and Technology (NIST) by         |
' | employees of the Federal Government in the course of their official duties. Pursuant to title 17    |
' | Section 105 of the United States Code. This software is not subject to copyright protection and     |
' | is in the public domain. NIST assumes no responsibility whatsoever for use by other parties of      |
' | its source code or open source server, and makes no guarantees, expressed or implied, about its     |
' | quality, reliability, or any other characteristic.                                                  |
' |                                                                                                     |
' | Specific hardware and software products identified in this open source project were used in order   |
' | to perform technology transfer and collaboration. In no case does such identification imply         |
' | recommendation or endorsement by the National Institute of Standards and Technology, nor            |
' | does it imply that the products and equipment identified are necessarily the best available for the |
' | purpose.                                                                                            |
' 

Option Strict On

Imports System.Xml.Serialization

Namespace Mbark.Sensors

    Public Class SensorTaskFactory

        <XmlType("TaskFactory"), Serializable()> _
        Public Shadows Class Definition
            Implements ICloneable

#Region "Private"
            Private mName As String
            Private mCategory As SensorTaskCategory
            Private mTaskCount As Integer = 1
            Private mSensorConfiguration As SensorConfiguration
            Private mReassignableCategoriesDefinition As New SensorTaskCategoryCollectionDefinition
            Private mConditionFactoryDefinitions As New ConditionFactoryCollectionDefinition

            <XmlElement("GeneratedTasks")> Public GeneratedTasksDefinition As New SensorTaskCollectionDefinition
            <XmlElement("DisplayColors")> Public ColorDefinitions As SensorTaskColorsDefinition

            <XmlElement("SensorType")> Public SensorTypeDefinitionString As String

            <XmlElement("TaskPrerequisite")> Public PrerequisiteDefinition As PredicateDefinition
#End Region

            Public Property Name() As String
                Get
                    Return mName
                End Get
                Set(ByVal value As String)
                    mName = value
                End Set
            End Property
            Public Property Category() As SensorTaskCategory
                Get
                    Return mCategory
                End Get
                Set(ByVal value As SensorTaskCategory)
                    mCategory = value
                End Set
            End Property
            Public Property TaskCount() As Integer
                Get
                    Return mTaskCount
                End Get
                Set(ByVal value As Integer)
                    mTaskCount = value
                End Set
            End Property
            Public Property SensorConfiguration() As SensorConfiguration
                Get
                    Return mSensorConfiguration
                End Get
                Set(ByVal value As SensorConfiguration)
                    mSensorConfiguration = value
                End Set
            End Property
            <XmlElement("ReassignableCategories")> Public Property ReassignableCategoriesDefinition() As SensorTaskCategoryCollectionDefinition
                Get
                    Return mReassignableCategoriesDefinition
                End Get
                Set(ByVal value As SensorTaskCategoryCollectionDefinition)
                    mReassignableCategoriesDefinition = Value
                End Set
            End Property
            <XmlElement("ConditionFactories")> Public Property ConditionFactoryDefinitions() As ConditionFactoryCollectionDefinition
                Get
                    Return mConditionFactoryDefinitions
                End Get
                Set(ByVal value As ConditionFactoryCollectionDefinition)
                    mConditionFactoryDefinitions = value
                End Set
            End Property

            Public Function Clone() As Object Implements ICloneable.Clone
                Dim newDef As New Definition
                newDef.Name = Name

                newDef.Category = Category
                newDef.ReassignableCategoriesDefinition = ReassignableCategoriesDefinition.DeepCopy

                newDef.ConditionFactoryDefinitions = ConditionFactoryDefinitions.DeepCopy()
                newDef.GeneratedTasksDefinition = GeneratedTasksDefinition.DeepCopy
                newDef.TaskCount = TaskCount
                newDef.ColorDefinitions = ColorDefinitions.DeepCopy
                newDef.SensorTypeDefinitionString = SensorTypeDefinitionString

                If Not PrerequisiteDefinition Is Nothing Then
                    newDef.PrerequisiteDefinition = PrerequisiteDefinition.DeepCopy
                End If

                If Not SensorConfiguration Is Nothing Then
                    newDef.SensorConfiguration = SensorConfiguration.DeepCopy
                End If

                Return newDef
            End Function

            Public Function DeepCopy() As Definition
                Return DirectCast(Clone(), SensorTaskFactory.Definition)
            End Function

            Public Function CreateFactory(ByVal parentSensorTaskFactories As SensorTaskFactoryCollection) As SensorTaskFactory
                If parentSensorTaskFactories Is Nothing Then Throw New ArgumentNullException("parentSensorTaskFactories")

                Dim factory As New SensorTaskFactory
                With factory
                    .mDefinition = Me.DeepCopy()
                    .mParentSensorTaskFactories = parentSensorTaskFactories
                    If Not PrerequisiteDefinition Is Nothing Then
                        .mPrerequisite = PrerequisiteDefinition.CreatePredicate(parentSensorTaskFactories.CurrentConditions)
                    End If
                    .mConditionFactories = ConditionFactoryDefinitions.CreateConditionFactoryCollection()

                    ' We *must* restore the sensor before creating the task collection (which needs this data)
                    .mSensor = parentSensorTaskFactories.Sensors.ByDefinitionString(SensorTypeDefinitionString)
                    .mGeneratedTasks = GeneratedTasksDefinition.CreateSensorTaskCollection(factory)
                    .mReassignableCategories = ReassignableCategoriesDefinition.CreateTaskCategoryCollection()

                    If .mSensor Is Nothing Then Debugging.Break()
                    .mColors = ColorDefinitions.CreateSensorTaskColors
                End With

                Return factory
            End Function

        End Class

        Private mDefinition As New Definition
        Private mReassignableCategories As New SensorTaskCategoryCollection
        Private mParentSensorTaskFactories As SensorTaskFactoryCollection
        Private mPrerequisite As Predicate

        Private mConditionFactories As New ConditionFactoryCollection
        Private mGeneratedTasks As New SensorTaskCollection

        Private mSensor As ISensor
        Private mColors As SensorTaskColors


        Friend Function CreateDefinition() As Definition

            If Not mPrerequisite Is Nothing Then mDefinition.PrerequisiteDefinition = mPrerequisite.CreateDefinition

            mDefinition.ConditionFactoryDefinitions = mConditionFactories.CreateDefinition()
            mDefinition.GeneratedTasksDefinition = mGeneratedTasks.CreateDefinition()
            mDefinition.ReassignableCategoriesDefinition = ReassignableSensorTaskCategories.CreateDefinition
            mDefinition.SensorTypeDefinitionString = SensorTypeDefinition.CreateDefinition(mSensor).ToString
            mDefinition.ColorDefinitions = mColors.Definition

            Return mDefinition.DeepCopy
        End Function


        Public Property Name() As String
            Get
                Return mDefinition.Name
            End Get
            Set(ByVal value As String)
                mDefinition.Name = value
            End Set
        End Property

        Friend Sub WireParentSensorTaskFactories(ByVal factories As SensorTaskFactoryCollection)
            mParentSensorTaskFactories = factories
        End Sub

        Friend ReadOnly Property ParentSensorTaskFactories() As SensorTaskFactoryCollection
            Get
                Return mParentSensorTaskFactories
            End Get
        End Property

        Public ReadOnly Property ReassignableSensorTaskCategories() As SensorTaskCategoryCollection
            Get
                Return mReassignableCategories
            End Get
        End Property

        Public Property Colors() As SensorTaskColors
            Get
                Return mColors
            End Get
            Set(ByVal value As SensorTaskColors)
                mColors = value
            End Set
        End Property

        Public Property Sensor() As ISensor
            Get
                Return mSensor
            End Get
            Set(ByVal value As ISensor)
                mSensor = value
            End Set
        End Property



        Public Property Prerequisite() As Predicate
            Get
                If mPrerequisite Is Nothing Then Return BuiltIns.Predicates.TrueLiteral
                Return mPrerequisite
            End Get
            Set(ByVal value As Predicate)
                mPrerequisite = value
            End Set
        End Property
        Public Property TaskCount() As Integer
            Get
                Return mDefinition.TaskCount
            End Get
            Set(ByVal value As Integer)
                mDefinition.TaskCount = value
            End Set
        End Property

        Public ReadOnly Property GeneratedTasks() As SensorTaskCollection
            Get
                Return mGeneratedTasks
            End Get
        End Property
        Public Property ConditionFactories() As ConditionFactoryCollection
            Get
                Return mConditionFactories
            End Get
            Set(ByVal value As ConditionFactoryCollection)
                mConditionFactories = value
            End Set
        End Property
        Public Property Category() As SensorTaskCategory
            Get
                Return mDefinition.Category
            End Get
            Set(ByVal value As SensorTaskCategory)
                mDefinition.Category = value
            End Set
        End Property

        Public Property SensorConfiguration() As SensorConfiguration
            Get
                Return mDefinition.SensorConfiguration
            End Get
            Set(ByVal value As SensorConfiguration)
                mDefinition.SensorConfiguration = value
            End Set
        End Property

        Public ReadOnly Property SatisfactoryTaskCount() As Integer
            Get
                Dim count As Integer

                ' Build a set of conditions to test with
                Dim testConditions As ConditionCollection = ConditionFactories.CreateConditionCollection()

                For i As Integer = 0 To mGeneratedTasks.Count - 1
                    Dim task As SensorTask = mGeneratedTasks(i)
                    If task.TargetCategory = Category AndAlso task.Conditions.EqualsSubset(testConditions) Then
                        count += 1
                    End If

                Next
                Return count
            End Get
        End Property


        Public Sub AddExtraTask(ByVal currentConditions As ConditionCollection, ByVal currentInaccessibleParts As BodyParts)
            Dim oldTaskCount As Integer = TaskCount
            TaskCount = GeneratedTasks.Count + 1
            PopulateTargets(currentConditions, currentInaccessibleParts)
            TaskCount = oldTaskCount
        End Sub

        Public Sub TrimExcessTasks()

            Dim toRemove As Integer = Math.Max(0, GeneratedTasks.Count - TaskCount)
            If toRemove = 0 Then Return

            Dim excessTasks As New ArrayList

            ' Mark those tasks that can be removed
            For i As Integer = 0 To GeneratedTasks.Count - 1
                If GeneratedTasks(i).IsExcess AndAlso GeneratedTasks(i).Attempts.Count = 0 Then
                    excessTasks.Add(GeneratedTasks(i))
                End If
            Next

            ' Remove those tasks
            For i As Integer = 0 To excessTasks.Count - 1
                Dim task As SensorTask = DirectCast(excessTasks(i), SensorTask)
                GeneratedTasks.Remove(task)
            Next


        End Sub

        Public Sub PopulateTargets(ByVal currentConditions As ConditionCollection, ByVal currentInaccessibleParts As BodyParts)

            If currentConditions Is Nothing Then Throw New ArgumentNullException("currentConditions")
            If currentInaccessibleParts Is Nothing Then Throw New ArgumentNullException("currentInaccessibleParts")

            Dim satisfactoryCount As Integer = SatisfactoryTaskCount
            Dim toCreate As Integer = TaskCount - satisfactoryCount - 1


            For i As Integer = 0 To toCreate
                Dim task As New SensorTask

                task.TargetCategory = Category
                task.Sensor = Sensor
                task.Colors = Colors
                task.TargetInaccessibleParts = currentInaccessibleParts.DeepCopy
                task.Conditions.WireParentTask(task)
                task.WireOriginatingFactory(Me)
                If Not SensorConfiguration Is Nothing Then task.SensorConfiguration = SensorConfiguration.DeepCopy
                task.Conditions = ConditionFactories.CreateConditionCollection()
                GeneratedTasks.Add(task)
            Next

            For i As Integer = 0 To GeneratedTasks.Count - 1

                Dim task As SensorTask = GeneratedTasks(i)
                task.TargetInaccessibleParts.AddEachPartIfContainedInOtherSet( _
                    task.TargetParts, currentInaccessibleParts)
                task.TargetInaccessibleParts.RemoveEachPartIfNotContainedInOtherSet( _
                        currentInaccessibleParts, task.targetparts)
            Next

            Return

        End Sub

        Public Sub Clear()
            GeneratedTasks.Clear()
        End Sub

        'Mainly use for changing status for skipping or not skipping
        Public Sub ChangeAllTaskStatus(ByVal NewStatus As SensorTaskStatus)
            With mGeneratedTasks
                For i As Integer = 0 To .Count - 1
                    If NewStatus = SensorTaskStatus.Skipped Then
                        If .Items(i).IsSkippable() Then mGeneratedTasks.Items(i).Status = NewStatus
                    ElseIf NewStatus = SensorTaskStatus.Unstarted Then
                        If .Items(i).Status = SensorTaskStatus.Skipped Then mGeneratedTasks.Items(i).Status = NewStatus
                    End If
                Next
            End With
        End Sub

    End Class

    <XmlType("SensorType"), Serializable()> Public Class SensorTypeDefinition
        Inherits TypeDefinition

        Private mHasConfigurationClass As Boolean
        Public Property HasConfigurationClass() As Boolean
            Get
                Return mHasConfigurationClass
            End Get
            Set(ByVal value As Boolean)
                mHasConfigurationClass = value
            End Set
        End Property

        Private mConfigurationClassName As String
        Public Property ConfigurationClassName() As String
            Get
                Return mConfigurationClassName
            End Get
            Set(ByVal value As String)
                mConfigurationClassName = value
            End Set
        End Property

        Public Overloads Shared Function CreateDefinition(ByVal sensor As ISensor) As SensorTypeDefinition

            If sensor Is Nothing Then Throw New ArgumentNullException("sensor")

            Dim newDef As New SensorTypeDefinition

            Dim asBaseSensor As BaseSensor = DirectCast(sensor, BaseSensor)
            newDef.ClassName = asBaseSensor.GetType.FullName
            newDef.LibraryFileName = IO.Path.GetFileName(asBaseSensor.GetType.Assembly.Location)
            If sensor.HasConfigurationClass Then
                newDef.HasConfigurationClass = True
                newDef.ConfigurationClassName = sensor.ConfigurationClassName
            Else
                newDef.HasConfigurationClass = False
                newDef.ConfigurationClassName = Nothing
            End If
            Return newDef
        End Function

        Public Shared Function CreateSensor(ByVal definition As SensorTypeDefinition) As ISensor
            If definition Is Nothing Then Throw New ArgumentNullException("definition")
            Return DirectCast(definition.InstantiateObject(), ISensor)
        End Function

        Public Overrides Function InstantiateObject() As Object
            Dim newObject As Object
            Try
                Subdirectory = "Sensors"
                newObject = MyBase.InstantiateObject()
            Catch ex As IO.DirectoryNotFoundException
                Subdirectory = String.Empty
                newObject = MyBase.InstantiateObject()
            End Try
            Return newObject
        End Function


    End Class


End Namespace
