'
'  Multimodal Biometric Applicaiton Resource Kit (MBARK)
'
'  File author(s):
'       Ross J. Micheals (rossm@nist.gov)
'       Kayee Kwong (kayee@nist.gov)
'
' 
' | LICENSE & DISCLAIMER                                                                                |
' |                                                                                                     |
' | This software was developed at the National Institute of Standards and Technology (NIST) by         |
' | employees of the Federal Government in the course of their official duties. Pursuant to title 17    |
' | Section 105 of the United States Code. This software is not subject to copyright protection and     |
' | is in the public domain. NIST assumes no responsibility whatsoever for use by other parties of      |
' | its source code or open source server, and makes no guarantees, expressed or implied, about its     |
' | quality, reliability, or any other characteristic.                                                  |
' |                                                                                                     |
' | Specific hardware and software products identified in this open source project were used in order   |
' | to perform technology transfer and collaboration. In no case does such identification imply         |
' | recommendation or endorsement by the National Institute of Standards and Technology, nor            |
' | does it imply that the products and equipment identified are necessarily the best available for the |
' | purpose.                                                                                            |
' 

Option Strict On

Imports System.Drawing
Imports System.IO
Imports System.Globalization


Namespace Mbark.Sensors

    Public Enum SensorTaskFailureCategory
        InitializationError
        CaptureError
        ConfigurationError
        UnspecifiedCaptureTimeout
        SubjectRefuses
        SubjectLeftSession
        DownloadError
        OtherUnspecifiedError
    End Enum

    <Serializable()> Public Class SensorTaskFailure
        Implements ICloneable

        Private mCategory As SensorTaskFailureCategory
        Private mExceptionMessages As String
        Private mOperatorNotes As String
        Private mMachineNotes As String
        Private mTimestamp As DateTime
        Private mCategoryToDisplay As String

        Public Property Category() As SensorTaskFailureCategory
            Get
                Return mCategory
            End Get
            Set(ByVal value As SensorTaskFailureCategory)
                mCategory = value
            End Set
        End Property
        Public Property ExceptionMessages() As String
            Get
                Return mExceptionMessages
            End Get
            Set(ByVal value As String)
                mExceptionMessages = value
            End Set
        End Property
        Public Property OperatorNotes() As String
            Get
                Return mOperatorNotes
            End Get
            Set(ByVal value As String)
                mOperatorNotes = value
            End Set
        End Property
        Public Property MachineNotes() As String
            Get
                Return mMachineNotes
            End Get
            Set(ByVal value As String)
                mMachineNotes = value
            End Set
        End Property
        Public Property Timestamp() As DateTime
            Get
                Return mTimestamp
            End Get
            Set(ByVal value As DateTime)
                mTimestamp = value
            End Set
        End Property

        Public Sub New()
            mTimestamp = DateTime.UtcNow
        End Sub

        Public Sub New(ByVal ex As Exception)
            mTimestamp = DateTime.UtcNow

            If Not ex Is Nothing Then

                ExceptionMessages = StringUtilities.UnwindAllMessages(ex)
                Category = SensorTaskFailureCategory.OtherUnspecifiedError

                If TypeOf ex Is PollingCanceledException Then
                    PollingCancelledNew(DirectCast(ex, PollingCanceledException))
                End If

                If TypeOf ex Is InitializationFailureException Then
                    InitializationFailureNew(DirectCast(ex, InitializationFailureException))
                End If

                If TypeOf ex Is InitializationTimeoutException Then
                    InitializationTimeoutNew(DirectCast(ex, InitializationTimeoutException))
                End If

                If TypeOf ex Is CaptureFailureException Then
                    CaptureFailureNew(DirectCast(ex, CaptureFailureException))
                End If

                If TypeOf ex Is CaptureTimeoutException Then
                    CaptureTimeoutNew(DirectCast(ex, CaptureTimeoutException))
                End If

                If TypeOf ex Is ConfigurationFailureException Then
                    ConfigurationFailureNew(DirectCast(ex, ConfigurationFailureException))
                End If

                If TypeOf ex Is ConfigurationTimeoutException Then
                    ConfigurationTimeoutNew(DirectCast(ex, ConfigurationTimeoutException))
                End If

                If TypeOf ex Is DownloadFailureException Then
                    DownloadFailureNew(DirectCast(ex, DownloadFailureException))
                End If

                If TypeOf ex Is DownloadTimeoutException Then
                    DownloadTimeoutNew(DirectCast(ex, DownloadTimeoutException))
                End If


                If TypeOf ex Is SensorException Then
                    Dim se As SensorException = DirectCast(ex, SensorException)
                    Dim sensorName As String = "Unknown sensor"
                    If Not se.Sensor Is Nothing Then sensorName = se.Sensor.FriendlyName()
                    mMachineNotes = sensorName & " / " & mCategoryToDisplay & " / " & se.MachineNotes
                End If

            End If


        End Sub

        Private Sub SensorExceptionNew(ByVal ex As SensorException)
            mTimestamp = ex.Timestamp
        End Sub

        Private Sub InitializationFailureNew(ByVal ife As InitializationFailureException)
            SensorExceptionNew(ife)
            mCategory = SensorTaskFailureCategory.InitializationError
            mCategoryToDisplay = "Initialization failure" 'i18n
        End Sub

        Private Sub InitializationTimeoutNew(ByVal ite As InitializationTimeoutException)
            SensorExceptionNew(ite)
            mCategory = SensorTaskFailureCategory.InitializationError
            mCategoryToDisplay = "Initialization timeout" 'i18n
        End Sub

        Private Sub CaptureFailureNew(ByVal cfe As CaptureFailureException)
            SensorExceptionNew(cfe)
            mCategory = SensorTaskFailureCategory.CaptureError
            mCategoryToDisplay = "Capture failure" 'i18n
        End Sub

        Private Sub CaptureTimeoutNew(ByVal cte As CaptureTimeoutException)
            SensorExceptionNew(cte)
            mCategory = SensorTaskFailureCategory.UnspecifiedCaptureTimeout
            mCategoryToDisplay = "Capture timeout" 'i18n
        End Sub

        Private Sub PollingCancelledNew(ByVal pce As PollingCanceledException)
            SensorExceptionNew(pce)
            mCategory = SensorTaskFailureCategory.CaptureError
            mCategoryToDisplay = "Capture failure" 'i18n
        End Sub

        Private Sub ConfigurationTimeoutNew(ByVal cte As ConfigurationTimeoutException)
            SensorExceptionNew(cte)
            mCategory = SensorTaskFailureCategory.ConfigurationError
            mCategoryToDisplay = "Configuration timeout"
        End Sub

        Private Sub ConfigurationFailureNew(ByVal cfe As ConfigurationFailureException)
            SensorExceptionNew(cfe)
            mCategory = SensorTaskFailureCategory.ConfigurationError
            mCategoryToDisplay = "Configuration failure"
        End Sub

        Private Sub DownloadFailureNew(ByVal dfe As DownloadFailureException)
            SensorExceptionNew(dfe)
            mCategory = SensorTaskFailureCategory.DownloadError
            mCategoryToDisplay = "Download failure"
        End Sub

        Private Sub DownloadTimeoutNew(ByVal dte As DownloadTimeoutException)
            SensorExceptionNew(dte)
            mCategory = SensorTaskFailureCategory.DownloadError
            mCategoryToDisplay = "Download timeout"
        End Sub


        Public Function Clone() As Object Implements System.ICloneable.Clone
            Dim failure As New SensorTaskFailure

            failure.mCategory = Me.Category
            failure.mExceptionMessages = Me.ExceptionMessages
            failure.mOperatorNotes = Me.OperatorNotes
            failure.mMachineNotes = Me.MachineNotes
            failure.mTimestamp = Me.Timestamp

            Return failure
        End Function

        Public Function DeepCopy() As SensorTaskFailure
            Return DirectCast(Clone(), SensorTaskFailure)
        End Function

        Public Overloads Function Equals(ByVal failure As SensorTaskFailure) As Boolean
            If failure Is Nothing Then Throw New ArgumentNullException("failure")
            Return _
                failure.mCategory = Me.mCategory AndAlso _
                failure.mExceptionMessages = Me.mExceptionMessages AndAlso _
                failure.mOperatorNotes = Me.mOperatorNotes AndAlso _
                failure.mMachineNotes = Me.mMachineNotes AndAlso _
                failure.mTimestamp = Me.mTimestamp
        End Function

    End Class

    Public Enum SensorModality
        Face
        Fingerprint
        Iris
        Voice
        HandGeometry
        Face3D
        Gait
    End Enum

    Public Module SensorModalitySupport

        Public Function ToString(ByVal culture As CultureInfo, ByVal modality As SensorModality) As String
            Select Case modality
                Case SensorModality.Face : Return Mbark.SensorMessages.Messages.Face(culture)
                Case SensorModality.Face3D : Return Mbark.SensorMessages.Messages.Face3D(culture)
                Case SensorModality.Fingerprint : Return Mbark.SensorMessages.Messages.Fingerprint(culture)
                Case SensorModality.Gait : Return Mbark.SensorMessages.Messages.Gait(culture)
                Case SensorModality.HandGeometry : Return Mbark.SensorMessages.Messages.HandGeometry(culture)
                Case SensorModality.Iris : Return Mbark.SensorMessages.Messages.Iris(culture)
                Case SensorModality.Voice : Return Mbark.SensorMessages.Messages.Voice(culture)
            End Select
        End Function

    End Module

    Public Enum SensorStatus
        Uninitialized
        Offline
        Initializing
        Capturing
        Configuring
        Downloading
        Online
    End Enum


    Public Class CaptureActivatedEventArgs
        Inherits EventArgs
        Private mOriginatingSensor As ISensor
        Public ReadOnly Property OriginatingSensor() As ISensor
            Get
                Return mOriginatingSensor
            End Get
        End Property
        Public Sub New(ByVal originatingSensor As ISensor)
            mOriginatingSensor = originatingSensor
        End Sub
    End Class

    Public Interface ISensor
        Inherits IHasUICulture

        ReadOnly Property TypeName() As String

        ReadOnly Property FriendlyName() As String
        ReadOnly Property Modality() As SensorModality
        ReadOnly Property IsActive() As Boolean

        Property Disabled() As Boolean

        ReadOnly Property HasLivePreview() As Boolean
        Sub StartLivePreview()
        Sub StopLivePreview()

        ReadOnly Property LatestStatus() As SensorStatus

        ReadOnly Property InitializationCommandTemplate() As Threading.AsyncCommandTemplate
        ReadOnly Property DeferInitialization() As Boolean
        Function StartInitialize() As Guid
        ReadOnly Property IsOfflineOrUninitialized() As Boolean

        Sub Uninitialize()

        ReadOnly Property ConfigurationCommandTemplate() As Threading.AsyncCommandTemplate
        ReadOnly Property HasConfigurationClass() As Boolean
        ReadOnly Property ConfigurationClassName() As String
        ReadOnly Property RequiresConfiguration() As Boolean
        Function StartConfiguration(ByVal configuration As SensorConfiguration) As Guid

        ReadOnly Property CaptureCommandTemplate() As Threading.AsyncCommandTemplate
        Sub CancelCapture()
        Property TargetParts() As BodyParts
        Property TargetCategory() As SensorTaskCategory
        Property InaccessibleParts() As BodyParts
        ReadOnly Property PollingIsCancelable() As Boolean
        ReadOnly Property PollingWasCanceled() As Boolean
        Sub ActivateSensorNow()
        Event CaptureActivatedEvent(ByVal sender As Object, ByVal e As CaptureActivatedEventArgs)

        Property LatestReviewImageConsideredAcceptable() As Boolean

        ReadOnly Property LatestThumbnail() As Bitmap

        ReadOnly Property RequiresReview() As Boolean

        Property RequiresRecovery() As Boolean

        ReadOnly Property LatestCaptureSuccessful() As Boolean

        ReadOnly Property AsControl() As System.Windows.Forms.Control
        ReadOnly Property SensorId() As Guid

        ReadOnly Property DownloadCommandTemplate() As Threading.AsyncCommandTemplate
        Function StartDownload(ByVal guid As Guid) As Guid
        ReadOnly Property DownloadIsCancelable() As Boolean
        Function CancelDownload() As Boolean
        ReadOnly Property LatestDownloadWasSuccessful() As Boolean
        ReadOnly Property LatestDownloadWasCanceled() As Boolean
        ReadOnly Property PercentDownloadedIsMeaningful() As Boolean
        ReadOnly Property PercentDownloaded() As Single
        Sub DeleteInternalImages()

        Property ActivateSensorButtonEnabled() As Boolean
        Property ConditionsLabelText() As String



    End Interface


    Public Class ImageProperties

        Private mDefinition As New ImagePropertiesDefinition
        Private mImage As Image
        Public Property Image() As Image
            Get
                Return mImage
            End Get
            Set(ByVal value As Image)
                mImage = value
            End Set
        End Property

        Public Property Timestamp() As DateTime
            Get
                Return mDefinition.Timestamp
            End Get
            Set(ByVal value As DateTime)
                mDefinition.Timestamp = value
            End Set
        End Property

        Public Function Definition() As ImagePropertiesDefinition


            Dim newDef As New ImagePropertiesDefinition
            newDef.Timestamp = Timestamp
            If Not Image Is Nothing Then
                Dim memStream As New MemoryStream
                Image.Save(memStream, Imaging.ImageFormat.Png)
                memStream.Close()
                ' Notice the use of MemoryStream.ToArray() instead of the (larger) MemoryStream.GetBuffer()
                newDef.ImageAsBase64String = Convert.ToBase64String(memStream.ToArray())
            Else
                newDef.ImageAsBase64String = String.Empty
            End If

            Return newDef

        End Function

    End Class

    <Serializable()> Public Class ImagePropertiesDefinition
        Implements ICloneable

        Private mImageAsBase64String As String
        Private mTimestamp As DateTime

        Public Property ImageAsBase64String() As String
            Get
                Return mImageAsBase64String
            End Get
            Set(ByVal value As String)
                mImageAsBase64String = value
            End Set
        End Property
        Public Property Timestamp() As DateTime
            Get
                Return mTimestamp
            End Get
            Set(ByVal value As DateTime)
                mTimestamp = value
            End Set
        End Property

        Public Function CreateImageProperties() As ImageProperties

            Dim newProp As New ImageProperties

            If ImageAsBase64String <> String.Empty Then
                Dim imageAsBytes() As Byte = Convert.FromBase64String(ImageAsBase64String)
                Dim memStream As New MemoryStream(imageAsBytes)
                ' Notice we don't close the memory stream as noted in KB 814675
                Dim newImage As Image = Image.FromStream(memStream)
                newProp.Image = newImage
            End If

            Return newProp
        End Function

        Public Function Clone() As Object Implements System.ICloneable.Clone
            Dim newDef As New ImagePropertiesDefinition
            newDef.ImageAsBase64String = ImageAsBase64String
            newDef.Timestamp = Timestamp
            Return newDef
        End Function

        Public Function DeepCopy() As ImagePropertiesDefinition
            Return DirectCast(Clone(), ImagePropertiesDefinition)
        End Function
    End Class

    <Serializable()> Public Class SensorConfiguration
        Implements ICloneable

        Public Overridable Function Clone() As Object Implements ICloneable.Clone
            Throw New MissingSpecializationException("Clone()")
        End Function

        Public Function DeepCopy() As SensorConfiguration
            Return DirectCast(Clone(), SensorConfiguration)
        End Function


    End Class


End Namespace
