'
'  Multimodal Biometric Applicaiton Resource Kit (MBARK)
'
'  File author(s):
'       Ross J. Micheals (rossm@nist.gov)
'       Kayee Kwong (kayee@nist.gov)
'
' 
' | LICENSE & DISCLAIMER                                                                                |
' |                                                                                                     |
' | This software was developed at the National Institute of Standards and Technology (NIST) by         |
' | employees of the Federal Government in the course of their official duties. Pursuant to title 17    |
' | Section 105 of the United States Code. This software is not subject to copyright protection and     |
' | is in the public domain. NIST assumes no responsibility whatsoever for use by other parties of      |
' | its source code or open source server, and makes no guarantees, expressed or implied, about its     |
' | quality, reliability, or any other characteristic.                                                  |
' |                                                                                                     |
' | Specific hardware and software products identified in this open source project were used in order   |
' | to perform technology transfer and collaboration. In no case does such identification imply         |
' | recommendation or endorsement by the National Institute of Standards and Technology, nor            |
' | does it imply that the products and equipment identified are necessarily the best available for the |
' | purpose.                                                                                            |
' 

Option Strict On

Imports System.Globalization
Imports System.Xml.Serialization

Imports Mbark.SensorMessages

Namespace Mbark.Sensors

    <Serializable()> Public Enum BodyPart
        Unknown
        Face
        LeftIris
        RightIris
        LeftThumb = 12
        LeftIndex = 19
        LeftMiddle = 20
        LeftRing = 21
        LeftLittle = 22
        RightThumb = 11
        RightIndex = 15
        RightMiddle = 16
        RightRing = 17
        RightLittle = 18
    End Enum

    <Serializable()> Public Class BodyParts

        Private mReadOnly As Boolean
        <NonSerialized()> Private mInsideUpdate As Boolean
        <NonSerialized()> Private mPartsAddedDuringUpdate As BodyParts
        <NonSerialized()> Private mPartsRemovedDuringUpdate As BodyParts
        Private mDescription As String

        Public Property Description() As String
            Get
                Return mDescription
            End Get
            Set(ByVal value As String)
                mDescription = value
            End Set
        End Property

        Public Sub BeginUpdate()
            If mInsideUpdate = True Then Throw New UnfinishedBodyPartsUpdateException
            mInsideUpdate = True
            mPartsAddedDuringUpdate = New BodyParts
            mPartsRemovedDuringUpdate = New BodyParts
        End Sub

        Public Sub EndUpdate()
            mInsideUpdate = False
            If mPartsAddedDuringUpdate.Count > 0 Or mPartsRemovedDuringUpdate.Count > 0 Then
                mPartsAddedDuringUpdate.mParts.Sort()
                mPartsRemovedDuringUpdate.mParts.Sort()
                SyncLock Me
                    RaiseEvent BodyPartsChangeEvent(Me, New BodyPartsChangeEventArgs(mPartsAddedDuringUpdate, mPartsRemovedDuringUpdate))
                End SyncLock
            End If
        End Sub

        Private Shared smLeftSlap As BodyParts
        Private Shared smLeftHand As BodyParts
        Private Shared smThumbsSlap As BodyParts

        Private Shared smRightHand As BodyParts
        Private Shared smRightSlap As BodyParts

        Private Shared smFace As BodyParts

        Private Shared smLeftIris As BodyParts
        Private Shared smRightIris As BodyParts
        Private Shared smBothIrises As BodyParts

        Private Shared smLeftThumb As BodyParts
        Private Shared smLeftIndex As BodyParts
        Private Shared smLeftMiddle As BodyParts
        Private Shared smLeftRing As BodyParts
        Private Shared smLeftLittle As BodyParts

        Private Shared smRightThumb As BodyParts
        Private Shared smRightIndex As BodyParts
        Private Shared smRightMiddle As BodyParts
        Private Shared smRightRing As BodyParts
        Private Shared smRightLittle As BodyParts

        Private Shared smEmpty As BodyParts

        Public Shared ReadOnly Property ForLeftSlapTask() As BodyParts
            Get
                If smLeftSlap Is Nothing Then
                    smLeftSlap = New BodyParts
                    With smLeftSlap
                        .Add(BodyPart.LeftIndex)
                        .Add(BodyPart.LeftMiddle)
                        .Add(BodyPart.LeftRing)
                        .Add(BodyPart.LeftLittle)
                        .mReadOnly = True
                    End With

                End If
                Return smLeftSlap
            End Get
        End Property
        Public Shared ReadOnly Property ForLeftHand() As BodyParts
            Get
                If smLeftHand Is Nothing Then
                    smLeftHand = New BodyParts
                    With smLeftHand
                        .Add(BodyPart.LeftIndex)
                        .Add(BodyPart.LeftMiddle)
                        .Add(BodyPart.LeftRing)
                        .Add(BodyPart.LeftLittle)
                        .Add(BodyPart.LeftThumb)
                        .mReadOnly = True
                    End With

                End If
                Return smLeftHand
            End Get
        End Property
        Public Shared ReadOnly Property ForRightHand() As BodyParts
            Get
                If smRightHand Is Nothing Then
                    smRightHand = New BodyParts
                    With smRightHand
                        .Add(BodyPart.RightIndex)
                        .Add(BodyPart.RightMiddle)
                        .Add(BodyPart.RightRing)
                        .Add(BodyPart.RightLittle)
                        .Add(BodyPart.RightThumb)
                        .mReadOnly = True
                    End With

                End If
                Return smRightHand
            End Get
        End Property

        Public Shared ReadOnly Property ForTask(ByVal category As SensorTaskCategory) As BodyParts
            Get
                Select Case category
                    Case SensorTaskCategory.LeftSlap : Return ForLeftSlapTask
                    Case SensorTaskCategory.RightSlap : Return ForRightSlapTask
                    Case SensorTaskCategory.ThumbsSlap : Return ForThumbsSlapTask
                    Case SensorTaskCategory.LeftIris : Return ForLeftIrisTask
                    Case SensorTaskCategory.RightIris : Return ForRightIrisTask
                    Case SensorTaskCategory.Face : Return ForFaceTask

                    Case SensorTaskCategory.FlatLeftThumb : Return ForFlatLeftThumb
                    Case SensorTaskCategory.FlatLeftIndex : Return ForFlatLeftIndex
                    Case SensorTaskCategory.FlatLeftMiddle : Return ForFlatLeftMiddle
                    Case SensorTaskCategory.FlatLeftRing : Return ForFlatLeftRing
                    Case SensorTaskCategory.FlatLeftLittle : Return ForFlatLeftLittle
                    Case SensorTaskCategory.FlatRightThumb : Return ForFlatRightThumb
                    Case SensorTaskCategory.FlatRightIndex : Return ForFlatRightIndex
                    Case SensorTaskCategory.FlatRightMiddle : Return ForFlatRightMiddle
                    Case SensorTaskCategory.FlatRightRing : Return ForFlatRightRing
                    Case SensorTaskCategory.FlatRightLittle : Return ForFlatRightLittle

                    Case SensorTaskCategory.RolledLeftThumb : Return ForRolledLeftThumb
                    Case SensorTaskCategory.RolledLeftIndex : Return ForRolledLeftIndex
                    Case SensorTaskCategory.RolledLeftMiddle : Return ForRolledLeftMiddle
                    Case SensorTaskCategory.RolledLeftRing : Return ForRolledLeftRing
                    Case SensorTaskCategory.RolledLeftLittle : Return ForRolledLeftLittle
                    Case SensorTaskCategory.RolledRightThumb : Return ForRolledRightThumb
                    Case SensorTaskCategory.RolledRightIndex : Return ForRolledRightIndex
                    Case SensorTaskCategory.RolledRightMiddle : Return ForRolledRightMiddle
                    Case SensorTaskCategory.RolledRightRing : Return ForRolledRightRing
                    Case SensorTaskCategory.RolledRightLittle : Return ForRolledRightLittle


                    Case Else
                        Throw New NotImplementedException
                End Select
            End Get
        End Property

        Public Shared ReadOnly Property ForRightSlapTask() As BodyParts
            Get
                If smRightSlap Is Nothing Then
                    smRightSlap = New BodyParts
                    With smRightSlap
                        .Add(BodyPart.RightIndex)
                        .Add(BodyPart.RightMiddle)
                        .Add(BodyPart.RightRing)
                        .Add(BodyPart.RightLittle)
                        .mReadOnly = True
                    End With
                End If
                Return smRightSlap
            End Get
        End Property
        Public Shared ReadOnly Property ForThumbsSlapTask() As BodyParts
            Get
                If smThumbsSlap Is Nothing Then
                    smThumbsSlap = New BodyParts
                    With smThumbsSlap
                        .Add(BodyPart.LeftThumb)
                        .Add(BodyPart.RightThumb)
                        .mReadOnly = True
                    End With
                End If
                Return smThumbsSlap
            End Get
        End Property
        Public Shared ReadOnly Property ForFaceTask() As BodyParts
            Get
                If smFace Is Nothing Then
                    smFace = New BodyParts
                    With smFace
                        .Add(BodyPart.Face)
                        .mReadOnly = True
                    End With
                End If
                Return smFace
            End Get
        End Property
        Public Shared ReadOnly Property ForLeftIrisTask() As BodyParts
            Get
                If smLeftIris Is Nothing Then
                    smLeftIris = New BodyParts
                    With smLeftIris
                        .Add(BodyPart.LeftIris)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftIris
            End Get
        End Property
        Public Shared ReadOnly Property ForRightIrisTask() As BodyParts
            Get
                If smRightIris Is Nothing Then
                    smRightIris = New BodyParts
                    With smRightIris
                        .Add(BodyPart.RightIris)
                        .mReadOnly = True
                    End With
                End If
                Return smRightIris
            End Get
        End Property
        Public Shared ReadOnly Property ForBothIrises() As BodyParts
            Get
                If smBothIrises Is Nothing Then
                    smBothIrises = New BodyParts
                    With smBothIrises
                        .Add(BodyPart.RightIris)
                        .Add(BodyPart.LeftIris)
                        .mReadOnly = True
                    End With
                End If
                Return smBothIrises
            End Get
        End Property

        Public Shared ReadOnly Property ForFlatLeftThumb() As BodyParts
            Get
                If smLeftThumb Is Nothing Then
                    smLeftThumb = New BodyParts
                    With smLeftThumb
                        .Add(BodyPart.LeftThumb)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftThumb
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledLeftThumb() As BodyParts
            Get
                Return ForFlatLeftThumb
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatLeftIndex() As BodyParts
            Get
                If smLeftIndex Is Nothing Then
                    smLeftIndex = New BodyParts
                    With smLeftIndex
                        .Add(BodyPart.LeftIndex)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftIndex
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledLeftIndex() As BodyParts
            Get
                Return ForFlatLeftIndex
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatLeftMiddle() As BodyParts
            Get
                If smLeftMiddle Is Nothing Then
                    smLeftMiddle = New BodyParts
                    With smLeftMiddle
                        .Add(BodyPart.LeftMiddle)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftMiddle
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledLeftMiddle() As BodyParts
            Get
                Return ForFlatLeftMiddle
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatLeftRing() As BodyParts
            Get
                If smLeftRing Is Nothing Then
                    smLeftRing = New BodyParts
                    With smLeftRing
                        .Add(BodyPart.LeftRing)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftRing
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledLeftRing() As BodyParts
            Get
                Return ForFlatLeftRing
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatLeftLittle() As BodyParts
            Get
                If smLeftLittle Is Nothing Then
                    smLeftLittle = New BodyParts
                    With smLeftLittle
                        .Add(BodyPart.LeftLittle)
                        .mReadOnly = True
                    End With
                End If
                Return smLeftLittle
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledLeftLittle() As BodyParts
            Get
                Return ForFlatLeftLittle
            End Get
        End Property

        Public Shared ReadOnly Property ForFlatRightThumb() As BodyParts
            Get
                If smRightThumb Is Nothing Then
                    smRightThumb = New BodyParts
                    With smRightThumb
                        .Add(BodyPart.RightThumb)
                        .mReadOnly = True
                    End With
                End If
                Return smRightThumb
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledRightThumb() As BodyParts
            Get
                Return ForFlatRightThumb
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatRightIndex() As BodyParts
            Get
                If smRightIndex Is Nothing Then
                    smRightIndex = New BodyParts
                    With smRightIndex
                        .Add(BodyPart.RightIndex)
                        .mReadOnly = True
                    End With
                End If
                Return smRightIndex
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledRightIndex() As BodyParts
            Get
                Return ForFlatRightIndex
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatRightMiddle() As BodyParts
            Get
                If smRightMiddle Is Nothing Then
                    smRightMiddle = New BodyParts
                    With smRightMiddle
                        .Add(BodyPart.RightMiddle)
                        .mReadOnly = True
                    End With
                End If
                Return smRightMiddle
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledRightMiddle() As BodyParts
            Get
                Return ForFlatRightMiddle
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatRightRing() As BodyParts
            Get
                If smRightRing Is Nothing Then
                    smRightRing = New BodyParts
                    With smRightRing
                        .Add(BodyPart.RightRing)
                        .mReadOnly = True
                    End With
                End If
                Return smRightRing
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledRightRing() As BodyParts
            Get
                Return ForFlatRightRing
            End Get
        End Property
        Public Shared ReadOnly Property ForFlatRightLittle() As BodyParts
            Get
                If smRightLittle Is Nothing Then
                    smRightLittle = New BodyParts
                    With smRightLittle
                        .Add(BodyPart.RightLittle)
                        .mReadOnly = True
                    End With
                End If
                Return smRightLittle
            End Get
        End Property
        Public Shared ReadOnly Property ForRolledRightLittle() As BodyParts
            Get
                Return ForFlatRightLittle
            End Get
        End Property

        Public Shared ReadOnly Property Empty() As BodyParts
            Get
                If smEmpty Is Nothing Then
                    smEmpty = New BodyParts
                    smEmpty.mReadOnly = True
                End If
                Return smEmpty
            End Get
        End Property

        Dim mParts As New ArrayList

        Default Public ReadOnly Property Items(ByVal index As Integer) As BodyPart
            Get
                Return DirectCast(mParts(index), BodyPart)
            End Get
        End Property

        <XmlArrayItem(GetType(BodyPart))> Public ReadOnly Property AsArrayList() As IList
            Get
                Return mParts
            End Get
        End Property

        Public Sub Assign(ByVal parts As BodyParts)
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            mParts.Clear()
            mDescription = parts.mDescription
            Me.AddRange(parts)
        End Sub

        Public Sub Add(ByVal part As BodyPart)
            If mReadOnly Then Throw New ReadOnlyException
            If Not mParts.Contains(part) Then
                mParts.Add(part)
                mParts.Sort()
                If Me.mInsideUpdate Then
                    mPartsAddedDuringUpdate.mParts.Add(part)
                Else
                    Dim partsAdded As New BodyParts
                    partsAdded.mParts.Add(part)
                    SyncLock Me
                        RaiseEvent BodyPartsChangeEvent(Me, New BodyPartsChangeEventArgs(partsAdded, New BodyParts))
                    End SyncLock
                End If
            End If
        End Sub

        Public Sub AddIfContainedInOtherSet(ByVal otherSet As BodyParts, ByVal part As BodyPart)
            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If otherSet.Contains(part) Then Add(part)
        End Sub

        Public Sub AddIfNotContainedInOtherSet(ByVal otherSet As BodyParts, ByVal part As BodyPart)
            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If Not otherSet.Contains(part) Then Add(part)
        End Sub

        Public Sub AddEachPartIfContainedInOtherSet(ByVal otherSet As BodyParts, ByVal parts As BodyParts)

            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If parts Is Nothing Then Throw New ArgumentNullException("parts")

            For i As Integer = 0 To parts.Count - 1
                AddIfContainedInOtherSet(otherSet, parts(i))
            Next
        End Sub

        Public Sub AddEachPartIfNotContainedInOtherSet(ByVal otherSet As BodyParts, ByVal parts As BodyParts)
            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            For i As Integer = 0 To parts.Count - 1
                AddIfNotContainedInOtherSet(otherSet, parts(i))
            Next
        End Sub

        Public Sub RemoveIfContainedInOtherSet(ByVal otherSet As BodyParts, ByVal part As BodyPart)
            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If otherSet.Contains(part) Then Remove(part)
        End Sub

        Public Sub RemoveIfNotContainedInOtherSet(ByVal otherSet As BodyParts, ByVal part As BodyPart)
            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If Not otherSet.Contains(part) Then Remove(part)
        End Sub

        Public Sub RemoveEachPartIfContainedInOtherSet(ByVal otherSet As BodyParts, ByVal parts As BodyParts)

            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If parts Is Nothing Then Throw New ArgumentNullException("parts")

            For i As Integer = 0 To parts.Count - 1
                RemoveIfContainedInOtherSet(otherSet, parts(i))
            Next
        End Sub

        Public Sub RemoveEachPartIfNotContainedInOtherSet(ByVal otherSet As BodyParts, ByVal parts As BodyParts)

            If otherSet Is Nothing Then Throw New ArgumentNullException("otherSet")
            If parts Is Nothing Then Throw New ArgumentNullException("parts")

            For i As Integer = 0 To parts.Count - 1
                RemoveIfNotContainedInOtherSet(otherSet, parts(i))
            Next
        End Sub

        Public Sub Remove(ByVal part As BodyPart)
            If mReadOnly Then Throw New ReadOnlyException
            If mParts.Contains(part) Then
                mParts.Remove(part)
                mParts.Sort()
                If Me.mInsideUpdate Then
                    mPartsRemovedDuringUpdate.Add(part)
                Else
                    Dim partsRemoved As New BodyParts
                    partsRemoved.mParts.Add(part)
                    SyncLock Me
                        RaiseEvent BodyPartsChangeEvent(Me, New BodyPartsChangeEventArgs(New BodyParts, partsRemoved))
                    End SyncLock
                End If

            End If
        End Sub

        Public Sub AddRange(ByVal parts As BodyParts)
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            If mReadOnly Then Throw New ReadOnlyException
            For i As Integer = 0 To parts.Count - 1
                Add(parts(i))
            Next
        End Sub

        Public Sub RemoveRange(ByVal parts As BodyParts)
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            If mReadOnly Then Throw New ReadOnlyException
            For i As Integer = 0 To parts.Count - 1
                Remove(parts(i))
            Next
        End Sub

        Public ReadOnly Property Count() As Integer
            Get
                Return mParts.Count
            End Get
        End Property

        Public Shared Function OnLeftHand(ByVal part As BodyPart) As Boolean
            Return _
                part = BodyPart.LeftThumb OrElse _
                part = BodyPart.LeftIndex OrElse _
                part = BodyPart.LeftMiddle OrElse _
                part = BodyPart.LeftRing OrElse _
                part = BodyPart.LeftLittle
        End Function

        Public Shared Function OnRightHand(ByVal part As BodyPart) As Boolean
            Return _
                           part = BodyPart.RightThumb OrElse _
                           part = BodyPart.RightIndex OrElse _
                           part = BodyPart.RightMiddle OrElse _
                           part = BodyPart.RightRing OrElse _
                           part = BodyPart.RightLittle
        End Function

        Public Function ContainsSet(ByVal parts As BodyParts) As Boolean
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            ContainsSet = True
            For i As Integer = 0 To parts.Count - 1
                If Not Contains(parts(i)) Then Return False
            Next
        End Function

        Public Function ContainsAny(ByVal parts As BodyParts) As Boolean
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            ContainsAny = False
            For i As Integer = 0 To parts.Count - 1
                If Contains(parts(i)) Then Return True
            Next
        End Function

        Public Function ContainsNone(ByVal parts As BodyParts) As Boolean
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            ContainsNone = True
            For i As Integer = 0 To parts.Count - 1
                If Contains(parts(i)) Then Return False
            Next
        End Function

        Public Overloads Function Equals(ByVal parts As BodyParts) As Boolean
            If parts Is Nothing Then Throw New ArgumentNullException("parts")
            Return Me.ContainsSet(parts) AndAlso parts.ContainsSet(Me)
        End Function

        Public Function Contains(ByVal part As BodyPart) As Boolean
            Return mParts.Contains(part)
        End Function

        Public Event BodyPartsChangeEvent(ByVal sender As Object, ByVal e As BodyPartsChangeEventArgs)


        Public Overloads Function ToString() As String
            Return ToString(CultureInfo.CurrentUICulture)
        End Function

        Public Overloads Function ToString(ByVal culture As CultureInfo) As String

            If mParts.Count = 0 Then Return InfrastructureMessages.Messages.None(culture)

            Dim builder As New System.Text.StringBuilder
            Dim workingCopy As BodyParts = Me.DeepCopy

            If workingCopy.ContainsSet(BodyParts.ForLeftHand) Then
                builder.Append(Messages.LeftHand(culture))
                workingCopy.RemoveRange(BodyParts.ForLeftHand)
            End If

            If workingCopy.ContainsSet(BodyParts.ForRightHand) Then
                builder.Append(Messages.RightHand(culture))
                workingCopy.RemoveRange(BodyParts.ForRightHand)
            End If


            For i As Integer = 0 To workingCopy.Count - 1
                Dim part As BodyPart = workingCopy(i)
                builder.Append(ToString(culture, part))
                If i <> mParts.Count - 1 Then
                    builder.Append(InfrastructureMessages.Messages.CommaSpace(culture))
                End If
            Next
            Return builder.ToString
        End Function

        Public Overloads Shared Function ToString(ByVal culture As CultureInfo, ByVal part As BodyPart) As String
            Select Case part
                Case BodyPart.Face
                    Return Messages.Face(culture)

                Case BodyPart.LeftIndex
                    Return Messages.LeftIndexFinger(culture)
                Case BodyPart.LeftIris
                    Return Messages.LeftIris(culture)

                Case BodyPart.LeftLittle
                    Return Messages.LeftLittleFinger(culture)

                Case BodyPart.LeftMiddle
                    Return Messages.LeftMiddleFinger(culture)

                Case BodyPart.LeftRing
                    Return Messages.LeftRingFinger(culture)

                Case BodyPart.LeftThumb
                    Return Messages.LeftThumb(culture)

                Case BodyPart.RightIndex
                    Return Messages.RightIndexFinger(culture)

                Case BodyPart.RightIris
                    Return Messages.RightIris(culture)

                Case BodyPart.RightLittle
                    Return Messages.RightLittleFinger(culture)

                Case BodyPart.RightMiddle
                    Return Messages.RightMiddleFinger(culture)

                Case BodyPart.RightRing
                    Return Messages.RightRingFinger(culture)

                Case BodyPart.RightThumb
                    Return Messages.RightThumb(culture)
                Case Else
                    Return part.ToString
            End Select

        End Function

        Public Function BodyCategory(ByVal culture As CultureInfo) As String
            If ContainsSet(ForLeftSlapTask) Then
                Return Messages.LeftSlap(culture)
            ElseIf ContainsSet(ForRightSlapTask) Then
                Return Messages.RightSlap(culture)
            ElseIf ContainsSet(ForThumbsSlapTask) Then
                Return Messages.ThumbsSlap(culture)
            ElseIf ContainsSet(ForLeftIrisTask) Then
                Return Messages.LeftIris(culture)
            ElseIf ContainsSet(ForRightIrisTask) Then
                Return Messages.RightIris(culture)
            ElseIf ContainsSet(ForFaceTask) Then
                Return Messages.Face(culture)
            End If
        End Function

        Public Sub New()

        End Sub

        Public Function DeepCopy() As BodyParts
            Dim bp As New BodyParts
            bp.AddRange(Me)
            bp.Description = Description
            Return bp
        End Function

        Public Sub New(ByVal part As BodyPart)
            MyClass.New()
            Add(part)
        End Sub


    End Class

    Public Class BodyPartsChangeEventArgs
        Inherits EventArgs
        Private mIsAddition As Boolean
        Private mPartsAdded As BodyParts
        Private mPartsRemoved As BodyParts

        Public ReadOnly Property ChangeWasAnAddition() As Boolean
            Get
                Return mIsAddition
            End Get
        End Property


        Public Property PartsAdded() As BodyParts
            Get
                Return mPartsAdded
            End Get
            Set(ByVal value As BodyParts)
                mPartsAdded = value
            End Set
        End Property

        Public Property PartsRemoved() As BodyParts
            Get
                Return mPartsRemoved
            End Get
            Set(ByVal value As BodyParts)
                mPartsRemoved = value
            End Set
        End Property


        Public Sub New(ByVal partsAdded As BodyParts, ByVal partsRemoved As BodyParts)
            mPartsAdded = partsAdded
            mPartsRemoved = partsRemoved
        End Sub

    End Class

End Namespace

