/*
* This software was developed at the National Institute of Standards and
* Technology (NIST) by employees of the Federal Government in the course
* of their official duties. Pursuant to title 17 Section 105 of the
* United States Code, this software is not subject to copyright protection
* and is in the public domain. NIST assumes no responsibility  whatsoever for
* its use by other parties, and makes no guarantees, expressed or implied,
* about its quality, reliability, or any other characteristic.
*/

#include <stdint.h>
#include <math.h>
#include "interpolate.h"

/*
Version		0.5.2
Last modified:	August 27, 2008
*/

/*
 * Convert rectilinear image to polar format.
 */
void rectilinear_to_polar(const uint8_t *rect_buffer, const unsigned int width, const unsigned int height,
                         uint8_t *polar_buffer,const unsigned int num_circum_samples,
                         const unsigned int num_radial_samples,
                         const unsigned int polar_center_x, const unsigned int polar_center_y,
                         const unsigned int inner_radius, const unsigned int outer_radius)
{

   /* nested iterators */
   double radial_index, angle_index;

   /* Each radial unit spans this many pixels in the rectilinear image */
   const double radial_resolution = (outer_radius - inner_radius) / (double)num_radial_samples;

   /* Each circumferal unit spans this many radians */
   const double circumferal_resolution = (M_PI + M_PI) / (double)num_circum_samples;

   /* do parameter checking */

   if (inner_radius >= outer_radius)
      ERR_EXIT("Inner radius larger than or equal to outer radius");

   /* foreach circumferal unit (rotate around conter-clockwise) */
   for (angle_index = 0; angle_index < num_circum_samples; angle_index++)
   {
      /* add pi/2 to the angle so that zero degrees is at 6 o'clock rather than 3 o'clock */
      const double angle = M_PI_2 - angle_index * circumferal_resolution;

      const double COS = cos(angle),
                   SIN = sin(angle);

      /* foreach radial unit (moving outward from center) */
      for (radial_index = 0; radial_index < num_radial_samples; radial_index++)
      {
         const double radius = inner_radius + radial_index * radial_resolution;

         /* convert polar coordinates to rectilinear */
         const double x = polar_center_x + COS * radius,
                      y = polar_center_y + SIN * radius;

         if (x < 1 || y < 1 || x >= width - 1 || y >= height - 1)
         {
            /* coordinates are not within the rectilinear bounday, so pad with zero */
            *polar_buffer = 0;

         } else
         {
            /* prepare for interpolation at (x,y) */

            /* define some variables for easy reference to four neighbouring pixels */
            const unsigned int xl = (int)x,
                               yl = (int)y,
	                       xh = xl + 1,
	                       yh = yl + 1;

            /* get fractional remainder */
            const double delta_x = x - xl,
                         delta_y = y - yl;

            /* do bilinear interpolation */
            *polar_buffer = (uint8_t) (.5 + delta_x * delta_y * rect_buffer[xh + yh * width] +
                                      (1 - delta_x) * delta_y * rect_buffer[xl + yh * width] +
                                      delta_x * (1 - delta_y) * rect_buffer[xh + yl * width] +
                                (1 - delta_x) * (1 - delta_y) * rect_buffer[xl + yl * width]);
         }

         polar_buffer += num_circum_samples;

      } /* end radial units loop */

      polar_buffer -= (num_radial_samples * num_circum_samples) - 1;

   } /* end circumferal units loop */

}

/*
 * Convert polar image to rectilinear format.
 * Assumes space for the rectilinear image has already been allocated, and that the
 * width and height are both 2 * outer radius distance.
 */
void polar_to_rectilinear(const uint8_t *polar_buffer,
                          const unsigned int num_circum_samples, const unsigned int num_radial_samples,
                          unsigned char *rect_raster_data,
                          const unsigned int inner_radius, const unsigned int outer_radius)
{
   /* nested iterators */
   unsigned int y, x;

   /* Assume width and height of rectilinear image are twice the outer radius distance */
   const unsigned int width = 2 * outer_radius,
                     height = 2 * outer_radius;

   /* Place the center of the polar image at the center of the rectilinear image */
   const int polar_center_x =  (width + 1) / 2,
             polar_center_y = (height + 1) / 2;

   /* Each radial unit spans this many pixels in the rectilinear image */
   const double radial_resolution = (outer_radius - inner_radius) / (double)num_radial_samples;

   /* Each circumferal unit spans this many radians */
   const double circum_resolution = (M_PI + M_PI) / (double)num_circum_samples;

   /* do parameter checking */

   if (inner_radius >= outer_radius)
      ERR_EXIT("Inner radius larger than or equal to outer radius");

   for (y = 0; y < height; y++)
   {
      const int ydiff = y - polar_center_y;

      for (x = 0; x < width; x++)
      {
         const int xdiff = x - polar_center_x;
         const double radius = pow(ydiff * ydiff + xdiff * xdiff, .5);
         const double radial_index = (radius - inner_radius) / radial_resolution;

         if (radial_index < 1 || radial_index > num_radial_samples - 2)
         {
            /* cordinates are not within the polar image, so pad with zero */
            *rect_raster_data = 0;

         } else {

            /* compute angle for (x, y), and put it into the range [0, 2*pi) */
            const double angle = fmod(M_PI + M_PI + M_PI_2 - atan2(ydiff, xdiff), M_PI + M_PI);

            const double circum_index = angle / circum_resolution;

            /* prepare for interpolation at (circum_index, radial_index) */

            /* define some variables for easy access to four neighbouring pixels */
            const unsigned int rl = (int)radial_index,
		 	       rh = rl + 1,
                               cl = (int)circum_index,
                               ch = (cl == num_circum_samples - 1) ? 0 : cl + 1;

            /* get fractional remainders */
            const double dc = circum_index - cl,
                         dr = radial_index - rl;

            /* do bilinear interpolation */
            *rect_raster_data = (uint8_t)(.5 + dc * dr * polar_buffer[ch + rh * num_circum_samples] +
                                         (1 - dc) * dr * polar_buffer[cl + rh * num_circum_samples] +
                                         dc * (1 - dr) * polar_buffer[ch + rl * num_circum_samples] +
                                   (1 - dc) * (1 - dr) * polar_buffer[cl + rl * num_circum_samples]);

         } /* end if */

         rect_raster_data++;

      } /* end foreach x */

   } /* end foreach y */

}

