/*
* This software was developed at the National Institute of Standards and
* Technology (NIST) by employees of the Federal Government in the course
* of their official duties. Pursuant to title 17 Section 105 of the
* United States Code, this software is not subject to copyright protection
* and is in the public domain. NIST assumes no responsibility  whatsoever for
* its use by other parties, and makes no guarantees, expressed or implied,
* about its quality, reliability, or any other characteristic.
*/

#include <stdio.h>
#include <errno.h>
#include <stdint.h>
#include <string.h>
#include "interpolate.h"

/*
Version		0.5.2
Last modified:	August 27, 2008
*/

/* Loads an uncompressed, binary PGM (Pixel Gray Map) file into memory.

   PGM format specification:
      1) A Magic number ("P5")
      2) Whitespace
      3) Image width
      4) Whitespace
      5) Image Height
      6) Whitespace
      7) Maximum Gray Value (should be 255)
      8) Whitespace
      9) Raster Data
*/
int loadPGM(const char *filename, uint8_t **img_raster, unsigned int *width, unsigned int *height)
{
   FILE *fp = fopen(filename, "rb");

   if (!fp)
      ERR_EXIT("Open of %s failed: %s\n", filename, strerror(errno));

   /* magic number test */
   {
      char magicNumber[9];
      fscanf(fp, "%s ", magicNumber);

      if (strcmp(magicNumber, "P5"))
         ERR_EXIT("Not a PGM file.\n");
   }

   fscanf(fp, "%u %u", width, height);

   /* read in max gray value */
   {
      unsigned int maxGrayValue;
      fscanf(fp, "%u", &maxGrayValue);

      if (maxGrayValue > 255)
         ERR_EXIT("Cannot process images with more than 8 bits per pixel.\n");
   }

   /* read in the raster data */
   {
      const unsigned int num_pixels = *width * *height;
      int num_bytes_read;

      *img_raster = malloc(num_pixels);

      if (! *img_raster)
         ERR_EXIT("Malloc failed.\n");

      num_bytes_read = fread(*img_raster, 1, num_pixels, fp);

      if (num_bytes_read != num_pixels)
         ERR_EXIT("Read incorrect number of bytes.\n");
   }

   return fclose(fp);
}

/* Stores an uncompressed raster image as a PGM file. */
int savePGM(const char *filename, const uint8_t *img_raster, const unsigned int width, const unsigned int height)
{
   FILE *fp = fopen(filename, "wb");

   if (!fp)
      ERR_EXIT("Open of %s failed: %s\n", filename, strerror(errno));

   fprintf(fp, "P5\n%u %u\n255\n", width, height);

   {
      const size_t num_pixels = width * height;

      size_t bytes_written = fwrite(img_raster, 1, num_pixels, fp);

      if (bytes_written != num_pixels)
         ERR_EXIT("Did not write correct number of bytes: %s\n", filename);
   }

   return fclose(fp);
}


int usage() {
   printf("usage:\n\tconvertToPolar pgmFile\n\n");
   return 1;
}

int main(int argc, const char *argv[])
{
   /* define "width" and "height" of polar image to be generated */
   const int num_radial_samples = 64,
             num_circum_samples = 512;

   const unsigned int inner_radius = 30,
                      outer_radius = 136;

   /* Specify what location in the rectilinear image should become
      the center of the polar image. */
   const unsigned int polar_center_x = 242,
                      polar_center_y = 295;


   /* will store the polar image pixel intensities */
   uint8_t *polar_raster;

   if (argc < 2)
      return usage();

   polar_raster = malloc(num_radial_samples * num_circum_samples);

   if (!polar_raster)
      ERR_EXIT("Could not allocate space for polar image.\n");


   /* Step 1: Rectilinear to polar conversion */
   {
      /* will store rectilinear image pixel intensities */
      uint8_t *rect_raster;

      /* width and height of rectilinear image */
      unsigned int width, height;

      loadPGM(argv[1], &rect_raster, &width, &height);

      printf("Converting rectilinear image to polar format.\n");

      rectilinear_to_polar(rect_raster, width, height,
                           polar_raster, num_circum_samples, num_radial_samples,
                           polar_center_x, polar_center_y,
                           inner_radius, outer_radius);

      /* Save "polarized" image to file */
      savePGM("polarized.pgm", polar_raster, num_circum_samples, num_radial_samples);

      free(rect_raster);
   }

   /* Step 2: Polar to rectilinear conversion */
   {
      /* generated rectilinear image is always 2 * outer radius distance  */
      const unsigned int width = 2 * outer_radius,
                        height = 2 * outer_radius;

      uint8_t *rect_raster = malloc(width * height);

      if (!rect_raster)
         ERR_EXIT("Could not allocate space for rectilinear image.\n");

      printf("Converting polar image to rectilinear format.\n");

      polar_to_rectilinear(polar_raster, num_circum_samples, num_radial_samples,
                           rect_raster, inner_radius, outer_radius);

      /* Save "depolarized" image to file */
      savePGM("depolarized.pgm", rect_raster, width, height);

      free(rect_raster);
   }

   free(polar_raster);

   return 0;
}

