#-----------------------------------------------------------------
# read LHPM/RIET7 ASCII data files
# the xray format is 10F8, neutron formats can be 10(I1,I9) or 10(I2,I8)
#------- define a command line option -----------------------------
# command line option 
lappend command(cmdopt) -lhpm
lappend command(cmdopt) -RIET7
# proc to use 
set command(-lhpm)  readLHPMdata
set command(-RIET7) readLHPMdata
#--------define a dialog box entry
# menu label
lappend command(readtypes) "LHPM/RIET7 raw data"
# proc for this entry
lappend command(readproc) ReadLHPM
# allowed data types
if {$tcl_platform(platform) == "windows"} {
    lappend command(filterlist) {dat}
} else {
    lappend command(filterlist) {dat DAT}
}
# definitions for these data types
set command(ReadLHPM_dat_type) "LHPM/RIET7 data"
set command(ReadLHPM_DAT_type) "LHPM/RIET7 data"


proc ReadLHPM {file} {
    global command
    if {$file == ""} return
    pleasewait "reading file $file"
    set ret [readLHPMdata $file]
    donewait
    if {$ret != ""} {return $ret}
    showlastentry $command(read_filelist)
}

proc readLHPMdata {filename} {

    #puts "opening LHPM/RIET7 file $filename"
    global graph

    set lnum 0
    # signal errors by quitting
    if [catch {
	set fp [open $filename r]

	# Skip up to the first 4 lines until we get a valid scan range
	
	for {set i 0} {$i<=4} {incr i} {
	    incr lnum
	    set eof [gets $fp line]
	    if {$eof < 0} {return "early end-of-file"} 
	    # use the first line as the title whatever it is
	    if {$i == 0} {set title $line}
	    # is this a valid start, step and end angle? 
	    set start [string trim [string range $line 0 7]]
	    set step  [string trim [string range $line 8 15]]
	    set last [string trim [string range $line 16 23]]
	    # check for valid numbers
	    if ![catch {set npts [expr 1. + ($last - $start)/$step]}] {break}
	}
	
	# check for valid numbers
	if [catch {set npts [expr 1. + ($last - $start)/$step]}] {
	    return "Invalid scan range: $start, $step, $last"
	}
	if {$npts > 60000} {return "There are $npts points -- something is wrong!"}

	# Great!! Now we are positioned to read the data.

	# Initialize the data structure.
	set data [file root [file tail $filename]] 
	# eliminate spaces from the name
	regsub -all " " $data "_" data
	set data [initdata $data]
	global ${data}
	set ${data}(xlabel) "2theta"
	set ${data}(xunits) "2theta"
	set ${data}(ylabel) "counts"
	
	# No missing points
	set ${data}(skip) 0
	set ${data}(title) $title

	# Now, read the rest of the file.
	# until we get to the last data point, an end-of-file or error
	set ptsread 0
	while {[gets $fp line] >= 0} {
	    # on the 1st line get the format
	    if {$ptsread == 0} {
		if {[llength $line] > 10} {
		    set type neutron
		    # for I1 we should see blanks in column 2, 10, ...
		    set blank 0
		    foreach num {1  9 17 25 33 41 49 57 65 73} {
			if {[string range $line $num $num] == ""} {set blank 1}
		    }
		    set l0 {0  8 16 24 32 40 48 56 64 72}
		    set l3 {7 15 23 31 39 47 55 63 71 79}
		    if $blank {
			set l1 {0  8 16 24 32 40 48 56 64 72}
			set l2 {1  9 17 25 33 41 49 57 65 73}
		    } else {
			set l1 {1  9 17 25 33 41 49 57 65 73}
			set l2 {2 10 18 26 34 42 50 58 66 74}
		    }	 
		} else { 
		    set type xray
		}
	    }
	    incr lnum
	    if {$lnum > $::command(maxlines)} {
		return "Read > $::command(maxlines) lines. Something is wrong or change command(maxlines)!"
	    }
	    if {$type == "xray"} {
		# assume a space between counts
		foreach count $line {
		    if {$count >= 0} {
			lappend ${data}(x) [expr $start + ($ptsread-1.)*$step]
			lappend ${data}(y) [expr $count]
			lappend ${data}(esd) [expr sqrt(1.*$count)]
		    } else {
			# there are skipped points
			set ${data}(skip) 1
		    }
		    incr ptsread
		}
	    } else { 
		foreach n0 $l0 n1 $l1 n2 $l2 n3 $l3 {
		    set ndet [string trim [string range $line $n0 $n1]]
		    set count [string trim [string range $line $n2 $n3]]
		    if {$count == ""} continue
		    if {$count >= 0} {
			lappend ${data}(x) [expr $start + ($ptsread-1.)*$step]
			if {$ndet == "" || $ndet == 0} {set ndet 1}
			lappend ${data}(y) [expr $count]
			lappend ${data}(esd) [expr sqrt(1.*$count/$ndet)]
		    } else {
			# there are skipped points
			set ${data}(skip) 1
		    }
		    incr ptsread
		}
	    }
	}
	if {$ptsread != $npts} {return "$ptsread were read, but expected $npts points!"}
    } errmsg] {
	# try to process the data we have read
	catch {resetdata $data}
	return "LHPM/RIET7 error. Error reading line $lnum: $errmsg"
    }
    resetdata $data
    lappend graph(plotlist) $data
    catch {close $fp} test
    #puts "$ptsread valid points read"
    return {}
}
