#-----------------------------------------------------------------
# read GSAS .raw files in either standard or ESD format
# files may be direct access or sequential
# version Oct 5th 2010 (V West, M Suchomel)
#   new features - attempts to read wavelength, temp, name from FXYE header
#-----------------------------------------------------------------
# limitations: 
#     This only reads CONS and FXYE format files, there are other types
#     It also only reads the first bank from a multibank file
#------- define a command line option -----------------------------
# command line option 
lappend command(cmdopt) -gsas
# proc to use 
set command(-gsas) readgsasdata
#--------define a dialog box entry
# menu label
lappend command(readtypes) "GSAS (fxye,raw,gsas)"
# proc for this entry
lappend command(readproc) ReadGSASRAW
# allowed data types
if {$tcl_platform(platform) == "windows"} {
    lappend command(filterlist) {gsas raw fxye}
} else {
    lappend command(filterlist) {gsas GSAS raw RAW fxye FXYE}
}
# definitions for these data types
set command(ReadGSASRAW_gsas_type) "GSAS raw data"
set command(ReadGSASRAW_GSAS_type) "GSAS raw data direct access"
set command(ReadGSASRAW_raw_type) "GSAS raw data"
set command(ReadGSASRAW_RAW_type) "GSAS raw data direct access"

proc ReadGSASRAW {file} {
    global command
    if {$file == ""} return
    pleasewait "reading file $file"
    set ret [readgsasdata $file]
    donewait
    if {$ret != ""} {return $ret}
    showlastentry $command(read_filelist)
}

proc readgsasdata {filename} {
	set wavelength ""
	set temperature ""
	set samplename ""
    global graph
    # signal errors by quitting
    if [catch {

	set fp [open $filename r]
	set lnum 0

	# 1st line is title
	incr lnum
	set eof [gets $fp title]
	# is this a direct access file?
	if {$eof > 100} {
	    set direct 1
	    # rewind file
	    seek $fp 0
	    # read again
	    set title [read $fp 80]
	} else {
	    set direct 0
	}
	if {$eof < 0} {return "end-of-file before BANK found"}

	# second line may be "Instrument parameter" line followed by #comments
        set skip 1
        while {$skip} {
	    incr lnum
	    if $direct {
		set line [read $fp 80]
		if {[string length $line] < 80} {set eof -1}
	    } else {
		set eof [gets $fp line]
	    }
    # read from FXYE header
	    if {$eof < 0} {return "end-of-file before BANK found"}
	    if {[string first "Calibrated wavelength" $line] != -1} {
		set wavelength [string trim [lindex [split $line "="] 1]]
	    }
	    if {[string first "# Sample name" $line] != -1} {
		set samplename [string trim [lindex [split $line "="] 1]]
	    }
	    if {[string first "# User sample name" $line] != -1} {
		set samplename [string trim [lindex [split $line "="] 1]]
	    }
	    if {[string first "# Temp (K)" $line] != -1} {
		set temperature [string trim [lindex [split $line "="] 1]]
	    }
	 # skip line if it does not contain BANK starting in the 1st column
	    if {[string range $line 0 3] != "BANK"} {
	    if $direct {
		set line [read $fp 80]
		if {[string length $line] < 80} {set eof -1}
	    }
	    if {$eof < 0} {return "end-of-file before BANK found"} 
	  }
	  if {[string range $line 0 3] == "BANK"} { set skip 0 }
        }
	if {[string range [lindex $line 4] 0 3] != "CONS"} {
	    return "This is an unsupported file type: [lindex $line 4]"
	}
	set start [lindex $line 5]
	set step [lindex $line 6]
	set npts [lindex $line 2]
	if {[lindex $line 9] == "ESD"} {
	    set esdflag 1
	} else {
	    set esdflag 0
	}
	if {[lindex $line 9] == "FXYE"} {
	    set fxyeflag 1
	} else {
	    set fxyeflag 0
	}

	set ptnum 0
	set data [file root [file tail $filename]] 
	# eliminate spaces from the name
	regsub -all " " $data "_" data
	set data [initdata $data]
	global ${data}
	set ${data}(title) [string trim [string range $title 0 45]]
	set ${data}(xlabel) "2theta"
	set ${data}(xunits) "2theta"

	set ptsread 0
	# are there missing points?
	set ${data}(skip) 0
	if {$esdflag} {
	    set ${data}(ylabel) "scaled counts"
	    while {$ptnum < $npts} {
		incr lnum
		if $direct {
		    set line [read $fp 80]
		    if {[string length $line] < 80} {set eof -1}
		} else {
		    set eof [gets $fp line]
		}
		if {$lnum > 120000} {
		    return "Read too many lines. Something is wrong!"
		}
		if {$eof < 0} {return "end-of-file before all data read"} 
		foreach num {0 16 32 48 64} num7 {7 23 39 55 71} \
			num8 {8 24 40 56 72} num15 {15 31 47 63 79} {
		    if {$ptnum < $npts} {
			set count [string trim [string range $line $num $num7]]
			set esd [string trim [string range $line $num8 $num15]]
			if {[string trim $count] == ""} {set count 0}
			if {$count >= 0 && $esd > 0} {
			    lappend ${data}(x) \
				    [expr ($start+$step*$ptnum)/100.]
			    lappend ${data}(y) $count
			    lappend ${data}(esd) $esd
			    incr ptsread
			} else {
			    # missing points
			    set ${data}(skip) 1
			}
			incr ptnum
		    }
		}
	    }
	} else {
            if {$fxyeflag} {
              while {$ptnum < $npts} {
		if $direct {
		    set line [read $fp 80]
		    if {[string length $line] < 80} {set eof -1}
		} else {
		    set eof [gets $fp line]
		}
		if {$lnum > 120000} {
		   return "Read too many lines. Something is wrong!"
		}
		if {$eof < 0} {return "end-of-file before all data read"}
	        set pos [lindex $line 0]
	        set count [lindex $line 1]
                set sig [lindex $line 2]
                lappend ${data}(x) [expr $pos/100.]
                lappend ${data}(y) [expr $count]
                lappend ${data}(esd) [expr $sig]
                incr ptnum
              }
            } else {
	       set ${data}(ylabel) "counts"
	       while {$ptnum < $npts} {
		   incr lnum
		   if $direct {
		       set line [read $fp 80]
		       if {[string length $line] < 80} {set eof -1}
		   } else {
		      set eof [gets $fp line]
		   }
		   if {$lnum > 20000} {
		      return "Read too many lines. Something is wrong!"
		   }
		   if {$eof < 0} {return "end-of-file before all data read"}
		   foreach num  {0  8 16 24 32 40 48 56 64 72} \
			num1 {1  9 17 25 33 41 49 57 65 73} \
			num2 {2 10 18 26 34 42 50 58 66 74} \
			num7 {7 15 23 31 39 47 55 63 71 79} {
		     if {$ptnum < $npts} {
			set ndet [string trim [string range $line $num $num1]]
			set count [string trim [string range $line $num2 $num7]]
			lappend ${data}(x) \
				[expr ($start+$step*$ptnum)/100.]
			if {$ndet == "" || $ndet == 0} {set ndet 1}
			if {[string trim $count] == ""} {set count 0}
			lappend ${data}(y) [expr $count]
			incr ptsread
			if {$count >= 0} {
			    lappend ${data}(esd) [expr sqrt(1.*$count/$ndet)]
			} else {
			    lappend ${data}(esd) 0
			}
			incr ptnum
		     }
		  }
               }
	    }
	}
    } errmsg] {
	# try to process the data we have read
	catch {resetdata $data}
	return "GSAS raw error. Error reading line $lnum: $errmsg"
    }
    catch {
		set fxyetitle ""
		if { $samplename != "" } { 
			append fxyetitle $samplename 
		}
		if { $temperature != "" } {
		  if { $samplename != "" } {
		      append fxyetitle ", "
		  }
		  append fxyetitle $temperature
		  append fxyetitle " K"
		  append fxyetitle " ("		  
		  append fxyetitle [expr int($temperature-273)]
		  append fxyetitle " C)"		  
		  }
		if { $fxyetitle != "" } {set ${data}(title) $fxyetitle}
	if { $wavelength != "" } { 
		set ${data}(wavelength) $wavelength 
		set ${data}(xunits) "2theta"
		set ${data}(ylabel) "counts"
	}
	resetdata $data
	lappend graph(plotlist) $data
    }
    catch {close $fp} test
    #puts "$ptsread valid points read"
    return {}
}
