# open all output file after foreground and background run
# view with peak list or raw data

# $Revision: 458 $ $Date: 2010-10-31 10:49:29 -0500 (Sun, 31 Oct 2010) $
#--------------------------------------------------------------------
# GUI for autoindexing routines
#
# initial conditions
set aindex(sort) -2
set aindex(peaklist) {}
set aindex(font) "Courier"
set aindex(xmin) 0
set aindex(xmax) 60
set aindex(ymin) 0
set aindex(ymax) 100

array set ITO_INFO {
    MAN "Prints instructions, unless MAN=9"
    INSTR "Prints instructions for special features if INSTR=1"
    INTENS "Reads intensities besides line positions (8(f7.3,A3)) if INTENS=1"
    NSOLMX "Number of solutions printed out extensively."
    TOL2 "Tolerance (range) on 2-dimensional search."
    TOL3 "Tolerance(range) on 3-dimensional search"
    LINCO "If LINCO .gt. 0, Enter line combinations or zones (4F10.2), End with a blank card,or a card with a zero in the first column."
    LZERCK "Gives a check on zero-errors when LZERCK > 0"
    K(ENL) "Gives output of intermediate results, if K .ge. 1"
    NQ1 "NQ1 and NQ2 determine the numbers of lines to be used in combinations for finding zones."
    NQ2 "NQ1 and NQ2 determine the numbers of lines to be used in combinations for finding zones."
    NZ1 "NZ1 and NZ2 determine the numbers of lines to be used in combinations for finding lattices from zones"
    NZ2 "NZ1 and NZ2 determine the numbers of lines to be used in combinations for finding lattices from zones"
    NR  "Refinement and evaluation only if NR .gt. 0.  Trial lattices required"
    INDAT "Number of the unit on which to read the line positions"
    LIST  "Prints a list of all calc and obs lines for *list* lattices"
    JTEST  "Test output is  printed when J.ge.1"
    WMOL "Molecular weight"
    DOBS "Observed density"
    TOLG "Tolerance on the match between calculated and observed two thetas."
    NTST "Number indicating where in the main program test output should start on second card"
    PRNTMR "The minimum figure of merit for a lattice to be printed"
    PRNTLN "The minimum number of indexed lines for a lattice to be printed"
}
# default ITO values
array set ITO {
	MAN    9
	INSTR  0
	INTENS 0
	NSOLMX 4
	TOL2 3.0
	TOL3 4.5
	LINCO  0
	LZERCK 0
	K(ENL) 0
	NQ1 3
	NQ2 6
	NZ1 6
	NZ2 6
	NR  0
	INDAT 0
	LIST  1
	JTEST 0
	DOBS 0.0
	WMOL 0.0
	TOLG 6.0
	NTST  0
	PRNTMR   4.0
	PRNTLN  14.0
}


array set TREOR_INFO {
 KH "=4 MAX H FOR CUBIC BASE LINE." 
 KK "=4 MAX K FOR CUBIC BASE LINE."
 KL "=4 MAX L FOR CUBIC BASE LINE."
 KS "=6 MAX H+K+L FOR THIS LINE. OBS. IF KS=0 CUBIC TEST OMITTED."
 THH "=4 MAX H FOR TETRAGONAL AND HEXAGONAL BASE LINES."
 THK "=4 MAX K FOR TETRAGONAL AND HEXAGONAL BASE LINES."
 THL "=4 MAX L FOR TETRAGONAL AND HEXAGONAL BASE LINES."
 THS "=4 MAX H+K+L FOR THESE LINES. OBS. IF THS=0 TETRAGONAL AND
HEXAGONAL TESTS OMITTED."
 OH1 "=2 MAX H FOR THE FIRST ORTHORHOMBIC BASE LINE."
 OK1 "=2 MAX K FOR THE FIRST ORTHORHOMBIC BASE LINE."
 OL1 "=2 MAX L FOR THE FIRST ORTHORHOMBIC BASE LINE."
 OS1 "=3 MAX H+K+L FOR THIS LINE. OBS. IF OS1=0 ORTHORHOMBIC TEST
OMITTED."
 OH2 "=2 MAX H FOR THE SECOND ORTHORHOMBIC BASE LINE."
 OK2 "=2 MAX K FOR THE SECOND ORTHORHOMBIC BASE LINE."
 OL2 "=2 MAX L FOR THE SECOND ORTHORHOMBIC BASE LINE."
 OS2 "=4 MAX H+K+L FOR THIS LINE."
 OH3 "=2 MAX H FOR THE THIRD ORTHORHOMBIC BASE LINE."
 OK3 "=2 MAX K FOR THE THIRD ORTHORHOMBIC BASE LINE."
 OL3 "=2 MAX L FOR THE THIRD ORTHORHOMBIC BASE LINE."
 OS3 "=4 MAX H+K+L FOR THIS LINE."
 MH1 "=2 MAX ABS(H) FOR THE FIRST MONOCLINIC BASE LINE."
 MK1 "=2 MAX K FOR THE FIRST MONOCLINIC BASE LINE."
 ML1 "=2 MAX L FOR THE FIRST MONOCLINIC BASE LINE."
 MS1 "=2 MAX ABS(H)+K+L FOR THIS LINE.THE NORMAL (AND FAST) WAY TO TEST
ONE EXPECTED CELL EDGE PARAMETER IS TO PUT IT IN AS Q NUMBER ONE (CARD
SET TWO) AND SET MH1=1,MK1=1,ML1=0,MS1=1."
 MH2 "=2 MAX ABS(H) FOR THE SECOND MONOCLINIC BASE LINE."
 MK2 "=2 MAX K FOR THE SECOND MONOCLINIC BASE LINE."
 ML2 "=2 MAX L FOR THE SECOND MONOCLINIC BASE LINE."
 MS2 "=3 MAX ABS(H)+K+L FOR THIS LINE."
 MH3 "=2 MAX ABS(H) FOR THE THIRD MONOCLINIC BASE LINE."
 MK3 "=2 MAX K FOR THE THIRD MONOCLINIC BASE LINE."
 ML3 "=2 MAX L FOR THE THIRD MONOCLINIC BASE LINE."
 MS3 "=3 MAX ABS(H)+K+L FOR THIS LINE."
 MH4 "=2 MAX ABS(H) FOR THE FOURTH MONOCLINIC BASE LINE."
 MK4 "=2 MAX K FOR THE FOURTH MONOCLINIC BASE LINE."
 ML4 "=2 MAX L FOR THE FOURTH MONOCLINIC BASE LINE."
 MS4 "=4 MAX ABS(H)+K+L FOR THIS LINE."
 MONOSET "=0 THIS PARAMETER MAKES IT POSSIBLE TO USE MORE THAN 3 SETS OF
BASE LINES IN THE MONOCLINIC TRIALS. IF MONOSET IS:
GREATER THAN 3 THE BASE LINE SET (1,3,4,5) WILL BE USED
GREATER THAN 4 THE BASE LINE SET (1,2,3,6) WILL BE USED
GREATER THAN 5 THE BASE LINE SET (2,3,4,5) WILL BE USED
GREATER THAN 6 THE BASE LINE SET (1,2,3,7) WILL BE USED
THUS MAX 7 BASE LINE SETS CAN BE USED."
 MONOGAM "=1 THE BEST 5 TRIAL PARAMETER SETS STORE (SEE PARAMETER 'IQ')
FOR EACH BASE LINE SET WILL BE
REFINED BEFORE NEXT BASE LINE SET IS TRIED. IF MONOGAM=0 ALL BASE LINE
SETS ARE TRIED BEFORE ANY
REFINEMENT IS PERFORMED. THE PARAMETER IS ONLY USED IN THE MONOCLINIC
TESTS."
 MONO "=0 MAX BETA ANGLE ALLOWED IN A MONOCLINIC CELL. OBS. NO
MONOCLINIC TEST IF MONO=0"
 SHORT "=1 SHORT AXIS TEST. THE PARAMETER IS ONLY VALID FOR MONOCLINIC
TESTS. THE FIRST SIX
LINES ARE TESTED FOR THE OCCURRENCE OF A COMMON ZERO INDEX IN THE SIX
FIRST LINES. IF SHORT=0 NO SHORT
AXIS TEST. IF YOU WANT TO MAKE THIS TEST WITHOUT REPEATING OTHER
MONOCLINIC TESTS YOU MAY GIVE THE
PARAMETER MONO A NEGATIVE SIGN."
 USE "=19 -OR EQ. TO THE NUMBER OF INPUT LINES IF THERE ARE LESS THAN 19
LINES, -OR EQ. TO THE NUMBER OF LINES WITH SINE SQUARE(THETAS) LESS THAN
0.327 -'USE' IS THE NUMBER OF LINES USED IN THE TRIAL-INDEXING PART OF
THE CALCULATIONS. OBS. MAX USE=20 OBS. IF YOU WANT TO CHANGE THIS
PARAMETER YOU SHOULD ALSO CHANGE THE PARAMETER IQ."
 IQ "=USE-3  THE NUMBER OF INDEXABLE LINES REQUIRED IN THE
TRIAL-INDEXING PROCEDURE IF THE CELL SHOULD BE STORED FOR EV.
LEAST-SQUARES REFINEMENT. THESE RECIPROCAL CELL PARAMETERS ARE
PRINTED IF THE PARAMETER LIST=1"
 LIST "=0 SEE IQ."
 SELECT "=0 IF 'SELECT' IS NON ZERO THE ORTHORHOMBIC BASE LINES ARE
(SELECT,1,2) (SELECT,1,3) AND (SELECT,2,3) IF 'SELECT' IS GREATER THAN 5
THE MONOCLINIC BASE LINES ARE (SELECT,1,2,3) (SELECT,1,2,4) AND
(SELECT,1,3,4)"
 MERIT "=10 FIGURE OF MERIT REQUIRED AS STOP LIMIT. THE FIGURE OF MERIT
IS FOR AN ORTHORHOMBIC CELL DEFINED BY DE WOLFF,P.M., J.APPL.CRYST.
1(1968)108-113. ( FOR THE CUBIC ,TETRAGONAL AND HEXAGONAL SYMMETRIES ARE
THE DIFFERENT QUADRATIC FORMS AS GIVEN IN INT. TABL. X-RAY CRYST.(1968)
VOL.2 P.109-145 USED IN THE CALCULATION OF THE NUMBER OF THEORETICAL
LINES.) OBS. THE FIGURE OF MERIT CALCULATIONS ARE NOT STRICTLY VALID
UNLESS ALL TWENTY FIRST LINES ARE INDEXED."
 NIX "=1 IF A CELL AFTER LEAST SQUARES REFINEMENT HAS A FIGURE OF MERIT
EQ. TO OR GREATER THAN THE PARAMETER  'MERIT' AND THE NUMBER OF NOT
INDEXABLE LINES AMONG THE 'USE' FIRST LINES ARE LESS THAN OR EQ. TO
'NIX' THE CALCULATIONS ARE STOPPED.  OBS. OTHERWISE THE CALCULATIONS
WILL END WITH A DIFFERENCE ANALYSIS (PROGRAM I1.  WERNER, P.-E. 
Z.KRISTALLOGR. 120(1964)375-378)"
 IDIV "=1 THE 7 FIRST LINES ARE ADJUSTED BY (EVENTUALL OCCURRING) HIGHER
ORDER LINES.IF IDIV=0  NO CORRECTIONS.USUALLY THE DEFAULT VALUE 1 IS
O.K. THERE ARE EXEPTIONS, HOWEVER. IF INDEXING IS NOT SUCCESSFUL IT IS
RECOMMENDED TO TRY IDIV=0."
 VOL "=2000 MAX CELL VOLUME (IN ANGSTROEM**3) A new option available in
TREOR90 is to give a negative value of VOL, ex. VOL=-2000."
 CEM "=25 MAX CELL EDGE (IN ANGSTROEM) THE COMPUTING TIME IS STRONGLY
DEPENDENT ON THE VOL AND CEM PARAMETERS. THEREFORE,IF POSSIBLE PUT IN
SMALLER VALUES (AT LEAST IN MONOCLINIC TRIALS)"
 D1 "=0.0002 DEFINED AS FOR PROGRAM PIRUM. (WERNER P.E. ARKIV KEMI
31(1969)513-516)"
 SSQTL "=0.05 DEFINED AS FOR PROGRAM PIRUM."
 D2 "=0.0004 DEFINED AS FOR PROGRAM PIRUM. A LINE IS CONSIDERED AS
INDEXED IF.. SINE SQUARE THETA IS LESS THAN 0.05 AND ABS(SINE SQUARE
THETA OBSERVED MINUS SINE SQUARE THETA CALCULATED) IS LESS THAN D1
OR.... IF SINE SQUARE THETA IS GREATER THAN 0.05 AND THE CORRESPONDING
DIFFERENCE IS LESS THAN D2. OBS. THE PARAMETERS D1,SSQTL AND D2 ARE USED
IN THE TRIAL INDEXING PART AS WELL AS THE LEAST-SQUARES REFINEMENTS.
OBS. 'D1,SSQTL AND D2' ARE DEPENDENT ON 'WAVE' OBS. THE PARAMETER D2 IS
ALSO USED IN THE DIFFERENCE ANALYSIS."
 CHOICE "=0 INDICATOR DEFINING 'SQ' ON CARD SET TWO.
                        CHOICE=0   SQ=SINE SQUARE THETA
                              =1   SQ=1/(D*D)    ('D'-SPACING IN
ANGSTROEM)
                              =2   SQ=THETA      ('THETA'=BRAGG ANGLE IN
DEG.)
                              =3   SQ=2*THETA
                              =4   SQ=D
                        OBS. IF 'CHOICE' IS NON ZERO IT IS ALWAYS
POSSIBLE TO USE WAVE=1.54051 (THE NORMAL VALUE)"
 DENS "=0 DENSITY NOT USED.IF ONLY AN INTEGRAL NUMBER OF MOLECULES IN
THE UNIT CELL IS ACCEPTED THE PARAMETERS DENS,EDENS AND MOLW MAY BE
USED.      (ON YOUR OWN RESPONSIBILITY) DENS EQ. DENSITY IN GRAM PER
CM**3"
 EDENS "=0 NOT USED UNLESS DENS EQ. NON ZERO. EDENS EQ. MAX. DEVIATION
IN THE PARAMETER DENS. OBS. DENS AND EDENS ARE USED IN TRIAL
CALCULATIONS I.E. THEY ARE USED IN TESTS ON NON REFINED UNIT CELLS. IT
IS THEREFORE RECOMMENDED TO USE AN EDENS WHICH IS THE EXPECTED MAX.
DEVIATION IN DENS PLUS 5-10 PER CENT OF DENS. OBS. THE CHOICE OF EDENS
SHOULD BE DEPENDENT ON THE QUALITY OF YOUR DIFFRACTION DATA."
 MOLW "=0 NOT USED UNLESS DENS ( AND EDENS ) EQ. NON ZERO MOL. WEIGHT IN
A.U. (OBS CRYSTAL WATER INCLUDED) THE PARAMETERS DENS,EDENS AND MOLW
(IF
KNOWN) MAY BE USED IN MONOCLINIC AND TRICLINIC  TESTS IN ORDER TO
REDUCE
THE COMPUTER TIME NEEDED. IT IS NOT RECOMMENDED TO USE THESE
PARAMETERS
FOR ORTHORHOMBIC AND HIGHER SYMMETRIES."
 TRIC "=0 NO TRICLINIC TEST. IF TRIC=1 ALL HIGHER SYMMETRY TESTS ARE
OMITTED AND A (TIMECONSUMING) TRICLINIC TEST IS MADE. IT IS PRESUPPOSED
THAT ALL HIGHER SYMMETRIES HAVE BEEN TRIED IN EARLIER RUNS. ALTHOUGH IT
IS IN PRINCIPLE POSSIBLE TO INDEX ANY PATTERN AS TRICLINIC, THE INDEXING
ALGORITHM USED HERE IS MORE EFFECTIVE FOR TRUE TRICLINIC THAN FOR
PSEUDO- TRICLINIC PATTERNS. FOR EXAMPLE A MONOCLINIC PATTERN MAY BE
CORRECTLY INDEXED BY A TRICLINIC CELL (USING TRIC=1) BUT THIS IS NOT THE
RECOMMENDED PROCEDURE. FURTHERMORE, THE TRICLINIC TEST IS
TIMECONSUMING
(TYPICAL 10-20 MINUTES CPU-TIME ON A VAX 11/750). OBS. FOR A TRUE
TRICLINIC CELL THE PARAMETER VOL  MAY BE GIVEN THE ESTIMATED VALUE (C.F.
THE WARNINGS GIVEN ABOVE -SEE  S T R A T E G Y....) PLUS 200 A**3."
LIMIT "=10 LIMIT can be any integer between 1 and 10.
The parameter LIMIT can be given a value  less than 10 in order
to reduce the maximum number of zero shifts tested. (The number
of zero shifts applied will be  dependent on the result of the
the calculations and is therefore usually less than 10).   
If you have good reason to  belive that the zero point error  is
negligible, you may save a lot of computer time by limit=1"
}
foreach item [array names TREOR_INFO] {set TREOR($item) ""}
# are there any non-blank TREOR defaults? (CHOICE is set in WriteTREORFile)
array set TREOR_Compound {
    .synchrotron {
	{"" "Sets commonly used options for indexing synchrotron patterns"}
	{LIMIT 1}
	{CHOICE 3}
	{MERIT 50}
	{NIX 0}
	{VOL 6000}
    }
}

array set DICVOL_INFO {
    N_USE "Number of lines to be used for searching, typically 20"
 AMAX "MAXIMUM VALUE OF UNIT CELL DIMENSION A IN ANGSTROMS. (IF AMAX= 0.0 DEFAULT= 25. ANGSTROMS)"
 BMAX "MAXIMUM VALUE OF UNIT CELL DIMENSION B IN ANGSTROMS. (IF BMAX= 0.0 DEFAULT= 25. ANGSTROMS)"
 CMAX "MAXIMUM VALUE OF UNIT CELL DIMENSION C IN ANGSTROMS. (IF CMAX= 0.0 DEFAULT= 25. ANGSTROMS)"
 VOLMIN "MINIMUM VOLUME FOR TRIAL UNIT CELLS IN ANGSTROMS**3."
 VOLMAX  "MAXIMUM VOLUME FOR TRIAL UNIT CELLS IN ANGSTROMS**3.(IF VOLMAX= 0.0 DEFAULT= 2500. ANGSTROMS**3)"
 BEMIN "MINIMUM ANGLE FOR UNIT CELL IN DEGREES (IF BEMIN= 0.0 DEFAULT= 90. DEGREES)."
 BEMAX "MAXIMUM ANGLE FOR UNIT CELL IN DEGREES (IF BEMAX= 0.0 DEFAULT= 125. DEGREES)."
 POIMOL "MOLECULAR WEIGHT OF ONE FORMULA UNIT IN A.M.U. (DEFAULT =0.0 IF FORMULA WEIGHT NOT KNOWN)."
 DENS    "MEASURED DENSITY IN G.CM(-3) (DEFAULT =0.0 IF DENSITY NOT KNOWN)."
 DELDEN  "ABSOLUTE ERROR IN MEASURED DENSITY."
 EPS "=0.0  THE ABSOLUTE ERROR ON EACH OBSERVED LINE IS TAKEN TO .03 DEG. 2THETA, WHATEVER THE SPACING DATA TYPE (ITYPE IN CARD 2).
                    =1.0  THE ABSOLUTE ERROR ON EACH OBSERVED LINE IS INPUT INDIVIDUALLY IN THE FOLLOWING CARDS, TOGETHER WITH THE OBSERVED 'D(I)', ACCORDING WITH THE SPACING DATA UNIT.
                    EPS NE 0.0 AND 1.0 THE ABSOLUTE ERROR IS TAKEN AS A CONSTANT (=EPS),IN DEG. 2THETA, WHATEVER THE SPACING DATA TYPE (ITYPE IN CARD 2)."
 FOM "LOWER FIGURE OF MERIT M(N) REQUIRED FOR PRINTED SOLUTION(S) (DEFAULT=0.0 IF LOWER M(N)=5.0)."
 N_IMP "MAX NUM. OF IMPURITY/SPURIOUS LINES ALLOWED. IF N_IMP<0  THE SEARCH STARTS WITH 0 IMPURITY LINES, AND THE NUMBER IS INCREASED UNTIL |N_IMP| IMPURITY LINES ARE TRIED."
 ZERO  "SEARCH FOR A ZERO-POINT ERROR IN INPUT DATA: =0 MEANS NO SEARCH; =1 MEANS SEARCH; IF ANY OTHER VALUE IS SUPPLIED, THE VALUE IS USED AS THE ZERO CORRECTION IN DEGREES 2THETA."
 ZERO_REF "REFINE ZERO?  =0 MEANS NO FIT; =1 PERFORMS A ZERO-POINT LEAST-SQUARES REFINEMENT."
 OPTION   "OPTIMIZED/EXTENDED SEARCH: =0: OPTIMIZED STRATEGY SEARCH WITH DECREASING CELL VOLUMES; =1: EXHAUSTIVE SEARCH IN VOLUME DOMAINS CONTAINING MATHEMATICAL SOLUTION(S) [Slower]"
}
foreach item [array names DICVOL_INFO] {set DICVOL($item) "0"}
array set DICVOL {EPS 0.03 N_USE 20 ZERO_REF 1} 

proc MakeIndex {page} {
    global graph gaussian aindex
    pack [frame $page.a] -side left -fill y -anchor n
    grid [frame $page.a.p -bd 4 -relief groove] -column 2 -columnspan 2 -row 0
    pack [label $page.a.p.1 -text {Select a peak list}] -side top -fill both

    pack [frame $page.a1 -borderwidth 3 -relief groove]  -side left -fill y -anchor n -expand yes
    #################################################################
    # put indexing parameters here inside $page.a1 using global 
    # variables in aindex()
    #################################################################
    set aindex(wavelength) {}
    set aindex(zerocorrection) 0.0
    set aindex(cubic) 1
    set aindex(tetragonal) 1
    set aindex(hexagonal) 1
    set aindex(orthorhombic) 1
    set aindex(monoclinic) 1
    set aindex(triclinic) 0

    set rown 0
    grid [label $page.a1.c1$rown -text {Wavelength:}] -column 0 -row $rown
    grid [entry $page.a1.c2$rown -width 8 \
	    -textvariable aindex(wavelength)] \
	    -column 1 -row $rown -sticky w
    incr rown
    grid [label $page.a1.c1$rown -text "Zero offset:"] -column 0 -row $rown
    grid [entry $page.a1.c2$rown -width 8 \
	    -textvariable aindex(zerocorrection)] \
	    -column 1 -row $rown -sticky w
    incr rown
    grid [label $page.a1.c1$rown -text "Allowed cell symmetries:"] -column 0 -row $rown -columnspan 2
    incr rown
    grid [checkbutton $page.a1.c2$rown -text Cubic \
	    -variable aindex(cubic)] -column 0 -row $rown -sticky w
    grid [checkbutton $page.a1.c4$rown  -text Orthorhombic \
	    -variable aindex(orthorhombic)] -column 1 -row $rown -sticky w

    incr rown
    grid [checkbutton $page.a1.c2$rown -text Tetragonal \
	    -variable aindex(tetragonal)] -column 0 -row $rown -sticky w
    grid [checkbutton $page.a1.c4$rown -text Monoclinic \
	    -variable aindex(monoclinic)] -column 1 -row $rown -sticky w
    incr rown
    grid [checkbutton $page.a1.c2$rown -text Hexagonal \
	    -variable aindex(hexagonal)] -column 0 -row $rown -sticky w
    grid [checkbutton $page.a1.c4$rown -text Triclinic \
	    -variable aindex(triclinic)] -column 1 -row $rown -sticky w

    incr rown
    grid [frame $page.a1.f0 -bd 2 -relief groove] \
	    -column 0 -row $rown -columnspan 2
    grid rowconfig $page.a1 $rown -weight 1
    grid [label $page.a1.f0.c0 -text "Program-specific options" -anchor c] \
	    -column 0 -row 0 -sticky ew -columnspan 3
    grid [frame $page.a1.f0.f1] \
	    -column 0 -row 1 -columnspan 4 -sticky ew 

    grid [radiobutton $page.a1.f0.f1.vito -text ITO \
	    -variable aindex(setvar) -value ITO \
	    -command "FillIndexHelpList $page ITO"\
	    ] -column 0 -row 1 -sticky w

    grid [radiobutton $page.a1.f0.f1.vtreor -text TREOR \
	    -variable aindex(setvar) -value TREOR \
	    -command "FillIndexHelpList $page TREOR"\
	    ] -column 1 -row 1 -sticky w

    grid [radiobutton $page.a1.f0.f1.vdicvol -text DICVOL \
	    -variable aindex(setvar) -value DICVOL \
	    -command "FillIndexHelpList $page DICVOL"\
	    ] -column 2 -row 1 -sticky w


    grid [listbox $page.a1.f0.lb  -height 4 -width 10 -bd 2 \
	    -yscrollcommand "$page.a1.f0.sb set" -relief raised] \
	    -column 0 -row 2 -rowspan 2
    grid [scrollbar $page.a1.f0.sb -command "$page.a1.f0.lb yview"] \
	    -column 1 -row 2 -sticky ns -rowspan 2
    grid [label $page.a1.f0.lab -textvariable aindex(helplbl)] \
	    -column 2 -row 2
    button $page.a1.f0.entbutton -text "Set values"
    grid [entry $page.a1.f0.ent -width 10]  -column 2 -row 3 -columnspan 2
    grid [text $page.a1.f0.msg -wrap word -width 25 -height 3 \
	    -yscrollcommand "$page.a1.f0.msb set"] \
	    -column 0 -row 4 -columnspan 3 -sticky ewns
    grid [scrollbar $page.a1.f0.msb -command "$page.a1.f0.msg yview"] \
	    -column 4 -row 4 -sticky ns
    bind $page.a1.f0.lb <ButtonRelease-1> "FillIndexHelpMsg $page.a1.f0.lb $page.a1.f0.msg $page.a1.f0.ent"
    grid columnconfig $page.a1.f0 2 -weight 1
    grid rowconfig $page.a1.f0 4 -weight 1

    pack [frame $page.a2] -side left -fill both -anchor n -expand yes
    pack [frame $page.a2.fg -bd 2 -relief groove] -side top -fill x -anchor n
    grid [label $page.a2.fg.lb3 -text "Run in:" ] \
	    -row 1 -column 1 -sticky w
    grid [radiobutton $page.a2.fg.b3 -text foreground -anchor w\
	      -variable aindex(runmode) -value 0] \
	-row 1 -column 2 -sticky w
    set aindex(runmode) 0
    grid [radiobutton $page.a2.fg.b4 -text background -anchor w\
	      -variable aindex(runmode) -value 1] \
	-row 1 -column 3 -sticky w
    pack [frame $page.a2.a -bd 2 -relief groove] -side top -fill x -anchor n
    grid [label $page.a2.a.l0 -text "Run in directory:"] -column 0 -row 0 -sticky w
    grid [button $page.a2.a.b0 -text Browse \
	    -command {set command(pwd) [getDirectory $command(pwd)]}\
	    ] -column 1 -row 0 -sticky e
    grid [entry $page.a2.a.e0 -textvariable command(pwd) -width 30] \
	    -column 0 -row 1 -columnspan 2
    #################################################################

    pack [frame $page.a2.b -borderwidth 2 -relief groove] -side top -fill both -anchor n -expand yes
#    pack [frame $page.a2.c -bd 2 -relief groove] -side top -fill x -anchor n
    trace variable aindex(sort) w "FillIndexPeaks $page.a2.b sort"
    trace variable aindex(peaklist) w "FillIndexPeaks $page.a2.b set"

    set row 1
    grid [label $page.a.ll -text "Sort peaks by:"] \
	    -row [incr row] -column 2 -sticky w
    grid [radiobutton $page.a.d1 -text "2theta" -value 1 \
	    -variable aindex(sort)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [radiobutton $page.a.d2 -text "Intensity" -value -2 \
	    -variable  aindex(sort)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [radiobutton $page.a.d3 -text "Peak number" -value 0 \
	    -variable  aindex(sort)] -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [button $page.a.d4 -text "Filter Peaks" \
	      -command IndexDiscardPeaks] -row [incr row] -column 2 -columnspan 2 -sticky w
    # expand the empty row, pushing the above to the top
    grid rowconfigure $page.a [incr row] -weight 2
    if {$::tcl_platform(platform) == "windows"} {
	set suffix .exe
    } else {
	set suffix {}
    }
    grid [label $page.a.l2 -text "Programs:" ] \
	    -row [incr row] -column 1 -columnspan 2 -sticky w
    grid [checkbutton $page.a.ch1 -text "ITO" \
	    -variable  aindex(writeITO)] \
	-row [incr row] -column 2 -sticky w
    set ::command(prog_ito) [file join $::command(exedir) ito$suffix]
    if {! [file exists $::command(prog_ito)]} {
	$page.a.ch1 config -state disabled
    }
    grid [checkbutton $page.a.ch2 -text "TREOR" \
	    -variable  aindex(writeTREOR)] \
	-row [incr row] -column 2 -sticky w
    set ::command(prog_treor) [file join $::command(exedir) ntreor$suffix]
    if {! [file exists $::command(prog_treor)]} {
	$page.a.ch2 config -state disabled
    }
    grid [checkbutton $page.a.ch3 -text "DICVOL" \
	    -variable  aindex(writeDICVOL)] \
	-row [incr row] -column 2 -sticky w
    set ::command(prog_dicvol) [file join $::command(exedir) dicvol06$suffix]
    if {! [file exists $::command(prog_dicvol)]} {
	$page.a.ch3 config -state disabled
    }

#    grid [checkbutton $page.a.cb1 -text "Run program(s)" \
#	    -variable aindex(run)] \
#	    -row [incr row] -column 1 -columnspan 3
    grid [button $page.a.b1 -text "Run program(s)" -command WriteIndexFiles] \
	-row [incr row] -column 1 -columnspan 3
    grid [button $page.a.b2 -text "Review Output" \
	      -command ParseAutoIndexOutput] \
	-row [incr row] -column 1 -columnspan 3 
}

proc PostPageIndex {page}  {
    global graph aindex
    set aindex(setvar) {}
    FillIndexHelpList $page ""
    set peakonlylist {}
    foreach data $graph(datalist) {
	global $data
	if {[set ${data}(type)] == "peaks"} {
	    set units [set ${data}(xunits)]
	    if {$units == "2theta" || $units == "dspace" || \
		    $units == "Q"} {
		lappend peakonlylist $data
	    } elseif {[llength [set ${data}(dspaces)]] > 2} {
		lappend peakonlylist $data
	    }

	}
    }
    catch {destroy $page.a.p.a}
    if {$peakonlylist == ""} {
	pack [label $page.a.p.a -text "no peaklists" -bg red -fg white]
	return
    }
    eval tk_optionMenu $page.a.p.a aindex(peaklist) $peakonlylist
    if {[llength $peakonlylist] == 1} {
	set aindex(peaklist) [lindex $peakonlylist 0]
    } else {
	set aindex(peaklist) {}
    }
    pack $page.a.p.a
    # resize the window
    update idletasks
    wm geom . ""
}

################### MakeScrollFrame ###############################
#                                                                 #
#   create a scrollable frame in a frame or toplevel $boxtop      #
#   $boxtop is the frame or toplevel                              #
#   proc $fillproc is used to create the contents of the frame    #
#     note that the last arg to fillproc is the frame name,       #
#     but the additional args can be passed as in                 #
#            proc myproc {a b c frame} {...}                      #
#            MakeScrollFrame .a "myproc 1 2 3" 250                #
#   height is the vertical size of the scrollable                 #
#                                                                 #
#   returns the name of scrollable frame                          #
#                                                                 #
###################################################################
proc MakeScrollFrame {boxtop fillproc "height 300"} {
    catch {eval destroy [grid slaves $boxtop]}
    # create the canvas and a scrollbar
    grid [canvas $boxtop.c \
	    -scrollregion {0 0 5000 500} -width 50 -height $height \
	    -yscrollcommand "$boxtop.yscroll set" -bg lightgrey] \
	    -sticky  news -row 0 -column 0
    grid [scrollbar $boxtop.yscroll -orient vertical \
	    -command "$boxtop.c yview"] \
	    -sticky ns -row 0 -column 1
    grid columnconfigure $boxtop 0 -weight 1
    grid rowconfigure $boxtop 0 -weight 1
    # create a scrollable frame called scrollframe inside the canvas
    set scrollframe [frame $boxtop.c.f]
    $boxtop.c create window 0 0 -anchor nw  -window $scrollframe
    
    #####################################################
    # fill the frame
    #####################################################
    eval $fillproc $scrollframe

    # resize the scroll window to match the actual
    update idletasks
    $boxtop.c config -scrollregion [grid bbox $scrollframe]
    $boxtop.c config -width [lindex [grid bbox $scrollframe] 2]
    update idletasks
    return $scrollframe
}

proc PeakUseList {data frame} {
    global $data aindex
    set order $aindex(sort)
    set j 0
    grid [label $frame.la0 -text use] -column [incr j] -row 0 -columnspan 2
    incr j
    set xunits [set ${data}(xunits)]
    set xvalues [set ${data}(x)]
    if {$xunits != "2theta" && $xunits != "dspace" && $xunits != "Q"} {
	set xunits dspace
	set xvalues [set ${data}(dspaces)]
    }
    foreach item {x y j} lbl "[list $xunits] { height } { width }" {
	grid [label $frame.l${item}0 -text $lbl] -column [incr j] -row 0
    }
    # create a list of items to be sorted later
    set list {}
    set i 0
    foreach x $xvalues \
	y [set ${data}(heights)] \
	f [set ${data}(widths)] {
	    incr i
	    if {[catch {expr $x}] && abs($order) == 1} {set x 0}
	    if {[catch {expr $y}] && abs($order) == 2} {set y 0}
	    lappend list [list $i $x $y $f]
	}
    set i 0
    set j 0
    set mode -increasing
    if {$order < 0} {
	set order [expr abs($order)]
	set mode -decreasing
    }	
    foreach tuple [lsort $mode -index $order -real $list] {
	set j 0
	incr i
	set num [lindex $tuple 0]
	grid [label $frame.la$i -text $num] -column [incr j] -row $i
	set aindex(index$i) $num
	# set all checkbuttons by default
	if {[catch {set aindex(peak$num)}]} {set aindex(peak$num) 1}
	grid [checkbutton $frame.c$i -variable aindex(peak$num)] -column [incr j] -row $i
	#$frame.c$i select
	foreach item {x y f} fmt {%.3f %.1f %.3f} indx {1 2 3} {
	    set val [lindex $tuple $indx]
	    set $item $val
	    catch {if {$val != ""} {set val [format $fmt $val]}}
	    grid [label $frame.l${item}$i -text $val] -column [incr j] -row $i
	}
	# do some sanity checking on input
	if {$y < 0} {set aindex(peak$num) 0}
	if {$xunits == "2theta"} {
	    if {$x < 0.1 || $x > 180} {set aindex(peak$num) 0}
	} elseif {$xunits == "Q"} {
	    if {$x < 0.05 || $x > 15} {set aindex(peak$num) 0}
	} else {
	    if {$x < 0.5 || $x > 100} {set aindex(peak$num) 0}
	}
    }
}

proc FillIndexPeaks {frame mode args} {
    global aindex
    eval destroy [grid slaves $frame]
    if {$aindex(peaklist) == ""} return
    set data $aindex(peaklist)
    if {$mode == "set"} {array unset aindex {peak[0-9]*} }
    MakeScrollFrame $frame "PeakUseList $aindex(peaklist)" 240
    global $data
    if {[set ${data}(wavelength)] != ""} {
	set aindex(wavelength) [set ${data}(wavelength)]
    }
    ResizeNotebook   
}

proc FillIndexHelpList {page var} {
    global aindex
    $page.a1.f0.lb delete 0 end
    $page.a1.f0.msg delete 0.0 end
    set aindex(helplbl) {}
    $page.a1.f0.ent config -textvariable {}
    $page.a1.f0.ent delete 0 end    
    $page.a1.f0.ent config -state disabled
    if {$var == ""} return
    global $var ${var}_Compound ${var}_INFO
    set varlist [array names ${var}_INFO]
    set list [array names ${var}_Compound]
    if {[llength $list] > 0} {lappend varlist [array names ${var}_Compound]}
    eval $page.a1.f0.lb insert 0 [lsort $varlist]
}

proc FillIndexHelpMsg {list edit ent} {
    global aindex
    set var $aindex(setvar) 
    global $var ${var}_INFO ${var}_Compound
    $edit delete 0.0 end
    set i [$list curselection]
    if {$i == ""} return
    set elm [$list get $i]
    set aindex(helplbl) $elm
    set help "no information for this variable"
    catch {
	set string [set ${var}_Compound($elm)]
	set morehelp "\nValues set:"
	foreach val $string {
	    if {[lindex $val 0] == ""} {
		set help [lindex $val 1]
	    } else {
		append morehelp " [lindex $val 0]=[lindex $val 1];"
	    }
	}
	append help $morehelp
    }
    catch {
	set help [set ${var}_INFO($elm)]
    }
    set help [regsub -all "\n" $help " "]
    set help [regsub -all "  " $help " "]
    set help [regsub -all "  " $help " "]
    $edit insert 0.0 $help
    # is this a special variable?
    if {[string first . $elm] == 0} {
	grid forget ${ent}
	grid ${ent}button -column 2 -row 3 -columnspan 2
	${ent}button config -command "IndexSetMultipleValues $var $elm"
    } else {
	grid forget ${ent}button
	grid ${ent} -column 2 -row 3 -columnspan 2
	$ent config -state normal -textvariable ${var}($elm)
    }
}

proc IndexSetMultipleValues {var elm} {
    global $var ${var}_INFO ${var}_Compound
    foreach val [set ${var}_Compound($elm)] {
	if {[lindex $val 0] != ""} {
	    set ${var}([lindex $val 0]) [lindex $val 1]
	}
    }
}

proc WriteIndexFiles {} {
    global aindex command
    set msg {}
    if {[catch {expr $aindex(wavelength)}]} {
	set msg "wavelength (=\"$aindex(wavelength)\") is invalid\n"
    }
    if {[catch {expr $aindex(zerocorrection)}]} {
	set msg "Zero offset ($aindex(zerocorrection)) is invalid\n"
    }
    if {$msg != ""} {
	MyMessageBox -parent . -title "Invalid input" -message $msg \
	    -icon error -type {"Try again"} -default "try again" 
	return
    }
    # compile a list of two-theta values that have been selected
    set data $aindex(peaklist)
    if {$data == ""} {
	MyMessageBox -parent . -title "Invalid peak list" \
	    -message "Select a peak list first" \
	    -icon error -type {"Try again"} -default "try again" 
	return
    }
    global $data
    set xunits [set ${data}(xunits)]
    set xvalues [set ${data}(x)]
    if {$xunits != "2theta" && $xunits != "dspace" && $xunits != "Q"} {
	set xunits dspace
	set xvalues [set ${data}(dspaces)]
    }
    set ttlist {}
    set i 0
    set init 1
    foreach value $xvalues y [set ${data}(heights)] bkg [set ${data}(bkgs)] {
	incr i
	if {$aindex(peak$i)} {
	    # convert Q values to d-space
	    if {$xunits == "Q"} {
		if {[catch {
		    set value [expr 2 * 3.14159 / $value]		
		}]} continue
	    }
	    # do some sanity checking on input
	    if {$xunits == "2theta"} {
		if {$value < 0.1 || $value > 180} continue
	    } else {
		if {$value < 0.5 || $value > 100} continue
	    }
	    # get data limits for plotting
	    set y1 0
	    set y2 0
	    catch {
		set y1 [expr $bkg]
		set y2 [expr $bkg+$y]
	    }
	    if {$init} {
		set aindex(xmin) $value
		set aindex(xmax) $value
		set aindex(ymin) $y1
		set aindex(ymax) $y2
		set init 0
	    } else {
		catch {if {$aindex(xmin) > $value} {set aindex(xmin) $value}}
		catch {if {$aindex(xmax) < $value} {set aindex(xmax) $value}}
		catch {if {$aindex(ymin) > $y1} {set aindex(ymin) $y1}}
		catch {if {$aindex(ymax) < $y2} {set aindex(ymax) $y2}}
	    }
	    lappend ttlist $value
	}
    }
    if {$xunits == "2theta"} {
	set units tt
    } elseif {$xunits == "dspace"} {
	set units dspace
    } elseif {$xunits == "Q"} {
	set units dspace
    }
    set runlist {}
    set err 0
    foreach prog {ito treor dicvol} {
	set PROG [string toupper $prog]
	if {$aindex(write${PROG})} {
	    if {! [Validate${PROG} $ttlist]} {
		set err 1
	    }
	    if {$runlist != ""} {append runlist " & "}
	    append runlist $prog
	}
    }
    if $err return
    if {$runlist ==""} {
	MyMessageBox -parent . -title "Select error" \
	    -message "No indexing programs selected to run" \
	    -icon error -type "OK" -default "ok" 
	return
    }
    # now launch the programs
    set msg {}
    set outmsg {}
    set outfmtlist {}
    set outlist {}
    # save current directory and set the directory to the work area
    set initcd [pwd]
    cd $command(pwd)
    foreach prog {ito treor dicvol} {
	set PROG [string toupper $prog]
	if {$aindex(write${PROG})} {
	    if {[catch {
		append msg "   [Write${PROG}File $ttlist]\n"
		if {$aindex(write${PROG})} {
		    set out [ExecIndex $prog]
		    if {$outmsg != ""} {append outmsg "\n"}
		    append outmsg $prog ": " [lindex $out 0]
		    lappend outfmtlist $out
		    lappend outlist [lindex $out 0]
		}
	    } errmsg]} {
		MyMessageBox -parent . -title "Write error" \
		    -message "Error in Index write/run -- $errmsg" \
		    -icon error -type "OK" -default "ok" 
	    }
	} else {
	    lappend outlist {}
	}
    }
    if {$aindex(runmode)} {
	# launch a separate script that runs in the background 
	# captures errors and provides a "done" box
	# dump the search info to disk for 
	set fileroot [string range $data 0 14]
	set filename [file join $command(pwd) ${fileroot}.indexsum]
	set fp [open $filename w]
	puts $fp "set outfmtlist [list $outfmtlist]"
	puts $fp "set data \[initpeaks \$data\]"
 	puts $fp "global \${data}"
	foreach item {title wavelength  xunits xlabel ylabel \
		spg           extcodes x y dspaces heights etas bkgs} {
	    puts $fp "set \${data}(${item}) \t[list [set ${data}(${item})]]"
	}
	puts $fp "resetdata \$data"
 	puts $fp "global aindex"
	foreach item {xmin xmax ymin ymax wavelength} {  
	    puts $fp "set aindex(${item}) \t[list $aindex($item)]"
	}
	close $fp
	set impfile [file join $command(pwd) ${fileroot}.indexinp]
	set fp [open $impfile w]
	puts $fp $filename
	foreach item $outfmtlist {
	    puts $fp $item
	}
	close $fp
	exec [info nameofexecutable] [file join $::scriptdir runindex.tcl] < $impfile &
	MyMessageBox -parent . -title "Indexing Launched" \
	    -message "The indexing program(s) ($runlist) have been launched in a background process running in directory $command(pwd). A separate notification box will appear when this completes. Then use the \"Review Output\" button to load file [file tail $filename] to see the results." \
	    -type "OK" -default "ok" 
    } else {
	set i 0
	set msg "Indexing program(s) completed. The following output\nfiles(s) were created:\n\n"
	foreach out $outfmtlist {
	    append msg "Program [lindex $out 1]: [lindex $out 0]\n"
	}
	append msg "\nPress Continue to view these output files"
	MyMessageBox -parent . -title "Indexing Completed" \
	    -message $msg -type "Continue" -default "continue" 
	IndexOutputParse [lindex $outlist 0] [lindex $outlist 2] [lindex $outlist 1]
    }
    # restore directory
    cd $initcd
}

proc WriteITOFile {ttlist "units tt"} {
    global aindex ITO command
    set data $aindex(peaklist)
    global $data
    set fileroot [string range $data 0 14]
    set filename [file join $command(pwd) ${fileroot}.ito]
    set fil [open $filename w]
 
    ############################## WRITE ITO ##############################
    #1.   A title card, containing up to 80 characters.
    set ttl "ITO: $data [set ${data}(title)]"
    #card 1
    puts $fil [format "%s" $ttl]

    #2.   A parameter card, which may be completely blank.
    set line {}
    foreach var "ITO(MAN) ITO(INSTR) ITO(INTENS) ITO(NSOLMX) \
	    aindex(orthorhombic) aindex(monoclinic) aindex(triclinic) \
	    ITO(TOL2) ITO(TOL3) aindex(wavelength) ITO(LINCO) ITO(LZERCK) \
	    ITO(K(ENL)) ITO(NQ1) ITO(NQ2) ITO(NZ1) ITO(NZ2) \
	    ITO(NR) ITO(INDAT) ITO(LIST) ITO(JTEST) ITO(WMOL) \
	    ITO(DOBS) ITO(TOLG) ITO(NTST)" \
	    fmt "%1d %1d %1d %1d %2d %2d %2d %5.2f %5.2f %10.5f %2d %1d %1d \
	    %2d %2d %2d %2d %2d %2d %2d %2d %10.5f %10.5f %8.4f %2d" {
	append line [format $fmt [set $var]]
    }

    puts $fil $line
    #3.   Another parameter card. Can also be blank.
    puts $fil [format "%10.5f%10.5f%10.5f" \
	    $aindex(zerocorrection) $ITO(PRNTMR) $ITO(PRNTLN)]

    #card 4+ 2theta values (must sorted by increasing 2theta)
    if {$units == "tt"} {
	foreach pos [lsort -real $ttlist] {
	    puts $fil [format "%.4f" $pos]
	}
    } else {
	foreach pos [lsort -real -decreasing $ttlist] {
	    puts $fil [format "%.4f" $pos]
	}
    }
    puts $fil "0.0"
    puts $fil "END"
    close $fil
    return $filename
}
 
proc ValidateITO {ttlist} {
    global aindex ITO
    set data $aindex(peaklist)
    global $data
    set errmsg {}
    foreach var "MAN INSTR INTENS NSOLMX \
	    TOL2 TOL3 LINCO LZERCK \
	    K(ENL) NQ1 NQ2 NZ1 NZ2 \
	    NR INDAT LIST JTEST WMOL \
	    DOBS TOLG NTST PRNTMR PRNTLN" \
	    fmt "%1d %1d %1d %1d %5.2f %5.2f %2d %1d %1d \
	    %2d %2d %2d %2d %2d %2d %2d %2d %10.5f %10.5f \
	    %8.4f %2d %10.5f %10.5f" {
	if [catch {format $fmt $ITO($var)}] {
	    append errmsg " value $ITO($var) is invalid for $var\n"
	}
    }
    set msg "Invalid ITO input:\n$errmsg"
    if {$errmsg != ""} {
	MyMessageBox -parent . -title "Invalid ITO input" \
	    -message $msg \
	    -icon error -type "OK" -default "ok" 
	return 0
    }
    return 1
}

proc WriteTREORFile {ttlist "units tt"} {
    global aindex TREOR command
    set data $aindex(peaklist)
    global $data
    set fileroot [string range $data 0 14]

    if {$units == "tt"} {
	set TREOR(CHOICE) 3
    } else {
	set TREOR(CHOICE) 4
    }
    set filename [file join $command(pwd) ${fileroot}.treor]
    set fil [open $filename w]
 
    ############################## WRITE TREOR ##############################
    #1.   A title card, containing up to 80 characters.
    set ttl "  TREOR: $data [set ${data}(title)]"
    #card 1
    puts $fil [format "%s" $ttl]

    # LINE SET TWO.      FORMAT(F16.6,3X,A4) # output twotheta
    foreach pos [lrange [lsort -real $ttlist] 0 99] {
             puts $fil [format "%16.6f " $pos] 
    }
    # a blank line
    puts $fil {}

    # LINE SET THREE.    GENERAL INSTRUCTIONS.
    puts $fil " WAVE=[expr 1.00000 * $aindex(wavelength)],"
    if { $aindex(triclinic) == 1 } {
	puts $fil " TRIC=1,"
    } elseif { $aindex(monoclinic) == 1 && $TREOR(MONO) == ""} {
	puts $fil " MONO=130.,"
    }

    foreach item [array names TREOR] {
	if {[string trim $TREOR($item)] != ""} {
	    puts $fil "$item=$TREOR($item),"
	}
    }
    puts $fil "END*"
    puts $fil "  0.00"
    close $fil
    return $filename
}

proc ValidateTREOR {ttlist} {
    global aindex TREOR
    set data $aindex(peaklist)
    global $data

    set errmsg {}
    foreach var [array names TREOR] {
	if {[string trim $TREOR($var)] != ""} {
	    if [catch {expr $TREOR($var)}] {
		append errmsg " value $TREOR($var) is invalid for $var\n"
	    }
	}
    }
    set msg "Invalid TREOR input:\n$errmsg"
    if {$errmsg != ""} {
	MyMessageBox -parent . -title "Invalid TREOR input" \
	    -message $msg \
	    -icon error -type "OK" -default "ok" 
	return 0
    }
    if {$aindex(triclinic) && ( \
	    $aindex(cubic) || $aindex(tetragonal) || $aindex(hexagonal) \
	    || $aindex(orthorhombic) || $aindex(monoclinic) )} {
	MyMessageBox -parent . -title "Incompatible TREOR input" \
	    -message 	"Note: TREOR will not search for a higher symmetry unit cell, since triclinic is selected" \
	    -icon warning -type "OK" -default "ok" 
    }
    if {[llength $ttlist] > 100} {
	MyMessageBox -parent . -title "Incompatible TREOR input" \
	    -message "Note: TREOR can only search with the first 100 lines." \
	    -icon warning -type "OK" -default "ok" 
    }
    return 1
}

proc WriteDICVOLFile {ttlist "units tt"} {
    global aindex DICVOL command
    set data $aindex(peaklist)
    global $data
    set fileroot [string range $data 0 14]

    set filename [file join $command(pwd) ${fileroot}.dic]
    set fil [open $filename w]
    set ZERO  [expr 1.00000 * $aindex(zerocorrection)]
    set WAVE [expr 1.00000 * $aindex(wavelength)]
    # N  - NUMBER OF LINES USED.
    set N [llength $ttlist]
    # =2  2-THETA ANGLE IN DEGREES.
    # =3  D-SPACING IN ANGSTROMS.
    if {$units == "tt"} {
	set ITYPE 2
    } else {
	set ITYPE 3
    }
    # JC =0  CUBIC SYSTEM IS NOT TESTED.
    set JC $aindex(cubic)
    # JT=0  TETRAGONAL SYSTEM IS NOT TESTED.
    set JT $aindex(tetragonal)
    # JH=0  HEXAGONAL SYSTEM IS NOT TESTED.
    set JH $aindex(hexagonal)
    # JO=0  ORTHORHOMBIC SYSTEM IS NOT TESTED.
    set JO $aindex(orthorhombic)
    # JM=0  MONOCLINIC SYSTEM IS NOT TESTED.
    set JM $aindex(monoclinic)
    # JTR=0  TRICLINIC SYSTEM IS NOT TESTED.
    set JTR $aindex(triclinic)
    #1.   A title card, containing up to 80 characters.
    set ttl "DICVOL: $data [set ${data}(title)]"
    #card 1
    puts $fil [format "%s" $ttl]
    #CARD 2  N,ITYPE,JC,JT,JH,JO,JM,JTR                 FREE FORMAT
    puts $fil "$DICVOL(N_USE) $ITYPE $JC $JT $JH $JO $JM $JTR"
    #CARD 3  AMAX,BMAX,CMAX,VOLMIN,VOLMAX,BEMIN,BEMAX   FREE FORMAT
    puts $fil "$DICVOL(AMAX) $DICVOL(BMAX) $DICVOL(CMAX) \
	    $DICVOL(VOLMIN) $DICVOL(VOLMAX) $DICVOL(BEMIN) $DICVOL(BEMAX)"
    #CARD 4  WAVE,POIMOL,DENS,DELDEN                    FREE FORMAT
    puts $fil "$WAVE $DICVOL(POIMOL) $DICVOL(DENS) $DICVOL(DELDEN)"
    #CARD 5  EPS,FOM,ZERO                               FREE FORMAT
    #CARD 5  EPS,FOM,N_IMP,ZERO,ZERO_REF,OPTION           FREE FORMAT
    puts $fil "$DICVOL(EPS) $DICVOL(FOM) $DICVOL(N_IMP) $DICVOL(ZERO) $DICVOL(ZERO_REF) $DICVOL(OPTION)"
    #CARD 6 TO 6+N  D(I),EPSIL(I)                       FREE FORMAT
    foreach pos [lsort -real $ttlist] {
	puts $fil [format "%10.5f" $pos]
    }
    close $fil
    return $filename
}

proc ValidateDICVOL {ttlist} {
    global aindex DICVOL
    set data $aindex(peaklist)
    global $data

    set errmsg {}
    foreach var [array names DICVOL] {
	if {[string trim $DICVOL($var)] != ""} {
	    if [catch {expr $DICVOL($var)}] {
		append errmsg " value $DICVOL($var) is invalid for $var\n"
	    }
	}
    }
    set msg "Invalid DICVOL input:\n$errmsg"
    if {$errmsg != ""} {
	MyMessageBox -parent . -title "Invalid DICVOL input" \
	    -message $msg \
	    -icon error -type "OK" -default "ok" 
	return 0
    }
    return 1
}

proc ExecIndex {prog} {
    global aindex command
    set data $aindex(peaklist)
    set fileroot [string range $data 0 14]
    set fp [open ${prog}.inp w]
    if {$::tcl_platform(platform) == "windows"} {
	set image $prog.exe
    } else {
	set image $prog
    }
    set fmt $prog
    if {$prog == "ito"} {
	set out [file join $command(pwd) ${fileroot}.imp]
	set image $::command(prog_ito)
	puts $fp go
	puts $fp ${fileroot}.ito
	puts $fp ${fileroot}.imp
    } elseif {$prog == "treor"} {
	set out [file join $command(pwd) ${fileroot}.trrs]
	set fmt "treor short"
	set image $::command(prog_treor)
	puts $fp "N"
	puts $fp ${fileroot}.treor  ; # input
	puts $fp ${fileroot}.trout  ; # output
	puts $fp ${fileroot}.trr    ; # condensed output
	puts $fp ${fileroot}.trrs   ; # short output
	puts $fp "0.0"
	if { $aindex(triclinic) == 1 } {
	    puts $fp "N"
	} else {
	    puts $fp "Y"
	}
	puts $fp "Y"; # don't continue if a solution is found
    } elseif {$prog == "dicvol"} {
	set out [file join $command(pwd) ${fileroot}.vol]
	set image $::command(prog_dicvol)
	puts $fp ${fileroot}.dic
	puts $fp ${fileroot}.vol
    }
    close $fp
    if {$aindex(runmode)} {
	# return the info that will be used in the background script
	return [list $image $prog $out]
    } else {
	pleasewait "Now running [string toupper $prog]..."
	exec $image < ${prog}.inp > ${prog}.out
	donewait
	catch {file delete ${prog}.inp}
	return [list $out $fmt]
    }
}

proc ParseAutoIndexOutput {} {
    global command
    set typelist {
	{"Background run output" .indexsum}
	{"ITO output" .imp}
	{"DICVOL full output" .vol}
	{"DICVOL short output" .ord}
	{"TREOR short output (preferred)" .trrs}
	{"TREOR condensed output" .trr}
	{"TREOR full output (no scan)" .trout}
    }
    set filename [tk_getOpenFile -filetypes $typelist \
	    -initialdir $command(pwd) -title "Review Autoindex Output"]
    if {[string trim $filename] == ""} return
    if {[file extension $filename] == ".imp"} {
	IndexOutputParse $filename "" ""
    } elseif {[file extension $filename] == ".vol"} {
	IndexOutputParse "" $filename ""
    } elseif {[file extension $filename] == ".ord"} {
	IndexOutputParse "" $filename ""
    } elseif {[file extension $filename] == ".trout"} {
	IndexOutputParse "" "" "" "" $filename 
    } elseif {[file extension $filename] == ".trrs"} {
	IndexOutputParse "" "" $filename
    } elseif {[file extension $filename] == ".trr"} {
	IndexOutputParse "" "" "" $filename
    } elseif {[file extension $filename] == ".indexsum"} {
	set data [file root [file tail $filename]]
	if {[catch {source $filename} err]} {
	    MyMessageBox -parent . -title "Error reading" \
		-message "Error reading the summary file:\n$err" \
		-icon error -type "OK" -default "ok" 
	    return
	} 
	foreach prog {ito dicvol treor} {
	    set filenam($prog) {}
	}
	foreach item $outfmtlist {
	    set filenam([lindex $item 1]) [lindex $item 2]
	}
	set ::aindex(peaklist) $data
	IndexOutputParse $filenam(ito) $filenam(dicvol) $filenam(treor)
    } else {
	IndexOutputParse "" "" "" "" $filename 
    }
}


proc ParseIto {file txt} {
    set fp [open $file r]
    set list1 {}
    set list2 {}
    set list3 {}
    set lnum -1
    while {[gets $fp line] >= 0} {
	incr lnum
	$txt.t insert end "$line\n"
	# beginning of initial solutions
	if {[string first {Q(A)} $line] > 0} {
	    set i 0
	    # 1st block has FOM
	    while {[gets $fp line] >= 0 && [string trim $line] != ""} {
		incr lnum
		incr i
		$txt.t insert end "$line\n"
		lappend list1 $line
		lappend list3 $lnum
	    }
	    while {[gets $fp line] >= 0 && [string first {A        B} $line] < 0} {
		incr lnum
		$txt.t insert end "$line\n"
	    }	
	    # 2nd block has lattice constants
	    set j 0
	    while {[gets $fp line] >= 0 && [string trim $line] != "0" \
		       && [string trim $line] != ""} {
		incr lnum
		incr j
		$txt.t insert end "$line\n"
		if {$j > $i} break
		lappend list2 $line
	    }
	    break
	}
    }
    # 2nd section; apply zeroshift to best set from before
    set zero {}
    set fomline {}
    while {[gets $fp line] >= 0} {
	incr lnum
	$txt.t insert end "$line\n"
	if {[string first {SOLUTION NR} $line] > 0} {
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    set fomline $line
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    if {[string first {ZEROSHIFT} $line] > 0} {
		set zero [lindex $line 1]
	    }
	}
	if {[string first {A        B} $line] > 0} {
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    # hitting an error here where fomline has not been set
	    if {$fomline != ""} {
		lappend list1 [concat $fomline "P" $zero] 
		lappend list3 $lnum
		lappend list2 $line
	    }
	}
    }
    set list4 {}
    foreach l1 $list1 l2 $list2 lnum $list3 {
	lappend list4 "[list [lindex $l1 8]] $l2 [lindex $l1 7] [list [lindex $l1 9]] $lnum"
    }
    close $fp
    return $list4
}

proc ParseDicvol {file txt} {
    set fp [open $file r]
    set list1 {}
    set lnum -1
    set mode ?
    set replace -1
    while {[gets $fp line] >= 0} {
	incr lnum
	$txt.t insert end "$line\n"
	if {[string first {TO INDEX ALL INPUT} $line] > 0} {
	    set replace $lnum
	}
	if {[string first {C U B I C} $line] > 0} {
	    set mode cubic
	    set loop {A= A= A= 90 90 90 VOLUME=} 
	} elseif {[string first {T E T R A G O N A L} $line] > 0} {
	    set mode tet
	    set loop {A= A= C= 90 90 90 VOLUME=} 
	} elseif {[string first {H E X A G O N A L} $line] > 0} {
	    set loop {A= A= C= 90 120 90 VOLUME=} 
	    set mode hex
	} elseif {[string first {O R T H O R H O M B I C} $line] > 0} {
	    set loop {A= B= C= 90 90 90 VOLUME=} 
	    set mode ortho
	} elseif {[string first {M O N O C L I N I C} $line] > 0} {
	    set loop {A= B= C= 90 BETA= 90 VOLUME=} 
	    set mode mono
	} elseif {[string first {T R I C L I N I C} $line] > 0} {
	    set loop {A= B= C= ALP= BET= GAM= VOLUME=} 
	    set mode tri
	}
	if {[string first {DIRECT PARAMETERS} $line] > 0} {
	    set list {P}		
	    foreach str $loop {
		if {$str == 90 || $str == 120} {
		    lappend list $str
		    continue
		} 
		lappend list [lindex [string range $line [expr [string first $str $line]+[string length $str]] end] 0]
	    }
	    gets $fp line
	    $txt.t insert end "$line\n"
	    gets $fp line
	    $txt.t insert end "$line\n"
	    if {[string first "ZERO-POINT" $line] >= 0} {
		set zero [lindex [string range $line [expr [string first : $line]+1] end] 0]
	    } else { 
		set zero 0
	    }
	    lappend list $zero $lnum
	    incr lnum 2
	    if {$replace >0 && abs($replace - $lnum) < 5} {
		set list1 [lreplace $list1 end end $list]
	    } else {
		lappend list1 $list
	    }
	    set replace -1
	}
    }
    close $fp
    return $list1
}

proc ParseTreorLong {file txt} {
    set fp [open $file r]
    set list1 {}
    set lnum -1
    while {[gets $fp line] >= 0} {
	incr lnum
	$txt.t insert end "$line\n"
	foreach {a b c alpha beta gamma vol unindexed} {? ? ? ? ? ? ? ?} {}
	if {[string first {A =} $line] > 0} {
	    regexp {A += *([0-9]+\.[0-9]+).* = *([0-9]+\.[0-9]+)} \
		$line x a alpha
	    set slnum $lnum
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    regexp {B += *([0-9]+\.[0-9]+).* = *([0-9]+\.[0-9]+)} \
		$line x b beta
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    regexp {C += *([0-9]+\.[0-9]+).* = *([0-9]+\.[0-9]+)} \
		$line x c gamma
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    regexp { = *([0-9]+\.[0-9]+)} $line x vol
	    
	    while {[gets $fp line] >= 0} {
		incr lnum
		$txt.t insert end "$line\n"
		if {[string first {lines are unindexed} $line] > 0} {
		    set unindexed [lindex $line 0]
		    break
		}
	    }
	    lappend list1 [list $a $b $c $alpha $beta $gamma $vol $unindexed $slnum]
	}
    }
    close $fp
    return $list1
}
proc ParseTreorShort {file txt} {
    set fp [open $file r]
    set list1 {}
    set lnum -1
    while {[gets $fp line] >= 0} {
	incr lnum
	$txt.t insert end "$line\n"
	foreach {a b c alpha beta gamma vol unindexed} {? ? ? ? ? ? ? ?} {}
	if {[string first {Refined cell} $line] > 0} {
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    regexp {A *= *([0-9]+\.[0-9]+).*B *= *([0-9]+\.[0-9]+).*C *= *([0-9]+\.[0-9]+)} \
		$line x a b c
	    gets $fp line
	    incr lnum
	    set slnum $lnum
	    $txt.t insert end "$line\n"
	    regexp {Alpha *= *([0-9]+\.[0-9]+).*Beta *= *([0-9]+\.[0-9]+).*Gamma *= *([0-9]+\.[0-9]+)} \
		$line x alpha beta gamma
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    gets $fp line
	    incr lnum
	    $txt.t insert end "$line\n"
	    regexp {= *([0-9]+\.[0-9]+)} $line x vol
	    set i 0
	    while {[gets $fp line] >= 0 && $i < 10} {
		incr lnum
		$txt.t insert end "$line\n"
		incr i
		if {[string first {unindexed} $line] > 0} {
		    regexp { = *([0-9]+)} $line x unindexed
		    break
		}
	    }
	    lappend list1 [list $a $b $c $alpha $beta $gamma $vol $unindexed $slnum]
	}
    }
    close $fp
    return $list1
}

proc MakeAIndexTextbox {txt} {
    global aindex tcl_version
    eval destroy [winfo children $txt]
    grid [text $txt.t -width 100 -wrap none \
	      -yscrollcommand "$txt.ys set" \
	      -xscrollcommand "$txt.xs set"] -row 1 -column 1
    if {$tcl_version >= 8.0} {$txt.t config -font $aindex(font)}
    grid [scrollbar $txt.ys -command "$txt.t yview"] \
	-row 1 -column 2 -sticky ns
    grid [scrollbar $txt.xs -command "$txt.t xview" -orient horizontal] \
	-row 2 -column 1 -sticky ew
    grid columnconfigure $txt 1 -weight 1
    grid rowconfigure $txt 1 -weight 1
    grid [frame $txt.f] -column 1 -columnspan 2 -row 3 -sticky ew
    if {$tcl_version >= 8.0} {
	set fontbut [tk_optionMenu $txt.f.0 aindex(font) ""]
	pack $txt.f.0 -side left
	$fontbut delete 0 end
	foreach f {5 6 7 8 9 10 11 12 13 14 15 16} {
	    $fontbut add command -label "Courier $f" \
		-command "set aindex(font) \"Courier $f\"; \
		    $txt.t config -font \$aindex(font)"
	}
    }
    pack [button $txt.f.1 -text Close -command "destroy $txt"] -side right
    bind $txt <KeyPress-Prior> "$txt.t yview scroll -1 page"
    bind $txt <KeyPress-Next> "$txt.t yview scroll 1 page"
    bind $txt <KeyPress-Up> "$txt.t yview scroll -1 unit"
    bind $txt <KeyPress-Down> "$txt.t yview scroll 1 unit"
    bind $txt <KeyPress-Home> "$txt.t yview 0"
    bind $txt <KeyPress-End> "$txt.t yview end"
}

# creates a table that is scrollable in both x and y, use ResizeScrollTable
# to set sizes after gridding the widgets
proc MakeScrollTable {box} {
    proc sync2boxes {cmd master slave scroll args} {
	$slave $cmd moveto [lindex [$master $cmd] 0]
	eval $scroll set $args
    }
    proc move2boxes {cmd box1 box2 args} {
	eval $box1 $cmd $args
	eval $box2 $cmd $args
    }
    grid [label $box.0] -column 0 -row 1
    grid [canvas $box.top -scrollregion {0 0 10 10} \
	    -xscrollcommand "sync2boxes xview $box.top $box.can $box.scroll" \
	    -width 10 -height 10] -sticky sew -row 1 -column 1
    grid [canvas $box.side -scrollregion {0 0 10 10} \
	    -yscrollcommand "sync2boxes yview $box.side $box.can $box.yscroll" \
	    -width 10 -height 10] -sticky nes -row 2 -column 0
    grid [canvas $box.can -scrollregion {0 0 10 10} \
	    -yscrollcommand "sync2boxes yview $box.can $box.side $box.yscroll" \
	    -xscrollcommand "sync2boxes xview $box.can $box.top $box.scroll" \
	    -width 200 -height 200 -bg lightgrey] -sticky news -row 2 -column 1
    grid [set sxbox [scrollbar $box.scroll -orient horizontal \
			 -command "move2boxes xview $box.can $box.top"]] \
	    -sticky ew -row 3 -column 1
    grid [set sybox [scrollbar $box.yscroll \
			 -command "move2boxes yview $box.can $box.side"]] \
	    -sticky ns -row 2 -column 2

    $box.top create window 0 0 -anchor nw  -window [frame $box.top.f -bd 0]
    $box.can create window 0 0 -anchor nw  -window [frame $box.can.f -bd 2]
    $box.side create window 0 0 -anchor nw  -window [frame $box.side.f -bd 2]

    grid columnconfig $box 1 -weight 1
    grid rowconfig $box 2 -weight 1
    return [list  $box.top.f  $box.can.f $box.side.f $box.0]
}

proc ResizeScrollTable {box} {
    update idletasks
    for {set i 0} {$i < [lindex [grid size $box.can.f] 0]} {incr i} {
	set x1 [lindex [grid bbox $box.can.f $i 0] 2]
	set x2 [lindex [grid bbox $box.top.f $i 0] 2]
	if {$x2 > $x1} {set x1 $x2}
	grid columnconfigure $box.top.f $i -minsize $x1
	grid columnconfigure $box.can.f $i -minsize $x1
    }
    for {set i 0} {$i < [lindex [grid size $box.can.f] 1]} {incr i} {
	set x1 [lindex [grid bbox $box.can.f 0 $i] 3]
	set x2 [lindex [grid bbox $box.side.f 0 $i] 3]
	if {$x2 > $x1} {set x1 $x2}
	grid rowconfigure $box.can.f $i -minsize $x1
	grid rowconfigure $box.side.f $i -minsize $x1
    }
    update idletasks
    set sizes [grid bbox $box.can.f]
    $box.can config -scrollregion $sizes
    $box.side config -scrollregion $sizes
    $box.top config -scrollregion $sizes
    $box.side config -width [lindex [grid bbox $box.side.f] 2]
    $box.top config -height [lindex [grid bbox $box.top.f] 3]
    # remove the scroll when not needed
    if {[lindex $sizes 3] > [winfo height $box.can]} {
	grid $box.yscroll -sticky ns -column 2 -row 2
    } else {
	grid forget $box.yscroll 
    }
    if {[lindex $sizes 2] > [winfo width $box.can]} {
	grid $box.scroll -sticky ew -column 1 -row 3
    } else {
	grid forget $box.scroll 
    }
}

proc IndexOutputParse {itofile dicvolfile treorshortfile "treorfile {}" "otherfile {}"} {
    global aindex tcl_version
    set cellbox .cells
    catch {toplevel $cellbox}
    eval destroy [winfo children $cellbox]
    bind $cellbox <Configure> ""
    wm title $cellbox "Indexed Cells"

    set cellrow 0
    set textboxlist {}
    pleasewait
    foreach prog {ito dicvol "treor short" treor ""} file [list $itofile $dicvolfile $treorshortfile $treorfile $otherfile] {
	incr cellrow
	set txt .file$cellrow
	destroy $txt
	if {$file == ""} continue
	catch {toplevel $txt}
	eval destroy [winfo children $txt]
	MakeAIndexTextbox $txt
	lappend textboxlist $txt
	raise $::PleaseWaitWindow

	grid [frame [set box $cellbox.s$cellrow]] -row $cellrow -column 0 -columnspan 3 -sticky news
	grid rowconfig $cellbox $cellrow -weight 1
	grid columnconfig $cellbox 0 -weight 1
	foreach {top main side lbl} [MakeScrollTable $box] {}
	[winfo parent $main] config -height 100 -width 500
	grid [label $box.l0 -anchor w -bg white -text \
		  "[string toupper [lindex $prog 0]] output $file" \
		 ] -row 0 -column 0 -columnspan 99 -sticky we
	set row 1
	set j 1
	foreach i {a b c} {
	    incr j
	    grid [label $top.$j -text $i] -row $row -column $j
	}
	foreach i {a b g} {
	    incr j
	    grid [label $top.$j -text $i -font symbol] -row $row -column $j
	}
	incr j
	grid [label $top.$j -text Vol] -row $row -column $j
	
	
	wm title $txt "[string toupper [lindex $prog 0]] output: $file"
	wm iconname $txt $file
	if {$prog == "ito"} {
	    set list1 [ParseIto $file $txt]
	    incr j
	    grid [label $top.$j -text "FOM"] -row $row -column $j
	    incr j
	    grid [label $top.$j -text "D2q" -font symbol] -row $row -column $j
	    if {[llength $list1] == 0} {
		eval destroy [winfo children $top]
		grid [label $main.0 -text "(none found)"] \
			  -row 0 -column 0 -columnspan 5
	    }
	    foreach line $list1 {
		incr row
		set k 0
		set zero [lindex $line 9]
		set lnum [lindex $line 10]
		grid [button $side.$row -text View \
			  -command "LoadAutoIndexCell [lrange $line 1 6] [lindex $line 0] [list $zero] [list $lnum] $txt"]\
		    -row $row -column 0
		foreach i {0 1 2 3 4 5 6 7 8 9} {
		    incr k
		    incr j
		    grid [label $main.$j -text " [lindex $line $i]"]\
			-row $row -column $k
		}
	    }
	} elseif {$prog == "dicvol"} {
	    incr j
	    grid [label $top.$j -text "D2q" -font symbol] -row $row -column $j
	    set list1 [ParseDicvol $file $txt]
	    if {[llength $list1] == 0} {
		eval destroy [winfo children $top]
		grid [label $main.0 -text "(none found)"] \
			  -row 0 -column 0 -columnspan 5
	    }
	    foreach line $list1 {
		incr row
		set k 0
		set prm [lrange $line 1 6]
		set zero [lindex $line 8]
		set cell [lindex $line 0]
		set lnum [lindex $line 9]
		grid [button $side.$row -text View \
			  -command "LoadAutoIndexCell $prm $cell $zero $lnum $txt"] \
		    -row $row -column 0
		foreach i {0 1 2 3 4 5 6 7 8} {
		    incr k
		    incr j
		    grid [label $main.$j -text " [lindex $line $i]"]\
			-row $row -column $k
		}
	    }
	} elseif {$prog == "treor"} {
	    set list1 [ParseTreorLong $file $txt]
	    incr j
	    grid [label $top.$j -text "lines not\nindexed"] -row $row -column $j
	    if {[llength $list1] == 0} {
		eval destroy [winfo children $top]
		grid [label $main.0 -text "(none found)"] \
			  -row 0 -column 0 -columnspan 5
	    }
	    foreach line $list1 {
		incr row
		set k 1
		set zero 0.0
		set cell "P"
		set lnum [lindex $line end]
		grid [button $side.$row -text View \
			  -command "LoadAutoIndexCell [lrange $line 0 5] $cell $zero $lnum $txt"]\
		    -row $row -column $k    
		foreach i {0 1 2 3 4 5 6 7} {
		    incr k
		    incr j
		    grid [label $main.$j -text " [lindex $line $i]"]\
			-row $row -column $k
		}
	    }
	} elseif {$prog == "treor short"} {
	    set list1 [ParseTreorShort $file $txt]
	    incr j
	    grid [label $top.$j -text "lines not\nindexed"] -row $row -column $j
	    if {[llength $list1] == 0} {
		eval destroy [winfo children $top]
		grid [label $main.0 -text "(none found)"] \
			  -row 0 -column 0 -columnspan 5
	    }
	    foreach line $list1 {
		incr row
		set k 1
		set zero 0.0
		set cell "P"
		set lnum [lindex $line end]
		grid [button $side.$row -text View \
			  -command "LoadAutoIndexCell [lrange $line 0 5] $cell $zero $lnum $txt"] \
		    -row $row -column $k    
		foreach i {0 1 2 3 4 5 6 7} {
		    incr k
		    incr j
		    grid [label $main.$j -text " [lindex $line $i]"]\
			-row $row -column $k
		}
	    }
	} else {
	    eval destroy [winfo children $top]
	    grid [label $main.0 -text "(file is not parsed)"] \
		-row 0 -column 0 -columnspan 5
	    set fp [open $file r]
	    while {[gets $fp line] >= 0} {
		$txt.t insert end "$line\n"
	    }
	    close $fp
	    #destroy $box
	}
	$txt.t configure -state disabled
	bind $cellbox <Configure> "+ResizeScrollTable $box"
    }
    if {[llength $textboxlist] == 0} {
	grid [label $cellbox.b1 -text "No results found to report"] \
	    -row 0 -column 2
    }
    set datalist {}
    lappend datalist $aindex(peaklist)
    set aindex(plotfile) $aindex(peaklist)
    foreach data $::graph(datalist) {
	global $data
	if {[set ${data}(type)] != "peaks"} {
	    lappend datalist $data
	}
    }
    catch {destroy $cellbox.f}
    grid [frame $cellbox.f] -row 99 -column 0 -sticky ew
    eval tk_optionMenu $cellbox.f.b1 aindex(plotfile) $datalist
    grid $cellbox.f.b1 -row 99 -column 1 -sticky e
    grid [label $cellbox.f.l1 -text "File to plot:"] -row 99 -column 0 -sticky e
    grid [button $cellbox.b2 -text "Close" \
	      -command "destroy $cellbox; catch \"destroy $textboxlist\""] \
	-row 99 -column 2
    donewait
    update
    raise $cellbox
}

# set the unit cell parameters and compute the hkl positions
proc LoadAutoIndexCell {a b c alpha beta gamma "cell {}" "zero 0" "lnum {}" "txt {}"} {
    global cellparm aindex
    set data autoindex
    catch {unset ::$data}
    # create the peaklist
    set data [initpeaks $data 1]
    global ${data}
    set ${data}(laue) 1
    set ${data}(wavelength) $aindex(wavelength)
    # view the cell in the output file
    if {$lnum != "" && $txt != ""} {
	catch {
	    $txt.t see $lnum.0
	    raise $txt
	}
    }
    # trigger calc of a cell
    editnew
    set command(pagenow) {}
    DoCommand EditCell
    # load the cell 
    foreach item {a b c alpha beta gamma} {
	set ${data}($item) [set $item]
    }
    set ${data}(laue) 1
    if {$alpha == 90 && $beta == 90 && $gamma == 90} {
	if {$a == $b && $b == $c} {set ${data}(laue) 8}
	if {$a == $b} {set ${data}(laue) 6}
	set ${data}(laue) 4
    } elseif {$alpha == 90 && $beta == 90 && $gamma == 120 \
	    && $a == $b} {
	set ${data}(laue) 14
    } elseif {$alpha == 90 && $gamma == 90} {
	set ${data}(laue) 2
    }
    # set an initial spacegroup
    #set ${data}(spg) {P 1}
    # set some values in EditCell if we are passing a file to plot
    if {$aindex(plotfile) != ""} {
	global $aindex(plotfile)
	#set 2Theta Max
	catch {set cellparm(twothetamax) $::aindex(xmax)} err
	# set I min/max
	if {[set ${aindex(plotfile)}(type)] == "peaks"} {
	    # for peaks plot downwards
	    catch {set cellparm(ymin) $::aindex(ymin)} err
	    catch {set cellparm(ymax) [expr {-0.1*$::aindex(ymax)}]} err
	} else {
	    set yvec [set ${aindex(plotfile)}(yvector)]
	    set cellparm(ymin) 0
	    catch {
		set ymax [set ::${yvec}(max)]
		set cellparm(ymax) $ymax
	    }
	}
    }
    # select the peak file for display
    set i 0
    $cellparm(editdatalist) select clear 0 end
    foreach data [$cellparm(editdatalist) get 0 end] {
	if {$data == $aindex(plotfile)} {
	    $cellparm(editdatalist) select set $i
	}
	incr i
    }
    set cellparm(editsetting) ${data}
    #SetEditData 0 0 0
}

proc IndexDiscardPeaks {} {
    global aindex
    set data $aindex(peaklist)
    if {$data == ""} return
    global $data
    # determine x-units 
    set xunits [set ${data}(xunits)]
    if {$xunits != "2theta" && $xunits != "dspace" && $xunits != "Q"} {
	set xunits dspace
    }

    set frm .pks
    catch {destroy $frm}
    toplevel $frm
    wm title $frm "Filter peaks"
    grid [label $frm.title -text "Select peaks to be deselected"] -row 0 -column 0 -columnspan 3
    grid [frame $frm.1] -row 2 -column 0 -columnspan 3
    grid [frame $frm.2] -row 3 -column 0 -columnspan 3
    grid [frame $frm.3] -row 4 -column 0 -columnspan 3

    # set defaults
    if {[catch {set aindex(setuseby)}]} {set aindex(setuseby) 1}
    if {[catch {set aindex(threshold)}]} {set aindex(threshold) 0.0}
    if {[catch {set aindex(thresholdtype)}]} {set aindex(thresholdtype) below}

    set row 0
    grid [label $frm.1.ll -text "Filter peaks by:"] \
	    -row [incr row] -column 2 -sticky w
    grid [radiobutton $frm.1.d1 -text $xunits -value 1 \
	    -variable aindex(setuseby)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [radiobutton $frm.1.d2 -text "Intensity" -value 2 \
	    -variable  aindex(setuseby)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [radiobutton $frm.1.d3 -text "Width" -value 3 \
	    -variable  aindex(setuseby)] -row [incr row] -column 2 -columnspan 2 -sticky w
    
    grid [label $frm.2.ll -text "Ignore peaks:"] \
	    -row [incr row] -column 2 -sticky w
    grid [radiobutton $frm.2.d1 -text "Above threshold" -value above \
	    -variable aindex(thresholdtype)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w
    grid [radiobutton $frm.2.d2 -text "Below threshold" -value below \
	    -variable aindex(thresholdtype)] \
	    -row [incr row] -column 2 -columnspan 2 -sticky w

    grid [label $frm.3.ll -text "Threshold:"] \
	    -row 1 -column 1 -sticky w
    grid [entry $frm.3.d2 -textvariable aindex(threshold)] \
	    -row 1 -column 2 -columnspan 2 -sticky w

    grid [button $frm.reset -text "Set all peaks as included" -command IndexIncludeAllPeaks] -row 1 -column 1 -columnspan 2
    grid [button $frm.go -text "Deselect Peaks\nOutside Criterion" -command "IndexFilterPeaks $frm"] -row 99 -column 1
    grid [button $frm.stop -text "Cancel" -command "destroy $frm"] -row 99 -column 2
    putontop $frm
    tkwait window $frm
    afterputontop
}

proc IndexIncludeAllPeaks {} {
    global aindex
    foreach var [array names aindex {peak[0-9]*}] {
	set aindex($var) 1
    }
}

proc IndexFilterPeaks {frm} {
    global aindex
    set msg {}
    if {[catch {expr $aindex(threshold)}]} {
	set msg "Threshold (=\"$aindex(threshold)\") is invalid\n"
    }
    if {$msg != ""} {
	tk_dialog $frm.err "Invalid input" $msg error 0 "Try again"
	return
    }
    # compile a list of two-theta values that have been selected
    set data $aindex(peaklist)
    # should never happen, but...
    if {$data == ""} {
	tk_dialog $frm.err "Invalid peak list" \
		"Select a peak list first" error 0 "Try again"
	return
    }
    global $data
    if {$aindex(setuseby) == 1} {
	set xunits [set ${data}(xunits)]
	set list [set ${data}(x)]
	if {$xunits != "2theta" && $xunits != "dspace" && $xunits != "Q"} {
	    set xunits dspace
	    set list [set ${data}(dspaces)]
	}
    } elseif {$aindex(setuseby) == 2} {
	set list [set ${data}(heights)]
    } elseif {$aindex(setuseby) == 3} {
	set list [set ${data}(widths)]
    } else {
	tk_dialog $frm.err "Invalid option" \
		"Invalid filter by option" error 0 "Try again"
	return
    }
    if {$aindex(thresholdtype) == "above"} {
	set type 0
    } elseif {$aindex(thresholdtype) == "below"} {
	set type 1
    } else {
	tk_dialog $frm.err "Invalid option" \
		"Invalid ignore by option" error 0 "Try again"
	return
    }

    set i 0
    foreach val $list {
	incr i
	if {$type == 0 && $val > $aindex(threshold)} {
	    set aindex(peak$i) 0
	} elseif {$type == 1 && $val < $aindex(threshold)} {
	    set aindex(peak$i) 0
	}
    }
    catch {destroy $frm}
}