#-------------------------------------------------------------------------
# notes:
# all "data" or "peaks" entries have the following data items defined 
# (for entry "dat"):
#    dat(type) datatype == xy (data) or peaks
#    dat(xvector) name of x vector (dat_x)
#    dat(yvector) name of y vector (dat_y)
#    dat(x) x data -- as read
#    dat(y) intensity data (or peak areas)
#    dat(line) line type be plotted
#    dat(color) color to be plotted
#    dat(symbol) symbol type  be plotted
#    dat(symsize) relative symbol size be plotted
#    dat(title) descriptive text
#    dat(xlabel) x axis label for raw data
#    dat(ylabel) y axis label for raw data
#    dat(cxlabel) x axis label for data in current units
#    dat(cylabel) y axis label for data in current units
#
#  data "file" dat has the following array elements defined:
#    dat(esdvec) name of esd vector (dat_esd)
#    dat(y)  y data -- as read
#    dat(esd)  esd data -- as read
#  data "file" dat has the following vectors defined:
#     dat_x     x axis
#     dat_y     y axis
#     dat_esd   esd on y axis
#        these are derived from dat(x), dat(y) and dat(z)
# 
# files of peaks are handled a bit differently
#   peak "file" dat has the following array elements defined:
#     dat(h) dat(k) dat(l) hkl for reflections
#     dat(ttvector) name of two-theta vector, dat_tt
    # _tt is used for reflection labeling (lblhkl)
#     dat(extinctions) {} 1 for extinct peaks
#     dat(dspaces) {}
#     dat(wavelength) ""
#     dat(wavelength2) ""
#     dat(a) dat(b) dat(c) dat(alpha) dat(beta) dat(gamma) 
#     dat(old_a) dat(old_b) dat(old_c) 
#     dat(old_alpha) dat(old_beta) dat(old_gamma) 
#     dat(heights) peak heights 
#     dat(widths) intensity data (or peak areas)
#     dat(etas) 0 = Gaussian; 1 = Cauchy
#     dat(bkgs)  background values
#  peak "file" dat has the following vectors defined:
#     dat_x     x axis
#     dat_y     y axis
#     dat_tt
# note that the name "dat" must be added to list graph(datalist) to be used
#
#########################################################################
proc initdata {data} {
    global graph
    # is data a valid name?
    regsub -all  {\.} $data {} data
    set data_orig $data
    set getnewdataname 1
    while {$getnewdataname} {
	set getnewdataname 0
	# is this name in use somewhere?
	if {[info vars ::$data] != ""} {
	    if {[lsearch {graph command makenew gpls} $data] != -1} {
		# test for global arrays that should not be overwritten (probably incomplete)
		set getnewdataname 1
		tk_messageBox -type ok -parent . \
		    -message "Dataset name \"$data\" is not allowed. Select a new name." 
	    } elseif {[array exists ::$data]} {
		# if it is an array assume this is OK to overwrite
		set answer [tk_messageBox -message "Dataset $data already exists! Overwrite?" -type yesno -parent .]
		if {$answer == "no"} {
		    set getnewdataname 1
		}
	    } else {
		set getnewdataname 1
		tk_messageBox -type ok -parent . \
		    -message "Dataset name \"$data\" is not allowed. Select a new name." 
	    }
        }
	# Get new name if needed
	if {$getnewdataname} {
	    set data [modalDialog "New Data Name" "New Name" [append data "_1"]]
	    regsub -all  {\.} $data {} data
	    # If cancel was selected in the modal Dialogue, start loop over again and reprompt yesorno
	    if {[string compare $data "ID_CANCEL"] == 0} { 
		set getnewdataname 1
		set data $data_orig
	    } else {
		regsub -all  {\.} $data {} data
	    }
	}
    }
    global $data 
    regsub -all  {[\*\+-]} $data {_} cdat
    # define vectors for the data
    set ${data}(type) xy
    set ${data}(xvector) ${cdat}_x
    set ${data}(yvector) ${cdat}_y
    set ${data}(esdvec) ${cdat}_esd
    global [set ${data}(xvector)] [set ${data}(yvector)] [set ${data}(esdvec)]
    catch {vector [set ${data}(xvector)]}
    catch {vector [set ${data}(yvector)]}
    catch {vector [set ${data}(esdvec)]}
    set ${data}(wavelength) ""
    set ${data}(wavelength2) ""
    set ${data}(xunits) ""
    set ${data}(xlabel) ""
    set ${data}(ylabel) ""
    set ${data}(cxlabel) ""
    set ${data}(cylabel) ""
    set ${data}(title) ""
    # initialize arrays
    set ${data}(x) ""
    set ${data}(y) ""
    set ${data}(esd) ""
    # set the color and line type
    incr graph(colorindex) 
    if {$graph(colorindex) == $graph(ncolors)} {set graph(colorindex) 0}
    incr graph(lineindex) 
    if {$graph(lineindex) == $graph(nlines)} {set graph(lineindex) 0}
    set ${data}(line) 1
    set ${data}(color) [lindex $graph(colorlist) $graph(colorindex)]
    set ${data}(symbol) [lindex $graph(linelist) $graph(lineindex)]
    set ${data}(symsize) 1.0
    # add to list of defined arrays
    if {[lsearch -exact $graph(datalist) $data] == -1} {
	lappend graph(datalist) $data
    }
    return $data
}

proc initpeaks {data "overwrite 0"} {
    global graph
    # is data a valid name?
    regsub -all  {\.} $data {} data
    set getnewdataname 1
    while {$getnewdataname} {
	set getnewdataname 0
	# is this name in use somewhere?
	if {[info vars ::$data] != ""} {
	    if {[lsearch {graph command makenew gpls} $data] != -1} {
		# test for global arrays that should not be overwritten (probably incomplete)
		set getnewdataname 1
		tk_messageBox -type ok -parent . \
		    -message "Dataset name \"$data\" is not allowed. Select a new name." 
	    } elseif {$overwrite && [array exists ::$data]} {
		set getnewdataname 0
	    } elseif {[array exists ::$data]} {
		# if it is an array assume this is OK to overwrite
		set answer [tk_messageBox -message "Dataset $data already exists! Overwrite?" -type yesno -parent .]
		if {$answer == "no"} {
		    set getnewdataname 1
		}
	    } else {
		set getnewdataname 1
		tk_messageBox -type ok -parent . \
		    -message "Dataset name \"$data\" is not allowed. Select a new name." 
	    }
        }
	# Get new name if needed
	if {$getnewdataname} {
	    set data [modalDialog "New Data Name" "New Name" [append data "_1"]]
	    regsub -all  {\.} $data {} data
	    # If cancel was selected in the modal Dialogue, start loop over again and reprompt yesorno
	    if {[string compare $data "ID_CANCEL"] == 0} { 
		set getnewdataname 1
		set data $data_orig
	    } else {
		regsub -all  {\.} $data {} data
	    }
	}
    }
    global $data 
    # define vectors for the data
    set ${data}(type) peaks
    regsub -all  {\.} $data {} cdat
    regsub -all  {[\*\+-]} $cdat {_} cdat
    set ${data}(xvector) ${cdat}_x
    set ${data}(yvector) ${cdat}_y
    set ${data}(ttvector) ${cdat}_tt
    global [set ${data}(xvector)] [set ${data}(yvector)] [set ${data}(ttvector)]
    catch {vector [set ${data}(xvector)]}
    catch {vector [set ${data}(yvector)]}
    catch {vector [set ${data}(ttvector)]}
    # initialize arrays
    foreach var {x y h k l extinctions dspaces widths etas bkgs heights \
	    wavelength xunits xlabel ylabel cxlabel cylabel title spg \
	    extcodes } {
	set ${data}($var) {}
    }
    # set the color and line type
    set ${data}(line) 1
    incr graph(colorindex) 
    if {$graph(colorindex) == $graph(ncolors)} {set graph(colorindex) 0}
    incr graph(lineindex) 
    if {$graph(lineindex) == $graph(nlines)} {set graph(lineindex) 0}
    set ${data}(color) [lindex $graph(colorlist) $graph(colorindex)]
    set ${data}(symbol) [lindex $graph(linelist) $graph(lineindex)]
    set ${data}(symsize) 1.0
    # add to list of defined arrays
    if {[lsearch -exact $graph(datalist) $data] == -1} {
	lappend graph(datalist) $data
    }
    foreach var {a b c alpha beta gamma} {
	set ${data}(old_$var) {}
	set ${data}($var) {}
    }
    return $data
}

proc peaktovector {data} {
    global $data cellparm
    set xtmp {}
    set ytmp {}
    # put the peaks in order
    set l {}
    foreach peak [set ${data}(x)] \
	    h [set ${data}(heights)] \
	    b [set ${data}(bkgs)] {
	if ![catch {expr $peak}] {
	    lappend l [list $peak $h $b]
	}
    }
    foreach triple [lsort -index 0 -real $l] {
	set peak [lindex $triple 0]
	set h [lindex $triple 1]
	set b [lindex $triple 2]
	if {$peak > 0} {
	    lappend xtmp $peak $peak $peak
	    set ymin $cellparm(ymin)
	    set ymax $cellparm(ymax)
	    catch {
		if {$h != "" &&  $h > 0} {
		    set ymin $b
		    set ymax [expr $h+$b]
		}
	    }
	    lappend ytmp $ymin $ymax $ymin
	}
    }
    [set ${data}(ttvector)] set [set ${data}(x)]
    [set ${data}(xvector)] notify never
    [set ${data}(yvector)] notify never
    [set ${data}(xvector)] set $xtmp
    [set ${data}(yvector)] set $ytmp
    [set ${data}(xvector)] notify now
    [set ${data}(yvector)] notify now
    [set ${data}(xvector)] notify always
    [set ${data}(yvector)] notify always
}

proc resetdata {data} {
    global $data 
    if {[set ${data}(type)] == "xy"} {
	[set ${data}(xvector)] set [set ${data}(x)]
	# make sure that all data arrays are the same length as the data
	foreach elem {y esd} {
	    for  {set i [llength [set ${data}($elem)]]} \
		    {$i < [llength [set ${data}(x)]]} {incr i} {
		lappend ${data}($elem) -1
	    }
	}
	[set ${data}(yvector)] set [set ${data}(y)]
	[set ${data}(esdvec)] set [set ${data}(esd)]
    } elseif {[set ${data}(type)] == "peaks"} {
	# make sure that all peak arrays are the same length as the data
	foreach elem {y widths etas heights bkgs extinctions h k l dspaces} {
	    for  {set i [llength [set ${data}($elem)]]} \
		    {$i < [llength [set ${data}(x)]]} {incr i} {
		lappend ${data}($elem) {}
	    }
	}
	peaktovector $data
    }
    set ${data}(cxlabel) [set ${data}(xlabel)]
    set ${data}(cylabel) [set ${data}(ylabel)]
}

proc scaledata {data xoffset xmult yoffset ymult} {
    # scale x values as xoffset + (xmult * data)
    global $data
    if {$xmult != 1} {[set ${data}(xvector)] set [[set ${data}(xvector)] * $xmult]}
    if {$xoffset != 0} {[set ${data}(xvector)] set [[set ${data}(xvector)] + $xoffset]}
    if {$ymult != 1} {
	[set ${data}(yvector)] set [[set ${data}(yvector)] * $ymult]
	if {[set ${data}(type)] == "xy"} {
	    [set ${data}(esdvec)] set [[set ${data}(esdvec)] * $ymult]
	}
    }
    if {$yoffset != 0} {[set ${data}(yvector)] set [[set ${data}(yvector)] + $yoffset]}
}

proc ConvertY {data mode} {
    # convert intensities to sqrt(I), log(I), I/sig(I)
    global $data
    if {$mode == ""} {
	return
    }
    if {$mode == "sqrt"} {
	foreach y [[set ${data}(yvector)] range 0 end] {
	    set yt 0.0
	    catch {set yt [expr sqrt($y)]}
	    lappend ytmp $yt
	}
	[set ${data}(yvector)] set $ytmp
	set ${data}(cylabel) "sqrt(I)"
    } elseif {$mode == "log"} {
	foreach y [[set ${data}(yvector)] range 0 end] {
	    set yt 0.0
	    catch {set yt [expr log10($y)]}
	    lappend ytmp $yt
	}
	set ${data}(cylabel) "log(I)"
	[set ${data}(yvector)] set $ytmp
    } elseif {$mode == "s-n"} {
	foreach y   [[set ${data}(yvector)] range 0 end] \
		err [[set ${data}(esdvec)]  range 0 end] {
	    set yt 0.0
	    catch {set yt [expr $y/$err]}
	    lappend ytmp $yt
	}
	set ${data}(cylabel) "I/sigma_I"
	[set ${data}(yvector)] set $ytmp
    } elseif {$mode == "unscale"} {
	foreach y   [[set ${data}(yvector)] range 0 end] \
		err [[set ${data}(esdvec)]  range 0 end] {
	    set yt 0.0
	    catch {set yt [expr {($y/$err)*($y/$err)}]}
	    lappend ytmp $yt
	    set et 0.0
	    catch {set et [expr {($y/$err)}]}
	    lappend etmp $et
	}
	set ${data}(cylabel) "unscaled counts"
	[set ${data}(yvector)] set $ytmp
	[set ${data}(esdvec)] set $etmp
    }
}

proc ConvertX {data mode {wave ""}} {
    # convert x-axis to 2theta, Q or dspace 
    global command
    global $data
    if {$mode == ""} {
	return
    }
    if {$mode == "2theta"} {
	# check the conversion wavelength
	if [catch {expr $wave}] {
	    tk_dialog .warn "Invalid wavelength" \
	"Unable to convert data to 2-theta units: \
	Invalid conversion wavelength, \"$wave\"" warning 0 OK
	return
	}
    }
    if {[set ${data}(xunits)] == "2theta"} {
	while {[catch "expr [set ${data}(wavelength)]"]} {
	    catch {destroy .wave}
	    toplevel .wave
	    wm title .wave "Get wavelength"
	    pack [label .wave.0 -text \
		    "Enter a wavelength for dataset $data"]
	    pack [frame .wave.1] -side top -expand yes -fill both
	    pack [label .wave.1.a -text Wavelength] -side left 
	    pack [entry .wave.1.b -textvariable ${data}(wavelength)] -side left 
	    pack [button .wave.2 -text OK -command "destroy .wave"] -side top 
	    # reuse the last value as the next default
	    catch {set [set data](wavelength) $command(lastwavelength)}	    
	    grab .wave
	    tkwait window .wave
	    set command(lastwavelength) [set ${data}(wavelength)]
	}
	if {$mode == "2theta"} {
	    set torad [expr 3.14159 / 360.]
	    set wrat [expr $wave / [set ${data}(wavelength)]]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 180.0
		catch {set xt [expr asin($wrat * sin($x*$torad))/$torad ]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "2 Theta @ $wave"
	} elseif {$mode == "Q"} {
	    set torad [expr 3.14159 / 360.]
	    set pi4ow [expr  4. * 3.14159 / \
		    [set ${data}(wavelength)]]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 0.0
		catch {set xt [expr $pi4ow * sin($x*$torad)]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "Q"
	} elseif {$mode == "dspace"} {
	    set torad [expr 3.14159 / 360.]
	    set wo2 [expr 0.5 * [set ${data}(wavelength)]]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 0.0
		catch {set xt [expr $wo2 / sin($x*$torad) ]} err
		lappend xtmp $xt
	    }
	    set ${data}(cxlabel) "d-space"
	    [set ${data}(xvector)] set $xtmp
	}
	return
    } elseif {[set ${data}(xunits)] == "dspace"} {
	if {$mode == "2theta"} {
	    set torad [expr 3.14159 / 360.]
	    set wo2 [expr $wave / 2.]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 180.0
		catch {set xt [expr asin( $wo2/$x )/$torad ]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "2 Theta @ $wave"
	} elseif {$mode == "Q"} {
	    set pi2 [expr  2. * 3.14159 ]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 0.0
		catch {set xt [expr $pi2 / $x]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "Q"
	} elseif {$mode == "dspace"} {
	    return
	}
    } elseif {[set ${data}(xunits)] == "EDSKEV"} {
	while {[catch "expr [set ${data}(wavelength)]"]} {
	    catch {destroy .wave}
	    toplevel .wave
	    wm title .wave "Get 2Theta"
	    pack [label .wave.0 -text \
		    "Enter the 2Theta value for dataset $data"]
	    pack [frame .wave.1] -side top -expand yes -fill both
	    pack [label .wave.1.a -text 2Theta] -side left 
	    pack [entry .wave.1.b -textvariable ${data}(wavelength)] -side left 
	    pack [button .wave.2 -text OK -command "destroy .wave"] -side top 
	    # reuse the last value as the next default
	    catch {set [set data](wavelength) $command(lastwavelength)}	    
	    grab .wave
	    tkwait window .wave
	    set command(lastwavelength) [set ${data}(wavelength)]
	}
	if {$mode == "2theta"} {
	    set torad [expr 3.14159 / 360.]
	    set wo2 [expr $wave / 2.]
	    # wavelength is 2theta for EDS
	    set cnv [expr 6.199 / sin ($torad * [set ${data}(wavelength)])]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 180.0
		catch {set xt [expr asin( $wo2*$x/$cnv )/$torad ]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "d-space"
	} elseif {$mode == "Q"} {
	    set torad [expr 3.14159 / 360.]
	    # wavelength is 2theta for EDS
	    set cnv [expr 6.199 / sin ($torad * [set ${data}(wavelength)])]
	    set pi4 [expr  4. * 3.14159 ]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 0.0
		catch {set xt [expr $pi4 * $x / $cnv]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "Q"
	} elseif {$mode == "dspace"} {
	    set torad [expr 3.14159 / 360.]
	    # wavelength is 2theta for EDS
	    set cnv [expr 6.199 / sin ($torad * [set ${data}(wavelength)])]
	    foreach x [[set ${data}(xvector)] range 0 end] {
		set xt 9999
		catch {set xt [expr $cnv / $x]}
		lappend xtmp $xt
	    }
	    [set ${data}(xvector)] set $xtmp
	    set ${data}(cxlabel) "d-space"
	}
	return
    } else {
	tk_dialog .warn "No conversion" \
	"Unable to convert [set ${data}(xunits)] data to \
	$mode units: conversion not programmed" warning 0 OK
	return
    }
}

# Used to ask user for new data name if it already exists
# Based on code available at http://bioinf.wehi.edu.au/~wettenhall/RTclTkExamples/modalDialog.html
proc modalDialog {title question entryInit {entryWidth 30} {returnValOnCancel "ID_CANCEL"}} {
  toplevel .newdatawindow
  raise .newdatawindow
  wm deiconify .newdatawindow
  grab set .newdatawindow
  focus .newdatawindow
  wm title .newdatawindow "$title"
  grid [label .newdatawindow.l1 -text "       "] -row 1 -column 1 -columnspan 2
  grid [label .newdatawindow.ql -text "$question"] -row 2 -column 1
  grid [entry .newdatawindow.tew -width $entryWidth -textvariable entryInit] -row 2 -column 2
  grid [label .newdatawindow.l2 -text "       "] -row 3 -column 1 -columnspan 2
  global rv_modal
  set rv_modal $returnValOnCancel
  # TMM: Hack that is needed for some reason
  .newdatawindow.tew delete 0 end
  .newdatawindow.tew insert 0 "$entryInit"
  proc onOK {} {
    global rv_modal
    set rv_modal [.newdatawindow.tew get]
    grab release .newdatawindow
    destroy .newdatawindow
  }
  proc onCancel {} {
    grab release .newdatawindow
    destroy .newdatawindow
  }
  grid [button .newdatawindow.ok -text "   OK   " -command "onOK"] -row 4 -column 1
  grid [button .newdatawindow.cancel -text " Cancel " -command "onCancel"] -row 4 -column 2
  grid [label .newdatawindow.l3 -text "       "] -row 5 -column 1 -columnspan 2

  bind .newdatawindow "<Destroy>" onCancel
  bind .newdatawindow "<Return>" onOk
  wm geometry .newdatawindow ""
  focus .newdatawindow
  tkwait window .newdatawindow

  return $rv_modal
}

#-------------------------------------------------------------------------
# file box stuff
#-------------------------------------------------------------------------
# make a box to select the data sets
proc makeselectbox {box} {
    grid [listbox $box.1 -yscrollcommand "$box.scroll set" \
	      -xscrollcommand "$box.xscroll set" \
	      -height 8 -width 15 -bd 2 -relief raised ] -column 0 -row 0 \
	-sticky news
    $box.1 config -selectmode extended -exportselection 0
    grid [scrollbar $box.scroll -command "$box.1 yview"] \
	    -column 1 -row 0 -sticky ns
    grid [scrollbar $box.xscroll -command "$box.1 xview" \
	      -orient horizontal ] -column 0 -row 1 -sticky ew
    bind $box.1 <Double-Button-1> "getfilelist $box; displaylist"
    bind $box.1 <Triple-Button-1> \
	    "$box.1 selection set 0 end; getfilelist $box; displaylist"
    bind $box.1 <Button-3> \
	    "$box.1 selection set 0 end; getfilelist $box; displaylist"
    bind $box.1 <Control-Button-3> "deletelist $box; getfilelist $box; displaylist"
    # bindings to allow + and - keys to reorder list
    bind $box.1 <KeyPress-plus> "ReorderBox $box.1 1"
    bind $box.1 <KeyPress-minus> "ReorderBox $box.1 0"
    # 
    bind $box.1 <Enter> "focus $box.1"
}

# reorder the listbox 
proc ReorderBox {box dir} {
    set selected [$box curselection]
    if {[llength $selected] != 1} {
	#puts "can't move more than one"
	bell
	return
    }
    # moving up
    set target $selected
    set value [lindex $::graph(datalist) $selected]
    if {$dir == 1} {
	if {$selected == 0} {
	    #puts "can't move up"
	    bell
	    return
	}
	incr target -1
    }
    if {$dir == 0} {
	incr target
	if {$target == [llength $::graph(datalist)]} {
	    #puts "can't move down"
	    bell
	    return
	}
    }
    # reorder the list
    set ::graph(datalist) [linsert \
				[lreplace $::graph(datalist) $selected $selected]\
				$target $value]
    # show the change
    updateselectbox [winfo parent $box]
    # select the moved item
    $box selection clear 0 end
    $box selection set $target
    $box see $target
}

set helplist(FileSelectionTricks) {
Where a list of datasets is shown, entries are selected by 
a left-click. Holding control down and a left-click 
(control-left-click) causes the selected to be selected 
or unselected. Holding the shift down and dragging the 
left mouse button (shift-left-drag) causes a consecutive 
list of datasets to be selected.

Double-left click causes the current dataset(s) to be 
selected and plotted (OK to combine with shift or control). 

A triple-left click or a single right-click (Mac: Apple+click) 
causes all datasets to be selected and plotted.

Control-right-click (Mac: Control+Apple+click) causes the selected 
datasets to be removed from the program list (not deleted from disk). 
You are prompted before datasets are removed.

+ causes the selected dataset to be moved up in the list
- causes the selected dataset to be moved down in the list
(note that these options only work if a single data set is selected).
}
# update the file list
proc updateselectbox {box {mode extended}} {
    global graph
    $box.1 delete 0 end
    eval  $box.1 insert 0 $graph(datalist)
    $box.1 config -selectmode $mode
}

# update and scroll so the last entry is showing
proc showlastentry {filebox} {
    if {$filebox == ""} return
    global command
    updateselectbox $filebox
    $filebox.1 see end 
}
# set graph(plotlist) to the names of the selected entries
proc getfilelist {box} {
    global graph
    set graph(plotlist) {}
    foreach num [$box.1 curselection ] {
	lappend graph(plotlist) [$box.1 get $num]
    }
}
# deletelist deletes selected files
proc deletelist {box} {
    global graph 
    set filelist {}
    foreach entry [$box.1 curselection] {
	append filelist "\n" [lindex $graph(datalist) $entry]	
    }
    if {[MyMessageBox -parent . -title "OK to delete?" \
	     -message "OK to delete dataset(s):\n$filelist" \
	     -icon warning -type {Yes No} -default yes] != "yes"} return
    foreach entry [lsort -decreasing -integer [$box.1 curselection]] {
	set data [lindex $graph(datalist) $entry]
	deletedata $data
    }
    updateselectbox $box
}

proc deletedata {data} {
    global $data graph
    # delete vectors
    catch {vector destroy [set ${data}(xvector)]} errmsg
    catch {vector destroy [set ${data}(yvector)]} errmsg
    catch {vector destroy [set ${data}(esdvec)]} errmsg
    catch {vector destroy [set ${data}(ttvector)]} errmsg
    # delete the array
    catch {unset $data} errmsg
    # remove from the list
    set entry [lsearch $graph(datalist) $data]
    if {$entry == -1} return
    set graph(datalist) [lreplace $graph(datalist) $entry $entry]
}

proc copydata {data newdata} {
    global $data
    if {[set ${data}(type)] == "xy"} {
	set newdata [initdata $newdata]
    } elseif {[set ${data}(type)] == "peaks"} {
	set newdata [initpeaks $newdata]
    }
    global $newdata
    foreach name [array names $data] {
	if {[string match "*vec*" $name]} {
	    if {[info command [set ${data}($name)]] != ""} {
		catch {[set ${data}($name)] dup [set ${newdata}($name)]} err
	    }
	} else {
	    set ${newdata}($name) [set ${data}($name)]	
	}
    }
}

# reverse the order of a list
proc ReverseList {list} {
    set _temp {}
    for {set i [ expr [ llength $list ] - 1 ] } \
	{$i >= 0} \
	{incr i -1} {
	    lappend _temp [ lindex $list $i ]
 	}
    return  $_temp
}

# displaylist plots the entries in graph(plotlist)
proc displaylist {} {
    global graph command
    eval $graph(blt) element delete [$graph(blt) element names]
    eval $graph(blt) marker delete  [$graph(blt) marker names]
    set firstpass 1
    if {$graph(ReversePlotOrder)} {
	set itemstoplot [ReverseList $graph(plotlist)]
    } else {
	set itemstoplot $graph(plotlist) 
    }
    foreach data $itemstoplot {
	global $data
	# this next change will be needed for Tcl/Tk8.5 & BLT
	catch {global [set ${data}(xvector)] [set ${data}(yvector)]}
	if {$firstpass} {
	    $graph(blt) xaxis configure -title [set ${data}(cxlabel)]
	    $graph(blt) yaxis configure -title [set ${data}(cylabel)]
	    set firstpass 0
	}
	catch {$graph(blt) element delete $data}
	$graph(blt) element create $data \
		-linewidth [set ${data}(line)] \
		-color [set ${data}(color)] -symbol [set ${data}(symbol)] \
		-xdata [set ${data}(xvector)] -ydata [set ${data}(yvector)] \
		-pixels [expr 0.125*[set ${data}(symsize)]]i
#		-activecolor black 
	if {$graph(LabelByTitle)} {
	    catch {
		$graph(blt) element config $data -label [set ${data}(title)]
	    }
	}
	 [set ${data}(xvector)] notify now
	 [set ${data}(yvector)] notify now
    }
}

#------------------------------------------------------------------------------
#	Message box code that centers the message box over the parent.
#          or along the edge, if too close, 
#          but leave a border along +x & +y for reasons I don't remember
#       It also allows the button names to be defined using 
#            -type $list  -- where $list has a list of button names
#       larger messages are placed in a scrolled text widget
#       capitalization is now ignored for -default
#       The command returns the name button in all lower case letters
#       otherwise see  tk_messageBox for a description
#
#       This is a modification of tkMessageBox (msgbox.tcl v1.5)
#
proc MyMessageBox {args} {
    global tkPriv tcl_platform

    set w tkPrivMsgBox
    upvar #0 $w data

    #
    # The default value of the title is space (" ") not the empty string
    # because for some window managers, a 
    #		wm title .foo ""
    # causes the window title to be "foo" instead of the empty string.
    #
    set specs {
	{-default "" "" ""}
        {-icon "" "" "info"}
        {-message "" "" ""}
        {-parent "" "" .}
        {-title "" "" " "}
        {-type "" "" "ok"}
        {-helplink "" "" ""}
    }

    tclParseConfigSpec $w $specs "" $args

    if {[lsearch {info warning error question} $data(-icon)] == -1} {
	error "bad -icon value \"$data(-icon)\": must be error, info, question, or warning"
    }
    if {![string compare $tcl_platform(platform) "macintosh"]} {
      switch -- $data(-icon) {
          "error"     {set data(-icon) "stop"}
          "warning"   {set data(-icon) "caution"}
          "info"      {set data(-icon) "note"}
	}
    }

    if {![winfo exists $data(-parent)]} {
	error "bad window path name \"$data(-parent)\""
    }

    switch -- $data(-type) {
	abortretryignore {
	    set buttons {
		{abort  -width 6 -text Abort -under 0}
		{retry  -width 6 -text Retry -under 0}
		{ignore -width 6 -text Ignore -under 0}
	    }
	}
	ok {
	    set buttons {
		{ok -width 6 -text OK -under 0}
	    }
          if {![string compare $data(-default) ""]} {
		set data(-default) "ok"
	    }
	}
	okcancel {
	    set buttons {
		{ok     -width 6 -text OK     -under 0}
		{cancel -width 6 -text Cancel -under 0}
	    }
	}
	retrycancel {
	    set buttons {
		{retry  -width 6 -text Retry  -under 0}
		{cancel -width 6 -text Cancel -under 0}
	    }
	}
	yesno {
	    set buttons {
		{yes    -width 6 -text Yes -under 0}
		{no     -width 6 -text No  -under 0}
	    }
	}
	yesnocancel {
	    set buttons {
		{yes    -width 6 -text Yes -under 0}
		{no     -width 6 -text No  -under 0}
		{cancel -width 6 -text Cancel -under 0}
	    }
	}
	default {
#	    error "bad -type value \"$data(-type)\": must be abortretryignore, ok, okcancel, retrycancel, yesno, or yesnocancel"
	    foreach item $data(-type) {
		lappend buttons [list [string tolower $item] -text $item -under 0]
	    }
	}
    }

    if {[string compare $data(-default) ""]} {
	set valid 0
	foreach btn $buttons {
	    if {![string compare [lindex $btn 0] [string tolower $data(-default)]]} {
		set valid 1
		break
	    }
	}
	if {!$valid} {
	    error "invalid default button \"$data(-default)\""
	}
    }

    # 2. Set the dialog to be a child window of $parent
    #
    #
    if {[string compare $data(-parent) .]} {
	set w $data(-parent).__tk__messagebox
    } else {
	set w .__tk__messagebox
    }

    # 3. Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $data(-title)
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW { }
    wm transient $w $data(-parent)
    if {![string compare $tcl_platform(platform) "macintosh"]} {
	unsupported1 style $w dBoxProc
    }

    frame $w.bot
    pack $w.bot -side bottom -fill both
    frame $w.top
    pack $w.top -side top -fill both -expand 1
    if {$data(-helplink) != ""} {
#	frame $w.help
#	pack $w.help -side top -fill both
	pack [button $w.top.1 -text Help -bg yellow \
		-command "MakeWWWHelp $data(-helplink)"] \
		-side right -anchor ne
	bind $w <Key-F1> "MakeWWWHelp $data(-helplink)"
    }
    if {[string compare $tcl_platform(platform) "macintosh"]} {
	$w.bot configure -relief raised -bd 1
	$w.top configure -relief raised -bd 1
    }

    # 4. Fill the top part with bitmap and message (use the option
    # database for -wraplength and -font so that they can be
    # overridden by the caller).

    option add *Dialog.msg.wrapLength 6i widgetDefault

    if {[string length $data(-message)] > 300} {
	if {![string compare $tcl_platform(platform) "macintosh"]} {
	    option add *Dialog.msg.t.font system widgetDefault
	} else {
	    option add *Dialog.msg.t.font {Times 18} widgetDefault
	}
	frame $w.msg
	grid [text  $w.msg.t  \
		-height 20 -width 55 -relief flat -wrap word \
		-yscrollcommand "$w.msg.rscr set" \
		] -row 1 -column 0 -sticky news
	grid [scrollbar $w.msg.rscr  -command "$w.msg.t yview" \
		] -row 1 -column 1 -sticky ns
	# give extra space to the text box
	grid columnconfigure $w.msg 0 -weight 1
	grid rowconfigure $w.msg 1 -weight 1
	$w.msg.t insert end $data(-message)
    } else {
	if {![string compare $tcl_platform(platform) "macintosh"]} {
	    option add *Dialog.msg.font system widgetDefault
	} else {
	    option add *Dialog.msg.font {Times 18} widgetDefault
	}
	label $w.msg -justify left -text $data(-message)
    }
    pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 3m -pady 3m
    if {[string compare $data(-icon) ""]} {
	label $w.bitmap -bitmap $data(-icon)
	pack $w.bitmap -in $w.top -side left -padx 3m -pady 3m
    }

    # 5. Create a row of buttons at the bottom of the dialog.

    set i 0
    foreach but $buttons {
	set name [lindex $but 0]
	set opts [lrange $but 1 end]
      if {![llength $opts]} {
	    # Capitalize the first letter of $name
          set capName [string toupper \
		    [string index $name 0]][string range $name 1 end]
	    set opts [list -text $capName]
	}

      eval button [list $w.$name] $opts [list -command [list set tkPriv(button) $name]]

	if {![string compare $name [string tolower $data(-default)]]} {
	    $w.$name configure -default active
	}
      pack $w.$name -in $w.bot -side left -expand 1 -padx 3m -pady 2m

	# create the binding for the key accelerator, based on the underline
	#
	set underIdx [$w.$name cget -under]
	if {$underIdx >= 0} {
	    set key [string index [$w.$name cget -text] $underIdx]
          bind $w <Alt-[string tolower $key]>  [list $w.$name invoke]
          bind $w <Alt-[string toupper $key]>  [list $w.$name invoke]
	}
	incr i
    }

    # 6. Create a binding for <Return> on the dialog if there is a
    # default button.

    if {[string compare $data(-default) ""]} {
      bind $w <Return> [list tkButtonInvoke $w.[string tolower $data(-default)]]
    }

    # 7. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set wp $data(-parent)
    # center the new window in the middle of the parent
    set x [expr [winfo x $wp] + [winfo width $wp]/2 - \
	    [winfo reqwidth $w]/2 - [winfo vrootx $wp]]
    set y [expr [winfo y $wp] + [winfo height $wp]/2 - \
	    [winfo reqheight $w]/2 - [winfo vrooty $wp]]
    # make sure that we can see the entire window
    set xborder 10
    set yborder 25
    if {$x < 0} {set x 0}
    if {$x+[winfo reqwidth $w] +$xborder > [winfo screenwidth $w]} {
	incr x [expr \
		[winfo screenwidth $w] - ($x+[winfo reqwidth $w] + $xborder)]
    }
    if {$y < 0} {set y 0}
    if {$y+[winfo reqheight $w] +$yborder > [winfo screenheight $w]} {
	incr y [expr \
		[winfo screenheight $w] - ($y+[winfo reqheight $w] + $yborder)]
    }
    wm geom $w +$x+$y
    wm deiconify $w

    # 8. Set a grab and claim the focus too.

    catch {set oldFocus [focus]}
    catch {set oldGrab [grab current $w]}
    catch {
	grab $w
	if {[string compare $data(-default) ""]} {
	    focus $w.[string tolower $data(-default)]
	} else {
	    focus $w
	}
    }

    # 9. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    tkwait variable tkPriv(button)
    catch {focus $oldFocus}
    destroy $w
    catch {grab $oldGrab}
    return $tkPriv(button)
}

#-------------------------------------------------------------------------
# export current plot to Grace
#-------------------------------------------------------------------------
if {$tcl_platform(platform) == "unix"} {
    set graph(gracefile) /tmp/cmpr_grace.agr
} else {
    set graph(gracefile) C:/cmprgrac.agr
}

proc makeGraceExport {} {
    global tcl_platform graph
    catch {toplevel .export}
    raise .export
    eval destroy [grid slaves .export]
    set col 5
    grid [label .export.1a -text Title:] -column 1 -row 1
    grid [entry .export.1b -width 60 -textvariable graph(title)] \
	    -column 2 -row 1 -columnspan 4
    grid [label .export.2a -text Subtitle:] -column 1 -row 2
    grid [entry .export.2b -width 60 -textvariable graph(subtitle)] \
	    -column 2 -row 2 -columnspan 4
    grid [label .export.3a -text "File name:"] -column 1 -row 3
    grid [entry .export.3b -width 60 -textvariable graph(gracefile)] \
	    -column 2 -row 3 -columnspan 4
    grid [button .export.c -text "Close" -command "destroy .export"] \
	    -column [incr col -1] -row 4
    if {$tcl_platform(platform) == "unix"} {
	grid [button .export.d -text "Export & \nstart grace" \
		-command "WriteGrace 1"] \
		-column [incr col -1] -row 4
    }
    grid [button .export.e -text "Export" -command "WriteGrace 0"] \
	    -column [incr col -1] -row 4
}

proc WriteGrace {fork} {
    global graph
    ExportGraceList
    if $fork {
	set err [catch {exec xmgrace $graph(gracefile) &} errmsg]
	if $err {
	    tk_dialog .err "Grace Error" "File $graph(gracefile) created\n\nError running Grace: $errmsg" \
		    error 0 Continue
	}    
    } else {
	tk_dialog .ok " Export Grace" "File $graph(gracefile) created" \
		"" 0 OK
    }
}
proc ExportGraceList {} {
    global graph
    # the colors from yellow4 on are only approximate
    array set gracecolormap {
	black 1 red 2 green 3 blue 4 yellow 5 cyan 9 magenta 10
	yellow4 5 navy 4 purple 10 red4 2 darkolivegreen 3 darkcyan 9 
	royalblue4 4
    }
    array set gracesymbolmap {
	none 0 square 2 circle 1 diamond 3
	plus 8 cross 9 splus 8 scross 9
    }
    array set gracesymbolwidthmap {
	plus 4.0 cross 4.0
    }

    set fp [open $graph(gracefile) w]
    puts $fp {# Grace project file written by CMPR}
    puts $fp {@version 50005}
    puts $fp {#}
    set filenum 0
	# use opposite order from BLT
    if {! $graph(ReversePlotOrder)} {
	set itemstoplot [ReverseList $graph(plotlist)]
    } else {
	set itemstoplot $graph(plotlist) 
    }
    foreach data $itemstoplot {
	global $data 
	global [set ${data}(xvector)] [set ${data}(yvector)]
	set nam s$filenum
	puts $fp "@$nam comment \"[set ${data}(title)]\""
	puts $fp "@$nam legend \"$data\""
	puts $fp "@$nam errorbar off"
	puts $fp "@$nam symbol fill pattern 1"
	# color
	set colnum 12
	catch {set colnum $gracecolormap([set ${data}(color)])}
	puts $fp "@$nam symbol color $colnum"
	puts $fp "@$nam symbol fill color $colnum"
	puts $fp "@$nam line color $colnum"
	if {[set ${data}(type)] == "xy"} {
	    # line type
	    if {[set ${data}(line)] == 0} {
		puts $fp "@$nam line type 0"
	    } else {
		puts $fp "@$nam line type 1"
		puts $fp "@$nam line linewidth [set ${data}(line)]"
	    }
	    # symbol type/width/size
	    set swidth 1.0
	    set sym 10
	    catch {set sym $gracesymbolmap([set ${data}(symbol)])}
	    catch {set swidth $gracesymbolwidthmap([set ${data}(symbol)])}
	    puts $fp "@$nam symbol $sym"
	    puts $fp "@$nam symbol linewidth $swidth"
	    puts $fp "@$nam symbol size [set ${data}(symsize)]"
	    puts $fp "@type xydy"
	    foreach x [ [set ${data}(xvector)] range 0 end] \
		    y [ [set ${data}(yvector)] range 0 end] \
		    e [ [set ${data}(esdvec)] range 0 end] {
		if {$e == ""} {set e 0}
		puts $fp "$x $y $e"
	    }
	    puts $fp "&"
	} elseif {[set ${data}(type)] == "peaks"} {
	    # line type
	    puts $fp "@$nam line type 0"
 	    puts $fp "@$nam baseline type 0"
	    if {[set ${data}(line)] == 0} {
		puts $fp "@$nam symbol pattern 0"
		puts $fp "@$nam symbol fill pattern 0"
	    } else {
		puts $fp "@$nam symbol pattern 1"
		puts $fp "@$nam symbol fill pattern 1"
		puts $fp "@$nam symbol size 0.[set ${data}(line)]"
	    
	    }
	    # symbols are ignored
	    puts $fp "@type bar"
	    set i 0
	    foreach x [ [set ${data}(xvector)] range 0 end] \
		    y [ [set ${data}(yvector)] range 0 end] {
		incr i
		if {$i %3 == 2} {puts $fp "$x $y"}
	    }
	    puts $fp "&"
	}
	if {$filenum == 0} {
	    # save axes labels from first file
	    set xlbl [set ${data}(cxlabel)]
	    set ylbl [set ${data}(cylabel)]
	}
	incr filenum
    }
    # now do scaling
    puts $fp {@with g0}
    puts $fp {@autoscale}
    puts $fp {# is this needed?}
    puts $fp {@    stack world 0, 0, 0, 0}
    puts $fp {@    view xmin 0.150000}
    puts $fp {@    view xmax 1.150000}
    puts $fp {@    view ymin 0.150000}
    puts $fp {@    view ymax 0.850000}
    foreach opt  {-min  -max   -min  -max} \
	    axis {xaxis xaxis yaxis yaxis} \
	    var  { xmin  xmax  ymin  ymax} {
	set val [$graph(blt) $axis cget $opt]
	if {$val != ""} {puts $fp "@world $var $val"}
    }
    puts $fp {@    legend loctype view}
    puts $fp {@    legend x1 0.9}
    puts $fp {@    legend y1 0.8}
    puts $fp "@title \"$graph(title)\""
    puts $fp "@subtitle \"$graph(subtitle)\""
    regsub "2theta" $xlbl {2\f{Symbol}q} xlbl
    regsub "2 Theta" $xlbl  {2\f{Symbol}q} xlbl
    if {$xlbl == "Q"} {set xlbl { Q, \305}}
    puts $fp "@xaxis label \"$xlbl\""
    puts $fp "@yaxis label \"$ylbl\""
    if {$graph(legend)} {
	puts $fp "@legend on"
    } else {
	puts $fp "@legend off"
    }
    puts $fp "@g0 hidden false"
    puts $fp "@autoticks"
    close $fp
}

#-------------------------------------------------------------------------
# export current plot to a CSV file
#-------------------------------------------------------------------------
proc exportPlotSpreadsheet {parent} {
    set file [tk_getSaveFile -title "Select output file" -parent $parent \
	    -defaultextension .csv \
	    -filetypes {{"Comma separated variables" .csv}}]
    if {$file == ""} return
    if {[catch {
	set fp [open $file w]
	output_csv .plot.gr "csv" $fp
	close $fp
    } errmsg]} {
	MyMessageBox -parent $parent -title "Export Error" \
		-message "An error occured during the export: $errmsg" \
		-icon error -type Ignore -default ignore
	return
    }

    MyMessageBox -parent $parent -title "OK" \
	    -message "File $file created" \
	    -type OK -default ok
}

# tcl code to dump the contents of a BLT graph in a file
# based on output_grace by John Cowgill
proc output_csv {graph_name fmt fp} {
    global blt_version
    # trap pre 2.4 BLT versions, where options have different names
    # but beware, really old versions of blt don't have a version number
    if [catch {set blt_version}] {set blt_version 0}
    if {$blt_version <= 2.3 || $blt_version == 8.0} {
	# version 8.0 is ~same as 2.3
	tk_dialog .tooOld "Old BLT" \
		"Sorry, you are using a version of BLT that is too old for this routine" \
		"" 0 OK
	return
    }
    set element_count 0

    # define field separator
    if {$fmt == "csv"} {
	set sep ","
    } else {
	set sep \t
    }

    # get title of graph
    puts $fp "title:${sep}[$graph_name cget -title]"
	
    # get x and y axis limits & labels
    foreach v {x y} {
	set limit_data [$graph_name ${v}axis limits]
	set ${v}min [lindex $limit_data 0]
	set ${v}max [lindex $limit_data 1]
	puts $fp "${v}-axis label:${sep}[$graph_name ${v}axis cget -title]"
	puts $fp "${v}-axis range:${sep}[lindex $limit_data 0]${sep}[lindex $limit_data 1]"
    }

    # loop through each element in the graph but reverse order, so that
    # elements on the bottom are done first
    set element_list [$graph_name element names] 
    set index [llength $element_list]
    set i 0
    set variablelist {}
    set headers {}
    while {[incr index -1] >= 0} {
	set element_name [lindex $element_list $index]
	set element_cmd "$graph_name element cget $element_name"
	
	# get xy data for this element
	set data_list [eval $element_cmd -data]
	
	#if there is no data, skip this set
	if {[llength $data_list] == 0} continue
	incr i
	
	# save the legend name for this element
	set lbl [eval $element_cmd -label]

	# save xy data
	set f 1
	set data(${i}_x) {}
	set data(${i}_y) {}
	lappend variablelist data(${i}_x) data(${i}_y)
	append headers "x-${lbl}${sep}y-${lbl}${sep}"
	foreach item $data_list {
	    if {$f} {
		lappend data(${i}_x) $item
		set f 0
	    } else {
		lappend data(${i}_y) $item
		set f 1
	    }
	}
	
	# check to see if there is -edata defined for this element
	# should work for versions of BLT that do not support -edata
	if {[catch \
		"$graph_name element configure $element_name -edata" edata_list] || \
		[string compare "" [lindex $edata_list 4]] == 0} {
	    # no error data present, just use xy data
	    set data(${i}_ey) ""
	} else {
	    # error data present, check for error vector
	    set edata_list [lindex $edata_list 4]
	    if {[llength $edata_list] == 1} {
		# found a vector name instead of a list, so get the values
		set edata_list [$edata_list range 0 end]
	    }
	    set data(${i}_ey) ""
	    # get xy data for this element
	    set data_list [eval $element_cmd -data]
	    set max [expr {[llength $data_list] / 2}]
	    set data(${i}_ey) ""
	    if {[llength $edata_list] >= [expr {[llength $data_list] * 2}]} {
		for {set i 0} {$i < $max} {incr i} {
		    lappend data(${i}_ey) [lindex $edata_list [expr {4*$i + 1}]]
		}
	    }
	    lappend variablelist data(${i}_ey)
	    append headers "sig(y)-${lbl}${sep}"
	}
    }
    # now get graph markers
    set i 0
    foreach m [$graph_name marker names] {
	if {[$graph_name marker type $m] == "line"} {
	    if {$i == 0} {
		append headers "marker-x1${sep}marker-x2${sep}marker-y1${sep}marker-y2${sep}"
		set i 1
		foreach v {x1 y1 x2 y2} {
		    lappend variablelist data(${i}_marker$v)
		    set data(${i}_marker$v) {}
		}
	    }
	    set coords [$graph_name marker cget $m -coords]
	    if {[lindex $coords 0] < $xmin || [lindex $coords 0] > $xmax} \
		    continue 
	    regsub -all -- "\\+Inf" $coords $ymax coords
	    regsub -all -- "-Inf" $coords $ymin coords
	    while {[llength $coords] >= 4} {
		foreach c [lrange $coords 0 3] v {x1 y1 x2 y2} {
		    lappend data(${i}_marker$v) $c
		}
		set coords [lrange $coords 2 end]
	    }
	} elseif {[$graph_name marker type $m] == "text"} {
	    # at least for now, ignore text markers
	}
    }    
    puts $fp $headers
    set max 0
    foreach var $variablelist {
	set l [llength [set $var]]
	if {$max < $l} {set max $l}
    }
    for {set i 0} {$i < $max} {incr i} {
	set line {}
	foreach var $variablelist {
	    append line [lindex [set $var] $i] $sep
	}
	puts $fp $line
    }
    return
}
#-------------------------------------------------------------------------
# rework later
#-------------------------------------------------------------------------

proc definecolors {} {
    global graph 
    # test if colors are defined on the monitor -- I wonder why I did this, did I have a B&W monitor once?
    if {[regexp .*color [winfo visual .]]} {
	set graph(linelist) "none"
    } { 
	#set linelist "square circle diamond plus cross splus scross"
	set graph(linelist) "diamond splus scross"
	set graph(colorlist) "black"
    }
    set graph(ncolors) [llength $graph(colorlist)]
    set graph(nlines) [llength $graph(linelist)]
    set graph(colorindex) -1
    set graph(lineindex) -1
}
