namespace eval Vlistbox {
    # ------------------------
    # widget init
    # ------------------------
    proc Vlistbox {rootwin} {
	variable Vlbox
	set ns [namespace current]
	if {$rootwin == "."} {
	    set root ""
	} else {
	    set root $rootwin
	}
	set Vlbox(selection) {}                ; # clear previous selections

	set canvas_width    [expr {$Vlbox(visible_width) * $Vlbox(char_width)}]
	set scroll_width    [expr {$Vlbox(scroll_width)  * $Vlbox(char_width)}]
	set canvas_height   [expr {$Vlbox(visible_height) * $Vlbox(char_height)}]
	canvas $root.c -width $canvas_width -height $canvas_height \
		-bg white
	if {$Vlbox(visible_width) < $Vlbox(scroll_width)} {
	    $root.c configure  -xscrollcommand "$root.x set" \
		    -scrollregion "0 0 $scroll_width $canvas_height"
	    set Vlbox(xscroll) [scrollbar $root.x -orient horizontal \
		    -command "$root.c xview"]
	}
	set Vlbox(canvas) $root.c
	set y 4
	set width [expr {$canvas_width + 4}]
	for {set i 0} {$i < $Vlbox(visible_height)} {incr i} {
	    $root.c create text 6 $y -tags text$i -anchor nw -font $Vlbox(font)
	    $root.c create rectangle 4 $y $width [incr y $Vlbox(char_height)] \
		    -tags rect$i -outline ""
	    $root.c raise text$i
	}

	# ------------------------
	# interface
	# ------------------------

	#label $root.l -text "enter an item number"
	#entry $root.e -textvariable ${ns}::Vlbox(list_start)
	set ${ns}::Vlbox(list_start) 0
	#bind $root.e <KeyPress-Return> ${ns}::display
	#button $root.b -text display -command ${ns}::display
	
	# ------------------------
	# start
	# ------------------------
	
	grid $root.c -column 1 -row 1 -sticky news
	grid columnconfig $rootwin 1 -weight 1
	#focus -force $root.e

	# scrolling virtual items
	# ------------------------
	# scroll bar init
	# ------------------------
	
	set Vlbox(yscroll) [scrollbar $root.s -command ${ns}::yview]
	grid $root.s -column 2 -row 1 -sticky ns
	if {$Vlbox(visible_width) < $Vlbox(scroll_width)} {
	    grid $root.x -column 1 -row 2 -sticky ew
	}

	# ------------------------
	# button command
	# ------------------------

	#$root.b config -command " ${ns}::yset; ${ns}::display "

	# ------------------------
	# set bindings
	# ------------------------
	bind $root.c <1> " ${ns}::select %y "
	bind $root.c <Double-1> " ${ns}::select %y double"
	bind $rootwin <Key-Home> "${ns}::yview moveto 0.0"
	bind $rootwin <Key-End>  "${ns}::yview moveto 1.0"
	bind $rootwin <Key-Next> "${ns}::yview scroll 1 page"
	bind $rootwin <Key-Prior> "${ns}::yview scroll -1 page"
	bind $rootwin <Key-Up> "${ns}::yview scroll -1"
	bind $rootwin <Key-Down> "${ns}::yview scroll 1"
	# shift-page is 5%
	set 5p [expr int($Vlbox(num_items)/20)]
	if {$5p < 1} {set 5p 1}
	bind $rootwin <Shift-Key-Next>  "${ns}::yview scroll $5p"
	bind $rootwin <Shift-Key-Prior> "${ns}::yview scroll -$5p"

	display
	yset
    }


    # ------------------------
    # mass-configure items
    # ------------------------

    # parm1: first item to configure
    # parm2: last item to configure (optional)
    # parm3: modified options from last config
    proc config {first last args} {
	variable Vlbox
	# get args
	if {[string index $last 0] == "-"} { 
	    set args [linsert $args 0 $last]; set last "" 
	}
	if {$last == ""} { set last $first }
	# modify & register config
	set CID [llength $Vlbox(configs)]; incr CID -1
	if {$args != ""} {
	    set config [lindex $Vlbox(configs) $CID]
	    set changed 0
	    foreach {background foreground selbackground selforeground} $config break
	    foreach {key value} $args {
		switch -glob -- $key {
		    -bg       -
		    -bac*     { set background $value; set changed 1 }
		    -fg       -
		    -for*     { set foreground $value; set changed 1 }
		    -selb*    -
		    -selectb* { set selbackground $value; set changed 1 }
		    -self*    -
		    -selectf* { set selforeground $value; set changed 1 }
		}
	    }
	    if {$changed} {
		# create a new config
		set config [list $background $foreground $selbackground $selforeground]
		set CID [newconfig $config]
	    }
	}
	# associate items range and config ID
	newrange $first $last $CID
    }

    # ------------------------
    # retrieve a config
    # ------------------------

    # parm1: item ID
    # return: registered config
    proc retrieve {item} {
	variable Vlbox
	set CID 0
	foreach range $Vlbox(ranges) {
	    foreach {start newID} $range break
	    if {$start > $item} { break }
	    set CID $newID
	}
	return [lindex $Vlbox(configs) $CID]
    }


    # selecting items


    # ------------------------
    # select proc
    # ------------------------
    # parm1: y list box coordinate
    proc select {y "mode {}"} {
	variable Vlbox
	set ns [namespace current]
	# get item
	set item [expr {$Vlbox(list_start) + $y / $Vlbox(char_height)}]
	if {$mode == "double"} {
	    set Vlbox(doubleselection) $item
	    return
	}
	# toggle select state
	set n [lsearch $Vlbox(selection) $item]
	if {$Vlbox(selectionmode) == "single"} {
	    if {$n == -1} { 
		set Vlbox(selection) $item
	    } else {
		set Vlbox(selection) {}
	    }
	} elseif {$n == -1} { 
	    lappend Vlbox(selection) $item 
	} else { 
	    set Vlbox(selection) [lreplace $Vlbox(selection) $n $n] 
	}
	# display
	display
    }

    # ------------------------
    # display visible items
    # ------------------------
    proc display {} {
	variable Vlbox
	set ns [namespace current]
	# get first item
	set item $Vlbox(list_start)
	# display all visible items
	for {set i 0} {$i < $Vlbox(visible_height)} {incr i} {
	    # retrieve config
	    set config [retrieve $item]
	    foreach {background foreground selbackground selforeground} $config break
	    # set colors
	    if {[lsearch $Vlbox(selection) $item] != -1} {
		set background $selbackground
		set foreground $selforeground
	    }
	    # map item
	    if {$item >= $Vlbox(num_items)} {
		$Vlbox(canvas) itemconf text$i -text {} -fill $foreground
	    } else {
		$Vlbox(canvas) itemconf text$i \
			-text [$Vlbox(generateitem) $item] -fill $foreground
	    }
	    $Vlbox(canvas) itemconf rect$i -fill $background
	    incr item
	}
    }


    # ------------------------
    # scroll procs
    # ------------------------

    # setting the scroll bar
    proc yset {} {
	variable Vlbox
	set first $Vlbox(list_start)
	set last [expr {$first + $Vlbox(visible_height)}]
	$Vlbox(yscroll) set [expr {double($first) / $Vlbox(num_items)}] \
		[expr {double($last) / $Vlbox(num_items)}]
    }

    # setting the visible area
    # parm1: scroll command (moveto or scroll)
    # parm2: scroll command args
    proc yview {cmd args} {
	variable Vlbox
	set ns [namespace current]
	switch -- $cmd {
	    moveto {
		# absolute movement
		set first [expr {int($args * $Vlbox(num_items))}]
	    }
	    scroll {
		# relative movement
		set count [lindex $args 0]
		set units [lindex $args 1]
		if {[string match p* $units]} {
		    # paging
		    set count [expr {$count * $Vlbox(visible_height)}]
		}
		set first $Vlbox(list_start)
		incr first $count
	    } else {
		return
	    }
	}
	# setting the scroll bar & displaying the visible area
	if {$first < 0} { set first 0 }
	if {$first > $Vlbox(num_items) - $Vlbox(visible_height)} { 
	    set first [expr {$Vlbox(num_items) - $Vlbox(visible_height)}] 
	}
	set Vlbox(list_start) $first
	yset
	display
    }



    # configuring virtual items

    # ------------------------
    # new config
    # ------------------------
    # parm1: config to register
    # return: config ID
    proc newconfig {config} {
	variable Vlbox
	set ID [llength $Vlbox(configs)]
	lappend Vlbox(configs) $config
	return $ID
    }

    # ------------------------
    # new range
    # ------------------------

    # parm1: first item
    # parm2: last item
    # parm3: config ID
    proc newrange {first last CID} {
	variable Vlbox
	if {$first == [lindex [lindex $Vlbox(ranges) end] 0]} {
	    # add a new range
	    set Vlbox(ranges) [lreplace $Vlbox(ranges) end end [list $first $CID] [incr last]]
	} else {
	    # replace a range
	    set step 1
	    foreach range $Vlbox(ranges) {
		foreach {start newID} $range break
		switch $step {
		    1 {
			# before first
			if {$start < $first} {
			    lappend newVlbox(ranges) $range
			    if {$newID != ""} { set curID $newID }
			} else {
			    lappend newranges [list $first $CID]
			    if {$newID != ""} { set curID $newID }
			    set step 2
			}
		    }
		    2 {
			# between first & last
			if {$start > $last} {
			    lappend newranges [list [incr last] $curID]
			    if {$newID != ""} { set curID $newID }
			    set step 3
			} elseif {$newID != ""} { set curID $newID }
		    }
		    3 {
			# after last
			lappend newranges $range
		    }
		}
	    }
	    if {$step == 2} { lappend newranges [list [incr last] $curID] }
	    lappend newranges $Vlbox(num_items)
	    set Vlbox(ranges) $newranges
	}
    }


    # ------------------------
    # parameters
    # ------------------------

    set Vlbox(scroll_width)    20        ; # scrolled area width, in chars
    set Vlbox(visible_width)   20        ; # visible area width, in chars
    set Vlbox(visible_height)  20        ; # visible area height, in rows
    set Vlbox(char_width)      8         ; # char width, in pixels
    set Vlbox(char_height)     16        ; # char height, in pixels
    set Vlbox(num_items)       {}      ; # items count
    set Vlbox(selectionmode)  "multiple" ; # selection mode
    set Vlbox(font)            "Helvetica -12" ; # default font
    #set Vlbox(selectionmode)  "single"
    set Vlbox(generateitem)  puts

    # ------------------------
    # items init
    # ------------------------
    set defconf {"" black navy white}      ; # default config
    set Vlbox(configs) [list $defconf]     ; # registered configs
    set Vlbox(ranges) {{0 0}}              ; # config ID associated with each item
    set Vlbox(selection) {}                ; # selected items list

}

# ------------------------
# Sample code to create, 
# configure & display
# a virtual listbox
# ------------------------

#set Vlistbox::Vlbox(visible_width)   100        ; # width in chars
#set Vlistbox::Vlbox(visible_height)  10         ;  # height in rows
#set Vlistbox::Vlbox(selectionmode)  "single"
#set Vlistbox::Vlbox(num_items)       100000     ; # of items

# ------------------------
# sample routine to 
# generate a display line
# ------------------------
#proc generateitem {item} {
#    return "this is item #$item"
#}
#set Vlistbox::Vlbox(generateitem)  generateitem

# Make the listbox
#Vlistbox::Vlistbox .

# set global background
#Vlistbox::config 0 [expr $Vlbox(num_items) - 1]  -bg beige
# now redisplay
#Vlistbox::display
