# initialize the data needed for the periodic table & special symbols
array set PeriodicTable {
    row0	"1A 2A 3B 4B 5B 6B 7B    8B    1B 2B 3A 4A 5A 6A 7A 8A"
    row1	"H                                                  He"
    row2	"Li Be                               B  C  N  O  F  Ne"
    row3	"Na Mg                               Al Si P  S  Cl Ar"
    row4	"K  Ca Sc Ti V  Cr Mn Fe Co Ni Cu Zn Ga Ge As Se Br Kr"
    row5	"Rb Sr Y  Zr Nb Mo Tc Ru Rh Pd Ag Cd In Sn Sb Te I  Xe"
    row6	"Cs Ba La Hf Ta W  Re Os Ir Pt Au Hg Tl Pb Bi Po At Rn"
    row7	"Fr Ra Ac                                             "
    row8	"         Ce Pr Nd Pm Sm Eu Gd Tb Dy Ho Er Tm Yb Lu   "
    row9	"         Th Pa U  Np Pu Am Cm Bk Cf Es Fm Md No Lr   "
    group_1A	{Li Na K Rb Cs Fr}
    group_2A	{Be Mg Ca Sr Ba Ra}
    group_3A	{B Al Ga In Tl}
    group_4A	{C Si Ge Sn Pb}
    group_5A	{N P As Sb Bi}
    group_6A	{O S Se Te Po}
    group_7A	{F Cl Br I At}
    group_8A	{He Ne Ar Kr Xe Rn}
    group_1B	{Cu Ag Au}
    group_2B	{Zn Cd Hg}
    group_3B	{Sc Y La Ac}
    group_4B	{Ti Zr Hf}
    group_5B	{V Nb Ta}
    group_6B	{Cr Mo W}
    group_7B	{Mn Tc Re}
    group_8B	{Fe Co Ni Ru Rh Pd Os Ir Pt}
    group_LN	{La Ce Pr Nd Pm Sm Eu Gd Tb Dy Ho Er Tm Yb Lu}
    group_AD	{Ac Th Pa U  Np Pu Am Cm Bk Cf Es Fm Md No Lr}
    1A	1
    2A	2
    3A	3
    4A	4
    5A	5
    6A	6
    7A	7
    8A	8
    1B	9
    2B	10
    3B	11
    4B	12
    5B	13
    6B	14
    7B	15
    8B	16
    LN	17
    AD	18
    elements	{}
}
# generate a list of elements
set cribsheet(Ce) 58; set cribsheet(Th) 90; set cribsheet(Hf) 72
set A 1
for {set i 1} { $i<=9 } { incr i 1} {
    for {set j 0} { $j<18 } { incr j 1} {
	set elem [string trim [string range $PeriodicTable(row$i) \
		[expr $j*3] [expr $j*3+1]]]
	if {$elem != ""} {
	    lappend PeriodicTable(elements) $elem
	    if {![catch "set cribsheet($elem)"]} { set A $cribsheet($elem) }
	    set PeriodicTable([string tolower $elem]) $A
	    set PeriodicTable($A) $elem
	    incr A
	}
    }
}
# generate a list of special symbols
set PeriodicTable(grouplist) {}
for {set j 0} { $j<18 } { incr j 1} {
    set elem [string trim \
	    [string range $PeriodicTable(row0) [expr $j*3] [expr $j*3+1]] ]
    if {$elem != ""} {
	lappend PeriodicTable(grouplist) $elem
	set A [expr 103 + $PeriodicTable([string toupper $elem]) ]
	set PeriodicTable($A) $elem
    }
}
lappend PeriodicTable(grouplist) LN
lappend PeriodicTable(grouplist) AD

# parse a list of elements and return their atomic numbers
# for an invalid token, a list containing a 0 and a error messsage is returned
# examples: 
#     parselem "li,AL si"    returns  {3 13 14}
#     parselem "li,AL si O2" returns  { 0 {invalid element 'O2'} }
#
proc parselem {listin} {
    variable PeriodicTable
    # break up the listin into individual tokens and check them
    set listout {}
    foreach word [split $listin " ,"] {
	if {$word == {}} continue
	# do upcase match
	if {![catch "set PeriodicTable([string tolower $word])" A]} {
	    lappend listout $A
	} elseif {![catch "set PeriodicTable([string toupper $word])" A]} {
	    lappend listout [expr 103 + $A]
	} else {
	    return "invalid element '$word'"
	}
    }
    return  $listout
}

proc reportSelected {tbl elemlist} {
    variable PeriodicTable
    variable $elemlist
    set selected {}
    foreach elem $PeriodicTable(elements) {
	if {$PeriodicTable(set$tbl.$elem) && \
		!$PeriodicTable(disable$tbl.$elem)} {
	    set selected "$selected $elem"
	}
    }
    foreach elem $PeriodicTable(grouplist) {
	if {$PeriodicTable(set$tbl.$elem)} {
	    set selected "$selected $elem"
	}
    }
    set $elemlist $selected
}

proc clearelemarray {tbl} {
    variable PeriodicTable
    foreach elem $PeriodicTable(elements) {
	set PeriodicTable(set$tbl.$elem) 0
	set PeriodicTable(disable$tbl.$elem) 0
    }
    # special elements
    foreach elem $PeriodicTable(grouplist) {
	set PeriodicTable(set$tbl.$elem) 0
    }
}

proc ShowElemButton {elem tbl} {
    variable PeriodicTable
    # regular element
    if {$PeriodicTable(set$tbl.$elem) != 0} {
	$PeriodicTable(cell$tbl.$elem) config \
		-background black -fg white \
		-activeforeground white -activebackground #666666
    } else {
	$PeriodicTable(cell$tbl.$elem) config \
		-background #d9d9d9 -fg black \
		-activeforeground black -activebackground #ececec
    }
}

proc ToggleElemButton {elem tbl elemvar} {
    variable PeriodicTable
    set PeriodicTable(set$tbl.$elem) [expr ! $PeriodicTable(set$tbl.$elem)]
    ShowElemButton $elem $tbl
    reportSelected $tbl $elemvar
}

proc ShowGroupButton {elem tbl} {
    variable PeriodicTable
    # special element button
    if {$PeriodicTable(set$tbl.$elem) != 0} {
	$PeriodicTable(cell$tbl.$elem) config -relief sunken \
		-background black -fg pink \
		-activeforeground red -activebackground #666666 
    } else {
	$PeriodicTable(cell$tbl.$elem) config -relief sunken \
		-background #d9d9d9 -fg red \
		-activeforeground darkred -activebackground #ececec
    }
}

proc ToggleGroupButton {elem tbl elemvar} {
    variable PeriodicTable 
    variable $elemvar
    set PeriodicTable(set$tbl.$elem) [expr ! $PeriodicTable(set$tbl.$elem)]
    ShowGroupButton $elem $tbl
    if {$PeriodicTable(set$tbl.$elem) != 0} {
	set A 1
    } else {
	set A 0
    }
    foreach subelem $PeriodicTable(group_$elem) {
	set PeriodicTable(disable$tbl.$subelem) $A
	if {$PeriodicTable(disable$tbl.$subelem)} {
	    $PeriodicTable(cell$tbl.$subelem) config -state disabled \
		    -background #666666
	} else {
	    $PeriodicTable(cell$tbl.$subelem) config -state normal
	    ShowElemButton $subelem $tbl
	}
    }
    reportSelected $tbl $elemvar
}

proc makePeriodicTable {tbl elemvar label} {
    variable PeriodicTable

    set buttonsize "-width 1 -borderwidth 1 -padx 6 -pady 2" 
    catch "destroy $tbl"
    toplevel $tbl
    wm title $tbl $label
    #    frame $tbl -borderwidth 2 -relief raised
    #    pack $tbl -side top 
    #    pack [label $tbl.top -text $label ] -side top

    clearelemarray $tbl

    # save a blank frame at the top for element groups
    pack [frame $tbl.row0 ] -side top -fill x

    # main periodic table
    for {set i 1} { $i<=9 } { incr i 1} {
	pack [frame $tbl.row$i ] -side top -fill x
	for {set j 0} { $j<18 } { incr j 1} {
	    set elem [string trim [string range $PeriodicTable(row$i) \
		    [expr $j*3] [expr $j*3+1]]]
	    set ec $tbl.row$i.f$j
	    if {$elem != ""} {
		set PeriodicTable(cell$tbl.$elem) $ec.b
		frame  $ec   -relief sunken -borderwidth 1
		set ns [namespace current]
		pack  [button $ec.b -text $elem \
			-command "${ns}::ToggleElemButton $elem $tbl $elemvar"]
	    } else {
		pack [frame  $ec -borderwidth 1 -relief flat ] -fill x 
		pack [button $ec.b -relief flat -state disabled] 
	    }
	    eval $ec.b config $buttonsize
	    pack $ec -side left -fill y
	}
    }
    
    # now do the top row
    for {set j 0} { $j<18 } { incr j 1} {
	set elem [string trim [string range $PeriodicTable(row0) \
		[expr $j*3] [expr $j*3+1]]]
	set ec $tbl.row0.f$j
	if {$elem != ""} {
	    set PeriodicTable(cell$tbl.$elem) $ec.b
	    frame  $ec   -relief sunken -borderwidth 1
	    set ns [namespace current]
	    pack  [button $ec.b -text $elem -relief sunken \
		    -command "${ns}::ToggleGroupButton $elem $tbl $elemvar" \
		    -fg red -activeforeground darkred]
	} {
	    pack [frame  $ec -borderwidth 1 -relief flat ] -fill x 
	    pack  [button $ec.b -relief flat -state disabled] 
	}
	eval $ec.b config $buttonsize
	pack $ec -side left -fill y
    }
    # and reset the special elements
    $tbl.row8.f17 config -relief sunken
    set PeriodicTable(cell$tbl.LN) $tbl.row8.f17.b 
    set ns [namespace current]
    $tbl.row8.f17.b config -text LN -state normal \
	    -command "${ns}::ToggleGroupButton LN $tbl $elemvar" \
	    -fg red -activeforeground darkred -relief sunken
    $tbl.row9.f17 config -relief sunken
    set PeriodicTable(cell$tbl.AD) $tbl.row9.f17.b 
    $tbl.row9.f17.b config -text AD -state normal \
	    -command "${ns}::ToggleGroupButton AD $tbl $elemvar" \
	    -fg red -activeforeground darkred -relief sunken
    $tbl.row0.f7.b config -text "<--" -disabledforeground red
    $tbl.row0.f9.b config -text "-->" -disabledforeground red
    # bottom row
    pack [frame $tbl.bottom ] -side top -fill x
    pack [button $tbl.bottom.d -text Close \
	    -command "destroy $tbl"] -side right
    # set the buttons from the main menu string
    setarray $tbl $elemvar
}

proc setarray {tbl elemlist} {
    variable PeriodicTable 
    variable $elemlist
    set var [set $elemlist]
    if {$var == ""} return
    # loop for elements
    foreach word $var {
	if {![catch "set PeriodicTable([string tolower $word])" A]} {
	    set elem $PeriodicTable($A)
	    set PeriodicTable(set$tbl.$elem) 1
	    ShowElemButton $elem $tbl
	} 
    }
    # loop for groups
    foreach word $var {
	if {![catch "set PeriodicTable([string toupper $word])" A]} { 
	    set A [expr $A + 103]
	    set elem $PeriodicTable($A)
	    set PeriodicTable(set$tbl.$elem) 1
	    ShowGroupButton $elem $tbl
	    foreach subelem $PeriodicTable(group_$elem) {
		set PeriodicTable(disable$tbl.$subelem) 1
		if {$PeriodicTable(disable$tbl.$subelem)} {
		    $PeriodicTable(cell$tbl.$subelem) config -state disabled \
			    -background #666666
		}
	    }
	}
    }
    reportSelected $tbl $elemlist
}
