#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "logiclib.h"

/*  Program findpeak

    This program locates the entries that have one or more peaks 
    a specified d-space range. The resulting entries are applied to 
    the specified bitmap using the selected boolean operation.

    Warnings are written to standard out, if any occur.

  Command syntax:
        findpeak <file> 
	         -peak "<pos1>, <pos2> ..." -error "<err1> <err2> ..." 
		 [-wave <lambda> | -q ]
	         [-any | -all] 
		 [-fast | -exact] 
		 [ -or | -and | -remove | -not ]
		 [-intmin <int> | -3strongest]
    where 
       -peak            precedes a list of peak positions as d-spaces (unless 
                        -wave or -q is used)
       -error           specifies the minimum error in the peak positions
       -wave <lambda>   positions are 2theta angles instead of d-spaces
       -q               positions are in Q (2 pi/d-space) instead of d-spaces
       -any             any one of the positions must be matched
       -all             all of the positions must be matched
       -intmin <int>    peaks must be above a threshold of <int>% 
       -3strongest      peaks must be one of the three strongest in the pattern
       -fast            expand the errors or lower "intmin" to speed the search
       -exact           use the limits exactly as specified
       -or -and -remove -not
                        logical options
*/
int main (int argc, char *argv[]) {
  int error;			/* error flag */
  int j, jpeak, jerr; 
  INTEGER4 allflag, intflag, fastflag, ilogic, numpeaks, retval;
  int k;
  REAL4 wave;
  REAL4 *peaklist = NULL;
  REAL4 *errlist = NULL;

  if (PSL_INIT() != 0) {
    fprintf(stderr,"error initializing the ICDD database\n");
    return 1;
  }

  error = 0;
  numpeaks = allflag = intflag = fastflag = ilogic = -1;
  wave = 0.0;
  jerr = jpeak = -1;
  j = 2;
  while (j < argc) {
    if (!strcmp(argv[j],"-peak")) {
      j++;
      jpeak = j;
    } else if (j+1 < argc && !strcmp(argv[j],"-wave")) {
      j++;
      if (wave != 0) {
	fprintf(stderr,
		"error: -wave or -q repeated or in conflict\n");
	error=1; /* flag error if there is a conflict */
      } else {
	wave = strtod(argv[j], NULL);
	if (wave <= 0) {
	  fprintf(stderr,
		  "error: wavelength of %f is illegal\n", wave);
	  error=1; /* flag error if there is a conflict */
	}
      }
    } else if (!strcmp(argv[j],"-q")) {
	if (wave != 0.0) {
	  fprintf(stderr,
		  "error: -wave or -q repeated or in conflict\n");
	  error = 1; /* flag error if there is a conflict */
	}
	wave = -1.;
    } else if (j+1 < argc && !strcmp(argv[j],"-error")) {
      jerr = ++j;  /* keep track of the error position */
    } else if (!strcmp(argv[j],"-any") || !strcmp(argv[j],"-all")) {
      if (allflag != -1) {
	fprintf(stderr,
		"error: -any or -all repeated or in conflict\n");
	error=1; /* flag error if there is a conflict */
      }
      if (!strcmp(argv[j],"-any"))
	allflag = 1;
      else
	allflag = 0;
    } else if (j+1 < argc && !strcmp(argv[j],"-intmin")) {
      if (intflag != -1) {
	fprintf(stderr,
		"error: -intmin or -3strongest repeated or in conflict\n");
	error=1; /* flag error if there is a conflict */
      }	
      j++;
      k = strtol(argv[j], NULL, 10);
      intflag = (k>100) ? 99 : k; /* limit the intensity to 99 */
      intflag = (k<  0) ?  0 : k; /* limit the intensity to  0 */
    } else if (!strcmp(argv[j],"-3strongest")) {
      if (intflag != -1) {
	fprintf(stderr,
		"error: -intmin or -3strongest repeated or in conflict\n");
	error=1; /* flag error if there is a conflict */
      }
      intflag = 100;
    } else if (!strcmp(argv[j],"-fast") || !strcmp(argv[j],"-exact")) {
      if (fastflag != -1) {
	fprintf(stderr,
		"error: -fast or -exact repeated or in conflict\n");
	error=1; /* flag error if there is a conflict */
      }
      if (!strcmp(argv[j],"-fast")) 
	fastflag = 0;
      else
	fastflag = 1;
    } else if (!strcmp(argv[j],"-or") || !strcmp(argv[j],"-and") ||
	       !strcmp(argv[j],"-remove") || !strcmp(argv[j],"-not")) {
      if (ilogic != -1) {
	fprintf(stderr,
		"option -or -and -remove or -not repeated\n");
	error = 1;
      }
      if (!strcmp(argv[j],"-and"))
	ilogic = 0;
      else if (!strcmp(argv[j],"-or"))
	ilogic = 1;
      else if (!strcmp(argv[j],"-remove"))
	ilogic = 2;
      else if (!strcmp(argv[j],"-not"))
	ilogic = 3;
    } else {
      fprintf(stderr,"unknown option: %s\n",argv[j]);
      error = 1;
    }
    j++;
  }
  /* check that we have values for everything */
  if (allflag == -1 || intflag == -1 || fastflag == -1 || ilogic == -1 ||
      jpeak == -1 || jerr == -1) {
    fprintf(stderr,"error: required option missing\n");
    error = 1;
  }
  /* now process the peaks and errors */
  if (!error) {
    int listlen1, listlen2;
    REAL4 val1, val2;
    listlen1 = ParseRealList(argv[jpeak], &peaklist);
    listlen2 = ParseRealList(argv[jerr], &errlist);
    numpeaks = 0;
    if (wave == 0) {
      for (j=0; j< listlen1; j++) {
	if (j < listlen2) val2 = errlist[j];
	errlist[j] = val2;
      }
      numpeaks = listlen1;
    } else {
      INTEGER4 i = 0;
      for (j=0; j< listlen1; j++) {
	REAL4 pos,pos1,pos2;
	pos = val1 = peaklist[j];
	if (j < listlen2) val2 = errlist[j];
	peaklist[numpeaks] = PSL_PEAKTOD(&pos,&wave,&i);
	if (peaklist[numpeaks] > 0) {
	  pos = val1 + val2;
	  pos1 = PSL_PEAKTOD(&pos,&wave,&i);
	  pos = val1 - val2;
	  pos2 = PSL_PEAKTOD(&pos,&wave,&i);
	  if (pos1 > 0 && pos2 > 0) 
	    errlist[numpeaks] = 0.5 * fabs (pos2 - pos1); 
	  else if (pos1 > 0)
	    errlist[numpeaks] = 0.5 * fabs (pos1 - peaklist[numpeaks]);
	  else
	    errlist[numpeaks] = 0.5 * fabs (pos2 - peaklist[numpeaks]);
	  numpeaks++;
	} else {
	  printf("Ignoring out of range peak: %f\n",pos);
	}
      }
    }
  }
  
  if (!error) {
    /*  Load the starting file */
    INTEGER4 il=4;    /* Replace bitmap */
    error = PSL_READBITMAP(argv[1],&il);
    if (error) {      
      fprintf(stderr, "error reading subfile %s\n",argv[1]);
      return 1;
    }
    retval = PSL_PEAKSUB(&numpeaks, peaklist, errlist, 
			 &allflag, &intflag, &fastflag, &ilogic, &wave); 
/*         for (j=0;j<numpeaks; j++)  */
/*   	printf("%f %f ",peaklist[j], errlist[j]);  */
/*       printf( */
/*   "\nnumpeaks=%ld, allflag=%ld, intflag=%ld, fastflag=%ld, ilogic=%ld\n", */
/*        numpeaks, allflag, intflag, fastflag, ilogic); */
  } else {
    retval = 0;
  }

  if (peaklist != NULL) free(peaklist);
  if (errlist != NULL) free (errlist);

  if (retval < 0) {
    fprintf(stderr, "Error in peak input\n");
    return 1;
  }
  if (error) {
    fprintf(stderr, "\nusage:\n");
    fprintf(stderr,"\t findpeak <file>");
    fprintf(stderr,"\t\t-peak \"<pos1> <pos2> ...\" -error \"<err1> <err2> ...\"\n");
    fprintf(stderr,"\t\t [-wave <lambda> | -q ]\n");
    fprintf(stderr,"\t\t [-any | -all]\n");
    fprintf(stderr,"\t\t [-fast | -exact]\n");
    fprintf(stderr,"\t\t [-intmin <int> | -3strongest]\n");
    fprintf(stderr,"\t\t [ -or | -and | -remove | -not ]\n");
    return 1;
  }

  error = PSL_SAVEBITMAP(argv[1]);
  if (error) {      
    fprintf(stderr, "error writing subfile %s\n",argv[1]);
  }/*   else { */
/*      INTEGER4 i = 0; */
/*      printf("%d\n",PSL_REPHITS(&i) ); */
/*    } */

  return 0;
}
