#-----------------------------------------------------------------
# read BT1 multidetector data from a .raw (proprep output) file
# or process through proprep if a .bt1 file
#------- define a command line option -----------------------------
# command line option 
lappend command(cmdopt) -MD -md
# proc to use 
set command(-MD) readmultidata 
set command(-md) readmultidata
#--------define a dialog box entry
# menu label
lappend command(readtypes) "BT-1 multi-detector"
# proc for this entry
lappend command(readproc) ReadBT1
# allowed data types
if {$tcl_platform(platform) != "windows"} {
    lappend command(filterlist) {bt1 raw bt1gz}
} else {
    lappend command(filterlist) {bt1 raw bt1}
}
# definitions for these data types
set command(ReadBT1_bt1_type) "BT-1 as-collected"
set command(ReadBT1_bt1gz_type) "BT-1 as-collected compressed"
set command(ReadBT1_raw_type) "BT-1 proprep"

set bt1(firstmon) 0

proc ReadBT1 {file} {
    global command
    if {$file == ""} return
    pleasewait "reading file $file"
    set ret [readmultidata $file]
    donewait
    if {$ret != ""} {return $ret}
    showlastentry $command(read_filelist)
}

#--------------------------------------------------------------------------
# load multi detector data in BT1 .bt1 or .raw format
proc readmultidata {filename "zerocorr 1" "multcorr 1"} {
#   zerocorr != 0 ==> apply detector zero correction
#   multcorr != 0 ==> apply detector sensitivity correction
#
    global command
    # signal errors by quitting
    if [catch {
	# is this a bt1 file?
	set ext [file extension $filename]
	if {$ext == ".bt1"} {
	    if {$command(debug)} {puts "opening BT1 file $filename"}
	    set fl [open $filename r]
	    set ret [readbt1multidata $filename $fl $zerocorr $multcorr]
	} elseif {$ext == ".bt1gz"} {
	    if {$command(debug)} {puts "opening compressed BT1 file $filename"}
	    set fl [open "| gunzip < $filename" r]
	    set ret [readbt1multidata $filename $fl $zerocorr $multcorr]
	} elseif {$ext == ".raw"} {
	    if {$command(debug)} {puts "opening raw file $filename"}
	    set ret [readrawmultidata $filename $zerocorr $multcorr]
	} else {
	    return "file $filename has an invalid extension for BT1 data"
	}
    } errmsg] {
	return $errmsg
    }
    if {$ret != ""} {return $ret}
}

# load multi detector data in BT1 .raw (proprep) format
proc readrawmultidata {filename "zerocorr 1" "multcorr 1"} {
#   zerocorr != 0 ==> apply detector zero correction
#   multcorr != 0 ==> apply detector sensitivity correction
#
    global bt1 graph
    global command

    if [catch {

	set lnum 0
	set fp [open $filename r]
	#skip first 2 lines
	incr lnum
	gets $fp line
	set title [string trim [string range $line 0 50]]
	set prefact [string range $line 70 end]
	if {[llength $prefact] > 1} {
	    return "Invalid prefactor. Is this a BT1 .raw file?"
	}
	if [catch {set bt1(firstmon)}] {set bt1(firstmon) 0}
	if {$bt1(firstmon) == 0} {
	    set bt1(firstmon) $prefact
	}
	if {$prefact == 0.0 || $bt1(firstmon) == 0.0 || $prefact == "" } {
	    set norm 1
	} else {
	    set norm [expr  $bt1(firstmon) / (1.* $prefact) ]
	}
	if {$command(debug)} {puts "normalizing by a factor of $norm"}
	incr lnum
	gets $fp line

	set bank 0
	set ptsread 0
	while {$bank < 32} {
	    incr bank
	    # read bank header
	    incr lnum
	    gets $fp line
	    set start [lindex $line 0]
	    set step [lindex $line 1]
	    set end [lindex $line 2]
	    set scale [lindex $line 3]
	    set offset [lindex $line 4]
	    set npts [lindex $line 6]
	    set ptnum 0
	    set data [file root [file tail $filename]]_$bank 
	    # eliminate spaces from the name
	    regsub -all " " $data "_" data
	    set data [initdata $data]
	    global $data
	    set ${data}(xlabel) "2theta"
	    set ${data}(xunits) "2theta"
	    set ${data}(ylabel) "counts"
	    set zero 0
	    set ${data}(title) "Bank $bank $title"
	    if {$zerocorr} {set zero $offset}
	    set mult 1
	    if {$multcorr} {set mult $scale}
	    #	puts "$bank $mult $scale * $norm"
	    while {$ptnum < $npts} {
		incr lnum
		gets $fp line
		if {$lnum > $::command(maxlines)} {
		    return "Read > $::command(maxlines) lines. Something is wrong or change command(maxlines)!"
		}
		foreach count $line {
		    if {$count > 0} {
			lappend ${data}(x) \
				[expr ($start-$zero + $step*$ptnum)/100.]
			lappend ${data}(y) [expr $mult*$count*$norm]
			lappend ${data}(esd) [expr $mult*sqrt($count)*$norm]
			incr ptsread
		    }
		    incr ptnum
		}
	    }
	    resetdata $data
	    lappend graph(plotlist) $data
	}
    } errmsg] {
	# try to process the last data we have read
	catch {resetdata $data; lappend graph(plotlist) $data}
	return "BT1m raw error. Error reading line $lnum: $errmsg"
    }

    catch {close $fp} test
    if {$command(debug)} {puts "$ptsread non-zero points read"}
    return {}
}

# load multi detector data in BT1 .bt1 (as collected) format
proc readbt1multidata {filename fl "zerocorr 1" "multcorr 1"} {
#   zerocorr != 0 ==> apply detector zero correction
#   multcorr != 0 ==> apply detector sensitivity correction
#
    global bt1 graph
    global command

    set lnum 0

    # signal errors by quitting
    if [catch {
	# get a line and turn single quotes into double
	incr lnum
	gets $fl line
	regsub -all {'} $line {"} line
	set prefact [expr [lindex $line 8] * [lindex $line 9]]
	set npts [lindex $line 11]
	incr lnum
	gets $fl line
	incr lnum
	gets $fl line
	set title [string trim $line]
	incr lnum
	gets $fl line
	regsub -all {'} $line {"} line
	set wave  [lindex $line 7]
	set ndet [lindex $line 11]
	incr lnum
	gets $fl line
	incr lnum
	gets $fl line
	incr lnum
	gets $fl line
	# get the scan range for motor 4
	set start [lindex $line 1]
	set step  [lindex $line 2]
	incr lnum
	gets $fl line
	# get the normalization constants
	incr lnum
	gets $fl line
	set j 0
	for {set i 1} {$i <=$ndet} {incr i} {
	    set norm_array($i) [lindex $line $j]
	    if {[string trim $norm_array($i)] == ""} {
		set j 0
		incr lnum
		gets $fl line
		set norm_array($i) [lindex $line $j]
	    }
	    incr j
	    #	puts "$i $j $norm_array($i)"
	}
	incr lnum
	gets $fl line
	# get the zero corrections
	incr lnum
	gets $fl line
	set j 0
	for {set i 1} {$i <=$ndet} {incr i} {
	    set zero_array($i) [lindex $line $j]
	    if {[string trim $zero_array($i)] == ""} {
		set j 0
		incr lnum
		gets $fl line
		set zero_array($i) [lindex $line $j]
	    }
	    incr j
	    #    puts "$i $j $zero_array($i)"
	}
	incr lnum
	gets $fl line
	incr lnum
	gets $fl line
	for {set i 1} {$i <=$ndet} {incr i} {set pt($i) ""}
	# read through the data add to the pt array
	set i 1
	while {[gets $fl line] >= 0} {
	    incr lnum
	    if {$lnum > $::command(maxlines)} {
		return "Read > $::command(maxlines) lines. Something is wrong or change command(maxlines)!"
	    }
	    if {[string range $line 0 0] != "$"} {
		# change commas to spaces
		regsub -all {,} $line { } line
		set j 0
		# anything left to read?
		while {[lindex $line $j] != ""} {
		    lappend pt($i) [lindex $line $j] 
		    incr i
		    if {$i > $ndet} {set i 1}
		    incr j
		}
	    }
	}
	close $fl
	if {$i != 1} {
	    return "Unable to parse, not enough points for all $ndet detectors"
	}
	if {[llength $pt(1)] != $npts || [llength $pt(32)] != $npts} {
	    return "could not read all $npts data points, [llength $pt(32)] read"
	}
	
	if [catch {set bt1(firstmon)}] {set bt1(firstmon) 0}
	if {$bt1(firstmon) == 0} {
	    set bt1(firstmon) $prefact
	}
	if {$prefact == 0.0 || $bt1(firstmon) == 0.0 || $prefact == ""} {
	    set norm 1
	} else {
	    set norm [expr $bt1(firstmon) / (1.* $prefact) ]
	}
	if {$command(debug)} {puts "normalizing by a factor of $norm"}
	
	set bank 0
	# is this old or new BT1 data?
	if {$ndet == 5} {set bankoffset 20} {set bankoffset 5}
	set ptsread 0
	while {$bank < $ndet} {
	    incr bank
	    set data [file root [file tail $filename]]_$bank 
	    # eliminate spaces from the name
	    regsub -all " " $data "_" data
	    set data [initdata $data]
	    global $data
	    set ${data}(xlabel) "2theta"
	    set ${data}(xunits) "2theta"
	    set ${data}(ylabel) "counts"
	    set ${data}(title) "Bank $bank $title"
	    set zero 0
	    if {$zerocorr} {set zero $zero_array($bank)}
	    set mult $norm
	    if {$multcorr} {set mult [expr $norm_array($bank) * $norm]}
	    #	puts "$bank $mult $norm_array($bank) * $norm"
	    set ptnum 0
	    foreach count $pt($bank) {
		if {$count > 0} {
		    lappend ${data}(x) \
			    [expr $start + ($bank-1.)*$bankoffset - $zero/100.\
			    + $step*$ptnum]
		    lappend ${data}(y) [expr $mult*$count]
		    lappend ${data}(esd) [expr $mult*sqrt($count)]
		    incr ptsread
		}
		incr ptnum
	    }
	    resetdata $data
	    lappend graph(plotlist) $data
	}
    } errmsg] {
	# try to process the last data we have read
	catch {resetdata $data}
	return "BT1m bt1 error. Error reading line $lnum: $errmsg"
    }
    catch {close $fp} test
    if {$command(debug)} {puts "$ptsread non-zero points read"}
    return {}
}
