namespace eval logicGUI {
    # center a new window (w) on window wp (defaults to parent of $w)
    proc centerontop {w "wp {}"} {
	wm withdraw $w
	if {$wp == ""} {set wp [winfo toplevel [winfo parent $w]]}
	update idletasks
	# center the new window in the middle of the parent
	set x [expr [winfo x $wp] + [winfo width $wp]/2 - \
		[winfo reqwidth $w]/2 - [winfo vrootx $wp]]
	if {$x < 0} {set x 0}
	set xborder 10
	if {$x+[winfo reqwidth $w] +$xborder > [winfo screenwidth $w]} {
	    incr x [expr \
		    [winfo screenwidth $w] - \
		    ($x+[winfo reqwidth $w] + $xborder)]
	}
	set y [expr [winfo y $wp] + [winfo height $wp]/2 - \
		[winfo reqheight $w]/2 - [winfo vrooty $wp]]
	if {$y < 0} {set y 0}
	set yborder 25
	if {$y+[winfo reqheight $w] +$yborder > [winfo screenheight $w]} {
	    incr y [expr \
		    [winfo screenheight $w] - \
		    ($y+[winfo reqheight $w] + $yborder)]
	}
	wm geom $w +$x+$y
	wm deiconify $w
    }

    proc warnbox {message} {
	variable settings
	set root $settings(root)
	# kill any window with this name
	catch {destroy $root.warn}
	toplevel .warn
	pack [message $root.warn.1 -width 8c -justify left -bd 2 \
		-text $message]          
	pack [button $root.warn.2 -text OK -command {destroy .warn}] -side bottom
	centerontop $root.warn
    }

    proc waitbox {message} {
	variable settings
	set root $settings(root)
	# kill any window with this name
	catch {destroy $root.wait}
	toplevel $root.wait
	pack [message $root.wait.1 -width 8c -justify left -bd 6 -relief raised \
		-text $message]          
	centerontop $root.wait
    }

    #----------------------------------------------------------------
    #create a commands from the menu results
    #----------------------------------------------------------------
    proc makesubcommand {listbox} {
	variable settings
	variable Logic_option
	set root $settings(root) 
	set logic $Logic_option
	if {$listbox == ""} {
	    set listval 0
	} else {
	    set listval [$listbox curselection]
	}
	if {$Logic_option == "all" || $Logic_option == "none"} {
	    saveundo
	    if {$Logic_option == "all"} {catch {destroy $root.index $root.browse}}
	    loadsub -$Logic_option
	    update_logic
	    return
	}
	if {$listval == ""} return
	saveundo
	if {$Logic_option == "or" || $Logic_option == "replace" \
		|| $Logic_option == "all" } {catch {destroy $root.index $root.browse}}
	loadsub -$Logic_option -subfile [expr 1+$listval]
	update_logic
    }

    proc makecountcommand {} {
	variable countvar
	variable Logic_option
	if {$Logic_option == ""} return
	set countlist {}
	for {set i 0} {$i < 10} {incr i} {
	    if {$countvar($i)} {lappend countlist $i}
	}
	if {$countlist == {}} return
	saveundo
	elemcount -count $countlist -$Logic_option
	update_logic
    }

    proc makeelementcommand {} {
	variable Logic_option
	variable required_opt
	variable optional_opt
	variable req_list
	variable opt_list
	if {$Logic_option == ""} return
	if {$required_opt == ""} return
	if {$optional_opt == ""} return
	set list1 [list [parselem $req_list]]
	saveundo
	if {$optional_opt == "alloptional"} {
	    eval element -$Logic_option -$required_opt $list1 -$optional_opt
	} else {
	    set list2 [list [parselem $opt_list]]
	    eval element -$Logic_option -$required_opt $list1 \
		    -$optional_opt $list2
	}
	update_logic
    }

    proc makepdfnumcommand {top} {
	variable pdfnum
	variable Logic_option
	if {$Logic_option == ""} return
	if {$pdfnum(searchtype) == 1} {
	    set valid {}
	    set invalid {}
	    foreach item {min max} lbl {Start End} {
		set val [string trim $pdfnum($item)]
		if {[llength [set sval [split $val ",-"]]] == 2} {
		    foreach "v1 v2" $sval {}
		    foreach var "v1 v2" {set $var [string trim [set $var]]}
		    if {[catch {set v [expr $v1*10000 + $v2]} errmsg]} {
			append invalid "  The value of \"$val\" is invalid for $lbl\n"
			continue
		    }
		    set val $v
		}
		if {[catch {expr $val}]} {
		    append invalid "  The value of \"$val\" is invalid for $lbl\n"
		    continue
		}
		if {$val <=0 || $val > 999999} {
		    append invalid "  The value of \"$val\" is invalid for $lbl\n"
		    continue
		}
		set $item $val
	    }
	    if {$invalid != ""} {
		tk_dialog $top.msg "Invalid number" \
			"The following entries are invalid:\n $invalid" \
			error 0 Abort
		return
	    }
	    saveundo
	    findnumber -range $min $max -$Logic_option
	} elseif {$pdfnum(searchtype) == 2} {
	    set valid {}
	    set invalid {}
	    for {set i 1} {$i < $pdfnum(nextrow)} {incr i} {
		if {[set val [string trim $pdfnum(entry$i)]] == ""} {continue}
		if {[llength [set sval [split $val ",-"]]] == 2} {
		    foreach "v1 v2" $sval {}
		    foreach var "v1 v2" {set $var [string trim [set $var]]}
		    if {[catch {set v [expr $v1*10000 + $v2]} errmsg]} {
			append invalid "\t" $val "\n"
			continue
		    }
		    set val $v
		}
		if {[catch {expr $val}]} {
		    append invalid "\t" $val "\n"
		    continue
		}
		if {$val <=0 || $val > 999999} {
		    append invalid "\t" $val "\n"
		    continue
		}
		if {$valid != ""} {append valid ","}
		append valid $val
	    }
	    if {$invalid != ""} {
		set ans [tk_dialog $top.msg "Invalid number" \
			"The following entries are invalid:\n $invalid\nSkip these entries or abort the search?" \
			error 0 Skip Abort]
		if {$ans != 0} {return}
	    }
	    saveundo
	    findnumber -list $valid -$Logic_option
	} else {
	    return
	}
	update_logic
    }

    #----------------------------------------------------------------
    # multi-purpose macros
    #----------------------------------------------------------------
    proc options_list {frame {listtype load}} {
	variable Logic_option
	if {$listtype == "load"} {
	    pack [ label $frame.label -text "Combine tables by:" \
		    ] -side top -anchor w
	    set options {"OR file with previous" "AND file with previous" \
		    "REMOVE file from previous" \
		    "REMOVE previous from file" "REPLACE previous with file"}
	    set qual {or and remove not replace}
	    set default replace
	} elseif {$listtype == "search"} {
	    pack [ label $frame.label -text "Combine tables by:" \
		    ] -side top -anchor w
	    set options {"OR search with previous" "AND search with previous" \
		    "REMOVE search from previous" \
		    "REMOVE previous from search"}
	    set qual {or and remove not}
	    set default and
	} else {
	    set options {"set table to all entries" \
		    "clear all entries from table"}
	    set qual {all none}
	    set default all
	}
	set optnum 0
	set ns [namespace current]
	foreach item $options {
	    set name [lindex $qual $optnum]
	    pack [ radiobutton $frame.$name -variable ${ns}::Logic_option \
		    -text $item -value $name ] -side top -anchor w
	    incr optnum
	}
	# set default option -- and except in init manu
	set Logic_option $default
    }

    proc logic_list {frame} {
	# create a listbox
	set listbox \
		[listbox $frame.list -yscrollcommand \
		"$frame.scroll set" -selectmode single]
	# create the scroll bar
	set scroll \
		[scrollbar $frame.scroll \
		-command "$frame.list yview"]
	# add a line for each entry from the sublist command
	foreach item [sublist] { $listbox insert end $item }
	pack $frame.list $frame.scroll -side left -fill y
	# start with the first entry set
	$frame.list selection set 1
    }

    #----------------------------------------------------------------
    #----------------------------------------------------------------
    proc loadsub_menu {top} {
	variable leftbox
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	
	wm title $top "Load predefined subfiles"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	#
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox load
	# create frame for the subfile menu and then the menu
	set leftbox [frame $top.a.sublist -relief raised -bd 2] 
	logic_list $leftbox
	pack $rightbox $leftbox -side right
	
	set ns [namespace current]
	button $botbox.doit -text Apply \
		-command "${ns}::makesubcommand $leftbox.list"
	button $botbox.end -text Close -command "destroy $top"
	pack $botbox.doit $botbox.end -side left
    }

    proc initsub_menu {top} {
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Initialize table"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	#
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox init
	pack $rightbox -side left
	
	set ns [namespace current]
	button $botbox.doit -text Apply \
		-command "${ns}::makesubcommand {}; destroy $top"
	button $botbox.end -text Close -command "destroy $top"
	pack $botbox.doit $botbox.end -side left
	centerontop $top
    }

    proc loadfile_menu {top} {
	set ns [namespace current]
	# destroy any window with this name
	catch {destroy $top}
	set file [tk_getOpenFile -defaultextension .pdftbl \
		-filetypes {{"ICDD/PDF file" .pdftbl}}]
	if {$file == ""} return
	toplevel $top
	wm title $top "Select load logic"
	label $top.h -text "Select logic for loading file $file" -wrap 250
	options_list [frame $top.logic -bd 2 -relief groove] load
	frame $top.bot
	button $top.bot.l -text "Load" \
		-command "${ns}::makeloadmapcommand $top $file"
	button $top.bot.c -text "Cancel" \
		-command "destroy $top"
	pack $top.h $top.logic $top.bot -side top
	pack $top.bot.l $top.bot.c -side left
	centerontop $top
    }

    proc makeloadmapcommand {top file} {
	variable Logic_option
	saveundo
	loadmap $file -$Logic_option
	update_logic
	destroy $top
    }

    proc savefile_menu {top} {
	set file [tk_getSaveFile -defaultextension .pdftbl \
		-filetypes {{"ICDD/PDF file" .pdftbl}}]
	if {$file == ""} return
	savemap $file
    }

    proc elemcount_menu {top} {
	variable countvar 
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Select by number of elements"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox search
	# create frame for the subfile menu and then the menu
	set leftbox [frame $top.a.sublist -relief raised -bd 2] 
	pack $rightbox $leftbox -side right

	set ns [namespace current]
	for {set i 0} {$i < 10} {incr i} {
	    pack [checkbutton $leftbox.$i -variable ${ns}::countvar($i) \
		    -text "$i elements"] -side top -anchor w
	}

	button $botbox.doit -text Apply -command ${ns}::makecountcommand
	button $botbox.end -text Close -command "destroy $top"
	pack $botbox.doit $botbox.end -side left
    }

    proc pdfnum_menu {top} {
	variable countvar 
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Select entries by PDF number"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox search
	# create frame for the subfile menu and then the menu
	set leftbox [frame $top.a.sublist] 
	pack $rightbox $leftbox -side right

	pack [frame $leftbox.a -relief groove -bd 2] -side top -anchor center -pady 10
	pack [frame $leftbox.b -relief raised -bd 2] -side top -anchor center
	pack [frame $leftbox.b.t -width 150 -height 100]
	set ns [namespace current]
	pack [radiobutton $leftbox.a.1 -variable ${ns}::pdfnum(searchtype) \
		-command "${ns}::MakePDFnumbox  $leftbox.b $botbox.doit" \
		-value 1 -text "Range of entries "] -side top -anchor w
	pack [radiobutton $leftbox.a.2 -variable ${ns}::pdfnum(searchtype) \
		-command "${ns}::MakePDFnumbox  $leftbox.b $botbox.doit" \
		-value 2 -text "Specific entries"] -side top -anchor w
	variable pdfnum
	set pdfnum(searchtype) {}
	#array unset pdfnum entry*
	#array unset pdfnum max
	#array unset pdfnum min
	button $botbox.doit -text Apply -state disabled \
		-command "${ns}::makepdfnumcommand $top"
	button $botbox.end -text Close -command "destroy $top"
	pack $botbox.doit $botbox.end -side left
    }

    proc AddEntryBox {win var width} {
	variable $var
	set ns [namespace current]
	set row [set ${var}(nextrow)]
	set fr $win.canvas.fr
	grid [label $fr.l$row -text $row] \
		-column 0 -row $row -sticky e
	grid [entry $fr.$row -textvariable ${ns}::${var}(entry$row) -width $width] \
		-column 1 -row $row -sticky e
	incr ${var}(nextrow)
	update
	set sizes [grid bbox $win.canvas.fr]
	$win.canvas config -scrollregion $sizes -width [lindex $sizes 2]
	# use the scroll for BIG lists
	if {[lindex $sizes 3] > [winfo height $win.canvas]} {
	    grid $win.scroll -sticky ns -column 1 -row 1
	    # scroll to end
	    $win.canvas yview moveto 1
	} else {
	    grid forget $win.scroll 
	}
    }

    proc MakePDFnumbox {frame button} {
	variable pdfnum
	eval destroy [winfo children $frame]
	set ns [namespace current]
	$button config -state normal
	if {$pdfnum(searchtype) == 2} {
	    set win $frame
	    grid [label $frame.lbl -text "Enter PDF numbers"] \
		    -column 0 -columnspan 2 -row 0 -sticky w
	    grid [canvas $win.canvas \
		    -scrollregion {0 0 120 500} -width 120 -height 80 \
		    -yscrollcommand "$win.scroll set"] \
		    -column 0 -row 1 -sticky nse
	    frame [set pdfnum(window) $win.canvas.fr]
	    $win.canvas create window 0 0 -anchor nw -window $pdfnum(window)
	    grid columnconfigure $win 0 -weight 1
	    grid rowconfigure $win 1 -weight 1
	    scrollbar $win.scroll \
		    -command "$win.canvas yview"
	    grid [label $frame.caption -font {Helvetica -12 italic} -text \
		    "PDF numbers may be specified as 120034, 12-34 or 12,34" \
		    -wraplength 150 -justify left] \
		    -column 0 -columnspan 2 -row 2 -sticky e
	    set ns [namespace current]
	    grid [button $win.add -text "Add boxes" \
		    -command "${ns}::AddEntryBox $win pdfnum 8"] \
		    -column 0 -columnspan 2 -row 3 -sticky s
	    set pdfnum(nextrow) 1
	    AddEntryBox $win pdfnum 8
	    AddEntryBox $win pdfnum 8
	    AddEntryBox $win pdfnum 8
	} else {
	    set row 0
	    grid [label $frame.$row -text "Starting PDF number"] \
		    -column 0 -row [incr row] -sticky w
	    grid [entry $frame.$row -textvariable ${ns}::pdfnum(min) -width 8] \
		    -column 0 -row [incr row] -sticky e
	    grid [label $frame.$row -text "Ending PDF number"] \
		    -column 0 -row [incr row] -sticky w
	    grid [entry $frame.$row -textvariable ${ns}::pdfnum(max) -width 8] \
		    -column 0 -row [incr row] -sticky e
	    grid [label $frame.$row -font {Helvetica -12 italic} -text \
		    "PDF numbers may be specified as 120034, 12-34 or 12,34" \
		    -wraplength 150 -justify left] \
		    -column 0 -row [incr row] -sticky e
	}
    }

    proc string_menu {top} {
	variable countvar 
	variable Logic_option
	variable strsea
	set ns [namespace current]
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Select entries to match by text"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option]
	pack [frame $rightbox.a -relief raised -bd 2] -side top -pady 5
	pack [label $rightbox.a.top -text "Fields to search"] -side top
	pack [frame $rightbox.b -relief raised -bd 2] -side top -pady 5
	options_list $rightbox.b search
	# not all options allowed
	if {$Logic_option == "and" || $Logic_option == "not"} {
	    set Logic_option and
	}
	$rightbox.b.or config -state disabled
	$rightbox.b.not config -state disabled
	# create frame for the subfile menu and then the menu
	set leftbox [frame $top.a.sublist] 
	pack $rightbox $leftbox -side right

	pack [frame $leftbox.a -relief groove -bd 2] -side top -anchor center -pady 10
	pack [frame $leftbox.b -relief raised -bd 2] -side top -anchor center
	if {[catch {set strsea(searchtype)}]} {set strsea(searchtype) 1}
	pack [radiobutton $leftbox.a.1 -variable ${ns}::strsea(searchtype) \
		-value 1 -text "Match any one of:"] -side top -anchor w
	pack [radiobutton $leftbox.a.2 -variable ${ns}::strsea(searchtype) \
		-value 2 -text "Match all of:"] -side top -anchor w
	set ns [namespace current]
	button $botbox.doit -text Apply \
		-command "${ns}::makestrseacommand $top"
	button $botbox.end -text Close -command "destroy $top"
	pack $botbox.doit $botbox.end -side left
	foreach var {formula chemname mineral common authors coden} {
	    pack [checkbutton $rightbox.a.$var -variable ${ns}::strsea($var) \
		    -command "${ns}::EnableSearch $botbox.doit" \
		    -text $var] -anchor w -side top
	}

	set win $leftbox.b
	grid [label $win.lbl -text "Strings to find"] \
		-column 0 -columnspan 2 -row 0 -sticky w
	grid [canvas $win.canvas \
		-scrollregion {0 0 120 500} -width 120 -height 80 \
		-yscrollcommand "$win.scroll set"] \
		-column 0 -row 1 -sticky nse
	frame [set strsea(window) $win.canvas.fr]
	$win.canvas create window 0 0 -anchor nw -window $strsea(window)
	grid columnconfigure $win 0 -weight 1
	grid rowconfigure $win 1 -weight 1
	scrollbar $win.scroll \
		-command "$win.canvas yview"
	grid [label $win.caption -font {Helvetica -12 italic} -text \
		"Case of letters is ignored; commas may not be included in strings" \
		-wraplength 150 -justify left] \
		-column 0 -columnspan 2 -row 2 -sticky e
	set ns [namespace current]
	grid [button $win.add -text "More strings" \
		-command "${ns}::AddEntryBox $win strsea 15"] \
		-column 0 -columnspan 2 -row 3 -sticky s
	set strsea(nextrow) 1
	AddEntryBox $win strsea 15
	EnableSearch $botbox.doit
    }

    proc EnableSearch {button} {
	variable strsea  
	set flag disabled
	foreach var {formula chemname mineral common authors coden} {
	    if {$strsea($var)} {
		set flag normal
		break
	    }
	}
	$button config -state $flag
    }

    proc makestrseacommand {top} {
	variable strsea
	variable Logic_option
	set msg {}
	set stringlist {}
	for {set i 1} {$i < $strsea(nextrow)} {incr i} {
	    if {[string first , $strsea(entry$i)] != -1} {
		append msg "Comma is not allowed in string #$i \"$strsea(entry$i)\"\n"
	    }
	    if {[string trim $strsea(entry$i)] != ""} {
		if {$stringlist == ""} {
		    set stringlist $strsea(entry$i)
		} else {
		    append stringlist "," $strsea(entry$i)
		}
	    }
	}
	set fields {}
	foreach var {formula chemname mineral common authors coden} {
	    if {$strsea($var)} {
		append fields " -$var"
	    }
	}
	if {$Logic_option != "and" && $Logic_option != "remove"} {
	    append msg "Invalid Combine option\n"
	}
	if {$msg != ""} {
	    tk_dialog $top.msg "Invalid input" \
		    "Error: \n$msg" \
		    error 0 Continue
	    return
	}
	saveundo
	if {$strsea(searchtype) == 1} {
	    eval findstring -$Logic_option -oneof \"$stringlist\" $fields
	} else {
	    eval findstring -$Logic_option -allof \"$stringlist\" $fields
	}
	update_logic
    }


    proc element_menu {top} {
	set ns [namespace current]
	variable rightbox
	variable settings
	set root $settings(root)
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Select by composition"
	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox search
	set leftbox [frame $top.a.l] 
	pack $rightbox $leftbox -side left
	set lefttop [frame $top.a.ltop -relief raised -bd 2] 
	set leftbot [frame $top.a.lbot -relief raised -bd 2] 
	pack $lefttop $leftbot -in $leftbox -side top
	pack [label $lefttop.0 -text "Required elements"] -side top
	pack [radiobutton $lefttop.1 -variable ${ns}::required_opt \
		-value oneof -text "One of:"] -side top -anchor w
	pack [radiobutton $lefttop.2 -variable ${ns}::required_opt \
		-value allof -text "All of:"] -side top -anchor w
	pack [entry $lefttop.3 -textvariable ${ns}::req_list]  -side top 
	pack [button $lefttop.4 -text "Use table" \
		-command "${ns}::makePeriodicTable $root.req req_list {Required Elements}"]\
		-side top -anchor center
	$lefttop.1 invoke

	pack [label $leftbot.0 -text "Optional elements"] -side top
	pack [radiobutton $leftbot.1 -variable ${ns}::optional_opt \
		-value alloptional\
		-command "${ns}::hideelembox 1 $leftbot" \
		-text "All elements optional"] \
		-side top -anchor w
	pack [radiobutton $leftbot.2 -variable ${ns}::optional_opt  \
		-value optional\
		-command "${ns}::hideelembox 0 $leftbot" \
		-text "Allowed elements:"] \
		-side top -anchor w
	pack [entry $leftbot.3 -textvariable ${ns}::opt_list]  -side top 
	pack [button $leftbot.4 -text "Use table" \
		-command "${ns}::makePeriodicTable $root.opt opt_list {Optional Elements}"] \
		-side top -anchor center
	$leftbot.1 invoke
	button $botbox.doit -text Apply -command ${ns}::makeelementcommand
	button $botbox.end -text Close \
		-command "destroy $top; catch \"destroy $root.req\"; catch \"destroy $root.opt\""
	pack $botbox.doit $botbox.end -side left
    }

    proc hideelembox {onoff box} {
	variable settings
	set root $settings(root)
	if {$onoff} {
	    $box.3 config -state disabled -relief flat -fg [$box.3 cget -bg]
	    $box.4 config -state disabled
	    catch {destroy $root.opt}
	} else {
	    $box.3 config -state normal -relief sunken -fg black
	    $box.4 config -state normal
	}
    }

    proc saveboxastext {box {print ""}} {
	variable settings
	set root $settings(root)
	if [catch {
	    if {$print == ""} {
		set file [tk_getSaveFile -defaultextension $root.txt]
		if {[string trim $file] == ""} return
		set fp [open $file a]
		puts $fp [clock format [clock seconds]]
		puts $fp [$box get 1.0 end]
		close $fp
	    } else {
		set file tmp.txt
		set fp [open $file w]
		puts $fp [clock format [clock seconds]]
		puts $fp [$box get 1.0 end]
		close $fp
		if {$::tcl_platform(platform) != "windows"} {
		    eval exec cat $file | $settings(printer)
		    file delete $file
		} else {
		    eval exec $settings(printer) $file
		}
	    }
	} errmsg ] {
	    if {$errmsg == "" && $file == ""} return
	    tk_dialog $root.msg "error writing file" \
		    "Error writing file: $errmsg" \
		    error 0 OK
	}
    }

    proc savepeaks {} {
	variable settings
	set root $settings(root)
	if [catch {
	    set file [tk_getSaveFile -defaultextension .pks]
	    if {[string trim $file] == ""} return
	    set fp [open $file a]
	    puts $fp [clock format [clock seconds]]
	    set list [getpdf1 -seq $settings(seqno) -common -mineral -chemname]
	    puts -nonewline $fp "[getpdf1 -seq $settings(seqno) -entry]: "
	    puts -nonewline $fp \
		    "[lindex [getpdf1 -seq $settings(seqno) -formula] 0] ("
	    foreach item $list {
		if {$item != ""} {
		    puts -nonewline  $fp "$item "
		}
	    }
	    puts $fp ")"
	    puts $fp "d-space\tint"
	    set peaks [lindex [getpdf1 -seq $settings(seqno) -peaks ] 0]
	    if {$settings(peaktype) == "-q" } {
		set peaks1 [lindex [eval getpdf1 -seq $settings(seqno) -peaks -q] 0]
		puts $fp "\tQ"
	    } elseif {$settings(peaktype) == "-wave"} {
		set peaks1 [lindex [eval getpdf1 -seq $settings(seqno) -peaks \
			-wave $settings(wave) ] 0]
		puts $fp "\t2Theta, (wave = $settings(wave))"
	    } else {
		set peaks1 ""
	    }
	    set i 0
	    while {$i < [llength $peaks]} {
		set pos [lindex $peaks $i]
		set tt [lindex $peaks1 $i]
		incr i
		set int [lindex $peaks $i]
		puts $fp "$pos\t$int\t$tt"
		incr i
	    }
	    close $fp
	} errmsg ] {
	    tk_dialog $root.msg "error writing peaks" \
		    "Error writing file: $errmsg" \
		    error 0 OK
	}
    }

    # restart the browse window, if the window is open
    proc UpdateBrowse {} {
	variable settings
	set root $settings(root)
	# see if the browse window exists
	if [catch { wm state $root.browse} ] {return}
	first_entry
	set button $settings(browse_viewas)
	if {[$button cget -text] == "view as pdf2"} {
	    display_entry_pdf1 
	} else {
	    display_entry_pdf2
	}
    }
    # create the browse dialog box
    proc make_browse { {first 0} } {
	variable settings
	set root $settings(root)
	variable txtbx
	# reuse previous browse window
	set top $root.browse
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	set txtbx [text $top.t -relief raised -bd 2 -height 10 -width 50 \
		-wrap none \
		-yscrollcommand "$top.s set" \
		-xscrollcommand "$top.a set"]
	scrollbar $top.s -command "$top.t yview"
	scrollbar $top.a -command "$top.t xview" -orient horizontal
	frame $top.bar -relief groove -bd 2
	pack $top.bar $top.a -side bottom -fill x
	pack $top.t -side left -fill both -expand yes
	pack $top.s -side left -fill y 
	pack [button $top.bar.next -text Next] -side left
	set settings(browse_previous) $top.bar.prev
	pack [button $top.bar.prev -text Previous] -side left
	set settings(browse_viewas) $top.bar.fmt 
	pack [button $top.bar.fmt -text "view as pdf2"] -side left 

	set ns [namespace current]
	pack [button $top.bar.saveas -text "Save as..." \
		-command "${ns}::saveboxastext $top.t "] -side left
	pack [button $top.bar.print -text "Print" \
		-command "${ns}::saveboxastext $top.t print"] -side left
	pack [button $top.bar.export -text "Export..." \
		-command ${ns}::savepeaks] -side left
	pack [button $top.bar.end -text Close -command "destroy $top"] -side right
	$top.bar.fmt config -command ${ns}::switch_pdf_view
	$top.bar.next config -command ${ns}::next_entry
	$top.bar.prev config -command ${ns}::prev_entry 
	if {$first} first_entry
	display_entry_pdf1
    }

    proc switch_pdf_view {} {
	variable settings
	set button $settings(browse_viewas)
	if {[$button cget -text] == "view as pdf2"} {
	    $button config -text "view as pdf1"
	    display_entry_pdf2
	} else {
	    $button config -text "view as pdf2"
	    display_entry_pdf1
	}
    }

    proc first_entry {} {
	variable settings
	set settings(hitnum) 0
	set settings(seqno) [nexthit -first]
	incr settings(hitnum)
	$settings(browse_previous) config -state disabled
    }

    proc next_entry {} {
	variable settings
	set button $settings(browse_viewas)
	set settings(seqno) [nexthit]
	incr settings(hitnum)
	if {$settings(seqno) < 0} first_entry
	if {[$button cget -text] == "view as pdf2"} {
	    display_entry_pdf1 
	} else {
	    display_entry_pdf2
	}
	if {$settings(hitnum) == 1} {
	    $settings(browse_previous) config -state disabled
	} else {
	    $settings(browse_previous) config -state normal
	}
    }

    proc prev_entry {} {
	variable settings
	set button $settings(browse_viewas)
	set settings(seqno) [nexthit -backward]
	incr settings(hitnum) -1
	if {$settings(seqno) < 0} first_entry
	if {$settings(hitnum) == 1} {
	    $settings(browse_previous) config -state disabled
	}
	if {[$button cget -text] == "view as pdf2"} {
	    display_entry_pdf1 
	} else {
	    display_entry_pdf2
	}
    }

    proc display_entry_pdf1 {} {
	variable settings
	set root $settings(root)
	variable txtbx 
	if {$settings(peaktype) == "-wave"} {
	    set list [eval getpdf1 -seq $settings(seqno) -entry -formula \
		    -chemname -common -mineral -reference -peaks \
		    $settings(peaktype) $settings(wave) \
		    $settings(peakorder)]
	} else {
	    set list [eval getpdf1 -seq $settings(seqno) -entry -formula \
		    -chemname -common -mineral -reference -peaks \
		    $settings(peaktype) $settings(peakorder)]
	}
	$txtbx configure -state normal
	if {$::tcl_version >= 8.0} {$txtbx config -font $settings(font)}
	$txtbx delete 1.0 end
	set patnumb "[lindex $list 0]\t"
	regsub "D\t" $patnumb " (Deleted)" patnumb
	$txtbx insert end \
		"Entry: $patnumb\t(hit $settings(hitnum)/[report -current])\n"
	set tmp [lindex $list 1] 
	if {$tmp != {}} {$txtbx insert end "Chemical formula:\t$tmp\n"}
	set tmp [lindex $list 2]
	if {$tmp != {}} {$txtbx insert end "Chemical name:\t$tmp\n"}
	set tmp [lindex $list 3]
	if {$tmp != {}} {$txtbx insert end "Common name:\t$tmp\n"}
	set tmp [lindex $list 4]
	if {$tmp != {}} {$txtbx insert end "Mineral name:\t$tmp\n"}
	set tmp [lindex $list 5]
	if {$tmp != {}} {$txtbx insert end "Reference:\t$tmp\n"}
	if {$settings(peaktype) == "-q" } {
	    $txtbx insert end "Peaks, Q (A-1):\n"
	} elseif {$settings(peaktype) == "-wave"} {
	    $txtbx insert end "Peaks, 2Theta, (degrees, wave = $settings(wave)):\n"
	} else {
	    $txtbx insert end "Peaks, d-spaces (A):\n"
	}
	# format peaks
	set peaks [lindex $list 6]
	set ncol 4
	set nrow [expr ([llength $peaks]+ 2*$ncol - 1) / ( 2*$ncol ) ]
	for {set row 0} {$row < $nrow} {incr row} {
	    $txtbx insert end "  "
	    for {set col 0} {$col < $ncol} {incr col} {
		set pos [expr 2*($row + $nrow*$col)]
		if {$pos >= [llength $peaks]} break
		$txtbx insert end \
			[format "%8.3f %3d" \
			[lindex $peaks $pos] [lindex $peaks [expr $pos+1]]]
	    }
	    $txtbx insert end "\n"
	}
	#    $txtbx insert end "\t[lindex $list 6]\n"
	$txtbx configure -state disabled  -height 10 -width 50
    }

    proc display_entry_pdf2 {} {
	variable settings
	set root $settings(root)
	variable txtbx

	if {$settings(peaktype) == "-q" } {
	    set fmt Q
	} elseif {$settings(peaktype) == "-wave"} {
	    set fmt $settings(wave)
	} else {
	    set fmt D
	}
	set list [getpdf2 -seq $settings(seqno) -file pdf2.fmt -format $fmt]

	set fl [open pdf2.fmt r]
	set fmt [read $fl]
	close $fl
	file delete pdf2.fmt
	$txtbx configure -state normal
	if {$::tcl_version >= 8.0} {$txtbx config -font $settings(font)}
	$txtbx delete 1.0 end
	$txtbx insert end \
		"\t\t\t\t\t\t\t\t\t\t\t\t\t\t ($settings(hitnum)/[report -current])\n"
	$txtbx insert end $fmt
	if {$settings(showaids)} {
	    $txtbx insert end \n
	    foreach line $list {
		$txtbx insert end "$line\n"
	    }
	    set dbg [open "lastaids.rec" w]
	    foreach line $list {
		puts $dbg $line
	    }
	    close $dbg
	}
	$txtbx configure -state disabled -width $settings(pdf2width) -height 24  
    }

    proc validatewave {} {
	variable settings
	set settings(temp) d
	switch -- $settings(peaktype) {
	    -q {set settings(temp) q} 
	    -wave {
		set settings(temp) O
		if {$settings(wave) == 1.5418} {set settings(temp) C}
	    }
	}
    }

    proc testwave {top} {
	variable settings
	switch $settings(temp) {
	    q {set settings(peaktype) -q}
	    C {
		set settings(peaktype) -wave
		set settings(wave) 1.5418
	    }
	    O {
		if {[catch {expr $settings(wave)*1}]} {
		    $top.a.warn config -text "Invalid Wavelength"; return
		}
		if {$settings(wave) <= 0} {
			$top.a.warn config -text "Invalid Wavelength"; return
		}
		set settings(peaktype) -wave
	    }
	    default { set settings(peaktype) ""}
	}
	destroy $top
    }

    proc settings_menu {top} {
	variable settings 
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Display settings"
	pack [frame $top.a]
	set topa $top.a
	set ns [namespace current]
	grid [button $topa.d -text Done -command "${ns}::testwave $top"] -row 2 -column 0 -columnspan 2
	grid [frame $topa.a -bd 4 -relief groove] -row 0 -column 0 -rowspan 2 
	
	grid [frame $topa.b -bd 4 -relief groove] -row 0 -column 1 -sticky news
	pack [label $topa.b.1 -text "PDF-1/Index Peak Ordering"]
	pack [radiobutton  $topa.b.2 -value {} -text "Decreasing d-space" \
		-variable ${ns}::settings(peakorder) ] -side top -anchor w 
	pack [radiobutton  $topa.b.3 -value -maxi -text "Decreasing intensity" \
		-variable ${ns}::settings(peakorder) ] -side top -anchor w
	
	grid [frame $topa.c -bd 4 -relief groove] -row 1 -column 1 -sticky news
	pack [label $topa.c.1 -text "PDF-2 Display"]
	pack [checkbutton  $topa.c.2 -text "Include AIDS*83 records" \
		-variable ${ns}::settings(showaids) ] -side top -anchor w 
	
	validatewave
	set enable     {if {$settings(temp) == "O"} }
	append enable " \{ $topa.a.w config -state normal -relief sunken; "
	append enable  " $topa.a.w config -fg \[lindex \[$topa.a.w config -fg\] 3\]\}"
	append enable " \{ $topa.a.w config -state disabled -relief flat ;" 
	append enable   " $topa.a.w config -fg \[ $topa.a.w cget -bg \] ;"
	append enable   " $topa.a.warn config -text {} \}"
	pack [label $topa.a.l -text "Display Peaks as"] -side top
	pack [radiobutton  $topa.a.1 -value d -text d-space \
		-command  $enable \
		-variable ${ns}::settings(temp) ] -side top -anchor w 
	pack [radiobutton  $topa.a.2 -value q -text "Q (A-1)" \
		-command  $enable \
		-variable ${ns}::settings(temp) ] -side top -anchor w
	pack [radiobutton  $topa.a.3 -value C -text "2theta, CuKa" \
		-command  $enable \
		-variable ${ns}::settings(temp) ] -side top -anchor w
	pack [radiobutton  $topa.a.4 -value O -text "2theta, other" \
		-command  $enable \
		-variable ${ns}::settings(temp) ] -side top -anchor w
	pack [entry $topa.a.w -textvariable ${ns}::settings(wave) -width 8]
	pack [label $topa.a.warn -text ""]
	eval $enable
	centerontop $top
    }
    
    proc printsettings_menu {top} {
	variable settings 
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Print settings"
	pack [button $top.d -text Done -command "destroy $top"] -side bottom 
	pack [frame $top.a -bd 4 -relief groove] -side left 
	pack [frame $top.b -bd 4 -relief groove] -side left
	pack [label $top.b.1 -text "Printer command line"]
	pack [entry $top.b.2 -textvariable ${ns}::settings(printer) -width 30] -side top -anchor w 
	centerontop $top
    }

    proc disableindexentry {val top dummy} {
	variable settings
	if {$settings(len$val) > 5 } {
	    $top.f.$val.t config -state normal
	} else { 
	    $top.f.$val.t config -state disabled 
	}
    }

    proc indexsettings_menu {top} {
	variable settings 
	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Index contents"

	pack [frame $top.a ] -side top
	pack [label $top.a.1 -text \
		"Select the entries to include in the index and their lengths"]

	pack [frame $top.f ] -side top
	foreach val {1 2 3 4 5 6} {
	    pack [frame $top.f.$val -bd 4 -relief groove] -side left -fill both
	    set ns [namespace current]
	    scale $top.f.$val.s -from 0 -to 50 -orient horizontal -label length \
		    -variable ${ns}::settings(len$val) \
		    -command "${ns}::disableindexentry $val $top"
	    eval tk_optionMenu $top.f.$val.t settings(ent$val) $settings(options)
	    $top.f.$val.t config -takefocus {}
	    pack [label $top.f.$val.l -text "entry $val" -width 15] 
	    pack $top.f.$val.s  $top.f.$val.t 
	}
	pack [frame $top.b ] -side bottom
	pack [button $top.b.2 -text Done \
		-command "destroy $top;${ns}::UpdateIndex"] -side right
	pack [button $top.b.1 -text Apply \
		-command ${ns}::UpdateIndex] -side left
	centerontop $top
    }

    # writes ~/.logicrc to save options
    proc savesettings {} {
	variable settings
	set fp [open ~/.logicrc w]
	puts $fp "array set settings {"
	foreach var {1 2 3 4 5 6} {
	    puts $fp "\t len$var \t$settings(len$var)\t ent$var [list $settings(ent$var)]"
	}
	foreach item {peakorder peaktype temp wave font printer showaids} {
	    puts $fp "\t $item [list $settings($item)]"
	}
	puts $fp "}"
	close $fp
    }

    # index_to_browse is used to display an entry from the index by
    # double clicking
    proc index_to_browse {textbx coord} {
	variable settings
	set linenum [lindex [split [$textbx index @$coord] .] 0]
	# jump to the selected entry
	set num [expr $linenum - 1]
	if {$num < 1} {set num 1}
	if {$num > [report -total]} {set num [report -total]}
	set settings(seqno) [nexthit [set settings(hitnum) $num]]
	if [catch {
	    set button $settings(browse_viewas)
	    
	    if {[$button cget -text] == "view as pdf2"} {
		display_entry_pdf1 
	    } else {
		display_entry_pdf2
	    }  
	}] {make_browse}
    }

    # reload the index contents, if the window is open
    proc UpdateIndex {} {
	# see if an index window exists
	variable settings
	set root $settings(root)
	if [catch { wm state $root.index} ] {return}
	make_index
    }

    proc make_index { } {
	variable settings
	set root $settings(root)
	variable indexbx
	set ns [namespace current]

	# reuse previous index window
	set top $root.index
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	pack [frame $top.bar -relief groove -bd 2] -side bottom -fill x
	pack [scrollbar $top.a -command "$top.t xview" -orient horizontal] \
		-side bottom -fill x
	set indexbx [text $top.t -relief raised -bd 2 -height 10 -width 50 \
		-wrap none \
		-yscrollcommand "$top.s set" \
		-xscrollcommand "$top.a set"]
	if {$::tcl_version >= 8.0} {$indexbx config -font $settings(font)}
	bind $indexbx <Double-1> "${ns}::index_to_browse $indexbx %x,%y"
	pack $indexbx -side left -fill both -expand yes
	pack [scrollbar $top.s -command "$top.t yview"] -side left -fill y 
	set settings(abort) 0
	pack [button $top.bar.stop -text stop \
		-command "set settings(abort) 1"] -side right
	pack [label $top.bar.label -text ""] -side left
	pack [button $top.bar.saveas -text "Save as..." \
		-command "${ns}::saveboxastext $top.t "] -side left
	pack [button $top.bar.print -text "Print" \
		-command "${ns}::saveboxastext $top.t print"] -side left

	update

	set lengths {}
	set seqloc {}
	set seqlen 0
	set header {}
	set entries {}
	foreach val {1 2 3 4 5 6} {
	    if { $settings(len$val) > 5 } {
		lappend header $settings(ent$val)
		set entnum [lsearch $settings(options) $settings(ent$val)]
		if {$entnum == 0} {
		    set seqloc [llength $lengths] 
		    set seqlen $settings(len$val)
		} else {
		    lappend lengths $settings(len$val)
		    lappend entries [lindex $settings(flag) $entnum]
		}
	    }
	}
	if {$seqlen > 0} {
	    set  lengths [linsert $lengths $seqloc $seqlen]
	}
	# start entering info into the listbox
	$indexbx configure -state normal
	$indexbx delete 1.0 end
	# make a header
	set line {}
	for {set i 0} {$i < [llength $settings(options)]} {incr i} {
	    set x [lindex $header $i]
	    append x "                                                 "
	    append line [string range $x 0 [expr [lindex $lengths $i]-1]] " "
	}
	$indexbx insert end $line
	#    $indexbx insert end "\n"
	set seq 1
	set settings(seqno) [nexthit -first]
	set total [report -current]
	set prevdone 0
	while {$settings(seqno) > 0} {
	    set done [expr 100 * $seq / $total]
	    if {$done > $prevdone} {
		set prevdone $done 
		$top.bar.label config -text "$done % complete"
		update
		if {$settings(abort)} break
	    }
	    set line {}
	    # get the pdf1 info
	    if {$settings(peaktype) == "-wave"} {
		set list [eval getpdf1 -seq $settings(seqno) $entries \
			$settings(peaktype) $settings(wave) \
			$settings(peakorder)]
	    } else {
		set list [eval getpdf1 -seq $settings(seqno) $entries \
			$settings(peaktype) $settings(peakorder)]
	    }
	    # add the sequence number
	    if {$seqlen > 0} {set list [linsert $list $seqloc $seq]}
	    # create a line from the entry
	    for {set i 0} {$i < [llength $list]} {incr i} {
		set x [lindex $list $i]
		append x "                                                 "
		append line [string range $x 0 [expr [lindex $lengths $i]-1]] " "
	    }
	    set settings(seqno) [nexthit]
	    incr seq
	    $indexbx insert end "\n"
	    $indexbx insert end $line
	}
	# remove counting label
	if {! $settings(abort)} {pack forget $top.bar.label}
	# reconfigure the stop button
	$top.bar.stop config -text Close -command "destroy $top"
	$indexbx configure -state disabled  -height 10 -width 50
	UpdateBrowse
    }



    #----------------------------------------------------------------
    # enter information into logic (main) screen
    #----------------------------------------------------------------
    proc update_logic {} {
	variable settings
	set root $settings(root)
	$root.l2 configure -text "[report -current] of [report -total] entries"
	$root.l3.t configure -state normal
	$root.l3.t delete 1.0 end
	set num 1
	$root.l3.t insert end "Command history\n\n"
	foreach item [report -history] {
	    $root.l3.t insert end " $num) $item\n"
	    incr num
	}
	$root.l3.t configure -state disabled
	# since history is changed update browse & index
	UpdateIndex
    }

    proc saveundo {} {
	variable settings
	storemap
	eval $settings(undo) -state normal
    }

    proc undo {} {
	variable settings
	set root $settings(root) 
	set before [report -current]
	restoremap
	set after [report -current]
	if {$after > 1.5*$before && $after > 1000} {catch {destroy $root.index $root.browse}}
	eval $settings(undo) -state disabled
	update_logic
    }

    proc peakmenudone {top} {
	variable peakopt 
	variable Logic_option
	variable settings
	set root $settings(root) 
	switch $peakopt(temp) {
	    q {set peakopt(peaktype) -q}
	    C {
		set peakopt(peaktype) -wave
		set peakopt(wave) 1.5418
	    }
	    O {
		if {[catch {expr $peakopt(wave)*1}]} {
		    warnbox "Invalid Wavelength: $peakopt(wave)"; return
		}
		if {$peakopt(wave) <= 0} {
		    warnbox "Invalid Wavelength: $peakopt(wave)"; return
		}
		set peakopt(peaktype) -wave
	    }
	    default { set peakopt(peaktype) ""}
	}
	set peaklist {}
	foreach var {1 2 3 4 5 6 7 8} {
	    if {$peakopt(peak$var) != ""} {	
		if {[catch {expr $peakopt(peak$var)*1}]} {
		    warnbox "Invalid peak value: $peakopt(peak$var)"; return
		}
		if {$peakopt(peak$var) <= 0} {
		    warnbox "Invalid peak value: $peakopt(peak$var)"; return
		}
		lappend peaklist $peakopt(peak$var)
	    }
	}
	if {[llength $peaklist] <= 0} {
	    warnbox "No valid peaks to search!"; return
	}
	set errorlist {}
	foreach var {1 2 3 4 5 6 7 8} {
	    if {$peakopt(err$var) != ""} {	
		if {[catch {expr $peakopt(err$var)*1}]} {
		    warnbox "Invalid error value: $peakopt(err$var)"; return
		}
		if {$peakopt(err$var) <= 0} {
		    warnbox "Invalid error value: $peakopt(err$var)"; return
		}
		lappend errorlist $peakopt(err$var)
	    }
	}
	if {[llength $errorlist] <= 0} {
	    warnbox "No valid error range!"; return
	}
	saveundo
	waitbox {Search in progress, please wait}
	update
	if {$peakopt(peaktype) == "-wave"} {
	    set list [eval findpeak \
		    -peak [list $peaklist] -error [list $errorlist] \
		    $peakopt(0) $peakopt(1) $peakopt(3) -$Logic_option \
		    $peakopt(peaktype) $peakopt(wave) ]
	} else {
	    set list [eval findpeak \
		    -peak [list $peaklist] -error [list $errorlist] \
		    $peakopt(0) $peakopt(1) $peakopt(3) -$Logic_option \
		    $peakopt(peaktype) ]
	}
	catch {destroy $root.wait}
	update_logic
	catch {destroy $root.warn}
    }

    proc peakMenu_enable {leftmid} {
	variable peakopt
	if {$peakopt(temp) == "O"} { 
	    $leftmid.w config -state normal -relief sunken
	    $leftmid.w config -fg [lindex [$leftmid.w config -fg] 3]
	} else { 
	    $leftmid.w config -state disabled -relief flat
	    $leftmid.w config -fg [$leftmid.w cget -bg]
	}
    }

    proc peak_menu {top} {
	variable rightbox 
	variable peakopt
	set ns [namespace current]
	variable settings
	set root $settings(root)

	# reuse any window with this name
	catch {toplevel $top}
	raise $top
	eval destroy [pack slaves $top]
	wm title $top "Select by peaks"

	set leftmid  $top.1 
	pack [frame $leftmid -relief raised -bd 2] -fill x -side top

	pack [label $leftmid.l -text "Enter peaks as "] -side top -anchor w
	pack [radiobutton  $leftmid.1 -value d -text d-space \
		-command  "peakMenu_enable $leftmid" \
		-variable ${ns}::peakopt(temp) ] -side left -anchor w 
	pack [radiobutton  $leftmid.2 -value q -text "Q (A-1)" \
		-command  "peakMenu_enable $leftmid" \
		-variable ${ns}::peakopt(temp) ] -side left -anchor w
	pack [radiobutton  $leftmid.3 -value C -text "2theta, CuKa" \
		-command  "peakMenu_enable $leftmid" \
		-variable ${ns}::peakopt(temp) ] -side left -anchor w
	pack [radiobutton  $leftmid.4 -value O -text "2theta, wavelength:" \
		-command  "peakMenu_enable $leftmid" \
		-variable ${ns}::peakopt(temp) ] -side left -anchor w
	pack [entry $leftmid.w -textvariable ${ns}::peakopt(wave) -width 8] -side left -anchor e

	peakMenu_enable $leftmid

	pack [frame $top.2 -relief raised -bd 2] -fill x -side top 
	pack [label $top.2.l -text "Enter one or more peaks:"] \
		-side top -anchor w
	foreach var {1 2 3 4 5 6 7 8} {
	    pack [entry $top.2.$var -textvariable ${ns}::peakopt(peak$var) -width 8] \
		    -side left -padx 1 -fill x -expand yes
	}
	pack [frame $top.3 -relief raised -bd 2] -fill x -side top 
	pack [label $top.3.l -text "Enter one or more errors"] \
		-side top -anchor w
	foreach var {1 2 3 4 5 6 7 8} {
	    pack [entry $top.3.$var -textvariable ${ns}::peakopt(err$var) -width 8] \
		    -side left -padx 1 -fill x -expand yes
	}

	set topbox [frame $top.a]; # frame for all boxes
	set botbox [frame $top.b]; # frame for buttons
	pack $topbox $botbox -side top
	# create frame for the options menu and then the menu
	set rightbox [frame $top.a.option -relief raised -bd 2]
	options_list $rightbox search
	pack $rightbox -side left

	set peakmid [frame $top.a.mid -relief raised -bd 2] 
	pack $peakmid -side left -fill x
	pack [label $peakmid.t -text "Peak Threshold"] -side top
	pack [radiobutton $peakmid.1 -variable ${ns}::peakopt(0) -value -3strongest\
		-text "One of the 3 strongest"]\
		-side top -anchor w
	pack [frame $peakmid.r ] -side right
	pack [frame $peakmid.l ] -side left
	foreach val {0 10 20 30 40} {
	    pack [radiobutton $peakmid.l.$val -variable ${ns}::peakopt(0) \
		    -value "-intmin $val" -text ">= $val%"]\
		    -side top -anchor w
	}
	foreach val {50 60 70 80 90} {
	    pack [radiobutton $peakmid.r.$val -variable ${ns}::peakopt(0) \
		    -value "-intmin $val" -text ">= $val%"]\
		    -side top -anchor w
	}
	$peakmid.l.30 invoke

	set leftbox [frame $top.a.l] 
	pack $leftbox -side left


	set lefttop [frame $top.a.ltop -relief raised -bd 2] 
	pack $lefttop -in $leftbox -side top -fill x
	pack [label $lefttop.0 -text "Multiple peaks"] -side top
	pack [radiobutton $lefttop.1 -variable ${ns}::peakopt(1) -value -any\
		-text "Require one of list"] -side top -anchor w
	pack [radiobutton $lefttop.2 -variable ${ns}::peakopt(1) -value -all\
		-text "Require all of list"] -side top -anchor w
	$lefttop.1 invoke


	set leftbot [frame $top.a.lbot -relief raised -bd 2] 
	pack $leftbot -in $leftbox -side top -fill x
	pack [label $leftbot.0 -text "Search speed"] -side top
	pack [radiobutton $leftbot.1 -variable ${ns}::peakopt(3) -value -fast\
		-text "Expand errors as needed"]\
		-side top -anchor w
	pack [radiobutton $leftbot.2 -variable ${ns}::peakopt(3) -value -exact\
		-text "Use exact errors (slow)"]\
		-side top -anchor w
	$leftbot.1 invoke

	button $botbox.doit -text Apply -command "${ns}::peakmenudone $top"
	button $botbox.end -text Close \
		-command "destroy $top; catch {destroy $root.warn}"
	pack $botbox.doit $botbox.end -side left
    }
    proc CreateLogicSubwin {rootwin} {
	variable settings
	if {$rootwin == "."} {
	    set root ""
	} else {
	    set root $rootwin
	}
	set settings(root) $root
	bind all <Control-KeyPress-c> "destroy $rootwin"
	# create the main window
	wm title $rootwin "Program Logic"
	frame $root.l1 -relief groove -bd 4
	label $root.l2 -text <>
	frame $root.l3
	pack $root.l1 -side top -fill x
	pack $root.l2 -anchor e -side top 
	pack $root.l3 -side top -fill both -expand yes
	text  $root.l3.t -relief raised -bd 2 -height 10 -width 50 \
		-wrap none \
		-yscrollcommand "$root.l3.s set" \
		-xscrollcommand "$root.l3.a set" 
	scrollbar $root.l3.s -command "$root.l3.t yview" 
	scrollbar $root.l3.a -command "$root.l3.t xview" -orient horizontal
	pack $root.l3.a -side bottom -fill x
	pack $root.l3.t -side left -fill both -expand yes
	pack $root.l3.s -side left -fill y 
	
	foreach bar {file view searches options} {
	    menubutton $root.l1.$bar -text $bar -underline 0 \
		    -menu $root.l1.$bar.menu
	    pack $root.l1.$bar -side left
	    menu $root.l1.$bar.menu
	}

	# file menu
	set menu $root.l1.file.menu
	foreach lbl {Init Save Reload} \
		cmd {initsub_menu savefile_menu loadfile_menu} {
	    $menu add command -label $lbl \
		    -command "::logicGUI::$cmd $root.$cmd"
	}
	if {$root == ""} {
	    $menu add command -label  Quit -command {destroy .}
	} else {
	    $menu add command -label Close -command "destroy $rootwin"
	}
	# view menu
	set menu $root.l1.view.menu
	$menu add command -label  Browse -command {::logicGUI::make_browse 1}
	$menu add command -label  Index -command {::logicGUI::make_index} 

	# search menu
	set menu $root.l1.searches.menu
	foreach lbl {Subfile Chemistry "Element count" \
		Peaks "PDF number" "String search"} \
		cmd {loadsub_menu element_menu elemcount_menu \
		peak_menu pdfnum_menu string_menu} {
	    $menu add command -label $lbl \
		    -command "::logicGUI::$cmd $root.$cmd"
	}
	$menu add separator
	$menu add command -label  Undo -command {::logicGUI::undo} 
	set settings(undo) "$menu entryconfigure Undo"
	
	# options menu
	set menu $root.l1.options.menu
	foreach lbl {"Display settings" "Index contents" "Printer settings"} \
		cmd {settings_menu indexsettings_menu printsettings_menu} {
	    $menu add command -label $lbl \
		    -command "::logicGUI::$cmd $root.$cmd"
	}
	if {$::tcl_version >= 8.0} {
	    $menu add cascade -label "Font" -menu $menu.font
	    menu $menu.font
	    foreach size {5 6 7 8 9 10 11 12 13 14 15 16} {
		$menu.font add command -label "Courier $size" \
			-command "set settings(font) \"Courier $size\"; \
			catch {\$txtbx config -font \$settings(font)}; \
			catch {\$indexbx config -font \$settings(font)}"
	    }
	}
	$menu add command -label  SaveSettings -command ::logicGUI::savesettings
	ShowLogicCopyright 
	update_logic
	eval $settings(undo) -state disabled
    }

    # display copyright notice on first use
    proc ShowLogicCopyright {} {
	variable settings
	if [catch {set shownotice [file exists ~/icdd.notice]}] {
	    set shownotice 1
	}
	if [catch {set havenotice [file exists $settings(home)/copynot.txt]}] {
	    set havenotice 0
	}
	if {!$shownotice} {
	    if [catch {set havenotice [file exists $settings(home)/copynot.txt]}] {
		set havenotice 0
	    }
	    if {!$havenotice} {return}
	    set in [ open [file join $settings(home) copynot.txt]]
	    set msg [read $in]
	    close $in
	    if [tk_dialog $settings(root).warn "Copyright Notice" $msg {} 0 OK Quit] {
		exit
	    } else {
		catch {close [open ~/icdd.notice w]}
	    }  
	}
    }
}
