# This implements the commands in logicpkg.c but using 
# discrete programs 
# Commands are:
#     sublist    lists the name associated with a subfile or all subfiles.
#     loadsub    load a subfile into the current bitmap file
#     storemap   saves the current table for undo use
#     restoremap restores the last saved table as the current table 
#     report
#     element
#     nexthit
#     savemap
#     loadmap
#     getpdf1
#     getpdf2
#     findpeak
#     findnumber
#     elemcount

namespace eval logic {
    # where is this file?
    set savedvals(script) [info script]
    # translate links -- go six levels deep
    foreach i {1 2 3 4 5 6} {
	if {[file type $savedvals(script)] == "link"} {
	    set link [file readlink $savedvals(script)]
	    if { [file  pathtype  $link] == "absolute" } {
		set savedvals(script) $link
	    } {
		set savedvals(script) [file dirname $savedvals(script)]/$link
	    }
	} else {
	    break
	}
    }
    # fixup relative paths
    if {[file pathtype $savedvals(script)] == "relative"} {
	set savedvals(script) [file join [pwd] $savedvals(script)]
    }
    set savedvals(scriptdir) [file dirname $savedvals(script) ]
    set savedvals(exedir) [file normalize [file join [file dirname $savedvals(script) ] .. lexe]]
    
    if {$tcl_platform(platform) == "windows"} {
	set savedvals(exesuffix) ".exe"
	set savedvals(currentfile)  icddcurrent.pdftbl
	set savedvals(previousfile) icddprevious.pdftbl
    } else {
	set savedvals(exesuffix) ""
	set savedvals(currentfile) [file nativename ~/icddcurrent.pdftbl]
	set savedvals(previousfile) [file nativename ~/icddprevious.pdftbl]
    }
    set savedvals(debug) 0

    proc progname {prog} {
	return [file join $logic::savedvals(exedir) \
		${prog}$logic::savedvals(exesuffix)]
    }

    proc InitTable {} {
	set logic::savedvals(direction) 1
	set logic::savedvals(hitnum) 0
	set logic::savedvals(numhits) [report -current]
    }

    proc sublist {"num {}"} {
	set prog [logic::progname sublist]
	if {$num == ""} {
	    if {[catch {set list [exec $prog]} errmsg]} {
		if {$logic::savedvals(debug)} {error $errmsg}
		return
	    }
	    return [split $list "\n"]
	} else {
	    if {[catch {set list [exec $prog $num]} errmsg]} {
		if {$logic::savedvals(debug)} {error $errmsg}
		return
	    }
	    return $list
	}
    }

    proc loadsub {args} {
	set result {}
	set prog [logic::progname loadsub]
	if {[catch {set result [eval exec [list $prog] \
		$logic::savedvals(currentfile) $args]} errmsg]} {
	    if {$logic::savedvals(debug)} {error $errmsg}
	    return
	}
	logic::InitTable
	return $result
    }

    proc storemap {} {
	if {[file exists $logic::savedvals(currentfile)]} {
	    file copy -force \
		    $logic::savedvals(currentfile) \
		    $logic::savedvals(previousfile)
	} else {
	    if {$logic::savedvals(debug)} {error "No current table exists"}
	}
    }

    proc restoremap {} {
	if {[file exists $logic::savedvals(previousfile)]} {
	    file rename -force \
		    $logic::savedvals(previousfile) \
		    $logic::savedvals(currentfile) 
	    logic::InitTable
	} else {
	    if {$logic::savedvals(debug)} {error "No saved table exists"}
	}
    }


    proc report {arg} {
	set prog [logic::progname report]
	if {$arg == "-total"} {
	    if {[catch {set num [exec $prog -total]} errmsg]} {
		if {$logic::savedvals(debug)} {error $errmsg}
		return
	    }
	} else {
	    if {![file exists $logic::savedvals(currentfile)]} {
		if {$logic::savedvals(debug)} {error "no current table exists"}
		return ""
	    }
	    if {[catch {set num [exec $prog \
		    $logic::savedvals(currentfile) $arg]} errmsg]} {
		if {$logic::savedvals(debug)} {error $errmsg}
		return
	    }
	}
	return [split $num "\n"]
    }

    proc  element  {args} {
	set result {}
	set prog [logic::progname element]
	if {[catch {set result \
		[eval exec [list $prog] \
		$logic::savedvals(currentfile) $args]} errmsg]} {
	    if {$logic::savedvals(debug)} {error $errmsg}
	    return
	}
	logic::InitTable
	return $result
    }

    #  Implement the nexthit command using hit2seq. Command syntax:
    #    nexthit -first        sets scroll direction to + and returns the 
    #                          sequence number of the first entry
    #    nexthit -last         sets scroll direction to - and returns the
    #                          sequence number of the last entry
    #    nexthit -forward      returns the sequence number of the next entry
    #                          after the current entry (scroll direction
    #                          is ignored)
    #                          in the +direction even after nexthit -last
    #    nexthit -backward     returns the sequence number of the entry
    #                          preceeding the current entry (scroll direction
    #                          is ignored)
    #    nexthit <num>         returns the sequence number of the <num>th entry
    #    nexthit               returns the sequence number of the next 
    #                          entry in the scroll direction
    #  A negative value is returned if there are no more entries.
    proc  nexthit {"arg {}"} {
	
	set prog [logic::progname hit2seq]

	if {$arg == ""} {
	    if {$logic::savedvals(direction) < 0} {
		set arg -backward
	    } else {
		set arg -forward
	    }
	}	

	if {$arg == "-first"} {
	    set logic::savedvals(direction) 1
	    set logic::savedvals(hitnum) 1
	    if {$logic::savedvals(numhits) < 1} {
		return -1
	    }
	} elseif {$arg == "-last"} {
	    set logic::savedvals(direction) -1
	    set logic::savedvals(hitnum) $logic::savedvals(numhits)
	    if {$logic::savedvals(numhits) < 1} {
		return -1
	    }
	} elseif {$arg == "-forward"} {
	    if {$logic::savedvals(hitnum) < $logic::savedvals(numhits)} {
		incr logic::savedvals(hitnum)
	    } else {
		set logic::savedvals(hitnum) 0
		return -1
	    }
	} elseif {$arg == "-backward"} {
	    if {$logic::savedvals(hitnum) > 1} {
		incr logic::savedvals(hitnum) -1
	    } elseif {$logic::savedvals(hitnum) == 0} {
		set logic::savedvals(hitnum) $logic::savedvals(numhits)
	    } else {
		set logic::savedvals(hitnum) 0
		return -1
	    }
	} else {
	    # is this a valid number?
	    scan $arg %d num
	    if {$num == ""} {
		return -1
	    }
	    if {$num < 1 || $num > $logic::savedvals(numhits)} {
		set logic::savedvals(hitnum) 0
		return -1
	    } else {
		set logic::savedvals(hitnum) $num
	    }
	}
	return [exec $prog $logic::savedvals(currentfile) \
		$logic::savedvals(hitnum) $logic::savedvals(hitnum)]
    }

    proc  savemap {arg} {
	file copy -force $logic::savedvals(currentfile) $arg    
    }
    proc  loadmap {file logic} {
	set prog [logic::progname cmbfiles]
	if {$logic == "-replace"} {
	    file copy -force $file $logic::savedvals(currentfile)
	} else {
	    exec $prog $logic::savedvals(currentfile) $file $logic
	}
	logic::InitTable
    }

    proc  getpdf1 {args} {
	set prog [logic::progname getpdf1]
	return [split [eval exec [list $prog] $args] "\n"]
    }

    proc  getpdf2 {args} {
	set prog [logic::progname getpdf2]
	return [split [eval exec [list $prog] $args] "\n"]
    }

    proc  elemcount {args} {
	set prog [logic::progname elemcount]
	eval exec [list $prog] $logic::savedvals(currentfile) $args
	logic::InitTable
	return
    }

    proc  findpeak {args} {
	set prog [logic::progname findpeak]
	if {[catch {
	    eval exec [list $prog] $logic::savedvals(currentfile) $args
	} errmsg]} {
	    set msg "Search aborted due to error:\n\n"
	    append msg $errmsg
	    tk_dialog .warn "Error occurred" $msg {} 0 Continue
	} elseif {$errmsg != ""} {
	    set msg "Warning generated in search:\n\n"
	    append msg $errmsg
	    tk_dialog .warn "Warning occurred" $msg {} 0 Continue
	}
	logic::InitTable
	return
    }

    proc  findnumber {args} {
	set prog [logic::progname findnumber]
	eval exec [list $prog] $logic::savedvals(currentfile) $args
	logic::InitTable
	return
    }

    proc findstring {args} {
	set prog [logic::progname findstring]
	eval exec [list $prog] $logic::savedvals(currentfile) $args
	logic::InitTable
	return
    }

    # create a list of h, k & l values from the PDF-2 database
    proc hkldilist {seqno} {
	set seqno
	set pdf2rec [getpdf2 -seq $seqno]
	foreach v {d i  h  k  l} {set $v {}}
	foreach line $pdf2rec {
	    # look for I records only
	    if {[string range $line 79 79 ] == "I"} {
		foreach j {0 23 46} jj {22 45 68} {
		    set hkldi [string range $line $j $jj]
		    if {[string trim $hkldi] != ""} {
			foreach s {0 7 11 14 17} \
				e {6 9 13 16 19} \
				v {d i  h  k  l} {
			    lappend $v [string trim [string range $hkldi $s $e]]
			}
		    }
		}
	    }
	}   
	return [list $d $i $h $k $l]
    }

    namespace export elemcount element findnumber findpeak getpdf1 getpdf2 \
	    loadmap loadsub nexthit report restoremap savemap storemap \
	    sublist findstring hkldilist

    set status unexpected
    # is the ICDD database installed?
    if {[catch {exec [logic::progname loadsub]} errmsg]} {
	# should always produce an error, but did the database fail to load?
	if {[string first "error initializing" $errmsg] == -1} {
	    set status OK
	} else {
	    set status fail
	    set reason "loadsub did not initialize"
	}
    }
    
    # initialize to last result or all entries if no file found
    set result {}
    if {$status == "OK" && [file exists $logic::savedvals(currentfile)]} {
	set result [logic::InitTable]
	# if InitTable failed, perhaps the old file is incompatible
	if {$result == ""} {set result [loadsub -all]}
    } elseif {$status == "OK"} {
	set result [loadsub -all]
    }
    # if there was no return value, there was an error running the 
    # logic programs
    if {$result == ""} {
	set status error
	set reason "loadsub -all failed"
    }
}
#namespace import logic::*
package provide logic 2.0
