/* =============================================== */
/* Descriptions of the top-level LOGIC subroutines */
/* =============================================== */
/* note mapping of names in logicmap.h */ 
#include "logicmap.h"

/*  Routine to parse element list arguments */
int ParseElementList (const char * strg, INTEGER4 *elem, INTEGER4 *group);

/* Routine to parse a list of integers */
int ParseIntegerList (const char * arg, INTEGER4 **array);

/* Routine to parse a list of floating point numbers */
int ParseRealList (const char * arg, REAL4 **array);

/* psl_init
	Initialize the arrays and open files.
	Returns 0 if successful
	*/
INTEGER4 PSL_INIT(void);

/* psl_tblload
	Reads a predefined subfile and applies logic 
	    ilogic is a number between 0 and 6
		0	 Match with table
		1	 Add to table
		2	 Remove from table
		3	 Subfile without current table
		4	 Replace table with subfile
		5	 Clear all entries from table
		6	 Set table with all entries
	    isubfile is a number between 1 and ?: see psl_tbllist
	Returns number of hits if successful, <0 if not
	*/
INTEGER4 PSL_TBLLOAD(INTEGER4 * ilogic, INTEGER4 *isubfile);

/* psl_tbllist
	Sets string to a description of subfile nsub. 
	Returns the record number of the subfile or zero if no 
	subfile is defined. Returns -1 if there are no more defined
	subfiles.
	*/	
INTEGER4 PSL_TBLLIST(INTEGER4 * nsub, CHAR * string);


/* psl_elemsub
	Create a table of entries based on chemical constraints.
	Elements are specified in two arrays (reqelem[103], optelem[103])
	and element groups are specified in two arrays (reqgrp[18], 
	optgrp[18]). Variables reqval & optval determine how the 
	elements are combined; ilogic determines how the new table 
	is combined with the previous table.
	    reqval = 0 all elements in reqelem & reqgrp are required
		   = 1 at least 1 element in reqelem & reqgrp is required
	    optval = 0 all elements are allowed (ignore optelem & optgrp)
		   = 1 only the selected elements are allowed
	    ilogic 
		0	 Match new table against old table
		1	 Add new table to old table
		2	 Remove new table from old table
		3	 Remove old table from new table
		*/
INTEGER4 PSL_ELEMSUB(INTEGER4 * ilogic, INTEGER4 * reqval, INTEGER4 * optval,
		 INTEGER4 * reqelem, INTEGER4 * reqgrp, INTEGER4 * optelem, 
		 INTEGER4 * optgrp);

/* psl_elemcount
	Create a table of entries based on constraints om the 
	number of unique element in an entry's chemical 
	composition.

	count[i] is 1 if element with i elements are to be used, 
	            0 otherwise
	
	    ilogic 
		0	 Match new table against old table
		1	 Add new table to old table
		2	 Remove new table from old table
		3	 Remove old table from new table
		*/
INTEGER4 PSL_ELEMCOUNT(INTEGER4 * ilogic, INTEGER4 * count);

/* psl_peaksub
      locate entries based on peak positions.

 numpeaks is the number of d-spaces in peaklist with search window in errlist
 ilogic determines the logic to be applied to the tables
	0	 Match with table
	1	 Add to table
	2	 Remove from table
 
 allflag  = 0 all specified peaks must be located; = 1 ok to locate any 1 peak

 intlevel = 00-99 -- use dspXX.bit index file 
          = 100 -- use 3-strongest

 fastflag = 0 fast search; = 1 exact search

 wave     = 0 original input as d-spaces
          < 0 original input as Q
          > 0 original input as 2theta angles
*/
INTEGER4 PSL_PEAKSUB(INTEGER4 * numpeaks, REAL4 * peaklist, REAL4 * errlist,
		 INTEGER4 * allflag, INTEGER4 * intlevel, INTEGER4 * fastflag, 
		 INTEGER4 * ilogic, REAL4 * wave);

/* psl_setbynumber
	Create a table of entries based on a list of entries

	numberlist[i] is the list of entry numbers, where there are 
          numberofentries values in the array
	
	    ilogic 
		0	 Match new table against old table
		1	 Add new table to old table
		2	 Remove new table from old table
		3	 Remove old table from new table

        returns the number of entries that are located, which will 
        not agree with numberofentries if there are invalid numbers
        or duplicates in the list.
		*/
INTEGER4 PSL_SETBYNUMBER(INTEGER4 * ilogic, INTEGER4 * numberlist, INTEGER4 * numberofentries);

/* psl_setbyrange
	Create a table of entries based on a range of entries

	All entries between num1 and num2 will be included.
	
	    ilogic 
		0	 Match new table against old table
		1	 Add new table to old table
		2	 Remove new table from old table
		3	 Remove old table from new table

        returns the number of entries that are located, or an error
	if there are no entries
		*/
INTEGER4 PSL_SETBYRANGE(INTEGER4 * ilogic, INTEGER4 * num1, INTEGER4 * num2);

/* psl_rephits
	Returns the number of hits in the current table for option=0 
	or total entries for option = 1
	*/
INTEGER4 PSL_REPHITS(INTEGER4 * option);

/* psl_rephistory
	Sets string to the contents of history line # iline. 
	Returns 0 if iline is out of range, 1 otherwise.
	*/
INTEGER4 PSL_REPHISTORY(INTEGER4 * iline, CHAR * string);

/* psl_nexthit
	Returns the sequence number of the next hit in the current 
	table following (direction=0) or preceeding (direction=1)
	the current sequence number (iseq). Use iseq=0 to obtain the 
	first (direction=0) or last (direction=1) entry. Returns
	-1 if there are no more entries (iseq is the last/first entry)
	Variable hitnum is set to the index of the entry (1 to nentries).
	*/
INTEGER4 PSL_NEXTHIT(INTEGER4 * iseq, INTEGER4 * hitnum, INTEGER4 * direction);

/* psl_reppdf1
	Reports the PDF1 information corresponding to an entry
	Returns a negative number on error, 0 if successful.

	  (input)
		nseq 		sequence number 
	  (output)
		ncrd		JCPDS entry number iijjjj (ii = set #)
		ndsp		number of peaks
		INT[255]	peak intensity
		DSP[255]	peak d-space
		formula[256]	chemical formula
		MINNAM[256]	mineral name
		COMMNAM[256]	common name
		CHEMNAM[256]	chemical name
		year		year of publication
		coden[7]	journal coden
		VOLUME[6]	volume number
		page[6]		page number
		AUTHOR[256]	author(s) name
		*/
INTEGER4 PSL_REPPDF1(INTEGER4 * nseq, INTEGER4 * ncrd, INTEGER4 * ndsp, 
		 INTEGER4 * INT, REAL4 * DSP, CHAR * formula, CHAR * MINNAM, 
		 CHAR * COMMNAM, CHAR * CHEMNAM, INTEGER4 * year, 
		 CHAR * coden, CHAR * VOLUME, CHAR * page, CHAR * AUTHOR);

/* psl_reppdf2
	Report the PDF2 data as an AIDS*83 record. The subroutine
	must be first called with iline=0 to cause the record to 
	be loaded. Subsequent calls report the iline'th line in 
	the buffer.

	Returns the number of lines in the record (nlines) or -1 for 
	iline=0; returns 81 (number of characters in line or 0 
	if no more lines (iline> nline).
	*/
INTEGER4 PSL_REPPDF2(INTEGER4 * nseq, INTEGER4 * iline, CHAR * line, 
		     CHAR * format, CHAR * filename);

/* psl_savebitmap
	Saves the current bitmap in a sequential file named "filename".

	Returns 0 if successful, <0 if not.
	*/
INTEGER4 PSL_SAVEBITMAP(CHAR * filename);
	
/* psl_readbitmap
	Reads the current bitmap from a sequential file named "filename".

	Returns 0 if successful, <0 if not.
	      ilogic
		0	 Match file with table
		1	 Add file to table
		2	 Remove file from table
		3	 File without current table
		4	 Replace table with file
		*/
INTEGER4 PSL_READBITMAP(CHAR * filename, INTEGER4 * ilogic);

/* psl_peakfromd
    convert peak from d-space to two-theta or Q
       wave = 0 -- return d-space
       wave < 0 -- return Q
       wave > 0 -- return 2theta @ wave
     if conversion is not possible, return a negative value.
     */
REAL4 PSL_PEAKFROMD(REAL4 * dspace, REAL4 * wave);

/* psl_peaktod
    convert peak to d-space from two-theta or Q
       wave = 0 -- return d-space unchanged
       wave < 0 -- convert from Q to d-space
       wave > 0 -- convert from 2theta @ wave
     if conversion is not possible, return a negative value.
     if indflag is 0 (true) check if d-spaces are in range as index entries
     */
REAL4 PSL_PEAKTOD(REAL4 * dspace, REAL4 * wave, INTEGER4 * indflag);
				
/*  Stores the current table for potential reuse in the undo command
*/
void PSL_STOREBITMAP(INTEGER4);

/*  Restores the previous current table (undo command)
*/
void PSL_RESTOREBITMAP(INTEGER4);


/* psl_strngsrch
   loop through all entries and search for strings
		*/
INTEGER4 PSL_STRNGSRCH(INTEGER4 *, char *, char **, INTEGER4 *, int *);
