#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "logiclib.h"

/*  Program getpdf1

    This program returns the requested information for the entry # <int>.

  program arguments:
    getpdf1 -seq <int> [-common] [-mineral] [-formula] [-chemname] 
                       [-reference] [-entry]
                       [-peaks [-wave <lambda> | -q ] -maxi ]
  The options listed below determine the information that is reported:
    -common        common name
    -mineral       mineral name
    -formula       chemical formula
    -chemname      chemical name
    -reference     literature reference
    -entry         JCPDS entry number in form ssnnnn where ss is set number
    -peaks         show peaks as d-spaces/q/2theta and intensities 
  These options affect how the peaks are reported:
    -wave <lambda> peaks as 2theta angles instead of d-spaces
    -q             peaks as Q values (2 pi/d-space) instead of d-spaces
    -maxi          peaks sorted by maximum intensity rather than d-space

    The program returns the requested results in the order
    they are specified on the command line, each result on a separate line.
*/

int main (int argc, char *argv[])
{
  INTEGER4 iseq, ncrd, ndsp, maxi;
  INTEGER4 inten[255];
  REAL4 dsp[255],f;
  CHAR formula[256], minnam[256], commnam[256], chemnam[256];
  INTEGER4 year;
  CHAR coden[7], volume[6], page[6], author[256];
  REAL4 wave;

  int error,i,j,k;
  INTEGER4 l;
  char string[2550]; /* need room for 255 peaks */
  char string1[50];

  error = 0;
  wave = 0.0;
  iseq = -1;
  maxi = 0;

  if (PSL_INIT() != 0) {
    fprintf(stderr,"error initializing the ICDD database\n");
    error = 1;
  }

  i=1;
  /* Pass one -- look for some qualifiers we will need 1st */
  while (i<argc) {
    if (i+1 < argc && !strcmp(argv[i],"-seq")) {
      INTEGER4 l = 1;
      /*  now get the sequence number */
      i++;
      iseq = strtol(argv[i], NULL, 10);
      if (iseq < 1 || iseq > PSL_REPHITS(&l)) {
	fprintf(stderr,"error: invalid sequence number %s\n", argv[i]);
	error = 1;
      }
    } else if (i+1 < argc && !strcmp(argv[i],"-wave")) {
      i++;
      if (wave != 0) {
	fprintf(stderr,
		"error: -wave repeated or in conflict with -q\n");
	error=1; /* flag error if there is a conflict */
      }
      wave = strtod(argv[i], NULL);
      if (wave <= 0) {
	fprintf(stderr,"error: invalid wavelength %s\n", argv[i]);
	error = 1;
      }
    } else if (i < argc && !strcmp(argv[i],"-q")) {
      if (wave != 0.0) {
	fprintf(stderr,
		"error: -wave or -q repeated or in conflict\n");
	error = 1; /* flag error if there is a conflict */
      }
      wave = -1.;
    } else if (i < argc && !strcmp(argv[i],"-maxi")) {
      maxi = 1;
    } else if (strcmp(argv[i],"-peaks") && strcmp(argv[i],"-reference") &&
	       strcmp(argv[i],"-formula") && strcmp(argv[i],"-chemname") &&
	       strcmp(argv[i],"-mineral") && strcmp(argv[i],"-common") &&
	       strcmp(argv[i],"-entry")) {
      fprintf(stderr,"unknown option: %s\n",argv[i]);
      error = 1;
    }
    i++;
  }

  if (iseq == -1) {
    fprintf(stderr,"error: -seq and sequence number missing\n");
    error = 1;
  }

  /*  Now start reporting requested info */
  if (!error) {
    l = PSL_REPPDF1(&iseq, &ncrd, &ndsp, inten, dsp, formula, minnam, 
		    commnam, chemnam, &year, coden, volume, page, author);
    if (l < 0) {
      fprintf(stderr,"error processing request in PSL_REPPDF1\n");
      error = 1;
    }
  }

  /* review the qualifiers again and process as we go */
  i=1;
  while (i<argc && !error) {
    if (!strcmp(argv[i],"-seq")) /* skip over */
      i++;
    else if (!strcmp(argv[i],"-wave")) /* skip over */
      i++;
    else if (!strcmp(argv[i],"-q")) /* skip over */
      {}
    else if (!strcmp(argv[i],"-maxi")) /* skip over */
      {}
    else if (!strcmp(argv[i],"-peaks")) { /* list all peaks */
      strcpy(string, "");
      /* sort by d-spaces */
      if (maxi == 1) {
	for (j=0; j<ndsp-1; j++) {
	  for (k=j+1; k<ndsp; k++) {
	    if (inten[j] < inten[k]) {
	      f = dsp[j];
	      dsp[j] = dsp[k];
	      dsp[k] = f;
	      l = inten[j];
	      inten[j] = inten[k];
	      inten[k] = l;
	    }
	  }
	}
      }
      for (j=0; j<ndsp; j++) {
	if (wave == 0) {
	  sprintf(string1, "%.3f %d ",dsp[j], inten[j]);
	  strcat(string,string1);
	} else if (wave < 0) {
	  REAL4 temp;
	  if (dsp[j] != 0.0)
	    {
	      temp = 2 * 3.14159 / dsp[j];
	      sprintf(string1, "%.3f %d ", temp, inten[j]);
	      strcat(string,string1);
	    }
	} else {
	  REAL4 temp=0.;
	  temp = PSL_PEAKFROMD(&dsp[j],&wave);
	  if (temp > 0.) {
	    sprintf(string1, "%.2f %d ",temp, inten[j]);
	    strcat(string,string1);
	  }
	}
      }
      printf("%s\n",string);
    } else if (!strcmp(argv[i],"-reference")) {
      if (strcmp(author,"")) {
	sprintf(string,"%s, %s, %s, p%s (%d)",
		author, coden, volume, page, year);
      } else {
	sprintf(string,"%s, %s, p%s (%d)",
		coden, volume, page, year);
      }
      printf("%s\n",string);
    } else if (!strcmp(argv[i],"-formula")) 
      printf("%s\n",  formula);
    else if (!strcmp(argv[i],"-chemname"))
      printf("%s\n",  chemnam);
    else if (!strcmp(argv[i],"-mineral"))
      printf("%s\n",  minnam);
    else if (!strcmp(argv[i],"-common"))
      printf("%s\n",  commnam);
    else if (!strcmp(argv[i],"-entry")) {
      if (ncrd < 0) 
	sprintf(string,"%dD",-ncrd);
      else
	sprintf(string,"%d",ncrd);
      printf("%s\n",  string);
    } else {
      fprintf(stderr,"unknown option: %s\n",argv[i]);
      error = 1;
    }
    i++;
  }

  if (error) {
    fprintf(stderr, "\nusage:\n");
    fprintf(stderr, "\t getpdf1 -seq <int>\n");
    fprintf(stderr, "\t\t [-common] [-mineral] [-formula] [-chemname]\n");
    fprintf(stderr, "\t\t [-reference] [-entry]\n");
    fprintf(stderr, "\t\t [ -peaks] [-wave <lambda> | -q ] [-maxi]\n");
    return 1;
  }
  return 0;
}
