# read RAW Bruker/Rigaku data files
#
# This read is based on a dissection of the Bruker Format by Tyrel McQueen
# and of the Rigaku format by Vince West.  These only work for continuous
# scans with a single data set, and comes with no warranty! egads
#
# The Rigaku format that this reads is very old, and newer formats may exist,
# but are not yet supported.  Dr. Nikolaos Kourkoumelis, author of POWDll, has
# given me his Rigaku related source code, and in the future, if needed, I can
# modify this routing to support other RAW formats.  email:
# Vince West - barelytone@gmail.com
#
# version 0.000000000001 Nov 8, 2010
#
# ----------------------------------------------------------------
# ------- define a command line option -----------------------------
#  command line option 
lappend command(cmdopt) -binaryRaw
# proc to use 
set command(-binaryRaw)  ReadBinaryRawData
#--------define a dialog box entry
# menu label
lappend command(readtypes) ".RAW (Bruker/Siemens, Rigaku)"
# proc for this entry
lappend command(readproc) ReadBinaryRaw
# allowed data types
if {$tcl_platform(platform) == "windows"} {
    lappend command(filterlist) {raw}
    # definitions for these data types
    set command(ReadBinaryRaw_raw_type) "XY data"
} else {
    lappend command(filterlist) {raw RAW}
    # definitions for these data types
    set command(ReadBinaryRaw_raw_type) "XY data"
    set command(ReadBinaryRaw_RAW_type) "XY data"
}

proc ReadBinaryRaw {file} {
    global command
    if {$file == ""} return
    pleasewait "reading file $file"
    set ret [ReadBinaryRawData $file]
    donewait
    if {$ret != ""} {return $ret}
    showlastentry $command(read_filelist)

}
proc ReadBinaryRawData {filename} {

    global graph

    set lnum 0
    # signal errors by quitting
    if [catch {
	set fp [open $filename r]
		fconfigure $fp -translation binary
		set inBinData [read $fp]
		
		# Bruker Raw Formats are broken down by the following:
		# First seven bytes (a7) = 'RAW1.01' with other date information in the RAW Header
		# next 709 bytes include the rest of the header, and currently unknown stuff
		# at byte 716, there is a 4 byte 'longint' (i1) which is the number of steps in the scan
		# at byte 728, there is an 8 byte 'double' precision (d1) floating point number which is the initial 2Theta Value
		# at byte 888, there is an 8 byte 'double' (d1) - 2theta step size
		# at byte 1016 begins the data stored as 4 byte 'single' precision (f$numberOfSteps) floating point numbers
		# the following are the byte - value pairs for radition wavelength as double floats
		#		617 - 1.54184 (weighted average)
		#		625 - 1.5406 (Cu K-alpha1)
		#		633 - 1.54439 (Cu K-alpha2)
		
		# Rigaku Raw Formats are broken down by the following:
		# Bytes 1-64 are the header  (8-bit ASCII chars : a)
		# Bytes 81-84 are the End 2Theta (32 Bit Float : f)
		# Bytes 85-88 are the scan speen (deg / min) (32 Bit Float : f)
		# Bytes 137-140 are the Start 2Theta (32 Bit Float : f)
		# Bytes 141-144 are the step size (32 Bit Float : f)
		# Bytes 145-146 are the number of Steps (16 Bit Int : s)
		# Bytes 120, 124, 128, 132 are the 4 different values for Cu K-alpha wavelength (32 Bit Float : f)
		# 		121 - 1.54187095165 (weighted average)
		# 		125 - 1.54059803486 (Cu K-alpha1)
		# 		129 - 1.54441797733 (Cu K-alpha )
		# 		133 - 1.5390625

	binary scan $inBinData a7 brukerHeader
	binary scan $inBinData a4 rigakuHeader
	
	if { $brukerHeader == "RAW1.01" } {
		# currently, I know there are at least 40 bytes of header information (as determined from Rigaku Raw conversion),
		# but I don't know if there are more, so we'll cut it off at 40
		binary scan $inBinData @326a40 sampleTitle
			set sampleTitle [ string trim $sampleTitle ]
		binary scan $inBinData @616ddd wavelengthAve wavelength1 wavelength2
		# round off wavelengths to the fourth decimal place (stupid floats)
			set wavelengthAve [expr round($wavelengthAve*10000)/10000.0]
			set wavelength1 [expr round($wavelength1*10000)/10000.0]			
			set wavelength2 [expr round($wavelength2*10000)/10000.0]
		binary scan $inBinData @716i numberOfSteps
			if { $numberOfSteps == "" || $numberOfSteps < 0 || $numberOfSteps > 14000} { return "the file has '$numberOfSteps' steps, Probably not a Bruker Raw format" }
		binary scan $inBinData @728d init2Theta
			if { $init2Theta < 0.0 || $init2Theta > 140.0 } { return "initial 2Theta is '$init2Theta', probably not a Bruker Raw Format" }
		binary scan $inBinData @888d stepSize
			if { $stepSize < 0.001 || $stepSize > 1.0 } { return "step size is '$stepSize', probably not a Bruker Raw Format" }
		binary scan $inBinData @1016f$numberOfSteps counts
	} elseif { $rigakuHeader == ".RAW" } {
		binary scan $inBinData @24a40 sampleTitle
			set sampleTitle [ string trim $sampleTitle ]
		binary scan $inBinData @80ff end2Theta scanSpeed
		binary scan $inBinData @120fff wavelengthAve wavelength1 wavelength2
		# round off wavelengths to the fourth decimal place (stupid floats)
			set wavelengthAve [expr round($wavelengthAve*10000)/10000.0]
			set wavelength1 [expr round($wavelength1*10000)/10000.0]			
			set wavelength2 [expr round($wavelength2*10000)/10000.0]
		binary scan $inBinData @136ffs init2Theta stepSize numberOfSteps
			if { $init2Theta < 0.0 || $init2Theta > 140.0 } { return "initial 2Theta is '$init2Theta', probably not a Rigaku Raw Format" }
			if { $stepSize < 0.001 || $stepSize > 1.0 } { return "step size is '$stepSize', probably not a Rigaku Raw Format" }
		binary scan $inBinData @200f$numberOfSteps counts
	} else {
		return "RAW Header: '$brukerHeader' is not recognized. Currently, only Bruker (header: RAW1.01) and Rigaku (header: .RAW) formats are supported.  This file may be a newer format.  Please contact the developer (info inside comments of 'read_binaryRaw.tcl')"
	}
		
	set data [file root [file tail $filename]] 
	# eliminate spaces from the name
	regsub -all " " $data "_" data

	# Great!! Now we are positioned to read the data.

	# Initialize the data structure.
	set data [initdata $data]
	global ${data}
	set ${data}(xlabel) "2theta"
	set ${data}(xunits) "2theta"
	set ${data}(ylabel) "counts"
	set ${data}(title) $sampleTitle
	
	# Now, read the rest of the file.
	# until we get to the last data point, an end-of-file or error
	set i 0
	set xdata ""
		
	foreach count $counts {
		# now we set the 2-theta values, rounding off to the thousandth's place
		# in order to avoid step size of 0.02999999995748
		lappend xdata [expr round([expr $i*$stepSize+$init2Theta]*1000)/1000.0]
		incr i
	}
	
	set ${data}(x) $xdata
	set ${data}(y) $counts
	
	if { $wavelength1 < 0.01 && $wavelength1 > -0.01 } {
		if { $wavelengthAve < 0.01 && $wavelengthAve > -0.01 } {
		} else { set ${data}(wavelength) $wavelengthAve }
	} else {
		set ${data}(wavelength) $wavelength1
		if { $wavelength2 < 0.01 && $wavelength2 > -0.01 } {
		} else { set ${data}(wavelength2) $wavelength2 }
	}

    } errmsg] {
	# try to process the data we have read
	catch {resetdata $data}
	return "XY Data error. Error reading: $errmsg"
    }
    resetdata $data
    lappend graph(plotlist) $data
    catch {close $fp} 
    #puts "$ptsread valid points read"
    return {}
}
 
#-- 
#Alistair McGhie [DuPont Exp. Sta.  Wilm. De. 19880-0228]

