# this script is designed to be used with the liveplot tcl script in the 
# gsasgui package. It displays entries from the ICDD/JCPDS database on
# top of the GSAS difference plot.
#
# Brian Toby (NIST) completely revised 10/2003
#

# list of files needed by LOGIC
set RequiredFileList {logicpkg.tcl periodic.tcl vlistbox.tcl logicGUI.tcl}
if {$tcl_platform(platform) == "windows"} {
    set EXESUFFIX ".EXE"
} else {
    set EXESUFFIX {}
}
lappend RequiredFileList \
	sublist$EXESUFFIX loadsub$EXESUFFIX report$EXESUFFIX  \
	element$EXESUFFIX hit2seq$EXESUFFIX getpdf1$EXESUFFIX \
	getpdf2$EXESUFFIX findpeak$EXESUFFIX elemcount$EXESUFFIX \
	findnumber$EXESUFFIX findstring$EXESUFFIX  cmbfiles$EXESUFFIX

# where are we?
set scriptname [info script]
# translate links -- go six levels deep
foreach i {1 2 3 4 5 6} {
    if {[file type $scriptname] == "link"} {
	set link [file readlink $scriptname]
	if { [file  pathtype  $link] == "absolute" } {
	    set scriptname $link
	} {
	    set scriptname [file dirname $scriptname]/$link
	}
    } else {
	break
    }
}
set logicloc [file dirname $scriptname]


# search for a file with the ICDD database location
set filelist {}
# assemble search path
catch {lappend filelist $::env(LOGIC_LOC)}
catch {lappend filelist [file join $::env(HOME) .icdd_files_loc]}
lappend filelist /usr/local/icdd_files_loc.txt 
lappend filelist [file join $logicloc icddloc.txt]
lappend filelist C:/icddloc.txt
# look through the search path
foreach file $filelist {
    if {[file exists $file]} {
	set fp [open $file r]
	# the first line contains the location of the logic database files
	set diskpath [gets $fp]
	close $fp
	lappend logicloc [file dirname $diskpath]
	break
    }
}


foreach loc $logicloc {
    set missing {}
    set CMPR_OK 1
    foreach file $RequiredFileList {
	if {![file exists [file join $loc $file]]} {
	    set CMPR_OK 0
	    lappend missing $file
	    continue
	}
    }
    if {$CMPR_OK} {
	set logicloc $loc
	break
    }
}

if {!$CMPR_OK} {
    catch {puts "LOGIC files not found in ${loc}:\n$missing"}
}


proc MakeLogicWin {} {
    global graph 

    pleasewait "Loading Logic software"
    update
    if {[catch {set logic::status}]} {
	# finish loading Logic (gets done once)
	namespace eval logicGUI {
	    set settings(home) [file dirname [file join $::logicloc logicGUI.tcl]]
	    set settings(ymultICDD) 1.0
	    set settings(icddpage) .icdd
	}
	uplevel {
	    source [file join $::logicloc logicGUI.tcl]
	    #source [file join $::logicloc ../data.tcl]
	}
	set ::graph(pdflist) {}
	trace variable graph(xunits) w ::logicGUI::ICDDsetxunits
	if {[string first lblICDD [bind . H]] == -1} {
	    bind . <Key-h> "+lblICDD $graph(blt) %x"
	    bind . <Key-H> "+lblICDD $graph(blt) %x"
	}
    }
    catch {destroy $::logicGUI::settings(icddpage)}
    toplevel $::logicGUI::settings(icddpage)
    
    grid [frame $::logicGUI::settings(icddpage).a] \
	    -row 0 -column 0 -sticky news
    grid [frame $::logicGUI::settings(icddpage).b] \
	    -row 1 -column 0 -sticky news

    # name the index window
    set ::logicGUI::settings(indexwin) $::logicGUI::settings(icddpage).b
    ::logicGUI::CreateLogicEmbWin $::logicGUI::settings(icddpage).a liveplot
    foreach num $::graph(pdflist) {
	$::graph(blt) element delete $num
    }
    set ::graph(pdflist) {}
    ::logicGUI::ICDDsetxunits
    donewait
    tkwait window $::logicGUI::settings(icddpage)
    foreach num $::graph(pdflist) {
	$::graph(blt) element delete $num
    }
    set ::graph(pdflist) {}
}


namespace eval logicGUI {
    proc ICDDsetxunits {args} {
	variable settings
	set mode disabled
	set color gray
	if {$::graph(xunits) == 2} {
	    set settings(peaktype) "-q"
	} elseif {$::graph(xunits) == 1} {
	    set settings(peaktype) ""
	} else {
	    set settings(peaktype) "-wave"
	    if {[catch {expr $settings(wave)}]} {set settings(wave) 1.5418}
	    set mode normal
	    set color black
	}
	catch {
	    foreach w [winfo children $settings(liveplot_wave_box)] {
		catch {$w configure -state $mode}
		catch {$w configure -fg $color}
	    }
	}
	# recompute plots here
	foreach data $::graph(pdflist) {
	    global $data
	    # vector names
	    set ttv [set ${data}(ttvector)]
	    set xv [set ${data}(xvector)]
	    set yv [set ${data}(yvector)]
	    # get list of dspaces & heights
	    set dlist [set ${data}(x)]
	    set hlist [set ${data}(heights)]
	    set tt {}
	    set x {}
	    set y {}
	    
	    # convert to liveplot units (always from d-spaces)
	    if {$::graph(xunits) == 2} {
		foreach d $dlist h $hlist {
		    set xi 0
		    catch {
			set xi [expr 6.28318530718/$d]
			lappend y 0 $h 0
			lappend x $xi $xi $xi
		    }
		    lappend tt $xi
		}
		set ${data}(cxlabel) "Q"	
	    } elseif {$::graph(xunits) != 1} {
		set torad [expr asin(1) / 180.]
		set wo2 [expr $settings(wave) / 2.]
		foreach d $dlist h $hlist {
		    set xi 180
		    catch {
			set xi [expr asin( $wo2/$d )/$torad ]
			lappend y 0 $h 0
			lappend x $xi $xi $xi
		    }
		    lappend tt $xi
		}
		set ${data}(cxlabel) "2 Theta @ $settings(wave)"
	    } else {
		set tt $dlist
		foreach xi $dlist h $hlist {
		    lappend y 0 $h 0
		    lappend x $xi $xi $xi
		}
		set ${data}(cxlabel) [set ${data}(xlabel)]
	    }
	    $ttv set $tt
	    $xv set $x
	    $yv set $y
	    $xv notify now
	}
    }
    proc Logic2liveplot {} {
	variable settings
	set ns [namespace current]
	set c 0
	set ::graph(pdflist) {}
	foreach seq [lsort -integer $Vlistbox::Vlbox(selection)] {
	    set seqno [nexthit $seq]
	    if {$seqno <= 0} {continue}
	    incr c
	    if {$c > 4} {set c 0}
	    set color [lindex {cyan blue magenta green yellow} $c]
	    set pdfnum [getpdf1 -seq $seqno -entry]
	    set list [hkldilist $seqno]
	    foreach {d_list i_list  h_list  k_list  l_list} $list {}

	    set data $pdfnum
	    lappend ::graph(pdflist) $pdfnum
	    global $data
	    # define vectors for the data
	    set ${data}(type) peaks
	    regsub -all  {\.} $data {} cdat
	    regsub -all  {[\*\+-]} $cdat {_} cdat
	    set ${data}(xvector) ${ns}::${cdat}_x
	    set ${data}(yvector) ${ns}::${cdat}_y
	    set ${data}(ttvector) ${ns}::${cdat}_tt
	    global [set ${data}(xvector)] [set ${data}(yvector)] [set ${data}(ttvector)]
	    catch {vector [set ${data}(xvector)]}
	    catch {vector [set ${data}(yvector)]}
	    catch {vector [set ${data}(ttvector)]}
	    # initialize arrays
	    foreach var {x y h k l dspaces widths heights \
		    wavelength xunits xlabel ylabel cxlabel cylabel title spg \
		    extcodes } {
		set ${data}($var) {}
	    }
	    set ${data}(h) $h_list
	    set ${data}(k) $k_list
	    set ${data}(l) $l_list
	    set ${data}(x) $d_list
	    set ${data}(dspaces) $d_list
	    set ${data}(xlabel) "d-space"
	    set ${data}(xunits) "dspace"
	    foreach i $i_list {
		set int 0
		catch {set int [expr $i * $settings(ymultICDD)]}
		lappend ${data}(heights) $int
	    }
	    # make sure that all peak arrays are the same length as the data
	    foreach elem {y widths heights h k l dspaces} {
		for  {set i [llength [set ${data}($elem)]]} \
			{$i < [llength [set ${data}(x)]]} {incr i} {
		    lappend ${data}($elem) {}
		}
	    }
	}
	ICDDsetxunits
	foreach data $::graph(pdflist) {
	    catch {
		$::graph(blt) element create $data \
			-color $color \
			-linewidth 1  -symbol none \
			-xdata [set ${data}(xvector)] \
			-ydata [set ${data}(yvector)]
	    }
	}
    }
}

proc lblICDD {plot x} {
    global command blt_version expgui
    if {![winfo exists $::logicGUI::settings(icddpage)]} {return}

    # look for peaks within pixelregion pixels
    set pixelregion 5
    set xmin [$plot xaxis invtransform [expr $x - $pixelregion]]
    set xmax [$plot xaxis invtransform [expr $x + $pixelregion]]
    set peaklist {}
    foreach data $::graph(pdflist) {
	global $data
	set xnam [set ${data}(ttvector)]
	set peaknums [$xnam search $xmin $xmax]
	set ymax [lindex [$plot yaxis limits] 1]
	set xcen 0    
	foreach peak $peaknums {
	    set big 0
	    foreach i {h k l} {
		set $i [lindex [set ${data}($i)] $peak]
		if {[set $i] < -9 || [set $i] > 9} {set big 1}
	    }
	    set xpos [$xnam range $peak $peak]
	    set xcen [expr $xcen + $xpos]
	    if $big {
		set reflbl $h,$k,$l
		lappend peaklist $h,$k,$l
	    } else {
		set reflbl $h$k$l
		lappend peaklist $h$k$l
	    }
	    if {$expgui(hklbox)} {
		catch {
		    .hkl.txt insert end "\n$data"
		    .hkl.txt insert end "\t$reflbl"
		    .hkl.txt insert end "\t$xpos"
		    .hkl.txt see end
		}
	    }
	}
    }
    if {$peaklist == ""} return
    set xcen [expr $xcen / [llength $peaknums]]
    # avoid bug in BLT 2.3 where Inf does not work for text markers
    if {$blt_version == 2.3} {
	set ycen [lindex [$plot yaxis limits] 1]
    } else  {
	set ycen Inf
    }
    set mark [$plot marker create text -coords "$xcen $ycen" \
	    -rotate 90 -text $peaklist -anchor n -bg ""]
    if {$expgui(fadetime) > 0} {
	catch {
	    after [expr $expgui(fadetime) * 1000 ] \
	    "catch \{ $plot marker delete $mark \}"
	}
    }
}
