# note: hklgen bombs without an error message if the cell is too large and 
# the d-space limit is too high. This needs an error message.
#
# EditCell program flow: 
#    SetEditData is used to control the contents of the menu
#       editnew to setup for new computation
#            ComputeNewCell is called to process a new cell
#       editold is used for existing cells
#            the extinctions conditions are set/reset & displayed
#            data & peaks are displayed
#            the range for the cell sliders is set
#            RecalcLoop is started so that cell parameter changes are updated
#                  every 0.1 second as long as the active page is EditCell
#            shift-mouse1 is bound to label hkl values (using lblhkl)

###############################################################################
# Showcell is called by HKLGEN to compute and display the reflections generated
###############################################################################
proc showcell {} {
    global extinctlbl
    global cellparm
    set topbox $cellparm(top)
    set message {}

    foreach var {a b c alpha beta gamma} {
	set val [$topbox.cell.e$var get]
	if {[catch {expr $val}]} {
	    append message "The value '$val' for cell parameter $var is invalid\n"
	} elseif {$val<10} {
	    append line [format %8.5f $val]
	} elseif {$val<100} {
	    append line [format %8.4f $val]
	} else {
	    append line [format %8.3f $val]
	}
    }
    if {[catch {expr $cellparm(wavelength)}]} {
	append message "The value '$cellparm(wavelength)' for wavelength is invalid\n"
    }
    if {[catch {expr $cellparm(twothetamax)*1}]} {
	append message "value '$cellparm(twothetamax)' for 2theta(max) is invalid\n"
    }
    if {$message != ""} {
	tk_messageBox -message $message -type ok -icon warning
	return
    }
    set out [open hklgen.inp w]
    puts $out "Created by hklgen.tcl"
    puts $out $line
    set line {}
    for {set num 1} {$num < 14} {incr num} {
	append line [format %3i [lsearch $extinctlbl($num) $extinctlbl(r$num)]]
    }
    puts $out [format "%9.5f%9.2f%3d%s" \
	    $cellparm(wavelength) $cellparm(twothetamax) $cellparm(laue) $line]
    if {$cellparm(spacegroup) == "(none)"} {
	puts $out ""
    } else {
	puts $out $cellparm(spacegroup)
    }
    close $out
    catch {exec $cellparm(hklgenprog) < hklgen.inp > hklgen.out}
    set out [open hklgen.out r]
    incr cellparm(outcount)
    set top [toplevel .out$cellparm(outcount)]
    set txtbx [text $top.t -relief raised -bd 2 -height 10 -width 50 \
	-wrap none \
	-yscrollcommand "$top.s set" \
	-xscrollcommand "$top.a set"]
    global tcl_version
    if {$tcl_version >= 8.0} {$txtbx config -font Courier}
    scrollbar $top.s -command "$top.t yview"
    scrollbar $top.a -command "$top.t xview" -orient horizontal
    frame $top.bar -relief groove -bd 2
    pack $top.bar $top.a -side bottom -fill x
    pack $top.t -side left -fill both -expand yes
    pack $top.s -side left -fill y 
    pack [button $top.bar.print -text "Save as:" -command "savebox $top"] -side left
    pack [entry $top.bar.file] -side left
    $top.bar.file insert 0 hklgen$top
    pack [button $top.bar.done -text Close -command "destroy $top"] -side right
    $txtbx configure -state normal
    $txtbx delete 1.0 end
    while {[gets $out line] >= 0} {
	$txtbx insert end $line
	$txtbx insert end "\n"
    }
    $txtbx configure -state disabled -width 60 -height 20  
    
    close $out
    file delete -force hklgen.inp hklgen.out
}

# savebox writes the contents of the listbox to a file
proc savebox {top} {
    set out [open [$top.bar.file get] w]
    puts $out [$top.t get 1.0 end]
    close $out
}

# compute two-theta values from hkl values etc.
proc ttcalc {data} {
    global $data cellparm graph
    set base $cellparm(celleditbase)
    set flag 1
    # reevaluate extinctions if symmetry has changed
    if {$cellparm(extspg) != $cellparm(oldextspg)} {
	calcextns $data $cellparm(extspg)
	set cellparm(oldextspg) $cellparm(extspg)
	set flag 0
    }
    # has cell changed? Is it valid?
    foreach var {a b c alpha beta gamma} {
	set value [$base.e$var get]
	set $var $value
	if {[set ${data}(old_$var)] != $value} {set flag 0}
	if [catch {expr $value}] {return}
    }
    # have the peak values or wavelength changed? 
    foreach var {ymin ymax zero} {
	# look for changes
	if {$cellparm($var) != $cellparm(old_$var)} {set flag 0}
	# stop for invalid values
	if [catch {expr $cellparm($var)}] {return}
	# save the value
	set cellparm(old_$var) $cellparm($var)
    }
    if {$graph(units) != 0 && $graph(units) != $cellparm(old_wave)} {
	set flag 0
    } elseif {$graph(units) == 0 && \
	    [set ${data}(wavelength)] != $cellparm(old_wave)} {
	set flag 0
    }
    #-----------------------------
    # no reflections, don't bother
    #-----------------------------
    if {[llength [set ${data}(h)]] == 0} return

    #---------------------------
    # no changes -- don't bother
    #---------------------------
    if $flag {update; return}

    #---------------------------
    # validate wavelength
    #---------------------------
    if {$graph(units) == 0} {
	if [catch {expr [set ${data}(wavelength)]}] {return}
	set cellparm(old_wave) [set ${data}(wavelength)]
    } else {
	set cellparm(old_wave) $graph(units)
    }	

    #---------------------------
    # update the positions
    #---------------------------
    global ${data}
    [set ${data}(xvector)] notify cancel 
    [set ${data}(yvector)] notify cancel 
    set fil [open calc.inp w]
    puts $fil $data
    foreach var {a b c alpha beta gamma} {
	puts -nonewline $fil " [set $var]"
	set ${data}(old_$var) [set $var]
    }
    puts $fil {}
    puts $fil [set ${data}(wavelength)]
    set numdat [llength [set ${data}(dspaces)]]
    # get the limits to draw on the y-axes
    puts $fil "$cellparm(ymin) $cellparm(ymax) $cellparm(zero)"
    puts $fil $graph(units)
    puts $fil $numdat
    puts $fil [set ${data}(h)]
    puts $fil [set ${data}(k)]
    puts $fil [set ${data}(l)]
    puts $fil [set ${data}(extinctions)]
    close $fil
    #set fil [open "$cellparm(calc2tprog) < calc.inp" r]
    catch {exec $cellparm(calc2tprog) < calc.inp > calc.out}
    set fil [open calc.out r]
    set out [read $fil]
    close $fil
    file delete -force calc.inp calc.out

    [set ${data}(xvector)] notify never
    [set ${data}(yvector)] notify never
    xextinct notify never
    yextinct notify never
    xextinct set {}
    yextinct set {}

    eval $out

    # (ttvector) is used for reflection labeling (lblhkl)
    [set ${data}(ttvector)] set [set ${data}(x)]

    [set ${data}(xvector)] notify now
    [set ${data}(yvector)] notify now
    xextinct notify now
    yextinct notify now
    [set ${data}(xvector)] notify always
    [set ${data}(yvector)] notify always
    update
}

proc RecalcLoop {} {
    global command cellparm
    # are we running editcell?
    if {$command(pagenow) != "editcell"} return
    # don't compute new runs
    if {$cellparm(editsetting) == "(new)"} return
    if {$cellparm(celleditvar) != ""} {catch {ttcalc $cellparm(celleditvar)}}
    # cancel any other RecalcLoop
    after cancel RecalcLoop
    after cancel RecalcLoop
    # requeue
    after 100 RecalcLoop
}

# compute extinctions from hkl values & space groups
proc calcextns {data spacegroup} {
    global $data cellparm extinctlbl
    set fil [open tstext.inp w]
    puts $fil $data
    set line {}
    # manual extinction rules
    for {set num 1} {$num < 14} {incr num} {
	append line [format %3i [lsearch $extinctlbl($num) $extinctlbl(r$num)]]
    }
    puts $fil $line
    puts $fil $cellparm(extspg)
    set numdat [llength [set ${data}(dspaces)]]
    puts $fil $numdat
    puts $fil [set ${data}(h)]
    puts $fil [set ${data}(k)]
    puts $fil [set ${data}(l)]
    close $fil
    #set fil [open "$cellparm(tstextprog) < tstext.inp" r]
    catch {exec $cellparm(tstextprog) < tstext.inp > tstext.out}
    set fil [open tstext.out r]
    set out [read $fil]
    close $fil
    file delete -force tstext.inp tstext.out
    eval $out
}

# lblallhkl is called by a binding to shift-left-double-click
# All reflections grouped in regions pixels $pixelregion wide are labeled. 
proc lblallhkl {plot} {
    global cellparm command blt_version tcl_platform tcl_version
    # ignore unless EditCell is showing
    if {$command(pagenow) != "editcell"} return
    # look for peaks within pixelregion pixels
    set data $cellparm(celleditvar)
    if {$data == ""} return
    global $data
    #  regions in pixels
    if {$tcl_platform(platform) == "windows"} {
	set pixelregion 8
    } else {
	set pixelregion 5
    }
    set data $cellparm(celleditvar)
    # get the plot limits in pixels
    set xmin [$plot xaxis transform [lindex [$plot xaxis limits] 0]]
    set xmax [$plot xaxis transform [lindex [$plot xaxis limits] 1]]
    incr xmin $pixelregion
    for {set x $xmin} {$x < $xmax} {incr x $pixelregion} {
	set ttmin [$plot xaxis invtransform [expr $x - $pixelregion]]
	set ttmax [$plot xaxis invtransform [expr $x + $pixelregion]]
	set xnam [set ${data}(ttvector)]
	set peaknums [$xnam search $ttmin $ttmax]
	set peaklist {}
	set ymax [lindex [$plot yaxis limits] 1]
	set xcen 0    
	foreach peak $peaknums {
	    set big 0
	    foreach i {h k l} {
		set $i [lindex [set ${data}($i)] $peak]
		if {[set $i] < -9 || [set $i] > 9} {set big 1}
	    }
	    set xcen [expr $xcen + [$xnam range $peak $peak]]
	    set sym {}
	    if [lindex [set ${data}(extinctions)] $peak] {set sym e}
	    if $big {
		lappend peaklist $h,$k,$l$sym
	    } else {
		lappend peaklist $h$k$l$sym
	    }
	}
	if {$peaklist == ""} continue
	set xcen [expr $xcen / [llength $peaknums]]
	# avoid bug in BLT 2.3 where Inf does not work for text markers
	if {$blt_version == 2.3} {
	    set ycen [lindex [$plot yaxis limits] 1]
	} else  {
	    set ycen Inf
	}
	# older BLT versions can't rotate text in windows
	if {$tcl_platform(platform) == "windows" && \
		($blt_version <= 2.3 || $blt_version == 8.0)} {
	    regsub -all { } $peaklist "\n" peaklist
	    $plot marker create text -coords "$xcen $ycen" \
		    -text $peaklist -anchor n -bg "" -name hkl$x
	} else {
	    $plot marker create text -coords "$xcen $ycen" \
		    -rotate 90 -text $peaklist -anchor n -bg "" -name hkl$x
	}
	if {$tcl_version >= 8.0} {
	    $plot marker config hkl$x -font lblfont
	}
    }
}

# delete all hkl labels
proc delallhkllbl {plot} {
    catch {
	eval $plot marker delete [$plot marker names hkl*]
    }
}

# lblhkl is called by a binding to shift-left-mouse
# it finds all reflections within 10 pixels of the mouse and 
# labels them. The label is deleted after 10 seconds
proc lblhkl {plot x} {
    global cellparm command blt_version tcl_platform tcl_version
    # ignore unless EditCell is showing
    if {$command(pagenow) != "editcell"} return
    # look for peaks within pixelregion pixels
    if {$tcl_platform(platform) == "windows"} {
	set pixelregion 8
    } else {
	set pixelregion 5
    }
    set data $cellparm(celleditvar)
    if {$data == ""} return
    global $data
    set ttmin [$plot xaxis invtransform [expr $x - $pixelregion]]
    set ttmax [$plot xaxis invtransform [expr $x + $pixelregion]]
    set xnam [set ${data}(ttvector)]
    set peaknums [$xnam search $ttmin $ttmax]
    set peaklist {}
    set ymax [lindex [$plot yaxis limits] 1]
    set xcen 0    
    foreach peak $peaknums {
	set big 0
	foreach i {h k l} {
	    set $i [lindex [set ${data}($i)] $peak]
	    if {[set $i] < -9 || [set $i] > 9} {set big 1}
	}
	set xcen [expr $xcen + [$xnam range $peak $peak]]
	set sym {}
	if [lindex [set ${data}(extinctions)] $peak] {set sym e}
	if $big {
	    lappend peaklist $h,$k,$l$sym
	} else {
	    lappend peaklist $h$k$l$sym
	}
    }
    if {$peaklist == ""} return
    set xcen [expr $xcen / [llength $peaknums]]
    # avoid bug in BLT 2.3 where Inf does not work for text markers
    if {$blt_version == 2.3} {
	set ycen [lindex [$plot yaxis limits] 1]
    } else  {
	set ycen Inf
    }
    if {$tcl_platform(platform) == "windows"} {
	# at least right now, text can't be rotated in windows
	regsub -all { } $peaklist "\n" peaklist
	set mark [$plot marker create text -coords "$xcen $ycen" \
		-text $peaklist -anchor n -bg "" -name hkl$xcen]
    } else {
	set mark [$plot marker create text -coords "$xcen $ycen" \
		-rotate 90 -text $peaklist -anchor n -bg "" -name hkl$xcen]
    }
    if {$tcl_version >= 8.0} {
	$plot marker config hkl$xcen -font lblfont
    }
    if {$cellparm(fadetime) > 0} {
	catch {
	    after [expr $cellparm(fadetime) * 1000 ] \
		    "catch \{ $plot marker delete $mark \}"
	}
    }
}

# set up the editcell window to take input for a new dgen computation
# It is called by SetEditData whenever cellparm(editsetting) is set to "(new)"
proc editnew {} {
    global cellparm
    # set the button value; but don't trigger SetEditData
    set cellparm(skipSetEditData) 1
    set cellparm(editsetting) "(new)"
    set cellparm(skipSetEditData) 0
    set cellparm(celleditvar) "----"
    set page $cellparm(page)
    pack forget $page.a.1
    pack forget $page.a.2
    pack forget $page.a.h
    pack forget $page.a.p
    catch {button $page.a.5 -text Compute -command ComputeNewCell}
    pack $page.a.5 -side bottom
    $page.d.w.l config -fg black
    $page.d.w.e config -textvariable cellparm(wavelength) -fg black \
	    -state normal
    resetextinctions  
    set base $cellparm(celleditbase) 
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg black
	$base.e$var config -textvariable newcell($var) \
		-fg black -state normal
    }
    # set cell choices to all
    setsymbox $page.b.cell 1
    if {$cellparm(celllaue) == {}} {set cellparm(celllaue) 8}
    setcellsym $cellparm(celllaue) SetEditParms
    # disable zero
    $base.lzero config -fg #888
    $base.ezero config -fg #888 -state disabled
    $base.szero config -fg #888 -state disabled    
    # reset extinction codes
    $cellparm(extinctmenuE).l1 config -text "to remove"
    $cellparm(extinctmenuE).l2 config -text ""
    $cellparm(extinctmenuE).spg3 config -text ""
    $cellparm(extinctmenuE).extl config -text ""
}

# set up the editcell window to take input for a new dgen computation
# It is called by SetEditData whenever cellparm(editsetting) is set 
# to something other than "(new)"
proc editold {} {
    global cellparm
    set page $cellparm(page)
    pack forget $page.a.5 
    pack $page.a.1 -side top
    pack $page.a.2 -side top
    pack $page.a.h -side bottom
    pack $page.a.p -side bottom
    set base $cellparm(celleditbase) 
    $base.lzero config -fg black
    $base.ezero config -fg black -state normal
    $base.szero config -fg black -state normal
    $cellparm(extinctmenuE).l1 config -text "to label"
    $cellparm(extinctmenuE).l2 config -text "removed"
}

proc computehkllist {} {
    global extinctlbl cellparm
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase)
    set out [open hklgen.inp w]
    puts $out "Created by hklgen.tcl"
    foreach var {a b c alpha beta gamma} {
	set val [$base.e$var get]
	if {[catch {expr $val}]} {
	    return
	} elseif {$val<10} {
	    append line [format %8.5f $val]
	    set ${data}($var) $val
	} elseif {$val<100} {
	    append line [format %8.4f $val]
	    set ${data}($var) $val
	} else {
	    append line [format %8.3f $val]
	    set ${data}($var) $val
	}
    }
    puts $out $line
    set line {}
    # use old extinctions if present in the data
    if {[set ${data}(extcodes)] != ""} {
	set line [set ${data}(extcodes)]
    }
    puts $out [format "%9.5f%9.2f%3d%s" \
	    [set ${data}(wavelength)] \
	    $cellparm(twothetamax) \
	    [set ${data}(laue)] \
	    $line]
    # use the saved value, if any
    if {[set ${data}(spg)] == "(none)"} {
	puts $out ""
    } elseif {[set ${data}(spg)] != ""} {
	puts $out [set ${data}(spg)]
    }
    close $out
    catch {exec $cellparm(hklgenprog) < hklgen.inp > hklgen.out}
    set out [open hklgen.out r]
    foreach var {h k l dspaces x extinctions} {
	set ${data}($var) {}
    }
    while {[gets $out line] >= 0} {
	if {6 == [scan $line %*d%d%d%d%d%*s%*d%f%f h k l mult tt d]} {
	    lappend ${data}(h) $h
	    lappend ${data}(k) $k
	    lappend ${data}(l) $l
	    lappend ${data}(dspaces) $d
	    lappend ${data}(x) $tt
	    lappend ${data}(extinctions) 0
	    lappend ${data}(y) {}
	    lappend ${data}(widths) {}
	    lappend ${data}(etas) {}
	    lappend ${data}(heights) {}
	    lappend ${data}(bkgs) 0
	}
    }
    close $out
    file delete -force hklgen.inp hklgen.out
    if {[llength [set ${data}(h)]] == 0} {
	tk_messageBox  -type ok -icon error -message \
		"No reflections were computed.\nCheck the 2theta limit in Options."
	return
    }
    resetdata $data
}

proc ComputeNewCell {} {
    global extinctlbl cellparm
    # need a variable name for SetEditParms
    incr cellparm(outcount)
    set data dgen$cellparm(outcount)
    global $data
    set data [initpeaks $data]
    
    set cellparm(celleditvar) $data
    global $data
    set base $cellparm(celleditbase)
    set message [validatecell]
    if {$message != ""} {
	tk_messageBox -message $message -type ok -icon warning
	return
    }
    savecelldata
    # set a dummy reflection
    set ${data}(h) 0
    set ${data}(k) 0
    set ${data}(l) 0
    # update the cell page
    PostPageEditCell $cellparm(page)
    #   set the peaklist button value; and trigger SetEditData to compute 
    # reflection positions and display the plot
    set cellparm(celleditvar) $data
    set cellparm(editsetting) $data
}

# validate the input before computing hkl values
# called in ComputeNewCell
proc validatecell {} {
    global cellparm
    set base $cellparm(celleditbase)
    set message {}
    foreach var {a b c alpha beta gamma} {
	set val [$base.e$var get]
	if {[catch {expr $val}]} {
	    append message "The value '$val' for cell parameter $var is invalid\n"
	}
    }
    if {[catch {expr $cellparm(wavelength)}]} {
	append message "The value '$cellparm(wavelength)' for wavelength is invalid\n"
    }
    if {[catch {expr $cellparm(twothetamax)}]} {
	append message "value '$cellparm(twothetamax)' for 2theta(max) is invalid\n"
    }
    return $message
}

# save the initial cellgen information into the data array
# called in ComputeNewCell
proc savecelldata {} {
    global extinctlbl cellparm
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase)
    foreach var {a b c alpha beta gamma} {
	set val [$base.e$var get]
	set ${data}($var) [$base.e$var get]
    }
    for {set num 1} {$num < 14} {incr num} {
	append line [format %3i \
		[lsearch $extinctlbl($num) $extinctlbl(r$num)]]
	set ${data}(extcodes) $line
    }

    # use a previous value, if any

    if {$cellparm(extspg) == "(none)"} {
	set ${data}(spg) (none)
    } else {
	set ${data}(spg) $cellparm(extspg)
    }
    set ${data}(laue) $cellparm(celllaue)
    set ${data}(wavelength) $cellparm(wavelength)
    foreach var {h k l dspaces x extinctions} {
	set ${data}($var) {}
    }
}
