#--------------------------------------------------------------------
# GUI for fitting profile functions to peak width tables. 
# $Revision: 332 $ $Date: 2009-12-04 09:44:04 -0600 (Fri, 04 Dec 2009) $

# queue a command to execute later
lappend command(ExecuteLater) SetupFitWidths

proc SetupFitWidths {} {
    global scriptdir menulist helplist
    lappend menulist(compute) FitWidths
    lappend menulist(pages) FitWidths
    set helplist(FitWidths) {
	Fit profile function to peak widths
    }
}

# create the page (once)
proc MakeFitWidths {page} {
    global command

    if [catch {package require La} errmsg] {
	MyMessageBox -parent . -title "La Load Error" \
		-message "Error -- Unable to load the La (Linear Algebra) package; cannot fit profile" \
		-icon error -type OK -default ok
	return
    }

    grid [frame $page.a] -column 0 -row 0 -sticky nsw
    pack [label $page.a.t1 -text {Select peak list}] -side top
    pack [frame $page.a.2 ] -side top 
    pack [frame $page.a.3 -borderwidth 3 -relief groove] -side top -pady 20 -ipadx 5
    pack [label $page.a.3.b -text "Plot range" -anchor c -bg yellow] \
	-side top -pady 5 -anchor s
    pack [label $page.a.3.c -text "2theta min" -anchor c] \
	-side top
    pack [entry $page.a.3.d -width 6 -textvariable command(FitMin)] \
	-side top
    pack [label $page.a.3.e -text "2theta max" -anchor c] \
	-side top
    pack [entry $page.a.3.f -width 6 -textvariable command(FitMax)] \
	-side top

    set command(fit_button) $page.a.4
    pack [button $page.a.4    -text "Fit Profile" \
	    	-command "FitGSAS"] -side bottom

    set command(fit_widlst) $page.a1
    set command(fit_terms) $page.a2
    grid [frame $command(fit_widlst) -borderwidth 3 -relief groove] \
	-column 1 -row 0 -sticky nsew
    grid [frame $command(fit_terms)  -borderwidth 3 -relief groove]\
	-column 2 -row 0 -sticky nw
    grid columnconfig $page 0 -weight 1
    grid rowconfig $page 0 -weight 1

    set box $command(fit_widlst)
    grid [label $box.c0 -text "Peak list" -anchor c -bg yellow] \
	-column 0 -row 0 -sticky ew -columnspan 6

    set bx $command(fit_terms)
    grid [label $bx.c0 -text "Profile Function" -anchor c -bg yellow] \
	-column 0 -row 0 -sticky ewn -columnspan 6

    trace variable command(widthlist) w setupwidthlist
#    $command(fit_button) config -command "setupwidthlist 1 2 3"
}

# "cleanup" the page each time it is selected
proc PostPageFitWidths {page}  {
    global graph command

    eval destroy [winfo children $page.a.2]
    set command(widthlist) {}

    set peakonlylist {}
    foreach data $graph(datalist) {
	global $data
	if {[set ${data}(type)] == "peaks"} {lappend peakonlylist $data}
    }
    if {$peakonlylist == ""} {
	tk_optionMenu $page.a.2.a command(widthlist) {}
    } else {
	eval tk_optionMenu $page.a.2.a command(widthlist) $peakonlylist
    }
    $page.a.2.a.menu add separator
    $page.a.2.a.menu add command -command AddNewPeakList -label "(create new)"
    pack $page.a.2.a -side top
}

proc AddNewPeakList {} {
    global graph command
    set i 1
    while {[lsearch $graph(datalist) peaklist$i] != -1} {
	incr i
    }
    set name peaklist$i
    set name [initpeaks $name]
    set command(widthlist) $name
    # init as 2theta data
    global $name
    set ${name}(xlabel) "2theta"
    set ${name}(xunits) "2theta"    
}

# set the data file when the select button is used
proc setupwidthlist {a b c} {
    global command graph
    set box $command(fit_widlst)
    eval destroy [winfo children $box]
    set bx $command(fit_terms)
    eval destroy [winfo children $bx]
    if {$command(widthlist) == ""} return
    set data $command(widthlist)
    global $data
    grid [label $box.c0 -text "Peak list" -anchor c -bg yellow] \
	-column 0 -row 0 -sticky ew -columnspan 6
    grid [label $box.h0 -text "#" -anchor c] \
	-column 0 -row 1 -sticky ew
    grid [label $box.h1 -text "Pos" -anchor c] \
	-column 1 -row 1 -sticky w
    grid [label $box.h2 -text "Gauss\rFWHM" -anchor c] \
	-column 2 -row 1 -sticky w
    grid [label $box.h3 -text "Cauchy\rFWHM" -anchor c] \
	-column 3 -row 1 -sticky w
    grid [label $box.h4 -text "Omit" -anchor c] \
	-column 4 -row 1 -sticky w
    set i 99
    grid [button $box.add -text "Add" -command Add2WidthList] -column 0 -row $i
    grid [entry $box.p$i -width 8] -row $i -column 1
    $box.p$i delete 0 end
    grid [entry $box.g$i -width 7] -row $i -column 2
    $box.g$i delete 0 end
    grid [entry $box.l$i -width 7] -row $i -column 3
    $box.g$i delete 0 end

    grid [canvas $box.canvas \
	      -scrollregion {0 0 5000 600} -width 0 -height 100 \
	      -yscrollcommand "$box.scroll set"] \
	-column 0 -row 2 -sticky nsew -columnspan 5
    grid columnconfigure $box 0 -weight 1
    grid rowconfigure $box 2 -weight 1
    scrollbar $box.scroll \
	-command "$box.canvas yview"
    frame [set command(widthframe) $box.canvas.fr]
    $box.canvas create window 0 0 -anchor nw -window $command(widthframe)
    set i 0
    set command(last_width) $i
    foreach x [set ${data}(x)]  wid [set ${data}(widths)] eta [set ${data}(etas)] {
	foreach {L G e e} [ComputeGL $wid $eta 0 0] {}
	if {$G != ""} {
	    incr i
	    set command(last_width) $i
	    grid [label $command(widthframe).h$i -text $i] -row $i -column 0
	    grid [entry $command(widthframe).p$i -width 8] -row $i -column 1
	    $command(widthframe).p$i delete 0 end
	    $command(widthframe).p$i insert end $x
	    grid [entry $command(widthframe).g$i -width 7] -row $i -column 2
	    $command(widthframe).g$i delete 0 end
	    $command(widthframe).g$i insert end $G
	    grid [entry $command(widthframe).l$i -width 7] -row $i -column 3
	    $command(widthframe).l$i delete 0 end
	    $command(widthframe).l$i insert end $L
	    set command(Omit$i) 0
	    grid [checkbutton $command(widthframe).u$i \
		      -variable command(Omit$i) \
		      -anchor c -command PlotWidths] -row $i -column 4
	}
    }

    grid [label $bx.c0 -text "Profile Function" -anchor c -bg yellow] \
	-column 0 -row 0 -sticky ewn -columnspan 6

    grid [label $bx.c1 -text "GSAS -- Function 2 or 3" -anchor w] \
	-column 0 -row 1 -sticky ew -columnspan 6

    grid [frame $bx.c2 -bd 2 -relief sunken] \
	-column 0 -row 2 -sticky ew -columnspan 6
    grid [button $bx.c3    -text "Update Plot" \
	    	-command "PlotWidths"] \
	-column 0 -row 3 -sticky ew -columnspan 6

    set row 0
    grid [label $bx.c2.$row -text "Fit" -anchor c] \
	-column 2 -row $row -sticky w
    incr row
    grid [label $bx.c2.$row -text "Gaussian" -anchor c] \
	-column 0 -row $row -sticky ew -columnspan 3
    foreach lbl {U V W} {
	incr row
	grid [label $bx.c2.a$lbl -text $lbl -anchor c] \
		-column 0 -row $row -sticky ew
	grid [entry $bx.c2.b$lbl -width 8] \
		-column 1 -row $row -sticky ew
	grid [checkbutton $bx.c2.c$lbl \
	      -variable command(Refine$lbl)] \
	    -column 2 -row $row -sticky ew
	set command(Refine$lbl) 1
    }
    incr row
    grid [label $bx.c2.$row -text "Lorenztian\n(Cauchy)" -anchor c] \
	-column 0 -row $row -sticky ew -columnspan 3
    foreach lbl {X Y} {
	incr row
	grid [label $bx.c2.a$lbl -text $lbl -anchor c] \
		-column 0 -row $row -sticky ew
	grid [entry $bx.c2.b$lbl -width 8] \
		-column 1 -row $row -sticky ew
	grid [checkbutton $bx.c2.c$lbl \
	      -variable command(Refine$lbl)] \
	    -column 2 -row $row -sticky ew
	set command(Refine$lbl) 1
    }
    # get the range of 2theta values
    set min {}
    set max {}
    for {set i 1} {$i <= $command(last_width)} {incr i} {
	if {$command(Omit$i)} continue
	set p [$command(widthframe).p$i get]
	if {[catch {expr $p}]} continue
	if {$min == ""} {set min $p; set max $p}
	if {$min > $p} {set min $p}
	if {$max < $p} {set max $p}
    }
    set command(FitMin) {}
    set command(FitMax) {}
    catch {set command(FitMin) [format %5.2f $min]}
    catch {set command(FitMax) [format %5.2f $max]}
    ResizeWidthList
    PlotWidths
}

proc Add2WidthList {} {
    global command
    set box $command(fit_widlst)
    set newp [$box.p99 get]
    set newg [$box.g99 get]
    set newl [$box.l99 get]
    if {[string trim $newl] == ""} {set newl 0.0}
    if {[catch {expr $newp + $newg + $newl}]} {bell;return}
    set i [incr command(last_width)]
    grid [label $command(widthframe).h$i -text $i] -row $i -column 0
    grid [entry $command(widthframe).p$i -width 8] -row $i -column 1
    $command(widthframe).p$i delete 0 end
    $command(widthframe).p$i insert end $newp
    grid [entry $command(widthframe).g$i -width 7] -row $i -column 2
    $command(widthframe).g$i delete 0 end
    $command(widthframe).g$i insert end $newg
    grid [entry $command(widthframe).l$i -width 7] -row $i -column 3
    $command(widthframe).l$i delete 0 end
    $command(widthframe).l$i insert end $newl
    set command(Omit$i) 0
    grid [checkbutton $command(widthframe).u$i \
	      -variable command(Omit$i) \
	      -anchor c -command PlotWidths] -row $i -column 4

    set min {}
    set max {}
    for {set i 1} {$i <= $command(last_width)} {incr i} {
	if {$command(Omit$i)} continue
	set p [$command(widthframe).p$i get]
	if {[catch {expr $p}]} continue
	if {$min == ""} {set min $p; set max $p}
	if {$min > $p} {set min $p}
	if {$max < $p} {set max $p}
    }
    if {[catch {expr $command(FitMin)}]}  {
	catch {set command(FitMin) [format %5.2f $min]}
    } elseif {$newp <  $command(FitMin)} {
	catch {set command(FitMin) [format %5.2f $min]}
    }
    if {[catch {expr $command(FitMax)}]}  {
	catch {set command(FitMax) [format %5.2f $max]}
    } elseif {$newp > $command(FitMax)} {
	catch {set command(FitMax) [format %5.2f $max]}
    }
    ResizeWidthList
    $box.canvas yview moveto 1
    PlotWidths
}

proc PlotWidths {} {
    global command graph gpls
    set box $command(fit_widlst)

    # create a new plot window, if needed
    if {$gpls(fitinnewplot)} {
	catch {toplevel .fit}
	catch {
	    pack [set graph(fit) [graph .fit.gr]] -fill both -expand yes -side top
	    $graph(fit) config -title {} -plotbackground white
	    $graph(fit) legend config -font  *-Helvetica-Bold-R-Normal-*-10-*-*
	    bind .fit <Key-P> "addpeak $graph(fit) %x %y"
	    bind .fit <Key-p> "addpeak $graph(fit) %x %y"
	}
    } else {
	set graph(fit) $graph(blt)
    }

    # zoom reset that works (in later versions?) to clear incomplete zoom
    catch {blt::ResetZoom $graph(fit)}
    # reset Zoom
    catch {Blt_ZoomStack $graph(fit)}
    # delete captions
    eval $graph(fit) element delete [$graph(fit) element names]
    eval $graph(fit) marker delete  [$graph(fit) marker names]

    $graph(blt) xaxis config -min {} -max {}
    $graph(blt) yaxis config -min {} -max {}

    # plot the fit values
    set PG {}
    set PL {}
    set G {}
    set L {}
    for {set i 1} {$i <= $command(last_width)} {incr i} {
	if {$command(Omit$i)} continue
	set p [$command(widthframe).p$i get]
	if {[catch {expr $p}]} continue
	set g [$command(widthframe).g$i get]
	if {![catch {expr $g}]} {
	    lappend PG $p
	    lappend G $g
	}
	set l [$command(widthframe).l$i get]
	if {![catch {expr $l}]} {
	    lappend PL $p
	    lappend L $l
	}
    }
    set data $command(widthlist)
    global $data
    catch {
	$graph(fit) element create Gaussian -xdata $PG -ydata $G -symbol circle \
	    -linewidth 0 -pixels [expr 0.125*[set ${data}(symsize)]]i \
	    -color red
    }
    catch {
	$graph(fit) element create Cauchy -xdata $PL -ydata $L -symbol square \
	    -linewidth 0 -pixels [expr 0.125*[set ${data}(symsize)]]i \
	    -color blue
    }
    # now plot the fit -- GSAS mode
    set bx $command(fit_terms)
    foreach var {U V W X Y} {
	set $var 0
	set val [$bx.c2.b$var get]
	if {$val == ""} {$bx.c2.b$var insert end 0.0}
	if {![catch {expr $val}]} {set $var $val}
    }
    plotGSASWidths $U $V $W $X $Y

}


proc ResizeWidthList {} {
    global command
    set box $command(fit_widlst)

    update idletasks
    # sync the column widths
    grid columnconfig $box.canvas.fr 0 -min [lindex [grid bbox $box 0 0] 2]
    foreach col {1 2 3 4} {
	grid columnconfig $box $col -min \
	    [lindex [grid bbox $box.canvas.fr $col 0] 2]
    }

    update idletasks
    # set the scrollbar, if needed
    set sizes [grid bbox $box.canvas.fr]
    $box.canvas config -scrollregion $sizes -width [lindex $sizes 2]
    # use the scroll for BIG lists
    if {[lindex $sizes 3] > [winfo height $box.canvas]} {
	grid $box.scroll -sticky ns -column 5 -row 2
    } else {
	grid forget $box.scroll 
    }
}

proc plotGSASWidths {U V W X Y} {
    global command
    set Glist {}
    set Llist {}
    set incr 0
    catch {set incr [expr {($command(FitMax) - $command(FitMin))/100.}]}
    if {$incr == 0} return
    for {set tt $command(FitMin)} \
	{$tt <= $command(FitMax)} \
	{set tt [expr $tt + $incr]} {
	set G 0
	catch {set G [FWHM $tt $U $V $W 0]}
	if {$G != ""} {
	    lappend Glist $tt $G
	}
	set L 0
	catch {set L [LFWHM $tt $X $Y]}
	if {$L < 0} {set L 0}
	lappend Llist $tt $L
    }
    global graph
    catch {$graph(fit) element create Fit-Gaussian \
	       -linewidth 1 -color red -symbol none}
    catch {$graph(fit) element create Fit-Cauchy \
	       -linewidth 1 -color blue -symbol none}
    catch {
	$graph(fit) element config Fit-Gaussian -data $Glist
    }
    catch {
	$graph(fit) element config Fit-Cauchy -data $Llist 
    }
}

proc FWHM {tt U V W P} {
    set pi 3.14159
    set torad [expr $pi / 360.]
    # tan theta
    set tantt [expr tan($tt * $torad ) ]
    set costt [expr cos($tt * $torad ) ]
    return [expr sqrt \
	    (8.* log(2) * ($U * $tantt * $tantt + $V * $tantt + $W \
	    + $P / ($costt * $costt))) / 100.]
}
proc LFWHM {tt X Y} {
    set pi 3.14159
    set torad [expr $pi / 360.]
    # tan theta
    set tantt [expr {tan($tt * $torad )} ]
    set costt [expr {cos($tt * $torad )} ]
    return [expr ($X / $costt + $Y * $tantt) / 100.]
}

proc LFWHMderiv {tt} {
    set pi 3.14159
    set torad [expr $pi / 360.]
    # 1 / cos(theta)
    set sectt [expr {0.01/cos($tt * $torad )} ]
    # tan(theta)
    set tantt [expr {tan($tt * $torad )/100.} ]
    return [list $sectt $tantt]
}
proc FWHMderiv {tt} {
    set pi 3.14159
    set rad [expr {$tt * $pi / 360.}]
    set cons [expr {8.* log(2) / 10000.}]
    # tan theta
    set tantt [expr {$cons * tan($rad ) }]
    set tantt2 [expr {$cons * tan($rad ) * tan($rad ) }]
    return [list $tantt2 $tantt $cons]
}

proc FitGSAS {} {
    global command
    set bx $command(fit_terms)
    FitGSASXY
    FitGSASUVW
    # reset X or Y terms that refine negative
    set redo 0
    foreach lbl {X Y} {
	if {$command(Refine$lbl)} {
	    catch {
		if {[$bx.c2.b$lbl get] < 0.0} {
		    $bx.c2.b$lbl delete 0 end
		    $bx.c2.b$lbl insert end 0.0
		    set command(Refine$lbl) 0
		    set redo 1
		}
	    }
	}
    }
    if {$redo} FitGSASXY
    PlotWidths
}

# evaluate the best-fit GSAS profile terms to fit a set of peak widths
proc FitGSASXY { } {
    global command
    set bx $command(fit_terms)
    # fit the X & Y terms
    set o 0
    set sel {}
    foreach lbl {X Y} {
	if {$command(Refine$lbl)} {
	    incr o
	    lappend sel 1
	    $bx.c2.b$lbl delete 0 end
	} else {
	    lappend sel 0
	}
    }
    if {$o == 0} return
    # zero the matrix and vector
    set npts 0
    for {set j 0} {$j < $o} {incr j} {
	set sum($j) 0.
	for {set i 0} {$i <= $j} {incr i} {
	    set sum(${i}_$j) 0.
	}
    }
    for {set ii 1} {$ii <= $command(last_width)} {incr ii} {
	if {$command(Omit$ii)} continue
	set p [$command(widthframe).p$ii get]
	if {[catch {expr $p}]} continue
	set l [$command(widthframe).l$ii get]
	if {[catch {expr $l}]} continue
	incr npts

	# compute all derivatives at point p; select the ones needed
	set i 0
	foreach val [LFWHMderiv $p] s $sel {
	    if {$s} {
		set T($i) $val
		incr i
	    }
	}

	# compute matrix elements
	for {set j 0} {$j < $o} {incr j} {
	    set sum($j) [expr {$sum($j) + $l * $T($j)}]
	    for {set i 0} {$i <= $j} {incr i} {
		set sum(${i}_$j) [expr {$sum(${i}_$j) + $T($i) * $T($j)}]
	    }
	}
    }
    if {$npts <= $o} return
    # populate the matrix & vector in La format
    lappend V 2 $o 0
    lappend A 2 $o $o
    for {set i 0} {$i < $o} {incr i} {
	lappend V $sum($i)
	for {set j 0} {$j < $o} {incr j} {
	    if {$j < $i} {
		lappend A $sum(${j}_$i)
	    } else {
		lappend A $sum(${i}_$j)
	    }
	}
    }
    set termlist {}
    if {[catch {
	set termlist [lrange [La::msolve $A $V] 3 end]
	set i 0
	foreach s $sel lbl {X Y} {
	    if {$s} {
		$bx.c2.b$lbl delete 0 end
		$bx.c2.b$lbl insert end [lindex $termlist $i]
		incr i
	    }
	}
    }]} {
	tk_dialog .singlar "Singular Matrix" \
	    "Unable to fit function: singular matrix. Too many terms or something else is wrong." ""\
	    0 OK
    }
    return $termlist
}


# evaluate the best-fit GSAS profile terms to fit a set of peak widths
proc FitGSASUVW { } {
    global command
    set bx $command(fit_terms)
    # fit the U, V & W terms
    set o 0
    set sel {}
    foreach lbl {U V W} {
	if {$command(Refine$lbl)} {
	    incr o
	    lappend sel 1
	    $bx.c2.b$lbl delete 0 end
#	    $bx.c2.b$lbl insert end [lindex $termlist $i]
	} else {
	    lappend sel 0
	}
    }
    if {$o == 0} return
    # zero the matrix and vector
    set npts 0
    for {set j 0} {$j < $o} {incr j} {
	set sum($j) 0.
	for {set i 0} {$i <= $j} {incr i} {
	    set sum(${i}_$j) 0.
	}
    }
    for {set ii 1} {$ii <= $command(last_width)} {incr ii} {
	if {$command(Omit$ii)} continue
	set p [$command(widthframe).p$ii get]
	if {[catch {expr $p}]} continue
	set g [$command(widthframe).g$ii get]
	if {[catch {expr $g}]} continue
	incr npts

	# compute all derivatives at point p; select the ones needed
	set i 0
	foreach val [FWHMderiv $p] s $sel {
	    if {$s} {
		set T($i) $val
		incr i
	    }
	}

	# compute matrix elements
	for {set j 0} {$j < $o} {incr j} {
	    set sum($j) [expr {$sum($j) + $g * $g * $T($j)}]
	    for {set i 0} {$i <= $j} {incr i} {
		set sum(${i}_$j) [expr {$sum(${i}_$j) + $T($i) * $T($j)}]
	    }
	}
    }
    if {$npts <= $o} return
    # populate the matrix & vector in La format
    lappend V 2 $o 0
    lappend A 2 $o $o
    for {set i 0} {$i < $o} {incr i} {
	lappend V $sum($i)
	for {set j 0} {$j < $o} {incr j} {
	    if {$j < $i} {
		lappend A $sum(${j}_$i)
	    } else {
		lappend A $sum(${i}_$j)
	    }
	}
    }
    set termlist {}
    if {[catch {
	set termlist [lrange [La::msolve $A $V] 3 end]
	set i 0
	foreach s $sel lbl {U V W} {
	    if {$s} {
		$bx.c2.b$lbl delete 0 end
		$bx.c2.b$lbl insert end [lindex $termlist $i]
		incr i
	    }
	}
    }]} {
	tk_dialog .singlar "Singular Matrix" \
	    "Unable to fit function: singular matrix. Too many terms or something else is wrong." ""\
	    0 OK
    }
    return $termlist
}
