# things to do: 
#   map peaks from peaklist file onto peak window when data range is set
#   routine to dump data to a file

catch {vector xvec yovec ycvec}
set gpls(fwhm) .1
set gpls(eta) .5
set gpls(ncyc)  5
set gpls(damp)  0.8
set gpls(radius) 500.
set gpls(det_hgt) 5.
set gpls(sampl_hgt) 5.
if {$tcl_platform(platform) == "windows"} {
    set gpls(program) [file join $command(exedir) gpls.exe]
    set gpls(getvoigt) [file join $command(exedir) getvoigt.exe]
} else {
    set gpls(program) [file join $command(exedir) gpls]
    set gpls(getvoigt) [file join $command(exedir) getvoigt]
}
set gpls(asymflg) 0
set gpls(dataset) ""
set gpls(fitonplot) 1
set gpls(fitinnewplot) 0
set gpls(in_setupfitfile) 0 
set gpls(rangemsg) {}
set gpls(xlow) {}
set gpls(xhigh) {}
set gpls(incr) 5.0
set graph(fitnum) 0

# set the data file when the select button is used
proc setupfitfile {a b c} {
    global graph gpls
    if {$gpls(dataset) == ""} return
    if {$gpls(dataset) == "(none selected)"} return
    if {$gpls(in_setupfitfile) == 1} return
    set gpls(in_setupfitfile) 1
    if {$gpls(peaklist) == "(create new)"} {
	set i 1
	while {[lsearch $graph(datalist) peaklist$i] != -1} {
	    incr i
	}
	set name peaklist$i
	set peakonlylist {}
	foreach data $graph(datalist) {
	    global $data
	    if {[set ${data}(type)] == "peaks"} {lappend peakonlylist $data}
	}
	lappend peakonlylist {(create new)}
	set page $gpls(page)
	catch {destroy $page.a.p.a}
	eval tk_optionMenu $page.a.p.a gpls(peaklist) $peakonlylist
	pack $page.a.p.a
	set gpls(peaklist) $name
	set name [initpeaks $name]
	# set peaklist parameters from data
	set data $gpls(dataset)
	global $data 
	set peak $gpls(peaklist)
	global $peak 
	foreach item {wavelength xunits xlabel ylabel cxlabel cylabel} {
	    set ${peak}($item) [set ${data}($item)]
	}
    }
    set graph(plotlist) $gpls(dataset)
    displaylist
    set data $graph(plotlist)
    global $data 
    set gpls(min) [[set ${data}(xvector)] range 0 0]
    set gpls(max) [[set ${data}(xvector)] range end end]
    # zoom reset that works (in later versions?) to clear incomplete zoom
    catch {blt::ResetZoom $graph(blt)}
    # reset Zoom
    catch {Blt_ZoomStack $graph(blt)}
    $graph(blt) xaxis config -min {} -max {}
    $graph(blt) yaxis config -min {} -max {}
    $graph(blt) config -title "Select a data range to fit"
    set gpls(in_setupfitfile) 0
    set gpls(rangemsg) {No range set}
    # attach vars to the wavelength widgets
    $gpls(wave1box) configure -textvariable "[set data](wavelength)"
    $gpls(wave2box) configure -textvariable "[set data](wavelength2)"
}

# compute the first and last point to be fit from the data range
proc setdatarange {} {
    global gpls command graph
    if {$command(pagenow) != "fit"} return
    if {$gpls(dataset) == ""} return
    if {$gpls(dataset) == "(none selected)"} return
    catch {
	set data $gpls(dataset)
	global $data
	# set the point numbers for the max & minimum range
	set pointlist [[set ${data}(xvector)] search $gpls(min) $gpls(max)]
	if {[llength $pointlist] == 0} return
	set gpls(xlow) [lindex $pointlist 0]
	set gpls(xhigh) [lindex $pointlist end]
	set gpls(bkglow) [[set ${data}(yvector)] range $gpls(xlow) $gpls(xlow)]
	set gpls(bkghigh) [[set ${data}(yvector)] range $gpls(xhigh) $gpls(xhigh)]
	catch {vector ytmp}
	ytmp set [[set ${data}(yvector)] range $gpls(xlow) $gpls(xhigh)]
	ytmp sort
	set gpls(ymin) [ytmp range 0 0]
	set gpls(ymax) [ytmp range end end]
	set numobs [expr $gpls(xhigh) - $gpls(xlow) + 1]

	if {$numobs > 999} return

	# set peaks from peak list
	set peak $gpls(peaklist)
	global $peak 
	resetdata $peak
	for {set i 1} {$i <= 8} {incr i} {
	    set gpls(use$i) 0
	    set gpls(label$i) {}
	    set gpls(refpos$i) 0
	    set gpls(refarea$i) 0
	    enablepeak $gpls(peakframe) $i
	}
	set i 0
	# include peaks within a full-width of the window
	foreach p [[set ${peak}(ttvector)] search \
		[expr $gpls(min)-$gpls(fwhm)] \
		[expr $gpls(max)+$gpls(fwhm)] ] {
	    incr i
	    if {$i <= 8} {
		set gpls(use$i) 1
		set gpls(area$i) [lindex [set ${peak}(y)] $p]
		set gpls(pos$i)  [lindex [set ${peak}(x)] $p]
		set gpls(label$i) [expr $p + 1]
		enablepeak $gpls(peakframe) $i
	    }
	}
	$graph(blt) config -title "Add peaks: press P at maximum"
    }
}

# respond to the "Enable asymmetry" checkbutton
proc disableasym {a1 a2 a3} {
    global gpls
    set base $gpls(overallframe)
    if $gpls(asymflg) {
	set flag black
	$base.refineasym1 config -variable gpls(refine_asym1) -state normal
	$base.refineasym2 config -variable gpls(refine_asym2) -state normal
    } {
	set flag #888
	$base.refineasym1 config -variable {} -state disabled
	$base.refineasym2 config -variable {} -state disabled
    }
    foreach widget "$base.radlbl $base.radent $base.hgtlbl $base.hgtent \
	    $base.detlbl $base.detent" {
	$widget config -fg $flag
    }
}

# check to see if the input is valid -- disable the gplsfit button otherwise
proc enablegpls {a b c} {
    global gpls graph
    set page $gpls(page)
    catch {pack forget $page.a.6}
    catch {grid forget $page.b.plot}
    catch {grid forget $page.b.undo}
    catch {pack $page.a.8 -side bottom}
    set graph(gplsstatus) { }
    # resize
    update idletasks
    ResizeNotebook
    update
    wm geom . [winfo reqwidth .]x[winfo reqheight .]

    set data $gpls(dataset)
    if {$data == "" || $data == "(none selected)"} {
	set graph(gplsstatus) "invalid data set"
	return
    }
    if [catch {
	global $data
	# get the number of data points
	set numobs [expr $gpls(xhigh) - $gpls(xlow) + 1]
    } errmsg] {
	set graph(gplsstatus) "error: $errmsg"
	return
    }
    if {$numobs > 999} {
	set graph(gplsstatus) "data range too large"
	return
    } elseif {$numobs <= 0} {
	set graph(gplsstatus) "invalid data range"
	return
    }
    # get the number of peaks and validate them
    set peaks 0
    catch {
	for {set i 1} {$i < 9} {incr i} {
	    if $gpls(use$i) {
		expr $gpls(area$i)
		expr $gpls(pos$i)
		incr peaks
	    }
	}
    }
    if {$peaks == 0} {
	set graph(gplsstatus) "no peaks"
	return
    }
    # validate the rest of the input
    if [catch {
	expr [[set ${data}(xvector)] range $gpls(xlow) $gpls(xlow)]
	expr [[set ${data}(xvector)] range $gpls(xhigh) $gpls(xhigh)]
	expr $gpls(radius)
	expr $gpls(damp)
	expr $gpls(bkglow)
	expr $gpls(bkghigh)
	expr $gpls(fwhm)
	expr $gpls(eta)
	if {$gpls(asymflg)} {
	    expr $gpls(sampl_hgt)
	    expr $gpls(det_hgt)
	}
	expr $gpls(ncyc)
    }] {
	set graph(gplsstatus) "invalid input value"
	return
    }
    # all is OK
    catch {pack forget $page.a.8}
    catch {pack $page.a.6 -side bottom}
    $page.a.6 config -state normal -fg black
    # resize
    update
    ResizeNotebook
    wm geom . [winfo reqwidth .]x[winfo reqheight .]
}

# write the output to gplsfit run
proc wrgpls {} {
    global gpls graph blt_version command
    set data $gpls(dataset)
    global $data
    $graph(blt) config -title ""
    # avoid bug in BLT 2.3 where Inf does not work for text markers
    if {$blt_version == 2.3} {
	set xcen [lindex [$graph(blt) xaxis limits] 1]
	set ycen [lindex [$graph(blt) yaxis limits] 1]
    } else  {
	set xcen Inf
	set ycen Inf
    }
    catch {$graph(fit) marker delete GOF}
    # get the number of data points
    set numobs [expr $gpls(xhigh) - $gpls(xlow) + 1]
    if {$numobs > 999} {
	catch {$graph(blt) marker delete GOF}
	$graph(blt) marker create text -name "GOF" \
		-text "Too many points for GPLSFT" \
		-coords "$xcen $ycen" -anchor ne -bg {} 
	return
    }
    # get the number of peaks
    set peaks 0
    for {set i 1} {$i < 9} {incr i} {
	if $gpls(use$i) {
	    incr peaks
	    set gpls(refine_[expr 4+2*$peaks]) $gpls(refpos$i)
	    set gpls(refine_[expr 3+2*$peaks]) $gpls(refarea$i)
	}
    }
    set gpls(refine_[expr 5+2*$peaks]) $gpls(refine_asym1)
    set gpls(refine_[expr 6+2*$peaks]) $gpls(refine_asym2)
    pleasewait "refining"
    update
    set out [open GPLS.INP w]
    puts $out "GPLS run from CMPR"
    set mode 0
    if {[catch {
	if {[string trim [set ${data}(wavelength)]] == ""} break
	if {[string trim [set ${data}(wavelength2)]] == ""} break
	expr [set ${data}(wavelength)]
	expr [set ${data}(wavelength2)]
	expr $gpls(ratio21)
    } errmsg]} {
	if {$command(debug)} {puts "wave1/2 error $errmsg"} 
	puts $out $numobs
    } else {
	puts $out -$numobs
	set mode 1
    }
    puts $out 10
    puts $out [expr $peaks*2 + 4 + 2*$gpls(asymflg)]
    puts $out [[set ${data}(xvector)] range $gpls(xlow) $gpls(xlow)]
    puts $out [expr (\
	    [[set ${data}(xvector)] range $gpls(xhigh) $gpls(xhigh)] \
	    - [[set ${data}(xvector)] range $gpls(xlow) $gpls(xlow)] ) \
	    / ($numobs - 1)]
    puts $out [[set ${data}(xvector)] range $gpls(xhigh) $gpls(xhigh)]
    puts $out $gpls(asymflg)
    puts $out $gpls(radius)
    puts $out $gpls(damp)
    if {$mode} {
	puts $out [set ${data}(wavelength)]
	puts $out [set ${data}(wavelength2)]
	puts $out $gpls(ratio21)
    }
    puts $out [[set ${data}(yvector)] range $gpls(xlow) $gpls(xhigh)]
    puts $out [[set ${data}(esdvec)] range $gpls(xlow) $gpls(xhigh)]
    puts $out "      LOW  BKGD                          $gpls(bkglow)"
    puts $out "      HIGH BKGD                          $gpls(bkghigh)"
    puts $out "      FWHM                               $gpls(fwhm)"
    puts $out "      ETA                                $gpls(eta)"
    for {set i 1} {$i < 9} {incr i} {
	if $gpls(use$i) {
	    puts $out "      INT. INT.                          [set gpls(area$i)]"
	    puts $out "      POSITION                           [set gpls(pos$i)]"
	}
    }
    if {$gpls(asymflg)} {
	puts $out "      Sample Size                        $gpls(sampl_hgt)"
	puts $out "      Detector Height                    $gpls(det_hgt)"
    }
    for {set num 1} {$num <= 4 + 2*$peaks + 2*$gpls(asymflg)} {incr num} {
	puts -nonewline $out $gpls(refine_$num)
    }
    puts $out {}
    puts $out $gpls(ncyc)
    close $out
    catch {
	$graph(fit) delete GOF
	$graph(fit) marker create text -name "GOF" -text "Running GPLSFT" \
	    -coords "$xcen $ycen" -anchor ne -bg {} 
    }
    update

    if {[catch {
	exec $gpls(program) < GPLS.INP > GPLS.R
    } errmsg]} {
	if {$command(debug)} {puts "refine error $errmsg"} 
	$graph(blt) config -title "Error running fit"
	donewait
	return
    }
    donewait
    if {[catch {
	plotgpls
    } errmsg]} {
	$graph(blt) config -title "Error reading fit results"
	if {$command(debug)} {puts "read in error $errmsg"} 
    }
    set page $gpls(page)
    # clean up
    if {!$command(debug)} {
	catch {file delete GPLS.LST GPLS.PRM GPLS.INP gpls.out GPLS.R}
    }
    catch {grid $page.b.plot -row 12 -column 0 -columnspan 4}
}

# read and display the results from gpls
proc plotgpls {} {
    global gpls gpls_err graph blt_version

    # create a new plot window, if needed
    if {$gpls(fitinnewplot)} {
	catch {toplevel .fit}
	catch {
	    pack [set graph(fit) [graph .fit.gr]] -fill both -expand yes -side top
	    $graph(fit) config -title {} -plotbackground white
	    $graph(fit) legend config -font  *-Helvetica-Bold-R-Normal-*-10-*-*
	    bind .fit <Key-P> "addpeak $graph(fit) %x %y"
	    bind .fit <Key-p> "addpeak $graph(fit) %x %y"
	}
    } else {
	set graph(fit) $graph(blt)
    }
    # zoom reset that works (in later versions?) to clear incomplete zoom
    catch {blt::ResetZoom $graph(fit)}
    # reset Zoom
    catch {Blt_ZoomStack $graph(fit)}
    $graph(fit) yaxis  config -min {} -max {}
    $graph(fit) xaxis  config -min {} -max {}
    # delete captions
    eval $graph(fit) element delete [$graph(fit) element names]
    eval $graph(fit) marker delete  [$graph(fit) marker names]

    # save the current parameters
    set graph(lastcycle) {array set gpls }
    append graph(lastcycle) "{"
    foreach item {bkglow bkghigh fwhm eta asymflg radius sampl_hgt det_hgt} {
	append graph(lastcycle) " $item $gpls($item)"
    }
    for {set i 1} {$i <= 8} {incr i} {
	foreach item {label use area pos} {
	    append graph(lastcycle) " ${item}$i [list $gpls(${item}$i)]"
	}
    }
    append graph(lastcycle) "}"

    set in [open gpls.out r]
    gets $in npts
    global xvec yovec ycvec ydvec
    catch {unset xvec yovec  ycvec ydvec}
    vector  xvec($npts) yovec($npts)  ycvec($npts) ydvec($npts)
    set data [lindex $graph(plotlist) 0]
    global $data 
    
    for {set i 0} {$i < $npts && ([gets $in line] >= 0)} {incr i} {
	set xvec($i) [lindex $line 0]
	set yovec($i) [lindex $line 1]
	set ycvec($i) [lindex $line 2]
    }
    ydvec set [yovec - ycvec]
    ydvec set [ydvec - [expr (1.1*$ydvec(max)) - $yovec(min)]]
    gets $in line
    set gof "[lindex $line 1] [lindex $line 2]"
    set nparm [lindex $line 0]

    set line [gets $in]
    set gpls(bkglow) [lindex $line 0]
    set gpls_err(bkglow) [lindex $line 2]

    set line [gets $in]
    set gpls(bkghigh) [lindex $line 0]
    set gpls_err(bkghigh) [lindex $line 2]

    set line  [gets $in]
    set gpls(fwhm) [lindex $line 0]
    set gpls_err(fwhm) [lindex $line 2]

    set line  [gets $in]
    set gpls(eta) [lindex $line 0]
    set gpls_err(eta) [lindex $line 2]

    foreach item {fwhm eta bkghigh bkglow} {
	catch {set gpls($item) [format %.6f $gpls($item)]}
    }
    set nline [expr ($nparm - 4)/2 - $gpls(asymflg)]
    set i 0
    set error 1
    catch {
	set stepcons [expr sqrt (4 * log(2) / 3.14159) * \
			  ([xvec range end end] - [xvec range 0 0]) / ([xvec length] -1)]
	for {set j 1} {$j <= $nline} {incr j} {
	    # get the next used peak
	    incr i
	    while {! $gpls(use$i) } {incr i}
	    set line  [gets $in]
	    set gpls(area$i) [lindex $line 0]
	    catch {set gpls(area$i) [format %.4f $gpls(area$i)]}
	    set gpls_err(area$i) [lindex $line 2]
	    set line  [gets $in]
	    set gpls(pos$i) [lindex $line 0]
	    catch {set gpls(pos$i) [format %.6f $gpls(pos$i)]}
	    set gpls_err(pos$i) [lindex $line 2]
	    
	    # deal with the peak list
	    set peak $gpls(peaklist)
	    global $peak 
	    # approximate the height of the peak
	    set height [expr $stepcons * $gpls(area$i) / $gpls(fwhm)]
	    # compute background at this point
	    set b [expr \
		       (1 - (([xvec range end end] - $gpls(pos$i)) / \
				 ([xvec range end end] - [xvec range 0 0]))) * \
		       ($gpls(bkghigh) - $gpls(bkglow)) + $gpls(bkglow)]
	    if {$gpls(label$i) == ""} {
		# we need to assign a peak to this entry
		lappend ${peak}(x) $gpls(pos$i)
		lappend ${peak}(y) $gpls(area$i)
		lappend ${peak}(dspaces) {}
		lappend ${peak}(h) {}
		lappend ${peak}(k) {}
		lappend ${peak}(l) {}
		lappend ${peak}(widths) $gpls(fwhm)
		lappend ${peak}(etas) $gpls(eta)
		lappend ${peak}(heights) $height
		lappend ${peak}(extinctions) 0
		lappend ${peak}(bkgs) $b
		set gpls(label$i) [llength [set ${peak}(x)]]
	    } else {
		set k [expr $gpls(label$i) - 1]
		set ${peak}(x) [lreplace [set ${peak}(x)] $k $k $gpls(pos$i)]
		set ${peak}(y) [lreplace [set ${peak}(y)] $k $k $gpls(area$i)]
		set ${peak}(widths) [lreplace [set ${peak}(widths)] $k $k $gpls(fwhm)]
		set ${peak}(etas) [lreplace [set ${peak}(etas)] $k $k $gpls(eta)]
		set ${peak}(heights) [lreplace [set ${peak}(heights)] $k $k $height]
		set ${peak}(bkgs) [lreplace [set ${peak}(bkgs)] $k $k $b]
	    }
	    enablepeak $gpls(peakframe) $i
	}
	set error 0
    } errmsg
    resetdata $peak
    if {$gpls(asymflg)} {
	set line  [gets $in]
	set gpls(sampl_hgt) [lindex $line 0]
	set gpls_err(sampl_hgt) [lindex $line 2]
	set line  [gets $in]
	set gpls(det_hgt) [lindex $line 0]
	set gpls_err(det_hgt) [lindex $line 2]
    }
    close $in
    $graph(fit) element create obs  -xdata xvec -ydata yovec -symbol splus \
	-linewidth 0 -pixels [expr 0.125*[set ${data}(symsize)]]i
    if {! $error} {
	$graph(fit) element create calc -xdata xvec -ydata ycvec -color red \
	    -symbol none
	$graph(fit) element create diff -xdata xvec -ydata ydvec -color blue \
	    -symbol none
	set bkgx "$xvec(min) $xvec(end)"
	$graph(fit) element create bkg -xdata $bkgx \
	    -ydata "$gpls(bkglow) $gpls(bkghigh)" -color green \
	    -symbol none
    }
    xvec notify now
    yovec notify now
    ycvec notify now
    ydvec notify now

    # ajm 04/12/00 & bht+rlhI 9/10/02
    # determin Gauss and Lorentz portions of peak widths
    foreach {gpls(bc) gpls(bg) gpls_err(bc) gpls_err(bg)} \
	[ComputeGL $gpls(fwhm) $gpls(eta) $gpls_err(fwhm) $gpls_err(eta)] {}
    set legend "File = ${data}; GOF = $gof\n"
    append legend "Background = [formaterror $gpls(bkglow) $gpls_err(bkglow)]"
    append legend " to [formaterror $gpls(bkghigh) $gpls_err(bkghigh)]\n"
    append legend "FWHM = [formaterror $gpls(fwhm) $gpls_err(fwhm) ]"
    append legend " eta = [formaterror $gpls(eta)  $gpls_err(eta)]\n"
    catch {
	append legend "Integral breadth: L = [formaterror $gpls(bc) $gpls_err(bc)]"
	append legend " G = [formaterror $gpls(bg) $gpls_err(bg)]\n"
    }
    if {$gpls(asymflg)} {
	append legend "R = $gpls(radius); "
	append legend "H/2 = [formaterror $gpls(sampl_hgt) $gpls_err(sampl_hgt)]; "
	append legend "D/2 = [formaterror $gpls(det_hgt) $gpls_err(det_hgt)]\n"
    }
    for {set i 1} {$i <= $nline} {incr i} {
	append legend "Peak $i @ [formaterror $gpls(pos$i) $gpls_err(pos$i)], "
	append legend "area = [formaterror $gpls(area$i)  $gpls_err(area$i)]\n"
    }
    # avoid bug in BLT 2.3 where Inf does not work for text markers
    if {$blt_version == 2.3} {
	set xcen [lindex [$graph(fit) xaxis limits] 1]
	set ycen [lindex [$graph(fit) yaxis limits] 1]
    } else  {
	set xcen Inf
	set ycen Inf
    }
    if $gpls(fitonplot) {
	$graph(fit) marker create text -name "GOF" -text $legend \
		-coords "$xcen $ycen" -anchor ne -bg {} -justify right
    } else {
	catch {
	    toplevel .gpls
	    pack [label .gpls.l] -side top
	    pack [frame .gpls.bar] -side bottom -fill x
	    pack [button .gpls.bar.print -text "Save as:" -command {
		if [catch {
		    set fp [open [.gpls.bar.file get] a]
		    puts $fp [clock format [clock seconds]]
		    puts $fp [.gpls.l cget -text]
		    close $fp
		} errmsg ] {
		    tk_dialog .msg "error writing file" \
		    "Error writing file [.gpls.bar.file get]: $errmsg" \
		    error 0 OK
		}
	    }
	    ] -side left
	    pack [entry .gpls.bar.file] -side left
	    .gpls.bar.file delete 0 end
	    .gpls.bar.file insert 0 gpls.rpt
	    pack [button .gpls.bar.done -text Close -command "destroy .gpls"] -side right
	}
	.gpls.l config -text $legend
    }
    catch {grid $gpls(page).b.undo -row 11 -column 1 -columnspan 3}
}

# Voigt program to determin Gauss and Lorentz portions of peak
proc ComputeGL {fwhm eta err_fwhm err_eta} {
    global gpls
    foreach var {bc bg errbc errbg} {
	set $var 0
    }
    catch {
	set gvinp [open getvoit.inp w]
	puts $gvinp [format "%10.5f%10.5f%10.5f%10.5f" \
			 $fwhm $err_fwhm $eta $err_eta]
	close $gvinp
	exec $gpls(getvoigt) < getvoit.inp > getvoit.out
	set gvout [open getvoit.out r]
	set line [gets $gvout] 
	close $gvout
	file delete -force getvoit.out getvoit.inp
	scan $line "%10f%10f%10f%10f" \
	    bc errbc bg errbg
    }
    return [list $bc $bg $errbc $errbg]
}

# return a number with crystallographic formatted errors
proc formaterror {val err} {
    if {[string trim $err] == ""} {return $val}
    if {[catch {expr $val}] } {return $val}
    if {[catch {expr $err}] } {return "${val}(?)"}
    if {$err == 0} {return $val}
    # for errors above 1 treat as integers and report all digits
    if {$err > 1} {return [format %0.0f(%0.0f) $val $err]}
    set mult 10
    set dec 1
    while {[expr $mult * $err] < 2} {
	incr dec
	set mult [expr 10*$mult]
    }
    return [format "%.${dec}f(%0.0f)" $val  [expr $mult*$err]]
}

proc overalltable {} {
    global gpls
    set font {-*-Helvetica-Medium-R-Normal-*-*-120-*-*-*-*-*-*}
    set width 8
    set base $gpls(overallframe)
    label $base.label#1 -text Variable
    label $base.label#3 -text Value
    label $base.label#4 -text ref.

    label $base.label#5 -text {Low Bkg}
    entry $base.entry#2 -textvariable gpls(bkglow) -width $width
    catch {$base.entry#2 configure -font $font}
    checkbutton $base.checkbutton#7 -variable gpls(refine_1)

    label $base.label#6 -text {High Bkg}
    entry $base.entry#3 -textvariable gpls(bkghigh) -width $width
    catch {$base.entry#3 configure -font $font}

    checkbutton $base.checkbutton#8 -variable gpls(refine_2)
    label $base.label#7 -text FWHM
    entry $base.entry#4 -textvariable gpls(fwhm) -width $width
    catch {$base.entry#4 configure -font $font}
    checkbutton $base.checkbutton#10 -variable gpls(refine_3)
    label $base.label#9 -text eta
    entry $base.entry#5 -textvariable gpls(eta) -width $width
    catch {$base.entry#5 configure -font $font}
    checkbutton $base.checkbutton#11 -variable gpls(refine_4)

    label $base.radlbl -text Diameter
    entry $base.radent -textvariable gpls(radius) -width $width
    catch {$base.radent configure -font $font}
    
    label $base.hgtlbl -text "Sample Ht"
    entry $base.hgtent -textvariable gpls(sampl_hgt) -width $width
    catch {$base.hgtent configure -font $font}
    checkbutton $base.refineasym1 -variable gpls(refine_asym1)
    lappend gpls(asymlist) [label $base.detlbl -text "Detector Ht"]
    lappend gpls(asymlist) [entry $base.detent \
	    -textvariable gpls(det_hgt) -width $width]
    catch {$base.detent configure -font $font}
    checkbutton $base.refineasym2 -variable gpls(refine_asym2)

    checkbutton $base.useasym -text {Enable Asymmetry} \
		-variable gpls(asymflg) 

    # Geometry management

    grid $base.label#1 -in $base	-row 1 -column 1 
    grid $base.label#3 -in $base	-row 1 -column 2 
    grid $base.label#4 -in $base	-row 1 -column 3 
    grid $base.label#5 -in $base	-row 2 -column 1 
    grid $base.entry#2 -in $base	-row 2 -column 2 
    grid $base.checkbutton#7 -in $base -row 2 -column 3 
    grid $base.label#6 -in $base	-row 3 -column 1 
    grid $base.entry#3 -in $base	-row 3 -column 2 
    grid $base.checkbutton#8 -in $base	-row 3 -column 3 
    grid $base.label#7 -in $base	-row 4 -column 1 
    grid $base.entry#4 -in $base	-row 4 -column 2 
    grid $base.checkbutton#10 -in $base	-row 4 -column 3 
    grid $base.label#9 -in $base	-row 5 -column 1 
    grid $base.entry#5 -in $base	-row 5 -column 2 
    grid $base.checkbutton#11 -in $base	-row 5 -column 3 

    grid $base.useasym -in $base	-row 6 -column 1  \
	    -columnspan 3 -sticky ew

    grid $base.radlbl  -in $base 	-row 7 -column 1 
    grid $base.radent  -in $base	-row 7 -column 2 
    grid $base.hgtlbl  -in $base	-row 8 -column 1 
    grid $base.hgtent  -in $base	-row 8 -column 2 
    grid $base.refineasym1 -in $base	-row 8 -column 3 
    grid $base.detlbl  -in $base	-row 9 -column 1 
    grid $base.detent  -in $base	-row 9 -column 2 
    grid $base.refineasym2 -in $base	-row 9 -column 3 

    # Resize behavior management
    grid rowconfigure $base 10 -weight 1
    grid columnconfigure $base 0 -weight 0 -minsize 20
    grid columnconfigure $base 8 -weight 1
}
proc peaktable {} {
    global gpls
    set font {-*-Helvetica-Medium-R-Normal-*-*-120-*-*-*-*-*-*}
    set width 8
    eval destroy [grid slaves $gpls(peakframe)]
    set base $gpls(peakframe)

    grid [label $base.label#0  -text {# }] -row 1 -column 0
    grid [label $base.label#26 -text Use] -row 1 -column 1
    grid [label $base.label#18 -text position] -row 1 -column 2 
    grid [label $base.label#16 -text ref.] -row 1 -column 3 
    grid [label $base.label#19 -text area] -row 1 -column 4 
    grid [label $base.label#17 -text ref.] -row 1 -column 5 
    grid rowconfigure $base 1 -weight 0 -minsize 30
    grid columnconfigure $base 1 -weight 0 -minsize 20
    grid columnconfigure $base 2 -weight 0 -minsize 30
    grid columnconfigure $base 3 -weight 0 -minsize 20
    grid columnconfigure $base 4 -weight 0 -minsize 30
    grid columnconfigure $base 5 -weight 0 -minsize 20
    grid columnconfigure $base 6 -weight 0 -minsize 30
    for {set i 1} {$i < 9} {incr i} {
	set row [expr 1 + $i]
	grid [label $base.lbl_$i -textvariable gpls(label$i) \
		] -row $row -column 0
	grid [checkbutton $base.use_$i -variable gpls(use$i) \
		-command "enablepeak $base $i" \
		] -row $row -column 1
	grid [entry $base.pos_$i -textvariable gpls(pos$i) -width $width \
		] -row $row -column 2
	catch {$base.pos_$i configure -font $font}
	grid [checkbutton $base.refpos_$i -variable gpls(refpos$i) \
		] -row $row -column 3
	grid [entry $base.area_$i -textvariable gpls(area$i) -width $width \
		] -row $row -column 4
	catch {$base.area_$i configure -font $font}
	grid [checkbutton $base.refarea_$i -variable gpls(refarea$i) \
		] -row $row -column 5
	grid [button $base.set_$i -text set \
		-command "setpeakposition $base $i" \
		] -row $row -column 6
	set gpls(use$i) 0
	enablepeak $base $i
    }
    grid rowconfigure $base 11 -weight 1
}

proc enablepeak {base i} {
    global gpls graph
    catch {$graph(blt) marker delete peak$i}
    if $gpls(use$i) {
	$base.pos_$i config -state normal -fg black
	$base.refpos_$i config -state normal -variable gpls(refpos$i)
	$base.area_$i config -state normal -fg black
	$base.refarea_$i config -state normal -variable gpls(refarea$i)
	$graph(blt) marker create line -name peak$i \
		-coords "$gpls(pos$i) -Inf $gpls(pos$i) Inf" 
	set clr [lindex $graph(colorlist) \
		     [expr $i % [llength $graph(colorlist)]] \
		    ]
	$graph(blt) marker config peak$i $graph(MarkerColorOpt) $clr
	catch {
	    if {$graph(fit) != $graph(blt)} {
		$graph(fit) marker create line -name peak$i \
		    -coords "$gpls(pos$i) -Inf $gpls(pos$i) Inf" 
		$graph(fit) marker config peak$i $graph(MarkerColorOpt) $clr
	    }
	}
    } else {
	$base.pos_$i config -state disabled -fg #888
	$base.refpos_$i config -state disabled -variable {}
	$base.area_$i config -state disabled -fg #888
	$base.refarea_$i config -state disabled -variable {}
    }
}

proc setpeakposition {base i} {
    global gpls graph
    $graph(blt) config -title {[Re]defining peak, press P at maximum}
    set oldGrab [grab current .]
    if {$oldGrab != ""} {
        set grabStatus [grab status $oldGrab]
    }
    grab -global .plot
    focus $graph(blt)
    raise .plot .
    if [catch {set gpls(origcursor)}] {
	set gpls(origcursor) [$graph(blt) cget -cursor]
    }
    $graph(blt) config  -cursor sb_up_arrow
    # get binding for graph
    set bindtag $graph(blt)
    catch {
	if {[bind bltZoomGraph] != ""} {set bindtag bltZoomGraph}
    }
    bind $bindtag <Control-Button-1> "definepeak %W %x %y $i"
    bind .plot <Key-P> "definepeak $graph(blt) %x %y $i"
    bind .plot <Key-p> "definepeak $graph(blt) %x %y $i"
    tkwait variable gpls(area$i)
    set gpls(use$i) 1
    enablepeak $gpls(peakframe) $i
    grab release .plot
    catch {focus $oldFocus}
    if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
    }
    after 50
    lower .plot .
    $graph(blt) config -cursor $gpls(origcursor)
}

proc definepeak {plot x y num} {
    global gpls graph blt_version
#    addpeak $graph(peakframe)
    set gpls(pos$num) [$plot xaxis invtransform $x]
    # take a wild stab at a peak width
    set gpls(area$num) [expr {[$plot yaxis invtransform $y] / 0.05 }]
    # use a better one if available
    catch {set gpls(area$num) [expr {0.05 * $gpls(area$num) / $gpls(fwhm) }]}
    # get binding for graph
    set bindtag $graph(blt)
    catch {
	if {[bind bltZoomGraph] != ""} {set bindtag bltZoomGraph}
    }
    bind $bindtag <Control-Button-1> ""
    bind .plot <Key-P> "addpeak $graph(blt) %x %y"
    bind .plot <Key-p> "addpeak $graph(blt) %x %y"
#    $plot config -title {}
    $graph(blt) config -title "Add peaks: press P at maximum"
}

proc addpeak {plot x y} {
    global gpls graph blt_version command
    # are we running fit?
    if {$command(pagenow) != "fit"} return
    # set num to the 1st peak not used
    for {set i 1} {$i < 9} {incr i} {
	set num $i
	if {!$gpls(use$i)} break
    }
    set pos  [$plot xaxis invtransform $x]
    set hgt [$plot yaxis invtransform $y] 
    set gpls(pos$num) $pos
    set bkg 0
    catch {
	set bkg [expr {$gpls(bkglow) + ($gpls(bkghigh)-$gpls(bkglow)) * \
			   ($gpls(max)-$pos) / ($gpls(max)-$gpls(min))}]
    }
    set gpls(area$num) [expr ($hgt-$bkg) / $gpls(fwhm) ]
    # clear the "Set" binding
    # get binding for graph
    set bindtag $graph(blt)
    catch {
	if {[bind bltZoomGraph] != ""} {set bindtag bltZoomGraph}
    }
    bind $bindtag <Control-Button-1> " "
    bind .plot <Key-P> "addpeak $graph(blt) %x %y"
    bind .plot <Key-p> "addpeak $graph(blt) %x %y"
    #    $plot config -title {}
    $graph(blt) config -title "Add peaks: press P at maximum"
    set gpls(use$i) 1
    set gpls(label$i) {}
    enablepeak $gpls(peakframe) $num
}
proc putontop {w} {
    # center window $w above its parent and make it stay on top
    set wp [winfo parent $w]
#    wm transient $w [winfo toplevel $wp]
    wm withdraw $w
    update idletasks
    # center the new window in the middle of the parent
    set x [expr [winfo x $wp] + [winfo width $wp]/2 - \
	    [winfo reqwidth $w]/2 - [winfo vrootx $wp]]
    if {$x < 0} {set x 0}
    set xborder 10
    if {$x+[winfo reqwidth $w] +$xborder > [winfo screenwidth $w]} {
	incr x [expr \
		[winfo screenwidth $w] - ($x+[winfo reqwidth $w] + $xborder)]
    }
    set y [expr [winfo y $wp] + [winfo height $wp]/2 - \
	    [winfo reqheight $w]/2 - [winfo vrooty $wp]]
    if {$y < 0} {set y 0}
    set yborder 25
    if {$y+[winfo reqheight $w] +$yborder > [winfo screenheight $w]} {
	incr y [expr \
		[winfo screenheight $w] - ($y+[winfo reqheight $w] + $yborder)]
    }
    wm geom $w +$x+$y
    wm deiconify $w

    global makenew
    set makenew(OldFocus) [focus]
#    catch {
#	set makenew(OldGrab) [grab current $w]
#	if {$makenew(OldGrab) != ""} {
#	    set makenew(GrabStatus) [grab status $makenew(OldGrab)]
#	}
#	grab $w
#    }
}

proc afterputontop {} {
    # restore focus
    global makenew
    catch {focus $makenew(OldFocus)}
#    if {$makenew(OldGrab) != ""} {
#	catch {
#	    if {$makenew(GrabStatus) == "global"} {
#		grab -global $makenew(OldGrab)
#	    } else {
#		grab $makenew(OldGrab)
#	    }
#        }
#    }
}

proc SetFitRange {} {
    global gpls
    if {$gpls(dataset) == ""} return
    if {$gpls(dataset) == "(none selected)"} return
    set frm .file
    catch {destroy $frm}
    toplevel $frm
    wm title $frm "Set Fit Limits"
    grid [frame $frm.a -bd 4 -relief groove] -column 1 -row 1 -sticky ew
    pack [label $frm.a.a -text "Fit range: "] \
	    -side left -anchor center
    pack [entry $frm.a.b -width 12 -textvariable gpls(min)] \
	    -side left -anchor center
    pack [entry $frm.a.c -width 12 -textvariable gpls(max)] \
	    -side left -anchor center
    pack [button $frm.a.d -text "Set" -command ManualSetFitRange] \
	    -side left -anchor center
    grid [frame $frm.b -bd 4] -column 1 -row 2
    pack [button $frm.b.d -text "Set from zoom" -command GrabFromZoom] \
	    -side left -anchor center
    pack [button $frm.b.e -text "Reset zoom" -command ResetFitZoom] \
	    -side left -anchor center
    grid [frame $frm.c -bd 4 -relief groove] -column 1 -row 3 -sticky ew
    pack [label $frm.c.a -text "Increment: "] \
	    -side left -anchor center
    pack [entry $frm.c.c -width 8 -textvariable gpls(incr)] \
	    -side left -anchor center
    pack [button $frm.c.d -text "Add increment to limits" \
	    -command IncrFitRange] \
	    -side left -anchor center
    grid [frame $frm.e -bd 4 ] -column 1 -row 4
    pack [label $frm.e.a -text "Adjust\nRange: "] \
	    -side left -anchor center
    pack [button $frm.e.b1 -text "Back 1.0" \
	    -command "GrabFromZoom; IncrFitRange -1."] \
	    -side left -anchor center
    pack [button $frm.e.b -text "Back 0.1" \
	    -command "GrabFromZoom; IncrFitRange -.1"] \
	    -side left -anchor center
    pack [button $frm.e.e -text "Expand" \
	    -command "GrabFromZoom; IncrFitRange .05 expand"] \
	    -side left -anchor center
    pack [button $frm.e.f -text "Forward 0.1" \
	    -command "GrabFromZoom; IncrFitRange .1"] \
	    -side left -anchor center
    pack [button $frm.e.f1 -text "Forward 1.0" \
	    -command "GrabFromZoom; IncrFitRange 1."] \
	    -side left -anchor center
    grid [frame $frm.d -bd 4 ] -column 1 -row 5
    pack [button $frm.d.d -text "Close" \
	    -command "destroy $frm"] \
	    -side top -anchor center
    #putontop $frm
    focus $frm.b.d
    update
    #tkwait window $frm
    #afterputontop
}

proc ManualSetFitRange {} {
    global gpls graph
    set graph(plotlist) $gpls(dataset)
    displaylist
    if [catch {expr $gpls(min)}] return
    if [catch {expr $gpls(max)}] return
    $graph(blt) xaxis config -min $gpls(min) -max $gpls(max)
    update 
    setdatarange
    set numobs [expr $gpls(xhigh) - $gpls(xlow) + 1]
    set gpls(rangemsg) [format \
	    "limits %.2f to %.2f\n (%d points)" \
	    $gpls(min) $gpls(max) $numobs]
    #if {$numobs > 999 || $numobs < 10} return
}

proc GrabFromZoom {} {
    global gpls graph command
    if {$command(pagenow) != "fit"} return
    catch {set gpls(max) [format %.5f [$graph(blt) xaxis cget -max]]}
    catch {set gpls(min) [format %.5f [$graph(blt) xaxis cget -min]]}
    ManualSetFitRange
}

proc ResetFitZoom {} {
    global gpls graph
    set graph(plotlist) $gpls(dataset)
    displaylist
    # zoom reset that works (in later versions?) to clear incomplete zoom
    catch {blt::ResetZoom $graph(blt)}
    # reset zoom stack & reset limits
    catch {Blt_ZoomStack $graph(blt)}
    $graph(blt) xaxis config -min {} -max {}
    $graph(blt) yaxis config -min {} -max {}
}

proc IncrFitRange {"increment {}" "option {}"} {
    global gpls command graph
    if {$command(pagenow) != "fit"} return
    if [catch {expr $gpls(min)}] return
    if [catch {expr $gpls(max)}] return
    set expand 1
    if {$option == "expand"} {set expand -1}
    if {$increment == ""} {
	set increment $gpls(incr)
    } else {
	if [catch \
		{set increment [expr $increment*($gpls(max)-$gpls(min))]} \
		] return
    }
    set gpls(min) [expr $gpls(min) + $expand * $increment]
    set gpls(max) [expr $gpls(max) + $increment]
    ManualSetFitRange
    catch {$graph(blt) yaxis config -min $gpls(ymin) -max $gpls(ymax)}
}

# undo the last cycle
proc UndoFit {} {
    global gpls graph
    eval $graph(lastcycle)
    set graph(lastcycle) {}
    catch {grid forget $gpls(page).b.undo}
}

proc StoreFit {} {
    global gpls graph
    global $gpls(dataset)
    incr graph(fitnum)
    set data $gpls(dataset)_fit_$graph(fitnum)
    set data [initdata $data]
    global ${data}
    foreach item {xlabel xunits ylabel skip title} {
	set ${data}($item) [set $gpls(dataset)($item)]
    }
    set ${data}(x) [xvec range 0 end]
    set ${data}(y) [ycvec range 0 end]
    resetdata $data
    tk_dialog .msg "Created" \
		    "Created dataset $data" \
		    "" 0 OK
}

bind .plot <Key-s> GrabFromZoom
bind .plot <Key-f> "GrabFromZoom; IncrFitRange .1"
bind .plot <Key-b> "GrabFromZoom; IncrFitRange -.1"
bind .plot <Key-e> "GrabFromZoom; IncrFitRange .05 expand"
bind .plot <Key-S> GrabFromZoom
bind .plot <Key-F> "GrabFromZoom; IncrFitRange .1"
bind .plot <Key-b> "GrabFromZoom; IncrFitRange -.1"
bind .plot <Key-E> "GrabFromZoom; IncrFitRange .05 expand"
