#----------------------------------------------------------------
# set default values for lots of variables
if {$tcl_platform(platform) == "windows"} {
    set cellparm(hklgenprog) [file join $command(exedir) hklgen.exe]
    set cellparm(tstextprog) [file join $command(exedir) testextn.exe]
    set cellparm(calc2tprog) [file join $command(exedir) calcpos.exe]
} else {
    set cellparm(hklgenprog) [file join $command(exedir) hklgen]
    set cellparm(tstextprog) [file join $command(exedir) testextn]
    set cellparm(calc2tprog) [file join $command(exedir) calcpos]
}
set cellparm(spacegroup) (none)
set cellparm(extspg) (none)
set cellparm(zero) 0
set cellparm(old_zero) 0
set cellparm(old_ymin) 0
set cellparm(old_ymax) 0
set cellparm(old_wave) -1
set cellparm(oldextspg) {xxx} 
set cellparm(sens) 25
set cellparm(outcount) 0
set cellparm(twothetamax) 168
set cellparm(wavelength) "1.5418"
set cellparm(alpha) [set cellparm(beta) [set cellparm(gamma) 90]]
set cellparm(a) [set cellparm(b) [set cellparm(c) 10]]
set cellparm(laue) 8
set cellparm(sens) 25
set cellparm(fadetime) 10
set cellparm(ymin) 0
set cellparm(ymax) 1000
set cellparm(abc_tol) 0.01
set cellparm(alphabeta_tol) 0.01
set command(writeunits) 0
set command(readbuttons) {}
set command(readbuttonproc) {}
set command(helpmode) 1
# command(pagenow) is used to indicate which screen is currently shown
set command(pagenow) ""
set graph(outname) out.ps
set graph(outcmd) "lpr -h"
if {$tcl_platform(platform) == "windows"} {
    set graph(printout) 1
} else {
    set graph(printout) 0
}
set graph(title) ""
set graph(legend) 0
set cellparm(fontsize) 15
set graph(font) 14
set graph(ReversePlotOrder) 0
set graph(LabelByTitle) 0
set graph(editcellcolor) blue
set graph(extinctcolor) yellow
# default list of colors
set graph(colorlist) {black red green blue magenta cyan yellow4 \
	navy purple red4 darkolivegreen darkcyan royalblue4}
# define a font used for labels
if {$tcl_version >= 8.0} {
    catch {font delete lblfont}
    font create lblfont -family Helvetica -size [expr -$cellparm(fontsize)]
} 
set graph(units) 0
set menulist(options) {}
# maximum lines that are allowed to be read from most file formats
set command(maxlines) 50000

# called by a trace on cellparm(alphabeta_tol) and cellparm(abc_tol)
proc setcellSliderResolution {a b c} {
    global cellparm
    set base $cellparm(celleditbase)
    catch {
	foreach var {a b c} {
	    $base.s$var config -resolution $cellparm(abc_tol)
	}
	$base.szero config -resolution $cellparm(abc_tol)
    }
    catch {
	foreach var {alpha beta gamma} {
	    $base.s$var config -resolution $cellparm(alphabeta_tol)
	}
    }
}

# called by a trace on cellparm(fontsize)   
proc setfontsize {a b c} {
    global cellparm graph
    catch {
	font config lblfont -size [expr -$cellparm(fontsize)]
	# this forces a redraw of the plot by changing the title to itself
	$graph(blt) configure -title [$graph(blt) cget -title]
    }
}

#called by a trace on graph(title)
proc setgraphtitle {a b c} {
	global graph
	$graph(blt) configure -title $graph(title)
}
#-------------------------------------------------------------------------
#
# Convenient "quit" key
#
if {$tcl_platform(platform) == "unix"} {
    bind all <Control-KeyPress-c> {Setup Quit}
}
focus .
vector xextinct 
vector yextinct 
set graph(datalist) {}
set graph(plotlist) {}
array set command  {xoffset 0 xmult 1 yoffset 0 ymult 1 choice "" xstagger 0 ystagger 0}
#-------------------------------------------------------------------------
# create a window for the plot
catch {destroy .plot}
toplevel .plot
wm geometry .plot =900x600+0+0
# build the graph
pack [set graph(blt) [graph .plot.gr]] -fill both -expand yes -side top
#-------------------------------------------------------------------------
# define a binding to show the cursor location
proc ToggleLiveCursor {} {
    if {[bind .plot <Any-Motion>] == ""} {
	pack [frame .plot.bot -bd 2 -relief sunken] -side bottom -fill x
	pack [label .plot.bot.val1 -textvariable graph(position)] -side left
	pack [button .plot.bot.close -command ToggleLiveCursor -text "Close cursor display"] -side right
	bind .plot <Any-Motion> {FormatLiveCursor %x %y}
    } else {
	destroy .plot.bot
	bind .plot <Any-Motion> {}
    }
}
proc FormatLiveCursor {x y} {
    global graph
    set graph(position) \
	    "x=[format %.3f [$graph(blt) xaxis invtransform $x]] y=[format %.3f [$graph(blt) yaxis invtransform $y]]"
}
bind .plot <Key-l> {ToggleLiveCursor}
bind .plot <Key-L> {ToggleLiveCursor}
bind .plot <Key-Up> "ScanZoom .plot.gr %K .1"
bind .plot <Key-Left> "ScanZoom .plot.gr %K .1"
bind .plot <Key-Right> "ScanZoom .plot.gr %K .1"
bind .plot <Key-Down> "ScanZoom .plot.gr %K .1"
bind .plot <Control-Key-Up> "ScanZoom .plot.gr %K 1.0"
bind .plot <Control-Key-Left> "ScanZoom .plot.gr %K 1.0"
bind .plot <Control-Key-Right> "ScanZoom .plot.gr %K 1.0"
bind .plot <Control-Key-Down> "ScanZoom .plot.gr %K 1.0"
# move the zoom region around
proc ScanZoom {box key frac} {
    foreach var  {xl xh yl yh} axis {xaxis  xaxis  yaxis  yaxis} \
	    flg  {-min -max -min -max} {
	set $var [$box $axis cget $flg]
	if {$var == ""} return
    }
    catch {
	switch -- $key {
	    Right {set a x; set l $xl; set h $xh; set d [expr {$frac*($h-$l)}]}
	    Left {set a x; set l $xl; set h $xh; set d [expr {-$frac*($h-$l)}]}
	    Up   {set a y; set l $yl; set h $yh; set d [expr {$frac*($h-$l)}]}
	    Down {set a y; set l $yl; set h $yh; set d [expr {-$frac*($h-$l)}]}
	}
	$box ${a}axis configure -min [expr {$l + $d}] -max [expr {$h + $d}]
    }
}
#-------------------------------------------------------------------------
# manual zoom option
proc BLTmanualZoom {} {
    global graph
    catch {toplevel .zoom}
    eval destroy [grid slaves .zoom]
    raise .zoom
    wm title .zoom {Manual Scaling}
    grid [label .zoom.l1 -text minimum] -row 1 -column 2 
    grid [label .zoom.l2 -text maximum] -row 1 -column 3 
    grid [label .zoom.l3 -text x] -row 2 -column 1 
    grid [label .zoom.l4 -text y] -row 3 -column 1 
    grid [entry .zoom.xmin -textvariable graph(xmin) -width 10] -row 2 -column 2 
    grid [entry .zoom.xmax -textvariable graph(xmax) -width 10] -row 2 -column 3 
    grid [entry .zoom.ymin -textvariable graph(ymin) -width 10] -row 3 -column 2 
    grid [entry .zoom.ymax -textvariable graph(ymax) -width 10] -row 3 -column 3 
    grid [frame .zoom.b] -row 4 -column 1 -columnspan 3
    grid [button .zoom.b.1 -text "Set Scaling" \
	     -command "SetManualZoom set"]  -row 4 -column 1 -columnspan 2
    grid [button .zoom.b.2 -text Reset \
	    -command "SetManualZoom clear"] -row 4 -column 3
    grid [button .zoom.b.3 -text Close -command "destroy .zoom"] -row 4 -column 4 
    grid rowconfigure .zoom 1 -weight 1 -pad 5
    grid rowconfigure .zoom 2 -weight 1 -pad 5
    grid rowconfigure .zoom 3 -weight 1 -pad 5
    grid rowconfigure .zoom 4 -weight 0 -pad 5
    grid columnconfigure .zoom 1 -weight 1 -pad 20
    grid columnconfigure .zoom 1 -weight 1 
    grid columnconfigure .zoom 3 -weight 1 -pad 10
    foreach item {min min max max} \
	    format {3   2   3   2} \
	    axis   {x   y   x   y} {
	set val [$graph(blt) ${axis}axis cget -${item}]
	set graph(${axis}${item}) {(auto)}
	catch {set graph(${axis}${item}) [format %.${format}f $val]}
    }
}

proc SetManualZoom {mode} {
    global graph
    if {$mode == "clear"} {
	foreach item {xmin ymin xmax ymax} {
	    set graph($item) {(auto)}
	}
    }
    foreach item {xmin ymin xmax ymax} {
	set $item {}
	catch {set $item [expr $graph($item)]}	
    }
    # zoom reset that works (in later versions?) to clear incomplete zoom
    catch {blt::ResetZoom $graph(blt)}
    # reset the zoomstack
    catch {Blt_ZoomStack $graph(blt)}
    catch {$graph(blt) xaxis config -min $xmin -max $xmax}
    catch {$graph(blt) yaxis config -min $ymin -max $ymax}
}
bind .plot <Key-z> {BLTmanualZoom}
bind .plot <Key-Z> {BLTmanualZoom}
set helplist(Shortcuts) {\
There are a number of shortcuts implemented by pressing 
keys or mouse buttons. Keys are not case-sensitive.

in plot window/any page
   L -- shows "live" cursor position
   Z -- set zoom scaling manually
   left click -- zoom in
   right click -- zoom out (Mac: Apple+click)

in plot window/EditCell page only
   shift+left click -- label nearby reflections
   shift+double left click -- label all reflections
   shift+right click -- delete reflection labels 
                        (Mac: shift+Apple+click)
   H -- label nearby reflections
   A -- label all reflections
   D -- delete reflection labels

in plot window/Fit page only
   control+double left click -- add peak at current position
   S -- set fit range to current zoom
   F -- scroll the fit region forward
   B -- scroll thefit region backward
   E -- expand the scroll fit region
   P -- define peak position 

in any window/any page:
  control+C -- exits the program
  F1 key    -- show help on current page

Also see File Selection Tricks
}

#-------------------------------------------------------------------------
if [file exists [file join $scriptdir opts.tcl]] {
    source [file join $scriptdir opts.tcl]
}
set command(ExecuteLater) {}
foreach f { data.tcl read.tcl write.tcl symsubs.tcl hklgen.tcl \
    fit.tcl index.tcl addlogic.tcl linterp.tcl fitwidth.tcl \
    editfile.tcl \
    } {
    set file [file join $scriptdir $f]
    if {[file exists $file]} {source $file}
}
if {!$command(haveBW)} {source [file join $scriptdir notebook.tcl]}

# prepare lists of routines to read and write data (so they can be manipulated 
# in the local files)
set command(readlist)  [lsort [glob -nocomplain [file join $scriptdir read_*.tcl]]]
set command(writelist) [lsort [glob -nocomplain [file join $scriptdir write_*.tcl]]]

# process local customization 
set readfilelist {}
set filelist [list [file join $scriptdir localcmds.tcl]]
lappend filelist [list [file join $scriptdir localconfig.tcl]]
if {$tcl_platform(platform) == "windows"} {
    lappend filelist "C:/cmpr.init" [file normalize ~/cmpr.init]
} else {
    lappend filelist "~/.cmpr_init"
}
foreach file $filelist {
    # add any personalized defaults
    if [file exists $file] {
	if {[catch {
	    source $file
	    lappend readfilelist $file
	} err]} {
	    MyMessageBox -parent . -title "Error customizing" \
		-icon error -type Continue -default continue -message \
		"An error was encountered reading customization file $file:\n$err"
	}
    }
}
XferDefaults
if {[llength $readfilelist] > 0} {
    set ::command(initfiles) [concat $readfilelist $::command(initfiles)]
}

# there is an initial working directory defined -- use it
if {[array name command initwd] != ""} {
    catch {cd $command(initwd)}
    set command(pwd) [pwd]
}

######################################################################
# read in the read_ routines that define the read formats
######################################################################
# process preferred items first
set preflist {}
set skiplist {}
catch {set preflist $command(prefreadlist)}
catch {set skiplist $command(skipreadlist)}
foreach item $preflist {
    foreach file $command(readlist) {
	if {[string match -nocase *$item* [file tail $file]]} {
	    # remove the match from the list
	    set i [lsearch $command(readlist) $file]
	    set command(readlist) [lreplace $command(readlist) $i $i]
	    if {[catch {
		source $file
	    } err ]} {
		MyMessageBox -parent . -title "Error customizing" \
		    -icon error -type Continue -default continue -message \
		    "An error was encountered reading read format file $file:\n$err"
	    }
	}
    }
}
# remove the unwanted options
foreach item $skiplist {
    foreach file $command(readlist) {
	if {[string match -nocase *$item* [file tail $file]]} {
	    # remove the match from the list
	    set i [lsearch $command(readlist) $file]
	    set command(readlist) [lreplace $command(readlist) $i $i]
	}
    }
}

# now process the rest, skipping over anything handled before
foreach file $command(readlist) {
    if {[catch {
	source $file
    } err ]} {
	MyMessageBox -parent . -title "Error customizing" \
	    -icon error -type Continue -default continue -message \
	    "An error was encountered reading read format file $file:\n$err"
    }
}

######################################################################
# read in the write_ routines that define the write formats
######################################################################
# process preferred items first
set preflist {}
set skiplist {}
catch {set preflist $command(prefwritelist)}
catch {set skiplist $command(skipwritelist)}
foreach item $preflist {
    foreach file $command(writelist) {
	if {[string match -nocase *$item* [file tail $file]]} {
	    # remove the match from the list
	    set i [lsearch $command(writelist) $file]
	    set command(writelist) [lreplace $command(writelist) $i $i]
	    if {[catch {
		source $file
	    } err ]} {
		MyMessageBox -parent . -title "Error customizing" \
		    -icon error -type Continue -default continue -message \
		    "An error was encountered reading write format file $file:\n$err"
	    }
	}
    }
}
# remove the unwanted options
foreach item $skiplist {
    foreach file $command(writelist) {
	if {[string match -nocase *$item* [file tail $file]]} {
	    # remove the match from the list
	    set i [lsearch $command(writelist) $file]
	    set command(writelist) [lreplace $command(writelist) $i $i]
	}
    }
}
# now process the rest, skipping over anything handled before
foreach file $command(writelist) {
    if {[catch {
	source $file
    } err ]} {
	MyMessageBox -parent . -title "Error customizing" \
	    -icon error -type Continue -default continue -message \
	    "An error was encountered reading write format file $file:\n$err"
    }
}

# set the default sizes
catch {SetTkDefaultOptions $graph(font)}
# define the color map
definecolors   
#----------------------------------------------------------------
# and now on with the code
#----------------------------------------------------------------
proc DoCommand {cmd} {
    global command
    # draw the page
    if {[catch {Setup $cmd} error]} {
	if {$command(debug)} {puts $error}
    }
}

# post a command dialog page
proc Setup {cmd} {
    global command
    if {$cmd == "Quit"} {
	if {[tk_messageBox -message "Really quit?" -type yesno -icon question] \
		== "yes"} {exit}
    } elseif $command(haveBW) {
	set pagename [string tolower $cmd]
	$command(notebook) see $pagename
	$command(notebook) raise $pagename
    } else {
	Notebook:raise $command(notebook) $cmd
    }
    set command(pagenow) [string tolower $cmd]
}
# resize the notebook to fit all the tabs and the largest page
proc ResizeNotebook {} {
    global command
    if {$command(haveBW)} {
	$command(notebook) compute_size
    } else {
	Notebook:resize $command(notebook)
    }
}
#----------------------------------------------------------------
# final initializations
#----------------------------------------------------------------
# locations of the extinction condition sub-menus
set cellparm(extinctmenuH) {}
set cellparm(extinctmenuE) {}
resetextinctions  
trace variable extinctlbl w extinctlblset 
trace variable cellparm(fontsize) w setfontsize
trace variable cellparm(abc_tol) w setcellSliderResolution
trace variable cellparm(alphabeta_tol) w setcellSliderResolution
trace variable graph(title) w setgraphtitle

#----------------------------------------------------------------
#Quit
lappend menulist(file) Quit 
set helplist(Quit) {Quit the program}
#----------------------------------------------------------------
#Plot
lappend menulist(display) Plot
lappend menulist(pages) Plot
set helplist(Plot) {\
The plot page is used to display one or more buffers. To select 
a single buffer, click on the buffer name.  A range of buffers 
can be selected by dragging with the mouse. Individual buffers 
can be selected or deselected by holding the control key and 
clicking on the buffer name. The plot is updated when the 
"Update Plot" button is pressed or with a mouse double-click.

The "Postscript out" button causes a postscript copy of the 
plot window to be created. Settings on the Options page 
determine where the file is created or how it is printed. 
Plots can be exported to, a publication-quality graphics package 
where the displayed plot can be enchanced and annotated.

This page is also used to change the way a file is displayed. 
Do this by selecting the file(s) to be changed and then using 
the appropriate checkboxes to select the color, line type, 
symbol type and size. Changes will be made when the "Apply 
Changes" button is pressed.
}

proc MakePlot {page} {
#    pack [frame $page.a] -side left -anchor nw -fill y
    grid [frame $page.a] -column 0 -row 0 -rowspan 2 -sticky nw
    pack [label $page.a.t1 -text {Select dataset(s) to}] -side top
    pack [label $page.a.t2 -text {plot or change...}] -side top
    pack [frame $page.a.2 ] -side top 
    makeselectbox $page.a.2
	pack [checkbutton $page.a.viewpdf \
	    -text "AutoOpen PDF" -variable graph(viewpdfout)] -side bottom
    pack [button $page.a.pdf -text "PDF out" \
	    -command makepdfout] -side bottom
    pack [button $page.a.h -text "PostScript out" \
	    -command makepostscriptout] -side bottom
    pack [button $page.a.p -text "Update Plot" \
	    -command "getfilelist $page.a.2; displaylist"] -side bottom

    grid [frame $page.g] -column 1 -row 0 -sticky nsew
    grid [frame $page.g.a -bd 2 -relief groove] -column 1 -row 0 -sticky n
    grid [label $page.g.a.1 -text "Reorder"] -column 1 -row 1 -columnspan 2
    grid [button $page.g.a.2 -text "+" -padx 1m -pady 0 \
	    -command "ReorderBox $page.a.2.1 1"] -column 1 -row 2
    grid [button $page.g.a.3 -text "-"  -padx 1m -pady 0  \
	    -command "ReorderBox $page.a.2.1 0"] -column 2 -row 2
    grid [checkbutton $page.g.a1 \
	    -text "Plot in\nlisted order" -variable graph(ReversePlotOrder) \
	    -command displaylist]  -column 1 -row 1 -sticky n
    grid rowconfig $page.g 2 -weight 1
    grid [checkbutton $page.g.b1 \
	    -text "Plot\nLegend" -variable graph(legend) \
	    -command graphlegend]  -column 1 -row 6 -sticky s
    grid [checkbutton $page.g.b2 \
	    -text "Use Title\nfor Legend" -variable graph(LabelByTitle) \
	    -command displaylist]  -column 1 -row 7 -sticky s
    grid [button $page.g.1 -text "Export\nto Grace" \
	    -command makeGraceExport] -column 1 -row 8 -sticky s
    grid [button $page.g.2 -text "Export\nto CSV" \
	    -command "exportPlotSpreadsheet ."] -column 1 -row 9 -sticky s
    if {$::tcl_platform(platform) == "windows"} {    
	grid [button $page.g.3 -text "Export to\nClipboard" \
		  -command "$::graph(blt) snap -format emf CLIPBOARD"] \
	    -column 1 -row 5 -sticky s
    }
#    grid [button $page.g -text "Export\nto Grace" \
#	    -command makeGraceExport] -column 1 -row 1 -sticky sw

    grid columnconfigure $page 1 -weight 1
#    pack [frame $page.b -bd 4 -relief groove] -side top -anchor ne -fill y
    grid [frame $page.b -bd 4 -relief groove] \
	    -column 2 -columnspan 2 -row 0 -sticky ne
    pack [frame $page.b.1] -side top -anchor n -fill both
    pack [frame $page.b.2] -side top -anchor n -fill both
    pack [frame $page.b.1.b] -side left -anchor n -fill both
    pack [label $page.b.1.b.t -text Line] -side top 
    foreach symbol {"" 0 1 2 4} \
	    symbol_name {"no change" "no line" "thin line" "medium" "thick"} {
	pack [radiobutton $page.b.1.b.$symbol_name \
		-text $symbol_name -variable command(line) \
		-value $symbol] -side top -anchor w
    }

    pack [frame $page.b.1.c] -side left -anchor n
    grid [label $page.b.1.c.t -text Color] -row 0 -column 0 -columnspan 2 
    # define a scrolled canvas for color entries
    set colorcan $page.b.1.c.b
    set colorscr $page.b.1.c.y 
    set colorfrm $page.b.1.c.b.f
    set colorlist [concat {"no change"} $::graph(colorlist)]
    grid [canvas $colorcan \
	    -scrollregion {0 0 5000 500} -width 100 -height 200 \
	    -yscrollcommand "$colorscr set" -bg lightgrey] \
	    -sticky  news -row 1 -column 1
    grid [scrollbar $colorscr -orient vertical \
	    -command "$colorcan yview"] \
	    -sticky ns -row 1 -column 2
    frame $colorfrm
    $colorcan create window 0 0 -anchor nw  -window $colorfrm
    # fill the color canvas frame
    foreach color $colorlist {
	grid [radiobutton $colorfrm.$color \
	    -text $color -variable command(color) -value $color \
	    ] -sticky w
	if {$color != "no change"} {$colorfrm.$color config -fg $color}
    }
    pack [frame $page.b.1.d] -side left -anchor n
    pack [label $page.b.1.d.t -text "Symbol type"] -side top
    foreach symbol {"" none square circle diamond plus cross \
	    splus scross} \
	    symbol_name {"no change" none square circle diamond plus cross \
	    thin-plus thin-cross} {
	pack [radiobutton $page.b.1.d.$symbol \
	    -text $symbol_name -variable command(symbol) \
	    -value $symbol] -side top -anchor w
    }
    pack [frame $page.b.1.e] -side left -anchor n -fill y
    pack [label $page.b.1.e.l -text "Symbol Size"] -side top
    pack [scale $page.b.1.e.s -variable command(size) \
	    -from .1 -to 5 -resolution .1] -side top
    pack [button $page.b.2.b \
	      -text "Apply all changes" \
	      -command "ChangeColors $page.a.2"] -side right
    pack [button $page.b.2.c \
	      -text "Set Color Sequence" \
	      -command "ResetColors $page.a.2"] -side left
    
    # resize the color box
    update idletasks
    $colorcan config -scrollregion [grid bbox $colorfrm]
    $colorcan config -width [lindex  [grid bbox $colorfrm] 2]
}

proc PostPagePlot {page} {
    global command
    set command(line) ""
    set command(symbol) ""
    set command(color) "no change"
    set command(size) 1
    updateselectbox $page.a.2
}

proc ChangeColors {box} {
    global command
    foreach num [$box.1 curselection ] {
	set data [$box.1 get $num]
	global $data
	if {$command(line) != ""} {set ${data}(line) $command(line)}
	if {$command(color) != "no change"} {
	    set ${data}(color) $command(color)
	}
	if {$command(size) != ""} {set ${data}(symsize) $command(size)}
	if {$command(symbol) != ""} {set ${data}(symbol) $command(symbol)}
    }
    displaylist
}

proc ResetColors {box} {
    global graph command
    # get the selected color
    set graph(colorindex) [lsearch $graph(colorlist) $command(color)]
    if {$graph(colorindex) == -1} {set graph(colorindex) 0}
    # set colors to each buffer
    foreach num [$box.1 curselection] {
	set data [$box.1 get $num]
	global $data
	set ${data}(color) [lindex $graph(colorlist) $graph(colorindex)]
	incr graph(colorindex)
	if {$graph(colorindex) == $graph(ncolors)} {set graph(colorindex) 0}
    }
    displaylist
}
#--------------------------------------------------------------------
lappend menulist(display) Rescale
lappend menulist(pages) Rescale
set helplist(Rescale) {\
This page is used to modify the stored values in one or 
more buffers. Specify constants to multiply the x and y 
values and constants to add to the x and y values (addition 
is done after multiplication).

It is also possible to transform the x and y scales. The x-axis 
can be transformed between Q, d-space and 2-theta units for an 
arbitrary wavelength.  Intensities (y values) can be transformed 
to square roots [Sqrt(I)], base 10 logarithms [Log(I)]. The 
I/sig(I) option divides each  intensity by its estimated error, 
providing an estimate of the signal-to-noise. Conversion is done 
before values are multiplied or offset.

Changes are applied when the "Rescale" button is pressed.
If more than one scaling or transformation is applied, the 
operation is applied to the results from the previous operation.
If the  "Reset to original values" checkbox is selected, 
however, the x and y values are reset to the values that were read in,
before any scaling or transformation is done.
}

proc MakeRescale {page}  {
    global command graph
    pack [frame $page.a] -side left -anchor n -fill both
    pack [label $page.a.t1 -text {Select dataset(s) to}] -side top
    pack [label $page.a.t2 -text rescale] -side top
    pack [frame $page.a.2 ] -side top 
    makeselectbox $page.a.2

    pack [frame $page.b] -side left -anchor n -fill both
    pack [frame $page.b.1] -side top -fill both
    pack [frame $page.b.2 -bd 2 -relief groove] -side top -fill both

    set box $page.b.1.a
    pack [frame $box -bd 2 -relief groove] -side left -fill both
    pack [label $box.1 -text "X-axis units"] -side top
    pack [radiobutton $box.2 -text "no change\nor reset" -value {} \
	    -variable command(xmode)] -side top
    pack [radiobutton $box.3 -text d-space -value dspace \
	    -variable command(xmode)] -side top -anchor w
    pack [radiobutton $box.4 -text Q -value Q \
	    -variable command(xmode)] -side top -anchor w
    pack [radiobutton $box.5 -text "2Theta @" -value 2theta \
	    -variable command(xmode)] -side top -anchor w
    pack [frame $box.6] -side top -anchor e 
    pack [label $box.6.a -text A ] -side right
    pack [entry $box.6.b -textvariable command(newwave) -width 8]\
	    -side right

    set box $page.b.1.b
    pack [frame $box -bd 2 -relief groove] -side left -fill both
    pack [label $box.1 -text "Y-axis units"] -side top
    pack [radiobutton $box.2 -text "Linear\n or reset" -value {} \
	    -variable command(ymode)] -side top -anchor w
    pack [radiobutton $box.3 -text Sqrt(I) -value sqrt \
	    -variable command(ymode)] -side top -anchor w
    pack [radiobutton $box.4 -text Log(I) -value log \
	    -variable command(ymode)] -side top -anchor w
    pack [radiobutton $box.5 -text I/sig(I) -value s-n \
	    -variable command(ymode)] -side top -anchor w
    pack [radiobutton $box.6 -text I2/sig(I)2 -value unscale \
	    -variable command(ymode)] -side top -anchor w

    set box $page.b.1.c
    pack [frame $box -bd 2 -relief groove] -side left \
	    -anchor n -fill both
    pack [label $box.t -text Scaling] -side top
    foreach var {xoffset xmult yoffset ymult xstagger ystagger} \
	    label {"X offset" "X multiplier" \
	    "Y offset" "Y multiplier" "X stagger %" "Y stagger %"} {
	pack [frame $box.$var -bd 2 -relief groove] \
		-side top -fill x
	pack [label $box.$var.l -text $label] -side left
	pack [entry $box.$var.e -textvariable command($var) \
		-width 12] -side right
    }

    set box $page.b.2
    pack [checkbutton $box.3 \
	    -text "Normalize Y Values" \
	    -variable command(normalize)] -side top -anchor c
    pack [checkbutton $box.4 \
	    -text "Reset to original values" \
	    -variable command(rescale)] -side top -anchor c
    pack [button $box.5 \
	    -text "Rescale" -command "redefinescale $page.a.2"] \
	    -side top
}

proc PostPageRescale {page} {
    updateselectbox $page.a.2
} 

proc redefinescale {box} {
    global graph command
    # get x and y range
    set xl [$graph(blt) xaxis limits]
    set xr [expr {[lindex $xl 1] - [lindex $xl 0]}]
    set i 0
    if {$command(rescale) || $command(normalize)} {
		set ymin ""
		set ymax ""
		foreach num [$box.1 curselection ] {
			catch {
				set data [$box.1 get $num]
				global $data
				if $command(rescale) {resetdata $data}
				if $command(normalize) {
					set yvec [set ${data}(yvector)]
					global $yvec
					set ymin [set ${yvec}(min)]
					set ymax [set ${yvec}(max)]
					set range [expr {$ymax - $ymin}]
					set ymult [expr { 1000 / $range }]
					set yoff [expr {$ymult * -1 * $ymin}]
					scaledata $data 0 1 $yoff $ymult
					
				} else {
					set yvec [set ${data}(yvector)]
					global $yvec
					set yv [set ${yvec}(min)]
					if {$ymin == ""} {set ymin $yv}
					if {$yv < $ymin} {set ymin $yv}
					set yv [set ${yvec}(max)]
					if {$ymax == ""} {set ymax $yv}
					if {$yv > $ymax} {set ymax $yv}
					set yr [expr {$ymax - $ymin}]
				}
			}
		}
		# if data is being normalized with an additional multiplier, the yrange must
		# be adjusted to reflect that additional multiplier
		if $command(normalize) {set yr [expr { $command(ymult) * 1000 }]}
		
    } else {
	set yl [$graph(blt) yaxis limits]
	set yr [expr {[lindex $yl 1] - [lindex $yl 0]}]
    }
    foreach num [$box.1 curselection ] {
	set data [$box.1 get $num]
	catch {
	    #if $command(rescale) {resetdata $data}
	    ConvertX $data $command(xmode) $command(newwave)
	    ConvertY $data $command(ymode) 
	    set xoff [expr {$command(xoffset) + \
				($command(xstagger) * $xr * $i)/ 100.}]
	    set yoff [expr {$command(yoffset) + \
				($command(ystagger) * $yr * $i)/ 100.}]
	    scaledata $data $xoff $command(xmult) $yoff $command(ymult)
	}
	incr i
    }
    # force an update
    displaylist
}
#--------------------------------------------------------------------
#Combine
lappend menulist(display) Combine
lappend menulist(pages) Combine
set helplist(Combine) {\
The Combine page is used to create a new buffer set by 
adding or subtracting up to five other buffers. The 
buffers must have the same number of data points. No 
attempt is made to offset buffers, so they must all 
start (and end) with the same x value.
}

proc MakeCombine {page}  {
    global command
    pack [frame $page.a -bd 4 -relief groove] -side left -fill y -anchor n
    pack [frame $page.a.top] -side top -anchor center
    pack [frame $page.a.m] -side top -anchor center
    pack [frame $page.a.bot] -side top -anchor center
    grid [label $page.a.top.l -text "Name: "] -row 0 -column 0
    grid [entry $page.a.top.e -textvariable command(combinename) -width 40] -row 0 -column 1
    pack [button $page.a.bot.m -text "More Files" \
	      -command "incr command(combinemax); MakeCombineFileBox"] \
	      -side left -anchor center
    pack [button $page.a.bot.h -text "Sum Files" \
	    -command "CombineFiles"] -side left -anchor center
    # define a scrolled canvas for color entries
    set command(CmbCanvas) $page.a.m.b
    set command(Cmbscroll) $page.a.m.y 
    set command(CmbFrame) $page.a.m.b.f
    grid [canvas $command(CmbCanvas) \
	    -scrollregion {0 0 5000 500} -width 100 -height 200 \
	    -yscrollcommand "$command(Cmbscroll) set" -bg lightgrey] \
	    -sticky  news -row 1 -column 1
    grid [scrollbar $command(Cmbscroll) -orient vertical \
	    -command "$command(CmbCanvas) yview"] \
	    -sticky ns -row 1 -column 2
    set command(CmbNoUpdates) 0
}

proc SetCombineName {args} {
    global command
    if {$command(CmbNoUpdates)} return
    set command(CmbNoUpdates) 1
    set max $command(combinemax)
    set name {}
    set i 0
    for {set num 1} {$num <= $max} {incr num} {
	set data $command(combdataset$num)
	if {$data != "(none)" && $command(combmult$num) != 0} {
	    incr i
	    append name "$data*$command(combmult$num)+"
	}
    }
    set command(CmbNoUpdates) 0
    ResizeCombineFileBox
    if {$name == ""} return
    if {[string length $name] > 39} {set name "combine$i"}
    set command(combinename) [string trim $name "+"]
}

proc MakeCombineFileBox {} {
    global command graph
    set command(CmbNoUpdates) 1
    set max $command(combinemax)
    set frm $command(CmbFrame)
    set datalist {(none)}
    foreach data $graph(datalist) {
	global $data
	if {[set ${data}(type)] != "data"} {lappend datalist $data}
    }
    catch {destroy $command(CmbFrame)}
    frame $command(CmbFrame)
    $command(CmbCanvas) create window 0 0 -anchor nw  -window $command(CmbFrame)
    for {set num 1} {$num <= $max} {incr num} {
	grid [frame $frm.$num -bd 4 -relief groove] -row $num
	pack [label $frm.$num.1 -text "Select dataset $num"] -side left
	eval tk_optionMenu $frm.$num.2 command(combdataset$num) $datalist
	if {[trace vinfo  command(combdataset$num)] == ""} {
	    trace variable command(combdataset$num) w SetCombineName
	}
	pack $frm.$num.2 -side left
	pack [label $frm.$num.3 -text times] -side left
	if {[catch {set command(combmult$num)}]} {set command(combmult$num) 1.0}
	pack [entry $frm.$num.4 -textvariable command(combmult$num) -width 5] \
		-side left
	if {[trace vinfo  command(combmult$num)] == ""} {
	    trace variable command(combmult$num) w SetCombineName
	}
    }
    set command(CmbNoUpdates) 0
    ResizeCombineFileBox
}

proc ResizeCombineFileBox {} {
    if {$::command(CmbNoUpdates)} return
    # resize the file box
    global command
    update idletasks
    $command(CmbCanvas) config -scrollregion [grid bbox $command(CmbFrame)]
    $command(CmbCanvas) config -width [lindex  [grid bbox $command(CmbFrame)] 2]
}

proc PostPageCombine {page} {
    global command
    set command(combinemax) 2
    set max $command(combinemax)
    MakeCombineFileBox
    set command(CmbNoUpdates) 1
    # reset the multiplier values
    for {set num 1} {$num <= $max} {incr num} {
	set command(combmult$num) 1.0
	set command(combdataset$num) "(none)"
    }
    set command(combinename) ""
    set command(CmbNoUpdates) 0
}

proc CombineFiles {} {
    global command
    set OK 0
    set max $command(combinemax)
    for {set num 1} {$num <= $max} {incr num} {
	if [catch {
	    set data $command(combdataset$num)
	    set mult $command(combmult$num)
	}] continue
	if {$data != "(none)" && $mult != 0} {
	    global $data
	    incr OK 1
	    if {$OK == 1} {
		set minlength [[set ${data}(xvector)] length]
	    } else {
		set l [[set ${data}(xvector)] length]
		if {$l < $minlength} {set minlength $l}
	    }
	    if {$minlength < 2} {
		MyMessageBox -parent . -title "No points" \
		    -icon error -type Ignore -default ignore -message \
		    "There are only $minlength points in file $command(combdataset$num)"
		return
	    }
	}
    }
    if {$OK == 0} {
	MyMessageBox -parent . -title "No files" \
	    -icon error -type Ignore -default ignore \
	    -message "There no valid files to process"
	return
    }
    set lastind [expr $minlength - 1]
    set data $command(combinename)
    set data [initdata $data]
    global ${data}
    set ${data}(ylabel) "arbitrary"
    set ${data}(xlabel) "2theta"
    set ${data}(xunits) "2theta"
    vector tmpx
    vector tmpy
    vector tmpesd2
    vector sumx
    vector sumy
    vector sumesd2
    set OK 0
    for {set num 1} {$num <= $max} {incr num} {
	if [catch {
	    set datn $command(combdataset$num)
	    set mult $command(combmult$num)
	}] continue
	if {$datn != "(none)" && $mult != 0} {
	    set datn $command(combdataset$num)
	    global $datn 
	    incr OK 1
	    tmpx set [[set ${datn}(xvector)] range 0 $lastind]
	    tmpy set [[set ${datn}(yvector)] range 0 $lastind]
	    tmpy set [tmpy * $command(combmult$num)]
	    tmpesd2 set [[set ${datn}(esdvec)] range 0 $lastind]
	    tmpesd2 set [tmpesd2 * $command(combmult$num)]
	    tmpesd2 set [tmpesd2 * tmpesd2]
	    if {$OK == 1} {
		sumx set tmpx
		sumy set tmpy
		sumesd2 set tmpesd2
	    } else {
		sumx set [sumx + tmpx]
		sumy set [sumy + tmpy]
		sumesd2 set [sumesd2 + tmpesd2]
	    }
	}
    }
    if {$OK} {
	set ${data}(x) [sumx / $OK]
	set ${data}(y) [sumy range 0 end]
	set ${data}(esd) {}
	foreach num [sumesd2 range 0 end] {
	lappend ${data}(esd) [expr sqrt($num)]
	}
    }
    resetdata $data
    MyMessageBox -parent . -title "New buffer" \
	    -icon info -type OK -default ok \
	    -message "Buffer $data has been created"
}

#--------------------------------------------------------------------
lappend menulist(compute) HKLGEN
lappend menulist(pages) HKLGEN
set helplist(HKLGEN) {\
This page generates a list of the allowed reflections for a given 
set of unit cell constants and extinction conditions. It is possible
to specify symmetry by space group or by selecting individual 
translational symmetry operations. 

Space groups should be entered as specified in GSAS:
Enter the centering condition or cell type (P, R, A, B, C, I or F) 
and then the symmetry along each appropriate axis, with spaces between
each axis. Append a final "R" for rhombohedral settings. Some examples 
are: "P 1 21/c 1" or "P 21/c" and "R 3 m" for or "R 3 m R" for the 
hexagonal and rhombohedral settings  of R3m, respectively. If an 
incorrect space group is entered,  no error message is generated, 
but no extinctions are generated. Either direct or reciprocal cell 
parameters may be used.  

The reflections are generated and placed in a separate window when 
the "Compute" button is pressed. This table can be saved to a disk 
file by entering a file name and pressing the "Save as" button. 
}
proc MakeHKLGEN {page}  {
    global graph cellparm extinctlbl
    pack [frame $page.r] -side left -fill both -anchor n
    makecellbox $page.r "Unit Cell\nParameters"
    set cellparm(top) $page.r
    pack [button $page.r.calc -text Compute -command showcell] -side bottom
    makesymbox $page cellparm(laue) setcellbox
    setcellsym $cellparm(laue) setcellbox
    set top $page.m
    pack [frame $top ] -side left -fill both -anchor n
    pack [frame $top.e -relief groove -bd 4] -side top -fill both -anchor n
    set cellparm(extinctmenuH) $top.e 
    label $top.e.l -text "Extinctions to remove"
    grid $top.e.l -in $top.e -row 1 -column 1 -columnspan 2
    label $top.e.el -text "Individual\nconditions"
    grid $top.e.el -in $top.e -row 2 -column 1
    button $top.e.ext -text "0 Set" -command "extinctmenu .extmen"
    grid $top.e.ext -in $top.e -row 2 -column 2
    label $top.e.spgl -text "Space\nGroup:"
    grid  $top.e.spgl -in $top.e -row 3 -column 1
    entry $top.e.spge -textvariable cellparm(spacegroup) -width 8
    grid  $top.e.spge -in $top.e -row 3 -column 2
    grid rowconfigure $top.e 2 -weight 0 -minsize 42
    
    pack [frame $top.lam -relief groove -bd 4] -side top -fill both -anchor center
    pack [label $top.lam.l -anchor e -text "Wavelength:"] \
	    -padx 1m -pady 1m -side left -anchor e
    pack [entry $top.lam.e \
	    -width 8 -relief sunken -bd 2 -textvariable cellparm(wavelength)] \
	    -padx 1m -pady 1m  -side left
    pack [frame $top.ttmax -relief groove -bd 4] -side top -fill both -anchor center
    pack [label $top.ttmax.l -width 10 -anchor e -text "2theta max:"] \
	    -padx 1m -pady 1m -side left
    pack [entry $top.ttmax.e \
	    -width 8 -relief sunken -bd 2 -textvariable cellparm(twothetamax)] \
	    -padx 1m -pady 1m  -side left
}
proc PostPageHKLGEN {page} {
    resetextinctions  
} 
#--------------------------------------------------------------------
lappend menulist(compute) EditCell
lappend menulist(pages) EditCell
set helplist(EditCell) {\
This page generates a stick-figure plot of the allowed reflections 
for a given set of unit cell constants. Unit cells may be input 
with either direct or reciprocal cell parameters. Reflections can 
be omitted or highlighted if extinct due to symmetry. 

Space groups should be entered as specified in GSAS:
Enter the centering condition or cell type (P, R, A, B, C, I or F) 
and then the symmetry along each appropriate axis, with spaces between
each axis. Append a final "R" for rhombohedral settings. Some examples 
are: "P 1 21/c 1" or "P 21/c" and "R 3 m" for or "R 3 m R" for the 
hexagonal and rhombohedral settings  of R3m, respectively. If an 
incorrect space group is entered,  no error message is generated, 
but no extinctions are generated. 

Reflections are generated and displayed when "Compute" is pressed. 
Enter symmetry before pressing "Compute" to omit the extinct 
reflections or afterwards to highlight the reflections. Moving the
sliders or typing a new value causes the reflection positions to be 
recomputed and displayed. 

Individual reflections can be labeled by holding Shift and clicking 
with the left mouse button.  All reflections can be labeled by holding
Shift and double-clicking with the left mouse button and cleared with 
by holding Shift clicking with the right mouse button. Reflection 
labels typically disappear after a period of time (see Options).
Label sizes can similarly be adjusted.
}

proc MakeEditCell {page}  {
    global cellparm
    set cellparm(page) $page
    pack [frame $page.a] -side left -fill y -anchor n
    pack [frame $page.a.3 -bd 4 -relief groove] -side top
    pack [label $page.a.3.1 -text {Select peaklist to edit}] -side top
    pack [frame $page.a.3.peaks] -side top
    pack [label $page.a.1 -text {Select dataset(s) to plot}] -side top
    pack [frame $page.a.2 ] -side top
    pack [listbox $page.a.2.1 -yscrollcommand "$page.a.2.scroll set" \
	    -height 8 -width 15 -bd 2 -relief raised ] -side left
    set cellparm(editdatalist) $page.a.2.1
    $page.a.2.1 config -selectmode extended -exportselection 0
    pack [scrollbar $page.a.2.scroll -command "$page.a.2.1 yview"] \
	    -side left -fill y
    #bind $page.a.2.1 <Double-Button-1> "getfilelist $page.a.2; SetEditData 1 2 3"
    bind $page.a.2.1 <Double-Button-1> "SetEditData 1 2 3"
    pack [button $page.a.h -text "PostScript out" -command makepostscriptout] \
	    -side bottom
    pack [button $page.a.p -text "Recompute/\nUpdate Plot" \
	    -command "SetEditData 1 2 3"] -side bottom
#	    -command "getfilelist $page.a.2; SetEditData 1 2 3"] -side bottom
    pack [frame $page.b] -side left -fill y -anchor n
    pack [frame $page.b.cell] -side top -fill y
    makesymbox $page.b.cell cellparm(celllaue) SetEditParms

    pack [frame $page.c] -side left -fill y -anchor n
    set top $page.c.ext
    pack [frame $top ] -side top -fill both -anchor n
    set base [frame $page.c.1]
    set cellparm(celleditbase) $base
    pack $base -side top
    set i 0
    foreach var {a b c} {
	incr i
	label $base.l$var -text $var
	grid $base.l$var -in $base -row $i -column 1
	entry $base.e$var -width 7
	grid $base.e$var -in $base -row $i -column 2
	scale $base.s$var -showvalue 0 -orient h \
		-resolution $cellparm(alphabeta_tol)
	grid $base.s$var -in $base -row $i -column 3
    }
    foreach var {alpha beta gamma} lbl {a b g} {
	incr i
	label $base.l$var -text $lbl -font symbol
	grid $base.l$var -in $base -row $i -column 1
	entry $base.e$var -width 7
	grid $base.e$var -in $base -row $i -column 2
	scale $base.s$var -showvalue 0 -orient h \
		-resolution $cellparm(alphabeta_tol)
	grid $base.s$var -in $base -row $i -column 3
    }
    incr i
    label $base.lzero -text Zero
    grid $base.lzero -in $base -row $i -column 1
    entry $base.ezero -width 7 -textvariable cellparm(zero)
    grid $base.ezero -in $base -row $i -column 2
    scale $base.szero -showvalue 0 -orient h -resolution 0.01 \
	    -command "set cellparm(zero) " -from -0.5 -to 0.5
    grid $base.szero -in $base -row $i -column 3

    pack [frame $top.e -relief groove -bd 4] -side top -fill both -anchor n
    set cellparm(extinctmenuE) $top.e 
    label $top.e.l -text "Extinctions"
    grid $top.e.l -in $top.e -row 1 -column 1
    label $top.e.l1 -text ""
    grid $top.e.l1 -in $top.e -row 1 -column 2
    label $top.e.l2 -text "" -fg #888
    grid $top.e.l2 -in $top.e -row 1 -column 3
    label $top.e.el -text "Individual\nconditions"
    grid $top.e.el -in $top.e -row 2 -column 1
    button $top.e.ext -text "0 Set" -command "extinctmenu .extmen"
    grid $top.e.ext -in $top.e -row 2 -column 2
    label $top.e.extl -text "" -fg #888
    grid $top.e.extl -in $top.e -row 2 -column 3
    label $top.e.spgl -text "Space\nGroup:"
    grid  $top.e.spgl -in $top.e -row 3 -column 1
    entry $top.e.spge -textvariable cellparm(extspg) -width 8
    grid  $top.e.spge -in $top.e -row 3 -column 2
    label $top.e.spg3 -text "" -fg #888
    grid  $top.e.spg3 -in $top.e -row 3 -column 3
    grid rowconfigure $top.e 2 -weight 0 -minsize 42

    pack [frame $page.d] -side left -fill both -anchor center
    pack [frame $page.d.a -bd 4 -relief groove] -side top -fill x -anchor n
    pack [label $page.d.a.1 -justify center -text \
	    "Minimum &\nMaximum Y\nvalues for\ngenerated\npeaks" \
	    ] -side top
    pack [entry $page.d.a.min -width 8 -textvariable cellparm(ymin)] \
	    -side top -anchor center
    pack [entry $page.d.a.max -width 8 -textvariable cellparm(ymax)] \
	    -side top -anchor center
    pack [frame $page.d.w -bd 4 -relief groove] -side top -fill x -anchor n
    pack [radiobutton $page.d.w.u0 -text 2Theta -value 0 \
	     -variable graph(units)] -side top -anchor w
    pack [frame $page.d.w.tt -bd 4] -side top 
    pack [label $page.d.w.tt.1 -text "2T max"] -side left
    pack [entry $page.d.w.tt.2 -width 4 \
	    -textvariable cellparm(twothetamax)] -side left
    pack [label $page.d.w.l -text Wavelength] -side top
    pack [entry $page.d.w.e  -width 8] -side top
    pack [radiobutton $page.d.w.u1 -text d-space -value 1 \
	     -variable graph(units)] -side top -anchor w
    pack [radiobutton $page.d.w.u2 -text Q -value 2 \
	     -variable graph(units)] -side top -anchor w
    # disable SetEditData
    set cellparm(skipSetEditData) 1
    trace variable cellparm(editsetting) w SetEditData
}

proc PostPageEditCell {page} {
    global graph cellparm
    catch {destroy $page.a.3.peaks.1}
    set peakslist {}
    foreach data $graph(datalist) {
	global $data
	if {[set ${data}(type)] == "peaks"} {
	    global $data
	    # skip unindexed peak lists
	    if {[llength [set ${data}(h)]] == 0} continue
	    lappend peakslist $data
	}
    }
    lappend peakslist "(new)"
    eval tk_optionMenu $page.a.3.peaks.1 cellparm(editsetting) $peakslist
    pack $page.a.3.peaks.1
    $page.a.3.1 config -text {Select peaklist to edit}
    set base $cellparm(celleditbase)
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg #888
	$base.e$var config -fg #888 -state disabled
	$base.s$var config -fg #888 -state disabled -command {}
    }
    $page.d.w.l config -fg #888
    $page.d.w.e  config  -fg #888 -state disabled

    updateselectbox $page.a.2
    set cellparm(editsetting) {}
    set cellparm(celleditvar) {}
    setsymbox $page.b.cell {}
    # enable seteditdata
    set cellparm(skipSetEditData) 0
    if {$peakslist == "(new)"} {set cellparm(editsetting) "(new)"}
}

# Called to select peaklist and set it up for plotting
proc SetEditData {a1 a2 a3} {
    global cellparm graph
    set page $cellparm(page)
    if {$cellparm(editsetting) == ""} return
    if $cellparm(skipSetEditData) return
    #------------------------------------------------------------
    # allow a new dgen to be created
    if {$cellparm(editsetting) == "(new)"} {
	editnew
	return
    }
    #------------------------------------------------------------
    set cellparm(extspg) (none)
    set cellparm(celleditvar) $cellparm(editsetting)
    editold
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase) 
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg black
	$base.e$var config -textvariable ${data}($var) \
		-fg black -state normal
	set value [set ${data}($var)]
	catch {
	    $base.s$var config -fg black -state normal \
		    -from [expr $value * (1-$cellparm(sens)/100.)]  \
		    -to [expr $value * (1+$cellparm(sens)/100.)]
	    $base.s$var set $value
	    $base.s$var config -command "set ${data}($var) "
	}
	# -variable ${data}($var)
    }
    $page.d.w.l config -fg black
    $page.d.w.e config -textvariable ${data}(wavelength) -fg black \
	    -state normal
	if {$graph(units) == 0} {set ${data}(xunits) "2theta"}
	if {$graph(units) == 1} {set ${data}(xunits) Q}
	if {$graph(units) == 2} {set ${data}(xunits) dspace}	
    set laue 8
    catch {set laue [set ${data}(laue)]}
    set cellparm(celllaue) $laue

    # recompute the reflections using the current settings
    if {[validatecell] == ""} computehkllist
    # trigger a recalc of the extinctions
    set cellparm(oldextspg) {}
    $cellparm(extinctmenuE).spg3 config -text "[set ${data}(spg)]"
    set count 0
    foreach num [set ${data}(extcodes)] {
	if {$num > 0} {incr count}
    }
    $cellparm(extinctmenuE).extl config -text "$count Set"
    # get the plot list and add the peak file
    getfilelist $page.a.2
    # if the data array appears in the plot list remove it
    set num [lsearch -exact $graph(plotlist) $data]
    if {$num != -1} {
	set graph(plotlist) [lreplace $graph(plotlist) $num $num]
	# lappend graph(plotlist) $data
    }
    eval $graph(blt) element delete [$graph(blt) element names]
    eval $graph(blt) marker delete  [$graph(blt) marker names]
    foreach pldat $graph(plotlist) {
	global $pldat 
	global [set ${pldat}(xvector)] [set ${pldat}(yvector)]
	# expression to select each color from the list 
	$graph(blt) element create $pldat \
		-linewidth [set ${pldat}(line)] \
		-color [set ${pldat}(color)] -symbol [set ${pldat}(symbol)] \
		-xdata [set ${pldat}(xvector)] -ydata [set ${pldat}(yvector)] \
		-pixels [expr 0.125*[set ${pldat}(symsize)]]i
#		-activecolor black 
	[set ${pldat}(xvector)] notify now
	[set ${pldat}(yvector)] notify now
    }
    # put the computed lines on top of the extinct lines
    $graph(blt) element create $data \
	-color $graph(editcellcolor) -linewidth [set ${data}(line)] \
	-symbol none \
	-xdata [set ${data}(xvector)] -ydata [set ${data}(yvector)]
#	    -activecolor red
    $graph(blt) element create extinct \
	-color $graph(extinctcolor) -linewidth 2 \
	-xdata xextinct -ydata yextinct -symbol none
#	    -activecolor red
    xextinct notify now
    yextinct notify now
    [set ${data}(xvector)] notify now
    [set ${data}(yvector)] notify now
    setsymbox $page.b.cell $cellparm(celllaue)
    setcellsym $cellparm(celllaue) SetEditParms
    RecalcLoop
    # get binding for graph
    set bindtag $graph(blt)
    catch {
	if {[bind bltZoomGraph] != ""} {set bindtag bltZoomGraph}
    }
    bind $bindtag <Shift-Button-1> "lblhkl %W %x"
    bind $bindtag <Shift-Double-Button-1> "lblallhkl %W"
    bind $bindtag <Shift-Button-3> "delallhkllbl %W"
    bind .plot <Key-h> "lblhkl $graph(blt) %x"
    bind .plot <Key-H> "lblhkl $graph(blt) %x"
    bind .plot <Key-a> "lblallhkl $graph(blt)"
    bind .plot <Key-A> "lblallhkl $graph(blt)"
    bind .plot <Key-d> "delallhkllbl $graph(blt)"
    bind .plot <Key-D> "delallhkllbl $graph(blt)"
}

# this is called when the laue symmetry is changed or by setcellsym
proc SetEditParms {fix lock} {
    global cellparm
    if {$cellparm(editsetting) == ""} return
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase)
    
    set i -1
    foreach var {a b c alpha beta gamma} {
	incr i
	# variables to fix
	if {[lindex $fix $i]-0 > 0} {
	    $base.l$var config -fg #808080
	    $base.e$var config -state disabled -fg #808080 \
		    -textvariable ${data}($var)
	    $base.s$var config -state disabled -fg #808080 \
		    -command {}
	    set ${data}($var)  [lindex $fix $i]
	} else {
	    $base.l$var config -fg black
	    $base.e$var config -state normal -fg black \
		    -textvariable ${data}($var)
	    set value [set ${data}($var)]
	    catch {
		$base.s$var config -state disabled -fg black \
			-from [expr $value * (1-$cellparm(sens)/100.)]  \
			-to [expr $value * (1+$cellparm(sens)/100.)]
		$base.s$var set $value
		$base.s$var config -command "set ${data}($var)"
	    }
	    if {$cellparm(editsetting) != "(new)"} {
		$base.s$var config -state normal
	    }
	}
	# variables to lock to others
	if {"[lindex $lock $i]" != ""} {
	    $base.l$var config -fg #888 
	    $base.e$var config -state disabled -fg #888 \
		    -textvariable ${data}([lindex $lock $i])
	    $base.s$var config -state disabled -fg #888 -command {}
	}
    }
}

#--------------------------------------------------------------------
lappend menulist(compute) Fit
lappend menulist(pages) Fit
set helplist(Fit) {\
Perform a profile fit on a region of a dataset using a  pseudo-Voigt
and optionally Finger-Cox-Jephcoat low-angle asymmetry.

Steps:
  1) Select a dataset and peaklist (when needed)
  2) Define a fit region with the zoom and "Set Range to fit" 
       Shortcuts: use the S/Z/F/B/E keys (see shortcuts)
  3) Add peaks (when needed) using either "P" or with the mouse by 
       with a Control-double click at the peak
       or press a "Set" button and then use Control-click at the peak
  4) Select parameters to refine
        It is best to first refine the width and areas and later refine
        background, positions and eta
  5) Refine by pressing the "Run GPLSFT" button

After a refinement has been the refinement can be reversed once with 
the "Undo" button or the fit curve can be saved with "Store Fit".

Low angle asymmetry can be modeled using the Finger-Cox-Jephcoat 
correction. This requires the instrument diameter and the sample 
and detector heights, all in the same units.

Fit results can be placed in the same plot window that is used
to display the diffractogram plots or can be displayed in a separate 
window; likewise the fitted parameters can be superimposed on this plot 
or can be shown in a separate window; these choices are made with the 
"Plot fit results in separate window" and 
"Report fit results in plot window" checkboxes 
in the Options command.
}

proc MakeFit {page} {
    global graph gpls
    pack [frame $page.a] -side left -fill y -anchor n
    pack [frame $page.a.d -bd 4 -relief groove] -side top
    pack [label $page.a.d.1 -text {Select a dataset to fit}] -side top
    set base $page.a
    pack [frame $base.fr] -side top
    set gpls(wave1box) $base.fr.enw1
    set gpls(wave2box) $base.fr.enw2
    grid [label $base.fr.l1 -text "wavelength 1"] -column 1 -row 1 -sticky e
    grid [entry $gpls(wave1box) -width 8] -column 2 -row 1 -sticky w
    grid [label $base.fr.l2 -text "wavelength 2"] -column 1 -row 2 -sticky e
    grid [entry $gpls(wave2box) -width 8] -column 2 -row 2 -sticky w
    grid [label $base.fr.l3 -text "ratio of 2:1"] -column 1 -row 3 -sticky e
    grid [entry $base.fr.enr -width 5 -textvariable gpls(ratio21)] \
	-column 2 -row 3 -sticky w
    set gpls(ratio21) 0.5
    pack [frame $page.a.p -bd 4 -relief groove] -side top -fill both
    pack [label $page.a.p.1 -text {Select a peak list}] -side top -fill both
    pack [frame $page.a.3 -bd 4 -relief groove] -side top -fill both
    pack [button $page.a.3.a -text {Set range to fit} \
	    -command SetFitRange \
	    ] -side top -anchor center
    pack [label  $page.a.3.b -textvariable gpls(rangemsg)] -side top -anchor center
    pack [frame $page.a.4 ] -side top
    pack [label $page.a.4.a -text {Max cycles}] -side left
    pack [entry $page.a.4.b -justify center \
	    -textvariable gpls(ncyc) -width 4] -side left
    pack [frame $page.a.5 ] -side top
    pack [label $page.a.5.a -text {Damping factor}] -side left
    pack [entry $page.a.5.b -justify center \
	    -textvariable gpls(damp) -width 4] -side left
    button $page.a.6 -command wrgpls -text {Run GPLSFT} -width 12
    pack [frame $page.a.8 -bg red -bd 4] -side top -fill x
    pack [label  $page.a.8.a -text "Unable to refine:" -fg white -bg red] -side top -anchor w
    pack [label  $page.a.8.b -textvariable graph(gplsstatus)]
    set graph(gplsstatus) {}
    pack [frame $page.b -bd 4 -relief groove] -side left -fill y -anchor n
    frame $page.b.plot
    pack [button $page.b.plot.a -text "PostScript out" \
	    -command makepostscriptout] -side left
    pack [button $page.b.plot.b -text "Store Fit" -command StoreFit] -side right	    
    button $page.b.undo -text "Undo last cycle" -command UndoFit
    set gpls(overallframe) $page.b 
    overalltable
    pack [frame $page.c -bd 4 -relief groove] -side left -fill both -anchor n
    set gpls(peakframe) $page.c
    #peaktable

    if {[trace vinfo  gpls(dataset)] == ""} {
	trace variable gpls(dataset) w setupfitfile
    }
    if {[trace vinfo  gpls(peaklist)] == ""} {
	trace variable gpls(peaklist) w setupfitfile
    }
    if {[trace vinfo  gpls(asymflg)] == ""} {
	trace variable gpls(asymflg) w disableasym
    }
    set gpls(page) $page
    if {[trace vinfo  gpls] == ""} {
	trace variable gpls w enablegpls
    }
}

proc PostPageFit {page}  {
    global graph gpls
    set dataonlylist {}
    set peakonlylist {}
    $gpls(wave1box) configure -textvariable {}
    $gpls(wave2box) configure -textvariable {}
    foreach data $graph(datalist) {
	global $data
	if {[set ${data}(type)] != "peaks"} {lappend dataonlylist $data}
	if {[set ${data}(type)] == "peaks"} {lappend peakonlylist $data}
    }
    catch {pack forget $page.a.7}

    catch {destroy $page.a.d.a}
    catch {destroy $page.a.p.a}
    if {$dataonlylist != ""} {
	eval tk_optionMenu $page.a.d.a gpls(dataset) $dataonlylist
	pack $page.a.d.a
	peaktable
    } else {
	pack [label $page.a.d.a -text {no data to fit} -fg red]
	return
    }
    lappend peakonlylist {(create new)}
    eval tk_optionMenu $page.a.p.a gpls(peaklist) $peakonlylist
    pack $page.a.p.a

    set gpls(asymflg) $gpls(asymflg)
    foreach i {1 2 3 4} {set gpls(refine_$i) 0}
    # get binding for graph
    set bindtag $graph(blt)
    catch {
	if {[bind bltZoomGraph] != ""} {set bindtag bltZoomGraph}
    }
    bind .plot <Key-P> "addpeak $graph(blt) %x %y"
    bind .plot <Key-p> "addpeak $graph(blt) %x %y"
    bind $bindtag <Control-Double-Button-1> "addpeak %W %x %y"
    bind $bindtag <Control-Button-1> " "
    # if there is only 1 fit to fit choose it
    if {[llength $dataonlylist] == 1} {
	set gpls(dataset) $dataonlylist
    } else {
	$graph(blt) config -title "Select a dataset to fit"
	set gpls(dataset) "(none selected)"
	set graph(plotlist) {}
	displaylist
    }
    set gpls(rangemsg) {No range set}
}

#--------------------------------------------------------------------
# Smoothing and Peak finding
# GUI section by A.McGhie don't blame B.Toby
#      reorganized 1/21/00 -- do blame B.Toby
# Fortran program by R.Harlow
# mcghieaj@esvax.email.dupont.com
# 01/20/2000
#

# initial conditions
# gaussian(ndp)  = number of data point for gausian to fit
# gpls(fwhm) = full width half max (defaults in fit)
# gaussian(sens) = sensitivity
set gaussian(esd)  2.0000
set gaussian(sens) 3.0000
set gaussian(dp) 1
if {$tcl_platform(platform) == "windows"} {
    set gaussian(program) [file join $command(exedir) gaussuper.exe]
} else {
    set gaussian(program) [file join $command(exedir) gaussuper]
}

lappend menulist(compute) "PeaksSmooth PeakSearch"
lappend menulist(compute) "PeaksSmooth Smooth"
lappend menulist(pages) PeaksSmooth
set helplist(PeaksSmooth) {\
Used to smooth diffraction patterns with a Gaussian. 
The smoothed results are used to compute a first and 
second derivative of the data. Peak positions, as well 
as height and area estimates are made using the second 
derivative. From the peak positions, data away from the 
peak positions are used to estimate the background. 

FWHM sets the width of the Gaussian convolution function. 
It appears best to set this to the FWHM of the actual 
peaks.

The "Sensitivity" and "ESD" settings are used only for 
peak searching. Both are used to reject peaks below 
signal-to-noise criteria -- somehow.
}

proc MakePeaksSmooth {page} {
    global graph gaussian
    pack [frame $page.a] -side left -fill y -anchor n
    pack [label $page.a.t1 -text {Select dataset(s) }] -side top
    pack [frame $page.a.2 ] -side top 
    makeselectbox $page.a.2
    pack [frame $page.a1 -borderwidth 3 -relief groove]  -side left -fill y -anchor n
#    pack [frame $page.a1.b] -side left -fill y -anchor n

    grid [label $page.a1.l2 -text {FWHM } ] -column 1 -row 1 -columnspan 2 -sticky e
    grid [entry $page.a1.e2 -justify center \
	    -textvariable gpls(fwhm) -width 8] -column 3 -row 1
    grid [label $page.a1.j2  -justify left \
	    -text "defines the gaussian width for convolution.\nShould be about the same as the peak width." \
	    ] -column 4 -row 1 -sticky w

    grid [label $page.a1.l3 -text {Sensitivity } ] -column 1 -row 2 -columnspan 2 -sticky e
    grid [entry $page.a1.e3 -justify center \
	    -textvariable gaussian(sens) -width 8] -column 3 -row 2
    grid [label $page.a1.j3  -justify left \
	    -text "relative error threshold for peak location." \
	    ] -column 4 -row 2 -sticky w

    grid [label $page.a1.l1 -text {ESD } ] -column 1 -row 3 -columnspan 2 -sticky e 
    grid [entry $page.a1.e1 -justify center \
	    -textvariable gaussian(esd) -width 8] -column 3 -row 3
    grid [label $page.a1.j1  -justify left \
	    -text "reject peaks below this signal-to-noise ratio." \
	    ] -column 4 -row 3 -sticky w

    # expand the empty row, pushing the above to the top
    grid rowconfigure $page.a1 4 -weight 2
    grid [label $page.a1.ll -text Save:] -row 5 -column 1 -sticky w
    grid [checkbutton $page.a1.do -text "Smoothed curve" \
	    -variable gaussian(d0) ] -row 6 -column 2 -columnspan 2 -sticky w
    grid [checkbutton $page.a1.db -text "Background" \
	    -command "disablePeakVariables $page" \
	    -variable gaussian(db) ] -row 7 -column 2 -columnspan 2 -sticky w
    grid [checkbutton $page.a1.d1 -text "1st Derivative" \
	    -variable gaussian(d1) ] -row 8 -column 2 -columnspan 2 -sticky w
    grid [checkbutton $page.a1.d2 -text "2nd Derivative" \
	    -variable gaussian(d2) ] -row 9 -column 2 -columnspan 2 -sticky w
    grid [checkbutton $page.a1.dp -text "Peak positions" \
	    -command "disablePeakVariables $page" \
	    -variable gaussian(dp) ] -row 10 -column 2 -columnspan 2 -sticky w
    # stick in a bit more space
    grid rowconfigure $page.a1 11 -weight 1
    grid [button $page.a1.b1 -text Compute \
		-command "GaussianMyFile $page.a.2" ] -row 12 -column 1 -columnspan 3
    pack [button $page.a.4    -text "  Update Plot  " \
	    	-command "getfilelist $page.a.2; displaylist"] -side bottom
}

proc PostPagePeaksSmooth {page}  {
    global graph gpls
    updateselectbox $page.a.2
    disablePeakVariables $page
}

proc disablePeakVariables {page} {
    global gaussian
    if {$gaussian(dp) || $gaussian(db) } {
	foreach i {l1 j1 l3 j3} {
	    $page.a1.$i config -fg black
	}
	$page.a1.e3 config -state normal -fg black
	$page.a1.e1 config -state normal -fg black
    } else {
	foreach i {l1 j1 l3 j3} {
	    $page.a1.$i config -fg gray
	}
	$page.a1.e3 config -state disabled -fg gray
	$page.a1.e1 config -state disabled -fg gray
    }
}

proc GaussianMyFile {box} {
    global graph gaussian gpls command
    set opt 0
    foreach item {d0 db d1 d2 dp} {
	set opt [expr $opt || $gaussian($item)]
    }
    if !{$opt} {
	tk_dialog .msg "nothing to do" \
		"Nothing to do: you have not selected any output options" \
		error 0 OK
	return
    }

    set type 3
    if {$gaussian(dp) || $gaussian(db)} {set type 4}

    set data "(none)"
    # now loop over selected data sets
    foreach num [$box.1 curselection] {
	set data  [$box.1 get $num]
	global ${data}
	if {[set ${data}(type)] == "xy"} {

	    set fp [open gauss.inp w]
	    set number_of_data_points [[set ${data}(xvector)] length]
	    set max_two_theta  [[set ${data}(xvector)] range end end]
	    set min_two_theta  [[set ${data}(xvector)] range 0 0]
	    set delta [expr ($max_two_theta - $min_two_theta) \
		    / ($number_of_data_points - 1)]
	    puts $fp [format "%1d %d %f %f %f %f %f %f " \
		    $type $number_of_data_points \
		    $min_two_theta $max_two_theta $delta \
		    $gpls(fwhm) $gaussian(sens) $gaussian(esd)]
	    puts $fp [[set ${data}(xvector)] range 0 end]
	    puts $fp [[set ${data}(yvector)] range 0 end]
	    puts $fp [[set ${data}(esdvec)] range 0 end]
	    close $fp

	    #    run Harlow's fortran program
	    exec $gaussian(program) < gauss.inp > gauss.out

	    set opt 0
	    # get the x-axis data when needed
	    foreach item {d0 db d1 d2} {
		if {$gaussian($item)} {
		    source gauss.x
		    break
		}
	    }
	    foreach label {"smoothed intensity values" background \
		    "1st derivative" "2nd derivative"} \
		    lbl {smo bkg 1st 2nd} \
		    item {d0 db d1 d2} {
		if {$gaussian($item)} {
		    # create a name
		    set newdata ${data}_[expr round( $gpls(fwhm) * 1000)]
		    if {$lbl == "bkg"} {
			append newdata _[expr round($gaussian(sens) * 10)]
			append newdata _[expr round($gaussian(esd) * 10)]
		    }
		    append newdata _$lbl
		    source gauss.$lbl
		    
		    set newdata [initdata $newdata]
		    global ${newdata}
		    set ${newdata}(xlabel) "2theta"
		    set ${newdata}(xunits) "2theta"
		    set ${newdata}(ylabel) $label
		    set ${newdata}(skip) 0
		    set ${newdata}(x) $newx
		    set ${newdata}(y) $newy
		    set ${newdata}(esd) $newesd
		    resetdata $newdata
		    # add the new file to the plot list
		    if {[lsearch $graph(plotlist) $newdata] == -1} {
			lappend graph(plotlist) $newdata
		    }

		}
	    }
	    # now process peaks
	    if {$gaussian(dp)} {
		source gauss.pks
		set peakdata ${data}
		append peakdata _[expr round($gpls(fwhm) * 1000)]
		append peakdata _[expr round($gaussian(sens) * 10)]
		append peakdata _[expr round($gaussian(esd) * 10)]_pks
		set peakdata [initpeaks $peakdata]
		global $peakdata
		set ${peakdata}(xlabel) "2theta"
		set ${peakdata}(xunits) "2theta"
		set ${peakdata}(ylabel) [set ${data}(ylabel)]
		set ${peakdata}(x) $tt
		set ${peakdata}(y) $area
		set ${peakdata}(widths) $wid
		set ${peakdata}(etas) 0
		set ${peakdata}(heights) $height
		set ${peakdata}(bkgs) $bkg
		resetdata $peakdata
		# add the new file to the plot list
		if {[lsearch $graph(plotlist) $peakdata] == -1} {
		    lappend graph(plotlist) $peakdata
		}
	    }
	}
    }
    if {$data == "(none)"} {
	tk_dialog .msg "nothing to do" \
		"Error: you have not selected any files to process" \
		error 0 OK
	return
    }
    updateselectbox $box
    displaylist
    # cleanup files
    if {!$command(debug)} {
	foreach lbl {smo bkg 1st 2nd inp out x pks} {
	    file delete gauss.$lbl
	}
    }
}
#--------------------------------------------------------------------
lappend menulist(compute) Index
lappend menulist(pages) Index
set helplist(Index) {
An interface to autoindexing routines TREOR, ITO and DICVOL

Note that peaks must be in units of 2theta, Q or d-space.
}
# see index.tcl for routines
#--------------------------------------------------------------------
#lappend menulist(options) Options
#lappend menulist(pages) Options
set helplist(Options) {\
Used to set values used in other parts of the program.
  "Plot legend" causes a legend to be shown on the side of the plot.
      The legend box will list the name of the dataset, unless the 
      "Use Title for Legend" option is selected. When the title is blank, 
      the legend for the dataset is omitted. Note that clicking on the 
      legend for a file causes that graph to be highlighted. Also, 
      the title for a dataset can be edited in the EditFile panel.

  "Write PostScript" and "Print PostScript" (UNIX only) determine
      how hardcopy is generated.

  "Report fit results in plot window" places fit output in plot
     or a separate window.

  "Plot in listed order" causes the curves to be overlaid in the order they 
     are listed in the dataset listing. This means that the second selected data 
     set will be plotted "on top of" (obscuring) the first. The third obscures
     the first and second,... When the box is unchecked, the overlay order 
     is reversed.

  "2theta max" determines the range of reflections generated
  "max change" controls the slider range and sensitivity
  "HKL label erase time" specifies the time to display reflection
     labels. 0 means delete them only with the right-mouse button
  "HKL label size" specifies the reflection labels size in pixels
     (or points for negative values).

  "Help Options" determine if help is displayed in a Web browser
     or a separate window.

  "Install ICDD database" invokes the LOGIC encode script to install 
     the ICDD-JCPDS database for use in CMPR and LOGIC
}

proc MakeOptions {page}  {
    global command graph tcl_version
    #global tcl_platform
    catch {destroy $page}
    toplevel $page
    set command(options_page) $page
    pack [frame $page.1] -side left -anchor n
    pack [frame $page.1.a -bd 4 -relief groove] -side top -anchor w
    pack [label $page.1.a.1 -text {Select plot options}] -side top
    pack [frame $page.1.a.2] -side top -anchor w -fill x
    pack [checkbutton $page.1.a.2.a \
	    -text "Plot Legend" -variable graph(legend) \
	    -command graphlegend] -side left -anchor w
    pack [checkbutton $page.1.a.2.b \
	    -text "Use Title for Legend" -variable graph(LabelByTitle) \
	    -command displaylist] -side left -anchor e
	pack [frame $page.1.a.10] -side top -anchor w -fill x
	pack [entry $page.1.a.10.2 -textvariable graph(title)] -side right
	pack [label $page.1.a.10.1 -text "Plot Title:"] -side left
		pack [checkbutton $page.1.a.3 -text "Write PostScript files" \
	    -variable graph(printout) -offvalue 0 -onvalue 1 \
	    -command "setprintopt $page.1.a"] -side top -anchor w
    pack [frame $page.1.a.4] -side top -anchor w -fill x
    pack [entry $page.1.a.4.2 -textvariable graph(outname)] -side right
    pack [label $page.1.a.4.1 -text "PostScript file name:"] -side left
    pack [checkbutton $page.1.a.5 -text "Print PostScript files" \
	    -variable graph(printout) -offvalue 1 -onvalue 0 \
	    -command "setprintopt $page.1.a" ] -side top -anchor w
    pack [frame $page.1.a.6] -side top -anchor w -fill x
    pack [entry $page.1.a.6.2 -textvariable graph(outcmd)] -side right 
    pack [label $page.1.a.6.1 -text "Command to print files:"] -side left
    pack [checkbutton $page.1.a.7 \
	    -text "Plot in listed order" -variable graph(ReversePlotOrder) \
	    -command displaylist] -side top -anchor w

    pack [frame $page.1.b -bd 4 -relief groove] -side top -fill x -anchor w
    pack [label $page.1.b.1 -text {Peak fit options}] -side top
    pack [checkbutton $page.1.b.2 -text "Report fit results in plot window" \
	    -variable gpls(fitonplot)] -side top -anchor w
    pack [checkbutton $page.1.b.3 -text "Plot fit results in separate window" \
	    -variable gpls(fitinnewplot)] -side top -anchor w

    pack [frame $page.2] -side left -anchor n
    pack [frame $page.2.c -bd 4 -relief groove] -side top  -anchor w
    pack [label $page.2.c.1 -text {EditCell options}] -side top
    #pack [frame $page.2.c.y] -side top -fill both
    #pack [frame $page.2.c.a] -side top  -anchor w
    #pack [label $page.2.c.a.l -width 10 -anchor e -text "2theta max:"] \
	#    -padx 1m -pady 1m -side left
    #pack [entry $page.2.c.a.e \
	#    -width 6 -bd 2 -textvariable cellparm(twothetamax)] \
	#    -padx 1m -pady 1m  -side left
    pack [frame $page.2.c.b] -side top  -anchor w
    pack [label $page.2.c.b.l -text "Max change\nin cell:"] -side left
    pack [entry $page.2.c.b.e -textvariable cellparm(sens) -width 3] \
	    -side left
    pack [label $page.2.c.b.l1 -text %] -side left
    pack [frame $page.2.c.abc] -side top  -anchor w
    pack [label $page.2.c.abc.l -text "Cell edge\nresolution:"\
	    -justify left] -side left
    pack [entry $page.2.c.abc.e -textvariable cellparm(abc_tol) -width 6] \
	    -side left
    pack [label $page.2.c.abc.u -text "A"] -side left
    pack [frame $page.2.c.ang] -side top  -anchor w
    pack [label $page.2.c.ang.l -text "Cell angle\nresolution:"\
	    -justify left] -side left
    pack [entry $page.2.c.ang.e -textvariable cellparm(alphabeta_tol) \
	    -width 6] -side left
    
    pack [label $page.2.c.ang.u -text "deg."] -side left
    pack [frame $page.2.c.c] -side top  -anchor w
    pack [label $page.2.c.c.l -text "HKL label\nerase time:"\
	    -justify left] -side left
    pack [entry $page.2.c.c.e -textvariable cellparm(fadetime) -width 5] \
	    -side left
    pack [label $page.2.c.c.l1 -text seconds] -side left
    pack [frame $page.2.c.d] -side top  -anchor w
    pack [label $page.2.c.d.l -text "HKL label size:"] -side left
    pack [entry $page.2.c.d.e -textvariable cellparm(fontsize) -width 3] \
	    -side left
    pack [label $page.2.c.d.l1 -text pixels] -side left
    # old versions if tcl/tk don't support the font command
    if {$tcl_version < 8.0} {
	$page.2.c.d.l config -fg #888
    	$page.2.c.d.e config -fg #888 -state disabled
	$page.2.c.d.l1 config -fg #888
    }
    pack [frame $page.2.h -bd 4 -relief groove] -side top -fill x -anchor w
    pack [label $page.2.h.1 -text {Help options}] -side top
    pack [radiobutton $page.2.h.2 \
	      -text "Help in browser" -variable command(helpmode) \
	      -value 0] -side top -anchor w
    pack [radiobutton $page.2.h.3 \
	      -text "Help in help window" -variable command(helpmode) \
	      -value 1] -side top -anchor w

    pack [button $page.1.s -text "Save Options" -command "SaveDefaults"] \
	    -side top 
    pack [button $page.1.q -text Close -command "destroy $page"] \
	    -side top 
    raise $page
    PostPageOptions $page
    bind $page <Key-F1> {MakeHelp .help options}
    # seems to be needed in OSX
    update
    wm geom $page [winfo reqwidth $page]x[winfo reqheight $page]
}

proc setprintopt {page} {
    global graph tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	set graph(printout) 1
	$page.3 config -state disabled -fg black
	$page.5 config -fg #888 -state disabled
    }
    if $graph(printout) { 
	$page.4.1 config -fg black
	$page.4.2 config -fg black -state normal
	$page.6.1 config -fg #888 
	$page.6.2 config -fg #888 -state disabled
    } else {
	$page.4.1 config -fg #888 
	$page.4.2 config -fg #888 -state disabled
	$page.6.1 config -fg black
	$page.6.2 config -fg black -state normal
    }
}

proc PostPageOptions {page} {
    setprintopt $page.1.a
    graphlegend
}

proc SaveWorkingDirectory {} {
    if {! [info exists ::command(initwd)]} {
	MyMessageBox -parent . -title "No WD" \
	    -icon error -type Continue -default continue -message \
	    "Ignored: The working directory has not been set."
	return
    }
    namespace eval ::userdefaults {}
    set ::userdefaults::command(initwd) $::command(initwd)
    SaveDefaults
}

proc SetWorkingDirectory {} {
    global command
    set dir [tk_chooseDirectory -mustexist 1]
    if {$dir == ""} return
    catch {
	cd $dir
	set command(initwd) $dir
	set ::FilePane(DirButton) [pwd]
	FilePaneSelChoose
    }
    set command(pwd) [pwd]
}


#--------------------------------------------------------------------
#--------------------------------------------------------------------
#lappend menulist(?) X..X
#lappend menulist(pages) X..X
#set helplist(X..X) {}
#proc MakeX..X {page}  {}
#proc PostPageX..X {page} {} 
#--------------------------------------------------------------------
#foreach f {linterp.tcl} {
#    set file [file join $scriptdir $f]
#    if {[file exists $file]} {source $file}
#}
foreach cmd $command(ExecuteLater) {
    eval $cmd
}
#--------------------------------------------------------------------
# show the 'about' message
proc aboutcmpr {} {
    set version $::Revision
    catch {
	set version "[GetSVNVersion $::scriptdir]"
    }
    if $::command(cmprversion) {
	set vermsg "\n$version: Using development release"
    } else {
	set vermsg "\n$version: Using stable release"
    }
    if {[llength $::command(initfiles)] == 0} {
	append vermsg "\n\nNo initialization files used"
    } else {
	append vermsg "\n\nInitialization files:"
	foreach file $::command(initfiles) {
	    append vermsg "\n\t$file"
	}
    }
    MyMessageBox -parent . -title About \
	-message "Program CMPR\n\n\
Brian Toby, Advanced Photon Source, Argonne National Lab (Brian.Toby@ANL.gov)\n\
Not subject to copyright\n\
$vermsg
\nIncluded codes:\n\
* HKLGEN: C. Hubbard, J. Stalick, A. Mighell & others\n\
* GPLSFT: D. Cox, W. Hamilton, L. Finger & many others\n\
* Space group extinctions: A. Larson & NRC Canada\n\
* Gaussian smoothing/peak search: R. Harlow & A. McGhie\n\
* ITO: J. W. Visser\n\
* DICVOL: A. Boultif and D. Louër\n\
* TREOR: P.-E. Werner"
}
#--------------------------------------------------------------------
#--------------------------------------------------------------------
#lappend menulist(display) Help
#lappend menulist(pages) Help
proc MakeHelp {page "opt {}"}  {
    global helplist helpmsg graph command tcl_platform env scriptdir
    if {$opt == ""} {set opt $command(pagenow)}
    set helpmsg {}
    if {!$command(helpmode) || $page == ""} {
	NetHelp cmprdoc.html $opt [file join $scriptdir doc] \
	    www.ncnr.nist.gov/xtal/software/cmpr
	return
    }

    catch {toplevel $page}
    eval destroy [grid slaves $page]
    raise $page
    grid [label $page.0 -text \
	    "Click on a topic for information" ] \
	    -column 0 -row 0 -columnspan 3
    listbox $page.cmds -relief raised -bd 2 -yscrollcommand \
	    "$page.scroll set" -height 10 -width 0
    scrollbar $page.scroll -command "$page.cmds yview"
    bind $page.cmds <ButtonRelease-1> \
	    "+set helpmsg \
	    \$helplist(\[$page.cmds get \
	    \[$page.cmds curselection\]\])"
    bind $page.cmds <ButtonRelease-1> "$page.help delete 0.0 end; $page.help insert end \$helplist(\[$page.cmds get \[$page.cmds curselection\]\])"

    grid $page.scroll -column 0 -row 1 -sticky ns
    grid $page.cmds  -column 1 -row 1 -sticky news
    grid [text $page.help -yscrollcommand "$page.helpscroll set"\
	     -width 70 -height 15 -relief groove] \
	   -column 2 -row 1 -rowspan 2  -sticky nsew
    grid [scrollbar $page.helpscroll -command "$page.help yview"] \
	   -column 3 -row 1 -rowspan 2  -sticky nsew
    grid [button $page.q -text Close -command "destroy $page"] \
	    -column 0 -row 2 -columnspan 2
    PostPageHelp $page $opt
    grid columnconfigure $page 2 -weight 1
    grid rowconfigure $page 2 -weight 1
}

proc PostPageHelp {page "opt {}"} {
    global helplist command helpmsg
    $page.cmds delete 0 end
    set nameslist [lsort [array names helplist]]
    eval $page.cmds insert end $nameslist
    if {[$page.cmds curselection] == ""} {$page.cmds selection set 0}
    # see if we can find help for the current page
    set num [lsearch [string tolower $nameslist] $opt]
    if {$num == -1} return
    $page.cmds selection clear 0 end
    $page.cmds see $num
    $page.cmds selection set $num
    #set helpmsg $helplist([$page.cmds get $num])
    $page.help delete 0.0 end
    $page.help insert end $helplist([$page.cmds get $num])
}
bind . <Key-F1> {MakeHelp .help}

proc NetHelp {file anchor localloc netloc} {
    # use the file on-line, if it exists
    if {[file exists [file join $localloc $file]]} {
	set url "[file join $localloc $file]"
    } else {
	set url "http://$netloc/$file"
    }
    catch {
	pleasewait "Starting web browser..."
	after 2000 donewait
    }
    if {$anchor != ""} {
	append url # $anchor
    }
    urlOpen $url
}

# browse a WWW page with URL. The URL may contain a #anchor
# On UNIX assume netscape or mozilla is in the path or env(BROWSER) is loaded. 
# On Windows search the registry for a browser. 
proc urlOpen {url} {
    global env tcl_platform
    if {$tcl_platform(os) == "Darwin"} {
	# if this is an external URL or does not contain an anchor, take the 
	# easy approach for web URLs without an anchor
 	if {[string range $url 0 4] == "http:" || \
		[string first "#" $url] == -1} {
	    if {![catch {exec open $url}]} {
		return
	    }
	}
 	if {[string range $url 0 4] != "http:"} {
	    # fix web pages referenced by file name file
	    set url [file nativename $url]; # replace ~/ if present
	    if {[file pathtype $url] == "relative"} {
		set url [file join [pwd] $url]
	    }
	    set url "file://$url"
	}
	if [catch {
	    exec osascript -e "open location \"$url\" "
	}] {
	    MyMessageBox -parent . -title "Browser problem" -icon warning \
		-message "script 'open location <URL>' failed (older OS?). Please open URL\n\n$url\n\n in your favorite browser."	    
	}
    } elseif {$tcl_platform(platform) == "unix"} {
	set browserlist {}
	if {[info exists env(BROWSER)]} {
	    set browserlist $env(BROWSER)
	}
	lappend browserlist netscape mozilla 
	foreach p $browserlist {
	    set progs [auto_execok $p]
	    if {[llength $progs]} {
		if {[catch {eval exec $progs -remote openURL($url)}]} {
		    # perhaps browser doesn't understand -remote flag
		    if {[catch {exec $env(BROWSER) $url &} emsg]} {
			error "Error displaying $url in browser\n$emsg"
		    }
		}
		return
	    }
	}
	MyMessageBox -parent . -title "No Browser" \
	    -message "Could not find a browser. Netscape & Mozilla not found. Define environment variable BROWSER to be full path name of browser." \
	    -icon warning
    } elseif {$tcl_platform(platform) == "windows"} {
	package require registry
	# Look for the application under
	# HKEY_CLASSES_ROOT
	set root HKEY_CLASSES_ROOT
	
	# Get the application key for HTML files
	set appKey [registry get $root\\.html ""]
	
	# Get the command for opening HTML files
	set appCmd [registry get \
			$root\\$appKey\\shell\\open\\command ""]

	# Substitute the HTML filename into the command for %1
	# or stick it on the end
	if {[string first %1 $appCmd] != -1} {
	    regsub %1 $appCmd $url appCmd
	} else {
	    append appCmd " " $url
	}
	
	# Double up the backslashes for eval (below)
	regsub -all {\\} $appCmd  {\\\\} appCmd
	
	# Invoke the command
	eval exec $appCmd &
    } elseif {$tcl_platform(platform) == "macintosh"} {
	# preOSX -- this is not used
	if {0 == [info exists env(BROWSER)]} {
	    set env(BROWSER) "Browse the Internet"
	}
	if {[catch {
	    AppleScript execute\
		"tell application \"$env(BROWSER)\"
                         open url \"$url\"
                     end tell
                "} emsg]
	} then {
	    error "Error displaying $url in browser\n$emsg"
	}
    }
}

#--------------------------------------------------------------------
#--------------------------------------------------------------------
proc graphlegend {} {
    global graph blt_version
    if {$blt_version >= 2.3 && $blt_version < 8.0} {
	if $graph(legend) {
	    $graph(blt) legend config -hide no
	} else {
	    $graph(blt) legend config -hide yes
	}
    } else {
	if $graph(legend) {
	    $graph(blt) legend config -mapped yes
	} else {
	    $graph(blt) legend config -mapped no
	}
    }
}

proc makepostscriptout {} {
    global graph
    if !$graph(printout) {
	set out [open "| $graph(outcmd) >& plot.msg" w]
	catch {
	    puts $out [$graph(blt) postscript output -landscape 1 \
		-decorations no -height 7.i -width 9.5i]
	    close $out
	} msg
	catch {
	    set out [open plot.msg r]
	    if {$msg != ""} {append msg "\n"}
	    append msg [read $out]
	    close $out
	    file delete plot.msg
	}
	if {$msg != ""} {
	    tk_dialog .msg "file created" \
		    "Postscript file processed with command \
		    $graph(outcmd). Result: $msg" "" 0 OK
	} else {
	    tk_dialog .msg "file created" \
		    "Postscript file processed with command \
		    $graph(outcmd)" "" 0 OK
	}
    } else {
	
    set psoutname [file join [pwd] $graph(outname)]
	set psoutname [tk_getSaveFile -title "Select output file" -parent . \
			  -initialdir [file dirname $psoutname] \
			  -initialfile [file tail $psoutname]]
	
	$graph(blt) postscript output $psoutname -landscape 1 \
		-decorations no -height 7.i -width 9.5i    
	tk_dialog .msg "file created" \
		"Postscript file $graph(outname) created" "" 0 OK
    }
}

proc makepdfout {} {
	global graph command
    set pdfoutname [file join [pwd] $graph(outname)]
	
	#outname was set for .ps, and could be anything, so make sure it has a .pdf file extension
	if {[file extension $pdfoutname] != ".pdf"} { set pdfoutname [file rootname $pdfoutname].pdf }
	set pdfoutname [tk_getSaveFile -title "Select output file" -parent . \
			  -initialdir [file dirname $pdfoutname] \
			  -initialfile [file tail $pdfoutname]]
	
	# set a name for the temporary postscript file, and make sure that such a file doesn't exist
	# if that file does exist, find a new name until you find a winner
	set psoutname "$pdfoutname.ps"
	set i 0
	set renameps 1
	while { $renameps } {
		if {[file isfile $psoutname]} {
			set psoutname "$pdfoutname$i.ps"
			incr i
		} else { 
			set renameps 0
			break 
		}
	}
	
	# to avoid potentiall errors with ghostscript overwrite, delete the file with Tcl
	# if there is such a file, the user has already confirmed overwrite with tk_getSaveFile
	if {[file isfile $pdfoutname]} { file delete $pdfoutname }
	
	# I haven't figured out the postscript -fontmap property yet.  It isn't obvious
	# the only work around I can think of is to set the font-size, and then reset it
	# after the export
	set graphfont [$graph(blt) cget -font]
	set graphyaxisfont [$graph(blt) yaxis cget -titlefont]
	set graphxaxisfont [$graph(blt) xaxis cget -titlefont]
	$graph(blt) configure -font *-Times-R-Normal-*-32-* -plotpadx 2 -plotpady 2
	$graph(blt) yaxis configure -titlefont *-Times-R-Normal-*-26-*
	$graph(blt) xaxis configure -titlefont *-Times-R-Normal-*-26-* -tickfont *-Courier-R-Normal-*-18-*
	
	$graph(blt) postscript output $psoutname -landscape 1 -decorations no \
		-height 7.5i -width 10.i -padx 0.5i -pady 0.5i

	$graph(blt) configure -font {Arial 12}
	$graph(blt) yaxis configure -titlefont {Arial 8} -tickfont {Arial 8}
	$graph(blt) xaxis configure -titlefont {Arial 8} -tickfont {Arial 8}
	
	if {$::tcl_platform(platform) == "windows"} {	
		set gspath [file join $command(exedir) gswin32.exe] 
		exec $gspath -sDEVICE=pdfwrite -o $pdfoutname -dNOPAUSE $psoutname 	
		tk_dialog .msg "file created" \
			"PDF file $pdfoutname created" "" 0 OK	
		
		# automatically open the pdf if the user so desires
		if { $graph(viewpdfout) } { eval exec [auto_execok start] \"\" [list $pdfoutname]; }
		
	} elseif {$::tcl_platform(os) == "Darwin"} {
		exec pstopdf $psoutname -o $pdfoutname	
		tk_dialog .msg "file created" \
			"PDF file $pdfoutname created" "" 0 OK
		if { $graph(viewpdfout) } { exec open $pdfoutname }
		
	} elseif {$::tcl_platform(platform) == "unix"} {
		exec gs -sDEVICE=pdfwrite -o $pdfoutname -dNOPAUSE $psoutname 	
		tk_dialog .msg "file created" \
			"PDF file $pdfoutname created" "" 0 OK
		if { $graph(viewpdfout) } { exec see $pdfoutname }
		
	} else {
		tk_dialog .msg "unrecognized platform, pdf conversion not attempted" "" 0 OK
	}

	file delete $psoutname
	

}

proc OpenDirectory {file} {
    catch {cd $file}
}

# invoke the logic encode script
proc ExecEncode {} {
    global scriptdir
    exec [info nameofexecutable] \
	[file join $scriptdir logic encode encode.tcl] &
}
#-------------------------------------------------------------------------
#process command line arguments
if {[llength $argv] > 0 && [string range [lindex $argv 0] 0 0] != "-"} {
    set command(read_filelist) ""
    set filelist $argv
    set typelist {}
    set command(readproclist) {}
    set problems {}
    set multiple 0
    set gotdir 0
    foreach file $filelist {
	if {[file isdirectory $file]} {
	    if {$gotdir} {
		lappend command(readproclist) ""
		lappend typelist {(Only 1st dir is used)}
	    } else {
		lappend typelist {}
		lappend command(readproclist) "OpenDirectory"
		set gotdir 1
	    }
	} elseif {[file exists $file]} {
	    set ext [string range [file extension $file] 1 end]
	    if {$::tcl_platform(platform) == "windows"} {
		set ext [string tolower $ext]
	    }
	    set lbls {}
	    set procs {}
	    foreach lst $command(filterlist) lbl $command(readtypes) prc $command(readproc) {
		if {$::tcl_platform(platform) == "windows"} {
		    set lst [string tolower $lst]
		}
		if {[lsearch $lst $ext] != -1} {
		    lappend lbls $lbl 
		    lappend procs $prc
		}
	    }
	    lappend typelist $lbls
	    lappend command(readproclist) $procs
	    if {$lbls == ""} {
		lappend problems $file
	    }
	    if {[llength $lbls] > 1} {
		set multiple 1
	    }
	} else {
	    lappend typelist {File not found}
	    lappend command(readproclist) ""
	    lappend problems $file
	}
    }

    if {$multiple || $problems != ""} {
	wm withdraw .
	wm withdraw .plot
	catch {destroy .choose}
	toplevel [set page .choose]
	set lbl {}
	if {$multiple} {
	    set lbl "Choose formats for files with ambiguous extensions."
	} 
	if {$problems != ""} {
	    if {$lbl != ""} {append lbl "\n & "}
	    append lbl "Please review problems listed below."
	}
	grid [label $page.top -text $lbl -bg white ]\
	    -sticky  new -row 0 -column 1 -columnspan 2
	grid rowconfig $page 0 -pad 5
	grid [canvas $page.b \
		  -scrollregion {0 0 5000 500} -width 150 -height 200 \
		  -yscrollcommand "$page.yscroll set" -bg lightgrey] \
	    -sticky  news -row 1 -column 1
	grid [scrollbar $page.yscroll -orient vertical \
		  -command "$page.b yview"] \
	    -sticky nse -row 1 -column 2
	set fmtbox [frame $page.b.f]
	$page.b create window 0 0 -anchor nw  -window $fmtbox
	set row -1
	foreach file $filelist type $typelist prc $command(readproclist) {
	    incr row
	    grid rowconfig $fmtbox $row -pad 5
	    grid [label $fmtbox.${row}file -text $file] -column 0 -row $row -sticky w
	    if {$prc == "OpenDirectory"} {
		grid [label $fmtbox.${row}type -text "<Directory>"] -column 1 -row $row -sticky w
		
	    } elseif {$prc == "" && $type == ""} {
		grid [label $fmtbox.${row}type -text "Unknown extension" -fg red] -column 1 -row $row -sticky w
	    } elseif {$prc == ""} {
		grid [label $fmtbox.${row}type -text $type -fg red] -column 1 -row $row -sticky w
	    } elseif {[llength $prc] == 1} {
		grid [label $fmtbox.${row}type -text [lindex $type 0]] -column 1 -row $row -sticky w
	    } else {
		# create a menu button here
		set menu [tk_optionMenu $fmtbox.${row}menu command(read$row) x]
		set command(read$row) "(select format)"
		grid $fmtbox.${row}menu -column 1 -row $row -sticky w
		$menu delete 0 end
		foreach t $type p $prc {
		    $menu add radiobutton -value $t -label $t \
			-variable command(read$row) \
			-command "set command(readproclist) \
                           \[lreplace \$command(readproclist) $row $row $p]"
		}
	    }
	}
	grid columnconfig $fmtbox 1 -minsize 200
	grid [button $page.done -text "Continue" -command "destroy $page"]\
	    -row 2 -column 1 -columnspan 2
	update idletasks
	$page.b config -scrollregion [grid bbox $fmtbox]
	$page.b config -width [lindex [grid bbox $fmtbox] 2]
	tkwait window $page
    }

    foreach file $filelist prc $command(readproclist) {
	if {[llength $prc] == 1} {
	    $prc $file
	}
    }
} else {
    set cnt 0
    set format {}
    while {$cnt < [llength $argv]} {
	set fmt [lindex $argv $cnt]
	if {[string range $fmt 0 0] == "-"} {
	    incr cnt
	    set fmtnum [lsearch $command(cmdopt) $fmt]
	    if {$fmtnum < 0} {
		MyMessageBox -title "command line error" -type OK -default ok \
		    -message "Unknown command line option $fmt\nKnown formats are:\n$command(cmdopt)"
		exit
	    }
	    set format $fmt
	}
	if {$format == ""} {
	    MyMessageBox -title "command line error" -type OK -default ok \
		-message "No format specified\nKnown formats are:\n$command(cmdopt)"
	    exit
	}
	foreach file [lindex $argv $cnt] {
	    #puts "Reading $file with $command($format)"
	    if [file exists $file] {
		set ret [$command($format) $file]
		if {$ret != ""} {
		    tk_dialog .err "Read Error" \
			"Error reading $format file\n$file:\n$ret" \
			error 0 OK
		}
	    } else {
		MyMessageBox -title "File error" -type OK -default ok \
		    -message "File $file not found"
	    }
	}
	incr cnt
    }
}
#-------------------------------------------------------------------------
# set up zooming of the axes; modify zooming so that it feeds back
# to the fit page
if [catch {
    Blt_ZoomStack $graph(blt)
    Blt_ActiveLegend $graph(blt)
} errmsg] {
    tk_dialog .err "BLT Error" \
	    "Error in BLT setup, support routine problem: $errmsg \
	    The pkgIndex.tcl is probably not loading bltGraph.tcl" \
	    error 0 "Limp ahead"
}
$graph(blt) xaxis configure -title {}
$graph(blt) yaxis configure -title {}
$graph(blt) config -title {} -plotbackground white
$graph(blt) legend config -font  *-Helvetica-Bold-R-Normal-*-10-*-*
#-------------------------------------------------------------------------
# define a menubar
catch {destroy .a}
grid [frame .a -bd 4 -relief groove -class MenuFrame] \
	-column 0 -row 0 -sticky nsew

foreach menu {File Display Compute Options} {
    set m [string tolower $menu]
    pack [menubutton .a.$m -text $menu -underline 0 -menu .a.$m.menu] \
	    -side left
    menu .a.$m.menu
    foreach item $menulist($m) {
	set cmd [lindex $item 0]
	set lbl [lindex $item 1]
	if {$lbl == ""} {set lbl $cmd}
	.a.$m.menu add command \
		-label $lbl -command "DoCommand $cmd"
	if {$command(debug) && [catch {set helplist($cmd)}]} {puts "no help for $cmd"}
    }
}

proc SetVarColor {var} {
    set cl [tk_chooseColor -initialcolor [set $var]]
    if {$cl != ""} {
	set $var $cl
    }
}

# special menus
.a.options.menu add command -command "MakeOptions .options" -label "Display options"
.a.options.menu add cascade -menu .a.options.menu.font \
	-label "Screen font"
.a.options.menu add command -command SetWorkingDirectory -label "Set working directory"
.a.options.menu add command -command SaveWorkingDirectory -label "Save WD as starting directory"
.a.options.menu add command -label "Set EditCell color" \
    -command "SetVarColor ::graph(editcellcolor)"
.a.options.menu add command -label "Set Extinct color" \
    -command "SetVarColor ::graph(extinctcolor)"
if {[file exists [file join $scriptdir logic encode encode.tcl]]} {
    .a.options.menu add command -label "Install ICDD database" \
	-command ExecEncode
}

if {[file exists [file join $scriptdir fitmax.tcl]]} {
	source [file join $scriptdir fitmax.tcl]
}

menu .a.options.menu.font 
foreach f {10 11 12 13 14 16 18 20 22} {
    .a.options.menu.font add radiobutton \
	    -command {SetTkDefaultOptions $graph(font); ResizeFont .; ResizeNotebook} \
	-label $f -value $f -variable graph(font) -font "Helvetica -$f"
}

pack [menubutton .a.help -text Help -underline 0 -menu .a.help.menu] -side right
menu .a.help.menu -tearoff 0
.a.help.menu add command -command "MakeHelp .help" -label "Help on current page"
.a.help.menu add command -command "MakeHelp {}"  -label "CMPR web documentation"
if {![catch {package require tkcon} errmsg]} {
    .a.help.menu add command -label "Open tkcon console" \
	-command {tkcon show}
} 
if {$tcl_platform(platform) == "windows"} {
    .a.help.menu add command -label "Open console" \
	-command {console show}
}

#-------------------------------------------------------------------------
# create a frame for the notebook page
set command(notebook) .b
if $command(haveBW) {
    NoteBook $command(notebook) -bd 2
    # create a notebook page for each dialog box
    foreach page $menulist(pages) {
	set pagename [string tolower $page]
	set notepage [$command(notebook) insert end $pagename -text $page ]
	# create the largest page
	if {$page == "EditCell" || $page == "Index" || $page == "Plot"} {
	    Make$page $notepage
	} else {
	    $command(notebook) itemconfigure $pagename \
		-createcmd "Make$page $notepage" 
	}
	$command(notebook) itemconfigure $pagename -raisecmd \
		"SetPostPage $page [string tolower $pagename]; \
		PostPage$page $notepage; \
		$command(notebook) compute_size"
    }
} else {
    Notebook:create $command(notebook) -pages $menulist(pages)
    # fill the notebook page for each dialog box
    foreach page $menulist(pages) {
	set pagename [Notebook:frame $command(notebook) $page]
	Make$page $pagename
	Notebook:pageconfig $command(notebook) $page -command \
		"SetPostPage $page [string tolower $page]; \
                PostPage$page $pagename"
    }
    ResizeNotebook
}

# respond to the posting of a new notebook page
proc SetPostPage {page pagename} {
    global command graph
    # leaving the fit page? restore the plot window and hide the fit window
    #if {$command(pagenow) == "fit"} {
    #}
    set command(pagenow) $pagename
}

#######################################################################
# code to do updates from the SVN repository
#######################################################################
# can we find the svn program?
proc CheckSVNinstalled {} {
    # can we find svn in the path?
    if {[auto_execok svn] != ""} {return 1}
    # add a locally supplied svn version and add to path
    if {$::tcl_platform(platform) == "windows"} {
	set s [file attributes $::scriptdir -shortname]
    } else {
	set s $::scriptdir
    }
    # look for svn first in current directory and then assume the development version
    set localsvn [file join $s svn bin]
    if {! [file exists $localsvn]} {
	set localsvn [file normalize [file join $s .. svn bin]]
    }
    if {[file exists $localsvn]} {
	if {$::tcl_platform(platform) == "windows"} {
	    set localsvn [file nativename $localsvn]
	    set sep {;}
	} else {
	    set sep {:}
	}
	if {[lsearch [split $::env(PATH) $sep] $localsvn] == -1} {
	    append ::env(PATH) $sep $localsvn
	    # note that auto_reset breaks the tkcon package in Windows -- not sure why
	    auto_reset
	}
    }
    if {[auto_execok svn] != ""} {return 1}
    return 0
}

proc GetSVNVersion {scriptdir} {
    if {[CheckSVNinstalled]} {
	set SVN [auto_execok svn]
	if {! [catch {set res [eval exec $SVN info [list $scriptdir]]} err]} {
	    set infolist [split $res]
	    set pos [lsearch $infolist "Revision:"]
	    return "CMPR SVN version [lindex $infolist [incr pos]]"
	}
    }
    return "CMPR version: [lindex $::Revision 1] ([lindex $::Revision 4])"
}

proc CMPRversionSetMenu {args} {
    if $::command(cmprversion) {
	.a.help.menu.track entryconfig [.a.help.menu.track index Standard*] \
	    -state normal -label "Standard"
	.a.help.menu.track entryconfig [.a.help.menu.track index Development*] \
	    -state disabled -label "Development (selected)"
    } else {
	.a.help.menu.track entryconfig [.a.help.menu.track index Standard*] \
	    -state disabled -label "Standard (selected)"
	.a.help.menu.track entryconfig [.a.help.menu.track index Development*] \
	    -state normal  -label "Development"
    }
}

proc SetCMPRversion {} {
    global scriptdir
    if {! [CheckSVNinstalled]} {
	MyMessageBox -parent . -title "SVN not found" \
	    -message "Unable to upgrade CMPR: Could not locate a copy of the subversion program. It does not appear that one of self-updating CMPR releases was installed" \
	    -icon error
	return 0
    }

    # are we switching to the development version?
    if $::command(cmprversion) {
	if {![file exists [file join $scriptdir cmprdev cmprmain.tcl]]} {
	    # are we already running from the CMPRDEV area? (somehow we have gotten confused)
	    if {[file tail $scriptdir] == "cmprdev"} {
		MyMessageBox -parent . -title "Development version running?" \
			-message "Seems as if the development version of CMPR is already running." 
	    } elseif {[file exists [file join $scriptdir cmprdev]]} {
		# try adding the new dev version of CMPR using switch
		set newURL "https://subversion.xor.aps.anl.gov/CMPR/trunk"
		set devloc [file join $scriptdir cmprdev]
		if [catch {eval exec [auto_execok svn] switch $newURL [list $devloc]} err] {
		    MyMessageBox -parent . -title "Error in switch" \
			-message "Error adding development files:\n$err" \
			-icon error
		    set :command(cmprversion) 0
		    return 0
		} else {
		    MyMessageBox -parent . -title "Development version loaded" \
			-message "The development version of CMPR was not found, but has been loaded by subversion." 
		}
	    } else {
		# we have a newer version of cmpr, but no cmprdev directory
		# this is strange
		MyMessageBox -parent . -title "No Dev version" \
		    -message "Unable to upgrade CMPR to have the development version. If the Help/Update button is present, please try that. If not, you will need to install the self-updating CMPR release for this to work" \
		    -icon error
		set :command(cmprversion) 0
		return 0
	    }
	}
	set msg "CMPR will start next as the Development version.\n\n"
    } else {
	set msg "CMPR will start next as the Stable version.\n\n"
    }
    append msg {Press the "Restart" button to start with this version now. You will need to reread all files and any unsaved information will be lost. Press Cancel to restart later; the version change will still take place at next restart.}
    # save the version 
    namespace eval ::userdefaults {}
    set ::userdefaults::command(cmprversion) $::command(cmprversion)
    SaveDefaults
    if {[MyMessageBox -parent . -title "Ready to switch" \
	-message $msg \
		 -type {Cancel Restart} -default cancel -icon warning
	] == "cancel"} {
	if $::command(cmprversion) {
	    .a.help.menu.track entryconfig [.a.help.menu.track index Standard*] \
		-state disabled -label "Standard (running)"
	    .a.help.menu.track entryconfig [.a.help.menu.track index Development*] \
		-state disabled -label "Development (after restart)"
	} else {
	    .a.help.menu.track entryconfig [.a.help.menu.track index Standard*] \
		-state disabled -label "Standard (after restart)"
	    .a.help.menu.track entryconfig [.a.help.menu.track index Development*] \
		-state disabled  -label "Development (running)"
	}
	return
    } else {
	exec [info nameofexecutable] [file normalize $::scriptname] &
	exit
    }
}

proc CheckAndDoUpdate { } {
    global scriptdir
    if {! [CheckSVNinstalled]} {
	MyMessageBox -parent . -title "SVN not found" \
	    -message "Unable to upgrade CMPR: Could not locate a copy of the subversion program. It does not appear that one of self-updating CMPR releases was installed" \
	    -icon error
	set :command(cmprversion) 0
	return
    }
    #is there a svn directory in the source?
    if {! [file exists [file join $scriptdir .svn]]} {
	MyMessageBox -parent . -title "Not installed for update" \
	    -message "Unable to upgrade CMPR: It does not appear that one of self-updating CMPR releases was installed" \
	    -icon error
	set :command(cmprversion) 0
	return
    }
    if {$::command(cmprversion)} {
	set homedir [file normalize [file join $scriptdir ..]]
    } else {
	set homedir $scriptdir
    }
    # check for updates
    set SVN [auto_execok svn]
    if [catch {set res [eval exec $SVN status [list $homedir] -u]} err] {
	MyMessageBox -parent . -title "Error checking status" \
	    -message "Error checking for updates: $err" \
	    -icon error
	return
    } else {
	if {[string first "*" $res] == -1} {
	    MyMessageBox -parent . -title "No updates" \
		-message "CMPR appears up-to-date" \
		-icon info
	    return
	}
    }
    if {[MyMessageBox -parent . -title "Ready to Update" \
	     -message {
Updates to CMPR found.
		 
Press the "Update & Restart" button to begin the update process. After the update completes, CMPR will be restarted.  You will need to reread all files and any unsaved information will be lost. Press Cancel to upgrade some other time.
} \
	     -type {Cancel "Update & Restart"} -default cancel -icon warning
	] == "cancel"} {return}

     # special upgrade for windows, where the wish exec blocks upgrade of the exe directory
    if {$::tcl_platform(platform) == "windows" && $::tcl_platform(os) != "Windows 95"} {
	# use the batch file, create it if needed (older install)	
	set out [file normalize [file join $homedir StartCMPR.bat]]
	if {! [file exists $out]} {
	    # The batch file to start CMPR does not exist; no problem: created it.
	    set fp [open $out w]
	    puts $fp {@
@echo =======================================================
@echo                 This file starts CMPR
@echo = Run this from the directory where CMPR is installed =
@echo =======================================================
@
@REM Get this script's directory; make sure that the path ends 
@REM    with a single backslash
@set cmprloc=%~dp0\*
@set cmprloc=%cmprloc:\\*=\*%
@set cmprloc=%cmprloc:\*=\%
@
cmd.exe /c start "" "%cmprloc%exe\ncnrpack-win.exe" "%cmprloc%cmpr.tcl" %*
exit
	    }
	    close $fp
	}
	set out [file normalize [file join $homedir bootstrap.bat]]
	if {! [file exists $out]} {
	    # The batch file to upgrade CMPR does not exist; no problem: created it.
	    set fp [open $out w]
	    puts $fp {@echo ============================================================
@echo =               This file installs CMPR                    =
@echo = Run this from the directory where CMPR will be installed =
@echo ============================================================
@REM Get this script's directory
@set cmprloc=%~dp0
@REM make it the working directory
%~d0
cd %cmprloc%
@echo Preparing to load/update CMPR
@pause
@REM start 
@echo loading stable CMPR from repository
.\svn\bin\svn co https://subversion.xor.aps.anl.gov/CMPR/tags/stable/ .
@echo loading CMPR .exe files
.\svn\bin\svn switch https://subversion.xor.aps.anl.gov/CMPR/exe/win ./exe
@echo loading development CMPR 
.\svn\bin\svn switch https://subversion.xor.aps.anl.gov/CMPR/trunk/ ./cmprdev
@echo Files have been loaded. Press Return to start CMPR.
@pause
StartCMPR.bat
exit
}
	    close $fp
	}
	if {[file exists $out]} {
	    #run the batch file
	    exec $::env(COMSPEC) /c "start [file nativename [file attributes $out -shortname]]" &
	    exit
	} else {
	    MyMessageBox -parent . -title "Error in update" \
		-message "Error: the bootstrap.bat file was not found and was not created. Please reinstall CMPR from http://tinyurl.com/CMPRhome" \
		-icon warning
	    return
	}
    }
    # do a quiet cleanup. Sometimes needed after install, and never hurts
    if [catch {set res [eval exec $SVN cleanup [list $homedir]]} err] {
	MyMessageBox -parent . -title "Error in cleanup" \
	    -message "Error performing cleanup. Will try to continue anyway. Error:\n$err" \
	    -icon error
    }
    if [catch {set res [eval exec $SVN up [list $homedir]]} err] {
	MyMessageBox -parent . -title "Error updating" \
	    -message "Error performing update:\n$err" \
	    -icon error
	return
    } else {
	MyMessageBox -parent . -title "Updating done" \
	    -message "Results from update:\n$res\n\nPress OK to restart CMPR" \
	    -icon info 
    }
    exec [info nameofexecutable] [file normalize $::scriptname] &
    exit
}

#is there a svn directory in the source?
if {[file exists [file join $scriptdir .svn]]} {
    .a.help.menu add command -command CheckAndDoUpdate -label "Update CMPR"
    if {$::command(cmprversion) || \
	    [file exists [file join $scriptdir cmprdev]]} {
	.a.help.menu add cascade -menu .a.help.menu.track \
	    -label "Select CMPR version"
	menu .a.help.menu.track 
	.a.help.menu.track add radiobutton -command "SetCMPRversion" \
	    -label Standard    -value 0 -variable command(cmprversion)
	.a.help.menu.track add radiobutton -command "SetCMPRversion" \
	    -label Development -value 1 -variable command(cmprversion)
	trace variable command(cmprversion)  w CMPRversionSetMenu
	CMPRversionSetMenu
    } else {
	.a.help.menu add command -label "CMPR development version not installed" -state disabled
    }
} else {
    .a.help.menu add command -state disabled -label "Self-updating not installed"
}
.a.help.menu add command -command aboutcmpr -label "About CMPR"

wm deiconify .
wm deiconify .plot
grid $command(notebook) -column 0 -row 1 -sticky nsew
grid columnconfig . 0 -weight 1
grid rowconfig . 1 -weight 1
graphlegend
# set the page to the first command
DoCommand [lindex $menulist(pages) 0]
update
ResizeNotebook

wm protocol .plot  WM_DELETE_WINDOW {Setup Quit}
wm protocol .  WM_DELETE_WINDOW {Setup Quit}

catch {eval $command(finalcommand)}
# seems to be needed in OSX
update
wm geom . [winfo reqwidth .]x[winfo reqheight .]-0+32

	$graph(blt) configure -plotpadx 2 -plotpady 2
	$graph(blt) yaxis configure -subdivisions 5
	$graph(blt) xaxis configure -subdivisions 5