# this script is designed to be used with the liveplot tcl script in the 
# gsasgui package. It displays unit cells with or without extinctions
# top of the GSAS difference plot.
#
# Brian Toby (NIST) 3/2002
#
#

# list of files needed by CMPR
if {$tcl_platform(platform) == "windows"} {
    set RequiredFileList {data.tcl symsubs.tcl exe/hklgen.exe exe/testextn.exe calcpos.exe}
} else {
    set RequiredFileList {data.tcl symsubs.tcl exe/hklgen exe/testextn exe/calcpos}
}

# where are we reading this file from?
set scriptname [info script]
# translate links -- go six levels deep
foreach i {1 2 3 4 5 6} {
    if {[file type $scriptname] == "link"} {
	set link [file readlink $scriptname]
	if { [file  pathtype  $link] == "absolute" } {
	    set scriptname $link
	} {
	    set scriptname [file dirname $scriptname]/$link
	}
    } else {
	break
    }
}

set cmprloc [file dirname $scriptname]

set CMPR_OK 1
foreach file $RequiredFileList {
    if {![file exists [file join $cmprloc $file]]} {
	set CMPR_OK 0
	continue
    }
}

if {!$CMPR_OK} {
    catch {puts "CMPR files not found in $cmprloc"}
} else {
    source [file join $cmprloc data.tcl]
    # default list of colors
    set graph(colorlist) {black red green blue magenta cyan yellow4 \
	navy purple red4 darkolivegreen darkcyan royalblue4}
    definecolors
    source [file join $cmprloc symsubs.tcl]
    set graph(blt) $box
    # needed by initpeaks
    set graph(datalist) ""
    # set default values for lots of variables
    if {$tcl_platform(platform) == "windows"} {
	set cellparm(hklgenprog) [file join $cmprloc exe/hklgen.exe]
	set cellparm(tstextprog) [file join $cmprloc exe/testextn.exe]
	set cellparm(calc2tprog) [file join $cmprloc exe/calcpos.exe]
    } else {
	set cellparm(hklgenprog) [file join $cmprloc exe/hklgen]
	set cellparm(tstextprog) [file join $cmprloc exe/testextn]
	set cellparm(calc2tprog) [file join $cmprloc exe/calcpos]
    }
    set cellparm(spacegroup) (none)
    set cellparm(extspg) (none)
    set cellparm(zero) 0
    set cellparm(old_zero) 0
    set cellparm(old_ymin) 0
    set cellparm(old_ymax) 0
    set cellparm(oldextspg) {xxx} 
    set cellparm(sens) 25
    set cellparm(twothetamax) 168
    set cellparm(wavelength) "1.5418"
    set cellparm(old_wave) ""
    set cellparm(alpha) [set cellparm(beta) [set cellparm(gamma) 90]]
    set cellparm(a) [set cellparm(b) [set cellparm(c) 10]]
    set cellparm(laue) 8
    set cellparm(sens) 25
    set cellparm(ymin) 0
    set cellparm(ymax) 1000
    vector xextinct 
    vector yextinct 
    #----------------------------------------------------------------
    # final initializations
    #----------------------------------------------------------------
    # locations of the extinction condition sub-menus
    set cellparm(extinctmenuH) {}
    set cellparm(extinctmenuE) {}
    resetextinctions  
    trace var extinctlbl w extinctlblset 
}

proc cellgen {page} {
    global cellparm
    catch {deletecellgen}
    toplevel $page
    set cellparm(page) $page
    pack [frame $page.a] -side left -fill y -anchor n
    pack [frame $page.b] -side left -fill y -anchor n
    pack [frame $page.b.cell] -side top -fill y
    makesymbox $page.b.cell cellparm(celllaue) SetEditParms
    pack [frame $page.b.w] -side top
    pack [label $page.b.w.l -text Wavelength:] -side left
    pack [entry $page.b.w.e  -width 7] -side left
    pack [frame $page.b.s] -side top
    pack [label $page.b.s.l -text "Max change:"] -side left
    pack [entry $page.b.s.e -textvariable cellparm(sens) -width 3] \
	    -side left
    pack [label $page.b.s.l1 -text %] -side left
    
    pack [frame $page.c] -side left -fill y -anchor n
    set top $page.c.ext
    pack [frame $top ] -side top -fill both -anchor n
    set base [frame $page.c.1]
    set cellparm(celleditbase) $base
    pack $base -side top
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	label $base.l$var -text $var
	grid $base.l$var -in $base -row $i -column 1
	entry $base.e$var -width 8
	grid $base.e$var -in $base -row $i -column 2
	scale $base.s$var -showvalue 0 -orient h  -resolution 0.01
	grid $base.s$var -in $base -row $i -column 3
    }
    incr i
    label $base.lzero -text Zero
    grid $base.lzero -in $base -row $i -column 1
    entry $base.ezero -width 8 -textvariable cellparm(zero)
    grid $base.ezero -in $base -row $i -column 2
    scale $base.szero -showvalue 0 -orient h -resolution 0.01 \
	    -variable cellparm(zero) -from -0.5 -to 0.5
    grid $base.szero -in $base -row $i -column 3
    
    pack [frame $top.e -relief groove -bd 4] -side top -fill both -anchor n
    set cellparm(extinctmenuE) $top.e 
    label $top.e.l -text "Extinctions"
    grid $top.e.l -in $top.e -row 1 -column 1
    label $top.e.l1 -text ""
    grid $top.e.l1 -in $top.e -row 1 -column 2
    label $top.e.l2 -text "" -fg #888
    grid $top.e.l2 -in $top.e -row 1 -column 3
    label $top.e.el -text "Individual\nconditions"
    grid $top.e.el -in $top.e -row 2 -column 1
    button $top.e.ext -text "0 Set" -command "extinctmenu .extmen"
    grid $top.e.ext -in $top.e -row 2 -column 2
    label $top.e.extl -text "" -fg #888
    grid $top.e.extl -in $top.e -row 2 -column 3
    label $top.e.spgl -text "Space\nGroup:"
    grid  $top.e.spgl -in $top.e -row 3 -column 1
    entry $top.e.spge -textvariable cellparm(extspg) -width 8
    grid  $top.e.spge -in $top.e -row 3 -column 2
    label $top.e.spg3 -text "" -fg #888
    grid  $top.e.spg3 -in $top.e -row 3 -column 3
    grid rowconfigure $top.e 2 -weight 0 -minsize 42
    
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg #888
	$base.e$var config -fg #888 -state disabled
	$base.s$var config -fg #888 -state disabled -command {}
    }
    $page.b.w.l config -fg #888
    $page.b.w.e  config  -fg #888 -state disabled
    
    set cellparm(celleditvar) dgen1
    #    setsymbox $page.b.cell {}
    #    set cellparm(outcount) "0"
    editnew
}
    
proc deletecellgen {} {
    global graph cellparm
    #    bind $graph(blt)  <Shift-Button-1> {}
    catch {destroy $cellparm(page)}
    catch {$graph(blt) element delete extinct $cellparm(celleditvar)}
    catch {deletedata dgen1}
    after cancel RecalcLoop
    after cancel RecalcLoop
    set cellparm(celleditvar) {}
}

######################################################################
# set allowed cell parameters as called by setcellsym
# this is called when the laue symmetry is changed or by setcellsym
# globals: 
#    cellparm(celleditvar)   The variable name (dgen1 here) used for peaks
#    cellparm(celleditbase)  The location of the cell parameter widget
######################################################################
proc SetEditParms {fix lock} {
    global cellparm
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase)
    
    set i -1
    foreach var {a b c alpha beta gamma} {
	incr i
	# variables to fix
	if {[lindex $fix $i]-0 > 0} {
	    $base.l$var config -fg #808080
	    $base.e$var config -state disabled -fg #808080 \
		    -textvariable ${data}($var)
	    $base.s$var config -state disabled -fg #808080 \
		    -command {}
	    set ${data}($var)  [lindex $fix $i]
	} else {
	    $base.l$var config -fg black
	    $base.e$var config -state normal -fg black \
		    -textvariable ${data}($var)
	    set value [set ${data}($var)]
	    catch {
		$base.s$var config -state normal -fg black \
			-from [expr $value * (1-$cellparm(sens)/100.)]  \
			-to [expr $value * (1+$cellparm(sens)/100.)]
		$base.s$var set $value
		$base.s$var config -command "set ${data}($var)"
	    }
	}
	# variables to lock to others
	if {"[lindex $lock $i]" != ""} {
	    $base.l$var config -fg #888 
	    $base.e$var config -state disabled -fg #888 \
		    -textvariable ${data}([lindex $lock $i])
	    $base.s$var config -state disabled -fg #888 -command {}
	}
    }
}

proc editnew {} {
    #puts editnew
    global cellparm
    # need a variable name for SetEditParms
    set data dgen1
    set data [initpeaks $data]
    global $data
    set cellparm(celleditvar) $data
    set page $cellparm(page)
    catch {
	frame $page.b.6 
	pack [label $page.b.6.l -width 10 -anchor e -text "2theta max:"] \
		-padx 1m -pady 1m -side left
	pack [entry $page.b.6.e \
		-width 8 -relief sunken -bd 2 -textvariable cellparm(twothetamax)] \
		-padx 1m -pady 1m  -side left
    }
    pack $page.b.6 -side top -fill both -anchor center
    catch {
	button $page.b.5 -text Compute -command computecell
	pack $page.b.5 -side bottom
    }
    $page.b.w.l config -fg black
    $page.b.w.e config -textvariable cellparm(wavelength) -fg black \
	    -state normal
    resetextinctions  
    set base $cellparm(celleditbase) 
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg black
	$base.e$var config -textvariable newcell($var) \
		-fg black -state normal
#	set value [set ${data}($var)]
#	$base.s$var config -fg black -state normal \
#		-from [expr $value * (1-$cellparm(sens)/100.)]  \
#		-to [expr $value * (1+$cellparm(sens)/100.)]
#	$base.s$var set $value
	$base.s$var config 
    }
    # set cell choices to all
    setsymbox $page.b.cell 1
    if {$cellparm(celllaue) == {}} {set cellparm(celllaue) 8}
    setcellsym $cellparm(celllaue) SetEditParms
    $base.lzero config -fg #888
    $base.ezero config -fg #888 -state disabled
    $base.szero config -fg #888 -state disabled    
    # reset extinction codes
    $cellparm(extinctmenuE).l1 config -text "to remove"
    $cellparm(extinctmenuE).l2 config -text ""
    $cellparm(extinctmenuE).spg3 config -text ""
    $cellparm(extinctmenuE).extl config -text ""
}

proc computecell {} {
    global extinctlbl cellparm
    set data $cellparm(celleditvar)
    global $data
    set base $cellparm(celleditbase)
    set message {}

    foreach var {a b c alpha beta gamma} {
	set val [$base.e$var get]
	if {[catch {expr $val}]} {
	    append message "The value '$val' for cell parameter $var is invalid\n"
	} else {
	    append line [format %8.3f $val]
	    set ${data}($var) $val
	}
    }
    set val 
    if {[catch {expr $cellparm(wavelength)}]} {
	append message "The value '$cellparm(wavelength)' for wavelength is invalid\n"
    }
    if {[catch {expr $cellparm(twothetamax)*1}]} {
	append message "value '$cellparm(twothetamax)' for 2theta(max) is invalid\n"
    }
    if {$message != ""} {
	tk_messageBox -message $message -type ok -icon warning
	return
    }
    set out [open hklgen.inp w]
    puts $out "Created by hklgen.tcl"
    puts $out $line
    set line {}
    for {set num 1} {$num < 14} {incr num} {
	append line [format %3i [lsearch $extinctlbl($num) $extinctlbl(r$num)]]
    }
    set ${data}(extcodes) $line
    puts $out [format "%9.5f%9.2f%3d%s" \
	    $cellparm(wavelength) $cellparm(twothetamax) $cellparm(celllaue) $line]
    #puts [format "%9.5f%9.2f%3d%s" \
	    $cellparm(wavelength) $cellparm(twothetamax) $cellparm(celllaue) $line]
    if {$cellparm(extspg) == "(none)"} {
	puts $out ""
	set ${data}(spg) (none)
    } else {
	puts $out $cellparm(extspg)
	set ${data}(spg) $cellparm(extspg)
    }
    close $out
    catch {exec $cellparm(hklgenprog) < hklgen.inp > hklgen.out}
    set out [open hklgen.out r]
    set ${data}(laue) $cellparm(celllaue)
    set ${data}(wavelength) $cellparm(wavelength)
    while {[gets $out line] >= 0} {
	if {6 == [scan $line %*d%d%d%d%d%*s%*d%f%f h k l mult tt d]} {
	    lappend ${data}(h) $h
	    lappend ${data}(k) $k
	    lappend ${data}(l) $l
	    lappend ${data}(dspaces) $d
	    lappend ${data}(x) $tt
	    lappend ${data}(extinctions) 0
	    lappend ${data}(y) {}
	    lappend ${data}(widths) {}
	    lappend ${data}(etas) {}
	    lappend ${data}(heights) {}
	    lappend ${data}(bkgs) 0
	}
    }
    resetdata $data
    close $out
    file delete -force hklgen.inp hklgen.out
#    PostPageEditCell $cellparm(page)
    resetextinctions
    editcomputedcell
}

proc editcomputedcell {} {
    global cellparm
    set page $cellparm(page)
    destroy $page.b.6 $page.b.s.l $page.b.s.e $page.b.s.l1
    pack [label $page.b.s.l1 -text "Min:"] -side left
    pack [entry $page.b.s.e1 -textvariable cellparm(ymin) -width 4] \
	    -side left
    pack [label $page.b.s.l2 -text "Max:"] -side left
    pack [entry $page.b.s.e2 -textvariable cellparm(ymax) -width 6] \
	    -side left
    $page.b.5 config -text "Remove Cell" -command deletecellgen
    set base $cellparm(celleditbase) 
    $base.lzero config -fg black
    $base.ezero config -fg black -state normal
    $base.szero config -fg black -state normal
    $cellparm(extinctmenuE).l1 config -text "to label"
    $cellparm(extinctmenuE).l2 config -text "removed"
    set data $cellparm(celleditvar)
    global $data
    $cellparm(extinctmenuE).spg3 config -text "[set ${data}(spg)]"
    set count 0
    foreach num [set ${data}(extcodes)] {
	if {$num > 0} {incr count}
    }
    $cellparm(extinctmenuE).extl config -text "$count Set"
    # plot the results
    global graph
#    eval $graph(blt) element delete [$graph(blt) element names]
#    eval $graph(blt) marker delete  [$graph(blt) marker names]
    $graph(blt) element create extinct \
	    -color yellow -linewidth 2 \
	    -xdata xextinct -ydata yextinct -symbol none
#	    -activecolor red
    $graph(blt) element create $data \
	    -color cyan -linewidth [set ${data}(line)]  -symbol none \
	    -xdata [set ${data}(xvector)] -ydata [set ${data}(yvector)]
#	    -activecolor red
    xextinct notify now
    yextinct notify now
    [set ${data}(yvector)] notify now
    [set ${data}(xvector)] notify now
	 [set ${data}(yvector)] notify now
# setup constrain & constrain the sliders
    set base $cellparm(celleditbase) 
    set i 0
    foreach var {a b c alpha beta gamma} {
	incr i
	$base.l$var config -fg black
	$base.e$var config -textvariable ${data}($var) \
		-fg black -state normal
	set value [set ${data}($var)]
	catch {
	    $base.s$var config -fg black -state normal \
		    -from [expr $value * (1-$cellparm(sens)/100.)]  \
		    -to [expr $value * (1+$cellparm(sens)/100.)]
	    $base.s$var set $value
	    $base.s$var config -command "set ${data}($var) "
	}
	# -variable ${data}($var)
    }
    $page.b.w.l config -fg black
    $page.b.w.e config -textvariable ${data}(wavelength) -fg black \
	    -state normal
    set cellparm(celllaue) [set ${data}(laue)]
    setsymbox $page.b.cell $cellparm(celllaue)
    # restrict Symmetry changes to higher symmetry
    setcellsym $cellparm(celllaue) SetEditParms
    # Start the recomputation loop
    RecalcLoop
    #bind $graph(blt) <Shift-Button-1> "lblCellgen %W %x"
    if {[string first lblCellgen [bind . H]] == -1} {
	bind . <Key-h> "+lblCellgen $graph(blt) %x"
	bind . <Key-H> "+lblCellgen $graph(blt) %x"
    }
}

proc RecalcLoop {} {
    global command cellparm
    #    puts RecalcLoop
    # if {$command(pagenow) != "editcell"} return

    if {$cellparm(celleditvar) != ""} {catch {ttcalc $cellparm(celleditvar)}}
    # cancel any other RecalcLoop
    after cancel RecalcLoop
    after cancel RecalcLoop
    # requeue
    after 100 RecalcLoop
}

# compute two-theta values from hkl values etc.
proc ttcalc {data} {
    global $data cellparm graph
    set base $cellparm(celleditbase)
    set flag 1
    # reevaluate extinctions if symmetry has changed
    if {$cellparm(extspg) != $cellparm(oldextspg)} {
	calcextns $data $cellparm(extspg)
	set cellparm(oldextspg) $cellparm(extspg)
	set flag 0
    }
    # has cell changed? Is it valid?
    foreach var {a b c alpha beta gamma} {
	set value [$base.e$var get]
	set $var $value
	if {[set ${data}(old_$var)] != $value} {set flag 0}
	if [catch {expr $value} err] {return}
    }
    # has wavelength/units changed?
    if {$graph(xunits) != 0 && $graph(xunits) != $cellparm(old_wave)} {
	set flag 0
    } elseif {$graph(xunits) == 0 && \
	    [set ${data}(wavelength)] != $cellparm(old_wave)} {
	set flag 0
    }
    if {$cellparm(ymin) != $cellparm(old_ymin)} {
	set cellparm(old_ymin) $cellparm(ymin)
	set flag 0
    }
    if {$cellparm(ymax) != $cellparm(old_ymax)} {
	set cellparm(old_ymax) $cellparm(ymax)
	set flag 0
    }
    #-----------------------------
    # no reflections, don't bother
    #-----------------------------
    if {[llength [set ${data}(h)]] == 0} return
    #---------------------------
    # no changes -- don't bother
    #---------------------------
    if $flag {update; return}
    #---------------------------
    # validate wavelength
    #---------------------------
    if {$graph(xunits) == 0} {
	if [catch {expr [set ${data}(wavelength)]}] {return}
	set cellparm(old_wave) [set ${data}(wavelength)]
    } else {
	set cellparm(old_wave) $graph(xunits)
    }	
    if [catch {expr $cellparm(ymin)}] {return}
    if [catch {expr $cellparm(ymax)}] {return}
    #---------------------------
    # update the positions
    #---------------------------
    if {$cellparm(zero) != $cellparm(old_zero) } {set flag 0}
    if $flag {update; return}
    global ${data}
    [set ${data}(xvector)] notify cancel 
    [set ${data}(yvector)] notify cancel 
    set fil [open calc.inp w]
    puts $fil $data
    foreach var {a b c alpha beta gamma} {
	puts -nonewline $fil " [set $var]"
	set ${data}(old_$var) [set $var]
    }
    set cellparm(old_zero) $cellparm(zero)
    puts $fil {}
    puts $fil [set ${data}(wavelength)]
    set numdat [llength [set ${data}(dspaces)]]
    # get the limits to draw on the y-axes
    puts $fil "$cellparm(ymin) $cellparm(ymax) $cellparm(zero)"
    puts $fil $graph(xunits)
    puts $fil $numdat
    puts $fil [set ${data}(h)]
    puts $fil [set ${data}(k)]
    puts $fil [set ${data}(l)]
    puts $fil [set ${data}(extinctions)]
    close $fil
    catch {exec $cellparm(calc2tprog) < calc.inp > calc.out}
    set fil [open calc.out r]
    set out [read $fil]
    close $fil
    file delete -force calc.inp calc.out

    [set ${data}(xvector)] notify never
    [set ${data}(yvector)] notify never
    xextinct notify never
    yextinct notify never
    xextinct set {}
    yextinct set {}

    eval $out

    # (ttvector) is used for reflection labeling (lblCellgen)
    [set ${data}(ttvector)] set [set ${data}(x)]

    [set ${data}(xvector)] notify now
    [set ${data}(yvector)] notify now
    xextinct notify now
    yextinct notify now
    [set ${data}(xvector)] notify always
    [set ${data}(yvector)] notify always
    update
}

proc calcextns {data spacegroup} {
    global $data cellparm extinctlbl
    global ${data}
    set fil [open tstext.inp w]
    puts $fil $data
    set line {}
    # manual extinction rules
    for {set num 1} {$num < 14} {incr num} {
	append line [format %3i [lsearch $extinctlbl($num) $extinctlbl(r$num)]]
    }
    puts $fil $line
    puts $fil $cellparm(extspg)
    set numdat [llength [set ${data}(dspaces)]]
    puts $fil $numdat
    puts $fil [set ${data}(h)]
    puts $fil [set ${data}(k)]
    puts $fil [set ${data}(l)]
    close $fil
    catch {exec $cellparm(tstextprog) < tstext.inp > tstext.out}
    set fil [open tstext.out r]
    set out [read $fil]
    close $fil
    file delete -force tstext.inp tstext.out
    eval $out
}

proc lblCellgen {plot x} {
    global cellparm command blt_version tcl_platform tcl_version
    global expgui

    # if the cellgen window does not exist there is nothing to do
    if {![winfo exists $cellparm(page)]} {return}
    set data {}
    catch {set data $cellparm(celleditvar)}
    if {$data == ""} return
    global $data

    # look for peaks within pixelregion pixels
    set xmin [$plot xaxis invtransform [expr $x - $expgui(pixelregion)]]
    set xmax [$plot xaxis invtransform [expr $x + $expgui(pixelregion)]]
    set xnam [set ${data}(ttvector)]
    set peaknums [$xnam search $xmin $xmax]
    set peaklist {}
    set ymax [lindex [$plot yaxis limits] 1]
    set xcen 0    
    foreach peak $peaknums {
	set big 0
	foreach i {h k l} {
	    set $i [lindex [set ${data}($i)] $peak]
	    if {[set $i] < -9 || [set $i] > 9} {set big 1}
	}
	set xpos [$xnam range $peak $peak]
	set xcen [expr $xcen + $xpos]
	set sym {}
	if [lindex [set ${data}(extinctions)] $peak] {set sym e}
	if $big {
	    set reflbl $h,$k,$l$sym
	    lappend peaklist $h,$k,$l$sym
	} else {
	    set reflbl $h$k$l$sym
	    lappend peaklist $h$k$l$sym
	}
	if {$expgui(hklbox)} {
	    catch {
		.hkl.txt insert end "\ndgen"
		.hkl.txt insert end "\t$reflbl"
		.hkl.txt insert end "\t$xpos"
		.hkl.txt see end
	    }
	}
    }
    if {$peaklist == ""} return
    set xcen [expr $xcen / [llength $peaknums]]
    # avoid bug in BLT 2.3 where Inf does not work for text markers
    if {$blt_version == 2.3} {
	set ycen [lindex [$plot yaxis limits] 1]
    } else  {
	set ycen Inf
    }
    if {$tcl_platform(platform) == "windows"} {
	# at least right now, text can't be rotated in windows
	regsub -all { } $peaklist "\n" peaklist
	set mark [$plot marker create text -coords "$xcen $ycen" \
	-text $peaklist -anchor n -bg "" -name dgen$xcen]
    } else {
	set mark [$plot marker create text -coords "$xcen $ycen" \
	-rotate 90 -text $peaklist -anchor n -bg "" -name dgen$xcen]
    }
    if {$tcl_version >= 8.0} {
	$plot marker config dgen$xcen -font lblfont
    }
    if {$expgui(fadetime) > 0} {
	catch {
	    after [expr $expgui(fadetime) * 1000 ] \
	    "catch \{ $plot marker delete $mark \}"
	}
    }
}
