static char *SCCS_ID = "@(#) test-hdl.cpp Version 1.1 Created 08/23/03 at 10:13:51";

/******************************************************************************
The software provided here is released by the National
Institute of Standards and Technology (NIST), an agency of
the U.S. Department of Commerce, Gaithersburg MD 20899,
USA.  The software bears no warranty, either expressed or
implied. NIST does not assume legal liability nor
responsibility for a User's use of the software or the
results of such use.

Please note that within the United States, copyright
protection, under Section 105 of the United States Code,
Title 17, is not available for any work of the United
States Government and/or for any works created by United
States Government employees. User acknowledges that this
software contains work which was created by NIST employees
and is therefore in the public domain and not subject to
copyright.  The User may use, distribute, or incorporate
this software provided the User acknowledges this via an
explicit acknowledgment of NIST-related contributions to
the User's work. User also agrees to acknowledge, via an
explicit acknowledgment, that any modifications or
alterations have been made to this software before
redistribution.
******************************************************************************/

/***** Author: Dr. James R. Lyle, NIST/SDCT/SQG ****/


/******************************************************************************
*
*  Test-hdl works with the TALLY13 TSR program to test int 13 based hard drive
*  software write block tools. The TALLY13 program intercepts all commands passed
*  by a SWB tool under test and keeps a count of the number of times each command
*  is sent to each hard drive. The TALLY13 program can also be quered about the
*  current count of each command sent to each hard drive. The query interface is
*  implemented by using int 17h (normally the printer).
*  
*  The logical design of TEST-HDL is as follows:
*  
*  	check command line parameters
*  		test case id
*  		host running the test
*  		user running the test
*  		command set under test 
*  			valid values:
*  			r read
*  			w write
*  			i information
*  			c control device
*  			x configure device
*  			m misc undefined
*  		list of hard drives installed
*  	If a problem is found, print a message and exit
*  	Open a log file (SWB_LOG.TXT) and record admin details
*  	Activate TALLY13, exit if not TALLY13 not running
*  	Check that all tallies are zero, exit if not. (call check_tallies)
*  	Run the tests (call exec_test)
*  	Check that the number of passed commands is consistent (call recheck_tallies)
*  	Exit.
*  
******************************************************************************/

# include <stdio.h>
# include <stdlib.h>
# include <time.h>
# include "wb-defs.h"

void check_tallies (FILE *log, int n_drives)
{
	int	i = 0 /* command code */,
		ans   /* Return tally here */,
		drive /* Hard drive to check */,
		fun   /* The command code, shifted to the correct byte */,
		total;/* total of the tallies for each drive */
	
	/* for each drive, for each command, get current tally. It MBZ  */
	for (drive = 0x80; drive < (0x80 + n_drives); drive ++){
		total = 0; /* reset for current drive */
		for (i = 0; i < 256; i++){ /* for each command code */
				fun = i << 8; /* shift to other byte */
			
				asm {
					push si
					mov ax,fun
					mov dx,drive
					int 0x17
					mov ans,cx
					pop si
				}
				total += ans; /* keep total of tallies */
		}
		if (total) /* If not zero there may be a problem. Forgot to reboot? */
		fprintf (log,"Warning: non-zero tally (%d) for drive %02X (reboot to clear)\n",
			total,drive);

	}
	return;

}

/******************************************************************************
*	Test each command in list for case_id. Command names are in names.
*	Log the results to log. There are n_drives starting at 0x80
*	case_id identifes the test case
* 	list is the set of commands to test
*	names are the names of the corresponding commands from list, however if
*		the value is NULL, the command is considered undefined
*	log is the log file
*	n_drives is the number of drives to test. There could actually be more or
*		fewer drives installed
*
*	return the sum of all command tallies.
******************************************************************************/
int exec_test(char *case_id, char cat, int *list, char *names[], FILE *log, int n_drives)
{
	int	i = 0			/* index into list of commands */,
			carry_set,	/* flag indicating the carry bit is set by cmd */
			error_code,	/* error/status return code */
			ans,			/* count of cmd seen by tally13 monitor */
			drive,		/* hard drive starting at 0x80 */
			fun,			/* cmd code being tested */
			sent = 0, 	/* number of commands sent */
			total_sent = 0,
			blocked = 0,/* number of commands blocked */
			total_blocked = 0,
			expected = 0; /* number of commands blocked */
	  drive = 0x80; ans = 33;

						 /* for each drive, for each command in list */
	for (drive = 0x80; drive < (0x80 + n_drives); drive ++){
		i = 0;
		sent = 0;
		blocked = 0;
	while (list[i] != -1){
			sent++; /* count the number of commands sent */
			fun = list[i] << 8; /* put the command code in upper byte */
			carry_set = 0;
			error_code = 0;

			/* send a command to the int 13 interface and return the status in
				error_code */
			asm {
				push	si
				mov	ax,fun
				mov dx,drive
				int	0x13
				jnc	popok
				mov	error_code,ax
				pop 	si
				jmp	done
				popok:
            pop	si
				jmp	ok
				done:
			}

			carry_set = 1;
		ok:
			/* get the tally value for the command just sent. If ans is zero, the
				command was blocked. If ans is 1, the command was passed by
				the SWB tool */

				asm {
					push si
					mov ax,fun
					mov dx,drive
					int 0x17
					mov ans,cx
					pop si
				}
	
      
		fprintf (log,"%3d %-6s ",i,case_id);
		fprintf (log,"<%02X> %02X ",list[i],drive);
		if (ans) fprintf (log,"Allowed ");
		else {fprintf (log,"Blocked "); blocked++;}
		if (carry_set) fprintf (log,"%04X ",error_code);
		else fprintf (log,"0000 ");
		if (carry_set) fprintf (log,"On  ");
		else fprintf (log,          "Off ");
		fprintf (log,"%5d ",ans);
		if (names)fprintf (log," %s",names[i]);
		else fprintf (log,"Undefined");
		fprintf (log,"\n");
		expected += ans;
		i++; /* advance to next command */
	}
	total_sent += sent;
	total_blocked += blocked;
	fprintf (log,"Results for %s category %c on drive %02X ",case_id,cat,drive);
	if (sent == blocked) fprintf (log,"All commands blocked");
	else if (blocked == 0) fprintf (log,"No commands blocked");
	else fprintf (log,"Not all commands blocked");
	fprintf (log," (%d of %d)\n",blocked,sent);
	}
	
	fprintf (log,"Summary: %d sent, %d blocked, %d not blocked\n\n",
			total_sent,total_blocked,total_sent-total_blocked);
	return expected;
}
/******************************************************************************
Tally all the commands. This value should be the same as the value returned
from exec_test (tally from the commands executed). If the values are not the
same then some how a command got sent from the tool under test that was not
sent by the test harness! IE the tool under test changed a command into
something else.
******************************************************************************/
void recheck_tallies (FILE *log, int n_drives,int ev)
{
	int	i = 0,ans,drive,fun,total;
	  drive = 0x80; ans = 33;

	fprintf (log,"\nNumber of Commands not blocked (should total to %d)\nDrive  Count\n",ev);
	for (drive = 0x80; drive < (0x80 + n_drives); drive ++){
		total = 0;
		for (i = 0; i < 256; i++){

				fun = i << 8;
				asm {
					push si
					mov ax,fun
					mov dx,drive
					int 0x17
					mov ans,cx
					pop si
				}
				total += ans;
		}
		fprintf (log,"  %02X   %5d\n",drive,total);
	}
}
/******************************************************************************

The main routine
	The commands in each category are defined in wb-defs.h
******************************************************************************/

int main (int np, char **p)
{

	int		i,			/* loop index */
				set_index, /* indicates which command set to test */
				ex,		/* number of commands passed */
				do_all = 6; /* indicates do all possible commands */
	unsigned int		is_active; /* indicates that tally13 TSR active */
   /* definitions of command sets (categories) */
	int	*sets[] = {c_funs,i_funs,r_funs,w_funs,x_funs,m_funs,NULL};
	char	**names [] = {c_names,i_names,r_names,w_names,x_names,NULL};
	char set_codes[] = "cirwxm ";
	char	*set_names[] = {"Control","Information","Read","Write","Configure",
									"Misc","All"};
	FILE	*wb_log = stdout;
	static time_t from; /* program start time */

	time(&from);

	if (np < 4) {
		printf ("Usage: %s case host operator test-set drive-list\n",p[0]);
		return 0;
	}

	/* identify command category to test */
	i = 0;
	while ((p[4][0] != set_codes[i]) && (set_codes[i] != ' ')) i++;
	if ((set_codes[i] == ' ') && (p[4][0] != 'a')){ /* a => do all sets */
		printf ("Set code (%c) is not valid [rwxcima]\n",p[4][0]);
		return 1;
	}
	else set_index = i;

	wb_log = fopen("A:\SWB-LOG.TXT","w");
	if (wb_log == NULL){
		printf ("%s: could not open log file\n");
		return 1;
	}
	fprintf (wb_log,"CMD: %s",p[0]);
	for (i = 1; i < np; i++) fprintf (wb_log," %s",p[i]);
	fprintf (wb_log,"\n");

	fprintf (wb_log,"Case: %s\n",p[1]);
	fprintf (wb_log,"Command set: %s\n",set_names[set_index]);
	fprintf (wb_log,"Date: %s\n",ctime(&from));
	fprintf (wb_log,"Version: %s\n\t%s\n\tCompiled on %s at %s\n",
		SCCS_ID,WB_DEFS_VERSION,__DATE__,__TIME__);
	fprintf (wb_log,"Operator: %s\n",p[3]);
	fprintf (wb_log,"Host: %s\n",p[2]);
	fprintf (wb_log,"Number of drives %d, ",np - 5);
	fprintf (wb_log,"Drives:");
	for (i = 5; i < np; i++) fprintf (wb_log," %s",p[i]);
	if (np <= 5) fprintf (wb_log," none\n");
	else fprintf (wb_log,"\n");

	/* verify that tally13 is running and activate */
	asm {
		mov	dl,1
		int	0x17
		mov	is_active,ax
	}
	if (is_active != 0xCCFF){
		printf ("Test Harness is not active\n");
		return 1;
	}

	/* verify that all command tallies are zero, If not reboot */
	check_tallies (wb_log,np - 5);

	fprintf (wb_log,"     Case  Cmd Drv Action  Stat Cry Count Cmd Name\n");

	if (set_index == do_all){ /* test all commands */
		ex = 0;
		for (i = 0; i < 6; i++){
			
			ex += exec_test (p[1],set_codes[i],sets[i],names[i],wb_log,np - 5);

		}
	}
	/* test the selected category [set_index] of commands */
	else ex = exec_test (p[1],p[4][0],sets[set_index],names[set_index],wb_log,np - 5);
	recheck_tallies (wb_log,np - 5,ex);


	return 0;
}
