// MainFrm.cpp : implementation of the CMainFrame class
//

#include "stdafx.h"
#include "dbt.h"
#include "Writeblocker.h"
#include "WriteblockerDoc.h"
#include "StorageDevices.h"
#include "NetworkDrives.h"
#include "DriversInterface.h"
#include "Install.h"
#include "Uninstall.h"
#include "MainFrm.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern CStorageDevices LocalDevices;
extern CNetworkDrives NetworkDrives;
extern CDriversInterface DriversInterface;

/////////////////////////////////////////////////////////////////////////////
// CMainFrame

IMPLEMENT_DYNCREATE(CMainFrame, CFrameWnd)

BEGIN_MESSAGE_MAP(CMainFrame, CFrameWnd)
	//{{AFX_MSG_MAP(CMainFrame)
	ON_WM_CREATE()
	ON_WM_CLOSE()
	ON_WM_TIMER()
	ON_COMMAND(ID_TOOLS_REINSTALL, OnToolsReinstall)
	ON_COMMAND(ID_TOOLS_UNINSTALL, OnToolsUninstall)
	ON_UPDATE_COMMAND_UI(ID_TOOLS_UNINSTALL, OnUpdateToolsUninstall)
	ON_COMMAND(IDM_TOGGLE_WRITEBLOCK, OnToggleWriteblock)
	ON_UPDATE_COMMAND_UI(IDM_TOGGLE_WRITEBLOCK, OnUpdateToggleWriteblock)
	ON_COMMAND(ID_FILE_CLEAR, OnFileClear)
	ON_UPDATE_COMMAND_UI(ID_FILE_CLEAR, OnUpdateFileClear)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdateFileSave)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE_AS, OnUpdateFileSaveAs)
	ON_WM_DEVICECHANGE()
	//}}AFX_MSG_MAP

END_MESSAGE_MAP()

static UINT indicators[] =
{
	ID_SEPARATOR,           // status line indicator
	ID_INDICATOR_CAPS,
	ID_INDICATOR_NUM,
	ID_INDICATOR_SCRL,
};

/////////////////////////////////////////////////////////////////////////////
// CMainFrame construction/destruction

CMainFrame::CMainFrame()
{
	m_pTopView = NULL;
	m_pEventLogView = NULL;
	m_uiTimerID = 0;
	m_bDriversInstalled = FALSE;
	m_bEnableToggleBtn = FALSE;

	// initialize critical section
	InitializeCriticalSection(&CritSect);
}

CMainFrame::~CMainFrame()
{
	DeleteCriticalSection(&CritSect);
}

int CMainFrame::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CFrameWnd::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	if (!m_wndToolBar.CreateEx(this, TBSTYLE_FLAT, WS_CHILD | WS_VISIBLE | CBRS_TOP
		| CBRS_GRIPPER | CBRS_TOOLTIPS | CBRS_FLYBY | CBRS_SIZE_DYNAMIC) ||
		!m_wndToolBar.LoadToolBar(IDR_MAINFRAME))
	{
		TRACE0("Failed to create toolbar\n");
		return -1;      // fail to create
	}

	if (!m_wndStatusBar.Create(this) ||
		!m_wndStatusBar.SetIndicators(indicators,
		  sizeof(indicators)/sizeof(UINT)))
	{
		TRACE0("Failed to create status bar\n");
		return -1;      // fail to create
	}

	m_wndToolBar.EnableDocking(CBRS_ALIGN_ANY);
	EnableDocking(CBRS_ALIGN_ANY);
	DockControlBar(&m_wndToolBar);

	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);

	SetIcon(m_hIcon, TRUE);
	SetIcon(m_hIcon, FALSE);

	// initialize driver interface pointer to main frame
	DriversInterface.SetMainFrmPtr(this);
	return 0;
}

BOOL CMainFrame::PreCreateWindow(CREATESTRUCT& cs)
{
	if( !CFrameWnd::PreCreateWindow(cs) )
		return FALSE;
	// TODO: Modify the Window class or styles here by modifying
	//  the CREATESTRUCT cs

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CMainFrame diagnostics

#ifdef _DEBUG
void CMainFrame::AssertValid() const
{
	CFrameWnd::AssertValid();
}

void CMainFrame::Dump(CDumpContext& dc) const
{
	CFrameWnd::Dump(dc);
}

#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CMainFrame message handlers


BOOL CMainFrame::OnCreateClient(LPCREATESTRUCT lpcs, CCreateContext* pContext) 
{
	CMenu* pMenu;

	// open interface to NTWBPM driver
	DriversInterface.OpenNTWBPM();

	// opens the drivers and verify version numbers
	DRIVERSSTATUS lStatus = DriversInterface.Initialize();

	// Enumerate the storage devices that are currently present.  This can be
	// done even if the Write Blocker drivers have not been started -- we just
	// won't be able to get the device number and write block status of each
	// device.
	LocalDevices.EnumDevices();

	// if drivers haven't been installed, notify user to install drivers
	if (lStatus == DRIVERS_NOT_INSTALLED) {
		MessageBox("The Drivers for Windows 2000 Write Blocker were not installed.\n\nPlease install the drivers via the Tools menu.",
			"Windows 2000 Write Blocker Error!", MB_ICONERROR | MB_OK);
	}
	// if drivers were installed but not started, notify user of error
	else if (lStatus == DRIVERS_INSTALLED) {
		MessageBox("The Drivers for Windows 2000 Write Blocker were not started.\n Please check the Windows 2000 Administration Event Log for insight to this problem.",
			"Windows 2000 Write Blocker Error!", MB_ICONERROR | MB_OK);
	}
	// else drivers started successfully so create views
	else {
		// Enumerate the network drives that have been mapped to a drive
		// letter.  Unlike LocalDevices, this will only work if the Write
		// Blocker drivers have been started.
		NetworkDrives.EnumNetworkDrives();

		// create splitter window with two rows
		if (!m_Splitter.CreateStatic(this, 2, 1)) {
			TRACE0("CMainFrame: Failed to create splitter window!\n");
			return FALSE;
		}

		// create top view
		if (!m_Splitter.CreateView(0, 0, RUNTIME_CLASS(CTopView), CSize(500, 500),
			pContext)) {
			TRACE0("CMainFrame: Failed to create top view!\n");
			return FALSE;
		}

		// create event log view
		if (!m_Splitter.CreateView(1, 0, RUNTIME_CLASS(CEventLogView), 
			CSize(150, 150), pContext)) {
			TRACE0("CMainFrame: Failed to create event log view!\n");
			return FALSE;
		}

		// store pointers to the views
		m_pTopView = (CTopView*) m_Splitter.GetPane(0, 0);
		m_pEventLogView = (CEventLogView*) m_Splitter.GetPane(1, 0);

		// set flag indicating that drivers have been installed - this flag is
		// used to enable/disable the appropriate menu items
		m_bDriversInstalled = TRUE;

		// set timer to poll NTWBPM every 3 seconds to determine if a device has 
		// just been added or removed
		m_uiTimerID = SetTimer(1, 3000, NULL);
	}

	// enable or disable "Clear Event Log" and "Uninstall Write Blocker" menu 
	// items as appropriate
	pMenu = GetMenu();
	if (pMenu) {
		pMenu->EnableMenuItem(ID_TOOLS_CLEAREVENTLOG, m_bDriversInstalled);
		pMenu->EnableMenuItem(ID_TOOLS_UNINSTALL, m_bDriversInstalled);
	}

	return TRUE;
}

void CMainFrame::OnClose() 
{
	// kill the timer used to detect device arrivals/removals
	if (m_uiTimerID != 0)
		KillTimer(m_uiTimerID);

	DriversInterface.UnInitialize();

	// close the interface to NTWBPM
	DriversInterface.CloseNTWBPM();

	// close the drivers
	DriversInterface.CloseDriver( );

	CFrameWnd::OnClose();
}

BOOL CMainFrame::OnDeviceChange(UINT nEventType, DWORD dwData)
{
	// ************************************************************************
	// For network drives, we want to process volume arrivals and removals.
	// For physical media, we only want to process volume removal events.  We 
	// need to process volume removals for physical media using the 
	// ON_WM_DEVICE_CHANGE event rather than querying NTWBPM for volume 
	// removals because NTWBPM does not keep track of drive letters associated 
	// with a physical device.  Conversely, the ON_WM_DEVICE_CHANGE event knows 
	// which drive letters were removed but it does not know which physical 
	// device was removed.  This is OK though since we are only interested in 
	// updating the GUI with the drive letters that were just removed.
	// ************************************************************************
	if ((nEventType == DBT_DEVICEARRIVAL) || 
		(nEventType == DBT_DEVICEREMOVECOMPLETE)) {

		PDEV_BROADCAST_HDR pHdr = (PDEV_BROADCAST_HDR) dwData;

		// only process volume change events
		if (pHdr && pHdr->dbch_devicetype == DBT_DEVTYP_VOLUME) {
			PDEV_BROADCAST_VOLUME pVolumeHdr = (PDEV_BROADCAST_VOLUME) dwData;
			STORAGE_DEVICE StorageDevice;
			MEDIA_EVENT MediaEvent;
			DWORD dwDrivesBitMask;

			// get pointer to view's document
			CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pTopView->GetDocument();

			// store bit mask of drive letters that were added/removed
			dwDrivesBitMask = pVolumeHdr->dbcv_unitmask;

			// step through each drive letter in the drives bit mask
			for (char c='A'; c <= 'Z'; c++) {

				// if the drive letter is set
				if (dwDrivesBitMask & 0x1) {

					// if network drive, update network drives list
					if (pVolumeHdr->dbcv_flags == DBTF_NET) {

						// rebuild network drives list
						NetworkDrives.EnumNetworkDrives();

						if (nEventType == DBT_DEVICEARRIVAL)
							MediaEvent.dwEventType = EVENT_NETWORK_DRIVE_ARRIVAL;
						else
							MediaEvent.dwEventType = EVENT_NETWORK_DRIVE_REMOVAL;

						MediaEvent.Media.cNetworkDrive = c;

						// send event notification to all views
						pDoc->AddMediaEvent(MediaEvent);
						pDoc->UpdateAllViews(NULL);
					}
					// else drive letter of a physical device...
					else {

						// for physical devices, only process device removal
						// events
						if (nEventType == DBT_DEVICEREMOVECOMPLETE) {

							// find the device that contains the specified drive letter
							// and remove the drive letter from the device's drive
							// letters list
							if (LocalDevices.GetStorageDeviceContainingDriveLetter(c,
								&StorageDevice)) {

								LocalDevices.RemoveDriveLetterFromStorageDevice(StorageDevice.dnDevInst,
									c);

								MediaEvent.dwEventType = EVENT_PHYSICAL_VOLUME_REMOVAL;
								MediaEvent.Media.dnDevInst = StorageDevice.dnDevInst;

								// send event notification to all views
								pDoc->AddMediaEvent(MediaEvent);
								pDoc->UpdateAllViews(NULL);
							}
						}
					}
				}

				dwDrivesBitMask >>= 1;	// go to next drive letter
			}
		}
	}

	return CWnd::OnDeviceChange(nEventType, dwData);
}

void CMainFrame::OnTimer(UINT nIDEvent) 
{
	// process recent write attempts
	ProcessWriteAttempts();

	// 
	// More than one thread calling ProcessMediaEvents(),
	// protect access with critical section.
	//

	__try {
		// request ownership
		EnterCriticalSection(&CritSect);
		// process device arrivals and removals
		ProcessMediaEvents();
	}
	__finally 
	{
		// release critical section
		LeaveCriticalSection(&CritSect);
	}

	CFrameWnd::OnTimer(nIDEvent);
}

void CMainFrame::ProcessMediaEvents()
{
	NTWBPMMEDIAEVENTSTRUCT MediaEvent[MAX_MEDIA_EVENTS];
	BOOL bSuccess;

	// ************************************************************************
	// Query NTWBPM for info on devices that were added.  We need to do this
	// rather than use the ON_WM_DEVICE_CHANGE event because during device 
	// arrival events, if a device has more than one partition on it, only the
	// first partition is returned in the drive letter bitmask of the
	// DEV_BROADCAST_VOLUME structure.  Also, if a device does not have any
	// drive letters assigned to it, we will never receive a device arrival
	// notification in the ON_WM_DEVICE_CHANGE event.  However, we want to list
	// these drives too.
	// ************************************************************************
	if (DriversInterface.GetPnpMediaEvents(MediaEvent)) {
		MEDIA_EVENT Event;
		DEVINST dnDevInst;

		// get pointer to view's document
		CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pTopView->GetDocument();

		// loop through all media events
		for (int x=0; x < MAX_MEDIA_EVENTS; x++) {
			bSuccess = FALSE;

			// process event
			switch (MediaEvent[x].Event) {
			case EventDeviceArrival:
				bSuccess = AddPhysicalDevice(MediaEvent[x], &dnDevInst);
				Event.dwEventType = EVENT_PHYSICAL_DEVICE_ARRIVAL;
				break;
			case EventDeviceRemoval:
				bSuccess = RemovePhysicalDevice(MediaEvent[x], &dnDevInst);
				Event.dwEventType = EVENT_PHYSICAL_DEVICE_REMOVAL;
				break;
			case EventVolumeArrival:
				bSuccess = AddPhysicalVolume(MediaEvent[x], &dnDevInst);
				Event.dwEventType = EVENT_PHYSICAL_VOLUME_ARRIVAL;
				break;
			// EventVolumeRemoval case handled in OnDeviceChange event handler
			//case EventVolumeRemoval:
			//	break;
			}

			// if event processed successfully, send notification to all views
			if (bSuccess) {
				Event.Media.dnDevInst = dnDevInst;

				pDoc->AddMediaEvent(Event);
				pDoc->UpdateAllViews(NULL);
			}
		}
	}
}

void CMainFrame::ProcessWriteAttempts()
{
	// make sure Write Blocker drivers are installed
	if (DriversInterface.IsInstalled()) {
		NTWBFSATTEMPTSTRUCT AttemptData;

		// query drivers for latest write attempts
		if (DRIVERSSTATUS_SUCCESS(DriversInterface.GetEntry(&AttemptData))) {
			MEDIA_EVENT MediaEvent;

			// get pointer to view's document
			CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pTopView->GetDocument();

			// process all write attempts
			for (WORD i=0; i < AttemptData.wEntries; i++) {

				//****************************************************************
				// The AttemptData Structure has fields for write attempts to
				// Network Drives and Physical Media.  The following few lines of
				// code figures out what kind of media was blocked
				//****************************************************************
				if (AttemptData.Data[i].wDrvType == NTWBFS_DRVTYPE_NETWORK) {

					MediaEvent.dwEventType = EVENT_NETWORK_WRITE_ATTEMPT;
					strcpy(MediaEvent.Media.szNetworkPath, AttemptData.Data[i].UNCString);
				}
				else {
					MediaEvent.dwEventType = EVENT_PHYSICAL_WRITE_ATTEMPT;

					// if this is really a drive letter, figure out which
					// device this corresponds to
					if (AttemptData.Data[i].wDrvPart >= 65) {
						STORAGE_DEVICE StorageDevice;

						// if device found, get device number and type
						if (LocalDevices.GetStorageDeviceContainingDriveLetter((char) AttemptData.Data[i].wDrvPart,
							&StorageDevice)) {

							switch (StorageDevice.eType) {
							case FLOPPY_DISK:
								AttemptData.Data[i].wDrvType = NTWBFS_DRVTYPE_FLOPPY;
								break;
							case CDROM_DISK:
								AttemptData.Data[i].wDrvType = NTWBFS_DRVTYPE_CDROM;
								break;
							case IDE_DISK:
							case SCSI_DISK:
							case USB_DISK:
								AttemptData.Data[i].wDrvType = NTWBFS_DRVTYPE_HD;
								break;
							default:
								AttemptData.Data[i].wDrvType = NTWBFS_DRVTYPE_NONE;
							}

							AttemptData.Data[i].wDrvNum = StorageDevice.wDeviceNum;
						}					
					}

					// create string identifying device that this write attempt
					// corresponds to
					switch (AttemptData.Data[i].wDrvType) {
					case NTWBFS_DRVTYPE_FLOPPY:
						sprintf(MediaEvent.Media.szPhysicalMedia, "%s%u", 
							MEDIA_NAME_FLOPPY_DRIVE, AttemptData.Data[i].wDrvNum);
						break;
					case NTWBFS_DRVTYPE_CDROM:
						sprintf(MediaEvent.Media.szPhysicalMedia, "%s%u",
							MEDIA_NAME_CDROM_DRIVE, AttemptData.Data[i].wDrvNum);
						break;
					case NTWBFS_DRVTYPE_HD:
						sprintf(MediaEvent.Media.szPhysicalMedia, "%s%u",
							MEDIA_NAME_HARD_DRIVE, AttemptData.Data[i].wDrvNum);
						break;
					default:
						sprintf(MediaEvent.Media.szPhysicalMedia, "%s%u",
							MEDIA_NAME_UNKNOWN_DRIVE, AttemptData.Data[i].wDrvNum);
						break;
					}
				}

				// store info about write attempt
				MediaEvent.BlockedIrp.wMajorFunction = AttemptData.Data[i].wMajorFunction;
				MediaEvent.BlockedIrp.dIoControlCode = AttemptData.Data[i].dIoControlCode;
				MediaEvent.BlockedIrp.dUpper = AttemptData.Data[i].dUpper;
				MediaEvent.BlockedIrp.dLower = AttemptData.Data[i].dLower;

				// send event notification to all views
				pDoc->AddMediaEvent(MediaEvent);
				pDoc->UpdateAllViews(NULL);
			}
		}
	}
}

BOOL CMainFrame::AddPhysicalDevice(NTWBPMMEDIAEVENTSTRUCT MediaEvent, PDEVINST pdnDevInst)
{
	char szSymbolicLink[260];

	// convert symbolic link from \??\XXX#{YYY} to \\?\XXX#{YYY}
	strcpy(szSymbolicLink, MediaEvent.szSymbolicLinkName);
	szSymbolicLink[1] = '\\';

	return LocalDevices.AddStorageDevice(szSymbolicLink, pdnDevInst);
}

BOOL CMainFrame::RemovePhysicalDevice(NTWBPMMEDIAEVENTSTRUCT MediaEvent, PDEVINST pdnDevInst)
{
	STORAGE_DEVICE StorageDevice;
	char szSymbolicLink[260];

	// convert symbolic link from \??\XXX#{YYY} to \\?\XXX#{YYY}
	strcpy(szSymbolicLink, MediaEvent.szSymbolicLinkName);
	szSymbolicLink[1] = '\\';

	// if the storage device is in our list, remove it
	if (LocalDevices.GetStorageDeviceInfo(szSymbolicLink, &StorageDevice)) {
		
		*pdnDevInst = StorageDevice.dnDevInst;
		LocalDevices.RemoveStorageDevice(StorageDevice.dnDevInst);
		return TRUE;
	}

	return FALSE;
}

BOOL CMainFrame::AddPhysicalVolume(NTWBPMMEDIAEVENTSTRUCT MediaEvent, PDEVINST pdnDevInst)
{
	STORAGE_DEVICE StorageDevice;
	char szLink[128];
	char szDrive[3] = "A:";
	BOOL bFound = FALSE;

	// find drive letter corresponding to the device described in the media 
	// event
	for (char c='A'; c <= 'Z'; c++) {
		szDrive[0] = c;

		if (QueryDosDevice(szDrive, szLink, sizeof(szLink))) {

			if (stricmp(MediaEvent.szNtDeviceName, szLink) == 0) {
				bFound = TRUE;
				break;
			}
		}
	}

	// if we know what drive letter this corresponds to...
	if (bFound) {
		STORAGE_TYPE eType;

		// get device type
		switch (MediaEvent.wDrvType) {
		case NTWBPM_DRVTYPE_FLOPPY:
			eType = FLOPPY_DISK;
			break;
		case NTWBPM_DRVTYPE_CDROM:
			eType = CDROM_DISK;
			break;
		case NTWBPM_DRVTYPE_HD:
			eType = IDE_DISK;
			break;
		default:
			eType = UNKNOWN_DISK;
		}

		// get info about the device
		if (LocalDevices.GetStorageDeviceInfo(eType, MediaEvent.wDrvNum, 
			&StorageDevice)) {
			
			// add the drive letter to the device's drive letters list
			LocalDevices.AddDriveLetterToStorageDevice(StorageDevice.dnDevInst,
				c);
			*pdnDevInst = StorageDevice.dnDevInst;
			return TRUE;
		}
	}

	return FALSE;
}

void CMainFrame::OnToolsReinstall() 
{
	CInstall InstallDlg;

	// if Write Blocker drivers already installed, prompt user to continue
	if (DriversInterface.IsInstalled()) {
		CString csText;

		csText.Format("%s drivers are already installed.\n Do you wish to reinstall?",
			WRITEBLOCKER_APP_NAME);

		if (MessageBox(csText, WRITEBLOCKER_APP_NAME, MB_YESNO | 
			MB_ICONEXCLAMATION) == IDNO)
			return;
	}

	// display dialog to reinstall drivers
	InstallDlg.DoModal();
}

void CMainFrame::OnToolsUninstall() 
{
	// make sure Write Blocker drivers are installed
	if (DriversInterface.IsInstalled()) {
		CUninstall UninstallDlg;

		// display dialog to uninstall drivers
		UninstallDlg.DoModal();
	}
	else {
		CString csText;

		csText.Format("%s drivers are not installed.", WRITEBLOCKER_APP_NAME);
		MessageBox(csText, WRITEBLOCKER_APP_NAME, MB_OK | MB_ICONEXCLAMATION);
	}
}

void CMainFrame::OnUpdateToolsUninstall(CCmdUI* pCmdUI) 
{
	// enable/disable the menu item
	pCmdUI->Enable(m_bDriversInstalled);
}

void CMainFrame::OnToggleWriteblock() 
{
	// if top view has been created, call its routine to toggle the write block
	// status of the selected media
	if (m_pTopView)
		m_pTopView->ToggleWriteblockingForSelectedMedia();
}

void CMainFrame::EnableToggleButton(BOOL bEnable)
{
	CToolBarCtrl* pToolBarCtrl;

	// get pointer to underlying toolbar control
	pToolBarCtrl = &(m_wndToolBar.GetToolBarCtrl());

	// enable/disable the button to toggle write blocking status
	m_bEnableToggleBtn = bEnable;
	pToolBarCtrl->EnableButton(IDM_TOGGLE_WRITEBLOCK, m_bEnableToggleBtn);
}

void CMainFrame::OnUpdateToggleWriteblock(CCmdUI* pCmdUI) 
{
	// enable/disable the toolbar toggle button
	pCmdUI->Enable(m_bEnableToggleBtn);
}

void CMainFrame::OnFileClear() 
{
	// clear event log of write attempts
	if (m_pEventLogView)
		m_pEventLogView->ClearEventLog();
}

void CMainFrame::OnUpdateFileClear(CCmdUI* pCmdUI) 
{
	BOOL bEnable = FALSE;

	// check if there are any events in the event log
	if (m_pEventLogView) {
		CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pEventLogView->GetDocument();

		// if there is at least one event in the event log, enable the menu 
		// item
		if (pDoc->GetEventLogEntryCount() > 0)
			bEnable = TRUE;
	}
	
	pCmdUI->Enable(bEnable);
}

void CMainFrame::OnUpdateFileSave(CCmdUI* pCmdUI) 
{
	BOOL bEnable = FALSE;

	// check if there are any events in the event log
	if (m_pEventLogView) {
		CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pEventLogView->GetDocument();

		// if there is at least one event in the event log, enable the menu 
		// item
		if (pDoc->GetEventLogEntryCount() > 0)
			bEnable = TRUE;
	}
	
	pCmdUI->Enable(bEnable);
}

void CMainFrame::OnUpdateFileSaveAs(CCmdUI* pCmdUI) 
{
	BOOL bEnable = FALSE;

	// check if there are any events in the event log
	if (m_pEventLogView) {
		CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pEventLogView->GetDocument();

		// if there is at least one event in the event log, enable the menu 
		// item
		if (pDoc->GetEventLogEntryCount() > 0)
			bEnable = TRUE;
	}
	
	pCmdUI->Enable(bEnable);
}

LRESULT CMainFrame::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
{
	// process user messages from event thread
	switch(message)
	{
	case WM_USER + WM_PROC_EVENTS_MSG:
		ProtectedProcessMediaEvents();
		break;
	case WM_USER + WM_UPDATE_VIEWS_MSG:
		UpdateViewsForThread(lParam);	
		break;
	}

	return CFrameWnd::WindowProc(message, wParam, lParam);
}

void CMainFrame::UpdateViewsForThread(LONG lParam)
{
	PMEDIA_EVENT pMediaEvent = (PMEDIA_EVENT) lParam;
	CWriteblockerDoc* pDoc = (CWriteblockerDoc*) m_pEventLogView->GetDocument();

	pDoc->AddMediaEvent(*pMediaEvent);
	pDoc->UpdateAllViews(NULL);

	// free event memory
	free(pMediaEvent);
}

void CMainFrame::ProtectedProcessMediaEvents()
{
	__try {
		// request ownership
		EnterCriticalSection(&CritSect);
		// process device arrivals and removals
		ProcessMediaEvents();
	}
	__finally 
	{
		// release critical section
		LeaveCriticalSection(&CritSect);
		// release section object
	}

	HANDLE hMediaProcEvent = OpenEvent(EVENT_ALL_ACCESS, FALSE, "MediaEventsProcessed");
	SetEvent(hMediaProcEvent);
	CloseHandle(hMediaProcEvent);
}
